#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/signal.h>
#include <syslog.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <netdb.h>
#include <stdio.h>
#include <ctype.h>
#include <strings.h>
#include "socks.h"

#include "ident.h"
IDENT	*ident_lookup(), *idp;
#define IDENTD_TIMEOUT 15 /* 15 seconds */
int	use_identd = 0;
char 	real_user[128];

#define STREQ(a, b)	(strcmp(a, b) == 0)

char *c_cmd;
char socks_cmd[] = "connect";

char *sockd_conf = SOCKD_CONF;
char server_version[] = "4.0";
u_short socks_port;
#define NAMELEN	128
extern char	*porttoserv();
extern char	*saddrtoname();
char	src_name[NAMELEN], src_user[NAMELEN], real_user[NAMELEN];
char	dst_name[NAMELEN], dst_port[NAMELEN];

char log_msg[1024];

#ifdef DEBUG
char buf[1024];
#endif

#ifndef TEST

/*
**  Current version for response messages
*/
int		Version = 0;

die()
{
	exit(1);
}


main(argc, argv)
int	argc;
char	*argv[];
{
	char			c;
	int			inp, in, out, index=0;
	int			n, len = sizeof(struct sockaddr_in);
	struct sockaddr_in	sin, from, dstsin;
	int			fromlen = sizeof(struct sockaddr_in);
	Socks_t			dst;
	int			one = 1;
	struct servent	*sp;
	int	bad_id = 0;

	socks_port = htons(SOCKS_DEF_PORT);

	if (argc >= 2) {
		if (strcmp(argv[1],"-ver") == 0) {
		printf("\nSOCKS proxy server ver. %s.\n", server_version);
		printf("Uses protocol version %d.\n\n", SOCKS_VERSION);
		exit(1);
		} else if (strcmp(argv[1], "-i") == 0)
			use_identd = 1;
		else if (strcmp(argv[1], "-I") == 0)
			use_identd = 2; /* strict use of identd */
		else ;
	}
	strcpy(real_user,"unknown");

	if ((sp = getservbyname("socks", "tcp")) != NULL)
		socks_port = sp->s_port;

#ifndef LOG_DAEMON
	(void) openlog("sockd", LOG_PID);
#else
	(void) openlog("sockd", LOG_PID, LOG_DAEMON);
#endif

#ifdef NOT_THROUGH_INETD
	inp  = socket(AF_INET, SOCK_STREAM, 0);
	sin.sin_family = AF_INET;
	sin.sin_port = socks_port;
	sin.sin_addr.s_addr = htonl(INADDR_ANY);

	if (bind(inp, &sin, sizeof(sin)) < 0) {
		syslog(LOG_NOTICE, "error -- main bind() %m");
		exit(1);
	}

	if (listen(inp, MAX_CLIENTS) < 0) {
		syslog(LOG_NOTICE, "error -- main listen() %m");
		exit(1);
	}

	if ((in = accept(inp, &sin, &len)) < 0) {
		syslog(LOG_NOTICE, "error -- main accept() %m");
		exit(1);
	}
#else
	in = dup(0);
#endif

	if (getpeername(in, &from, &fromlen) < 0) {
		syslog(LOG_NOTICE, "error -- unable to get client address.");
		exit(1);
	}

	saddrtoname(&from.sin_addr, src_name, sizeof(src_name));

	GetDst(in, &dst);


	if (dst.version != SOCKS_VERSION) {
		syslog(LOG_NOTICE, "error -- wrong version (0x%2x) from host %s.",
			dst.version, src_name);
		exit(1);
	}

	if (dst.cmd != SOCKS_CONNECT && dst.cmd != SOCKS_BIND) {
		syslog(LOG_NOTICE, "error -- undefined command (0x%2x) from host %s",
			dst.cmd, src_name);
		exit(1);
	}

	dstsin.sin_family      = AF_INET;
	dstsin.sin_addr.s_addr = dst.host;
	dstsin.sin_port        = dst.port;

	while (read(in, &c, 1) == 1)
		if (c == '\0')
			break;
		else {
			if (index < sizeof(src_user) - 1)
			src_user[index++] = c;
		}
	src_user[index] = '\0';
	saddrtoname(&dstsin.sin_addr, dst_name, sizeof(dst_name));
	porttoserv(dstsin.sin_port, dst_port, sizeof(dst_port));


	if (use_identd) {
		if ((idp = ident_lookup(in, IDENTD_TIMEOUT)) == ((IDENT *)0)) {
			syslog(LOG_NOTICE, "cannot connect to identd on %s", src_name);
			check_sp_conf("#NO_IDENTD:", &from, &dstsin);
			if (use_identd == 2) {
				bad_id = 1;
				dst.cmd = SOCKS_NO_IDENTD;
			}
		} else {
			strncpy(real_user, idp->identifier, sizeof(real_user));
			if (strcmp(src_user, real_user) != 0) {
				syslog(LOG_NOTICE, "*Alert*: real user is %s, not %s", real_user, src_user);
				bad_id = 1;
				check_sp_conf("#BAD_ID:", &from, &dstsin);
				dst.cmd = SOCKS_BAD_ID;
			}
		}
		ident_free(idp);
	}

	if (dst.cmd == SOCKS_CONNECT) {
		strcpy(socks_cmd, "connect");
		sprintf(log_msg, "Connect from %s(%s)@%s to %s (%s)",
			src_user, real_user, src_name, dst_name, dst_port);
	} else {
		strcpy(socks_cmd, "bind");
		sprintf(log_msg, "Bind from %s(%s)@%s for %s",
			src_user, real_user, src_name, dst_name);
	}

	if (bad_id) {
		syslog(LOG_NOTICE, "refused -- %s", log_msg);
		SendDst(in, &dst);
		exit(1);
	}

#ifdef DEBUG
	strcpy(buf, inet_ntoa(from.sin_addr));
	syslog(LOG_NOTICE,"USER:%s,  SRC:%s,  DST:%s, PORT:%u",
		src_user, buf, inet_ntoa(dstsin.sin_addr),
		ntohs(dstsin.sin_port));
#endif
	if (!Validate(&from, &dstsin, src_user)) {
		syslog(LOG_NOTICE, "refused -- %s", log_msg);
		exit(1);
	}
	
/*
	syslog(LOG_NOTICE, "request -- %s", log_msg);
*/

	/*
	**  Kill a connecting off if bind or connect takes too
	**    long to complete
	*/
	signal(SIGALRM, die);
	alarm(60*5);		/* 5 minutes */

	if (dst.cmd == SOCKS_CONNECT) {
		DoConnect(in, &dst);
	}
	if (dst.cmd == SOCKS_BIND) {
		DoNewBind(in, &dst);
	}
}

socks_fail(str, in, ndst)
char	*str;
int	in;
Socks_t	*ndst;
{
	syslog(LOG_NOTICE, "failed -- %s.  Error code: %s %m", log_msg, str);
	ndst->cmd = SOCKS_FAIL;
	SendDst(in, ndst);
	exit(1);
}


/*
** Actually connect a socket to the outside world,
*/
DoConnect(in, dst)
int	in;
Socks_t	*dst;
{
	int			out;
	struct sockaddr_in	sin;
	Socks_t			ndst;

	if ((out = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		socks_fail("socket()", in, &ndst);

	sin.sin_family = AF_INET;
	sin.sin_port = dst->port;
	sin.sin_addr.s_addr = dst->host;

	ndst.version = Version;
	ndst.cmd = SOCKS_RESULT;

	if (connect(out, &sin, sizeof(struct sockaddr_in)) < 0)
		socks_fail("connect()", in, &ndst);

	syslog(LOG_NOTICE, "connected -- %s", log_msg);
	SendDst(in, &ndst);

	Pump(in, out);
}


/*
**  Set up a socket to be connected to from the outside world.
**   diffrence between this an the Version1 protocal is that
**   the socket has to be bound from a specific host that
**   is passed.
*/
DoNewBind(in, dst)
int	in;
Socks_t	*dst;
{
	int			new, out, len = sizeof(struct sockaddr_in);
	struct sockaddr_in	sin;
	Socks_t			ndst;
	char	dsthost[16];
	char	dst_name[NAMELEN], dst_serv[NAMELEN];

	if ((out = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		socks_fail("socket()", in, &ndst);

	sin.sin_family = AF_INET;
	ndst.version = Version;
	ndst.cmd  = SOCKS_RESULT;
	sin.sin_port = htons(0);
	sin.sin_addr.s_addr = htonl(INADDR_ANY);

	if (bind(out, &sin, sizeof(sin)) < 0)
		socks_fail("bind()", in, &ndst);
	if (getsockname(out, &sin, &len) < 0)
		socks_fail("getsockname()", in, &ndst);

	ndst.port = sin.sin_port;
	ndst.host = sin.sin_addr.s_addr;

	if (listen(out, 1) < 0)
		socks_fail("listen()", in, &ndst);

	SendDst(in, &ndst);

	len = sizeof(struct sockaddr_in);
	if ((new = accept(out, &sin, &len)) < 0)
		socks_fail("accept()", in, &ndst);

	close(out);

	
	saddrtoname(&sin.sin_addr, dst_name, sizeof(dst_name));
	porttoserv(sin.sin_port, dst_serv, sizeof(dst_serv));

	if (sin.sin_addr.s_addr != dst->host) {
		strncpy(dsthost, inet_ntoa(&dst->host), sizeof(dsthost));
		syslog(LOG_NOTICE, "failed -- %s. Error: connected to wrong host %s (%s)",
			log_msg, dst_name, dst_serv);
		ndst.cmd = SOCKS_FAIL;
		SendDst(in, &ndst);
		exit(1);
	}

	syslog(LOG_NOTICE, "connected -- %s (%s)", log_msg, dst_serv);
	ndst.port = sin.sin_port;
	ndst.host = sin.sin_addr.s_addr;
	SendDst(in, &ndst);

	Pump(in, new);
}

/*
**  Now just pump the packets/character through..
*/
Pump(in, out)
int	in, out;
{
	static char		buf[4096];
	fd_set			fds;
	int			n;
	int			nfds = getdtablesize();
	static struct timeval	tout = { SOCKS_TIMEOUT, 0 };

	alarm(0);

	FD_ZERO(&fds);

	while (1) {
		tout.tv_sec = SOCKS_TIMEOUT;
		tout.tv_usec = 0;
		FD_SET(in, &fds);
		FD_SET(out, &fds);
		if ((n = select(nfds, &fds, NULL, NULL, &tout)) > 0) {
			if (FD_ISSET(in, &fds)) {
				if ((n = read(in, buf, sizeof buf)) > 0) {
					if (write(out, buf, n) < 0) {
						goto bad;
					}
				} else {
					goto bad;
				}
			}
			if (FD_ISSET(out, &fds)) {
				if ((n = read(out, buf, sizeof buf)) > 0) {
					if (write(in, buf, n) < 0) {
						goto bad;
					}
				} else {
					goto bad;
				}
			}
		} else {
			if (n != 0)
				syslog(LOG_NOTICE, "select %m\n");
			goto bad;
		}
	}

bad:
	;	/* Make the goto happy */
}

#endif /* #ifndef TEST */

check_sp_conf(s, src, dst)
char	*s;
struct sockaddr_in	*src, *dst;
{
	FILE		*fd;
	char	buf[1024], *p;
#ifdef TEST
	int		linenum = 0;
#endif

	if ((fd = fopen(sockd_conf, "r")) == NULL) {
		syslog(LOG_ERR, "Unable to open config file (%s)", sockd_conf);
		return;
	}

	while (fgets(buf, sizeof(buf) - 1, fd) != NULL) {
		if ((p = index(buf, '\n')) != NULL)
			*p = '\0';
#ifdef TEST
		linenum++;
#endif
		if (strncmp(s, buf, strlen(s)) == 0) {
#ifdef TEST
		fprintf(stderr,"Line %d: %s\n", linenum, buf);
#endif
			shell_cmd(buf+strlen(s), src, dst);
			break;
		}
	}
	fclose(fd);
	return;
}

/*
**  Simple 'mkargs' doesn't handle \, ", or '.
*/
void mkargs(cp, argc, argv, max)
char	*cp;
int	*argc;
char	*argv[];
int	max;
{
	*argc = 0;
	while (isspace(*cp))
		cp++;

	while (*cp != '\0') {
		argv[(*argc)++] = cp;
		if (*argc >= max)
			return;

		while (!isspace(*cp) && (*cp != '\0'))
			cp++;
		while (isspace(*cp))
			*cp++ = '\0';
	}
}

/* 
**  Get address, either numeric or dotted quad, or hex.
 *  Result is in network byte order.
*/
int GetAddr(name, addr)
char		*name;
unsigned long	*addr;
{
	struct hostent	*hp;
	struct netent	*np;

	if ((hp = gethostbyname(name)) != NULL) {
		bcopy(hp->h_addr_list[0], addr, sizeof(*addr));
		return *addr;
	}
	if ((np = getnetbyname(name)) != NULL) {
		bcopy(&np->n_net, addr, sizeof(*addr));
		return *addr;
	}
	return *addr = inet_addr(name);
}

long GetPort(name)
char		*name;
/* result is in HOST byte order */
{
	struct servent	*sp;

	if ((sp = getservbyname(name, "tcp")) != NULL) {
		return ntohs(sp->s_port);
	}
	if (!isdigit(*name))
		return -1;
	return atol(name);
}


int check_user(userlist, src_user)

char	*userlist, *src_user;

/*
 * Unless userlist is a null pointer, in which case all users are
 * allowed (return 1), otherwise
 * userlist is a nonempty string containing userids separated by
 * commas, no other separators are allowed in the string.
 *
 * Return 1 if src_user is in the userlist; 
 * return 0 if not.
 */
{
	char	*p, *q;

	if (!(p = userlist)) {
		return 1;
	}
	do {
		if (q = index(p, ','))
			*q++ = '\0';
		if (strcmp(p, src_user) == 0) {
			return 1;
		}
	} while ( p = q);

	return 0;
}

Validate(src, dst, src_user)
struct sockaddr_in	*src, *dst;
char	*src_user;
{
	FILE		*fd;
	static char	buf[1024];
#ifdef TEST
	char		temp[1024];
#endif
	char		*bp;
	int		linenum = 0, permit;
	char		*argv[10];
	int		argc;
	unsigned long	saddr, smask, daddr, dmask;
	unsigned short	dport;
	enum 		{ e_lt, e_gt, e_eq, e_neq, e_le, e_ge, e_nil } tst;
	char		*userlist;
	int		next_arg;
	unsigned short	dst_sin_port = ntohs(dst->sin_port); /* dst->sin_port in host byte order */
	long	p;
	char	*cmdp;


	if ((fd = fopen(sockd_conf, "r")) == NULL) {
#ifdef TEST
		printf("Unable to open config file (%s)\n", sockd_conf);
#else /* #ifdef TEST */
		syslog(LOG_ERR, "Unable to open config file (%s)", sockd_conf);
#endif /* #ifdef TEST */

		return 0;
	}

	while (fgets(buf, sizeof(buf) - 1, fd) != NULL) {
		linenum++;
#ifdef TEST
		strcpy(temp, buf);
#endif
		/*
		**  Comments start with a '#' anywhere on the line
		*/
		cmdp = (char *)0;
		if ((bp = index(buf, '\n')) != NULL)
			*bp = '\0';
		for (bp = buf; *bp != '\0'; bp++) {
			if (*bp == ':') {
				*bp++ = '\0';
				cmdp = bp;
				break;
			} else if (*bp == '#') {
				*bp = '\0';
				break;
			} else if (*bp == '\t')
				*bp = ' ';
		}

		mkargs(buf, &argc, argv, 9);
		if (argc == 0)
			continue;
		if ((argc < 3) || (argc > 8)) {
#ifdef TEST
			printf("Invalid entry at line %d\n", linenum);
#else
			syslog(LOG_NOTICE, "Invalid entry at line %d", linenum);
#endif
			continue;
		}

		if (STREQ(argv[0], "permit")) {
			permit = 1;
		} else if (STREQ(argv[0], "deny")) {
			permit = 0;
		} else {
#ifdef TEST
			printf("Invalid permit/deny field at line %d\n", linenum);
#else
			syslog(LOG_NOTICE, "Invalid permit/deny field at line %d", linenum);
#endif
			continue;
		}

		userlist = (char *)0;
		next_arg = 1;

		if (strncmp(argv[1], "*=", 2) == 0) {
			next_arg++;
			if (argv[1][2]) userlist = argv[1] + 2;
		}

		GetAddr(argv[next_arg++], &saddr);
		GetAddr(argv[next_arg++], &smask);

		if ((argc > next_arg + 1) &&
			!(STREQ(argv[next_arg], "eq") || STREQ(argv[next_arg], "neq") ||
			  STREQ(argv[next_arg], "lt") || STREQ(argv[next_arg], "gt") ||
			  STREQ(argv[next_arg], "le") || STREQ(argv[next_arg], "ge"))) {
			GetAddr(argv[next_arg++], &daddr);
			GetAddr(argv[next_arg++], &dmask);

		} else {
			daddr = 0;
			dmask = ~0;
		}
		if (argc > next_arg + 1) {
			if (STREQ(argv[next_arg], "eq"))
				tst = e_eq;
			else if (STREQ(argv[next_arg], "neq"))
				tst = e_neq;
			else if (STREQ(argv[next_arg], "lt"))
				tst = e_lt;
			else if (STREQ(argv[next_arg], "gt"))
				tst = e_gt;
			else if (STREQ(argv[next_arg], "le"))
				tst = e_le;
			else if (STREQ(argv[next_arg], "ge"))
				tst = e_ge;
			else {
#ifdef TEST
				printf("Invalid comparison at line %d\n", linenum);
#else
				syslog(LOG_NOTICE, "Invalid comparison at line %d", linenum);
#endif
				continue;
			}
				
			if (((p = GetPort(argv[next_arg+1])) < 0) ||
				(p >= (1L << 16))) {
#ifdef TEST
				printf("Invalid port number at line %d\n", linenum);
#else
				syslog(LOG_NOTICE, "Invalid port number at line %d", linenum);
#endif
				continue;
			} else {
				dport = p;
			}
		} else {
			tst = e_nil;
			dport = 0;
		}

#ifdef DEBUG
		{
			char msg[1024];
			if (userlist) 
				sprintf(msg,"%s %s 0x%08x 0x%08x 0x%08x 0x%08x %s %u",
					permit ? "permit" : "deny",
					userlist,
					saddr, smask, daddr, dmask,
						tst == e_eq ? "==" :
						tst == e_neq ? "!=" :
						tst == e_lt ? "<" :
						tst == e_gt ? ">" :
						tst == e_le ? "<=" :
						tst == e_ge ? ">=" : "NIL",
						dport);
			else
				sprintf(msg,"%s 0x%08x 0x%08x 0x%08x 0x%08x %s %u",
					permit ? "permit" : "deny",
					saddr, smask, daddr, dmask,
						tst == e_eq ? "==" :
						tst == e_neq ? "!=" :
						tst == e_lt ? "<" :
						tst == e_gt ? ">" :
						tst == e_le ? "<=" :
						tst == e_ge ? ">=" : "NIL",
						dport);
			syslog(LOG_NOTICE, "%s", msg);
		}
#endif
	/* comparisons of port numbers must be done in host order */

		if((saddr & ~smask) == (src->sin_addr.s_addr & ~smask) &&
		    (daddr & ~dmask) == (dst->sin_addr.s_addr & ~dmask) &&
		    check_user(userlist, src_user)) {
			switch (tst) {
			case e_eq:
				if (dst_sin_port == dport) {
#ifdef TEST
					printf("Line %d: %s", linenum, temp);
#endif
					fclose(fd);
					if (cmdp != (char *)0)
						shell_cmd(cmdp, src, dst);
					return permit;
				}
				break;
			case e_neq:
				if (dst_sin_port != dport) {
#ifdef TEST
					printf("Line %d: %s", linenum, temp);
#endif
					fclose(fd);
					if (cmdp != (char *)0)
						shell_cmd(cmdp, src, dst);
					return permit;
				}
				break;
			case e_lt:
				if (dst_sin_port < dport) {
#ifdef TEST
					printf("Line %d: %s", linenum, temp);
#endif
					fclose(fd);
					if (cmdp != (char *)0)
						shell_cmd(cmdp, src, dst);
					return permit;
				}
				break;
			case e_gt:
				if (dst_sin_port > dport) {
#ifdef TEST
					printf("Line %d: %s", linenum, temp);
#endif
					fclose(fd);
					if (cmdp != (char *)0)
						shell_cmd(cmdp, src, dst);
					return permit;
				}
				break;
			case e_le:
				if (dst_sin_port <= dport) {
#ifdef TEST
					printf("Line %d: %s", linenum, temp);
#endif
					fclose(fd);
					if (cmdp != (char *)0)
						shell_cmd(cmdp, src, dst);
					return permit;
				}
				break;
			case e_ge:
				if (dst_sin_port >= dport) {
#ifdef TEST
					printf("Line %d: %s", linenum, temp);
#endif
					fclose(fd);
					if (cmdp != (char *)0)
						shell_cmd(cmdp, src, dst);
					return permit;
				}
				break;
			case e_nil:
#ifdef TEST
				printf("Line %d: %s", linenum, temp);
#endif
				fclose(fd);
				if (cmdp != (char *)0)
					shell_cmd(cmdp, src, dst);
				return permit;
			}
		}
	}

	fclose(fd);
#ifdef TEST
	printf("*** No match with any line. Access denied.\n");
#endif
	return 0;
}
/*-------------------------------------------------*/

#ifdef TEST

char 	*testpname;

error_and_quit()
{
	printf("Usage: %s [-C conf_file] user src_addr dst_addr {service,port}\n", testpname);
		exit(-1);
}

main(argc, argv)
int	argc;
char	*argv[];

/*
 * test_sockd_conf [-i] [-I] [-B] [-C conf_file] user src_addr dst_addr {service,port}
 *
 * Prints out the line in configuration file that are examined and
 * whether the access is permitted or denied. Sets exit status code
 * to 1 for permit, 0 for deny, -1 for error in command line.
 */

{
	int next = 1;
	long	p;
	u_short	port;
	struct sockaddr_in	from, dstsin;
	char	log_msg[1024];
	char	command = SOCKS_CONNECT;
	int	bad_id = 0;
	int	no_identd = 0;
	char	buf[1024];

	if(testpname = rindex(argv[0], '/'))
		++testpname;
	else
		testpname = argv[0];

	while (--argc && (argv[next][0] == '-')) {
		switch (argv[next++][1]) {
		case 'B':
			bad_id = 1;
			break;
		case 'C':
			if (--argc) {
				sockd_conf =argv[next++];
				break;
			} else
				error_and_quit();
		case 'I':
			no_identd = 2;
			break;
		case 'i':
			no_identd = 1;
			break;
		default:
			error_and_quit();
		}
	}

	if (argc != 4) 
			error_and_quit();

	strncpy(src_user, argv[next++], sizeof(src_user));
	GetAddr(argv[next++], &from.sin_addr.s_addr);
	GetAddr(argv[next++], &dstsin.sin_addr.s_addr);
	if (((p = GetPort(argv[next])) < 0) || (p >= (1L << 16))) {
		printf("%s: Unknown service or illegal port number '%s'\n",testpname, argv[next]);
		exit(-1);
	}
	dstsin.sin_port = htons(port = p);

	if (bad_id || no_identd)
		strcpy(real_user,"unknown");
	else
		strcpy(real_user, src_user);

	saddrtoname(&from.sin_addr, src_name, sizeof(src_name));
	saddrtoname(&dstsin.sin_addr, dst_name, sizeof(dst_name));
	porttoserv(dstsin.sin_port, dst_port, sizeof(dst_port));
	sprintf(log_msg, "connect from %s(%s)@%s to %s (%s)",
		src_user, real_user, src_name, dst_name, dst_port);
#ifdef LOG_DAEMON
	openlog(testpname, LOG_PID, LOG_DAEMON);
#else
	openlog(testpname, LOG_PID);
#endif

	strcpy(buf, inet_ntoa(from.sin_addr));
	printf("USER:%s,  SRC:%s,  DST:%s, PORT:%u\n",
		src_user, buf, inet_ntoa(dstsin.sin_addr),
		ntohs(dstsin.sin_port));

	if (bad_id) {
		syslog(LOG_NOTICE, "*Alert*: real user is %s, not %s", real_user, src_user);
		syslog(LOG_NOTICE, "refused -- %s", log_msg);
		check_sp_conf("#BAD_ID:", &from, &dstsin);
		fprintf(stderr, "Access denied: bad user-id.\n");
		return(0);
	} else if (no_identd == 2) {
		syslog(LOG_NOTICE, "cannot connect to identd on %s", src_name);
		check_sp_conf("#NO_IDENTD:", &from, &dstsin);
		syslog(LOG_NOTICE, "refused -- %s", log_msg);
		fprintf(stderr, "Access denied: cannot verify user-id.\n");
		return(0);
	}
	if (no_identd == 1) {
		syslog(LOG_NOTICE, "cannot connect to identd on %s", src_name);
		check_sp_conf("#NO_IDENTD:", &from, &dstsin);
	}
	if (Validate (&from, &dstsin, src_user)) {
		syslog(LOG_NOTICE, "connected -- %s", log_msg);
		return(1);
	} else {
		syslog(LOG_NOTICE, "refused -- %s", log_msg);
		return(0);
	}
	
}

#endif /* #ifdef TEST */
