=head1 NAME

Crypt::Ed25519 - bare-bones Ed25519 public key signing/verification system

=head1 SYNOPSIS

 use Crypt::Ed25519; # no symbols exported

 # generate a public/private key pair once
 ($pubkey, $privkey) = Crypt::Ed25519::generate_keypair;

 # sign messages
 $signature = Crypt::Ed25519::sign $message, $pubkey, $privkey;

 # verify message
 $valid = Crypt::Ed25519::verify $message, $pubkey, $signature;

 # verify, but croak on failure
 Crypt::Ed25519::verify_croak $message, $pubkey, $signature;

=head1 DESCRIPTION

This module implements Ed25519 public key generation, message signing and
verification. It is a pretty bare-bones implementation that implements
the standard Ed25519 variant with SHA512 hash, as well as a slower API
compatible with the upcoming EdDSA RFC.

The security target for Ed25519 is to be equivalent to 3000 bit RSA or
AES-128.

The advantages of Ed25519 over most other signing algorithms are:
small public/private key and signature sizes (<= 64 octets), good key
generation, signing and verification performance, no reliance on random
number generators for signing and by-design immunity against branch or
memory access pattern side-channel attacks.

More detailed praise and other info can be found at
L<http://ed25519.cr.yp.to/index.html>.

=cut

package Crypt::Ed25519;

BEGIN {
   $VERSION = '0.9';

   require XSLoader;
   XSLoader::load Crypt::Ed25519::, $VERSION;
}

=head1 Ed25519 API

=over 4

=item ($public_key, $private_key) = Crypt::Ed25519::generate_keypair

Creates and returns a new random public and private key pair. The public
key is always 32 octets, the private key is always 64 octets long.

=item $signature = Crypt::Ed25519::sign $message, $public_key, $private_key

Generates a signature for the given message using the public and private keys.

=item $valid = Crypt::Ed25519::verify $message, $public_key, $signature

Checks whether the C<$signature> is valid for the C<$message> and C<$public_ke>.

=item Crypt::Ed25519::verify_croak $message, $public_key, $signature

Same as C<Crypt::Ed25519::verify>, but instead of returning a boolean,
simply croaks with an error message when the signature isn't valid, so you
don't have to think about what the return value really means.

=back

=head1 EdDSA compatible API

The upcoming EdDSA draft RFC uses a slightly different (and slower)
API for Ed25519. This API is provided by the following functions:

=over 4

=item $secret_key = Crypt::Ed25519::eddsa_secret_key

Creates and returns a new secret key, which is always 32 octets
long. The secret key can be used to generate the public key via
C<Crypt::Ed25519::eddsa_public_key> and is not the same as the private key
used in the Ed25519 API.

=item $public_key = Crypt::Ed25519::eddsa_public_key $secret_key

Takes a secret key generated by C<Crypt::Ed25519::eddsa_secret_key> and
returns the corresponding C<$public_key>.

This public key corresponds to the public key in the Ed25519 API above.

=item $signature = Crypt::Ed25519::eddsa_sign $message, $public_key, $secret_key

Generates a signature for the given message using the public and secret
keys.

=item $valid = Crypt::Ed25519::eddsa_verify $message, $public_key, $signature

=item Crypt::Ed25519::eddsa_verify_croak $message, $public_key, $signature

Really the same as C<Crypt::Ed25519::verify> and
C<Crypt::Ed25519::verify_croak>, i.e. the functions without the C<eddsa_>
prefix. These aliases are provided so it's clear that you are using EdDSA
and not Ed25519 API.

=back

=head1 CONVERTING BETWEEN Ed25519 and EdDSA

The Ed25519 and EdDSA compatible APIs handle keys slightly
differently: The Ed25519 API gives you a public/private key pair, while
EdDSA takes a secret and generates a public key from it.

You can convert an EdDSA secret to an Ed25519 private/public key pair
using C<Crypt::Ed25519::generate_keypair>:

   ($public_key, $private_key) = Crypt::Ed25519::generate_keypair $secret

=head2 IMPLEMENTATIOIN

This module currently uses "Nightcracker's Ed25519" implementation, but
the interface is kept implementation-agnostic to allow usage of other
implementations in the future.

=head1 AUTHOR

 Marc Lehmann <schmorp@schmorp.de>
 http://sfotware.schmorp.de/pkg/Crypt-Ed25519.html

=cut

1

