# $Id: Collection.pm,v 1.14 2003/08/13 14:05:22 gene Exp $

package URI::Collection;
use vars qw($VERSION); $VERSION = '0.08';
use strict;
use Carp;
use Cwd;
use File::Spec;
use File::Find;
use File::Path;
use IO::String;
use Config::IniFiles;
use Netscape::Bookmarks;

sub _debug {  # {{{
    print @_, "\n" if shift->{debug};
}  # }}}

sub new {  # {{{
    my ($class, %args) = @_;
    my $self = {
        debug => $args{debug} || 0,
        links => $args{links} || [],
        ignore_dups => $args{ignore_dups} || 0,
        cat_links => {},
    };
    bless $self, $class;
    $self->_init($args{links}) if $args{links};
    return $self;
}  # }}}

sub _init {  # {{{
    my $self = shift;

    $self->{links} = [ $self->{links} ]
        if $self->{links} && !ref($self->{links}) &&
            (-f $self->{links} || -d $self->{links});

    for (@{ $self->{links} }) {
        croak "$_ does not exist.\n" unless -d || -f;

        if (-d) {
            # Handle a M$ Windows Favorites directory tree.
            $self->_traverse($_);
        }
        elsif (-f) {
            # Handle a Netscape style bookmark file.
            $self->_parse_file($_);
        }
    }
#use Data::Dumper;warn Dumper($self->{cat_links});
}  # }}}

sub as_bookmark_file {  # {{{
    my ($self, %args) = @_;
$self->_debug('entering as_bookmark_file');

    # Make the top level bookmark category.
    my $top = Netscape::Bookmarks::Category->new({
        folded      => 0,
        title       => __PACKAGE__ .' Bookmarks',
        add_date    => time,
        description => 'Bookmarks generated by '. __PACKAGE__,
        id          => '?',
    });
$self->_debug("Created $top");

    # Declare a hash for storing the category objects.
    my %categories;

    # Make bookmark categories for the internal category paths.
    for my $path (sort keys %{ $self->{cat_links} }) {
        # Current category.
        my $category;

        # Make a bookmark category for each title in the category
        # path.
        # NOTE: This split means, "split on any forward slashes that
        # are not preceeded by a backslash."
        for my $title (split /(?<!\\)\//, $path) {
            # Set the current category to the top level if we are
            # just starting out.
            $category = $top unless $category;

            # Add an useen category.
            unless (exists $categories{$title}) {
                $categories{$title} = Netscape::Bookmarks::Category->new({
                    folded      => 0,
                    title       => $title,
                    add_date    => time,
                    description => '',
                    id          => '?',
                });
$self->_debug("Created unseen $categories{$title}");

                $category->add($categories{$title});
            }

            # "Increment" the current category with the one just seen.
            $category = $categories{$title};
        }

        # Add links to the last seen category.
        for (@{ $self->{cat_links}{$path} }) {
            $category->add(_make_bookmark(%$_));
$self->_debug('Added link for '. join ', ', %$_);
        }
    }

    # Save the bookmarks as a file, if told to.
    if ($args{save_as}) {
        open BOOKMARKS, "> $args{save_as}"
            or croak "Can't write $args{save_as} - $!\n";
        print BOOKMARKS $top->as_string;
        close BOOKMARKS;

        # Return the bookmark filename.
$self->_debug('exiting as_bookmark_file');
        return $args{save_as};
    }
    else {
        # Return the bookmark file contents.
        return $top->as_string;
    }
}  # }}}

sub as_favorite_directory {  # {{{
    my ($self, %args) = @_;
    my $favs;

    if ($args{save_as}) {
        # Create a top level directory for our Favorites.
        $favs = $args{save_as};
        mkpath $favs;
        chdir $favs;
        $favs = getcwd;
    }
    else {
        # Initialize the favorites structure.
        $favs = {};
    }

    # Build the Favorites tree with Internet Shortcut files.
    for my $path (keys %{ $self->{cat_links} }) {
        if ($args{save_as}) {
            mkpath $path;
            chdir $path;
        }

        # Add links to the path category.
        for (@{ $self->{cat_links}{$path} }) {
            my ($title, $url) = %$_;
            my $obj = _make_favorite($title, $url);

            if ($args{save_as}) {
                $obj->WriteConfig("$title.url");
            }
            else {
                # Aha! The undocumented OutputConfig method is what I
                # want.  Of course, it sucks and prints to STDOUT,
                # instead of returning a scalar.
                my $str;
                my $str_fh = IO::String->new($str);
                my $old_fh = select $str_fh;
                $obj->OutputConfig;
                select $old_fh if defined $old_fh;
                push @{ $favs->{$path} }, $str;
            }
        }

        # Change back to the top level path category directory.
        chdir $favs if $args{save_as};
    }

    # Return the top level directory or favorites structure.
    return $favs;
}  # }}}

# Return all similar links or categories.
# Gene: $self->{cat_links} is NOT NESTED. HTH. HAND.
sub fetch_items {  # {{{
    my ($self, %args) = @_;
$self->_debug('entering fetch_items');
    my %items;

    # Step through every link.
    while (my ($category, $links) = each %{ $self->{cat_links} }) {
#$self->_debug("Cat: $category");
        # If we are given a title, url or matching category...
        if (!(keys %args) || $args{title} || $args{url} ||
            _matches($category, $args{category})
        ) {
            # Fetch an entire category's links.
#$self->_debug("cat match: $category") if _matches($category, $args{category});
            for (@$links) {
                my ($title, $url) = %$_;
                # XXX Yikes. This condition is... giant!
                if (!(keys %args) ||
                    (_matches($category, $args{category}) &&
                        (
                            !($args{title} || $args{url})  ||
                            _matches($title, $args{title}) ||
                            _matches($url, $args{url})
                        )
                    )
                    ||
                    (!$args{category} &&
                        (
                            _matches($title, $args{title}) ||
                            _matches($url, $args{url})
                        )
                    )
                ) {
$self->_debug("link: $title => $url");
                    push @{ $items{$category} }, $_;
                }
            }
        }
    }

$self->_debug('exiting fetch_items');
    return \%items;
}  # }}}

# Alias for fetch_items restricted to a single title/url pattern.
sub is_item {  # {{{
    my ($self, $pat) = @_;
    my $items = $self->fetch_items(title => $pat, url => $pat);
    return keys %$items > 0 ? $items : undef;
}  # }}}

sub _matches {  # {{{
    my ($string, $pattern) = @_;
    return $pattern && $string =~ /$pattern/i
}  # }}}

sub _title_and_url {  # {{{
    my $link = shift;
    my ($title, $url);

    if (ref $link eq 'Netscape::Bookmarks::Link') {
        ($title, $url) = ($link->title, $link->href);
    }
    elsif (ref $link eq 'Config::IniFiles') {
        ($title, $url) = (
            $link->val('Title', 'Title'),
            $link->val('InternetShortcut', 'URL')
        );
    }

    return $title => $url;
}  # }}}

# Step over the Favorites directory and add the categories and links
# to our internal categories and links structure.
sub _traverse {  # {{{
    my ($self, $dir) = @_;
$self->_debug("entering _traverse with $dir");
    find (
        sub {
            if (/^(.+?)\.url$/) {
                # The file name - sans extension - is the title.
                my $title = $1;

                # Remove the Favorites tree path from the category
                # name.
                (my $category = $File::Find::dir) =~ s/^$dir//;
                # Set the top level category if we are there.
                $category = 'Favorites' unless $category;
                # Convert the (platform dependent) path separators to
                # slashes.  I.e., forward slashes in category names
                # are replaced with "back-slash escaped" forward
                # slashes ("\/").
                $category = join '/',
                    map { s!\/!\\/!g; $_ } grep { $_ }
                        File::Spec->splitdir($category);

                # Create a Winblows url object.
                my $obj = Config::IniFiles->new(
                    '-file' => "$title.url"
                );
                # Add a title section to the url object.
                $obj->AddSection('Title');
                $obj->newval('Title', 'Title', $title);

                # Add the category and link!
                push @{ $self->{cat_links}{$category} }, {
                    _title_and_url($obj)
                };
$self->_debug($self->{cat_links}{$category}[-1]);
            }
        },
        $dir  # This dangley bit is the directory into which we descend.
    );
$self->_debug('exiting _traverse');
}  # }}}

# Parse the given bookmarks file into our internal categories and
# links structure.
sub _parse_file {  # {{{
    my ($self, $file) = @_;
$self->_debug("entering _parse_file with $file");

    # Declare our categories list and current category title.
    my (@category, $category);
    # Define the last seen level as the top.
    my $last_level = 0;

    # Define a Netscape bookmarks object.
    my $nb = Netscape::Bookmarks->new($file);
    $nb->recurse(
        sub {
            my ($obj, $level) = @_;
$self->_debug("$obj @ $level");

            if ($obj->isa('Netscape::Bookmarks::Category')) {
                # Find the current / separated category name.
                if ($level > 0) {
                    if ($level <= $last_level) {
                        # XXX splice would be more idiomatic...
                        pop @category for 1 .. $last_level - $level + 1;
                    }

                    # Add the category title.
                    push @category, $obj->title;
                }

                # Set the current category and level.
                # NOTE that / is forced as the "path separator" here.
                $category = join '/', @category;
                $last_level = $level;
            }
            elsif ($obj->isa('Netscape::Bookmarks::Link')) {
                # Add the category and link to our internal structure.
                push @{ $self->{cat_links}{$category} }, {
                    $obj->title => $obj->href
                };
            }
        }
    );

$self->_debug('exiting _parse_file');
}  # }}}

sub _make_favorite {  # {{{
    my ($title, $url) = @_;
    # Define an Internet Shortcut object based on the given Netscape
    # bookmark object.
    my $obj = Config::IniFiles->new;
    $obj->AddSection('DEFAULT');
    $obj->newval('DEFAULT', 'BASEURL', $url);
    $obj->AddSection('InternetShortcut');
    $obj->newval('InternetShortcut', 'URL', $url);
    return $obj;
}  # }}}

sub _make_bookmark {  # {{{
    my ($title, $href) = @_;
#warn "$title, $href\n";
    # Define a Netscape bookmark link based on the given Internet
    # Shortcut object.
    return Netscape::Bookmarks::Link->new({
        TITLE         => $title,
        DESCRIPTION   => '',
        HREF          => $href,
        ADD_DATE      => '',
        LAST_VISIT    => '',
        LAST_MODIFIED => '',
        ALIAS_ID      => '',
    });
}  # }}}

1;
__END__

=head1 NAME

URI::Collection - Input and output link collections in different formats

=head1 SYNOPSIS

  use URI::Collection;

  $c = URI::Collection->new(links => $bookmark_file);
  $c = URI::Collection->new(links => $favorite_directory);
  $c = URI::Collection->new(
      links => [ $bookmark_file, $favorite_directory ],
  );

  $links = $c->fetch_items(
      category => $regexp_1,
      title    => $regexp_2,
      url      => $regexp_3,
  );

  if ($items = $c->is_item($regexp)) {
      print Data::Dumper($items);
  }

  $bookmarks = $c->as_bookmark_file;
  $c->as_bookmark_file(save_as => $filename);

  $favorites = $c->as_favorite_directory;
  $c->as_favorite_directory(save_as => $directory);

=head1 DESCRIPTION

An object of class C<URI::Collection> represents the links and 
categories in parsed Netscape style bookmark files and Windows 
"Favorites" directories, with output in either style.

=head1 METHODS

=head2 new

  $c = URI::Collection->new(links => $bookmark_file);
  $c = URI::Collection->new(links => $favorite_directory);
  $c = URI::Collection->new(
      links => [ $bookmark_file, $favorite_directory ],
  );

Return a new C<URI::Collection> object.

This method mashes link store formats together, simultaneously.

=head2 as_bookmark_file

  $bookmarks = $c->as_bookmark_file;
  $c->as_bookmark_file(save_as => $filename);

Output a Netscape style bookmark file as a string with the file 
contents.

Optionally, save the bookmarks as a file to disk, if given a 
C<save_as> argument.

=head2 as_favorite_directory

  $favorites = $c->as_favorite_directory;
  $c->as_favorite_directory(save_as => $directory);

Output a M$ Windows "Favorites" tree as a hash reference, where the
keys are the subdirectories (categories) and the values are array 
references with the contents of M$ Windows C<*.url> files as string
elements.

This tree is one dimensional.  That is, the nested categories are 
represented by "slash separated paths".  An example will elucidate:

  { 'foo'     => [ $link1, $link2, ],
    'foo/bar' => [ $link3, $link4, ],
    'baz'     => [ $link5, ],
    'baz/x/y' => [ $link6, $link7, ], }

This path representation is magically converted to the proper Win32 
scheme when expressed with the C<as_favorite_directory> method.

If a C<save_as> argument is provided, this method will write the 
"Favorites" directory to disk with the value as the top level 
directory name (and then return that same name).

=head2 fetch_items

  $items = $c->fetch_items(
      category => $regexp_1,
      title    => $regexp_2,
      url      => $regexp_3,
  );

Return a list of links that have titles, urls or categories similar 
to the given regular expressions.

Note that if a category argument is supplied, only links under 
matching categories will be found.  If no category argument is 
provided, any link with a matching title or url will be returned.  If
no arguments are provided, all links are returned.

=head2 is_item

  $items = $c->is_item($regexp);

Return the items whose titles or urls match the given regular 
expression.

Note that this method is just C<fetch_items> with the no category 
argument and identical title and url pattern.

=head1 DEPENDENCIES

L<Carp>

L<Cwd>

L<File::Spec>

L<File::Find>

L<File::Path>

L<IO::String>

L<Config::IniFiles>

L<Netscape::Bookmarks::Link>

=head1 TO DO

Ignore redundant links.

Optionally return the M$ Favorites directory structure (as a
variable) instead of writing it to disk.

Allow input/output of file and directory handles.

Allow slicing of the category-links structure.

Allow link munging to happen under a given category or categories 
only.

Check if links are active.

Update link titles and URLs if changed or moved.

Mirror links?

Handle other bookmark formats (including some type of generic XML), 
and "raw" (CSV) lists of links, to justify such a generic package 
name.  This includes different platform flavors of every browser.

Move the Favorites input/output functionality to a seperate module
like C<URI::Collection::Favorites::IE::Windows> and 
C<URI::Collection::Favorites::IE::Mac>, or some such.  Do the same 
with the above mentioned "platform flavors", such as Opera and Mosaic 
"Hotlists", and OmniWeb bookmarks, etc.

=head1 SEE ALSO

C<http://www.cyanwerks.com/file-format-url.html>

There are an enormous number of web-based bookmark managers out there 
(see http://useful.webwizards.net/wbbm.htm), which I don't care about 
at all.  What I do care about are multi-format link converters.  Here 
are a few that I found:

Online manager: 
C<http://www.linkagogo.com/>

CDML Universal Bookmark Manager (for M$ Windows only):
C<http://www.cdml.com/products/ubm.asp>

OperaHotlist2HTML:
C<http://nelson.oit.unc.edu/~alanh/operahotlist2html/>

bk2site:
C<http://bk2site.sourceforge.net/>

Windows favorites convertor:
C<http://www.moodfarm.demon.co.uk/download/fav2html.pl>

bookmarker:
C<http://www.renaghan.com/pcr/bookmarker.html>

Columbine Bookmark Merge:
C<http://home.earthlink.net/~garycramblitt/>

XBEL Bookmarks Editor:
C<http://pyxml.sourceforge.net/topics/xbel/>

And here are similar perl modules:

L<URI::Bookmarks>

L<BabelObjects::Component::Directory::Bookmark>

L<WWW::Bookmark::Crawler>

L<Apache::XBEL>

=head1 THANK YOU

A special thank you goes to my friends on rhizo #perl for answering 
my random questions.  : )

=head1 AUTHOR

Gene Boggs E<lt>gene@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2003 by Gene Boggs

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself. 

=cut
