package Automate::Animate::FFmpeg;

use 5.006;
use strict;
use warnings;

our $VERSION = '0.02';

use utf8; # filenames can be in utf8

use File::Temp;
use File::Find::Rule;
use IPC::Run;
use Text::ParseWords;
use Data::Roundtrip qw/perl2dump no-unicode-escape-permanently/;

sub	new {
	my $class = $_[0];
	my $params = $_[1];

	my $parent = ( caller(1) )[3] || "N/A";
	my $whoami = ( caller(0) )[3];

	my $self = {
		'input-images' => [],
		'output-filename' => undef,
		'verbosity' => 0,
		# params to ffmpeg must be given as an arrayref
		# these are our standard params, they are read only and must
		# be supplied here as an ARRAYref
		'ffmpeg-standard-params' => ['-c', 'copy', '-c:v', 'copy', '-c:a', 'copy', '-q:a', '0', '-q:v', '1'],
		# extra params to ffmpeg can be specified by the caller
		# we store this as an ARRAYref, each option and value is an item of its own
		'ffmpeg-extra-params' => [],
		# the duration of each frame in seconds (fractional supported)
		# or <=0 for using FFmpeg's defaults
		'frame-duration' => 0,
		# this will be modified during perl Makefile.PL
		'ffmpeg-executable' => '/usr/local/bin/ffmpeg' # specify fullpath if not in path
		# end this will be modified during perl Makefile.PL
	};
	bless $self, $class;

	# sort out the easy params
	for my $ak (qw/
		verbosity
		output-filename
		frame-duration
		ffmpeg-executable
	/){
		if( exists($params->{$ak}) && defined($params->{$ak}) ){
			$self->{$ak} = $params->{$ak}
		}
	}

	# specify input images as scalar, arrayref or hashref (values)
	if( exists($params->{'input-images'}) && defined($params->{'input-images'})
		&& scalar($params->{'input-images'})
	){
		$self->input_images($params->{'input-images'});
	}

	# input images can be specified via a pattern and a search dir
	# like : 'input-pattern' => ['*.png', '/x/y/searchdir']
	if( exists($params->{'input-pattern'}) && defined($params->{'input-pattern'}) ){
		if( ref($params->{'input-pattern'})ne'ARRAY' ){ print STDERR perl2dump($params->{'input-pattern'})."${whoami} (via $parent), line ".__LINE__." : error, the argument to 'input-pattern' must be an ARRAYref of 1 or 2 items: the pattern and optionally the search dir. See above for what was provided.\n"; return undef }
		if( ! $self->input_pattern($params->{'input-pattern'}) ){ print STDERR perl2dump($params->{'input-pattern'})."${whoami} (via $parent), line ".__LINE__." : error, failed to find input files based on the above pattern and search dir.\n"; return undef }
	}
	if( exists($params->{'input-patterns'}) && defined($params->{'input-patterns'}) ){
		if( ref($params->{'input-patterns'})ne'ARRAY' ){ print STDERR perl2dump($params->{'input-patterns'})."${whoami} (via $parent), line ".__LINE__." : error, the argument to 'input-patterns' must be an ARRAYref of one or more ARRAYrefs each of 1 or 2 items: the pattern and optionally the search dir. See above for what it was provided.\n"; return undef }
		if( ! $self->input_patterns($params->{'input-patterns'}) ){ print STDERR perl2dump($params->{'input-patterns'})."${whoami} (via $parent), line ".__LINE__." : error, failed to find input files based on the above pattern and search dir.\n"; return undef }
	}
	# specify output filename
	if( exists($params->{'output-filename'}) && defined($params->{'output-filename'}) ){
		$self->output_filename($params->{'output-filename'})
	}
	# any extra ffmpeg params?
	# these are cmdline options to FFmpeg and must be
	# passed as an ARRAY, each flag, option and parameter is
	# a single array item, for example
	# ['-i', 'inputfile', '-o', 'out', '-p', '1', '2']
	# note that above -p is used as in -p 1 2
	if( exists($params->{'ffmpeg-extra-params'}) && defined($params->{'ffmpeg-extra-params'}) ){
		if( ! defined($self->ffmpeg_extra_params($params->{'ffmpeg-extra-params'})) ){ print STDERR perl2dump($params->{'ffmpeg-extra-params'})."${whoami} (via $parent), line ".__LINE__." : error, failed to parse/verify the above params to ffmpeg via '--ffmpeg-extra-params'.\n"; return undef }
	}
	return $self
}
# it spawns ffmpeg as external command via IPC::Run::run(@cmd)
# requires that at least 1 input image was specified before.
# returns 0 on failure, 1 on success
sub	make_animation {
	my $self = $_[0];
	my $parent = ( caller(1) )[3] || "N/A";
	my $whoami = ( caller(0) )[3];
	my $verbos = $self->verbosity();

	my $cmdret = $self->_build_ffmpeg_cmdline();
	if( ! defined $cmdret ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : error, failed to build ffmpeg command line, call to ".'_build_ffmpeg_cmdline()'." has failed.\n"; return 0 }
	my $cmdline = $cmdret->{'cmdline'};
	my $tmpfile = $cmdret->{'tmpfile'};
	if( $verbos > 0 ){ print "${whoami} (via $parent), line ".__LINE__." : executing system command:\n".join(' ',@$cmdline)."\n" }
	my ($in, $out, $err);
	my $ret = IPC::Run::run($cmdline, \$in, \$out, \$err);
	if( ! $ret ){ 
		print STDERR "$out\n$err\n${whoami} (via $parent), line ".__LINE__." : error, executing this command has failed (the list of input files is in '$tmpfile'):\n  ".join(' ', @$cmdline)."\n";
		return 0
	}
	if( $verbos > 0 ){
		if( $verbos > 1 ){ print $out."\n" }
		print "${whoami} (via $parent), line ".__LINE__." : done, success. Output is in '".$self->output_filename()."'.\n"
	}
	unlink($tmpfile);
	return 1;
}
sub	_build_ffmpeg_cmdline {
	my $self = $_[0];
	my $parent = ( caller(1) )[3] || "N/A";
	my $whoami = ( caller(0) )[3];
	my $verbos = $self->verbosity();

	if( $self->num_input_images() == 0 ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : error, no input images in store.\n"; return undef }
	if( ! defined $self->output_filename() ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : error, no output filename specified.\n"; return undef }

	# get a tmp file
	my ($fh, $tmpfile) = File::Temp::tempfile();
	my $duration_str = $self->frame_duration() > 0 ? "duration ".$self->frame_duration()."\n" : "";
	for (@{$self->{'input-images'}}){
		print $fh "file '".$_
			 ."'\n${duration_str}"
		;
	}
	close $fh;

	my @cmdline = (
		$self->ffmpeg_executable(),
		# the extra params to FFmpeg is just a command-line args style
		@{ $self->ffmpeg_extra_params() },
		# and the concats etc.
		'-f', 'concat',
		'-y',
		# this is about accepting relative filepaths to images
		'-safe', '0',
		# all images filepathas are in this file (one in each line)
		'-i', $tmpfile,
		@{ $self->ffmpeg_standard_params() },
		$self->output_filename()
	);
	return {
		'cmdline' => \@cmdline,
		# this can be unlinked by caller if needed
		'tmpfile' => $tmpfile
	}
}
# set the executable only via the constructor
sub	ffmpeg_executable { return $_[0]->{'ffmpeg-executable'} }
sub	verbosity {
	my $self = $_[0];
	my $m = $_[1];
	return $self->{'verbosity'} unless defined $m;
	$self->{'verbosity'} = $m;
	return $m
}
sub	frame_duration {
	my $self = $_[0];
	my $m = $_[1];
	return $self->{'frame-duration'} unless defined $m;
	$self->{'frame-duration'} = $m;
	return $m
}

sub	output_filename {
	my $self = $_[0];
	my $outfile = $_[1];
	return $self->{'output-filename'} unless defined $outfile;
	$self->{'output-filename'} = $outfile;
	return $outfile
}
sub	_cmdline2argsarray {
	my $m = $_[0];
	if( ref($m) eq 'ARRAY' ){
		return [ @$m ]
	} elsif( ref($m) eq '' ){
		return Text::ParseWords::shellwords($m)
	} elsif( ref($m) eq 'HASH' ){
		return [ %$m ];
	}
	print STDERR "_cmdline2argsarray() : an ARRAYref/HASHref/String-scalar containing command-line arguments was expected, not ".ref($m)."\n";
	return undef
}
sub	ffmpeg_extra_params {
	my $self = $_[0];
	my $m = $_[1];
	return $self->{'ffmpeg-extra-params'} unless defined $m;
	my $ret = _cmdline2argsarray($self->{'ffmpeg-extra-params'});
	if( ! defined $ret ){ print STDERR perl2dump($ret)."ffmpeg_extra_params() : error, failed to pass above arguments.\n"; return undef }
	$self->{'ffmpeg-extra-params'} = $ret;
	return $ret
}
sub	ffmpeg_standard_params { return $_[0]->{'ffmpeg-standard-params'} }
sub	num_input_images { return scalar @{$_[0]->{'input-images'}} }
# specify a text file which holds image filenames, one per line to be added
# hash-comments are understood, empty/only-space lines are removed
# returns 1 on success, 0 on failure
sub	input_file_with_images {
	my ($self, $infile) = @_;
	my $parent = ( caller(1) )[3] || "N/A";
	my $whoami = ( caller(0) )[3];
	my $verbos = $self->verbosity();
	if( ! defined $infile ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : error, an input filename of input image filenames is expected.\n"; return 0 }
	my $fh;
	if( ! open($fh, '<:encoding(UTF-8)', $infile) ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : error, could not open input file '$infile' for reading, $!\n"; return 0 }
	while( <$fh> ){
		chomp;
		s/#.*$//;
		s/^\s*$//;
		$self->input_images($_) unless /^\s*$/;
	} close $fh;
	return 1
}
sub	clear_input_images { $#{ $_[0]->{'input-images'} } = -1 }
# Add using a single pattern/searchdir
# add image files via a pattern and an input dir, e.g. '*.png', '/x/y/z/'
# make sure that the order you expect is what you get during the pattern materialisation
# the search dir is optional, default is Cwd::cwd
sub	input_pattern {
	my ($self, $_pattern, $indir) = @_;
	my $parent = ( caller(1) )[3] || "N/A";
	my $whoami = ( caller(0) )[3];
	my $verbos = $self->verbosity();
	if( ! defined $indir ){
		$indir = Cwd::cwd;
		if( $verbos > 0 ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : warning, no search dir was specified and using current dir '$indir'.\n" }
	}
	my $pattern;
	if( $_pattern =~ m!^regex\(/(.*?)/(.*?)\)$! ){
		# see https://www.perlmonks.org/?node_id=1210675
		my $pa = $1; my $mo = $2;
		if( $mo!~/^[msixpodualn]+$/ ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : error, illegal modifiers ($mo) to the specified regex detected.\n"; return 0 }
		$pattern = qr/(?${mo})${pa}/;
	} else { $pattern = $_pattern }
	if( ! defined $self->input_images([
		File::Find::Rule
			->file()
			->name($pattern)
			->in($indir)
	]) ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : error, call to input_images() has failed.\n"; return 0 }
	return 1 # success
}
# This adds many patterns:
# the input is an ARRAY of 1-or-2-item arrays
# each subarray must consist of a pattern and optionally a search dir (else current dir will be used)
sub	input_patterns {
	my ($self, $specs) = @_;
	my $parent = ( caller(1) )[3] || "N/A";
	my $whoami = ( caller(0) )[3];
	my $verbos = $self->verbosity();
	for my $as (@$specs){
		if( scalar(@$as)==0 ){ print STDERR perl2dump($as)."${whoami} (via $parent), line ".__LINE__." : error, the spec must contain at least a pattern and optionally a search-dir as an array, see above.\n"; return 0; }
		if( ! $self->input_pattern(@$as) ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : error, call to input_pattern() has failed for this spec: @$as\n"; return 0 }
	}
	return 1 # success
}
# if no parameter is specified then it returns the current list of input images as an arrayref
# Otherwise:
# specify one or more input filenames (of images) via a single scalar, an arrayref or
# a hashref whose values are image filenames, to convert them into video
# in this case returns undef on failure or the current, updated list of input images on success
sub	input_images {
	my ($self, $m) = @_;
	if( ! defined $m ){ return $self->{'input-images'} }
	my $parent = ( caller(1) )[3] || "N/A";
	my $whoami = ( caller(0) )[3];
	my $verbos = $self->verbosity();
	if( $verbos > 0 ){
		if( $verbos > 1 ){ print STDOUT perl2dump($m)."${whoami} (via $parent), line ".__LINE__." : called with above parameter ...\n" }
		else { print STDOUT "${whoami} (via $parent), line ".__LINE__." : called ...\n" }
	}
	my $rf = ref $m;
	if( $rf eq 'ARRAY' ){
		for my $af (@$m){
			if( ! -e $af ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : warning/1, input image '$af' does not exist on disk and will be ignored.\n"; next }
			push @{$self->{'input-images'}}, Cwd::abs_path($af)
		}
	} elsif( $rf eq 'HASH' ){
		for my $af (values %$m){
			if( ! -e $af ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : warning/2, input image '$af' does not exist on disk and will be ignored.\n"; next }
			push @{$self->{'input-images'}}, Cwd::abs_path($af)
		}		
	} elsif( $rf eq '' ){
		if( ! -e $m ){ print STDERR "${whoami} (via $parent), line ".__LINE__." : warning/3, input image '$m' does not exist on disk and will be ignored.\n"; }
		else { push @{$self->{'input-images'}}, Cwd::abs_path($m) }
	} else { print STDERR "${whoami} (via $parent), line ".__LINE__." : error, input can be an arrayref of input image filenames, a hashref whose values are filenames or a single filename in the form of a scalar."; return undef }
	if( $verbos > 1 ){ print STDOUT perl2dump($self->{'input-images'})."${whoami} (via $parent), line ".__LINE__." : called and added some images, above is a list of all known images.\n" }
	return $self->{'input-images'}
}

# that's the end, pod now starts
=pod

=head1 NAME

Automate::Animate::FFmpeg - Create animation from a sequence of images using ffmpeg

=head1 VERSION

Version 0.02


=head1 SYNOPSIS

This module creates an animation from a sequence of input
images using L<FFmpeg|https://ffmpeg.org>.
An excellent, open source program.

    use Automate::Animate::FFmpeg;
    my $aaFFobj = Automate::Animate::FFmpeg->new({
      # specify input images in any of these 3 ways:
      'input-images' => [
        'im1.png',
        'im2.png',
        ...
      ],
      'input-pattern' => ['*.png', './'],
      'input-images-from-file' => 'file-containing-a-list-of-pathnames-to-images.txt',
      # optionally specify the duration of each frame
      'frame-duration' => 5.3, # seconds
      'output-filename' => 'out.mp4',
    });

    # options can be set after construction as well:

    # optionally add some extra params to FFmpeg as an arrayref
    $aaFF->ffmpeg_extra_params(['-x', 'abc', '-y', 1, 2, 3]);
    # you can also add images here, order is important
    $aaFF->input_images(['img1.png', 'img2.png']) or die;
    # or add images via a search pattern and optional search dir
    $aaFF->input_pattern(['*.png', './']);
    # or add images via multiple search patterns
    $aaFF->input_patterns([
	['*.png', './'],
	['*.jpg', '/images'],
	['*.tiff'], # this defaults to current dir
    ]) or die;

    # and make the animation:
    die "make_animation() has failed"
      unless $aaFF->make_animation()
    ;

=head1 METHODS

=head2 C<new>

  my $ret = Automate::Animate::FFmpeg->new({ ... });

All arguments are supplied via a hashref with the following keys:

=over 4

=item * C<input-images> : an array of pathnames to input images. Image types can be what ffmpeg understands: png, jpeg, tiff, and lots more.

=item * C<input-pattern> : an arrayref of 1 or 2 items. The first item is the pattern
which complies to what L<File::Find::Rule> understands (See [https://metacpan.org/pod/File::Find::Rule#Matching-Rules]).
For example C<*.png>, regular expressions can be passed by enclosing them in C<regex(/.../modifiers)>
and should include the C<//>. Modifiers can be after the last C</>. For example C<regex(/\.(mp3|ogg)$/i)>.

The optional second parameter is the search path. If not specified, the current working dir will be used.

Note that there is no implicit or explicit C<eval()> in compiling the user-specified
regex (i.e. when pattern is in the form C<regex(/.../modifiers)>).
Additionally there is a check in place for the user-specified modifiers to the regex:
C<die "never trust user input" unless $modifiers=~/^[msixpodualn]+$/;>.
Thank you L<Discipulus|https://www.perlmonks.org/?node_id=174111>.

=item * C<input-patterns> : same as above but it expects an array of C<input-pattern>.

=item * C<input-images-from-file> : specify the file which contains pathnames to image files, each on its own line.

=item * C<ffmpeg-extra-params> : pass extra parameters to the C<ffmpeg> executable as an arrayref of arguments, each argument must be a separate item as in : C<['-i', 'file']>.

=item * C<frame-duration> : set the duration of each frame (i.e. each input image) in the animation in (fractional) seconds.

=item * C<ffmpeg-executable> : set the path to the C<ffmpeg> executable.

=item * C<qw/verbosity> : set the verbosity, 0 being mute.

=back

Return value:

=over 4

=item * C<undef> on failure or the blessed object on success.

=back

This is the constructor. It instantiates the object which does the animations. Its
input parameters can be set also via their own setter methods.
If input images are specified during construction then the list
of filenames is constructed and kept in memory. Just the filenames.

=head2 C<make_animation()>

  $aaFF->make_animation() or die "failed";

It initiates the making of the animation by shelling out to C<ffmpeg>
with all the input images specified via one or more calls to any of:

=over 2

=item * input_images($m)

=item * input_pattern($m)

=item * input_patterns($m)

=item * input_file_with_images($m)

=back

On success, the resultant animation will be
written to the output file
(specified using L<output_filename($m)> before the call.

It returns 0 on failure, 1 on success.

=head2 C<input_images($m)>

  my $ret = $aaFF->input_images($m);

Sets or gets the list (as an ARRAYref) of all input images currently in the list
of images to make up the animation. The optional input parameter, C<$m>,
is an ARRAYref of input images (their fullpath that is) to make up
the animation.

=head2 C<input_pattern($m)>

  $aaFF->input_pattern($m) or die "failed";

Initiates a search via L<File::Find::Rule> for the
input image files to make up the animation using
the pattern C<$m-E<gt>[0]> with starting search dir being C<$m-E<gt>[1]>,
which is optional -- default being C<Cwd::cwd> (current working dir).
So, C<$m> is an array ref of one or two items. The first is the search
pattern and the optional second is the search path, defaulting to the current
working dir.

The pattern (C<$m->[0]>) can be a shell wildcard, e.g. C<*.png>,
or a regex specified as C<regex(/REGEX-HERE/modifiers)>, for example
C<regex(/\.(mp3|ogg)$/i)> Both shell wildcards and regular expressions
must comply with what L<File::Find::Rule> expects, see [https://metacpan.org/pod/File::Find::Rule#Matching-Rules].

The results of the search will be added to the list of input images
in the order of appearance.

Multiple calls to C<input_pattern()> will load
input images in the order they are found.

C<input_pattern()> can be combined with C<input_patterns()>
and C<input_images()>. The input images list will increase
in the order they are called.

It returns 1 on success or 0 on failure.

=head2 C<input_patterns($m)>

  $aaFF->input_patterns($m) or die "failed";

Argument C<$m> is an array of arrays each composed of one or two items.
The first argument, which is mandatory, is the search pattern.
The optional second argument is the directory to start the search.
For each item of C<@$m> it calls L<input_pattern($m)>.

C<input_patterns()> can be combined with C<input_pattern()>
and C<input_images()>. The input images list will increase
in the order they are called.

It returns 1 on success or 0 on failure.

=head2 C<output_filename($m)>

  my $ret = $aaFF->output_filename($m);

Sets or gets the output filename of the animation.

=head2 C<input_file_with_images($m)>

  $aaFF->input_file_with_images($m) or die "failed";

Reads file C<$m> which must contain filenames, one filename
per line, and adds the up to the list of input images to make up the
animation.

It returns 0 on failure, 1 on success.

=head2 C<num_input_images()>

  my $N = $aaFF->num_input_images();

It returns the number of input images currently
in the list to make up the animation.

=head2 C<clear_input_images()>

  $aaFF->clear_input_images();

It clears the list of input images to make up an animation.
Zero, null, it's over for Bojo.

=head2 C<ffmpeg_executable()>

  my $ret = $aaFF->ffmpeg_executable();

It returns the path to C<ffmpeg> executable as it was set during construction.
You can not change the path to the executable mid-stream. Set it via
the constructor or rely on the installation to detect it
and hardcode it to the module file.

=head2 C<verbosity($m)>

  my $ret = $aaFF->verbosity($m);

It sets or gets the verbosity level. Zero being mute.

=head2 C<frame_duration($m)>

  my $ret = $aaFF->frame_duration($m);

It sets or gets the frame duration in (fractional) seconds.
Frame duration is the time that each frame(=image) appears
in the produced animation.


=head1 SCRIPTS 

A script for making animations from input images using C<ffmpeg>
is provided: C<automate-animate-ffmpeg.pl>.


=head1 AUTHOR

Andreas Hadjiprocopis, C<< <bliako at cpan.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-automate-animate-ffmpeg at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/NoAuth/ReportBug.html?Queue=Automate-Animate-FFmpeg>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Automate::Animate::FFmpeg


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<https://rt.cpan.org/NoAuth/Bugs.html?Dist=Automate-Animate>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Automate-Animate>

=item * CPAN Ratings

L<https://cpanratings.perl.org/d/Automate-Animate>

=item * Search CPAN

L<https://metacpan.org/release/Automate-Animate>

=back


=head1 ACKNOWLEDGEMENTS

=over 2

=item * A big thank you to L<FFmpeg|https://ffmpeg.org>, an
excellent, open source software for all things moving.

=item * A big thank you to L<PerlMonks|https://perlmonks.org>
for the useful L<discussion|https://perlmonks.org/?node_id=11156484>
on parsing command line arguments as a string. And an even bigger
thank you to L<PerlMonks|https://perlmonks.org> for just being there.

=item * On compiling a regex when pattern and modifiers are in
variables, L<discussion|https://www.perlmonks.org/?node_id=1210675>
at L<PerlMonks|https://perlmonks.org>.

=item * A big thank you to Ace, the big dog. Bravo Ace!

=back

=head1 LICENSE AND COPYRIGHT

Copyright 2019 Andreas Hadjiprocopis.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of Automate::Animate::FFmpeg
