/* UNIX specific extensions */


#include "os.h"
#include "mem.h"
#include "strings.h"

#include <stdio.h>


/*---------------------------------------------------------------------------*/

/*
 * This file contains C procedures that are to be made available to the
 * Scheme world.  A special naming convention is used for the procedure
 * names so that any Scheme identifier can be written out (including
 * identifiers with '#' characters).  The procedure 'c_id_to_symbol' in
 * 'strings.c' implements the naming convention.
 *
 * The arguments and results of these procedures are of type 'SCM_obj'.
 * Macros and procedures exist to convert from Scheme data representation
 * to C data representation (see 'gambit.h').
 *
 */


SCM_obj X23X23unixDsystem( str )
SCM_obj str;
{ SCM_obj result;
  char *mark = local_mark();              /* put mark on local C heap    */
  char *command = string_to_c_str( str ); /* convert Scheme string to C  */
  if (command == NULL)
    result = SCM_false;
  else
    result = SCM_int_to_obj( system( command ) );
  local_release( mark );                  /* get rid of converted string */
  return result;
}


/* defined in os.c */

#define MAX_NB_FDS 32
extern OS_FILE file[MAX_NB_OPEN_FILES];


static long pipe_open( command, type )
char *command, *type;
{ FILE *p;
  long f = -1;
  p = popen( command, type );
  if (p != NULL)
  { f = fileno(p);
    if ((f < 0) || (f >= MAX_NB_FDS))
    { close( f ); f = -1; }
  }
  return f;
}


SCM_obj X23X23osDpipeDopenDinput( cmd )
SCM_obj cmd;
{ char *command, *mark;
  SCM_obj result;
  long i;
  for (i=0; i<(long)MAX_NB_OPEN_FILES; i++) if (file[i] == -1) break;
  if (i == (long)MAX_NB_OPEN_FILES) return (long)SCM_false;
  mark = local_mark();
  command = string_to_c_str(cmd);
  if (command == NULL)
    result = (long)SCM_false;
  else
  { OS_FILE f = (OS_FILE)pipe_open( command, "r" );
    if (f == -1)
      result = (long)SCM_false;
    else
    { file[i] = f;
      result = SCM_int_to_obj(i);
    }
  }
  local_release( mark );
  return result;
}


SCM_obj X23X23osDpipeDopenDoutput( cmd )
SCM_obj cmd;
{ char *command, *mark;
  SCM_obj result;
  long i;
  for (i=0; i<(long)MAX_NB_OPEN_FILES; i++) if (file[i] == -1) break;
  if (i == (long)MAX_NB_OPEN_FILES) return (long)SCM_false;
  mark = local_mark();
  command = string_to_c_str(cmd);
  if (command == NULL)
    result = (long)SCM_false;
  else
  { OS_FILE f = (OS_FILE)pipe_open( command, "w" );
    if (f == -1)
      result = (long)SCM_false;
    else
    { file[i] = f;
      result = SCM_int_to_obj(i);
    }
  }
  local_release( mark );
  return result;
}


SCM_obj X23X23osDpipeDclose( ind )
SCM_obj ind;
{ long i = SCM_obj_to_int(ind);
  if ((i>=3) && (i<(long)MAX_NB_OPEN_FILES))
  { OS_FILE f = file[i];
    file[i] = -1;
    if (f != -1)
    { pclose( &_iob[f] ); return (long)SCM_true; }
    else
      return (long)SCM_false;
  }
  else
    return (long)SCM_false;
}


/* Sample use of pipes:


(define (##open-input-pipe command)
  (let ((descr (##os-pipe-open-input command)))
    (if descr
      (##make-port descr command 0
        ##os-file-read
        #f
        ##os-file-read-ready
        ##os-pipe-close
        (##make-string 64 #\space)
        #f)
      #f)))

(define (##open-output-pipe command)
  (let ((descr (##os-pipe-open-output command)))
    (if descr
      (##make-port descr command 2
        #f
        ##os-file-write
        #f
        ##os-pipe-close
        #f
        (##make-string 1 #\space))
      #f)))

(define (echo x)
  (let ((p (##open-input-pipe (string-append "echo \\\"`" x "`\\\""))))
    (let ((x (read p)))
      (close-input-port p)
      x)))

(echo "date")
(echo "pwd")


*/


/*---------------------------------------------------------------------------*/


void ext_init()
{ DEFINE_C_PROC(X23X23unixDsystem);
  DEFINE_C_PROC(X23X23osDpipeDopenDinput);
  DEFINE_C_PROC(X23X23osDpipeDopenDoutput);
  DEFINE_C_PROC(X23X23osDpipeDclose);
}


/*---------------------------------------------------------------------------*/
