use strict;
use warnings;

use File::Spec;
use Test::More;

use lib File::Spec->catdir( File::Spec->curdir, 't' );

BEGIN { require 'check_datetime_version.pl' }

plan tests => 180;

eval { DateTime::TimeZone::OffsetOnly->new( offset => 'bad' ) };
is( $@, "Invalid offset: bad\n",
    'test that OffsetOnly does not allow invalid offsets' );

my $off = DateTime::TimeZone::OffsetOnly->new( offset => '-0100' );
is( $off->name, '-0100', 'name is -0100' );

my @good_offsets = (
    [ '0',          'UTC' ],
    [ '0000',       'UTC' ],
    [ '000000',     'UTC' ],
    [ '1000',       '+1000' ],
    [ '100001',     '+100001' ],
    [ '10:00:02',   '+100002' ],
    [ '+0000',      'UTC' ],
    [ '+000000',    'UTC' ],
    [ '+000001',    '+000001' ],
    [ '+00:00:02',  '+000002' ],
    [ '-0000',      'UTC' ],
    [ '-000000',    'UTC' ],
    [ '-000001',    '-000001' ],
    [ '-00:00:02',  '-000002' ],
    [ '+9959',      '+9959' ],
    [ '+995959',    '+995959' ],
    [ '+99:59:58',  '+995958' ],
    [ '-9959',      '-9959' ],
    [ '-995959',    '-995959' ],
    [ '-99:59:58',  '-995958' ],
    [ '1:11:11',    '+011111' ],
    [ '+1:11:11',   '+011111' ],
    [ '-1:11:11',   '-011111' ],
    [ '1:11',       '+0111' ],
    [ '+1:11',      '+0111' ],
    [ '-1:11',      '-0111' ],
);

my @bad_offsets = (
    '+0', '-0',
    '1', '+1', '-1',
    '1:', '+1:', '-1:',
    ':1', '+:1', '-:1',
    '11', '+11', '-11',
    '11:', '+11:', '-11:',
    '1:1', '+1:1', '-1:1',
    ':11', '+:11', '-:11',
    '111', '+111', '-111',
    '111:', '+111:', '-111:',
    '11:1', '+11:1', '-11:1',
    ':111', '+:111', '-:111',
    ':11:1', '+:11:1', '-:11:1',
    '1:11:', '+1:11:', '-1:11:',
    '1111:', '+1111:', '-1111:',
    '111:1', '+111:1', '-111:1',
    '1:111', '+1:111', '-1:111',
    ':1111', '+:1111', '-:1111',
    ':11:11', '+:11:11', '-:11:11',
    '1:11:1', '+1:11:1', '-1:11:1',
    '11:11:', '+11:11:', '-11:11:',
    '11111', '+11111', '-11111',
    '11111:', '+11111:', '-11111:',
    '1111:1', '+1111:1', '-1111:1',
    '111:11', '+111:11', '-111:11',
    '11:111', '+11:111', '-11:111',
    '1:1111', '+1:1111', '-1:1111',
    ':11111', '+:11111', '-:11111',
    ':11:111', '+:11:111', '-:11:111',
    '11:11:1', '+11:11:1', '-11:11:1',
    '111:11:', '+111:11:', '-111:11:',
    '111111:', '+111111:', '-111111:',
    '11111:1', '+11111:1', '-11111:1',
    '1111:11', '+1111:11', '-1111:11',
    '111:111', '+111:111', '-111:111',
    '11:1111', '+11:1111', '-11:1111',
    '1:11111', '+1:11111', '-1:11111',
    ':111111', '+:111111', '-:111111',
    ':11:1111', '+:11:1111', '-:11:1111',
    '1:11:111', '+1:11:111', '-1:11:111',
    '111:11:1', '+111:11:1', '-111:11:1',
    '1111:11:', '+1111:11:', '-1111:11:',
    '1111111', '+1111111', '-1111111',
    '00:60', '+00:60', '-00:60',
    '00:99', '+00:99', '-00:99',
    '00:60:00', '+00:60:00', '-00:60:00',
    '00:99:00', '+00:99:00', '-00:99:00',
    '00:00:60', '+00:00:60', '-00:00:60',
    '00:00:99', '+00:00:99', '-00:00:99',
    '00:60:60', '+00:60:60', '-00:60:60',
    '00:99:99', '+00:99:99', '-00:99:99',
);

foreach ( @good_offsets ) {
    my $off = DateTime::TimeZone::OffsetOnly->new( offset => $_->[0] );
    is( $off->name, $_->[1], "name matches $_->[1]" );
}

foreach ( @bad_offsets ) {
    eval{ DateTime::TimeZone::OffsetOnly->new( offset => $_ ) };
    like( $@, qr/Invalid offset/, "$_ is invalid" );
}
