/*
 *  huffman - Encode/Decode files using Huffman encoding.
 *  http://huffman.sourceforge.net
 *  Copyright (C) 2003  Douglas Ryan Richardson
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "huffman.h"

#ifdef WIN32
#include <winsock2.h>
#include <malloc.h>
#define alloca _alloca
#else
#include <netinet/in.h>
#endif

#ifdef HEADER

#ifdef __GNUC__
#define UNUSED __attribute__ ((unused))
#define NOIGNORE __attribute__ ((warn_unused_result))
#else
#define UNUSED
#define NOIGNORE
#endif /* def __GNUC__ */

typedef enum 
{
    huffman_status_ok,
    huffman_status_memory_failure,
    huffman_status_null_pointer,
    huffman_status_bounds,
}
huffman_status_t;

#endif /* def HEADER */

/* Count mallocs and frees. */

//#define MEMCOUNT

#ifdef MEMCOUNT
static int n_mallocs;
#define MEMCLEAR n_mallocs = 0
#define MEMPLUS n_mallocs++
#define MEMFREE n_mallocs--
#define MEMCHECK(expect) {				\
	if (n_mallocs != expect) {			\
	    fprintf (stderr, "%s:%d: n_mallocs = %d, "	\
		     "expected %d\n",			\
		     __FILE__, __LINE__,		\
		     n_mallocs, expect);		\
	}						\
    }
#else
#define MEMCLEAR
#define MEMPLUS
#define MEMFREE
#define MEMCHECK(expect)
#endif /* MEMCOUNT */

#define DEBUG

#ifdef DEBUG
#define MESSAGE(x) {				\
	printf ("%s:%d: ", __FILE__, __LINE__);	\
	printf x;				\
	puts ("\n");				\
    }
#else
#define MESSAGE(x)
#endif

/* Used to build the Huffman code. */

typedef struct huffman_node_tag
{
    /* One for a leaf node. */
    unsigned char isLeaf;
    /* Zero for a leaf node. */
    unsigned long count;
    struct huffman_node_tag *parent;

    union
    {
	struct
	{
	    struct huffman_node_tag *zero, *one;
	};
	unsigned char symbol;
    };
}
huffman_node;

typedef struct huffman_code_tag
{
    /* The length of this code in bits. */
    unsigned long numbits;

    /* The bits that make up this code. The first
       bit is at position 0 in bits[0]. The second
       bit is at position 1 in bits[0]. The eighth
       bit is at position 7 in bits[0]. The ninth
       bit is at position 0 in bits[1]. */
    unsigned char *bits;
}
huffman_code;

/* Given "numbits" bits, return the minimum number of bytes necessary
   to hold that many bits. */

static unsigned long
numbytes_from_numbits (unsigned long numbits)
{
    return numbits / 8 + (numbits % 8 ? 1 : 0);
}

/*
 * get_bit returns the ith bit in the bits array
 * in the 0th position of the return value.
 */
static unsigned char
get_bit (unsigned char* bits, unsigned long i)
{
    return (bits[i / 8] >> i % 8) & 1;
}

/* Reverse "bits" in place. */

static void
reverse_bits (unsigned char* bits, unsigned long numbits)
{
    unsigned long numbytes = numbytes_from_numbits (numbits);
    unsigned char *tmp =
	(unsigned char*)alloca (numbytes);
    unsigned long curbit;
    long curbyte = 0;
	
    memset (tmp, 0, numbytes);

    for (curbit = 0; curbit < numbits; curbit++) {
	unsigned int bitpos = curbit % 8;

	if (curbit > 0 && curbit % 8 == 0) {
	    curbyte++;
	}

		
	tmp[curbyte] |= (get_bit (bits, numbits - curbit - 1) << bitpos);
    }

    memcpy (bits, tmp, numbytes);
}

/*
 * new_code builds a huffman_code from a leaf in
 * a Huffman tree.
 */
static huffman_code*
new_code (const huffman_node* leaf)
{
    /* Build the huffman code by walking up to
     * the root node and then reversing the bits,
     * since the Huffman code is calculated by
     * walking down the tree. */
    unsigned long numbits = 0;
    unsigned char* bits = NULL;
    huffman_code *p;

    while (leaf && leaf->parent) {
	huffman_node *parent = leaf->parent;
	unsigned char cur_bit = (unsigned char)(numbits % 8);
	unsigned long cur_byte = numbits / 8;

	/* If we need another byte to hold the code,
	   then allocate it. */
	if (cur_bit == 0) {
	    size_t newSize = cur_byte + 1;
	    if (! bits) {
		MEMPLUS;
	    }
	    bits = realloc (bits, newSize);
	    if (! bits) {
		return 0;
	    }
	    bits[newSize - 1] = 0; /* Initialize the new byte. */
	}

	/* If a one must be added then or it in. If a zero
	 * must be added then do nothing, since the byte
	 * was initialized to zero. */
	if (leaf == parent->one) {
	    bits[cur_byte] |= 1 << cur_bit;
	}


	numbits++;
	leaf = parent;
    }

    if (bits) {
	reverse_bits (bits, numbits);
    }

    p = malloc (sizeof (huffman_code));
    if (! p) {
	return 0;
    }
    MEMPLUS;
    p->numbits = numbits;
    p->bits = bits;
    return p;
}

/* The maximum number of symbols we allow. */

#define MAX_SYMBOLS 256
//typedef huffman_node* SymbolFrequencies[MAX_SYMBOLS];
//typedef huffman_code* SymbolEncoder[MAX_SYMBOLS];
typedef huffman_node** SymbolFrequencies;
typedef huffman_code** SymbolEncoder;

/* Make a new "huffman_node" associated with a symbol. */

static huffman_node*
new_leaf_node (unsigned char symbol)
{
    huffman_node *p;
    p = malloc (sizeof (huffman_node));
    if (! p) {
	return 0;
    }
    MEMPLUS;
    p->isLeaf = 1;
    p->symbol = symbol;
    p->count = 0;
    p->parent = 0;
    return p;
}

/* Make a new "huffman_node" associated with two child nodes, "zero"
   and "one", without a symbol, but with count "count". */

static huffman_node*
new_nonleaf_node (unsigned long count, huffman_node *zero, huffman_node *one)
{
    huffman_node *p;
    p = malloc (sizeof (huffman_node));
    if (! p) {
	return 0;
    }
    MEMPLUS;
    p->isLeaf = 0;
    p->count = count;
    p->zero = zero;
    p->one = one;
    p->parent = 0;
	
    return p;
}

static void
free_huffman_tree (huffman_node *subtree)
{
    if (subtree == NULL) {
	return;
    }

    if (! subtree->isLeaf) {
	free_huffman_tree (subtree->zero);
	free_huffman_tree (subtree->one);
    }
	
    free (subtree);
    MEMFREE;
}

static void
free_code (huffman_code* p)
{
    free (p->bits);
    MEMFREE;
    free (p);
    MEMFREE;
}

static void
free_encoder (SymbolEncoder se)
{
    unsigned long i;
    for (i = 0; i < MAX_SYMBOLS; i++) {
	huffman_code *p = se[i];
	if (p) {
	    free_code (p);
	}
    }

    free (se);
    MEMFREE;
}

static huffman_status_t NOIGNORE
init_frequencies (SymbolFrequencies *pSF)
{
    *pSF = calloc (MAX_SYMBOLS,  sizeof (huffman_node *));
    if (! *pSF) {
	return huffman_status_memory_failure;
    }
    return huffman_status_ok;
}

static huffman_status_t NOIGNORE
free_frequencies (SymbolFrequencies pSF)
{
    free (pSF);
    return huffman_status_ok;
}

/* Store of Huffman output in memory. */

typedef struct buf_cache_tag
{
    unsigned char *cache;
    /* Allocated bytes in "cache". */
    unsigned int cache_len;
    /* Bytes used in "cache". */
    unsigned int cache_cur;
    /* Output. */
    unsigned char **pbufout;
    /* Length of output. */
    unsigned int *pbufoutlen;
}
buf_cache;

/* Initialize "pc" to "cache_size" of memory, and set the other
   variables to zero. */

static huffman_status_t NOIGNORE
init_cache (buf_cache* pc,
	    unsigned int cache_size,
	    unsigned char **pbufout,
	    unsigned int *pbufoutlen)
{
    if (! pc || ! pbufout || ! pbufoutlen) {
	return huffman_status_null_pointer;
    }
	
    pc->cache = malloc (cache_size);
    if (! pc->cache) {
	return huffman_status_memory_failure;
    }
    MEMPLUS;
    pc->cache_len = cache_size;
    pc->cache_cur = 0;
    pc->pbufout = pbufout;
    *pbufout = NULL;
    pc->pbufoutlen = pbufoutlen;
    *pbufoutlen = 0;

    return huffman_status_ok;
}

static void free_cache (buf_cache* pc)
{
    if (pc->cache) {
	free (pc->cache);
	MEMFREE;
	pc->cache = NULL;
    }
}

/* Copy the contents of pc->cache into *pc->bufout, adjusting the
   sizes etc. */

static huffman_status_t NOIGNORE
flush_cache (buf_cache* pc)
{
    if (! pc) {
	return huffman_status_null_pointer;
    }
	
    if (pc->cache_cur > 0) {
	unsigned int newlen = pc->cache_cur + *pc->pbufoutlen;
	unsigned char* tmp;
	if (! *pc->pbufout) {
	    MEMPLUS;
	}
	tmp = realloc (*pc->pbufout, newlen);
	if (! tmp) {
	    return huffman_status_memory_failure;
	}

	memcpy (tmp + *pc->pbufoutlen, pc->cache, pc->cache_cur);

	*pc->pbufout = tmp;
	*pc->pbufoutlen = newlen;
	pc->cache_cur = 0;
    }

    return huffman_status_ok;
}

/* Write "to_write_len" bytes from "to_write" into the cache "pc". */

static huffman_status_t NOIGNORE
write_cache (buf_cache * pc,
	     const void *to_write,
	     unsigned int to_write_len)
{
    unsigned char* tmp;

    if (! pc || ! to_write) {
	return huffman_status_null_pointer;
    }
    if (pc->cache_len < pc->cache_cur) {
	return huffman_status_bounds;
    }
	
    /* If trying to write more than the cache will hold,
     * flush the cache and allocate enough space immediately,
     * that is, don't use the cache. */

    if (to_write_len > pc->cache_len - pc->cache_cur) {
	unsigned int newlen;
	huffman_status_t rc;
	rc = flush_cache (pc);
	if (rc != huffman_status_ok) {
	    return rc;
	}
	newlen = *pc->pbufoutlen + to_write_len;
	if (! *pc->pbufout) {
	    MEMPLUS;
	}
	tmp = realloc (*pc->pbufout, newlen);
	if (! tmp) {
	    return huffman_status_memory_failure;
	}

	memcpy (tmp + *pc->pbufoutlen, to_write, to_write_len);
	*pc->pbufout = tmp;
	*pc->pbufoutlen = newlen;
    }
    else {
	/* Write the data to the cache. */
	memcpy (pc->cache + pc->cache_cur, to_write, to_write_len);
	pc->cache_cur += to_write_len;
    }

    return huffman_status_ok;
}

static unsigned int
get_symbol_frequencies_from_memory (SymbolFrequencies sf,
				    const unsigned char *bufin,
				    unsigned int bufinlen)
{
    unsigned int i;
    unsigned int total_count = 0;
	
    /* Count the frequency of each symbol in the input file. */
    for (i = 0; i < bufinlen; i++) {
	unsigned char uc = bufin[i];
	if (! sf[uc]) {
	    sf[uc] = new_leaf_node (uc);
	}
	sf[uc]->count++;
	total_count++;
    }

    return total_count;
}

/*
  compare symbol table entries p1 and p2.

 * When used by qsort, SFComp sorts the array so that
 * the symbol with the lowest frequency is first. Any
 * NULL entries will be sorted to the end of the list.
 */
static int
SFComp (const void *p1, const void *p2)
{
    const huffman_node *hn1 = *(const huffman_node**)p1;
    const huffman_node *hn2 = *(const huffman_node**)p2;

    /* Sort all NULLs to the end. */
    if (hn1 == NULL && hn2 == NULL) {
	return 0;
    }
    if (hn1 == NULL) {
	return 1;
    }
    if (hn2 == NULL) {
	return -1;
    }
    if (hn1->count > hn2->count) {
	return 1;
    }
    else if (hn1->count < hn2->count) {
	return -1;
    }
    return 0;
}

static void UNUSED
print_freqs (const SymbolFrequencies sf)
{
    size_t i;
    for (i = 0; i < MAX_SYMBOLS; i++) {
	if (sf[i]) {
	    printf ("%d, %ld\n", sf[i]->symbol, sf[i]->count);
	}
	else {
	    printf ("NULL\n");
	}
    }
}

/*
 * build_symbol_encoder builds a SymbolEncoder by walking
 * down to the leaves of the Huffman tree and then,
 * for each leaf, determines its code.
 */
static huffman_status_t NOIGNORE
build_symbol_encoder (huffman_node *subtree, SymbolEncoder se)
{
    if (subtree == NULL) {
	return huffman_status_null_pointer;
    }

    if (subtree->isLeaf) {
	se[subtree->symbol] = new_code (subtree);
    }
    else {
	huffman_status_t rc;
	rc = build_symbol_encoder (subtree->zero, se);
	if (rc != huffman_status_ok) {
	    return rc;
	}
	rc = build_symbol_encoder (subtree->one, se);
	if (rc != huffman_status_ok) {
	    return rc;
	}
    }
    return huffman_status_ok;
}

/*
 * calculate_huffman_codes turns pSF into an array
 * with a single entry that is the root of the
 * huffman tree. The return value is a SymbolEncoder,
 * which is an array of huffman codes index by symbol value.
 */

static SymbolEncoder
calculate_huffman_codes (SymbolFrequencies sf)
{
    unsigned int i = 0;
    unsigned int n = 0;
    huffman_node *m1 = NULL, *m2 = NULL;
    SymbolEncoder pSE;
    huffman_status_t rc;
#if 0
    printf ("BEFORE SORT\n");
    print_freqs (sf);
#endif

    /* Sort the symbol frequency array by ascending frequency. */
    qsort (sf, MAX_SYMBOLS, sizeof (huffman_node *), SFComp);

#if 0
    printf ("AFTER SORT\n");
    print_freqs (sf);
#endif

    /* Get the number of symbols. */
    for (n = 0; n < MAX_SYMBOLS && sf[n]; n++) {
	;
    }

    /*
     * Construct a Huffman tree. This code is based
     * on the algorithm given in Managing Gigabytes
     * by Ian Witten et al, 2nd edition, page 34.
     * Note that this implementation uses a simple
     * count instead of probability.
     */
    for (i = 0; i < n - 1; i++) {
	/* Set m1 and m2 to the two subsets of least probability. */
	m1 = sf[0];
	m2 = sf[1];

	/* Replace m1 and m2 with a set {m1, m2} whose probability
	 * is the sum of that of m1 and m2. */
	sf[0] = m1->parent = m2->parent =
	    new_nonleaf_node (m1->count + m2->count, m1, m2);
	sf[1] = NULL;
		
	/* Put newSet into the correct count position in pSF. */
	qsort (sf, n, sizeof (huffman_node *), SFComp);
    }

    /* Build the SymbolEncoder array from the tree. */
    pSE = calloc (MAX_SYMBOLS, sizeof (huffman_code *));
    if (! pSE) {
	return 0;
    }
    MEMPLUS;
    rc = build_symbol_encoder (sf[0], pSE);
    if (rc != huffman_status_ok) {
	return 0;
    }
    return pSE;
}

/*
 * Write the huffman code table. The format is:
 * 4 byte code count in network byte order.
 * 4 byte number of bytes encoded
 *   (if you decode the data, you should get this number of bytes)
 * code1
 * ...
 * codeN, where N is the count read at the begginning of the file.
 * Each codeI has the following format:
 * 1 byte symbol, 1 byte code bit length, code bytes.
 * Each entry has numbytes_from_numbits code bytes.
 * The last byte of each code may have extra bits, if the number of
 * bits in the code is not a multiple of 8.
 */
/*
 * Allocates memory and sets *pbufout to point to it. The memory
 * contains the code table.
 */
static huffman_status_t NOIGNORE
write_code_table_to_memory (buf_cache * pc,
			    SymbolEncoder se,
			    uint32_t symbol_count)
{
    uint32_t i, count = 0;
    huffman_status_t rc;

    /* Determine the number of entries in se. */
    for (i = 0; i < MAX_SYMBOLS; i++) {
	if (se[i]) {
	    count++;
	}
    }

    /* Write the number of entries in network byte order. */
    i = htonl (count);
    rc = write_cache (pc, &i, sizeof (i));
    if (rc != huffman_status_ok) {
	return rc;
    }
    /* Write the number of bytes that will be encoded. */
    symbol_count = htonl (symbol_count);
    rc = write_cache (pc, &symbol_count, sizeof (symbol_count));
    if (rc != huffman_status_ok) {
	return rc;
    }

    /* Write the entries. */
    for (i = 0; i < MAX_SYMBOLS; i++) {
	huffman_code *p = se[i];
	if (p) {
	    unsigned int numbytes;
	    /* The value of i is < MAX_SYMBOLS (256), so it can
	       be stored in an unsigned char. */
	    unsigned char uc = (unsigned char)i;
	    /* Write the 1 byte symbol. */
	    rc = write_cache (pc, &uc, sizeof (uc));
	    if (rc != huffman_status_ok) {
		return rc;
	    }
	    /* Write the 1 byte code bit length. */
	    uc = (unsigned char)p->numbits;
	    rc = write_cache (pc, &uc, sizeof (uc));
	    if (rc != huffman_status_ok) {
		return rc;
	    }
	    /* Write the code bytes. */
	    numbytes = numbytes_from_numbits (p->numbits);
	    rc = write_cache (pc, p->bits, numbytes);
	    if (rc != huffman_status_ok) {
		return rc;
	    }
	}
    }
    return huffman_status_ok;
}

/* Put "readlen" bytes of "buf + *pindex" into "*bufout", checking
   that "readlen" does not overflow the end of "buf" at "buflen". */

static huffman_status_t NOIGNORE
memread (const unsigned char* buf,
	 unsigned int buflen,
	 unsigned int *pindex,
	 void* bufout,
	 unsigned int readlen)
{
    if (! buf || ! pindex || ! bufout) {
	return huffman_status_null_pointer;
    }
    if (buflen < *pindex) {
	return huffman_status_bounds;
    }

    if (readlen + *pindex >= buflen) {
	return huffman_status_bounds;
    }

    memcpy (bufout, buf + *pindex, readlen);
    *pindex += readlen;
    return huffman_status_ok;
}

/* "pindex" is the offset into bufin. */

static huffman_node*
read_code_table_from_memory (const unsigned char* bufin,
			     unsigned int bufinlen,
			     unsigned int *pindex,
			     uint32_t *pDataBytes)
{
    huffman_node *root = new_nonleaf_node (0, NULL, NULL);
    uint32_t count;

    if (! root) {
	return NULL;
    }
	
    /* Read the number of entries.
       (it is stored in network byte order). */
    if (memread (bufin, bufinlen, pindex, &count, sizeof (count))) {
	free_huffman_tree (root);
	return NULL;
    }

    count = ntohl (count);

    /* Read the number of data bytes this encoding represents. */
    if (memread (bufin, bufinlen, pindex, pDataBytes, sizeof (*pDataBytes))) {
	free_huffman_tree (root);
	return NULL;
    }

    *pDataBytes = ntohl (*pDataBytes);

    /* Read the entries. */
    while (count-- > 0) {
	unsigned int curbit;
	unsigned char symbol;
	unsigned char numbits;
	unsigned char numbytes;
	unsigned char *bytes;
	huffman_node *p = root;

	if (memread (bufin, bufinlen, pindex, &symbol, sizeof (symbol))) {
	    free_huffman_tree (root);
	    return NULL;
	}

	if (memread (bufin, bufinlen, pindex, &numbits, sizeof (numbits))) {
	    free_huffman_tree (root);
	    return NULL;
	}
		
	numbytes = (unsigned char) numbytes_from_numbits (numbits);
	bytes = malloc (numbytes);
	if (! bytes) {
	    return NULL;
	}
	MEMPLUS;
	if (memread (bufin, bufinlen, pindex, bytes, numbytes)) {
	    free (bytes);
	    MEMFREE;
	    free_huffman_tree (root);
	    return NULL;
	}

	/*
	 * Add the entry to the Huffman tree. The value
	 * of the current bit is used switch between
	 * zero and one child nodes in the tree. New nodes
	 * are added as needed in the tree.
	 */
	for (curbit = 0; curbit < numbits; curbit++) {
	    if (get_bit (bytes, curbit)) {
		if (p->one == NULL) {
		    p->one = curbit == (unsigned char)(numbits - 1)
			? new_leaf_node (symbol)
			: new_nonleaf_node (0, NULL, NULL);
		    p->one->parent = p;
		}
		p = p->one;
	    }
	    else {
		if (p->zero == NULL) {
		    p->zero = curbit == (unsigned char)(numbits - 1)
			? new_leaf_node (symbol)
			: new_nonleaf_node (0, NULL, NULL);
		    p->zero->parent = p;
		}
		p = p->zero;
	    }
	}
		
	free (bytes);
	MEMFREE;
    }

    return root;
}

static int
do_memory_encode (buf_cache *pc,
		  const unsigned char* bufin,
		  unsigned int bufinlen,
		  SymbolEncoder se)
{
    unsigned char curbyte = 0;
    unsigned char curbit = 0;
    unsigned int i;
    huffman_status_t rc;

    for (i = 0; i < bufinlen; i++) {
	unsigned char uc = bufin[i];
	huffman_code *code = se[uc];
	unsigned long j;
		
	for (j = 0; j < code->numbits; j++) {
	    /* Add the current bit to curbyte. */
	    curbyte |= get_bit (code->bits, j) << curbit;

	    /* If this byte is filled up then write it
	     * out and reset the curbit and curbyte. */
	    curbit++;
	    if (curbit == 8) {
		rc = write_cache (pc, &curbyte, sizeof (curbyte));
		if (rc != huffman_status_ok) {
		    return rc;
		}
		curbyte = 0;
		curbit = 0;
	    }
	}
    }

    /*
     * If there is data in curbyte that has not been
     * output yet, which means that the last encoded
     * character did not fall on a byte boundary,
     * then output it.
     */
    if (curbit > 0) {
	rc = write_cache (pc, &curbyte, sizeof (curbyte));
	if (rc != huffman_status_ok) {
	    return rc;
	}
    }
    return huffman_status_ok;
}

#define CACHE_SIZE 1024

huffman_status_t NOIGNORE
huffman_encode_memory (const unsigned char *bufin,
		       unsigned int bufinlen,
		       unsigned char **pbufout,
		       unsigned int *pbufoutlen)
{
    SymbolFrequencies sf;
    SymbolEncoder se;
    huffman_node *root = NULL;
    huffman_status_t rc;
    unsigned int symbol_count;
    buf_cache cache;

    MEMCLEAR;
    /* Ensure the arguments are valid. */
    if (! pbufout || ! pbufoutlen) {
	return huffman_status_null_pointer;
    }

    rc = init_cache (&cache, CACHE_SIZE, pbufout, pbufoutlen);
    if (rc != huffman_status_ok) {
	return rc;
    }

    /* Allocate memory for sf and set all its frequencies to 0. */
    rc = init_frequencies (& sf);
    if (rc != huffman_status_ok) {
	return rc;
    }

    /* Get the frequency of each symbol in the input memory. */
    symbol_count = get_symbol_frequencies_from_memory (sf, bufin, bufinlen);

    /* Build an optimal table from the symbolCount. */
    se = calculate_huffman_codes (sf);
    root = sf[0];

    /* Scan the memory again and, using the table
       previously built, encode it into the output memory. */
    rc = write_code_table_to_memory (& cache, se, symbol_count);
    if (rc == huffman_status_ok) {
	rc = do_memory_encode (& cache, bufin, bufinlen, se);
    }

    /* Flush the cache. */
    rc = flush_cache (& cache);
    if (rc != huffman_status_ok) {
	return rc;
    }
    /* Free the Huffman tree. */
    free_huffman_tree (root);
    free_encoder (se);
    free_cache (& cache);
    rc = free_frequencies (sf);
    MEMCHECK (1);
    return rc;
}

huffman_status_t NOIGNORE
huffman_decode_memory (const unsigned char *bufin,
		       unsigned int bufinlen,
		       unsigned char **pbufout,
		       unsigned int *pbufoutlen)
{
    huffman_node *root, *p;
    unsigned int data_count;
    unsigned int i = 0;
    unsigned char *buf;
    unsigned int bufcur = 0;

    MEMCLEAR;
    /* Ensure the arguments are valid. */
    if (! pbufout || ! pbufoutlen) {
	return huffman_status_null_pointer;
    }

    /* Read the Huffman code table. */
    root = read_code_table_from_memory (bufin, bufinlen, &i, &data_count);
    if (! root) {
	return huffman_status_null_pointer;
    }

    buf = malloc (data_count);
    if (! buf) {
	return huffman_status_memory_failure;
    }
    MEMPLUS;
    /* Decode the memory. */
    p = root;
    for (; i < bufinlen && data_count > 0; i++)  {
	unsigned char byte = bufin[i];
	// Bit mask
	unsigned char mask = 1;
	while (data_count > 0 && mask) {
	    p = byte & mask ? p->one : p->zero;
	    mask <<= 1;

	    if (p->isLeaf) {
		buf[bufcur] = p->symbol;
		bufcur++;
		p = root;
		data_count--;
	    }
	}
    }

    free_huffman_tree (root);
    *pbufout = buf;
    *pbufoutlen = bufcur;
    MEMCHECK (1);
    return huffman_status_ok;
}
