package Evo;
use Evo::Base -strict;
use Evo::Manager;
use Evo::Messenger;

use Exporter 'import';

my @MSN  = qw(msn_subscribe msn_broadcast msn_unsubscribe);
my @DSL  = qw(dsl_stash dsl_check_stash dsl_call dsl_depth);
my @FUNC = qw(asap);
our @EXPORT_OK = (@DSL, @FUNC, @MSN);
our %EXPORT_TAGS
  = (all => \@EXPORT_OK, dsl => \@DSL, msn => \@MSN, func => \@FUNC);

our $VERSION = '0.0152';    # VERSION

sub asap : prototype(&@) { Evo::Manager->singleton->asap(@_) }

sub dsl_depth : prototype()         { Evo::Manager->singleton->dsl_depth; }
sub dsl_call : prototype($;@)       { Evo::Manager->singleton->dsl_call(@_); }
sub dsl_stash : prototype(;$$)      { Evo::Manager->singleton->dsl_stash(@_); }
sub dsl_check_stash : prototype(;$) { Evo::Manager->singleton->dsl_stash(@_); }

sub _msn { state $msn = Evo::Messenger->new }
sub msn_broadcast : prototype($$)   { _msn->broadcast(@_); }
sub msn_subscribe : prototype($$)   { _msn->subscribe(@_); }
sub msn_unsubscribe : prototype($$) { _msn->unsubscribe(@_); }


1;

# ABSTRACT: Perl Evo design pattern

__END__

=pod

=encoding UTF-8

=head1 NAME

Evo - Perl Evo design pattern

=head1 VERSION

version 0.0152

=head1 SYNOPSIS

  use Evo::Base -strict;
  use Evo ':all';


  # next tick
  our $state = 'non blocking';

  # always nb
  asap {
    say $state;
    local $state = 'blocking?';
    asap { say $state };
  };


  # dsl
  # prints 1;2;foo
  my @args = (1, 2);
  my $print = sub(@args) { local $, = ';'; say @args, dsl_stash 'foo' };

  dsl_call({foo => 'foo'}, @args, $print);

  require Evo::Manager;
  my $mngr = Evo::Manager->new;

  # currying
  my $debug = sub($evo, $next) {
    sub(@args) {
      say 'Got args: ', join ';', @args;
      $next->(@args);
      }
  };

  my $safe = sub($evo, $next) {
    sub(@args) {
      local $@;
      eval { $next->(@args) };
      warn "catched: $@" if $@;
      }
  };

  my $die = sub {die};
  my $curry = $mngr->curry_wrappers($debug, $safe);
  $curry->(sub { say "going to die"; die })->(1, 2);

  my $way = $mngr->build_way;
  my $train = $mngr->build_train(way => $way);
  $way->curry($curry)->add(sub { say 'step1'; })->add(sub { die "step 2" })
    ->add(sub { say "step 3"; });

  $train->start;

=head1 DESCRIPTION

  Tiny framework for organization non-trivial code into a simple logic.
  It includes benefits from functional and OO styles.

  Write fast, reusable and testable code.
  Familiar with OO? - Use it. Familiar with higher-order functions? - Use it.
  Familiar with non-blocking style? - Use it. Want DSL? - Use it.

  Do it the right way.

=head1 DESCRIPTION

  Attention. Don't use it right now. Wait for the stable release

=head1 METHODS

=head2 msn_broadcast, msn_subscribe, msn_unsubscribe

Calls corresponding L<Evo::Messenger> methods passing a singleton as the first
argument

=head2 asap, dsl_depth, dsl_stash dsl_check_stash dsl_call dsl_depth

Calls corresponding L<Evo::Manager> methods passing a singleton as the first
argument

=head1 AUTHOR

alexbyk.com

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by alexbyk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
