
/* Copyright (c) 1992 Vincent Cate
 * All Rights Reserved.
 *
 * Permission to use and modify this software and its documentation
 * is hereby granted, provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative works
 * or modified versions, and any portions thereof, and that both notices
 * appear in supporting documentation.  This software or any derivate works
 * may not be sold or distributed without prior written approval from
 * Vincent Cate.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND VINCENT CATE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL
 * VINCENT CATE BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Users of this software agree to return to Vincent Cate any improvements
 * or extensions that they make and grant Vincent Cate the rights to
 * redistribute these changes.
 *
 *
 * Jan 1993 this file completely rewritten by
 *
 *           Juergen Hannken-Illjes (hannken@eis.cs.tu-bs.de)
 *
 * Exports:
 *
 * extern int
 * InTransit(char *pathName);
 *  return:
 *    TRUE:  if transfer of file is in progress
 *    FALSE: otherwise
 *
 *
 * extern int NumFinishedFTPs
 *  Number of succesful FTPs done so far
 *
 *
 * extern int
 * DoAnFtp(struct HostEntry *HostPtr, char *command, char *remPath,
 *         char *ResultPath);
 *  return:
 *    AOK:       if file or directory has been transferred successfully
 *    AOLD:      if file doesn't exist on remote host (any more)
 *                 this doesn't work for directorys, as the error is written
 *                 to the output file as '... not found'
 *    AFAILED:   if transfer failed and will likely fail, if command is redone
 *    AWORKING:  if transfer is in progress or no free slot for another FTP
 *    SOFTERROR: if there has been some sort of FTP error that might go away with retry
 *
 *  side effect:
 *    uses and updates the following fields of *HostPtr:
 *      Path:       used to create hostname
 *      Type:       used to determine host type, set, if type changes
 *      NumErrorsSinceOk: used and set to collect recent soft failures
 *      LastUsed:   used and set
 *
 *    updates NumFinishedFTPs
 *
 *
 * Imports:
 * debugLog
 * Log()
 * ATypeToString()
 *
 * PathToHostName()
 * TimeInSeconds()
 *
 * LeaveAnError()
 * RemoveError()
 *
 * Whenever you access a file in alex-cache, do it as follows:
 *
 *   if(ExistsInCacheAndSizeOk(pathname)) ||
 *           (InTransit(pathname) && BigEnough(pathname, NeededSize)) {
 *             use the file
 *    } else {
 *
 *         result = DoAnFtp(...);
 *         if(result == AOK)
 *           update stat info, proceed;
 *         if(result == AFAILED || result == AOLD)
 *           update stat info, return(AFAIL);
 *         if(result == SOFTERROR)
 *           return(AWORKING); (* try again next time *)
 *         if(result == AWORKING){
 *           if(SizeOk(pathname))
 *             proceed;
 *           else
 *             return(AWORKING);
 *       }
 *   }
 *   proceed as normal
 *
 */

#include "alexincs.h"
#include "alex.h"
#include "alexftp.h"

char passWord[MAXPATH];                 /* anon password: "user@host" */

/* parent code: the following code will be used from ALEX */

int NumFinishedFTPs = 0;		/* number of finished FTP's */

static struct ftptab{			/* structure for a child */
  int  InUse;				/* if 0, this child doesn't exist */
  int  LastUsed;			/* time of last Status update */
  struct HostEntry *HostPtr;		/* host description for this child */
  char HostName[MAXPATH];		/* name of remote host */
  int  Pid;				/* Pid of child */
  int  CntrlIn;				/* pipe to child ("r") */
  int  CntrlOut;			/* pipe to child ("w") */
  char CurCommand[MAXCOMMAND];		/* current command or ' ??', if idle */
  char ResultPath[MAXPATH];		/* name of local file or '\0??', -"- */
  char LastError[MAXPATH];		/* last error, if any */
  char Status;				/* current Status of child (FTP_???) */
  double StartTime;                     /* for performance statistics        */
}FtpCache[MAXFTPSOPEN];



RemoveError(ResultPath)
char *ResultPath;
{
    char AlexErrorPath[MAXPATH], DirPath[MAXPATH];

    if (strlen(ResultPath) > CACHEDIRLEN) {
        PathToDir(ResultPath, DirPath);

        (void) strcpy(AlexErrorPath, DirPath);
        (void) strcat(AlexErrorPath, SLASHALEXERROR);  /* add /.alex.error                  */
        (void) unlink(AlexErrorPath);                  /* if there we remove it not so what */
    }
}


/* Certain types of chars are really painful in filenames so remove them
 */
OnlyPathSafeChars(s)
char *s;
{
    if ((s==NULL) || (*s==0)) return;

    while (*s != 0) {
        if (isalnum(*s) || HasChar(".-=+_", *s)) {
            s++;                                   /* ok move to next */
        } else {
            *s++ = FILLCHAR;                       /* not ok so make FILLCHAR and move to next */
        }
    }
}



/*  Make a .alex.error for FtpNum
 *       Find directory from .ResultPath
 *       Use .LastBack for text of error message
 */
int LeaveAnError(ResultPath, LastBack)
char *ResultPath, *LastBack;
{
    char DirPath[MAXPATH], ErrorFilePath[MAXPATH], ErrorMessage[MAXPATH];
    char AlexInfoPath[MAXPATH];
    int Result;
    FILE *AIfile;
    static int LastTimeAdded= -2;
    static char LastPathAdded[MAXPATH];
    int    MTime;

    if (strlen(ResultPath) <= CACHEDIRLEN) {
        ToLog(DBERROR, "LeaveAnError bogus ResultPath |%s|\n", ResultPath);
        return(AFAIL);
    }

    ToLog(6, "LeaveAnError %s %s\n", ResultPath, LastBack);

    PathToDir(ResultPath, DirPath);

    (void) strcpy(ErrorFilePath, DirPath);
    (void) strcat(ErrorFilePath, SLASHALEXERROR);

    (void) strcpy(ErrorMessage, "ALEX FTP ERROR  ");
    (void) strcat(ErrorMessage, LastBack);

    OnlyPathSafeChars(ErrorMessage);                
    NoTrailingC(ErrorMessage, FILLCHAR);             

    Result=StringIntoFile(ErrorFilePath, ErrorMessage);

    if (Result == AOK) {
        (void) strcpy(AlexInfoPath, DirPath);
        (void) strcat(AlexInfoPath, SLASHALEXINFO);

        MTime = MTimeOfFile(AlexInfoPath);
        if (MTime < 0) {
            Log2("LeaveAnError not adding to .alex.info since non existant", AlexInfoPath);

        } else if (streql(LastPathAdded, AlexInfoPath) && (MTime == LastTimeAdded)) {
            Log2("LeaveAnError already added this error message to .alex.info ", ErrorMessage);

        } else {
            TotalErrorMsg++;

            AIfile=AFOPEN(AlexInfoPath, "a");
            if (AIfile == NULL) {
                Log2("LeaveAnError could not open ", AlexInfoPath);
                Result=AFAIL;
            } else {
                Result=AddAnyErrorToStream(AIfile, DirPath);
                (void) AFCLOSE(AIfile);
                if (Result != AFAIL) {
                     Result = SortFile(AlexInfoPath);
                } else {
                     (void) SortFile(AlexInfoPath);       /* we should sort anyway */
                }
            }
            strcpy(LastPathAdded, AlexInfoPath);
            LastTimeAdded=MTimeOfFile(AlexInfoPath);
        }
    } else {
        ToLog(DBERROR, "LeaveAnError could not do StringIntoFile to %s\n", ErrorFilePath);
    }

    LogT("LeaveAnError returning ", Result);
    return(Result);
}


/*
 * check for host specific anon ftp password
 *
 * The file ANON_PASSWD can have lines like:
 *    howland.reston.ans.net  ansco+re
 *
 * If you add a file when you did not have one before you 
 * need to do a "grepnkill alexd" to restart the server as
 * it caches the fact that the file does not exist.
 */
static char *
GetPassword(hostname)
     char *hostname;
{
  int hostlen;
  FILE *passwd;
  static char line[265];
  static int NoFile= 0;

  if (!NoFile) {                             /* if this fails we will only try that once */
      passwd = fopen(ANON_PASSWD, "r");
      if (passwd == NULL) {
          NoFile = 1;
      }
  }

  if (!NoFile) {
    hostlen = strlen(hostname);
    /*
     * scan passwd file for hostname
     */
    while (fgets(line, sizeof(line), passwd)) {

       if (hostlen + 1 >= strlen(line) - 1 ||
	   strncmp(line, hostname, hostlen) != 0 ||
	   !isspace(line[hostlen]))
	 continue;

       /*
	* skip whitespace
	*/
       while (line[hostlen] && isspace(line[hostlen]))
	 hostlen++;

       /*
	* rest of line is password string
	*/
       if (line[hostlen]) {
	 line[strlen(line) - 1] = '\0';
	 fclose (passwd);
	 return &line[hostlen];
       }
    }
    fclose(passwd);
  }
  return(NULL);
}

/*
 * create name of a child, for logging purpose only
 */

static char *
NameOf(f)
     struct ftptab *f;
{
  static char buf[MAXPATH+10];

  sprintf(buf,"[%d:%s(%c)]",f-FtpCache,f->HostName,f->Status);
  return(buf);
}

/*
 * kill this child so its slot may be reused
 */

static void
KillFtp(fp)
     struct ftptab *fp;
{
  struct ftptab *f;
  int Pid;

  ToLog(1, "DoAnFtp::KillFtp: %s\n",NameOf(fp));

  kill(fp->Pid,SIGHUP);
  while((Pid = wait3(NULL,WNOHANG|WUNTRACED,NULL)) > 0)
    if(Pid == fp->Pid)
      break;
    else
      for(f = FtpCache; f < FtpCache+MAXFTPSOPEN; f++)
	if(f->InUse && f->Pid == Pid && FTP_RUNNING(f->Status)){
	  ToLog(1, "DoAnFtp::KillFtp: %s did exit, Status updated\n",NameOf(f));
	  f->Status = FTP_FAILED;
	}

  if(fp->CntrlIn >= 0)
    close(fp->CntrlIn);
  if(fp->CntrlOut >= 0)
    close(fp->CntrlOut);
  bzero((char *)fp, sizeof(struct ftptab));
}

/*
 * create a new child
 */

static int
NewFtp(f, HostPtr, HostName, CurCommand, ResultPath, UidStr)
     struct ftptab *f;
     struct HostEntry *HostPtr;
     char *HostName;
     char *CurCommand;
     char *ResultPath;
     char *UidStr;
{
  int i,nfds;
  int pipeTo[2];
  int pipeFrom[2];
  char RemoteTypeArg;
  char *HostPassword;

  if((HostPtr->NumErrorsSinceOk >= NUMFTPFASTTRIES) &&    /* few fast tries/retries then slow */
	(HostPtr->LastUsed+(HostPtr->NumErrorsSinceOk * SECSTILLRETRY) > TimeInSeconds())) {
      ToLog(DBFTP, "NewFtp  have failed with this host recently so just returning AFAIL %d\n ", 
                             HostPtr->NumErrorsSinceOk);
      LeaveAnError(ResultPath, "Connect failed");
      return(AFAIL);                                  
  }

  f->InUse = 1;
  f->LastUsed = TimeInSeconds();
  f->HostPtr = HostPtr;
  f->HostPtr->LastUsed = TimeInSeconds();
  (void) strcpy(f->HostName, HostName);
  (void) strcpy(f->CurCommand, CurCommand);
  (void) strcpy(f->ResultPath, ResultPath);
  f->Status = FTP_WORKING;
  f->CntrlIn = f->CntrlOut = -1;
  ToLog(DBFTP, "DoAnFtp::NewFtp: creating %s\n",NameOf(f));

  if(pipe(pipeFrom) < 0){
    ToLog(1, "DoAnFtp::NewFtp: cannot setup pipes\n");
    KillFtp(f);
    return(AFAIL);
  }
  if(pipe(pipeTo) < 0){
    close(pipeFrom[0]);
    close(pipeFrom[1]);
    ToLog(1, "DoAnFtp::NewFtp: cannot setup pipes\n");
    KillFtp(f);
    return(AFAIL);
  }
 
  HostPassword = GetPassword(HostName);
  if (HostPassword) {
    (void) strncpy(passWord, HostPassword, sizeof(passWord) - 1);
  } else {
    (void) strcpy(passWord, UidStr);
  }


  f->Pid = fork();
    
  if(f->Pid == 0){				/* child */
    debugLog = NULL;
    if(dup2(pipeTo[0],0) < 0 || dup2(pipeFrom[1],1) < 0) {
      ToLog(1, "DoAnFtp::NewFtp: dup2 failed\n");
      exit(1);
    }
    /* setlinebuf(stdout);                      not portable to HPUX so we fflush()  */
    nfds = getdtablesize();
    for(i = 2; i < nfds; i++)
      close(i);
    switch (HostPtr->Type) {
      case UNIX:   RemoteTypeArg= SYS_UNIX;
                   break;

      case DECVMS: RemoteTypeArg = SYS_VMS;
                   break;

      default:     RemoteTypeArg = SYS_UNKNOWN;
                   break;
    }
    StartFtpClient(f->HostName, RemoteTypeArg);
    exit(1);
  }else if(f->Pid > 0){			/* parent */
    TotalFtpsForked++;
    close(pipeFrom[1]);
    close(pipeTo[0]);
    f->CntrlIn = pipeFrom[0];
    f->CntrlOut = pipeTo[1];
    f->LastError[0]=0;
    write(f->CntrlOut,CurCommand,strlen(CurCommand));
    f->StartTime = TimeInSeconds();
    f->Status = FTP_WORKING;
    ToLog(DBRPC, "DoAnFtp::NewFtp: created %s Pid is %d\n",NameOf(f),f->Pid);
    return(AWORKING);
  }else{
    ToLog(1, "DoAnFtp::NewFtp: cannot fork ftp child\n");
    KillFtp(f);
    return(AFAIL);
  }
  return(AFAIL);   /* for lint only */
}

/*
 * update Status for this child
 */

static void
UpdateStatus(f)
     struct ftptab *f;
{
  char buf[MAXPATH+4];
  char lastStatus;
  int nread, i, done, NeedUpdatePerfStats;

  ToLog(DBFTP, "DoAnFtp::UpdateStatus: %s", NameOf(f));

  NeedUpdatePerfStats=0;
  done = 0;
  while(!done){
    lastStatus = f->Status;
    if(ioctl(f->CntrlIn, FIONREAD, &nread) < 0 || nread < 2){
      done = 1;
      continue;
    }

    for(i = 0; i < nread; i++){
      read(f->CntrlIn, buf+i, 1);
      if(buf[i] == '\n')
	break;
    }
    buf[i] = '\0';                                            

    ToLog(DBFTP, " {%s} ", buf);                   /* send everything we read to log */

    f->LastUsed = TimeInSeconds();

    ToLog(DBFTP, "+'%c'", buf[0]);
 
    f->Status = buf[0]; 
    if (f->Status == FTP_CONNECTED) {
      switch (buf[1]) {
          case SYS_UNIX: f->HostPtr->Type = UNIX;
                         break;

          case SYS_VMS:  f->HostPtr->Type = DECVMS;                 
                         break;

          default:       f->HostPtr->Type = UNKNOWN;                 
                         break;
      }
      ToLog(DBFTP, "[%c->%s]", buf[1], ATypeToString(f->HostPtr->Type));
    }

    switch(f->Status){                            /* look at what we got back from child */
      case FTP_CONNECTED:
        f->HostPtr->LastUsed = TimeInSeconds();
        f->Status = FTP_WORKING;
        break;

      case FTP_OK:
        f->HostPtr->NumErrorsSinceOk = 0;
        f->HostPtr->LastUsed = TimeInSeconds();
        if (f->CurCommand[0] == FTP_DIR) {     /* only if we have updated .alex.dir */
            RemoveError(f->ResultPath);        /* otherwise .alex.info might not be updated */
        }                                      /* small chance now though since .alex.info used */
        f->LastError[0]=0;
        f->Status = FTP_IDLE;
        NeedUpdatePerfStats=1;
        break;

      case FTP_NOSUCHHOST:
        f->HostPtr->Type = NOSUCHHOST;
        (void) strcpy(f->LastError, "No such host");
        break;

      case FTP_NOANON:
        f->HostPtr->Type = NOANONFTP;
        (void) strcpy(f->LastError, "No anonymous FTP");
        (void) unlink(f->ResultPath);          /* no longer available whatever it was - flush from cache */
        break;

      case FTP_FAILED:
      case FTP_RETRY:
        if(lastStatus == FTP_QUIT){
            f->Status = FTP_EXITED;
        } else {
            if (strlen(buf)>2) {
              (void) strcpy(f->LastError, &buf[1]);               /* save error message from child */
            }
            f->HostPtr->NumErrorsSinceOk++;
            ToLog(DBFTP, "DoAnFtp::UpdateStatus NumErrorsSinceOk= %d\n", f->HostPtr->NumErrorsSinceOk);
        }
        break;

      case FTP_NOTTHERE:
        (void) strcpy(f->LastError, "No such file");
        f->HostPtr->Type = UNKNOWN;           /* maybe host changed types and we specified filename wrong */
        break;

      case FTP_EXITED:
        if(lastStatus == FTP_WORKING){
	  f->Status = FTP_RETRY;
	  f->HostPtr->NumErrorsSinceOk++;
        }
        break;

      default:
	ToLog(DBERROR, "\nDoAnFtp::UpdateStatus ERROR unexpected state '%c'\n", f->Status);
    }
    ToLog(DBFTP, "='%c'", f->Status);
  }

  ToLog(DBFTP, " done\n"); 

  if (NeedUpdatePerfStats) {
      UpdatePerfStats(f);
  }
}

/*
 * build a command to be sent to child
 */

static int
BuildCommand(command, remPath, ResultPath, CurCommand)
     char *command;
     char *remPath;
     char *ResultPath;
     char *CurCommand;
{
  int len;

  while(remPath[0] == '/')                  /* remove any leading /  */
    remPath++;

  if(!strcmp(command,"get"))
    sprintf(CurCommand,"%c ", FTP_GET);
  else if(!strcmp(command,"dir"))
    sprintf(CurCommand,"%c ", FTP_DIR);
  else
    return(AFAIL);

  (void) strcat(CurCommand, ResultPath);
  (void) strcat(CurCommand," ");

  if(remPath[0] == '\0' && CurCommand[0] == FTP_GET){
    ToLog(DBFTP, "DoAnFtp::BuildCommand: cannot get empty path\n");
    return(AFAIL);
  }
  (void) strcat(CurCommand,remPath);
  if(CurCommand[0] == FTP_DIR){
    len = strlen(CurCommand);
    if(CurCommand[len-1] !='/' && remPath[0] != '\0') {   /* if name add a slash */
      CurCommand[len++] = '/';
    }
    CurCommand[len++] = '.';                              /* always add a dot    */
    CurCommand[len] = '\0';

  }

  (void) strcat(CurCommand,"\n");
  return(AOK);
}

/*
 * kill old childs
 */

static void
UpdateAllFtps()
{
  struct ftptab *f;
  char tmp[MAXFTPSOPEN+1];
  int i;

  for(i = 0; i < MAXFTPSOPEN; i++) {
    tmp[i] = FtpCache[i].InUse ? 'X' : '-';
  }
  tmp[MAXFTPSOPEN] = '\0';

  ToLog(DBFTP, "DoAnFtp::UpdateAllFtps: starting [%s]\n",tmp);

  /* kill all ftp's that have exited or are not running since 2*CLIENTTIMEOUT */

  for(f = FtpCache; f < FtpCache+MAXFTPSOPEN; f++)
    if(f->InUse){
      UpdateStatus(f);
      if(f->Status == FTP_EXITED || (!FTP_RUNNING(f->Status) &&
	    (TimeInSeconds()-f->LastUsed) > 2*CLIENTTIMEOUT))
        KillFtp(f);
    }
}


UpdatePerfStats(f)
struct ftptab *f;
{
    double FtpTime, BytesPerSec, BytesFtped;

    NumFinishedFTPs++;

    ToLog(DBPERF, "UpdatePerfStats %s  %d\n", f->ResultPath, NumFinishedFTPs);

    BytesFtped = SizeOfCachedFile(f->ResultPath);
    if (BytesFtped >= 0) {
        switch (f->CurCommand[0]) {
           case FTP_DIR:  TotalDirBytesFTPed += BytesFtped;
                          TotalDirsFTPed++;
                          break;

           case FTP_GET:  TotalFileBytesFTPed += BytesFtped;
                          TotalFilesFTPed++;
                          break;
    
           default:       ToLog(DBERROR, "UpdatePerfStats BUG CurCommand=%c\n", f->CurCommand[0]);
                          break;
        }

        TotalBytesFTPed += BytesFtped;
        FtpTime = TimeInSeconds() - f->StartTime;
        TotalFTPTime += FtpTime;
        BytesPerSec = BytesFtped/FtpTime;
        ToLog(DBPERF, "UpdatePerfStats BytesFtped= %0.0f Time= %2.2f B/S= %0.0f TotalBytesFTPed= %0.0f \n",
        BytesFtped,  FtpTime, BytesPerSec, TotalBytesFTPed);
    } else {
        ToLog(DBPERF, "UpdatePerfStats missing ResultPath. (%c) %s \n", f->Status, f->ResultPath);
    }
}

/*
 * main function, see description on top of file
 */

int
DoAnFtp(HostPtr, command, remPath, ResultPath, UidStr)
     struct HostEntry *HostPtr;
     char *command;
     char *remPath;
     char *ResultPath;
     char *UidStr;                 
{
  char CurCommand[MAXCOMMAND];
  struct ftptab *f;
  int result, OneFtpAlready, Age;
#ifdef AT_DIR
  char RealHostName[MAXPATH];
  char *HostName=&RealHostName[2];


  PathToHostName(HostPtr->Path, RealHostName);
  if ((RealHostName[0]!='@') || (RealHostName[1]!='.')) {
      result=NOSUCHHOST;
      goto done;
  }
#else /* !AT_DIR */
  char HostName[MAXPATH];

  PathToHostName(HostPtr->Path, HostName);
#endif /* AT_DIR */
  ToLog(DBRPC, "DoAnFtp: %s %s %s %s\n", HostName, command, remPath, ResultPath);

  switch (HostPtr->Type) {
      case NOSUCHHOST:   result=NOSUCHHOST;
                         goto done;
  
      case NEVERDOFTP:   LeaveAnError(ResultPath, "This host hates ftp attempts");
                         result = NOANONFTP;
                         goto done;

      case NOANONFTP:    if ((RecentTime - HostPtr->TimeLastCheck) > TRUSTHOSTTYPE) {
                             HostPtr->TimeLastCheck = RecentTime;
                             HostPtr->Type = UNKNOWN;       /* been long enough, try again */
                             HostPtr->TimeZone = RemTimeZoneFAIL;
                         } else {
                             /* LeaveAnError(ResultPath, "No anonymous FTP"); */
                             result=NOANONFTP;
                             goto done;
                         }
                         break;

      case ADOMAIN:      if ((RecentTime - HostPtr->TimeLastCheck) > TRUSTHOSTTYPE) {
                             HostPtr->Type = UNKNOWN;       /* been long enough, try again */
                         } else {
                             result=NOSUCHHOST;
                             goto done;
                         }    
                         break;

      default:           ;
  }

  Age = SecsSinceWrite(ResultPath);
  if (Age>0 && Age<60 && !InTransit(ResultPath)) {
      ToLog(DBERROR, "DoAnFtp ERROR was asked to re-FTP something we just got %s\n", ResultPath);
      if (SizeOfCachedFile(ResultPath) > 1) {
          return(AOK);
      } else {
          return(AFAIL);
      }
  }

  if((result = BuildCommand(command, remPath, ResultPath, CurCommand)) != AOK)
    goto done;

  /* Now it's time to update all states */

  UpdateAllFtps();

  /* First lets see, if we are already running this command */
  OneFtpAlready=0;
  for(f = FtpCache; f < FtpCache+MAXFTPSOPEN; f++) {
    if(streql(f->HostName, HostName)) {
      if (!streql(f->CurCommand, CurCommand)) {
        OneFtpAlready=1;
        if(f->Status == FTP_WORKING) {
            ToLog(DBFTP, "DoAnFtp: already running another command to this site %s\n", HostName);
        }
      } else {
        ToLog(DBFTP, "DoAnFtp: this command was already running\n");
        switch(f->Status) {
        case FTP_OK:
        case FTP_IDLE:
          f->CurCommand[0] = ' ';
          f->ResultPath[0] = '\0';
          f->Status=FTP_IDLE;
	  result = AOK;
	  goto done;
        case FTP_NOTTHERE:
  	  f->CurCommand[0] = ' ';
	  f->ResultPath[0] = '\0';
	  f->Status = FTP_IDLE;
	  LeaveAnError(f->ResultPath, "possible stale directory information");
	  result = AOLD;
	  goto done;
        case FTP_FAILED:
	  LeaveAnError(f->ResultPath, f->LastError);
	  KillFtp(f);
	  result = AFAIL;
	  goto done;
        case FTP_RETRY:
          if((HostPtr->NumErrorsSinceOk > 2) &&    /* 2 fast retries then slow */
	      (HostPtr->LastUsed+(HostPtr->NumErrorsSinceOk*SECSTILLRETRY) > TimeInSeconds())) {
            LeaveAnError(ResultPath, "Connect failed");
	    KillFtp(f);
	    result = AFAIL;                            /* do not retry now */
	    goto done;
	  }
          ToLog(DBFTP, "DoAnFtp: FTP_RETRY with NumErrorsSinceOk= %d\n", HostPtr->NumErrorsSinceOk);
	  KillFtp(f);                                  
	  break;
        case FTP_WORKING:
	  result = AWORKING;
	  goto done;
        case FTP_NOSUCHHOST:
          HostPtr->Type=NOSUCHHOST;
          result = NOSUCHHOST;
	  /* LeaveAnError(f->ResultPath, "No such host"); */
          goto done;
        case FTP_NOANON:
	  /* LeaveAnError(f->ResultPath, "No Anonymous FTP"); */
          HostPtr->Type=NOANONFTP;
          result = NOANONFTP;
          goto done;
        default:
	  ToLog(DBERROR, "DoAnFtp: ERROR unexpected state '%c'\n", f->Status);
          HostPtr->NumErrorsSinceOk++;
          result = AFAIL;
                                 /* AlexAssert(FALSE); */
          goto done; 
        }
      }
    }
  }

  /* Lets see, if we have an ftp to this host, thats idle */

  for(f = FtpCache; f < FtpCache+MAXFTPSOPEN; f++) {
    if(!strcmp(f->HostName,HostName) && f->Status == FTP_IDLE){
      ToLog(DBFTP, "DoAnFtp: reusing an existing connection\n");
      (void) strcpy(f->CurCommand, CurCommand);
      (void) strcpy(f->ResultPath, ResultPath);
      write(f->CntrlOut, CurCommand, strlen(CurCommand));
      f->StartTime = TimeInSeconds();
      f->Status = FTP_WORKING;
      result = AWORKING;
      goto done;
    }
  }

  /* Ok, we must start a new one */
  if (!OneFtpAlready) {
    for(f = FtpCache; f < FtpCache+MAXFTPSOPEN; f++) {
      if(!f->InUse){
        result = NewFtp(f, HostPtr, HostName, CurCommand, ResultPath, UidStr);
        goto done;
      }
    }
  }

  /* No entry available, quit all children, that are idle */

  for(f = FtpCache; f < FtpCache+MAXFTPSOPEN; f++) {
    if(f->Status == FTP_IDLE){
      char buf[3];

      sprintf(buf,"%c\n",FTP_QUIT);
      write(f->CntrlOut, buf, strlen(buf));
      f->Status = FTP_QUIT;
    }
  }

  /* All slots are busy, no chance at this moment, try again later */

  result = AWORKING;                                     /* was SOFTERROR */

done:;
  if (HostPtr->Type == NOANONFTP) {
    (void) unlink(ResultPath);
  }

  ToLog(DBRPC, "DoAnFtp: result is %s\n", ATypeToString(result));
  return(result);
}

/*
 * main function, see description on top of file
 */

int InTransit(ResultPath)
char *ResultPath;
{
  int i;

             /*&& (FtpCache[i].Status == FTP_WORKING) but could be error ){ */
  for(i=0; i<MAXFTPSOPEN; i++) {
    if(streql(FtpCache[i].ResultPath, ResultPath)){
      ToLog(DBFTP, "InTransit: %s IS in transit %s\n", 
          ResultPath, NameOf(&FtpCache[i]));
      return(1);
    }
  }

  ToLog(DBFTP, "InTransit: %s IS NOT in transit\n", ResultPath);
  return(0);
}

