/*
 * PPP version 1.0.1
 * An implementation of the Point-to-Point Protocol (PPP) as specified by RFC 1134 (November 1989) for SPARCstation 1 running SunOS 4.0.3.
 * This uses functions from a slightly modified version of slcompress.c (Van Jacobson TCP/IP compression).
 */

#include "ppp.h"
#if NPPP > 0

/* timeout occurs if a Configure-Ack is not received within RESTART_TIMEOUT seconds after a Configure-Request is sent */

#define RESTART_TIMEOUT 3

#include <sys/types.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <sys/errno.h>
#include <sys/syslog.h>
#include <net/if.h>
#include <net/netisr.h>
#include <netinet/in.h>
#include <netinet/in_var.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include "slcompress.h"

/* layer indicates LCP or IPCP */

enum layer {LINK, NETWORK};

union conf_opt_data {
	struct {
		u_char codst_filler[3];
		u_char codst_char;                            /* configuration option character values are right justified in a long */
	} cod_st_char;
	struct {
		u_short codst_filler;
		u_short codst_short;                          /* configuration option short values are right justified in a long */
	} cod_st_short;
	u_long cod_long;
	u_char *cod_ptr;                                      /* if value exceeds 4 bytes than union conf_opt_data holds a pointer to value */ 
};

#define cod_char cod_st_char.codst_char
#define cod_short cod_st_short.codst_short

/* CO_SPECIAL indicates that special handling is required in configuration negotiation for this option and (*co_fun)() will be called.
 * CO_NOT_ACCEPTABLE indicates that only the default values is supported for this option. */

enum conf_opt_data_type {CO_ON_OFF, CO_TYPE_SHORT, CO_VALUE_SHORT, CO_VALUE_LONG, CO_SPECIAL, CO_NOT_ACCEPTABLE};

struct conf_opt {
	u_char co_type;
	u_char co_len;
	int co_data_type;
	union conf_opt_data *co_supported_datap;              /* supported values for this option in order by preference */
	union conf_opt_data co_default_data;                  /* default value for this option */
	int (*co_fun)();                                      /* used if this option requires special handling during negotiations */
};

/* struct conf_opt_for_initialization_only is used to initialize ppp_lcp_conf_opt and ppp_ipcp_conf_opt since Sun's C compiler cannot
 * initialize unions */

struct conf_opt_for_initialization_only {
	u_char co_type;
	u_char co_len;
	int co_data_type;
	u_long *co_supported_datap;
	u_long co_default_data;
	int (*co_fun)();
};

/* the first value in ppp_max_receive_unit will be initalized with the mtu of the lower layer during the open */

u_long ppp_max_receive_unit[] = {0, 1500, 0xffffffff};

enum lcp_conf_opt_types {CO_LCP_MAX_RECEIVE_UNIT = 1, CO_LCP_ASYNC_CTL_CHAR_MAP, CO_LCP_AUTHENTICATION, CO_LCP_ENCRYPTION, CO_LCP_MAGIC_NUM,
    CO_LCP_LINK_QUALITY, CO_LCP_PROTOCOL_FIELD_COMP, CO_LCP_ADDR_CTL_FIELD_COMP};

struct conf_opt_for_initialization_only ppp_lcp_conf_opt_for_initialization_only[] = {
	/* type                    len  data type          supported         def   fun */
        {0,                          0, 0,                 0,                0,    (int (*)())0},
	{CO_LCP_MAX_RECEIVE_UNIT,    4, CO_VALUE_SHORT,    ppp_max_receive_unit, 1500, 0},
	{CO_LCP_ASYNC_CTL_CHAR_MAP,  0, CO_NOT_ACCEPTABLE, 0,                0,    0},
	{CO_LCP_AUTHENTICATION,      0, CO_NOT_ACCEPTABLE, 0,                0,    0},
	{CO_LCP_ENCRYPTION,          0, CO_NOT_ACCEPTABLE, 0,                0,    0},
	{CO_LCP_MAGIC_NUM,           0, CO_NOT_ACCEPTABLE, 0,                0,    0},
	{CO_LCP_LINK_QUALITY,        0, CO_NOT_ACCEPTABLE, 0,                0,    0},
	{CO_LCP_PROTOCOL_FIELD_COMP, 0, CO_NOT_ACCEPTABLE, 0,                0,    0},
	{CO_LCP_ADDR_CTL_FIELD_COMP, 0, CO_NOT_ACCEPTABLE, 0,                0,    0},
	{0,                          0, 0,                 0,                0,    0}
};

struct conf_opt *ppp_lcp_conf_opt = (struct conf_opt *)ppp_lcp_conf_opt_for_initialization_only;

#define N_LCP_OPT (sizeof ppp_lcp_conf_opt_for_initialization_only / sizeof(struct conf_opt) - 2)

enum {IP = 0x0021, VJ_COMP_TCP_IP = 0x0037, IPCP = 0x8021, LCP = 0xc021};

/* ppp_ipcp_comp_types may be reinitialized by an ioctl() */

u_long ppp_ipcp_comp_types[] = {VJ_COMP_TCP_IP, 0, 0xffffffff};

enum ipcp_conf_opt_types {CO_IPCP_IP_ADDRESS = 1, CO_IPCP_COMPRESSION};

struct conf_opt_for_initialization_only ppp_ipcp_conf_opt_for_initialization_only[] = {
	{0,                   0, 0,                    0,               0, 0},
	{CO_IPCP_IP_ADDRESS,  0, CO_NOT_ACCEPTABLE,    0,               0, 0},
	{CO_IPCP_COMPRESSION, 4, CO_TYPE_SHORT,        ppp_ipcp_comp_types, 0, 0},
	{0,                   0, 0,                    0,               0, 0}
};

struct conf_opt *ppp_ipcp_conf_opt = (struct conf_opt *)ppp_ipcp_conf_opt_for_initialization_only;

#define N_IPCP_OPT (sizeof ppp_ipcp_conf_opt_for_initialization_only / sizeof(struct conf_opt) - 2)

enum co_fun_req {CO_FUN_SEND_CONFIGURE_REQUEST, CO_FUN_RECEIVE_CONFIGURE_REQUEST};

enum state {CLOSED = 0, LISTEN, REQ_SENT, ACK_RCVD, ACK_SENT, OPEN, CLOSING, MAX_STATE};

struct state_transition {
	int (*st_action)();
	enum state st_next_state;
};

int ppp_init(), ppp_ioctl(), ppp_input(), ppp_output(), ppp_open(), ppp_send_configure_request(), ppp_send_terminate_request(),
    ppp_send_terminate_ack(), ppp_send_code_reject(), ppp_send_protocol_reject(), ppp_send_echo_reply(), ppp_negotiate_local_configuration(),
    ppp_negotiate_remote_configuration(), ppp_verify_configure_ack(), ppp_negotiate_local_configuration_and_send_configure_request(), 
    ppp_negotiate_local_configuration_and_open_higher_layer(), ppp_verify_configure_ack_and_open_higher_layer(), ppp_open_higher_layer(),
    ppp_close_lower_layer(), ppp_timeout();

struct mbuf *ppp_build_xcp_frame();

/* The state transition table for both Link Control Protocol (LCP) and IP Control Protocol (IPCP) automaton. The IPCP state transition
 * table is a subset of the full table.  The table is indexed by event and state and specifies action and next state.  The state index
 * begins with 0 = CLOSED.  Note that this is different from RFC 1134 which begins with 1 = CLOSED.  The Receive-Configure-Request (RCR) event
 * is resolved into a Receive-Configure-Request-Good (RCR+) or a Receive-Configure-Request-Bad (RCR-) event by the action function.  The
 * calling sequence for the action functions is:
 *
 *     action(sc, layer, m)
 *     struct ppp_softc *sc;
 *     enum layer layer;
 *     struct mbuf *m;
 *
 * sc is a pointer to the struct ppp_softc for the corresponding interface.  layer specifies either LCP or IPCP.  m is a pointer to the struct
 * mbuf associated with the event and should be initialized to NULL if it is not applicable. */

struct state_transition ppp_lcp_state_trans[][MAX_STATE] = {
	{/* RUC (Receive-Unknown-Code) */
		/* CLOSED */ 	{ppp_send_code_reject, 						CLOSED},
		/* LISTEN */ 	{ppp_send_code_reject, 						CLOSED},
		/* REQ_SENT */ 	{ppp_send_code_reject, 						CLOSED},
		/* ACK_RCVD */ 	{ppp_send_code_reject, 						CLOSED},
		/* ACK_SENT */ 	{ppp_send_code_reject,						CLOSED},
		/* OPEN */ 	{ppp_send_code_reject,						CLOSED},
		/* CLOSING */ 	{ppp_send_code_reject, 						CLOSED}
	},
	{/* RCR (Receive-Configure-Request 1) */
		/* CLOSED */ 	{ppp_send_terminate_ack, 					0},
		/* LISTEN */ 	{ppp_negotiate_local_configuration_and_send_configure_request, 	0},
		/* REQ_SENT */ 	{ppp_negotiate_local_configuration, 				0},
		/* ACK_RCVD */ 	{ppp_negotiate_local_configuration_and_open_higher_layer, 	0},
		/* ACK_SENT */ 	{ppp_negotiate_local_configuration, 				0},
		/* OPEN */ 	{ppp_negotiate_local_configuration_and_send_configure_request, 	0},
		/* CLOSING */ 	{0, 								0}
	},
	{/* RCA (Receive-Configure-Ack 2) */
		/* CLOSED */ 	{ppp_send_terminate_ack, 					CLOSED},
		/* LISTEN */ 	{ppp_send_terminate_ack, 					LISTEN},
		/* REQ_SENT */ 	{ppp_verify_configure_ack, 					ACK_RCVD},
		/* ACK_RCVD */ 	{ppp_open, 							REQ_SENT},
		/* ACK_SENT */ 	{ppp_verify_configure_ack_and_open_higher_layer,		OPEN},
		/* OPEN */ 	{ppp_open, 							REQ_SENT},
		/* CLOSING */ 	{0, 								CLOSING}
	},
	{/* RCN (Receive-Configure-Nak 3) */
		/* CLOSED */ 	{ppp_send_terminate_ack, 					CLOSED},
		/* LISTEN */ 	{ppp_send_terminate_ack, 					LISTEN},
		/* REQ_SENT */ 	{ppp_negotiate_remote_configuration, 				REQ_SENT},
		/* ACK_RCVD */ 	{ppp_open, 							REQ_SENT},
		/* ACK_SENT */ 	{ppp_negotiate_remote_configuration, 				ACK_SENT},
		/* OPEN */ 	{ppp_open, 							REQ_SENT},
		/* CLOSING */ 	{0, 								CLOSING}
	},
	{/* RCN (Receive-Configure-Reject 4) */
		/* CLOSED */ 	{ppp_send_terminate_ack, 					CLOSED},
		/* LISTEN */ 	{ppp_send_terminate_ack, 					LISTEN},
		/* REQ_SENT */ 	{ppp_negotiate_remote_configuration, 				REQ_SENT},
		/* ACK_RCVD */ 	{ppp_open,			 				REQ_SENT},
		/* ACK_SENT */ 	{ppp_negotiate_remote_configuration, 				ACK_SENT},
		/* OPEN */ 	{ppp_open,			 				REQ_SENT},
		/* CLOSING */ 	{0, 								CLOSING}
	},
	{/* RTR (Receive-Terminate-Request 5) */
		/* CLOSED */ 	{ppp_send_terminate_ack, 					CLOSED},
		/* LISTEN */ 	{ppp_send_terminate_ack, 					LISTEN},
		/* REQ_SENT */ 	{ppp_send_terminate_ack, 					REQ_SENT},
		/* ACK_RCVD */ 	{ppp_send_terminate_ack,	 				REQ_SENT},
		/* ACK_SENT */ 	{ppp_send_terminate_ack,	 				REQ_SENT},
		/* OPEN */ 	{ppp_send_terminate_ack, 					LISTEN},
		/* CLOSING */ 	{ppp_send_terminate_ack, 					CLOSING}
	},
	{/* RTA (Receive-Terminate-Ack 6)  */
		/* CLOSED */ 	{0, 								CLOSED},
		/* LISTEN */ 	{0, 								LISTEN},
		/* REQ_SENT */ 	{0, 								REQ_SENT},
		/* ACK_RCVD */ 	{0, 								REQ_SENT},
		/* ACK_SENT */ 	{0, 								REQ_SENT},
		/* OPEN */ 	{0, 								CLOSED},
		/* CLOSING */ 	{ppp_close_lower_layer,						CLOSED}
	},
	{/* RCJ (Receive-Code-Reject 7)  */
		/* CLOSED */ 	{0,								CLOSED},
		/* LISTEN */ 	{0,	 							LISTEN},
		/* REQ_SENT */ 	{ppp_close_lower_layer,						CLOSED},
		/* ACK_RCVD */ 	{ppp_close_lower_layer,						CLOSED},
		/* ACK_SENT */ 	{ppp_close_lower_layer, 					CLOSED},
		/* OPEN */ 	{ppp_close_lower_layer,						CLOSED},
		/* CLOSING */ 	{ppp_close_lower_layer, 					CLOSED}
	},
	{/* (Receive-Protocol-Reject 8)  */
		/* CLOSED */ 	{0, 								CLOSED},
		/* LISTEN */ 	{0, 								LISTEN},
		/* REQ_SENT */ 	{0, 								REQ_SENT},
		/* ACK_RCVD */ 	{0, 								ACK_RCVD},
		/* ACK_SENT */ 	{0, 								ACK_SENT},
		/* OPEN */ 	{0, 								OPEN},
		/* CLOSING */ 	{0, 								CLOSING}
	},
	{/* RER (Receive-Echo-Request 9)  */
		/* CLOSED */ 	{ppp_send_terminate_ack,					CLOSED},
		/* LISTEN */ 	{ppp_send_terminate_ack, 					LISTEN},
		/* REQ_SENT */ 	{0, 								REQ_SENT},
		/* ACK_RCVD */ 	{0, 								ACK_RCVD},
		/* ACK_SENT */ 	{0, 								ACK_SENT},
		/* OPEN */ 	{ppp_send_echo_reply, 						OPEN},
		/* CLOSING */ 	{0, 								CLOSING}
	},
	{/* (Receive-Echo-Reply 10)  */
		/* CLOSED */ 	{0, 								CLOSED},
		/* LISTEN */ 	{0, 								LISTEN},
		/* REQ_SENT */ 	{0, 								REQ_SENT},
		/* ACK_RCVD */ 	{0, 								ACK_RCVD},
		/* ACK_SENT */ 	{0, 								ACK_SENT},
		/* OPEN */ 	{0, 								OPEN},
		/* CLOSING */ 	{0, 								CLOSING}
	},
	{/* (Receive-Discard-Request 11)  */
		/* CLOSED */ 	{0, 								CLOSED},
		/* LISTEN */ 	{0, 								LISTEN},
		/* REQ_SENT */ 	{0, 								REQ_SENT},
		/* ACK_RCVD */ 	{0, 								ACK_RCVD},
		/* ACK_SENT */ 	{0, 								ACK_SENT},
		/* OPEN */ 	{0, 								OPEN},
		/* CLOSING */ 	{0, 								CLOSING}
	},
	{/* RCR_GOOD (Receive-Configure-Request(Good)) */
		/* CLOSED */ 	{0, 								CLOSED},
		/* LISTEN */ 	{0, 								ACK_SENT},
		/* REQ_SENT */ 	{0, 								ACK_SENT},
		/* ACK_RCVD */ 	{0, 								OPEN},
		/* ACK_SENT */ 	{0, 								ACK_SENT},
		/* OPEN */ 	{0, 								ACK_SENT},
		/* CLOSING */ 	{0, 								CLOSING}
	},
	{/* RCR_BAD (Receive-Configure-Request(Bad)) */
		/* CLOSED */ 	{0, 								CLOSED},
		/* LISTEN */ 	{0, 								REQ_SENT},
		/* REQ_SENT */ 	{0, 								REQ_SENT},
		/* ACK_RCVD */ 	{0, 								ACK_RCVD},
		/* ACK_SENT */ 	{0, 								REQ_SENT},
		/* OPEN */ 	{0, 								REQ_SENT},
		/* CLOSING */ 	{0, 								CLOSING}
	},
	{/* AO (Active-Open) */
		/* CLOSED */ 	{ppp_open, 							REQ_SENT},
		/* LISTEN */ 	{ppp_open, 							REQ_SENT},
		/* REQ_SENT */ 	{0, 								REQ_SENT},
		/* ACK_RCVD */ 	{0, 								ACK_RCVD},
		/* ACK_SENT */ 	{0, 								ACK_SENT},
		/* OPEN */ 	{0, 								OPEN},
		/* CLOSING */ 	{ppp_open, 							REQ_SENT}
	},
	{/* C (Close) */
		/* CLOSED */ 	{0,	 							CLOSED},
		/* LISTEN */ 	{0, 								CLOSED},
		/* REQ_SENT */ 	{0, 								CLOSED},
		/* ACK_RCVD */ 	{0, 								CLOSED},
		/* ACK_SENT */ 	{ppp_send_terminate_request, 					CLOSING},
		/* OPEN */ 	{ppp_send_terminate_request, 					CLOSING},
		/* CLOSING */ 	{0, 								CLOSING}
	},
	{/* TO (Timeout) */
		/* CLOSED */ 	{0, 								CLOSED},
		/* LISTEN */ 	{0, 								LISTEN},
		/* REQ_SENT */ 	{ppp_send_configure_request, 					REQ_SENT},
		/* ACK_RCVD */ 	{ppp_send_configure_request, 					REQ_SENT},
		/* ACK_SENT */ 	{ppp_send_configure_request,					ACK_SENT},
		/* OPEN */ 	{0, 								OPEN},
		/* CLOSING */ 	{ppp_send_terminate_request, 					CLOSING}
	},
	{/* (Receive-Unknown-Protocol)  */
		/* CLOSED */ 	{0, 								CLOSED},
		/* LISTEN */ 	{0, 								LISTEN},
		/* REQ_SENT */ 	{0, 								REQ_SENT},
		/* ACK_RCVD */ 	{0, 								ACK_RCVD},
		/* ACK_SENT */ 	{0, 								ACK_SENT},
		/* OPEN */ 	{ppp_send_protocol_reject, 					OPEN},
		/* CLOSING */ 	{0, 								CLOSING}
	},
};

enum event {RUC = 0, CONF_REQ = 1, CONF_ACK, CONF_NAK, CONF_REJ, TERM_REQ, TERM_ACK, CODE_REJ, N_IPCP_CODE = CODE_REJ, PROTO_REJ, ECHO_REQ,
    ECHO_REPLY, DISCARD_REQ, N_LCP_CODE = DISCARD_REQ, RCR_GOOD = N_LCP_CODE + 1, RCR_BAD, AO, C, TO, RUP};

struct ppp_softc {
	struct ifnet sc_if;
	enum state sc_lcp_state;
	enum state sc_ipcp_state;                             /* IPCP state is valid only if LCP state = OPEN */
	enum event sc_event;                                  /* last event */
	u_char sc_id;
	u_char sc_scr_id;
	struct sc_conf_opt {
		union conf_opt_data sc_local_data;            /* current value for this option for the local interface */
		union conf_opt_data *sc_remote_datap;         /* pointer to current value for this option for the remote interface */
	} sc_lcp_conf_opt[N_LCP_OPT + 2];
	struct sc_conf_opt sc_ipcp_conf_opt[N_IPCP_OPT + 2];
	struct slcompress sc_vj_comp;
};

/* if there are more than 4 interfaces this initialization statement needs to be modified */

struct ppp_softc ppp_softc[NPPP] = {
	{{"ppp", 0, 1500, IFF_POINTOPOINT, 0, 0, 0, 0, {0}, ppp_init, ppp_output, ppp_ioctl, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, ppp_input}}
#if NPPP > 1
	, {{"ppp", 1, 1500, IFF_POINTOPOINT, 0, 0, 0, 0, {0}, ppp_init, ppp_output, ppp_ioctl, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, ppp_input}}
#endif
#if NPPP > 2
	, {{"ppp", 2, 1500, IFF_POINTOPOINT, 0, 0, 0, 0, {0}, ppp_init, ppp_output, ppp_ioctl, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, ppp_input}}
#endif
#if NPPP > 3
	, {{"ppp", 3, 1500, IFF_POINTOPOINT, 0, 0, 0, 0, {0}, ppp_init, ppp_output, ppp_ioctl, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, ppp_input}}
#endif
#if NPPP > 4
	, {ERROR}
#endif
};

struct frame_hdr {
	u_short fh_addr_ctl;
	u_short fh_proto;
};

struct packet_hdr {
	u_char ph_code;
	u_char ph_id;
	u_short ph_len;
};

struct xcp_frame {
	struct frame_hdr fr_frame_hdr;
	struct packet_hdr fr_packet_hdr;
};

#define fr_addr_ctl fr_frame_hdr.fh_addr_ctl
#define fr_proto fr_frame_hdr.fh_proto
#define fr_code fr_packet_hdr.ph_code
#define fr_id fr_packet_hdr.ph_id
#define fr_len fr_packet_hdr.ph_len

struct conf_opt_hdr {
	u_char ch_type;
	u_char ch_len;
};

extern int hz;

ppp_init(unit)
	int unit;
{
	if_attach(ppp_softc + unit);
}

/* Caution the following define may conflict with future SunOS releases.  It is compatible with SunOS 4.0.3 */

#define IFF_VJ_COMP_TCP 0x4000

/* ppp_ioctl() is called by socket ioctl and used to turn on/off Van Jacobson Compression and bring the interface up/down */

ppp_ioctl(ifp, cmd, data)
	struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	int s;

	switch (cmd) {
	    case SIOCSIFFLAGS:

		/* The next statement depends on the interrupt level of the hardware interface. */
		s = splzs();

		/* Use bit IFF_VJ_COMP_TCP of if_flags to specify Van Jacobson Compression */
		if (*(u_short *)((struct ifreq *)data)->ifr_data & IFF_VJ_COMP_TCP) {

			/* turn on Van Jacobson Compression */
			ppp_ipcp_comp_types[0] = VJ_COMP_TCP_IP;
			ppp_ipcp_comp_types[1] = 0;

		} else {

			/* turn off Van Jacobson Compression */
			ppp_ipcp_comp_types[0] = 0;
			ppp_ipcp_comp_types[1] = VJ_COMP_TCP_IP;
		}

		if (*(u_short *)((struct ifreq *)data)->ifr_data & IFF_UP) {

			/* bring the link up */
			((struct ppp_softc *)ifp)->sc_event = AO;
			if (((struct ppp_softc *)ifp)->sc_lcp_state != OPEN) { 
				if (ppp_lcp_state_trans[AO][((struct ppp_softc *)ifp)->sc_lcp_state].st_action)
					(*ppp_lcp_state_trans[AO][((struct ppp_softc *)ifp)->sc_lcp_state].st_action)(ifp, LINK, 0);
				((struct ppp_softc *)ifp)->sc_lcp_state = 
			            ppp_lcp_state_trans[AO][((struct ppp_softc *)ifp)->sc_lcp_state].st_next_state;
			} else {
				if (ppp_lcp_state_trans[AO][((struct ppp_softc *)ifp)->sc_ipcp_state].st_action)
					(*ppp_lcp_state_trans[AO][((struct ppp_softc *)ifp)->sc_ipcp_state].st_action)(ifp, NETWORK, 0);
				((struct ppp_softc *)ifp)->sc_ipcp_state = 
			            ppp_lcp_state_trans[AO][((struct ppp_softc *)ifp)->sc_ipcp_state].st_next_state;
			}

		} else {

			/* bring the link down */
			((struct ppp_softc *)ifp)->sc_event = C;
			if (((struct ppp_softc *)ifp)->sc_lcp_state == OPEN) { 
				if (ppp_lcp_state_trans[C][((struct ppp_softc *)ifp)->sc_ipcp_state].st_action)
					(*ppp_lcp_state_trans[C][((struct ppp_softc *)ifp)->sc_ipcp_state].st_action)(ifp, NETWORK, 0);
				((struct ppp_softc *)ifp)->sc_ipcp_state = 
			            ppp_lcp_state_trans[C][((struct ppp_softc *)ifp)->sc_ipcp_state].st_next_state;
			} else {
				if (ppp_lcp_state_trans[C][((struct ppp_softc *)ifp)->sc_lcp_state].st_action)
					(*ppp_lcp_state_trans[C][((struct ppp_softc *)ifp)->sc_lcp_state].st_action)(ifp, LINK, 0);
				((struct ppp_softc *)ifp)->sc_lcp_state = 
			            ppp_lcp_state_trans[C][((struct ppp_softc *)ifp)->sc_lcp_state].st_next_state;
			}
		}
		splx(s);
		return(0);
	    case SIOCSIFDSTADDR:
	    case SIOCSIFADDR:
		return(0);
	    default:
		return(EINVAL);
	}
}

/* ppp_open() starts the configuration negotiations from the beginning */

ppp_open(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	struct sc_conf_opt *sco;
	struct conf_opt *co;

	if (layer == LINK) {
		sco = sc->sc_lcp_conf_opt + 1;
		co = ppp_lcp_conf_opt + 1;
		ppp_lcp_conf_opt[CO_LCP_MAX_RECEIVE_UNIT].co_supported_datap[0].cod_short = sc->sc_if.if_lower->if_mtu - 4;
	} else {
		sco = sc->sc_ipcp_conf_opt + 1;
		co = ppp_ipcp_conf_opt + 1;
	}

	/* for each option reset the pointer to option value for the remote interface to the start of the list of supported values */
	for (; co->co_type; ++sco, ++co) {
		sco->sc_remote_datap = co->co_supported_datap;
	}
	if (m) {
		m_freem(m);
	}
	ppp_send_configure_request(sc, layer, 0);
}

ppp_send_configure_request(sc, layer, m0)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m0;
{
	struct sc_conf_opt *sco;
	struct conf_opt *co;
	struct mbuf *req_m, *req_m0;
	struct xcp_frame *req_fr;
	struct conf_opt_hdr *req_ch;

	if (sc->sc_if.if_lower && sc->sc_if.if_lower->if_output) {
		if ((req_m0 = req_m = ppp_build_xcp_frame(sc, layer)) == 0) {
			return;
		}
		if (layer == LINK) {
			sco = sc->sc_lcp_conf_opt + 1;
			co = ppp_lcp_conf_opt + 1;
		} else {
			sco = sc->sc_ipcp_conf_opt + 1;
			co = ppp_ipcp_conf_opt + 1;
		}

		/* fili in header */
		req_fr = mtod(req_m0, struct xcp_frame *);
		req_fr->fr_code = CONF_REQ;
		sc->sc_scr_id = req_fr->fr_id = ++sc->sc_id;

		req_ch = (struct conf_opt_hdr *)(req_fr + 1);
		for (; co->co_type; ++sco, ++co) {

			/* send only options whose values differ from the default */
			if (co->co_data_type != CO_NOT_ACCEPTABLE && sco->sc_remote_datap->cod_long != co->co_default_data.cod_long) {

				/* make sure option will completely fit in a mbuf; allocate another mbuf if neccessary*/
				if (co->co_len > MMAXOFF - (req_m->m_off + req_m->m_len)) {
					struct mbuf *m;
					MGET(m, M_DONTWAIT, MT_DATA);
					if (m == 0) {
						log(LOG_ERR, "ppp: no mbuf is available");
						return;
					}
					req_m->m_next = m;
					req_m = m;
					req_m->m_off = MMINOFF;
					req_m->m_len = 0;
					req_ch = mtod(req_m, struct conf_opt_hdr *);
				}

				req_ch->ch_type = co->co_type;
				req_ch->ch_len = co->co_len;
				switch (co->co_data_type) {
				    case CO_ON_OFF:
					break;

				    case CO_TYPE_SHORT:
				    case CO_VALUE_SHORT:

					/* bcopy neccessary since data may not be properly aligned in packets */
					bcopy((u_char *)&sco->sc_remote_datap->cod_short, (u_char *)(req_ch + 1), 2);
					break;

				    case CO_VALUE_LONG:
					bcopy((u_char *)&sco->sc_remote_datap->cod_long, (u_char *)(req_ch + 1), 4);
					break;

				    case CO_SPECIAL:
					(*co->co_fun)(CO_FUN_SEND_CONFIGURE_REQUEST, req_ch, sco->sc_remote_datap);
					break;
				}
				req_m->m_len += req_ch->ch_len;
				req_fr->fr_len += req_ch->ch_len;
				req_ch = (struct conf_opt_hdr *)((u_char *)req_ch + req_ch->ch_len);
			}
		}
		(*sc->sc_if.if_lower->if_output)(sc->sc_if.if_lower, req_m0, 0);
		++sc->sc_if.if_opackets;
	}
	sc->sc_if.if_flags &= ~(IFF_RUNNING | IFF_UP);

	/* start LCP/IPCP restart timer */
	untimeout(ppp_timeout, sc);
	timeout(ppp_timeout, sc, RESTART_TIMEOUT * hz);
}

ppp_send_terminate_request(sc, layer, m0)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m0;
{
	struct mbuf *m;
	struct xcp_frame *fr;

	/* build packet then send it */
	if ((m = ppp_build_xcp_frame(sc, layer)) == 0) {
		return;
	}
	fr = mtod(m, struct xcp_frame *);
	fr->fr_code = TERM_REQ;
	fr->fr_id = ++sc->sc_id;
	(*sc->sc_if.if_lower->if_output)(sc->sc_if.if_lower, m, 0);
	++sc->sc_if.if_opackets;

	/* bring link down */
	sc->sc_if.if_flags &= ~(IFF_RUNNING | IFF_UP);

	/* start LCP/IPCP restart timer */
	untimeout(ppp_timeout, sc);
	timeout(ppp_timeout, sc, RESTART_TIMEOUT * hz);
}

/* reserve space in mbuf for use by lower layer */

#define MRESERVED 16

struct mbuf *ppp_build_xcp_frame(sc, layer)
	struct ppp_softc *sc;
	enum layer layer;
{
	struct mbuf *m;
	struct xcp_frame *fr;

	MGET(m, M_DONTWAIT, MT_DATA);
	if (m == 0) {
		log(LOG_ERR, "ppp: no mbuf is available");
		return(0);
	}
	m->m_off = MMINOFF + MRESERVED;
	m->m_len = sizeof(struct xcp_frame);
	fr = mtod(m, struct xcp_frame *);
	fr->fr_addr_ctl = 0xff03;
	if (layer == LINK)
		fr->fr_proto = LCP;
	else
		fr->fr_proto = IPCP;
	fr->fr_len = sizeof(struct packet_hdr);
	return(m);
}

ppp_send_terminate_ack(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	struct xcp_frame *fr;
	u_char id;

	/* reuse input packet */
	id = mtod(m, struct xcp_frame *)->fr_id;
	m->m_off = MMAXOFF -  sizeof(struct xcp_frame);
	m->m_len = sizeof(struct xcp_frame);
	if (m->m_next)
		m_freem(m->m_next);
	fr = mtod(m, struct xcp_frame *);
	fr->fr_addr_ctl = 0xff03;
	if (layer == LINK)
		fr->fr_proto = LCP;
	else
		fr->fr_proto = IPCP;
	fr->fr_code = TERM_ACK;
	fr->fr_id = id;
	fr->fr_len = sizeof(struct packet_hdr);
	sc->sc_if.if_flags &= ~(IFF_RUNNING | IFF_UP);
	(*sc->sc_if.if_lower->if_output)(sc->sc_if.if_lower, m, 0);
	++sc->sc_if.if_opackets;
}

/* ppp_send_code_reject() is called when a LCP/IPCP packet with an unknown code is received.  A Code-Reject is sent and the interface is
 * brought down */

ppp_send_code_reject(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	static struct xcp_frame xcp_frame = {{0xff03}, {CODE_REJ}};

	/* modify and return input packet as a Code-Reject packet */
	bcopy((u_char *)&mtod(m, struct xcp_frame *)->fr_len, (u_char *)&xcp_frame.fr_len, sizeof(u_short));
	m_adj(m, sizeof (struct frame_hdr));

	/* allocate space for frame header */
	if (!M_HASCL(m) && m->m_off - MMINOFF >= sizeof(struct xcp_frame)) {
                m->m_off -= sizeof(struct xcp_frame);
		m->m_len += sizeof(struct xcp_frame);
	} else {
		struct mbuf *m0;
		MGET(m0, M_DONTWAIT, MT_DATA);
		if (m0 == 0) {
			log(LOG_ERR, "ppp: no mbuf is available");
			return;
		}
		m0->m_next = m;
		m = m0;
		m->m_off = MMAXOFF - sizeof(struct xcp_frame);
		m->m_len = sizeof(struct xcp_frame);
	}

	/* fill in frame header */
	if (layer == LINK)
		xcp_frame.fr_proto = LCP;
	else
		xcp_frame.fr_proto = IPCP;
	xcp_frame.fr_id = ++sc->sc_id;
	xcp_frame.fr_len += sizeof(struct packet_hdr);
	if (xcp_frame.fr_len > sc->sc_lcp_conf_opt[CO_LCP_MAX_RECEIVE_UNIT].sc_local_data.cod_short) 
		xcp_frame.fr_len = ppp_truncate_to_mru(sc, m);
	bcopy((u_char *)&xcp_frame, mtod(m, u_char *), sizeof(struct xcp_frame));

	/* take down interface */
	sc->sc_if.if_flags &= ~(IFF_RUNNING | IFF_UP);

	/* send packet */
	(*sc->sc_if.if_lower->if_output)(sc->sc_if.if_lower, m, 0);
	++sc->sc_if.if_opackets;
	log(LOG_ERR, "ppp%d: received unknown %s code", sc->sc_if.if_unit, (layer == LINK) ? "LCP" : "IPCP");
}

/* ppp_send_echo_reply() is called when an Echo-Request packet is recived form the remote interface */

ppp_send_echo_reply(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	u_short len;

	/* modify and return input packet as a Echo-Reply packet */
	bcopy((u_char *)&mtod(m, struct xcp_frame *)->fr_len, (u_char *)&len, sizeof(u_short));
	mtod(m, struct xcp_frame *)->fr_code = ECHO_REPLY;
	if (len > sc->sc_lcp_conf_opt[CO_LCP_MAX_RECEIVE_UNIT].sc_local_data.cod_short) {
		len = ppp_truncate_to_mru(sc, m);
		bcopy((u_char *)&len, (u_char *)&mtod(m, struct xcp_frame *)->fr_len, sizeof(u_short));
	}
	(*sc->sc_if.if_lower->if_output)(sc->sc_if.if_lower, m, 0);
	++sc->sc_if.if_opackets;
}

/* ppp_send_protocol_reject() is called when a packet witn unknown protocol is received;  a Protocol-Reject packet is sent */

ppp_send_protocol_reject(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	static struct {
		struct xcp_frame xcp_frame;
		u_short rej_proto;
	} header = {{{0xff03, LCP}, {PROTO_REJ}}};

	/* modify and return input packet as a Protocol-Reject packet */
	bcopy((u_char *)&mtod(m, struct frame_hdr *)->fh_proto, (u_char *)&header.rej_proto, sizeof(u_short));

	/* allocate space for frame header */
	m_adj(m, sizeof (struct frame_hdr));
	if (!M_HASCL(m) && m->m_off - MMINOFF >= sizeof(struct xcp_frame) + sizeof(u_short)) {
		m->m_off -= sizeof(struct xcp_frame) + sizeof(u_short);
		m->m_len += sizeof(struct xcp_frame) + sizeof(u_short);
	} else {
		struct mbuf *m0;
		MGET(m0, M_DONTWAIT, MT_DATA);
		if (m0 == 0) {
			log(LOG_ERR, "ppp: no mbuf is available");
			return;
		}
		m0->m_next = m;
		m = m0;
		m->m_off = MMAXOFF - (sizeof(struct xcp_frame) + sizeof(u_short));
		m->m_len = sizeof(struct xcp_frame) + sizeof(u_short);
	}

	/* fill in frame header and send packet */
	header.xcp_frame.fr_len = ppp_truncate_to_mru(sc, m);
	header.xcp_frame.fr_id = ++sc->sc_id;
	bcopy((u_char *)&header, mtod(m, u_char *), sizeof(header));
	(*sc->sc_if.if_lower->if_output)(sc->sc_if.if_lower, m, 0);
	++sc->sc_if.if_opackets;
}

ppp_truncate_to_mru(sc, m)
	struct ppp_softc *sc;
	struct mbuf *m;
{
	int len = -sizeof(struct frame_hdr);

	for (; m; m = m->m_next) {
		len += m->m_len;
		if (len > sc->sc_lcp_conf_opt[CO_LCP_MAX_RECEIVE_UNIT].sc_local_data.cod_short) {
			if (m->m_next)
				m_freem(m->m_next);
			m->m_len -= len - sc->sc_lcp_conf_opt[CO_LCP_MAX_RECEIVE_UNIT].sc_local_data.cod_short;			
			break;
		}
	}
	return(len);
}

/* ppp_negotiate_local_configuration is called when a Configure-Request packet is received;  the options are examined and a Configure-Ack,
 * Configure-Nak or Configure-Reject packet is returned */

ppp_negotiate_local_configuration(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	struct mbuf *ack_m, *ack_m0;
	struct mbuf *nak_m, *nak_m0;
	struct mbuf *rej_m, *rej_m0;
	struct xcp_frame *fr;
	struct xcp_frame *nak_fr;
	struct xcp_frame *rej_fr;
	struct conf_opt_hdr *ch;
	struct conf_opt_hdr *nak_ch;
	struct conf_opt_hdr *rej_ch;
	struct sc_conf_opt *sco0, *sco;
	struct conf_opt *co0, *co;
	int max_type;
	u_short len;
	union conf_opt_data *datap;

	if (layer == LINK) {
		sco0 = sc->sc_lcp_conf_opt;
		co0 = ppp_lcp_conf_opt;
		max_type = N_LCP_OPT;
	} else {
		sco0 = sc->sc_ipcp_conf_opt;
		co0 = ppp_ipcp_conf_opt;
		max_type = N_IPCP_OPT;
	}

	/* reset all local options to their default values */
	for (sco = sco0 + 1, co = co0 + 1; co->co_type; ++sco, ++co) {
		sco->sc_local_data = co->co_default_data;
	}

	/* save a copy of the original packet to use as a Configure-Ack packet if everything is ok */
	ack_m0 = ack_m = m_copy(m, 0, M_COPYALL);
	fr = mtod(ack_m0, struct xcp_frame *);
	fr->fr_code = CONF_ACK;
	bcopy(&fr->fr_len, &len, 2);
	len -= sizeof(struct packet_hdr);
	m_adj(m, sizeof(struct xcp_frame));

	/* begin a Configure-Nak packet in case we need one */
	if ((nak_m0 = nak_m = ppp_build_xcp_frame(sc, layer)) == 0) {
		return;
	}
	nak_fr = mtod(nak_m0, struct xcp_frame *);
	nak_fr->fr_code = CONF_NAK;
	nak_fr->fr_id = fr->fr_id;
	nak_ch = (struct conf_opt_hdr *)(nak_fr + 1);

	/* begin a Configure-Reject packet in case we need one */
	if ((rej_m0 = rej_m = ppp_build_xcp_frame(sc, layer)) == 0) {
		return;
	}
	rej_fr = mtod(rej_m0, struct xcp_frame *);
	rej_fr->fr_code = CONF_REJ;
	rej_fr->fr_id = fr->fr_id;
	rej_ch = (struct conf_opt_hdr *)(rej_fr + 1);

	while (len) {

		/* make sure the complete option is entirely inside a single mbuf */
		if (m->m_len < sizeof(struct conf_opt_hdr)) {
			if ((m = m_pullup(m, sizeof(struct conf_opt_hdr))) == 0) {
				return;
			}
		}
		ch = mtod(m, struct conf_opt_hdr *);
		if (m->m_len < ch->ch_len) {
			if ((m = m_pullup(m, ch->ch_len)) == 0) {
				return;
			}
			ch = mtod(m, struct conf_opt_hdr *);
		}

		if (ch->ch_type > max_type) {

			/* option type is invalid */
			if (ch->ch_len > MMAXOFF - (rej_m->m_off + rej_m->m_len)) {
				struct mbuf *m;
				MGET(m, M_DONTWAIT, MT_DATA);
				if (m == 0) {
					log(LOG_ERR, "ppp: no mbuf is available");
					return;
				}
				rej_m->m_next = m;
				rej_m = m;
				rej_m->m_off = MMINOFF;
				rej_m->m_len = 0;
				rej_ch = mtod(rej_m, struct conf_opt_hdr *);
			}
			bcopy(ch, rej_ch, ch->ch_len);
			rej_fr->fr_len += ch->ch_len;
			rej_m->m_len += ch->ch_len;
			rej_ch = (struct conf_opt_hdr *)((u_char *)rej_ch + rej_ch->ch_len);

		} else {

			/* option type is valid */
			co = co0 + ch->ch_type;
			sco = sco0 + ch->ch_type;
			switch(co->co_data_type) {
			    case CO_ON_OFF:
				sco->sc_local_data.cod_long = 1;
				break;

			    case CO_TYPE_SHORT:

				/* search for value in list of supported values for this option */
				for (datap = co->co_supported_datap; datap->cod_long != 0xffffffff; ++datap) {
					if (!bcmp((u_char *)(ch + 1), (u_char *)&datap->cod_short, 2))
						break;
				}
				if (datap->cod_long != 0xffffffff) {

					/* value supported for this option */
					sco->sc_local_data = *datap;

				} else {

					/* value not supported for this option */
					if (ch->ch_len > MMAXOFF - (nak_m->m_off + nak_m->m_len)) {
						struct mbuf *m;
						MGET(m, M_DONTWAIT, MT_DATA);
						if (m == 0) {
							log(LOG_ERR, "ppp: no mbuf is available");
							return;
						}
						nak_m->m_next = m;
						nak_m = m;
						nak_m->m_off = MMINOFF;
						nak_m->m_len = 0;
						nak_ch = mtod(nak_m, struct conf_opt_hdr *);
					}
					*nak_ch = *ch;
					*(u_char *)(nak_ch + 1) = *(u_char *)&co->co_supported_datap->cod_short;
					*((u_char *)(nak_ch + 1) + 1) = *((u_char *)&co->co_supported_datap->cod_short + 1);
					nak_fr->fr_len += sizeof(struct conf_opt_hdr) + 2;
					nak_m->m_len += ch->ch_len;
					nak_ch = (struct conf_opt_hdr *)((u_char *)nak_ch + sizeof(struct conf_opt_hdr) + 2);
				}
				break;

			    case CO_VALUE_SHORT:
				sco->sc_local_data.cod_long = 0;
				*(u_char *)&sco->sc_local_data.cod_short = *(u_char *)(ch + 1);
				*((u_char *)&sco->sc_local_data.cod_short + 1) = *((u_char *)(ch + 1) + 1);
				break;

			    case CO_VALUE_LONG:

				/* bcopy neccessary since data may not be properly aligned in packets */
				bcopy((u_char *)(ch + 1), (u_char *)&sco->sc_local_data.cod_long, 4);
				break;

			    case CO_SPECIAL:
				(*co->co_fun)(CO_FUN_RECEIVE_CONFIGURE_REQUEST, ch, &sco->sc_local_data);
				break;

			    case CO_NOT_ACCEPTABLE:
			    default:
				if (ch->ch_len > MMAXOFF - (rej_m->m_off + rej_m->m_len)) {
					struct mbuf *m;
					MGET(m, M_DONTWAIT, MT_DATA);
					if (m == 0) {
						log(LOG_ERR, "ppp: no mbuf is available");
						return;
					}
					rej_m->m_next = m;
					rej_m = m;
					rej_m->m_off = MMINOFF;
					rej_m->m_len = 0;
					rej_ch = mtod(rej_m, struct conf_opt_hdr *);
				}
				bcopy(ch, rej_ch, ch->ch_len);
				rej_fr->fr_len += ch->ch_len;
				rej_m->m_len += ch->ch_len;
				rej_ch = (struct conf_opt_hdr *)((u_char *)rej_ch + rej_ch->ch_len);
			}
		}
		len -= ch->ch_len;
		m_adj(m, ch->ch_len);
	}
	m_freem(m);
	if (rej_fr->fr_len > sizeof(struct packet_hdr)) {

		/* there is at least one not recognizable option so send Configure-Reject packet */
		sc->sc_event = RCR_BAD;
		m_freem(ack_m0);
		m_freem(nak_m0);
		m = rej_m0;

	} else if (nak_fr->fr_len > sizeof(struct packet_hdr)) {

		/* there is at least one not acceptable option so send Configure-Nak packet */
		sc->sc_event = RCR_BAD;
		m_freem(ack_m0);
		m_freem(rej_m0);
		m = nak_m0;

	} else {

		/* everything is ok so send a Configure-Ack packet */
		sc->sc_event = RCR_GOOD;
		m_freem(nak_m0);
		m_freem(rej_m0);
		m = ack_m0;
	}
	(*sc->sc_if.if_lower->if_output)(sc->sc_if.if_lower, m, 0);
	++sc->sc_if.if_opackets;
}

/* ppp_negotiate_remote_configuration() is called when the remote interface has rejected our Configure-Request and sent either a Configure-Nak
 * or a Configure-Reject packet;  the list of supported option values is ordered by preference;  we ignore the recommended value from the
 * remote interface and select the next value in our list */

ppp_negotiate_remote_configuration(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	struct xcp_frame *fr;
	struct conf_opt_hdr *ch;
	struct sc_conf_opt *sco0, *sco;
	struct conf_opt *co0, *co;
	int max_type;
	u_short len;
	union conf_opt_data *datap;
	u_char code;

	if (layer == LINK) {
		sco0 = sc->sc_lcp_conf_opt;
		co0 = ppp_lcp_conf_opt;
		max_type = N_LCP_OPT;
	} else {
		sco0 = sc->sc_ipcp_conf_opt;
		co0 = ppp_ipcp_conf_opt;
		max_type = N_IPCP_OPT;
	}
	fr = mtod(m, struct xcp_frame *);

	/* if identifier field does not match then discard packet */
	if (fr->fr_id != sc->sc_scr_id) {
		m_freem(m);
		return;
	} else {
		untimeout(ppp_timeout, sc);
	}

	code = fr->fr_code;
	len = fr->fr_len - sizeof(struct packet_hdr);
	m_adj(m, sizeof(struct xcp_frame));
	while (len) {

		/* make sure option fits completely inside a single mbuf */
		if (m->m_len < sizeof(struct conf_opt_hdr)) {
			if ((m = m_pullup(m, sizeof(struct conf_opt_hdr))) == 0) {
				return;
			}
		}
		ch = mtod(m, struct conf_opt_hdr *);
		if (m->m_len < ch->ch_len) {
			if ((m = m_pullup(m, ch->ch_len)) == 0) {
				return;
			}
			ch = mtod(m, struct conf_opt_hdr *);
		}

		if (ch->ch_type <= max_type) {
			co = co0 + ch->ch_type;
			sco = sco0 + ch->ch_type;
			if (co->co_data_type != CO_NOT_ACCEPTABLE) {
				if (code != CONF_REJ && sco->sc_remote_datap->cod_long != co->co_default_data.cod_long) {

					/* advance to next value on preference list */
					++sco->sc_remote_datap;

				} else {

					/* default value will always be accepted */
					sco->sc_remote_datap = &co->co_default_data;
 				}
			}
		}
		m_adj(m, ch->ch_len);
		len -= ch->ch_len;
	}
	m_freem(m);

	/* send out a Configure-Request packet with new option values to remote interface */
	ppp_send_configure_request(sc, layer, 0);
}

ppp_verify_configure_ack(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	struct xcp_frame *fr;

	/* If identifier field does not match then discard packet */
	fr = mtod(m, struct xcp_frame *);
	if (fr->fr_id != sc->sc_scr_id) {
		sc->sc_event = DISCARD_REQ;
	} else {
		untimeout(ppp_timeout, sc);
	}
	m_freem(m);
}

ppp_negotiate_local_configuration_and_send_configure_request(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	ppp_negotiate_local_configuration(sc, layer, m);
	ppp_open(sc, layer, 0);
}

ppp_negotiate_local_configuration_and_open_higher_layer(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	ppp_negotiate_local_configuration(sc, layer, m);
	if (sc->sc_event == RCR_GOOD) {
		ppp_open_higher_layer(sc, layer, 0);
	}
}

ppp_verify_configure_ack_and_open_higher_layer(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	ppp_verify_configure_ack(sc, layer, m);
	if (sc->sc_event == CONF_ACK) {
		ppp_open_higher_layer(sc, layer, 0);
	}
}

ppp_open_higher_layer(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	if (layer == LINK) {
		ppp_open(sc, NETWORK, m);
		sc->sc_ipcp_state = REQ_SENT;
	} else {
		sc->sc_if.if_mtu = sc->sc_lcp_conf_opt[CO_LCP_MAX_RECEIVE_UNIT].sc_local_data.cod_short;

		/* always be capable of receiving Van Jacobson Compressed TCP/IP packets */
		sl_compress_init(&sc->sc_vj_comp);

		/* but only transmit Van Jacobson Compressed TCP/IP packets if the remote interface has negotiated for it */
		if (sc->sc_ipcp_conf_opt[CO_IPCP_COMPRESSION].sc_local_data.cod_short == VJ_COMP_TCP_IP) {
			sc->sc_if.if_flags |= IFF_VJ_COMP_TCP;
		}
		sc->sc_if.if_flags |= IFF_RUNNING | IFF_UP;
	}
}

ppp_close_lower_layer(sc, layer, m)
	struct ppp_softc *sc;
	enum layer layer;
	struct mbuf *m;
{
	if (layer == NETWORK) {
		ppp_send_terminate_request(sc, LINK);
		sc->sc_lcp_state = CLOSING;
	}
		
}

/* ppp_timeout() is called when an Ack is not received from the remote interface within RESTART_TIMEOUT seconds after a Configure-Request
 * or Terminate-Request is sent;  the request is re-sent */

ppp_timeout(sc)
	struct ppp_softc *sc;
{
	int s;

	/* The next statement depends on the interrupt level of the hardware interface. */
	s = splzs();

	sc->sc_event = TO;
	if (sc->sc_lcp_state != OPEN) {
		if (ppp_lcp_state_trans[TO][sc->sc_lcp_state].st_action) {
			(*ppp_lcp_state_trans[TO][sc->sc_lcp_state].st_action)(sc, LINK, 0);
		}
		sc->sc_lcp_state = ppp_lcp_state_trans[TO][sc->sc_lcp_state].st_next_state;
	} else {
		if (ppp_lcp_state_trans[TO][sc->sc_ipcp_state].st_action) {
			(*ppp_lcp_state_trans[TO][sc->sc_ipcp_state].st_action)(sc, NETWORK, 0);
		}
		sc->sc_ipcp_state = ppp_lcp_state_trans[TO][sc->sc_ipcp_state].st_next_state;
	}
	splx(s);
}

/* ppp_input() is called from the lower layer everytime an input packet is received form the remote interface */

ppp_input(ifp, m)
	struct ifnet *ifp;
	struct mbuf *m;
{
	struct xcp_frame *fr;
	struct frame_hdr *fh;
	enum state prev_ipcp_state;

	++ifp->if_ipackets;
	if (m->m_len < sizeof(struct frame_hdr)) {
		if ((m = m_pullup(m, sizeof(struct frame_hdr))) == 0) {
			return;
		}
	}
	fh = mtod(m, struct frame_hdr *);
	switch (*(u_char *)(&fh->fh_proto) << 8 | *((u_char *)(&fh->fh_proto) + 1)) {

	    /* always capable of receiving Van Jacobson Compressed TCP/IP */
	    case VJ_COMP_TCP_IP: 
		if ((((struct ppp_softc *)ifp)->sc_if.if_flags & (IFF_RUNNING | IFF_UP)) != (IFF_RUNNING | IFF_UP)) {
			m_freem(m);
			return;
		} else {
			int c;
			m_adj(m, sizeof(struct frame_hdr));
			c = *mtod(m, u_char *) & 0xf0;
			if (c != IPVERSION << 4) {
				if (c & 0x80)
					c = TYPE_COMPRESSED_TCP;
				else if (c == TYPE_UNCOMPRESSED_TCP)
					*mtod(m, u_char *) &= 0x4f;
				if (sl_uncompress_tcp(&m, c, &((struct ppp_softc *)ifp)->sc_vj_comp)) {
					m_freem(m);
					return;
				}
			}
		}

		/* put back frame header since we will fall through to next case and it is expecting a frame header */
		m->m_off -= sizeof(struct frame_hdr);
		m->m_len += sizeof(struct frame_hdr);
		/* fall through and process as an IP packet */

	    case IP:
		if ((((struct ppp_softc *)ifp)->sc_if.if_flags & (IFF_RUNNING | IFF_UP)) != (IFF_RUNNING | IFF_UP)) {
			m_freem(m);
		} else {
			int s = splimp();
			if (IF_QFULL(&ipintrq)) {
				IF_DROP(&ipintrq);
				m_freem(m);
			} else {

				/* allocate space for struct ifnet * */
				m_adj(m, sizeof(struct frame_hdr));
				if (!M_HASCL(m) && m->m_off - MMINOFF >= sizeof(struct ifnet *)) {
					m->m_off -= sizeof(struct ifnet *);
					m->m_len += sizeof(struct ifnet *);
				} else {
					struct mbuf *m0;
					MGET(m0, M_DONTWAIT, MT_DATA);
					if (m0 == 0) {
						log(LOG_ERR, "ppp: no mbuf is available");
						return;
					}
					m0->m_next = m;
					m = m0;
					m->m_off = MMAXOFF - sizeof(struct ifnet *);
					m->m_len = sizeof(struct ifnet *);
				}

				*mtod(m, struct ifnet **) = ifp;	
				IF_ENQUEUE(&ipintrq, m);
				softcall(NETISR_IP, (caddr_t)0);
			}
			splx(s);
		}
		return;

	    case LCP:
		if (m->m_len < sizeof(struct xcp_frame)) {
			if ((m = m_pullup(m, sizeof(struct xcp_frame))) == 0) {
				return;
			}
		}
		fr = mtod(m, struct xcp_frame *);
		if (fr->fr_code > N_LCP_CODE) {
			((struct ppp_softc *)ifp)->sc_event = RUC;
		} else {
			((struct ppp_softc *)ifp)->sc_event = fr->fr_code;
		}

		/* do action */
		if (ppp_lcp_state_trans[((struct ppp_softc *)ifp)->sc_event][((struct ppp_softc *)ifp)->sc_lcp_state].st_action) {
			(*ppp_lcp_state_trans[((struct ppp_softc *)ifp)->sc_event][((struct ppp_softc *)ifp)->sc_lcp_state].st_action)
			    (ifp, LINK, m);
		} else {
			m_freem(m);
		}

		/* set next state */
		if (((struct ppp_softc *)ifp)->sc_lcp_state == OPEN) {
			prev_ipcp_state = ((struct ppp_softc *)ifp)->sc_ipcp_state;
		} else {
			prev_ipcp_state = 0;
		}
		((struct ppp_softc *)ifp)->sc_lcp_state =
		    ppp_lcp_state_trans[((struct ppp_softc *)ifp)->sc_event][((struct ppp_softc *)ifp)->sc_lcp_state].st_next_state;
		if (prev_ipcp_state == OPEN && ((struct ppp_softc *)ifp)->sc_lcp_state != OPEN) {
			log(LOG_ERR, "ppp%d: down", ((struct ppp_softc *)ifp)->sc_if.if_unit);
		}
		return;

	    case IPCP:
		if (m->m_len < sizeof(struct xcp_frame)) {
			if ((m = m_pullup(m, sizeof(struct xcp_frame))) == 0) {
				return;
			}
		}
		fr = mtod(m, struct xcp_frame *);
		if (((struct ppp_softc *)ifp)->sc_lcp_state == OPEN) {
			if (fr->fr_code > N_IPCP_CODE) {
				((struct ppp_softc *)ifp)->sc_event = RUC;
			} else {
				((struct ppp_softc *)ifp)->sc_event = fr->fr_code;
			}

			/* do action */
			if (ppp_lcp_state_trans[((struct ppp_softc *)ifp)->sc_event][((struct ppp_softc *)ifp)->sc_ipcp_state].st_action) {
				(*ppp_lcp_state_trans[((struct ppp_softc *)ifp)->sc_event][((struct ppp_softc *)ifp)->sc_ipcp_state]
				    .st_action) (ifp, NETWORK, m);
			} else {
				m_freem(m);
			}

			/* set next state */
			prev_ipcp_state = ((struct ppp_softc *)ifp)->sc_ipcp_state;
			((struct ppp_softc *)ifp)->sc_ipcp_state =
			    ppp_lcp_state_trans[((struct ppp_softc *)ifp)->sc_event][((struct ppp_softc *)ifp)->sc_ipcp_state].st_next_state;
			if (prev_ipcp_state == OPEN && ((struct ppp_softc *)ifp)->sc_ipcp_state != OPEN) {
				log(LOG_ERR, "ppp%d: down", ((struct ppp_softc *)ifp)->sc_if.if_unit);
			}
		} else {
			m_freem(m);
		}
		return;

	    default:
		((struct ppp_softc *)ifp)->sc_event = RUP;
		if (ppp_lcp_state_trans[((struct ppp_softc *)ifp)->sc_event][((struct ppp_softc *)ifp)->sc_lcp_state].st_action) {
			(*ppp_lcp_state_trans[RUP][((struct ppp_softc *)ifp)->sc_lcp_state].st_action)(ifp, LINK, m);
		}
		log(LOG_ERR, "ppp%d: received unknown protocol", ((struct ppp_softc *)ifp)->sc_if.if_unit);
		return;
	}
}

/* ppp_output() is called to send data packets to the remote interface;  it appends a frame header and sends the packet to the lower layer;
 * this is called only for data packets;  control packets are sent directly from the negotiation routines */

ppp_output(ifp, m, dst)
	struct ifnet *ifp;
	struct mbuf *m;
	struct sockaddr *dst;
{
	static struct frame_hdr frame_hdr = {0xff03};

	if ((((struct ppp_softc *)ifp)->sc_if.if_flags & (IFF_RUNNING | IFF_UP)) != (IFF_RUNNING | IFF_UP)) {
		m_freem(m);
		return(ENETDOWN);
	}
	switch (dst->sa_family) {
	    case AF_INET:

		/* do Van Jacobson Compression if it is enabled and packet is TCP/IP */
		if (((struct ppp_softc *)ifp)->sc_ipcp_conf_opt[CO_IPCP_COMPRESSION].sc_local_data.cod_short == VJ_COMP_TCP_IP
		     && mtod(m, struct ip *)->ip_p == IPPROTO_TCP) {
			int p = sl_compress_tcp(m, mtod(m, struct ip *), &((struct ppp_softc *)ifp)->sc_vj_comp, 1);
			*mtod(m, u_char *) |= p;
			frame_hdr.fh_proto = VJ_COMP_TCP_IP;
		} else {
			frame_hdr.fh_proto = IP;
		}
		break;

	    default:
		return(EAFNOSUPPORT);
	}

	/* allocate space for frame header */
	if (!M_HASCL(m) && m->m_off - MMINOFF >= sizeof(struct frame_hdr)) {
		m->m_off -= sizeof(struct frame_hdr);
		m->m_len += sizeof(struct frame_hdr);
	} else {
		struct mbuf *m0;
		MGET(m0, M_DONTWAIT, MT_DATA);
		if (m0 == 0) {
			log(LOG_ERR, "ppp: no mbuf is available");
			return(ENOBUFS);
		}
		m0->m_next = m;
		m = m0;
		m->m_off = MMAXOFF - sizeof(struct frame_hdr);
		m->m_len = sizeof(struct frame_hdr);
	}

	/* copy in frame header and send packet to lower layer */
	bcopy((u_char *)&frame_hdr, mtod(m, u_char *), sizeof(struct frame_hdr));
	(*ifp->if_lower->if_output)(ifp->if_lower, m, 0);
	++ifp->if_opackets;
	return(0);
}

#endif
