/*
 * WorldVoc.t
 * Copyright (c) 1999 The Children of Dana
 * Written by David Haire
 *
 * This file contains the vocabulary of WorldClass.
 *
 */

/*
 * Define compound prepositions. Since prepositions that appear in parsed
 *   sentences must be single words, we must define any logical
 *   prepositions that consist of two or more words here. Note that only
 *   two words can be pasted together at once; to paste more, use a second
 *   step.
 */
compoundWord 'on' 'to' 'onto';
compoundWord 'in' 'to' 'into';
compoundWord 'in' 'between' 'inbetween';
compoundWord 'down' 'in' 'downin';
compoundWord 'down' 'on' 'downon';
compoundWord 'up' 'on' 'upon';
compoundWord 'out' 'of' 'outof';
compoundWord 'off' 'of' 'offof';
compoundWord 'out' 'from' 'outfrom';
compoundWord 'outfrom' 'under' 'outfromunder';
compoundWord 'from' 'under' 'fromunder';
compoundWord 'outfrom' 'underneath' 'outfromunderneath';
compoundWord 'from' 'underneath' 'fromunderneath';
compoundWord 'outfrom' 'beneath' 'outfrombeneath';
compoundWord 'from' 'beneath' 'frombeneath';
compoundWord 'outfrom' 'below' 'outfrombelow';
compoundWord 'from' 'below' 'frombelow';
compoundWord 'outfrom' 'behind' 'outfrombehind';
compoundWord 'from' 'behind' 'frombehind';
compoundWord 'outfrom' 'between' 'outfrombetween';
compoundWord 'from' 'between' 'frombetween';

/*
 *  Prep: object
 *
 *  A preposition. The preposition property specifies the vocabulary word.
 */
class Prep: object ;

/*
 * PREPOSITIONS
 *
 * NOTE: Adding to the preposition vocab lists only affects verbs that take
 *   indirect objects. Verbs that have preps stuck in their verb vocab
 *   lists don't pick up all the synonyms. E.g.,
 *
 *  verb = 'look in'
 *
 * will *only* match input commands like
 *
 *  >look in bucket
 *
 * NOT commands like
 *
 *  >look inside bucket
 *  >look into bucket
 *
 * even though we make 'inside' and 'into' synonyms of 'in' for inPrep
 *   below.
 */
ofPrep: Prep preposition='of' sdesc="of";
aboutPrep: Prep preposition = 'about' sdesc = "about";
withPrep: Prep preposition = 'with' sdesc = "with";
toPrep: Prep preposition = 'to' sdesc = "to";
onPrep: Prep preposition = 'on' 'onto' 'downon' 'upon' sdesc = "on";
inPrep: Prep preposition = 'in' 'into' 'downin' 'inside' sdesc = "in";
offPrep: Prep preposition = 'off' 'offof' sdesc = "off";
outPrep: Prep preposition = 'out' 'outof' 'outside' 'outfrom' sdesc = "out";
fromPrep: Prep preposition = 'from' sdesc = "from";
betweenPrep: Prep preposition = 'between' 'inbetween' sdesc = "between";
overPrep: Prep preposition = 'over' sdesc = "over";
atPrep: Prep preposition = 'at' sdesc = "at";
upPrep: Prep preposition = 'up' sdesc = "up";
downPrep: Prep preposition = 'down' sdesc = "down";
aroundPrep: Prep preposition = 'around' sdesc = "around";
throughPrep: Prep preposition = 'through' 'thru' sdesc = "through";
behindPrep: Prep preposition = 'behind' sdesc = "behind";
forPrep: Prep preposition = 'for' sdesc = "for";
underPrep: Prep 
  preposition = 'under' 'underneath' 'beneath' 'below' 
  sdesc = "under"
;
outfromunderPrep: Prep
  preposition = 'outfromunderneath' 'fromunderneath' 'outfromunder' 'fromunder' 'outfrombeneath' 'frombeneath' 'outfrombelow' 'frombelow'
  sdesc = "out from under"
;
outfrombehindPrep: Prep
  preposition = 'outfrombehind' 'frombehind'
  sdesc = "out from behind"
;
outfrombetweenPrep: Prep
  preposition = 'outfrombetween' 'frombetween'
  sdesc = "out from between"
;

/*
 * Direction prepositions. We need these so we can have two-word verbs like
 *   'go north' and 'look east'.
 */
dirPrep: Prep
  preposition = 'north' 'n' 'south' 's' 'east' 'e' 'west' 'w' 'up' 'down' 'northeast' 'ne' 'northwest' 'nw' 'southeast' 'se' 'southwest' 'sw' 'u' 'd' 'upstream' 'downstream' 'us' 'ups' 'ds'
  sdesc = "direction"
;

/*
 * Articles
 */
articles: object
  article = 'the' 'a' 'an'
;

/*
 * Verbs
 */
class Verb: object
  sdesc = "verb"

  /*
   * This method prints the verb, marked according to the object that
   *   is the subject. E.g.,
   *
   *   "\^<<actor.thedesc>> <<openVerb.desc(actor)>> the book.";
   *
   * would become
   *
   *   "The troll opens the book.";
   *
   * if the actor is singular, but
   *
   *   "The trolls open the book.";
   *
   * if the actor is plural.
   *
   * Note that this method will need to be overridden for a particular
   *   verb if the plural form is something other than verb+s.
   */
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "<<self.sdesc>>s";
  }

  /*
   * Properties of the actor that must be true for this verb and
   *   object.
   *
   * The properties are called with parameters obj and silent, where
   *   obj is the object, and silent is a boolean indicating whether or
   *   not the error message (if any) should be printed. The properties
   *   must return nil if the access is not allowed; true otherwise.
   *
   * By default direct objects require the same methods as indirect
   *   objects, but this isn't true for all verbs.
   */
  requires = [[]]
  dorequires = { return self.requires; }
  iorequires = { return self.requires; }

  /*
   * If this boolean is true, we'll print an explanation of why the
   *   verb could not be applied to the object when a requirement is
   *   not satisified. We don't want this on for verbs whose only
   *   action is to satisfy the requirement.
   *
   * For example, the verb "take" requires &cantake, but we don't want
   *   to print
   *
   *  You can't do that, because you can't take the ...
   *
   * Since all the player is trying to do is take the thing, we'd
   *   rather just get
   *
   *  You can't take the ...
   *
   * For verbs whose action needs but is not equivalent to the
   *   requirements, we want a full explanation:
   *
   *  >eat kee
   *  You can't do that, because you can't take the cheez kee.
   */
  longexplanation = true
  dolongexplanation = { return self.longexplanation; }
  iolongexplanation = { return self.longexplanation; }

  /*
   * This function is called when nothing matched by the do/io
   *   vocabulary is accessible.
   */
  cantReach(actor, dolist, iolist, prep) = {
    local i, lbase, l, isdo;
    if (global.playtestfailtime = global.ticks) {
      "That command is only for playtesting!";
      return;
    }
    if (iolist <> nil) {
      lbase := iolist;
      isdo := nil;
    }
    else {
      lbase := dolist;
      isdo := true;
    }

    /*
     * Remove things that aren't known to the actor.
     */
    l := [];
    for (i := 1; i <= length(lbase); i++) {
      if (lbase[i].isknownto(actor))
        l += lbase[i];
    }

    /*
     * If there are no objects left, pretend we don't know what the
     *   player's talking about.
     *
     * If there's a single object in the list, print a full
     *   explanation.
     *
     * If there are any objects in the list that are in the actor's
     *   current location, print full explanations for these
     *   objects.
     *
     * If all else fails, tell the player that no objects with this
     *   vocabulary are acceptable in this context.
     */
    if (l = []) {
      "\^<<actor.subjprodesc>> <<actor.doesnt>> know what that is.";
      return;
    }
    else if (length(l) = 1) {
      self.invalidobject(actor, l[1], isdo);
      setit(l[1]);
    }
    else {
      local l2;
      l2 := [];
      for (i := 1; i <= length(l); i++)
        if (l[i].iscontained(actor.location, nil))
          l2 += l[i];
      if (l2 = [])
        "There is nothing here matching that vocabulary that <<actor.subjprodesc>> can do that to.";
      else if (length(l2) = 1) {
        self.invalidobject(actor, l2[1], isdo);
        setit(l2[1]);
      }
      else for (i := 1; i <= length(l2); i++)
        "<<l2[i].multisdesc>>: <<self.invalidobject(actor, l2[i], isdo)>>\n";
    }
  }

  /*
   * A hack is that we sometimes want to say "you can't do that because
   *   <blah blah blah>," whereas other times we just want to print the
   *   message that the checkDo method printed. The global flag
   *   global.canoutput, if true tells us to do the latter. This is set
   *   by the canX methods in Sensor.
   *
   * When global.canoutput is true (i.e., when we're not printing a
   *   generic message), we usually still want to explain which action
   *   failed. So for the methods listed in global.senses, we print
   *   "(trying to ____ the ____ first)". E.g.,
   *
   *  >eat papers
   *  (trying to take the papers first)
   *
   *  There are too many papers to carry.
   *
   * (It's clear from this example that we have to be a bit careful
   *   about what failure messages we put in verDoTake!)
   */
  invalidobject(actor, o, isdo) = {
    local v := self;
    
    /*
     * If the actor's not really an actor, call the actor's
     *   message. (This is for fools who try stuff like "cheez kee,
     *   examine sword")
     */
    if (not isclass(actor, Actor)) {
      actor.aamessage(v, o, nil, nil);
      return;
    }

    if (isdo)
      v.checkDo(actor, o, true);
    else
      v.checkIo(actor, o, true);
    if (isdo and v.dolongexplanation or not isdo and v.iolongexplanation) {
      if (not global.canoutput) {
        "\^<<actor.subjsdesc>> can't do that, because ";
      }
      else {
        local pos;
        pos := find(global.senses, global.failedsense);
        if (pos <> nil)
          "(trying to <<global.senses[pos+1]>> <<o.objthedesc(actor)>> first)\b";
        caps();
      }
    }
    else  
      caps();
    if (isdo)
      v.checkDo(actor, o, nil);
    else
      v.checkIo(actor, o, nil);
  }

  /*
   * Check object requirements common to both do and io cases.
   *
   * The actor must know about an object to do anything to it, unless
   *   this verb is a Systemverb.
   *
   * Room vocabulary will be disallowed unless we're using a system
   *   verb, as we don't want room nouns and adjectives to interfere
   *   with manipulable objects.
   */
  checkCommon(actor, obj, silent) = {
    if (isclass(obj, Room) and not isclass(obj, Nestedroom) and not isclass(self, Systemverb) and not silent) {
      "I don't know what you're referring to.";
      global.canoutput := true;
      return nil;
    }
    if (not obj.isknownto(actor) and not isclass(self, Systemverb)) {
      if (not silent)
        "I can't imagine what you're referring to.";
      global.canoutput := true;
      return nil;
    }
    else
      return true;
  }

  /*
   * Check requirements in our requirements list.
   *
   * lastprep will be checked to see if it is from this turn. If not,
   *   it's stale, which means that this command has no prep.
   *
   * If the command has a prep, the list of requirements for the prep
   *   will be retrieved. If the command has no prep, the first list in
   *   the requirements list will be used.
   *
   * The format for the requirements list is as follows:
   *
   * [ [&cantake ...] fromPrep withPrep [&cansee ...] toPrep [...] ]
   *
   */
  checkReq(actor, obj, silent, requirements) = {
    local i, j, len, loctype, prep, req;
    if (global.lastpreptime <> global.ticks)
      prep := nil;
    else
      prep := global.lastprep;
    if (prep = nil)
      loctype := nil;
    else switch (prep) {
      case inPrep:
      case toPrep:
        loctype := 'in';
        break;
      case onPrep:
        loctype := 'on';
        break;
      case underPrep:
        loctype := 'under';
        break;
      case behindPrep:
        loctype := 'behind';
        break;
      default:
        loctype := nil;
        break;
    }
    if (prep = nil)
      req := requirements[1];
    else {
      for (i := 1; i <= length(requirements); i++)
        if (requirements[i] = prep)
          break;
      if (i > length(requirements))
        req := requirements[1];
      else {
        for (j := i + 1; j <= length(requirements); j++)
          if (datatype(requirements[j]) = 7) {
            req := requirements[j];
            break;
          }
      }
    }
    len := length(req);
    for (i := 1; i <= length(req); i++) {
      if (not actor.(req[i])(obj, loctype, silent)) {
        global.failedsense := req[i];
        return nil;
      }
    }
    return true;
  }
  checkDo(actor, obj, silent) = {
    if (not self.checkCommon(actor, obj, silent))
      return nil;
    if (not self.checkReq(actor, obj, silent, self.dorequires))
      return nil;
    return true;
  }

  /*
   * Check to see if we're in playtesting mode. If not, don't allow
   *   Debugverbs to work. (We send a message to Verb.cantReach via
   *   global.playtestfailtime so that a message to this effect will be
   *   printed.)
   */
  playtestcheck = {
    if (isclass(self, Debugverb) and not global.playtesting) {
      global.playtestfailtime := global.ticks;
      return true;
    }
    else
      return nil;
  }

  validDo(actor, obj, seqno) = {
    local ret;
    global.lastverb := self;
    global.lastactor := actor;
    global.lastdo := obj;
    if (self.playtestcheck)
      return nil;
    ret := self.checkDo(actor, obj, true);
    return ret;
  }

  /*
   * The following method returns a list of validDo objects. This
   *   speeds up parsing. All objects in the list returned are
   *   submitted to validDo, so this can return too much.
   */
  validDoList(actor, prep, iobj) = {
    global.lastprep := prep;
    global.lastpreptime := global.ticks;
    return nil;
  }

  checkIo(actor, obj, silent) = {
    if (not self.checkCommon(actor, obj, silent))
      return nil;
    if (not self.checkReq(actor, obj, silent, self.iorequires))
      return nil;
    return true;
  }
  validIo(actor, obj, seqno) = {
    local ret;
    global.lastverb := self;
    global.lastactor := actor;
    global.lastio := obj;
    ret := self.checkIo(actor, obj, true);
    return ret;
  }

  /*
   * The following method returns a list of validIo objects. This
   *   speeds up parsing. All objects in the list returned are
   *   submitted to validIo, so this can return too much.
   */
  validIoList(actor, prep, dobj) = {
    global.lastprep := prep;
    global.lastpreptime := global.ticks;
    return nil;
  }

  /*
   * The parser calls this method when it can't find a ver... method
   *   for the specified object. This generally indicates the
   *   programmer has omitted something.
   */
  invalidObj(actor, obj, name) = {
    "\^<<actor.subjsdesc>> can't <<self.sdesc>> <<obj.objthedesc(actor)>>.";
  }

  /*
   * For most verbs we don't want to allow multiple direct objects
   *   (including "all"). Verbs for which multiple direct objects are
   *   to be allowed should have allok = true.
   */
  allok = nil

  /*
   * These methods return the list of objects for "all" and for
   *   defaulting when do's and io's are omitted.
   */
  doDefault(actor, prep, io) = {
    local i, l, r, loc;
    if (not self.allok) {
      if (objwords(1) = ['A']) {
        global.allerror := true;
        return [];
      }
    }
    loc := actor.location;
    l := [];
    for (;;) {
      l += loc.contents;
      if (not isclass(loc, Nestedroom))
        break;
      else if (loc.location = nil or loc.location = TOP)
        break;
      else
        loc := loc.location;
    }
    r := [];
    for (i := length(l); i > 0; i--)
      if (self.checkDo(actor, l[i], true))
        r += l[i];
    return r;
  }
  ioDefault(actor, prep) = {
    local i, l, r, loc;
    loc := actor.location;
    l := [];
    for (;;) {
      l += loc.contents;
      if (not isclass(loc, Nestedroom))
        break;
      else if (loc.location = nil or loc.location = TOP)
        break;
      else
        loc := loc.location;
    }
    r := [];
    for (i := length(l); i > 0; i--)
      if (self.checkIo(actor, l[i], true))
        r += l[i];
    return r;
  }

  /*
   * Parse unknown objects. By default, these simply return 'nil' and 
   *   allow the parser to perform the default error reporting.
   */
  parseUnknownDobj(actor, prep, iobj, wordlist) = {
    return nil;
  }
  parseUnknownIobj(actor, prep, dobj, wordlist) = {
    return nil;
  }

  /*
   * verbAction. Like roomAction and actorAction, only this method 
   *   allows the verb to perform special handling on a command before 
   *   the normal parser processing. The default is to do nothing.
   */
  verbAction(actor, dobj, prep, io) = {
  }
;

/*
 * Verbs that can be used with no objects.
 */
class Soloverb: Verb
  action(actor) = {
    if (isclass(self, Debugverb) and not global.playtesting) {
      "That verb is only for playtesting!";
      abort;
      return;
    }
    global.lastverb := self;
    global.lastactor := actor;
    global.lastdo := nil;
    global.lastio := nil;
    self.soloaction(actor);
  }
;

/*
 * againVerb is required by the parser, which fills in the action itself.
 */
againVerb: Verb
    verb = 'again' 'g' 'repeat'
;

/*
 * Incredible hack from hell.
 * TADS uses the verb "takeVerb" as a placeholder whenever there isn't
 *   actually a verb. The only place this seems to matter is when you give
 *   an actor a commnd, as in
 *
 *  troll, eat rabbit
 *
 * In this case, takeVerb.validDo will get called because the parser
 *   doesn't find a verb in the first part of the command (before the
 *   comma). The parser needs to do this to disambiguate the actor who's
 *   being commanded. (Which troll?)
 *
 * To hack around this, we define takeVerb to be a verb that's never
 *   actually used by the player. Its only function is to field the
 *   disambiguation queries for actors. We return true for an object if the
 *   actor can speak to that object.
 */
takeVerb: Verb
  sdesc = "takeVerb"
  verb = 'takeVerb'
  action = { "No verb here."; }
  validDo(actor, obj, seqno) = {
    return actor.canspeakto(obj, nil, true);
  }
  validIo(actor, obj, seqno) = { return nil; }
  cantReach(actor, dolist, iolist, prep) = {
    tellVerb.cantReach(actor, dolist, nil, toPrep);
  }
;

/*
 * The real takeVerb...
 */
XtakeVerb: Verb
  sdesc = "take"
  verb = 'get' 'take' 'grab' 'remove'
  doAction = 'Take'
  ioAction(fromPrep) = 'Takefrom'
  ioAction(offPrep) = 'Takeoff'
  ioAction(outPrep) = 'Takeout'
  ioAction(inPrep) = 'Takeout'
  ioAction(onPrep) = 'Takeoff'
  ioAction(underPrep) = 'Takeunder'
  ioAction(outfromunderPrep) = 'Takeunder'
  ioAction(behindPrep) = 'Takebehind'
  ioAction(outfrombehindPrep) = 'Takebehind'
  dorequires = [[&cantake]]
  iorequires = [[&cansee]]
  longexplanation = nil
  allok = true

  /*
   * Return a list of all Items contained in the io that match the
   *   preposition.
   * First we convert the prep to a location type (in, on, under,
   *   behind) if possible. Then we get all the contents of the
   *   indirect object and prune out the ones that the player can't
   *   take.
   */
  doDefault(actor, prep, io) = {
    local i, loctypes, l, r;
    if (io = nil)
      io := actor.location;
    switch (prep) {
        case offPrep:
        case onPrep:
      loctypes := ['on'];
      break;
        case outPrep:
        case inPrep:
      loctypes := ['in'];
      break;
        case underPrep:
        case outfromunderPrep:
      loctypes := ['under'];
        case behindPrep:
        case outfrombehindPrep:
      loctypes := ['behind'];
      break;
        case fromPrep:
      loctypes := global.loctake;
      break;
        default:
      loctypes := nil;
      break;
    }
    l := io.itemcontents(actor, loctypes) - actor.itemcontents(actor, nil);
    r := [];
    for (i := length(l); i > 0; i--)
      if (self.checkDo(actor, l[i], true))
        r += l[i];
    return r;
  }
;

dropVerb: Verb
  sdesc = "drop"
  verb = 'drop' 'put down'
  ioAction(onPrep) = 'Puton'
  ioAction(inPrep) = 'Putin'
  ioAction(underPrep) = 'Putunder'
  ioAction(behindPrep) = 'Putbehind'
  ioAction(outPrep) = 'Putthrough'
  ioAction(throughPrep) = 'Putthrough'
  doAction = 'Drop'
  dorequires = [[&cantake]]
  iorequires = [[&cansee]]
  allok = true
  doDefault(actor, prep, io) = {
    return actor.itemcontents(actor, nil);
  }
;

tasteVerb: Verb
  sdesc = "taste"
  verb = 'taste' 'lick'
  doAction = 'Taste'
  requires = [[&cantouch]]
;

inspectVerb: Verb
  sdesc = "inspect"
  verb = 'inspect' 'examine' 'look at' 'l at' 'x' 'ex'
  doAction = 'Inspect'
  requires = [[&cansee]]
  longexplanation = nil
  allok = nil
  disambigDobj(actor, prep, iobj, verprop, words, objs, flags, num, ambig, silent) = {
    local i, o, found := nil, ocount := 0, olist := [];
    for (i := 1; i <= length(objs); i++) {
      o := objs[i];
      if (isclass(o, Everywhere) or isclass(o, Nestedroom) or not isclass(o, Room)) {
        found := true;
        ocount++;
        olist += o;
      }
    }
    if (found) {
      if (ocount <> num)
        return [DISAMBIG_CONTINUE];
      else
        return [DISAMBIG_DONE] + olist;
    }
    else {
      return [DISAMBIG_CONTINUE];
    }
  }
;

lookVerb: Soloverb
  sdesc = "look at"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "looks at";
  }
  verb = 'look' 'l' 'look around' 'l around'
  soloaction(actor) = {
    actor.location.lookaround(actor, true);
  }
  doAction = 'Inspect'
  requires = [[&cansee]]
  longexplanation = nil
;

inventoryVerb: Soloverb
  sdesc = "take inventory of"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "takes inventory of";
  }
  verb = 'inventory' 'i'
  soloaction(actor) = { actor.inventory; }
;

touchVerb: Soloverb
  sdesc = "touch"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "touches";
  }
  verb = 'touch' 'feel' 'feel around'
  soloaction(actor) = {
    actor.location.feelaround(actor, true);
  }
  doAction = 'Touch'
  ioAction(withPrep) = 'Touchwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
  dolongexplanation = nil
  iolongexplanation = true
;

smellVerb: Soloverb
  sdesc = "smell"
  verb = 'smell' 'sniff' 'waft' 
  soloaction(actor) = {
    actor.location.smellaround(actor, true);
  }
  doAction = 'Smell'
  requires = [[&cansmell]]
  longexplanation = nil
;

listentoVerb: Soloverb
  sdesc = "listen to"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "listens to";
  }
  verb = 'listen' 'listen to' 'hear'
  soloaction(actor) = {
    actor.location.listenaround(actor, true);
  }
  doAction = 'Listento'
  requires = [[&canhear]]
  longexplanation = nil
;

eatVerb: Verb
  sdesc = "eat"
  verb = 'eat' 'consume' 'swallow'
  doAction = 'Eat'
  requires = [[&cantake]]
;

askaboutVerb: Verb
  sdesc = "ask about"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "asks about";
  }
  verb = 'ask' 'inquire' 'enquire' 'question' 'query'
  prepDefault = aboutPrep
  ioAction(aboutPrep) = 'Askabout'
  ioAction(forPrep) = 'Askfor'
  dorequires = [[&canspeakto, &canhear]]
  iorequires = [[&isatopic]]
;

sayVerb: Verb
  sdesc = "say"
  verb = 'say' 'speak' 'utter' 'answer'
  doAction = 'Say'
  ioAction(toPrep) = 'Sayto'
  dorequires = [[&canusealpha]]
  iorequires = [[&canspeakto]]
;

/* yell does the same thing as say by default */
yellVerb: Verb
  sdesc = "yell"
  verb = 'yell' 'shout' 'scream'
  doAction = 'Say'
  ioAction(atPrep) = 'Sayto'
  ioAction(toPrep) = 'Sayto'
  dorequires = [[&canusealpha]]
  iorequires = [[&canspeakto]]
;

lookinVerb: Verb
  sdesc = "look in"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "looks in";
  }
  verb = 'look in' 'look into' 'look inside' 'look downin' 'l in' 'l into' 'l inside' 'l downin'
  doAction = 'Lookin'
  requires = [[&cansee]]
;

lookonVerb: Verb
  sdesc = "look at what's on"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "looks at what's on";
  }
  verb = 'look on' 'look onto' 'look downon' 'look upon' 'l on' 'l onto' 'l downon' 'l upon'
  doAction = 'Lookon'
  requires = [[&cansee]]
;

lookunderVerb: Verb
  sdesc = "look under"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "looks under";
  }
  verb = 'look under' 'look beneath' 'l under' 'l beneath'
  doAction = 'Lookunder'
  requires = [[&cansee]]
;

lookbehindVerb: Verb
  sdesc = "look behind"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "looks behind";
  }
  verb = 'look behind' 'l behind'
  doAction = 'Lookbehind'
  requires = [[&cansee]]
;

lookthroughVerb: Verb
  sdesc = "look through"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "looks through";
  }
  verb = 'look through' 'look thru' 'l through' 'l thru' 'look out' 'look outof'
  doAction = 'Lookthrough'
  requires = [[&cansee]]
;

putVerb: Verb
  sdesc = "put"
  verb = 'put' 'place' 'position' 'insert'
  prepDefault = inPrep  
  ioAction(inPrep) = 'Putin'
  ioAction(onPrep) = 'Puton'
  ioAction(underPrep) = 'Putunder'
  ioAction(behindPrep) = 'Putbehind'
  ioAction(throughPrep) = 'Putthrough'
  dorequires = [[&cantake]]
  iorequires = [[&canputinto]]
  dolongexplanation = true
  iolongexplanation = nil
  allok = true
  doDefault(actor, prep, io) = {
    return actor.itemcontents(actor, nil);
  }
;

tellVerb: Verb
  sdesc = "tell"
  verb = 'tell'
  prepDefault = aboutPrep
  ioAction(aboutPrep) = 'Tellabout'
  ioAction(toPrep) = 'Tellto'
  dorequires = [[&canspeakto]
                toPrep [&isatopic]]
  iorequires = [[&isatopic]
                toPrep [&canspeakto]]
;

followVerb: Verb
  sdesc = "follow"
  verb = 'follow'
  doAction = 'Follow'
  requires = [[&cansee]]
;

digVerb: Verb
  sdesc = "dig in"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "digs in";
  }
  verb = 'dig' 'dig in' 'dig into' 'dig inside' 'dig downin' 'excavate' 'exhume'
  prepDefault = withPrep
  ioAction(withPrep) = 'Digwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

pushVerb: Verb
  sdesc = "push"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "pushes";
  }
  verb = 'push' 'press' 'depress'
  doAction = 'Push'
  ioAction(onPrep) = 'Typeon'
  dorequires = [[&cantouch]
                onPrep [&canusealphanum]]
  iorequires = [[]
                onPrep [&cantouch]]
;

attachVerb: Verb
  sdesc = "attach"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "attaches";
  }
  verb = 'attach' 'connect'
  prepDefault = toPrep
  ioAction(toPrep) = 'Attachto'
  requires = [[&cantouch]]
;

tieVerb: Verb
  sdesc = "tie"
  verb = 'tie' 'lash'
  prepDefault = toPrep
  ioAction(toPrep) = 'Tieto'
  ioAction(onPrep) = 'Tieto'
  requires = [[&cantouch]]
;

wearVerb: Verb
  sdesc = "wear"
  verb = 'wear' 'put on' 'don'
  doAction = 'Wear'
  requires = [[&cantake]]
;

unwearVerb: Verb
  sdesc = "take off"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "takes off";
  }
  verb = 'take off' 'doff'
  doAction = 'Unwear'
  ioAction(fromPrep) = 'Takefrom'
  dorequires = [[&cantake]]
  iorequires = [[&cansee]]
;

openVerb: Verb
  sdesc = "open"
  verb = 'open'
  doAction = 'Open'
  ioAction(withPrep) = 'Openwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

closeVerb: Verb
  sdesc = "close"
  verb = 'close'
  doAction = 'Close'
  requires = [[&cantouch]]
;

plugVerb: Verb
  sdesc = "plug"
  verb = 'plug'
  prepDefault = inPrep
  ioAction(inPrep) = 'Plugin'
  requires = [[&cantouch]]
;

lightVerb: Verb
  verb = 'light'
  sdesc = "light"
  allok = nil
  doAction = 'Light'
  ioAction(withPrep) = 'Lightwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

unlightVerb: Verb
  verb = 'extinguish' 'unlight' 'blow out'
  sdesc = "extinguish"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "extinguishes";
  }
  allok = nil
  doAction = 'Unlight'
  dorequires = [[&cantouch]]
;

screwVerb: Verb
  sdesc = "screw"
  verb = 'screw'
  doAction = 'Screw'
  ioAction(withPrep) = 'Screwwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

unscrewVerb: Verb
  sdesc = "unscrew"
  verb = 'unscrew'
  doAction = 'Unscrew'
  ioAction(withPrep) = 'Unscrewwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

setVerb: Verb
  sdesc = "set"
  verb = 'set'
  doAction = 'Turn'
  ioAction(toPrep) = 'Turnto'
  ioAction(inPrep) = 'Putin'
  ioAction(onPrep) = 'Puton'
  ioAction(underPrep) = 'Putunder'
  ioAction(behindPrep) = 'Putbehind'
  ioAction(throughPrep) = 'Putthrough'
  dorequires = [[&cantouch]]
  iorequires = [[&canputinto]
                toPrep [&canusenumberorlist]]
;

turnVerb: Verb
  sdesc = "turn"
  verb = 'turn' 'rotate' 'twist' 'dial' 'tune'
  doAction = 'Turn'
  ioAction(toPrep) = 'Turnto'
  dorequires = [[&cantouch]]
  iorequires = [[&canusenumberorlist]]
;

switchVerb: Verb
  sdesc = "switch"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "switches";
  }
  verb = 'switch'
  doAction = 'Switch'
  dorequires = [[&cantouch]]
;

flipVerb: Verb
  verb = 'flip' 'flick'
  sdesc = "flip"
  doAction = 'Flip'
  requires = [[&cantouch]]
;

turnonVerb: Verb
  sdesc = "turn on"
  verb = 'activate' 'turn on' 'switch on' 'enable' 'start' 'start up'
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "turns on";
  }
  doAction = 'Turnon'
  requires = [[&cansee]]
;

turnoffVerb: Verb
  sdesc = "turn off"
  verb = 'turn off' 'deactivate' 'switch off' 'disable'
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "turns off";
  }
  doAction = 'Turnoff'
  requires = [[&cansee]]
;

sitVerb: Soloverb
  sdesc = "sit"
  verb = 'sit' 'sit down'
  soloaction(actor) = {
    local o;
    o := actor.location.ground;
    if (o = true or o = nil)
      o := Ground;
    if (sitonVerb.validDo(actor, o, 0)) {
      Outhide(true);
      o.verDoSiton(actor);
      if (Outhide(nil))
        "You'll have to be more specific.";
      else
        o.doSiton(actor);
    }
    else
      "You'll have to be more specific.";
  }
;

sitonVerb: Verb
  sdesc = "sit on"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "sits on";
  }
  verb = 'sit on' 'sit downon' 'sit upon'
  doAction = 'Siton'
  requires = [[&cantouch]]
;

sitinVerb: Verb
  sdesc = "sit in"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "sits in";
  }
  verb = 'sit in' 'sit downin'
  doAction = 'Sitin'
  requires = [[&cantouch]]
;

lieVerb: Soloverb
  sdesc = "lie"
  verb = 'lie' 'lie down'
  soloaction(actor) = {
    local o;
    o := actor.location.ground;
    if (o = true or o = nil)
      o := Ground;
    if (lieonVerb.validDo(actor, o, 0)) {
      Outhide(true);
      o.verDoLieon(actor);
      if (Outhide(nil))
        "You'll have to be more specific.";
      else
        o.doLieon(actor);
    }
    else
      "You'll have to be more specific.";
  }
;

lieinVerb: Verb
  sdesc = "lie in"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "lies in";
  }
  verb = 'lie in' 'lie downin'
  doAction = 'Liein'
  requires = [[&cantouch]]
;

lieonVerb: Verb
  sdesc = "lie on"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "lies on";
  }
  verb = 'lie on' 'lie downon' 'lie upon'
  doAction = 'Lieon'
  requires = [[&cantouch]]
;

getoutVerb: Verb
  sdesc = "get out of"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "gets out of";
  }
  verb = 'get out' 'get outof'
  doAction = 'Getout'
  requires = [[&cantouch]]
;

getoffVerb: Verb
  sdesc = "get off of"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "gets off of";
  }
  verb = 'get off' 'get offof'
  doAction = 'Getoff'
  requires = [[&cantouch]]
;

getoutfrombehindVerb: Verb
  sdesc = "get out from behind"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "gets out from behind";
  }
  verb = 'get outfrombehind'
  doAction = 'Getoutfrombehind'
  requires = [[&cantouch]]
;

getoutfromunderVerb: Verb
  sdesc = "get out from under"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "gets out from under";
  }
  verb = 'get outfromunder' 'get outfromunderneath'
  doAction = 'Getoutfromunder'
  requires = [[&cantouch]]
;

getinVerb: Verb
  sdesc = "get in"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "gets in";
  }
  verb = 'get in' 'get into' 'get inside' 'get downin'
  doAction = 'Getin'
  requires = [[&cantouch]]
;

getonVerb: Verb
  sdesc = "get on"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "gets on";
  }
  verb = 'get on' 'get onto' 'stand on' 'go on' 'get downon'
  doAction = 'Geton'
  requires = [[&cantouch]]
;

getunderVerb: Verb
  sdesc = "get under"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "gets under";
  }
  verb = 'get under' 'get underneath' 'stand under' 'stand underneath'
     'go under' 'go underneath'
  doAction = 'Getunder'
  requires = [[&cantouch]]
;

getbehindVerb: Verb
  sdesc = "get behind"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "gets behind";
  }
  verb = 'get behind' 'stand behind' 'go behind' 'hide behind'
  doAction = 'Getbehind'
  requires = [[&cantouch]]
;

boardVerb: Verb
  verb = 'board'
  sdesc = "board"
  doAction = 'Board'
  requires = [[&cantouch]]
;

waitVerb: Soloverb
  sdesc = "wait for"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "waits for";
  }
  verb = 'wait' 'z' 'pause'
  soloaction(actor) = { "Time passes...\n"; }
;

attackVerb: Verb
  verb = 'attack' 'kill'
  sdesc = "attack"
  prepDefault = withPrep
  ioAction(withPrep) = 'Attackwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

hitVerb: Verb
  sdesc = "hit"
  verb = 'hit' 'punch' 'beat' 'pound' 'bang' 'bang on' 'bang upon' 'bang downon' 'bang onto' 'hit on' 'hit upon' 'hit downon' 'hit onto' 'punch on' 'punch upon' 'punch downon' 'punch onto' 'beat on' 'beat upon' 'beat downon' 'beat onto' 'pound on' 'pound upon' 'pound downon' 'pound onto'
  doAction = 'Hit'
  ioAction(withPrep) = 'Hitwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

kickVerb: Verb
  sdesc = "kick"
  verb = 'kick' 'boot'
  doAction = 'Kick'
  requires = [[&cantouch]]
;

breakVerb: Verb
  sdesc = "break"
  verb = 'break' 'destroy' 'damage' 'bust' 'mangle' 'smash' 'shatter'
  doAction = 'Break'
  ioAction(withPrep) = 'Hitwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

climbVerb: Verb
  sdesc = "climb"
  verb = 'climb' 'scale' 'clamber' 'clamber up' 'climb up' 'scale up' 'climb into' 'climb onto' 'climb over'
  doAction = 'Climb'
  requires = [[&cantouch]]
;

drinkVerb: Verb
  sdesc = "drink"
  verb = 'drink' 'quaff' 'imbibe'
  doAction = 'Drink'
  requires = [[&cantouch]]
;

giveVerb: Verb
  sdesc = "give"
  verb = 'give' 'offer'
  prepDefault = toPrep
  ioAction(toPrep) = 'Giveto'
  dorequires = [[&cantake]]
  iorequires = [[&cantouch, &canputinto]]
;

pullVerb: Verb
  sdesc = "pull"
  verb = 'pull' 'yank' 'pull on' 'yank on' 'pull upon' 'yank upon'
  doAction = 'Pull'
  requires = [[&cantouch]]
;

readVerb: Verb
  sdesc = "read"
  verb = 'read'
  doAction = 'Read'
  requires = [[&cansee]]
;

throwVerb: Verb
  sdesc = "throw"
  verb = 'throw' 'toss'
  prepDefault = atPrep
  ioAction(atPrep) = 'Throwat'
  ioAction(toPrep) = 'Throwto'
  ioAction(throughPrep) = 'Throwthrough'
  ioAction(underPrep) = 'Throwunder'
  ioAction(behindPrep) = 'Throwbehind'
  ioAction(inPrep) = 'Putin'
  ioAction(onPrep) = 'Puton'
  dorequires = [[&cantake]]
  iorequires = [[&cansee]
                inPrep onPrep [&canputinto]]
;

standVerb: Soloverb
  sdesc = "stand"
  verb = 'stand' 'stand up' 'get up'
  soloaction(actor) = {
    if (actor.position = 'standing')
      "\^<<actor.youre>> already standing.";
    else {
      Outhide(true);
      actor.location.verDoStand(actor);
      if (Outhide(nil))
        actor.location.verDoStand(actor);
      else
        actor.location.doStand(actor);
    }
  }
;

helloVerb: Soloverb
  sdesc = "say hello"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "says hello";
  }
  verb = 'hello' 'hi' 'greetings' 'howdy'
  soloaction(actor) = {
    if (actor = parserGetMe())
      "Talking to yourself won't help matters.";
    else
      "\^<<actor.subjthedesc>> nods.";
  }
;

showVerb: Verb
  sdesc = "show"
  verb = 'show'
  prepDefault = toPrep
  ioAction(toPrep) = 'Showto'
  dorequires = [[&cantake]]
  iorequires = [[&cansee]]
;

cleanVerb: Verb
  sdesc = "clean"
  verb = 'clean' 'polish' 'shine' 'buff' 'scrub' 'clean up' 'polish up' 'shine up' 'buff up' 'scrub up'
  doAction = 'Clean'
  ioAction(withPrep) = 'Cleanwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

moveVerb: Verb
  sdesc = "move"
  verb = 'move' 'lift' 'raise'
  doAction = 'Move'
  ioAction(withPrep) = 'Movewith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

fastenVerb: Verb
  sdesc = "fasten"
  verb = 'fasten' 'buckle' 'buckle up'
  doAction = 'Fasten'
  requires = [[&cantouch]]
;

unfastenVerb: Verb
  sdesc = "unfasten"
  verb = 'unfasten' 'unbuckle'
  doAction = 'Unfasten'
  requires = [[&cantouch]]
;

unplugVerb: Verb
  verb = 'unplug'
  sdesc = "unplug"
  doAction = 'Unplug'
  ioAction(fromPrep) = 'Unplugfrom'
  dorequires = [[&cantouch]]
  iorequires = [[&cantouch]]
;

typeVerb: Verb
  sdesc = "type"
  verb = 'type'
  prepDefault = onPrep
  ioAction(onPrep) = 'Typeon'
  dorequires = [[&canusealphanum]]
  iorequires = [[&cantouch]]
;

lockVerb: Verb
  sdesc = "lock"
  verb = 'lock'
  prepDefault = withPrep
  doAction = 'Lock'
  ioAction(withPrep) = 'Lockwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

unlockVerb: Verb
  sdesc = "unlock"
  verb = 'unlock'
  prepDefault = withPrep  
  doAction = 'Unlock'
  ioAction(withPrep) = 'Unlockwith'
  dorequires = [[&cantouch]]
  iorequires = [[&cantake]]
;

detachVerb: Verb
  sdesc = "detach"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "detaches";
  }
  verb = 'detach' 'disconnect'
  ioAction(fromPrep) = 'Detachfrom'
  doAction = 'Detach'
  dorequires = [[&cantouch]]
  iorequires = [[&cantouch]]
;

untieVerb: Verb
  sdesc = "untie"
  verb = 'untie' 'unlash'
  ioAction(fromPrep) = 'Untiefrom'
  doAction = 'Untie'
  dorequires = [[&cantouch]]
  iorequires = [[&cantouch]]
;

sleepVerb: Soloverb
  sdesc = "sleep"
  verb = 'sleep'
  soloaction(actor) = {
    "%You% <<self.desc(actor)>>.";
    self.sleep;
  }
;

pokeVerb: Verb
  sdesc = "poke"
  verb = 'poke' 'jab'
  doAction = 'Poke'
  ioAction(withPrep) = 'Pokewith'
  dorequires = [[&cantouch]]
  iorequres = [[&cantake]]
  dolongexplanation = nil
  iolongexplanation = nil
;

searchVerb: Verb
  sdesc = "search"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "searches";
  }
  verb = 'search'
  doAction = 'Search'
  requires = [[&cansee, &cantouch]]
;

shootVerb: Verb
  sdesc = "shoot"
  verb = 'shoot' 'blast' 'fire'
  prepDefault = withPrep
  ioAction(withPrep) = 'Shootwith'
  ioAction(atPrep) = 'Shootat'
  dorequires = [[&cansee]
                atPrep [&cantake]]
  iorequires = [[&cantake]
                atPrep [&cansee]]
;

knockonVerb: Verb
  verb = 'knock' 'knock on'
  sdesc = "knock on"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "knocks on";
  }
  allok = nil
  doAction = 'Knockon'
  dorequires = [[&cansee]]
;

emptyVerb: Verb
  verb = 'empty'
  sdesc = "empty"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "empties";
  }
  allok = nil
  doAction = 'Empty'
  dorequires = [[&cantake]]
;

fillVerb: Verb
  verb = 'fill'
  sdesc = "fill"
  allok = nil
  doAction = 'Fill'
  dorequires = [[&cantake]]
;

/*
 * Movement direction verbs.
 */
moveNVerb: Verb
  sdesc = "move north"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "moves north";
  }
  verb = 'move north' 'move n' 'push north' 'push n'
  doAction = 'MoveN'
;

moveSVerb: Verb
  sdesc = "move south"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "moves south";
  }
  verb = 'move south' 'move s' 'push south' 'push s'
  doAction = 'MoveS'
;

moveEVerb: Verb
  sdesc = "move east"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "moves east";
  }
  verb = 'move east' 'move e' 'push east' 'push e'
  doAction = 'MoveE'
;

moveWVerb: Verb
  sdesc = "move west"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "moves west";
  }
  verb = 'move west' 'move w' 'push west' 'push w'
  doAction = 'MoveW'
;

moveNEVerb: Verb
  sdesc = "move northeast"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "moves northeast";
  }
  verb = 'move northeast' 'move ne' 'push northeast' 'push ne'
  doAction = 'MoveNE'
;

moveNWVerb: Verb
  sdesc = "move northwest"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "moves northwest";
  }
  verb = 'move northwest' 'move nw' 'push northwest' 'push nw'
  doAction = 'MoveNW'
;

moveSEVerb: Verb
  sdesc = "move southeast"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "moves southeast";
  }
  verb = 'move southeast' 'move se' 'push southeast' 'push se'
  doAction = 'MoveSE'
;

moveSWVerb: Verb
  sdesc = "move southwest"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "moves southwest";
  }
  verb = 'move southwest' 'move sw' 'push southwest' 'push sw'
  doAction = 'MoveSW'
;

/*
 * Travel verbs.
 * We play some games here to make the syntax for the verify methods the
 *   same as it is for direct objects of other verbs.
 *
 * Note that the old noexit syntax is still supported, but the new
 *   goNowhere(actor) (no parallel ver method) is preferred.
 */
class Travelverb: Soloverb
  soloaction(actor) = {
    local dest := nil, lose := nil;
    if (not defined(self, &verprop)) {
      "Verb has no verprop.";
      return;
    }
    if (not defined(self, &doprop)) {
      "Verb has no doprop.";
      return;
    }
    if (actor.location = nil) {
      "\^<<actor.subjthedesc>> <<actor.is>> nowhere!";
      return;
    }
    if (not defined(actor, &travelto)) {
      "\^<<actor.subjthdesc>> can't go anywhere on <<actor.possessivedesc>> own!";
      return;
    }
    
    /*
     * First call the verify method, if there is one. If it prints
     *   anything, the move is not allowed.
     */
    if (defined(actor.location, self.verprop)) {
      Outhide(true);
      actor.location.(self.verprop)(actor);
      if (Outhide(nil)) {
        actor.location.(self.verprop)(actor);
        return;
      }
    }

    /*
     * Here we see what the travel method evaluates to. Since we
     *   want to allow the programmer to say
     *
     *  goNorth = hill
     *
     * as a shorthand for
     *
     *  goNorth(actor) = { return hill; }
     *
     * we have the check the property type of the method before
     *   evaluating it.
     *
     * If the travel method evaluates to non-nil, we move the actor
     *   to that location. Otherwise the move is not allowed.
     *
     * If no travel method is defined, try the old-style method.
     *   (This is for backwards compatbility with old TADS games.)
     *
     * NOTE: DO NOT put nil goDirection intializers in any base
     *   classes! This will prevent the old-style methods from ever
     *   getting checked.
     *
     * If you need to define a goDirection method, you'll have to
     *   check for the old method directly.
     */
    if (defined(actor.location, self.doprop)) {
      local t;
      t := proptype(actor.location, self.doprop);
      if (t = 2) {                                                  /* Simple */
        dest := actor.location.(self.doprop);
      }
      else if (t = 5) { }                                          /* Nothing */
      else if (t = 6) {                                               /* Full */
        dest := actor.location.(self.doprop)(actor);
      }
      else {
        "\nERROR: Travelverb can't evaluate property!";
        return;
      }
    }
    else if (self.oldprop <> nil) {
      if (defined(actor.location, self.oldprop))
        dest := actor.location.(self.oldprop);
      else
        lose := true;
    }
    else
      lose := true;

    /*
     * If we've found no method for this direction, try to call the
     *   noexit method. If no such exit exists, try the goNowhere
     *   method. If that doesn't exist either, print a default "You
     *   can't go that way" message.
     *
     * Note that since noexit is sometimes used to mean "all exits"
     *   we need to try its return value as a possible destination.
     */
    if (lose) {
      if (defined(actor.location, &noexit))
        dest := actor.location.noexit;
      else if (defined(actor.location, &goNowhere))
        dest := actor.location.goNowhere(actor);
      else {
        "\^<<actor.subjthedesc>> can't go that way.";
        return;
      }
    }
    
    /*
     * Travel to destination unless it's nil.
     */
    if (dest = nil) {
      return;
    }
    else if (datatype(dest) <> 2) {
      "\nERROR: travel method (<<self.sdesc>>) does not return an object or nil.";
    }
    else
      actor.travelto(dest);     
  }
;

northVerb: Travelverb
  sdesc = "go north"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes north";
  }
  verb = 'north' 'go north' 'n' 'go n'
  verprop = &verGoNorth
  doprop = &goNorth
  oldprop = &north
;

southVerb: Travelverb
  sdesc = "go south"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes south";
  }
  verb = 'south' 'go south' 's' 'go s'
  verprop = &verGoSouth
  doprop = &goSouth
  oldprop = &south
;

eastVerb: Travelverb
  sdesc = "go east"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes east";
  }
  verb = 'east' 'go east' 'e' 'go e'
  verprop = &verGoEast
  doprop = &goEast
  oldprop = &east
;

westVerb: Travelverb
  sdesc = "go west"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes west";
  }
  verb = 'west' 'go west' 'w' 'go w'
  verprop = &verGoWest
  doprop = &goWest
  oldprop = &west
;

neVerb: Travelverb
  sdesc = "go northeast"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes northeast";
  }
  verb = 'northeast' 'go northeast' 'ne' 'go ne'
  verprop = &verGoNortheast
  doprop = &goNortheast
  oldprop = &ne
;

nwVerb: Travelverb
  sdesc = "go northwest"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes northwest";
  }
  verb = 'northwest' 'go northwest' 'nw' 'go nw'
  verprop = &verGoNorthwest
  doprop = &goNorthwest
  oldprop = &nw
;

seVerb: Travelverb
  sdesc = "go southeast"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes southeast";
  }
  verb = 'southeast' 'go southeast' 'se' 'go se'
  verprop = &verGoSoutheast
  doprop = &goSoutheast
  oldprop = &se
;

swVerb: Travelverb
  sdesc = "go southwest"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes southwest";
  }
  verb = 'southwest' 'go southwest' 'sw' 'go sw'
  verprop = &verGoSouthwest
  doprop = &goSouthwest
  oldprop = &sw
;

upVerb: Travelverb
  sdesc = "go up"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes up";
  }
  verb = 'up' 'go up' 'u' 'go u'
  verprop = &verGoUp
  doprop = &goUp
  oldprop = &up
;

downVerb: Travelverb
  sdesc = "go down"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes down";
  }
  verb = 'down' 'go down' 'd' 'go d'
  verprop = &verGoDown
  doprop = &goDown
  oldprop = &down
;

inVerb: Travelverb
  sdesc = "go in"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes in";
  }
  verb = 'in' 'go in' 'enter' 'go inside' 'go into' 'go downin'
  verprop = &verGoIn
  doprop = &goIn
  oldprop = &in
  doAction = 'Enter'
  requires = [[&cantouch]]
;

outVerb: Travelverb
  sdesc = "go out"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes out";
  }
  verb = 'out' 'go out' 'exit' 'go outof'
  verprop = &verGoOut
  doprop = &goOut
  oldprop = &out
  doAction = 'Exit'
  requires = [[&cantouch]]
;

upstreamVerb: Travelverb
  sdesc = "go upstream"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes upstream";
  }
  verb = 'upstream' 'go upstream' 'us' 'go us' 'ups' 'go ups'
  verprop = &verGoUpstream
  doprop = &goUpstream
  oldprop = nil
;

downstreamVerb: Travelverb
  sdesc = "go downstream"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "goes downstream";
  }
  verb = 'downstream' 'go downstream' 'ds' 'go ds'
  verprop = &verGoDownstream
  doprop = &goDownstream
  oldprop = nil
;

jumpVerb: Travelverb
  sdesc = "jump"
  verb = 'jump' 'jump over' 'jump off' 'jump out' 'jump in'
     'jump behind'
  verprop = &verGoJump
  doprop = &goJump
  oldprop = nil
;

/*
 * System verbs.
 * These can be used even when normal verbs are disabled.
 */
class Systemverb: Verb
;

quitVerb: Systemverb, Soloverb
  verb = 'quit'
  soloaction(actor) = {
    local yesno;
    score_and_rank();
    "\bDo you really want to quit? (YES or NO) > ";
    yesno := yorn();
    "\b";
    if (yesno = 1) {
      terminate();                             /* allow user good-bye message */
      quit();
    }
    else
      "Okay. ";
    abort;
  }
;

verboseVerb: Systemverb, Soloverb
  verb = 'verbose' 'wordy'
  soloaction(actor) = {
    "Okay, now in VERBOSE mode.\n";
    global.verbose := true;
    actor.location.lookaround(actor, true);
    abort;
  }
;

terseVerb: Systemverb, Soloverb
  verb = 'brief' 'terse'
  soloaction(actor) = {
    "Okay, now in TERSE mode.\n";
    global.verbose := nil;
    abort;
  }
;

scoreVerb: Systemverb, Soloverb
  verb = 'score' 'status'
  soloaction(actor) = {
    score_and_rank();
    abort;
  }
;

saveVerb: Systemverb, Soloverb
  verb = 'save'
  sdesc = "save"
  doAction = 'Save'
  requires = [[&canusealpha]]
  soloaction(actor) = {
    savegame();
    abort;
  }
;

restoreVerb: Systemverb, Soloverb
  verb = 'restore' 'load'
  sdesc = "restore"
  doAction = 'Restore'
  requires = [[&canusealpha]]
  soloaction(actor) = {
    local savefile;
    savefile := askfile('File to restore game from', ASKFILE_PROMPT_OPEN, FILE_TYPE_SAVE);
    if (savefile = nil or savefile = '')
      "Failed. ";
    else mainRestore(savefile);
    abort;
  }
;

scriptVerb: Systemverb, Soloverb
  verb = 'script'
  sdesc = "script"
  doAction = 'Script'
  requires = [[&canusealpha]]
  soloaction(actor) = {
    local scriptfile;
    scriptfile := askfile('File to write transcript to', ASKFILE_PROMPT_SAVE, FILE_TYPE_LOG);
    if (scriptfile = nil or scriptfile = '')
      "Failed. ";
    else {
      logging(scriptfile);
      "All text will now be saved to the script file. Type UNSCRIPT at any time to discontinue scripting.";
    }
    abort;
  }
;

unscriptVerb: Systemverb, Soloverb
  verb = 'unscript'
  soloaction(actor) = {
    logging(nil);
    "Script closed.\n";
    abort;
  }
;

restartVerb: Systemverb, Soloverb
  verb = 'restart'
  soloaction(actor) = {
    local yesno;
    while (true) {
      "Are you sure you want to start over? (YES or NO) > ";
      yesno := yorn();
      if (yesno = 1) {
        "\n";
        init_statusline();
        restart(initrestart, global.restarting);
        score_statusline();
        abort;
       }
       else if (yesno = 0) {
         "\nOkay.\n";
         abort;
       }
         }
  }
;

versionVerb: Systemverb, Soloverb
  verb = 'version'
  soloaction(actor) = {
    versioninfo();
    abort;
  }
;

undoVerb: Systemverb, Soloverb
  verb = 'undo'
  sdesc = "undo"
  soloaction(actor) = {
    self.undocommands(actor, 1);
  }
  doAction = 'Undo'                                          /* for "undo 12" */
  requires = [[&canusenumber]]
  undocommands(actor, turns) = {
    local i;
    if (turns < 1) {
      "Please specify the number of turns you want to undo, or just type \"undo\" to undo a single turn.";
      abort;
      return;
    }
    else if (turns > global.undowarning) {
      "You cannot redo turns that you undo.\nAre you sure you want to undo that many commands? (YES or NO) > ";
      if (yorn() = 0) {
        "Okay, never mind.";
        abort;
        return;
      }
    }
    for (i := 0; i <= turns; i++) {
      if (not undo())
        break;
    }
    if (i = 2)
      "One command undone. ";
    else if (i > 2)
      "<<i-1>> commands undone. ";
    if (i <= turns)
      "There is not enough undo information to undo any more commands.";
    if (i > 1) {
      "\b";
      actor.location.lookaround(actor, true);
      score_statusline();
    }
    abort;
  }
;

spaceVerb: Systemverb, Soloverb
  sdesc = "space"
  verb = 'space'
  soloaction(actor) = {
    if (global.paragraphspacing) {
      "Now single-spacing text.";
      global.paragraphspacing := nil;
    }
    else {
      "Now double-spacing text.";
      global.paragraphspacing := true;
    }
    abort;
  }
;

indentVerb: Systemverb, Soloverb
  sdesc = "indent"
  verb = 'indent'
  soloaction(actor) = {
    if (global.indent) {
      "Paragraph indentation now off.";
      global.indent := nil;
    }
    else {
      "Paragraph indentation now on.";
      global.indent := true;
    }
    abort;
  }
;

deterministicverb: Systemverb, Soloverb
  verb = 'deterministic'
  soloaction(actor) = {
    "(Deterministic mode)\n";
    global.nondeterministic := nil;
    abort;
  }
;

/*
 * This verb loops through the class 'versionTag' and displays the 'id'
 *   property.
 */
sourcesVerb: Systemverb, Soloverb
  verb = 'sources' 'source'
  sdesc = "sources"
  desc = self.sdesc
  soloaction(actor) = {
    local o;
    for (o := firstobj(versionTag); o <> nil; o.id, o := nextobj(o, versionTag));
    abort;
  }
;

/*
 * This verb will give credit where it is due for modules that have been
 *   made available by others.
 */
creditsVerb: Systemverb, Soloverb
  verb = 'credits' 'credit'
  sdesc = "credits"
  desc = self.sdesc
  soloaction(actor) = {
    local o, ap;
    local alist := [];
    local flist := [];
    for (o := firstobj(versionTag); o <> nil; o := nextobj(o, versionTag)) {
      if (proptype(o, &author) = 3 and proptype(o, &func) = 3) {
        ap := find(alist, o.author);
        if (ap = nil) {
          alist += o.author; flist += o.func;
        }
        else { flist[ap] += ', ' + o.func; }
      }
    }
    if (car(alist)) {
      self.credit_header;
      self.credit_list(alist, flist);
      self.credit_trailer;
    }
    abort;
  }
  credit_header = "The following modules were provided by TADS developers who were prepared to share their work with others:"
  credit_trailer = "\bIf you are a TADS developer, please consider doing the same. All the above mentioned modules should be available for ftp from the interactive fiction archive maintained by \(Volker Blasius\) on \(ftp.gmd.de\)."
  credit_list(alist, flist) = {
    local modules, f, l, p;
    while (car(alist)) {
      f := car(flist); flist := cdr(flist); l := 0;
      modules := '';
      while (true) {
        p := find(f, ', ');
        if (p <> nil) {
          if (l++ > 0) modules += ', ';
          modules += substr(f, 1, p-1);
          f := substr(f, p+1, length(f));
          continue;
        }
        if (l++ > 0) modules += ' and ';
        modules += f;
        break;
      }
      self.credit_entry(car(alist), modules, l);
      alist := cdr(alist);
    }
  }
  credit_entry(author, modules, n) =
    "\b\^<<modules>> <<n > 1 ? "were" : "was">> provided by \(<<author>>\).\n"
;

/*
 * Autosaving verbs
 */
AutosaveonVerb: Systemverb, Soloverb
  verb = 'autosave on'
  sdesc = "autosave on"
  soloaction(actor) = {
    if (global.autosavestatus)
      "\b*** Autosave is already on ***\n";
    else
    {
      "\b*** Autosave is now on. Game will be saved every ";
      say( global.turnsbetweensaves );
      " commands entered ***\n";
      global.lastsave := global.ticks;
      global.autosavestatus := true;
      setdaemon(autosave, nil);
    }
    abort;
  }
;

AutosaveoffVerb: Systemverb, Soloverb
  verb = 'autosave off'
  sdesc = "autosave off"
  soloaction(actor) = {
    if (global.autosavestatus) {
      "\b*** Autosave is now turned off ***\n";
      remdaemon(autosave, nil);
      global.autosavestatus := nil;
    }
    else "\b*** Autosave is already turned off ***\n";
    abort;
  }
;

AutosaveVerb: Systemverb
  verb = 'autosave'
  sdesc = "autosave"
  doAction = 'Autosave'
  action(actor) =
  {
    "\bAutosave is currently <<global.autosavestatus ? "on" :
     "off">> and set to save every ";
    say (global.turnsbetweensaves);
    " turns.\bUse 'autosave on' or 'autosave off' to turn the autosave function on or off, and use 'autosave' followed by a value to set the turns between autosaves.\nDuring the first autosave you will be prompted for a filename, which will be subsequently used.\n";
    abort;
  }
;

footnoteVerb: Verb
  sdesc = "get footnote information for"
  desc(obj) = {
    if (obj.isplural) self.sdesc;
    else "gets footnote information for";
  }
  verb = 'footnote' 'note'
  doAction = 'Footnote'
  requires = [[&canusenumber]]
;

/*
 * Playtesting verbs
 *
 * Code in Verb and Soloverb guarantees that no Debugverb will work at run-
 *   time. However, the vocab for these verbs will still be recognized, so
 *   be careful to avoid conflicts with real verbs!
 */
class Debugverb: Verb
  allok = true
;

qVerb: Systemverb, Soloverb
  sdesc = "fastquit"
  verb = 'q'
  soloaction(actor) = {
    if (global.playtesting)
      quit();
    else
      quitVerb.soloaction(actor);
  }
;

locateVerb: Debugverb, Systemverb
  sdesc = "locate"
  verb = 'locate'
  doAction = 'Locate'
;

listcontentsVerb: Soloverb, Debugverb, Systemverb
  sdesc = "list contents"
  verb = 'listcontents' 'contents' 'list'
  doAction = 'Listcontents'
  soloaction(actor) = {
    actor.location.doListcontents(actor);
  }
;

dieVerb: Debugverb, Systemverb, Soloverb
  sdesc = "commit suicide"
  verb = 'die'
  soloaction(actor) = { die(); }
;

hungryVerb: Debugverb, Systemverb, Soloverb
  sdesc = "become (un)hungry"
  verb = 'hungry'
  soloaction(actor) = {
    if (actor.turnsuntilstarvation < 40)
      actor.turnsuntilstarvation := actor.mealtime;
    else
      actor.turnsuntilstarvation := 40;
  }
;

thirstyVerb: Debugverb, Systemverb, Soloverb
  sdesc = "become (un)thirsty"
  verb = 'thirsty'
  soloaction(actor) = {
    if (actor.turnsuntildehydration < 40)
      actor.turnsuntildehydration := actor.drinktime;
    else
      actor.turnsuntildehydration := 40;
  }
;

sleepyVerb: Debugverb, Systemverb, Soloverb
  sdesc = "become (un)sleepy"
  verb = 'sleepy'
  soloaction(actor) = {
    if (actor.turnsuntilsleep < 40)
      actor.turnsuntilsleep := actor.sleeptime;
    else
      actor.turnsuntilsleep := 40;
  }
;

gimmeVerb: Debugverb, Systemverb
  sdesc = "gimme"
  verb = 'gimme'
  doAction = 'Gimme'
;

warptoVerb: Debugverb, Systemverb
  sdesc = "warpto"
  verb = 'warp to'
  doAction = 'Warpto'
  validDo(actor, obj, seqno) = {
    if (self.playtestcheck)
      return nil;
    else
      return isclass(obj, Room) ? true : nil;
  }
  validIo(actor, obj, seqno) = {
    return isclass(obj, Room) ? true : nil;
  }
;

knowVerb: Debugverb, Systemverb
  sdesc = "know"
  verb = 'know'
  doAction = 'Know'
;

lightroomVerb: Debugverb, Systemverb, Soloverb
  sdesc = "light room"
  verb = 'lightroom'
  soloaction(actor) = {
    "Let there be light!";
    actor.location.lighten;
    "\b";
    actor.location.lookaround(actor, true);
  }
;

darkroomVerb: Debugverb, Systemverb, Soloverb
  sdesc = "dark room"
  verb = 'darkroom'
  soloaction(actor) = {
    "Let there be dark!";
    actor.location.darken;
    "\b";
    actor.location.lookaround(actor, true);
  }
;

debugreachVerb: Debugverb, Systemverb, Soloverb
  sdesc = "debug reach"
  verb = 'debugreach' 'reachdebug' 'debug reach' 'reach debug'
  soloaction(actor) = {
    global.debugreach := not global.debugreach;
    "Reach debugging is now ";
    say(global.debugreach ? 'ON' : 'OFF');
    ".";
    abort;
  }
;

debugtraceVerb: Debugverb, Systemverb, Soloverb
  on = nil
  verb = 'debugtrace' 'debug trace'
  soloaction(actor) = {
    if (self.on) {
      "debugTrace is now OFF.\n";
      debugTrace(1, nil);
    }
    else {
      "debugTrace is now ON.\n";
      debugTrace(1, true);
    }
    self.on := not self.on;
    abort;
  }
;

debugVerb: Debugverb, Systemverb, Soloverb
  verb = 'debug'
  soloaction(actor) = {
    if (debugTrace())
      "I don't see any bugs here.";
    abort;
  }
;

weightVerb: Debugverb, Systemverb
  sdesc = "weight"
  verb = 'weight'
  doAction = 'Weight'
;

bulkVerb: Debugverb, Systemverb
  sdesc = "bulk"
  verb = 'bulk'
  doAction = 'Bulk'
;

/*
 * Score notification verbs
 */
notifyonVerb: Systemverb, Soloverb
  verb = 'notify on'
  sdesc = "notify on"
  soloaction(actor) = {
    if (not global.silentincscore)
      "\b*** Score notification is already on ***\n";
    else
    {
      "\b*** Score notification is now turned on ***\n";
      global.silentincscore := nil;
    }
    abort;
  }
;

notifyoffVerb: Systemverb, Soloverb
  verb = 'notify off'
  sdesc = "notify off"
  soloaction(actor) = {
    if (not global.silentinscore) {
      "\b*** Score notification is now turned off ***\n";
      global.silentincscore := true;
    }
    else "\b*** Score notification is already turned off ***\n";
    abort;
  }
;

notifyVerb: Systemverb, Soloverb
  verb = 'notify'
  sdesc = "notify"
  soloaction(actor) = {
    "\b*** Score notification is currently turned 
    <<global.silentincscore ? "off" : "on">> ***\n";
    abort;
  }
;

talkVerb:  Verb
  verb = 'talk' 'talk to'
  sdesc = "talk"
  doAction = 'Talk'
;
