/* io.c */

#include "const.h"
#include "type.h"
#include "var.h"

#define NDEV 3
#define KEYDEV 0
#define TTYDEV 1
#define SCRDEV 2

#define IOCCOFF 0		/* cursor on (these s.b. in header file) */
#define IOCCON  1		/* cursor off */
#define IOCUSER 2		/* copy user state */

struct dev
{
	bool_t inopen;
	bool_t outopen;
	pfv_t close;
	pfi_t in;
	pfv_t ioctl;
	pfv_t open;
	pfv_t out;
};

PRIVATE struct dev devtable[NDEV] =
{
	{ TRUE, FALSE, kbdclose, kbdin, kbdioctl, kbdopen, kbdout },
	{ TRUE, FALSE, ttyclose, ttyin, ttyioctl, ttyopen, ttyout },
	{ FALSE, TRUE, scrclose, scrin, scrioctl, scropen, scrout }
};

PRIVATE bool_t forceupper;
PRIVATE bool_t pause = FALSE;
PRIVATE bool_t someupper = TRUE;
PRIVATE count_t stringcount = 0;
PRIVATE char *string_ptr = NULL;	/* stringptr ambiguous at 8th char */
PRIVATE char *stringstart = NULL;

/* cancel input device */

PUBLIC void can_input( devptr )
struct dev *devptr;
{
	if ( devptr->inopen )
	{
		if ( !devptr->outopen )
			(*devptr->close)();
		devptr->inopen = FALSE;
	}
}

/* cancel output device */

PUBLIC void can_output( devptr )
struct dev *devptr;
{
	if ( devptr->outopen )
	{
		if ( !devptr->inopen )
			(*devptr->close)();
		devptr->outopen = FALSE;
	}
}

/* cancel keyboard device */

PUBLIC void can_keyboard()
{
	can_input( &devtable[KEYDEV] );
}

/* cancel screen device */

PUBLIC void can_screen()
{
	can_output( &devtable[SCRDEV] );
}

/* cancel tty input device */

PUBLIC void can_itty()
{
	can_input( &devtable[TTYDEV] );
}

/* cancel tty output device */

PUBLIC void can_otty()
{
	can_output( &devtable[TTYDEV] );
}

/* close all devices */

PUBLIC void closeio()
{
	struct dev *devptr;

	for ( devptr = &devtable[0]; devptr < &devtable[NDEV]; ++devptr )
		if ( devptr->inopen || devptr->outopen )
			(*devptr->close)();
}

/* close string device */

PUBLIC void closestring()
{
	stringcount = 0;
	stringstart = string_ptr = NULL;
}

/* enable input device */

PUBLIC void enab_input( devptr )
struct dev *devptr;
{
	if ( !devptr->inopen )
	{
		if ( !devptr->outopen )
			(*devptr->open)();
		devptr->inopen = TRUE;
	}
}

/* enable output device */

PUBLIC void enab_output( devptr )
struct dev *devptr;
{
	if ( !devptr->outopen )
	{
		if ( !devptr->inopen )
			(*devptr->open)();
		devptr->outopen = TRUE;
	}
}

/* enable keyboard device */

PUBLIC void enab_keyboard()
{
	enab_input( &devtable[KEYDEV] );
}

/* enable screen device */

PUBLIC void enab_screen()
{
	enab_output( &devtable[SCRDEV] );
}

/* enable tty input device */

PUBLIC void enab_itty()
{
	enab_input( &devtable[TTYDEV] );
}

/* enable tty output device */

PUBLIC void enab_otty()
{
	enab_output( &devtable[TTYDEV] );
}

PUBLIC void flipcase()
{
	someupper = TRUE - someupper;
}

/* get 8 bits current instruction pointer and advance pointer */

PUBLIC u8_pt get8()
{
	u8_pt temp;

	temp = peek8( &uptr );
	++uptr.off;
	return temp;
}

/* get 16 bits from current instruction pointer and advance pointer */

PUBLIC u16_t get16()
{
	u16_t temp;

	temp = peek16( &uptr );
	uptr.off += 2;
	return temp;
}

/* get 32 bits from current instruction pointer and advance pointer */

PUBLIC u32_t get32()
{
	u32_t temp;

	temp = peek32( &uptr );
	uptr.off += 4;
	return temp;
}

/* input char from one of currently open input devices */

PUBLIC char16_t inchar()
{
	char16_t ch;

	ioctlio( IOCCON );
	while ( (ch = testchar()) == EOF )
		;
	ioctlio( IOCCOFF );
	return ch;
}

/* perform ioctl for all devices */

PUBLIC void ioctlio( command )
bool_pt command;
{
	struct dev *devptr;

	for ( devptr = &devtable[0]; devptr < &devtable[NDEV]; ++devptr )
		if ( devptr->inopen || devptr->outopen )
			(*devptr->ioctl)( command );
}

/* convert char to lower case */

PUBLIC char_pt mytolower( ch )
char_pt ch;
{
	if ( ch >= 'A' && ch <= 'Z' )
	/*
	if ( (ch - 'A') <= 'Z' - 'A' )
	*/
		ch += 'a' - 'A';
	return ch;
}

/* open all devices */

PUBLIC void openio()
{
	struct dev *devptr;

	for ( devptr = &devtable[0]; devptr < &devtable[NDEV]; ++devptr )
		if ( devptr->inopen || devptr->outopen )
			(*devptr->open)();
	pause = FALSE;
}

/* open string device */

PUBLIC void openstring( string )
char *string;
{
	stringcount = 0;
	stringstart = string_ptr = string;
}

/* print 2 spaces */

PUBLIC void out2space()
{
	outspace();
	outspace();
}

/* print char to currently open output devices */

PUBLIC void outbyte( byte )
char_pt byte;
{
	struct dev *devptr;

	if ( forceupper && byte >= 'a' && byte <= 'z' )
		byte += 'A' - 'a';
	if ( string_ptr != NULL )
	{
		if ( (*string_ptr++ = byte) == '\t' )
			stringcount = 8 * (stringcount / 8 + 1);
		else
			++stringcount;
	}
	else
		for ( devptr = &devtable[0]; devptr < &devtable[NDEV]; ++devptr )
			if ( devptr->outopen )
				(*devptr->out)( byte );
}

/* print colon */

PUBLIC void outcolon()
{
	outbyte( ':' );
}

/* print comma */

PUBLIC void outcomma()
{
	outbyte( ',' );
}

/* get 8 bit offset and print in hex */

PUBLIC void outget8()
{
	outh8( get8() );
}

/* get 16 bit offset and print in hex */

PUBLIC void outget16()
{
	outh16( get16() );
}

/* print 4 bits hex */

PUBLIC void outh4( num )
u4_pt num;
{
	static char hexdigits[] = "0123456789abcdef";

	forceupper = someupper;
	outbyte( hexdigits[num % 16] );
	forceupper = FALSE;
}

/* print 8 bits hex */

PUBLIC void outh8( num )
u8_pt num;
{
	outh4( num / 16 );
	outh4( num );
}

/* print 8 bits hex and 1 space */

PUBLIC void outh8s( num )
u8_pt num;
{
	outh8( num );
	outspace();
}

/* print 16 bits hex */

PUBLIC void outh16( num )
u16_t num;
{
	outh8( num / 256 );
	outh8( num );
}

/* print 32 bits hex */

PUBLIC void outh32( num )
u32_t num;
{
	outh16( (u16_t) (num >> 16) );
	outh16( (u16_t) num );
}

#ifdef UNUSED_FUNCTIONS

/* print offset, hex format */

PUBLIC void outhex( num )
offset_t num;
{
#ifdef HEXSTARTCHAR
	if ( num >= 10 )
		outbyte( HEXSTARTCHAR );
#endif
	outhexdigs( num );
#ifdef HEXENDCHAR
	if ( num >= 10 )
		outbyte( HEXENDCHAR );
#endif
}

#endif

#ifdef UNUSED_FUNCTIONS

/* print 8 bit offset, hex format */

PUBLIC void outhex8( num )
u8_pt num;
{
#ifdef HEXSTARTCHAR
	outbyte( HEXSTARTCHAR );
#endif
	outh8( num );
#ifdef HEXENDCHAR
	outbyte( HEXENDCHAR );
#endif
}

#endif

#ifdef UNUSED_FUNCTIONS

/* print 16 bit offset, hex format */

PUBLIC void outhex16( num )
u16_t num;
{
#ifdef HEXSTARTCHAR
	outbyte( HEXSTARTCHAR );
#endif
	outh16( num );
#ifdef HEXENDCHAR
	outbyte( HEXENDCHAR );
#endif
}

#endif

#ifdef UNUSED_FUNCTIONS

/* print offset, hex format with digits only (no hex designator) */

PUBLIC void outhexdigs( num )
register offset_t num;
{
	if ( num >= 16 )
	{
		outhexdigs( num / 16 );
		num %= 16;
	}
	outhdigit( num );
}

#endif

#ifdef UNUSED_FUNCTIONS

/* print character, then newline */

PUBLIC void outnbyte( byte )
char_pt byte;
{
	outbyte( byte );
	outnl();
}

#endif

#ifdef UNUSED_FUNCTIONS

/* print offset, hex format, then newline */

PUBLIC void outnhex( num )
offset_t num;
{
	outhex( num );
	outnl();
}

#endif

/* print newline */

PUBLIC bool_pt outnl()
{
	char_pt ch;

	outstr( "\015\012" );
	if ( !pause )
	{
		if ( (ch = testchar()) == CAN )
			return FALSE;
		if ( ch != XOFF )
			return TRUE;
	}
	pause = FALSE;
	if ( (ch = inchar()) == XOFF )
		pause = TRUE;
	return ch - CAN;
}

/* print string, then newline */

PUBLIC bool_pt outnstr( s )
char *s;
{
	outstr( s );
	return outnl();
}

#ifdef UNUSED_FUNCTIONS

/* print plus sign */

PUBLIC void outplus()
{
	outbyte( '+' );
}

#endif

/* print segmented address */

PUBLIC void outsegadr( adr )
struct adr *adr;
{
	outh16( adr->seg );
	outcolon();
	if ( db_processor == 386 )
		outh32( adr->off );
	else
		outh16( (u16_t) adr->off );
}

/* print 32 bit segmented address and 2 spaces */

PUBLIC void outssegadr( adr )
struct adr *adr;
{
	outsegadr( adr );
	out2space();
}

#ifdef UNUSED_FUNCTIONS

/* print signed, hex format */

PUBLIC void outshex( num )
offset_t num;
{
	if ( (soffset_t) num < 0 )
	{
		outbyte( '-' );
		num = -num;
	}
	outhex( num );
}

#endif

/* print space */

PUBLIC void outspace()
{
	outbyte( ' ' );
}

/* print string */

PUBLIC void outstr( s )
register char *s;
{
	while ( *s )
		outbyte( *s++ );
}

/* print tab */

PUBLIC void outtab()
{
	outbyte( '\t' );
}

/* print string, perhaps converting case to upper */

PUBLIC void outustr( s )
register char *s;
{
	forceupper = someupper;
	while ( *s )
		outbyte( *s++ );
	forceupper = FALSE;
}

#ifdef UNUSED_FUNCTIONS

/* print offset, decimal format */

PUBLIC void outudec( num )
register offset_t num;
{
	if ( num >= 10 )
	{
		register offset_t reduction;

		outudec( reduction = num / 10 );
		num -= 10 * reduction;
	}
	outbyte( (char_pt) num + '0' );
}

#endif

/* set tty parameters from user */

PUBLIC void set_tty()
{
	ioctlio( IOCUSER );
}

/* flip to the db screen (flip to user by closing */

PUBLIC void show_db_screen()
{
	if ( devtable[SCRDEV].outopen )
		(*devtable[SCRDEV].open)();
}

/* flip to the user screen */

PUBLIC void show_user_screen()
{
	if ( devtable[SCRDEV].outopen )
		(*devtable[SCRDEV].close)();
}

/* return current offset of string device */

PUBLIC count_t stringpos()
{
	return string_ptr - stringstart;
}

/* return current "tab" spot of string device */

PUBLIC count_t stringtab()
{
	return stringcount;
}

/* test for char from one of currently open input devices */

PUBLIC char_pt testchar()
{
	char_pt ch;
	struct dev *devptr;

	for ( devptr = &devtable[0]; devptr < &devtable[NDEV]; ++devptr )
		if ( devptr->inopen && (ch = (*devptr->in)()) != EOF )
			return ch;
	return EOF;
}

#ifdef LINT

/* prototypes, perhaps out of date */
/* prototypes for assembler functions */

/* ARGSUSED */
PUBLIC segment_t codeseg() { return 1; }
/* ARGSUSED */
PUBLIC segment_t dataseg() { return 1; }
/* ARGSUSED */
PUBLIC unsigned get_processor() { return 1; }
/* ARGSUSED */
PUBLIC u8_pt inportb( port ) port_t port; { return 1; }
/* ARGSUSED */
PUBLIC void oportb( port, val ) port_t port; u8_pt val; {}
/* ARGSUSED */
PUBLIC u8_pt peek8( adr ) struct adr *adr; { return 1; }
/* ARGSUSED */
PUBLIC u8_pt peekb( seg, off ) segment_t seg; u8_t *off; { return 1; }
/* ARGSUSED */
PUBLIC u16_t peek16( adr ) struct adr *adr; { return 1; }
/* ARGSUSED */
PUBLIC u16_t peekw( seg, off ) segment_t seg; u16_t *off; { return 1; }
/* ARGSUSED */
PUBLIC u32_t peek32( adr ) struct adr *adr; { return 1; }
/* ARGSUSED */
PUBLIC void poke8( adr, value ) struct adr *adr; u8_pt value; {}
/* ARGSUSED */
PUBLIC void pokeb( seg, off, val ) segment_t seg, u8_t *off, u8_pt val; {}
/* ARGSUSED */
PUBLIC void poke16( adr, value ) struct adr *adr; u16_t value; {}
/* ARGSUSED */
PUBLIC void pokew( seg, off, val ) segment_t seg, u16_t *off, u16_t val; {}
/* ARGSUSED */
PUBLIC void poke32( adr, value ) struct adr *adr; u32_t value; {}
PUBLIC void scrclose() {}
PUBLIC char_pt scrin() {}
/* ARGSUSED */
PUBLIC void scrioctl( c ) char_pt c; {}
PUBLIC void scropen() {}
/* ARGSUSED */
PUBLIC void scrout( c ) char_pt c; {}
PUBLIC void symswap();
PUBLIC void symswap( left, right, tableseg, length )
struct nlist *left; struct nlist *right; segment_t tableseg; unsigned length;
{}
PUBLIC void ttyclose() {}
PUBLIC char_pt ttyin() { return (char) 1; }
/* ARGSUSED */
PUBLIC void ttyioctl( c ) char_pt c; {}
PUBLIC void ttyopen() {}
/* ARGSUSED */
PUBLIC void ttyout( c ) char_pt c; {}

/* prototypes for library functions */

/* ARGSUSED */
PUBLIC char *memcpy( t, s, n ) char *s; char *t; unsigned n; { return t; }
/* ARGSUSED */
PUBLIC int strlen( s ) char *s; { return 1; }

/* prototypes for Minix kernel functions */

PUBLIC segment_t codeseg() { return 1; }
PUBLIC void get_con_state() {}
PUBLIC void map_dmp() {}
PUBLIC void p_dmp() {}
PUBLIC void reset_con_state() {}

#endif /* LINT */
