#include "ns.h"
#include <ctype.h>

/*
 * This code implements a simple extension to the NaviServer.
 * 
 * It defines a new operation called /SqlReport that will  find the file
 * corresponding the URL following /SqlReport  and fill in any <SELECTSQL>
 * elements with printed tables which result from executing the SQL statement
 * specified in the <SELECTSQL> tag.
 * 
 * For example, /SqlReport/reports/perms.html will look up  the file
 * corresponding to URL /reports/perms.html.
 * 
 * The SELECTSQL element may have 2 attributes. SRC specifies  the SQL of a
 * query and MAX specifies the maximum number of  rows. For example,
 * 
 * <SELECTSQL SRC="select  from ns_permissions;" MAX=40>
 * 
 * will build a table of up to 40 rows of the ns_permissions table.
 * 
 * NOTE:  This code is unsupported and for example purposes only.
 * 
 */


static char    *UpcaseString(char *source);
static Ns_Set  *TagToSet(char *source);
static void     SqlReport(Ns_OpContext context, Ns_Conn *conn);


void DllExport
SqlReportInit(void)
{
    Ns_RegisterOp("GET", "/SqlReport", SqlReport, NULL, NULL, 0);
}


static void
SqlReport(Ns_OpContext context, Ns_Conn *conn)
{
    Ns_DString      html, filename, tag;
    FILE           *fp;
    int		    i;
    char            c;

    Ns_DStringInit(&filename);
    Ns_DStringInit(&html);
    Ns_DStringInit(&tag);

    Ns_UrlToFile(&filename, Ns_SkipUrl(conn->request, 1));

    fp = fopen(filename.string, "r");
    if (fp == NULL) {
	Ns_ReturnNotFound(conn);
	goto done;
    }
    while ((i = getc(fp)) != EOF) {
	c = (char) i;
	if (c == '<') {
	    Ns_DStringTrunc(&tag, 0);
	    Ns_DStringNAppend(&tag, &c, 1);
	    while ((i = getc(fp)) != EOF) {
		c = (char) i;
		Ns_DStringNAppend(&tag, &c, 1);
		if (c == '>') {
		    break;
		}
	    }

	    if ((tag.string[1] != '/')
		&& (strlen(tag.string) >= 10)
		&& ((strncmp(tag.string, "<SELECTSQL", 9) == 0)
		    || strncmp(tag.string, "<selectsql", 9) == 0)) {

		Ns_Set         *tagset;
		char           *sql;

		tagset = TagToSet(tag.string);
		if ((tagset == NULL)
		    || ((sql = Ns_SetGet(tagset, "SRC")) == NULL)) {
		    Ns_ReturnError(conn, 500, "Malformed SELECTSQL;"
			" could not find SRC");
		    Ns_SetFree(tagset);
		    goto done;
		} else {
		    char           *maxrowsbuf;
		    int             maxrows;

		    maxrowsbuf = Ns_SetGet(tagset, "MAX");
		    if (maxrowsbuf == NULL) {
			maxrows = 40;
		    } else {
			maxrows = atoi(maxrowsbuf);
		    }

		    Ns_DbPrintTable(&html, conn, sql, maxrows);
		    Ns_SetFree(tagset);
		}
	    } else {
		Ns_DStringAppend(&html, tag.string);
	    }
	} else {
	    Ns_DStringNAppend(&html, &c, 1);
	}
    }
    Ns_ReturnHtml(conn, 200, html.string, html.length);
done:
    Ns_DStringFree(&filename);
    Ns_DStringFree(&html);
    Ns_DStringFree(&tag);
}


/*
 * TagToSet - Convert an HTML tag to an Ns_Set.
 * 
 * Returns NULL on closing tags (e.g., </A>).
 */
static Ns_Set  *
TagToSet(char *src)
{
    Ns_Set         *set;
    Ns_DString      dsbuf;

    if ((strlen(src) < 2)
	|| (src[1] == '/')) {
	return NULL;
    }
    Ns_DStringInit(&dsbuf);

    while (*src != '\0') {
	if ((*src == '>') || (*src == ' ')) {
	    break;
	}
	Ns_DStringNAppend(&dsbuf, src, 1);
	src++;
    }

    UpcaseString(dsbuf.string);
    set = Ns_SetCreate(dsbuf.string);

    while (1) {
	while ((*src != '\0')
	    && (isspace(*src))) {
	    src++;
	}

	if ((*src == '\0') || (*src == '>')) {
	    goto done;
	}
	Ns_DStringTrunc(&dsbuf, 0);

	while ((*src != '\0')
	    && !isspace(*src)
	    && (*src != '=')) {
	    Ns_DStringNAppend(&dsbuf, src, 1);
	    src++;
	}

	UpcaseString(dsbuf.string);

	while ((*src != '\0')
	    && (isspace(*src))) {
	    src++;
	}

	if (*src == '=') {
	    Ns_DString      dsvalue;

	    src++;

	    Ns_DStringInit(&dsvalue);

	    while ((*src != '\0')
		&& (isspace(*src))) {
		src++;
	    }
	    if (*src == '\"') {		/* " */
		src++;
		while ((*src != '\0')
		    && (*src != '\"')) {/* " */
		    Ns_DStringNAppend(&dsvalue, src, 1);
		    src++;
		}
		if (*src == '\"') {	/* " */
		    src++;
		}
	    } else {
		while ((*src != '\0')
		    && !isspace(*src)) {
		    Ns_DStringNAppend(&dsvalue, src, 1);
		    src++;
		}
	    }
	    Ns_SetPut(set, dsbuf.string, dsvalue.string);
	    Ns_DStringFree(&dsvalue);
	} else {
	    Ns_SetPut(set, dsbuf.string, NULL);
	}
    }
done:
    Ns_DStringFree(&dsbuf);
    return set;
}


static char    *
UpcaseString(char *source)
{
    char           *pos;

    pos = source;

    while (*pos != '\0') {
	if (isalpha(*pos)) {
	    *pos = toupper(*pos);
	}
	pos++;
    }
    return source;
}
