/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  PreparePayResultType,
  succeedOrThrow,
  TalerMerchantInstanceHttpClient,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

/**
 * Test the wallet-core payment API, especially that repeated operations
 * return the expected result.
 */
export async function runPaymentIdempotencyTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, merchant, merchantAdminAccessToken } =
    await createSimpleTestkudosEnvironmentV3(t);

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  const merchantClient = new TalerMerchantInstanceHttpClient(
    merchant.makeInstanceBaseUrl(),
  );

  // Set up order.

  const orderResp = succeedOrThrow(
    await merchantClient.createOrder(merchantAdminAccessToken, {
      order: {
        summary: "Buy me!",
        amount: "TESTKUDOS:5",
        fulfillment_url: "taler://fulfillment-success/thx",
      },
    }),
  );

  let orderStatus = succeedOrThrow(
    await merchantClient.getOrderDetails(merchantAdminAccessToken, orderResp.order_id),
  );

  t.assertTrue(orderStatus.order_status === "unpaid");

  const talerPayUri = orderStatus.taler_pay_uri;

  // Make wallet pay for the order

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: orderStatus.taler_pay_uri,
    },
  );

  const preparePayResultRep = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: orderStatus.taler_pay_uri,
    },
  );

  t.assertTrue(
    preparePayResult.status === PreparePayResultType.PaymentPossible,
  );
  t.assertTrue(
    preparePayResultRep.status === PreparePayResultType.PaymentPossible,
  );

  const proposalTransactionId = preparePayResult.transactionId;

  const confirmPayResult = await walletClient.call(
    WalletApiOperation.ConfirmPay,
    {
      transactionId: proposalTransactionId,
    },
  );

  console.log("confirm pay result", confirmPayResult);

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  // Check if payment was successful.

  orderStatus = succeedOrThrow(
    await merchantClient.getOrderDetails(merchantAdminAccessToken, orderResp.order_id),
  );

  t.assertTrue(orderStatus.order_status === "paid");

  const preparePayResultAfter = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri,
    },
  );

  console.log("result after:", preparePayResultAfter);

  t.assertTrue(
    preparePayResultAfter.status === PreparePayResultType.AlreadyConfirmed,
  );
  t.assertTrue(preparePayResultAfter.paid === true);

  await t.shutdown();
}

runPaymentIdempotencyTest.suites = ["wallet"];
