--
-- This file is part of TALER
-- Copyright (C) 2014--2022 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_refresh_transfer_keys(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_transfer_keys';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(rtc_serial BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',melt_serial_id INT8 PRIMARY KEY'
      ',transfer_pub BYTEA NOT NULL CHECK(LENGTH(transfer_pub)=32)'
      ',transfer_privs BYTEA NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (melt_serial_id)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Transfer keys of a refresh operation (the data revealed to the exchange).'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'needed for exchange-auditor replication logic'
    ,'rtc_serial'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the refresh commitment (rc) of the operation.'
    ,'melt_serial_id'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'transfer public key for the gamma index'
    ,'transfer_pub'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'array of TALER_CNC_KAPPA-1 transfer private keys that have been revealed, with the gamma entry being skipped'
    ,'transfer_privs'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_refresh_transfer_keys(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_transfer_keys';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_rtc_serial_key'
    ' UNIQUE (rtc_serial)'
  );
END
$$;


CREATE FUNCTION foreign_table_refresh_transfer_keys()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'refresh_transfer_keys';
BEGIN
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || 'foreign_melt_serial_id'
    ' FOREIGN KEY (melt_serial_id)'
    ' REFERENCES refresh_commitments (melt_serial_id) ON DELETE CASCADE'
  );
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('refresh_transfer_keys'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('refresh_transfer_keys'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('refresh_transfer_keys'
    ,'exchange-0002'
    ,'foreign'
    ,TRUE
    ,FALSE);
