/*  swinstall_lib.c -- the top-level routines of swinstall

 Copyright (C) 2004,2005,2006,2007 Jim Lowe
 All Rights Reserved.
  
 COPYING TERMS AND CONDITIONS:
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
*/

#define FILENEEDDEBUG 1
#undef FILENEEDDEBUG

#include "swuser_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "strob.h"
#include "cplob.h"
#include "vplob.h"
#include "swlib.h"
#include "usgetopt.h"
#include "ugetopt_help.h"
#include "swcommon0.h"
#include "swcommon.h"
#include "swparse.h"
#include "swfork.h"
#include "swgp.h"
#include "etar.h"
#include "swssh.h"
#include "progressmeter.h"
#include "swevents.h"
#include "to_oct.h"
#include "tarhdr.h"
#include "swinstall.h"
#include "swheader.h"
#include "swheaderline.h"
#include "swicat.h"
#include "strar.h"
#include "swi.h"
#include "atomicio.h"
#include "shlib.h"
#include "swutilname.h"
#include "swproglib.h"

extern struct g_pax_read_command g_pax_read_commands[];

#define SWBIS_DEFAULT_REVISION "0.0"
		
static
int
construct_controlsh_script(GB * G, STROB * buf, SWI * swi)
{
	char * tag;
	STROB * tmp;
	STROB * taglist;

	taglist = strob_open(100);
	tmp = strob_open(100);
	strob_strcpy(buf, "");
	swicat_write_auto_comment(buf, "control.sh");
	swicat_env(buf, swi, NULL, NULL);

	swicat_construct_controlsh_taglist(swi, "*", taglist);

	strob_sprintf(buf, STROB_DO_APPEND,
		"SWBCS_TAGLIST=\"%s\"\n"
		"case \"$#\" in\n"
		"	2)\n"
		"		;;\n"
		"	*)\n"
		"		echo \"usage: control.sh selection tag\" 1>&2\n"
		"		exit 1\n"
		"		;;\n"
		"esac\n"
		"SWBCS_SWSEL_PAT=\"$1\"\n"
		"SWBCS_SCRIPT_TAG=\"$2\"\n"
		"SWBCS_GOT_MATCH=n\n"
		"SWBCS_MATCH=\"\"\n"
		"for fqc in $SWBCS_TAGLIST\n"
		"do\n"
		"	case \"$fqc\" in\n"
		"		${SWBCS_SWSEL_PAT})\n"
		"		case $SWBCS_GOT_MATCH in y) echo SW_SELECTION_NOT_FOUND_AMBIG 1>&2; exit 22;; esac\n"
		"		SWBCS_MATCH=\"$fqc\"\n"
		"		SWBCS_GOT_MATCH=y\n"
		"		;;\n"
		"	esac\n"
		"done\n"
		"case $SWBCS_GOT_MATCH in\n"
        	"	n)\n"
		"		echo SW_SELECTION_NOT_FOUND 1>&2\n"
		"		exit 1\n"
		"		;;\n"
		"esac\n"
		, strob_str(taglist)
		);

	strob_sprintf(buf, STROB_DO_APPEND,
		"case \"$SWBCS_MATCH\" in\n"
		);

	tag = strob_strtok(tmp, strob_str(taglist), " ");
	while(tag) {
		swpl_write_case_block(swi, buf, tag);
		tag = strob_strtok(tmp, NULL, " ");
	}
	
	strob_sprintf(buf, STROB_DO_APPEND, "esac\n");

	/*
	 * Now finally the environment is set, so now
	 * run the script.
	 */

	strob_sprintf(buf, STROB_DO_APPEND,
		"\n"
		"# Now run the script\n"
		"\n"
		"sh \"${SW_CONTROL_DIRECTORY}/${SWBCS_SCRIPT_NAME}\"\n"
		"swbcs_ret=$?\n"
		"case \"$swbcs_ret\" in\n"
		"	0) ;;\n"
		"	1) ;;\n"
		"	2) ;;\n"
		"	3) ;;\n"
		"	*) swbcs_ret=1 ;;\n"
		"esac\n"
		"exit $swbcs_ret\n"
		);

	strob_close(taglist);
	strob_close(tmp);
	return 0;
}

static
int
write_tar_controlsh_file(GB * G, SWI * swi, int ofd,
	char * catalog_path,
	char * pax_read_command,
	int alt_catalog_root,
	int event_fd,
	char * id_str
	)
{
	int ret = 0;
	STROB * name;
	STROB * data;

	name = strob_open(100);
	data = strob_open(100);

	strob_strcpy(name, swi->swi_pkgM->catalog_entryM);
	swlib_unix_dircat(name, SW_A_controlsh);
	ret = construct_controlsh_script(G, data, swi);
	SWLIB_ASSERT(ret == 0);

	ret = swpl_load_single_file_tar_archive(G,
		swi,
		ofd, 
		catalog_path,
		pax_read_command,
		alt_catalog_root,
		event_fd,
		id_str,
		strob_str(name),
		strob_str(data));

	strob_close(name);
	strob_close(data);
	return ret;
}

static
int
write_tar_session_options_file(GB * G, SWI * swi, int ofd,
	char * catalog_path,
	char * pax_read_command,
	int alt_catalog_root,
	int event_fd,
	char * id_str
	)
{
	int ret = 0;
	STROB * name;
	STROB * data;

	name = strob_open(100);
	data = strob_open(100);

	swpl_write_session_options_filename(name, swi);
	swpl_write_session_options_file(data, swi);

	ret = swpl_load_single_file_tar_archive(G,
		swi,
		ofd, 
		catalog_path,
		pax_read_command,
		alt_catalog_root,
		event_fd,
		id_str,
		strob_str(name),
		strob_str(data));

	strob_close(name);
	strob_close(data);
	return ret;
}

static
int
check_is_all_digits(char * value) {
	char * terminating_chars;
	char * w;
	w = value;
	if (strlen(value) == 0) return 1;
	terminating_chars = " ";
	while (*w  &&  isdigit(*w)) w++;
	if (*w == '\0' || strchr(terminating_chars, *w)) return 0;
	return -1;
}

static
int
detect_and_correct_numeric_ids(AHS * ahs)
{
	struct new_cpio_header * h;
	char * username;
	char * groupname;
	unsigned long uid;
	unsigned long gid;

	h = ahs->file_hdrM;
	username = ahsStaticGetTarUsername(h);
	groupname = ahsStaticGetTarGroupname(h);
	uid = h->c_uid;
	gid = h->c_gid;

	if (check_is_all_digits(username) == 0) {
		if (swlib_atoi(username, NULL) == (int)uid) {
			ahsStaticSetTarUsername(h, "");
		} 
	}

	if (check_is_all_digits(groupname) == 0) {
		if (swlib_atoi(groupname, NULL) == (int)gid) {
			ahsStaticSetTarGroupname(h, "");
		} 
	}
	return 0;
}

static
int
sanity_compare(AHS * ahs1, AHS * ahs2, char * filename, int do_be_verbose)
{
	int ret = 0;
	int cpiotype1, cpiotype2;
	struct new_cpio_header * h1 = ahs1->file_hdrM;
	struct new_cpio_header * h2 = ahs2->file_hdrM;

	cpiotype1 = (h1->c_mode & CP_IFMT);
	cpiotype2 = (h2->c_mode & CP_IFMT);

	if (cpiotype1 != cpiotype2) {
		ret ++;
		fprintf(stderr, "%s: attribute mismatch: %s: att=type: storage=[%d] [%d]\n",
			swlib_utilname_get(), filename,
			(int)(cpiotype1),
			(int)(cpiotype2));
	}

       	if  ((h2->usage_maskM) & TARU_UM_MODE) {
		if ((07777 & h1->c_mode) != (07777 & h2->c_mode)) {
			ret ++;
			fprintf(stderr, "%s: attribute mismatch: %s: att=mode: storage=[%o] [%o]\n",
				swlib_utilname_get(), filename,
				(int)(h1->c_mode),
				(int)(h2->c_mode));
		}
	} else {
		if (do_be_verbose)
		fprintf(stderr, "%s: warning: INFO file does not contain a mode attribute: %s\n",
			swlib_utilname_get(), filename);
	}

       	if  ((h2->usage_maskM) & TARU_UM_UID) {
 	      	if (
			(h1->c_uid != ULONG_MAX && h2->c_uid != ULONG_MAX) &&
			h1->c_uid != h2->c_uid
		) {
			ret ++;
			fprintf(stderr, "%s: attribute mismatch: %s: att=uid: storage=[%d] INFO=[%d]\n",
				swlib_utilname_get(), filename,
				(int)(h1->c_uid),
				(int)(h2->c_uid));
		}
	} else {
		if (do_be_verbose)
		fprintf(stderr, "%s: warning: INFO file does not contain a uid attribute: %s\n",
			swlib_utilname_get(), filename);
	}

       	if  ((h2->usage_maskM) & TARU_UM_GID) {
 	      	if (
			(h1->c_gid != ULONG_MAX && h2->c_gid != ULONG_MAX) &&
			h1->c_gid != h2->c_gid
		) {
			ret ++;
			fprintf(stderr, "%s: attribute mismatch: %s: att=gid: storage=[%d] INFO=[%d]\n",
				swlib_utilname_get(), filename,
				(int)(h1->c_gid),
				(int)(h2->c_gid));
		}
	} else {
		if (do_be_verbose)
		fprintf(stderr, "%s: warning: INFO file does not contain a gid attribute: %s\n",
			swlib_utilname_get(), filename);
	}

       	if ( ((h2->usage_maskM) & TARU_UM_MTIME) && h1->c_mtime != h2->c_mtime) {
		/* This is now only a warning. */
		/* ret ++; FIXME:  policy at wrong level */
		fprintf(stderr, "%s: Warning: attribute mismatch: %s: att=mtime: storage=[%lu] INFO=[%lu]\n",
			swlib_utilname_get(), filename,
			(h1->c_mtime),
			(h2->c_mtime));
	}

       	if (h1->c_filesize != h2->c_filesize &&
		(
			(cpiotype1 != CP_IFLNK) &&
			(cpiotype2 != CP_IFLNK) &&
			(cpiotype2 != CP_IFLNK) &&
			(h1->c_is_tar_lnktype != 1) &&
			(h1->c_is_tar_lnktype != 1)
		)
	) {
		/*
		 * Don't do a file size comparison if the type is SYMTYPE or LNKTYPE
		 */
		ret ++;
		fprintf(stderr, "%s: attribute mismatch: %s: att=size: storage=[%d] INFO=[%d]\n",
			swlib_utilname_get(), filename,
			(int)(h1->c_filesize),
			(int)(h2->c_filesize));

	}

       	if ((h1->c_dev_maj || h2->c_dev_maj) && 
		h1->c_dev_maj != h2->c_dev_maj) {
		ret ++;
		fprintf(stderr, "%s: attribute mismatch: %s: att=major: storage=[%d] INFO=[%d]\n",
			swlib_utilname_get(), filename,
			(int)(h1->c_dev_maj),
			(int)(h2->c_dev_maj));
	}

       	if ((h1->c_dev_min || h2->c_dev_min) &&
       		h1->c_dev_min != h2->c_dev_min) {
		ret ++;
		fprintf(stderr, "%s: attribute mismatch: %s: att=minor: storage=[%d] INFO=[%d]\n",
			swlib_utilname_get(), filename,
			(int)(h1->c_dev_min),
			(int)(h2->c_dev_min));
	}

       	if  ((h2->usage_maskM) & TARU_UM_GROUP) {
		ret += swpl_compare_name(
			ahsStaticGetTarGroupname(h1),
			ahsStaticGetTarGroupname(h2),
			SW_A_group, filename);
	} else {
		if (do_be_verbose)
		fprintf(stderr, "%s: warning: INFO file does not contain a group attribute: %s\n",
			swlib_utilname_get(), filename);
	}


       	if  ((h2->usage_maskM) & TARU_UM_OWNER) {
		ret += swpl_compare_name(
			ahsStaticGetTarUsername(h1),
			ahsStaticGetTarUsername(h2),
			SW_A_owner, filename);
	} else {
		if (do_be_verbose)
			fprintf(stderr, "%s: warning: INFO file does not contain a owner attribute: %s\n",
			swlib_utilname_get(), filename);
	}	

	if (1 || (h1->c_is_tar_lnktype != 1 && h2->c_is_tar_lnktype != 1)) {
		/*
		 * If either is a hard link, don't do this comparison.
		 * The linkname in a tar header does not match the
		 * link_source attribute in the INFO file since the tar
		 * archive may have control directories in the linkname.
		 * This is done so that the IEEE layout tar archive is
		 * self consistent.
		 */

		/*
		 * Actually, don't do this test at all since
		 * the linkname is rewritten.
		 * FIXME there should be sanity check anyway.
		 */
		/*
		ret += swpl_compare_name(
			ahsStaticGetTarLinkname(h1),
			ahsStaticGetTarLinkname(h2),
			SW_A_link_source, filename);
		*/
	}

	if (strcmp(filename, ".") == 0)
		ret += swpl_compare_name(
			"/" /* filename */,
			ahsStaticGetTarFilename(h2),
			SW_A_path, "/");
	else	
		ret += swpl_compare_name(
			filename /* ahsStaticGetTarFilename(h1)*/ ,
			ahsStaticGetTarFilename(h2),
			SW_A_path, filename);

	return ret;
}

static
int
determine_if_do_configure(SWI * swi, struct extendedOptions * opta)
{
	char * value;
	char * target_path;
	SWI_PRODUCT * current_product;
	SWI_XFILE * current_fileset;
	SWHEADER * indexheader;
	SWHEADER_STATE state;

	target_path = swi->swi_pkgM->target_pathM;

	/*
	 * check is the target_path is "/"
	 */

	if (target_path == NULL || strcmp(target_path, "/") != 0) {
		/*
		 * requirements for running configure script by swinstall
		 * are not met
		 */
		return 0;
	}

	current_product = swi_package_get_product(swi->swi_pkgM, 0 /* FIXME the first one */);
	current_fileset = swi_product_get_fileset(current_product, 0 /* FIXME the first one */);

	indexheader = swi_get_global_index_header(swi);
	swheader_store_state(indexheader, &state);
	swheader_reset(indexheader);

	/*
	 * FIXME, support more than one fileset
	 */

	swheader_set_current_offset(indexheader, current_fileset->baseM.header_indexM);

	/*
	 * Now check the is_reboot attribute
	 */
	value = swheader_get_single_attribute_value(indexheader, SW_A_is_reboot);
	swheader_restore_state(indexheader, &state);
	if (value != NULL && swextopt_is_value_true(value)) {
		/*
		 * don't configure now, per spec
		 */
		return 0;
	}

	/*
	 * Now check the defer_configure extended option
	 */

	if (swextopt_is_option_true(SW_E_defer_configure, opta)) {
		return 0;
	}

	return 1; /* Do configure, if a configure script is included */
}

static
char * 
get_directory(SWI * swi)
{
	SWI_PRODUCT * product;
	char * value;
	SWHEADER_STATE state;
	SWHEADER * indexheader;
	char * ret = "/"; /* default value */

	indexheader = swi_get_global_index_header(swi);
	swheader_store_state(indexheader, &state);
	swheader_reset(indexheader);

	/* FIXME, select the correct product, not the first */
	product = swi_package_get_product(swi->swi_pkgM, 0); /* FIXME */
	swheader_set_current_offset(indexheader, product->baseM.header_indexM);

	value = swheader_get_single_attribute_value(indexheader, SW_A_directory);
	if (value == NULL) 
		value = ret;
	swheader_restore_state(indexheader, &state);
	return value;
}

static
uintmax_t
get_delivery_size(SWI * swi, int * result)
{
	uintmax_t size;
	uintmax_t ret;
	int count;
	char * obj;
	char * line;
	char * value;

	SWHEADER * indexheader = swi_get_global_index_header(swi);
	SWHEADER * infoheader = swi_get_fileset_info_header(swi, 0, 0);

	/*
	 * Get the size of the tar archive of the fileset
	 * which is the fileset size attribute plus 2048 bytes 
	 * for each file to account for the tar header and
	 * slack block.  This assumes the fileset size attribute
	 * is a sum of the file sizes.
	 */

	/*
	 * Get the size attribute from the fileset.
	 */
	*result = 0;
	swheader_reset(indexheader);
        obj = swheader_get_object_by_tag(indexheader, SW_A_fileset, "*" );
        line = swheader_get_attribute(indexheader, SW_A_size);
	value = swheaderline_get_value(line, NULL);

	if (!value) {
		SWLIB_FATAL("fileset.size attribute is missing");
	}

	size = strtoumax(value, NULL, 10);

	count = swpl_get_fileset_file_count(infoheader);
	if (count < 0) {
		SWLIB_FATAL("");
	}

	size = 	size + 			/* the fileset size value */
		(count * 2048) +	/* header and last block for every file + 1024 extra pad */
		(10 * 512);		/* trailer blocks, 10 is arbitrary */

	/*
	 * Now convert size to be evenly divisible by 512
	 * size = ((size / 512) + 1) * 512;
	 */
	ret = swpl_get_whole_block_size(size);
	return ret;
}

static
int
write_attribute_file(SWI * swi, ETAR * etar, int ofd, char * filename, char * file_data)
{
	int ret;
	int retval;

	retval = 0;
	SWLIB_ASSERT(strlen(file_data) < 64); /* Sanity check and enforce the Posix Limit */
	swpl_init_header_root(etar);
	etar_set_size(etar, strlen(file_data) + 1 /* +1 is the newline */);
	if (etar_set_pathname(etar, filename)) SWLIB_FATAL("name too long");
	etar_set_chksum(etar);

	/*
	 * write the tar header for the file.
	 * The file name is <filename>
	 */
	ret = atomicio((ssize_t (*)(int, void *, size_t))uxfio_write,
			ofd,
			(void*)(etar_get_hdr(etar)),
			(size_t)(TARRECORDSIZE));
	if (ret != TARRECORDSIZE) { SWBIS_ERROR_IMPL(); SWI_internal_fatal_error(); }
	retval += ret;

	/*
	 * now write the data block which the value of the vendor_tag.
	if (strlen(file_data) == 0)
		return retval;
	 */

	memset((void*)(swi->tarbufM), '\0', TARRECORDSIZE);
	strncpy((char*)(swi->tarbufM), file_data, TARRECORDSIZE - 2);
	strcat((char*)(swi->tarbufM), "\n");
	ret = atomicio((ssize_t (*)(int, void *, size_t))uxfio_write,
			ofd, (void*)(swi->tarbufM), (size_t)(TARRECORDSIZE));
	if (ret != TARRECORDSIZE) { SWBIS_ERROR_IMPL(); SWI_internal_fatal_error(); }
	retval += ret;
	return retval;
}

static
int
write_catalog_archive_member(SWI * swi, int ofd,
		int sig_block_start, int sig_block_end)
{
	int ret;
	int retval;
	XFORMAT	* xformat;
	STROB * buf;
	int size;
	struct tar_header * catalog_tar_hdr;
	ETAR * etar;
	char * qualifier;
	char * location;
	char * vendor_tag;

	catalog_tar_hdr = (struct tar_header *)malloc(TARRECORDSIZE);
	xformat = swi->xformatM;
	buf = strob_open(100);
	etar = etar_open(swi->xformatM->taruM);
	etar_init_hdr(etar);

	/*
	 * Set the basic tar header fields
	 */
	swpl_init_header_root(etar);

	/*
	 * set the tar header name and file size
	 */
	strob_strcpy(buf, "");
	swlib_unix_dircat(buf, SWINSTALL_INCAT_NAME);
	swlib_unix_dircat(buf, SWINSTALL_CATALOG_TAR);
	size = swpl_write_catalog_data(swi, swi->nullfdM, sig_block_start, sig_block_end);
	etar_set_size(etar, size);
	if (etar_set_pathname(etar, strob_str(buf))) SWLIB_FATAL("name too long");
	etar_set_chksum(etar);

	/*
	 * write the tar header
	 */
	ret = atomicio((ssize_t (*)(int, void *, size_t))uxfio_write,
				ofd,
				(void*)(etar_get_hdr(etar)),
				(size_t)(TARRECORDSIZE));

	if (ret != TARRECORDSIZE) {
		SWBIS_ERROR_IMPL();
		SWI_internal_fatal_error();
	}

	/*
	 * write the catalog data.
	 */
	if (swpl_write_catalog_data(swi, ofd, sig_block_start, sig_block_end) != size) {
		SWBIS_ERROR_IMPL();
		SWI_internal_error();
		retval = -1;
	} else {
		retval = size + ret;
	}
	
	/*
	 * write the location attribute file.
	 */
	location = swi->swi_pkgM->locationM;
	retval += write_attribute_file(swi, etar, ofd, SW_A_location, location);

	/*
	 * write the qualifier attribute file.
	 */
	qualifier = swi->swi_pkgM->qualifierM;
	retval += write_attribute_file(swi, etar, ofd, SW_A_qualifier, qualifier);
	
	/*
	 * write the vendor_tag attribute file.
	 */
	vendor_tag = strar_get(swi->distdataM->vendor_tagsM, 0 /*product index*/);
	if (!vendor_tag || strlen(vendor_tag) == 0) {
		vendor_tag = "";
	}
	retval += write_attribute_file(swi, etar, ofd, SW_A_vendor_tag, vendor_tag);

	etar_close(etar);
	strob_close(buf);
	free(catalog_tar_hdr);
	return retval;
}

int
send_catalog_tarfile(SWI * swi, int ofd, 
	char * catalog_path, char * pax_read_command,
	int alt_catalog_root, int event_fd,
	struct tar_header * ptar_hdr,
	int sig_block_start, int sig_block_end)
	
{
	int ret = 0;
	int dataret = 0;
	int stdin_file_size;
	STROB * tmp;
	STROB * namebuf;

	tmp = strob_open(10);
	namebuf = strob_open(10);

	/*
	 * Here is the task scriptlet.
	 */
	strob_sprintf(tmp, 0,
		"dd 2>/dev/null | %s\n"
		"sw_retval=$?\n"	/* This line is required */
		"dd of=/dev/null 2>/dev/null\n"
		,
		pax_read_command
		);

	stdin_file_size = write_catalog_archive_member(swi, swi->nullfdM, sig_block_start, sig_block_end);
	if (stdin_file_size < 0) {
		SWBIS_ERROR_IMPL();
		return -1;
	}
	stdin_file_size += (TARRECORDSIZE+TARRECORDSIZE); /* trailer blocks */

	if (stdin_file_size % TARRECORDSIZE) {
		SWBIS_ERROR_IMPL();
		SWI_internal_error();
		return -1;
	}

	/*
	 * Here is the directory that is chdir'ed into.
	 */
	strob_strcpy(namebuf, catalog_path);
	if (alt_catalog_root == 0) {
		/*
	 	 * Unless alt_catalog_root is true,
		 * the catalog path is relative to the
		 * target path.
		 */
		swlib_squash_all_leading_slash(strob_str(namebuf));
	}

	swicol_set_task_idstring(swi->swicolM, SWBIS_TS_Load_catalog);
	ret = swicol_rpsh_task_send_script2(
		swi->swicolM,
		ofd, 
		stdin_file_size,
		strob_str(namebuf),
		strob_str(tmp), SWBIS_TS_Load_catalog
		);

	/*
	 * Now write the actual data to the stdin of the
	 * posix shell, It *must* be exactly stdin_file_size
	 * bytes in length.
	 */
	if (ret == 0) {
		ret = write_catalog_archive_member(swi, ofd, 
			sig_block_start, sig_block_end);

		if (ret < 0) {
			SWBIS_ERROR_IMPL();
			SWI_internal_error();
			return -1;
		}
		dataret += ret;

		/*
		 * Now write the trailer blocks
		 */	
		ret = etar_write_trailer_blocks(NULL, ofd, 2);
		if (ret < 0) {
			SWBIS_ERROR_IMPL();
			SWI_internal_error();
			return -1;
		}
		dataret += ret;

		/*
		 * Assert what must be true.
		 */
		if (dataret != stdin_file_size) {
			SWBIS_ERROR_IMPL();
			SWI_internal_error();
			return -1;
		}

		/*
		 * Reap the events from the event pipe.
		 */
		ret = swicol_rpsh_task_expect(swi->swicolM,
					event_fd,
					SWICOL_TL_9 /*time limit*/);
		if (swi->debug_eventsM)
			swicol_show_events_to_fd(swi->swicolM, STDERR_FILENO, -1);
	}	
	strob_close(tmp);
	strob_close(namebuf);
	return ret;
}


static
int
construct_execution_phase_script(GB * G, SWI * swi,
			STROB * buf,
			char * pax_read_command
			)
{
	int do_configure;
	int ret;
	char * xxx;
	STROB * tmp;

	tmp = strob_open(1000);

	strob_sprintf(buf, STROB_DO_APPEND,
	"( # subshell 001\n"
	);

	/*
	 * determine if its proper to run the configure scripts
	 */
	do_configure = determine_if_do_configure(swi, G->optaM);

	/*
	 * construct the preinstall portion of the execution script
	 */
	ret = swpl_construct_script(G, buf, swi, SW_A_preinstall);

	strob_sprintf(buf, STROB_DO_APPEND,
	"# FIXME: why isn't sw_retval being set\n"
	"sw_retval=\"${sw_retval:=0}\"\n"
	"exit \"$sw_retval\"  # sw_retval not set here\n"
	") 1</dev/null # subshell 001\n"
	"sw_retval=$?\n"
	);

	/*
	 * Now write the actual command to install the fileset
	 */
	xxx = TEVENT(2, -1, SWI_TASK_CTS, "Clear to Send: status=0");

	strob_sprintf(buf, STROB_DO_APPEND,
		"\n"
		"cd \"%s\"\n"
		"case $? in\n"
		"	0)\n"
		"	%s\n"
		"	dd 2>/dev/null | %s\n"
		"	sw_retval=$?\n"
		"	dd of=/dev/null 2>/dev/null\n"
		"	;;\n"
		"	*)\n"
		"	sw_retval=1\n"
		"	;;\n"
		"esac\n"
		"\n"
		,
		swi->swi_pkgM->target_pathM,
		xxx,
		pax_read_command);

	/*
	 * construct the postinstall portion of the execution script
	 */
	strob_sprintf(buf, STROB_DO_APPEND,
		"case $sw_retval in # Case005\n"
		"0)\n"
		);

	strob_sprintf(buf, STROB_DO_APPEND,
	"( # subshell 002\n"
	);
	
	ret = swpl_construct_script(G, buf, swi, SW_A_postinstall);

	swpl_construct_configure_script(G, buf, swi, do_configure);

	strob_sprintf(buf, STROB_DO_APPEND,
	"exit \"$sw_retval\"\n"
	") 1</dev/null # subshell 002\n"
	"sw_retval=$?\n"
	);

	strob_sprintf(buf, STROB_DO_APPEND,
		";;\n"
		"esac  # Case005\n"
		);

	return 0;
}

int
run_analysis_script(GB * G, SWI * swi, int ofd, int event_fd)
{
	int ret;
	STROB * buf;
	SWI_CONTROL_SCRIPT * script;
	int stdin_file_size;
	int this_index;
	int script_status;
	char * checkinstall_message;

	buf = strob_open(400);

	ret = swpl_construct_analysis_script(G, buf, swi, &script);
	SWLIB_ASSERT(ret == 0);
	
	stdin_file_size = 512;

	swicol_set_task_idstring(swi->swicolM, SWBIS_TS_Analysis_002);
	ret = swicol_rpsh_task_send_script2(
		swi->swicolM,
		ofd, 
		512,
		".",
		strob_str(buf),
		SWBIS_TS_Analysis_002
		);

	if (ret == 0) {
		/* 
		 * Send the payload, in this case, it is a gratuitous block of nuls.
		 */
		ret = etar_write_trailer_blocks(NULL, ofd, 1);

		/*
		 * wait for the script to finish
		 */
		ret = swicol_rpsh_task_expect(swi->swicolM,	
				event_fd,
				SWICOL_TL_9 /*timelimit*/);
		/*
		 * this shows only the events for this task script
		 */	
		if (swi->debug_eventsM)
			swicol_show_events_to_fd(swi->swicolM, STDERR_FILENO, -1);
	
		checkinstall_message = swicol_rpsh_get_event_message(swi->swicolM,
       	         			SW_CONTROL_SCRIPT_BEGINS,
					-1, 
					&this_index);

		if (checkinstall_message) {
			/*
			 * Perform a sanity check on the message
			 */
	
			SWLIB_ASSERT(strstr(checkinstall_message, SW_A_checkinstall) != NULL);
	
			/*
			 * record the result
			 */
	
			script_status = swicol_rpsh_get_event_status(swi->swicolM, NULL, SW_CONTROL_SCRIPT_ENDS, this_index, &this_index);
			script->resultM = script_status;
		} else {
			/*
			 * This is OK
			 * There must not have been a checkinstall script
			 */
			;
		}
	}
	strob_close(buf);
	return ret;
}

int
swinstall_lib_determine_catalog_path(STROB * buf, SWI_DISTDATA * distdata,
		int product_number, char * isc_path, int instance_id)
{
	char * isc_relpath;
	char * prodtag;
	char * prodrev;
	char * prodvendortag;
	int idx;
	int ret;
	STROB * tmp;
	
	tmp = strob_open(32);
	isc_relpath = swlib_return_relative_path(isc_path);
	strob_strcpy(tmp, isc_relpath);
	swlib_unix_dircat(tmp, distdata->catalog_bundle_dir1M);

	idx = product_number;  /* for now, always zero */
	prodtag = strar_get(distdata->product_tagsM, idx);
	prodrev = strar_get(distdata->product_revisionsM, idx);
	prodvendortag = strar_get(distdata->vendor_tagsM, idx);

	if (prodtag != NULL) {
		strob_strcpy(buf, strob_str(tmp));
		if (swlib_is_sh_tainted_string(prodtag)) { 
			return 7;
		}
		if (swlib_is_sh_tainted_string(prodrev)) { 
			return 8;
		}
		swlib_unix_dircat(buf, prodtag);
		if (prodrev) {
			swlib_unix_dircat(buf, prodrev);
		} else {
			return 9;
		}
		strob_sprintf(buf, STROB_DO_APPEND, "/%d", instance_id);
		ret = 0;
	} else {
		strob_strcpy(buf, "");
		ret = 1;
	}
	strob_close(tmp);
	return ret;
}

static
int
arfinstall_tar(GB * G, SWI * swi, int ofd,
	int * g_signal_flag, char * target_path, char * catalog_path, VPLOB * swspecs,
	int altofd, int opt_preview, char * pax_read_command_key,
	int alt_catalog_root, int event_fd, struct extendedOptions * opta,
	int keep_old_files, uintmax_t * pstatbytes, int allow_missing_files)
{
	int format;
	int output_format;
	int vofd;
	int padamount;
	int read_header_ret;
	int retval = 0;
	int cmpret = 0;
	int size_result;
	int taru_ls_verbose_level = -1;
	char * current_path;
	char * l_current_path;
	char * current_type;
	char * current_link_source;
	char * name;
	char * pax_read_command;
	int ifd;
	int fileset_fd;
	int ret;
	int parseret;
	int is_catalog;
	int data_amount;
	int files_loading_verbose;
	int filecount;
	int optacc_a;
	int atoi_error;
	int files_missing_from_storage;
	STROB * resolved_path;
	STROB * namebuf;
	STROB * newnamebuf;
	STROB * catalogdir;
	STROB * tmp;
	STROB * pathtmp;
	STROB * tmp_swpath_ex;
	STROB * execution_phase_script;
	STROB * current_path_expanded;
	STROB * relocated_current_path;
	STROB * shell_lib_buf;
	AHS * info_ahs2 = ahs_open();
	char * obj;
	int tarheaderflags;
	struct new_cpio_header * file_hdr;
	XFORMAT * xformat = swi->xformatM;
	SWPATH * swpath = swi->swpathM;
	SWPATH_EX * swpath_ex = NULL;
	SWHEADER * infoheader = (SWHEADER*)NULL;
	int sig_block_start;
	int sig_block_end;
	int package_is_signed = -1;
	struct tar_header *ptar_hdr;
	int did_catalog_tar = 0;
	int did_sigfiles = 0;
	int curpos;
	int did_fileset = 0;
	uintmax_t fileset_write_ret = 0;
	uintmax_t payload_size;
	uintmax_t filesize;
	int eoa;
	int header_ret;
	int current_file_offset;
	char * log_level_str;
	int log_level;
	char * md5buf;
	char * filemd5;
	char * is_volatile;
	SWI_PRODUCT * current_product;
	SWI_XFILE * current_fileset;
	char * newname_suffix;
	int product_count = 0;  /* This is always zero until multiple products are supported */
	char * p_location;
	char * p_qualifier;
	char * p_directory;
	SWVERID * swverid;

	files_missing_from_storage = 0;

	swpath_ex = swpath_shallow_create_export();
	shell_lib_buf = strob_open(100);
	catalogdir = strob_open(100);
	newnamebuf = strob_open(100);
	tmp_swpath_ex = strob_open(100);
	namebuf = strob_open(100);
	tmp = strob_open(10);
	current_path_expanded = strob_open(10);
	relocated_current_path = strob_open(10);
	pathtmp = strob_open(10);
	resolved_path = strob_open(10);
	execution_phase_script = strob_open(10);
	tarheaderflags = xformat_get_tarheader_flags(xformat);
	file_hdr = taru_make_header();

	swpath_reset(swpath);
	
	swi->swicolM->verbose_levelM = swi->verboseM;
	swicol_set_verbose_level(swi->swicolM, swi->verboseM);
	output_format = xformat_get_output_format(xformat);
	format = xformat_get_format(xformat);
	ifd = xformat_get_ifd(xformat);
	if (ifd < 0) {
		SWI_internal_error();
		return -32;		
	}
	taru_set_header_recording(xformat->taruM, 1/*ON*/);
	xformat->taruM->linkrecord_disableM = 1;
	
	if (opt_preview && altofd >= 0) {
		fileset_fd = altofd;
	} else if (opt_preview) {
		fileset_fd = swi->nullfdM;
	} else {
		fileset_fd = ofd;
	}
	
	if (opt_preview) {
		vofd = swi->nullfdM;
	} else {
		vofd = ofd;
	}

	xformat_set_ofd(xformat, vofd);

	ptar_hdr = (struct tar_header*)(strob_str(xformat->taruM->headerM));
	
	log_level_str = get_opta(opta, SW_E_loglevel);
	SWLIB_ASSERT(log_level_str != NULL);
	log_level = swlib_atoi(log_level_str, &atoi_error);	

	files_loading_verbose = (
				(keep_old_files ? keep_old_files : swi->verboseM > SWC_VERBOSE_3) ||
				(log_level >= 2) ||
				(0)
				);

	/* if files_loading_verbose is true, then a 1>&2 is added to the tar command
	   and this results in the file names being compared to error messages, this
	   is harmless but it gives a false appearance of error */
	
	pax_read_command = swc_get_pax_read_command(g_pax_read_commands, pax_read_command_key, 
					files_loading_verbose, 
					keep_old_files, DEFAULT_PAX_R);


	if (swextopt_is_option_true(SW_E_swbis_enforce_file_md5, opta)) {
		md5buf = (char*)(xformat->taruM->md5bufM->str_);
	} else {
		md5buf = NULL;
	}

	/*
	 * FIXME: Eventually support multiple products and filesets.
	 */
	swpl_enforce_one_prod_one_fileset(swi);

	/*
	 * Set the current product and fileset
	 */
	current_product = swi_package_get_product(swi->swi_pkgM, 0 /* the first one */);
	current_fileset = swi_product_get_fileset(current_product, 0 /* the first one */);

	/*
	 * Get the fileset size.
	 */
	payload_size = get_delivery_size(swi, &size_result);
	if (size_result) {
		SWBIS_ERROR_IMPL();
		return -31;
	}

	/* get the location spec */
	p_location = (char*)NULL;
	if ((swverid=vplob_val(swspecs, 0)) != NULL) {
		p_location = swverid_get_verid_value(swverid, SWVERID_VERIDS_LOCATION, 1);
	}
	if (p_location == (char*)NULL || strlen(p_location) == 0)
		p_location = "/";

	/* get the qualifier spec */
	if ((swverid=vplob_val(swspecs, 0)) != NULL) {
		p_qualifier = swverid_get_verid_value(swverid, SWVERID_VERIDS_QUALIFIER, 1);
		if (p_qualifier == NULL)
			p_qualifier = "";
	} else {
		p_qualifier = "";
	}
	
	swi->swi_pkgM->locationM = strdup(p_location);	
	swi->swi_pkgM->qualifierM = strdup(p_qualifier);	

	/* get the value of the product.directory attribute */
	p_directory = get_directory(swi);

	infoheader = swi_get_fileset_info_header(swi, 0, 0);
	filecount = swpl_get_fileset_file_count(infoheader);

	fileset_write_ret = 0;
	is_catalog = -1;
	optacc_a = 0;	
	
	/*
	 * ================================
	 * make the catalog entry directory
	 * ================================
	 */

	if (opt_preview == 0 && retval == 0 && G->g_to_stdout == 0) {
		swicol_set_task_idstring(swi->swicolM, SWBIS_TS_make_catalog_dir);
		strob_sprintf(tmp, 0,
			"(umask 002; mkdir -p \"%s\")</dev/null\n"
			"sw_retval=$?\n"
			"dd count=1 bs=512 of=/dev/null 2>/dev/null\n"
			,
			swi->swi_pkgM->catalog_entryM);

		ret = swicol_rpsh_task_send_script2(
			swi->swicolM,
			ofd, 
			512, 			/* stdin_file_size */
			target_path, 		/* directory to run in*/
			strob_str(tmp),  	/* the task script */
			SWBIS_TS_make_catalog_dir
			);

		if (ret != 0) {
			/* FIXME */
		}

		ret = etar_write_trailer_blocks(NULL, ofd, 1);
		if (ret < 0) {
			SWBIS_ERROR_IMPL();
			SWI_internal_error();
			return -1;
		}

		ret = swicol_rpsh_task_expect(swi->swicolM,
					event_fd,
					SWICOL_TL_5 /*time limit*/);

		if (ret != 0) {
			return -1;
		}
	}

	if (opt_preview == 0 && retval == 0 && G->g_to_stdout == 0) {
		ret = swpl_session_lock(G, swi->swicolM, target_path, ofd,  event_fd);
		if (ret < 0) {
			sw_e_msg(G, "error from swpl_session_lock: status=%d\n", ret);
			return -1;
		} else if (ret > 0) {
			if (G->g_force_locks == 0)
				return -1;
		}
	}

	/*
	 * =====================================
	 * Loop over the Catalog section files.
	 * =====================================
	 */

	while ((read_header_ret = xformat_read_header(xformat)) > 0 &&
			(!g_signal_flag || (g_signal_flag && *g_signal_flag == 0))) {
		
		E_DEBUG("HERE");
		if (xformat_is_end_of_archive(xformat)){
			/*
			 * This is an unexpected error unless their is no
			 * storage section such as the case of the installed
			 * software catalog or a package with no files.
			 */
			if (swi->does_have_payloadM == 0) {
				break;
			} else {
				fprintf(stderr, "%s: premature end of archive\n",
					swlib_utilname_get());
				return -36;
			}
		}

		optacc_a++;  /* Used for Optimization Hack */

		curpos = uxfio_lseek(ifd, 0L, SEEK_CUR);
		if (curpos < 0) {
			SWBIS_ERROR_IMPL();
		}
		xformat_get_name(xformat, namebuf);
		name = strob_str(namebuf);
		data_amount = xformat_get_filesize(xformat);	
	
		/* fprintf(stderr, "eraseme %d %s %d\n", curpos, name, data_amount);  */

		parseret = swpath_parse_path(swpath, name);
		if (parseret < 0) {
			SWBIS_ERROR_IMPL();
			return -38;
		}

		/*
		 * Show the parsed path components
		 */
		swpath_shallow_fill_export(swpath_ex, swpath); 
		if (swi->verboseM >= SWC_VERBOSE_0) {
			sw_d_msg(G,"%s", swpath_ex_print(swpath_ex, tmp_swpath_ex, ""));
		}
		is_catalog = swpath_get_is_catalog(swpath);

		if (is_catalog == SWPATH_CTYPE_DIR) {
			/*
			 * Leading package directories
			 */
			;
		} else if (did_catalog_tar == 0 &&
			is_catalog == SWPATH_CTYPE_CAT &&
			strstr(swpath_get_pathname(swpath), SW_A_INDEX)
		) {
			/*
			 * Write out the catalog.tar file
			 *  <catalog_path>/<export>/catalog.tar
			 */

			/*
			 * Examine the catalog for signature.
			 */
			E_DEBUG("HERE");
			did_catalog_tar = 1;
			swi_examine_signature_blocks(swi->swi_pkgM->dfilesM,
					&sig_block_start, &sig_block_end);
			if (sig_block_start < 0) {
				package_is_signed = 0;
			} else {
				package_is_signed = 1;
			}

			if (opt_preview == 0 && retval == 0) {
				/* TS_catalog_load */
				ret = send_catalog_tarfile(swi, vofd, 
					catalog_path, pax_read_command,
					alt_catalog_root, event_fd, ptar_hdr,
					sig_block_start, sig_block_end);
			} else {
				ret = 0;
			}

			if (ret) {
				retval++;
				E_DEBUG2("retval=%d", retval);
			}
	
			curpos = uxfio_lseek(ifd, curpos, SEEK_SET);
			if (curpos < 0) {
				SWBIS_ERROR_IMPL();
			}

			/*
			 * Exhaust the member data to /dev/null
			 */
			ret = xformat_copy_pass(xformat, swi->nullfdM, ifd);
			if (ret < 0) {
				SWBIS_ERROR_IMPL();
			}
		} else if (
			did_sigfiles == 0 &&
			package_is_signed == 0 &&
			is_catalog == SWPATH_CTYPE_CAT 
		) {
			/*
			 * =====================================
			 * Send empty payload to the task shell if
			 * there are no signatures.
			 * =====================================
			 */
			E_DEBUG("HERE");
			did_sigfiles = 1;
			if (opt_preview == 0 && retval == 0) {
				ret = swpl_send_null_task(swi->swicolM, vofd, event_fd, SWBIS_TS_Load_signatures, SW_SUCCESS);
			} else {
				ret = 0;
			}
			if (ret) {
				retval++;
				E_DEBUG2("retval=%d", retval);
			}
			ret = xformat_copy_pass(xformat, swi->nullfdM, ifd);
		} else if (
			did_sigfiles == 0 &&
			package_is_signed > 0 &&
			is_catalog == SWPATH_CTYPE_CAT &&
			strstr(swpath_get_pathname(swpath), "signature")
		) {
			/*
			 * Write out all the signatures.
			 *	 <catalog_path>/export/catalog.tar.sig
			 *	 <catalog_path>/export/catalog.tar.sigN
			 */
			E_DEBUG("HERE");
			did_sigfiles = 1;
			filesize = xformat_get_filesize(xformat);	

			if (opt_preview == 0 && retval == 0) {
				/* TS_Load_signatures */
				ret = swpl_send_signature_files(swi, vofd,
					catalog_path, pax_read_command,
					alt_catalog_root, event_fd, ptar_hdr,
					sig_block_start, sig_block_end, filesize);
			} else {
				ret = 0;
			}
			if (ret) {
				retval++;
				E_DEBUG2("retval=%d", retval);
			}

			curpos = uxfio_lseek(ifd, curpos, SEEK_SET);
			if (curpos < 0) {
				SWBIS_ERROR_IMPL();
			}

			E_DEBUG("");
			ret = xformat_copy_pass(xformat, swi->nullfdM, ifd);
			if (ret < 0) {
				SWBIS_ERROR_IMPL();
			}
		} else if (is_catalog == SWPATH_CTYPE_CAT) {
			/*
			 * =====================================
			 * Read past the remainder of the catalog files.
			 * =====================================
			 */
			E_DEBUG("HERE");
			ret = xformat_copy_pass(xformat, swi->nullfdM, ifd);
			if (ret < 0) {
				SWBIS_ERROR_IMPL();
			}
		} else if (did_fileset == 0 && is_catalog == SWPATH_CTYPE_STORE) {
			/*
			 * Now ready to deal with the storage section.
			 */		
			E_DEBUG("HERE");
			break;
		}
	} /* while */

	E_DEBUG("HERE");
	if (read_header_ret < 0) {
		SWBIS_ERROR_IMPL();
		return -50;
	}
	E_DEBUG("HERE");

	/*
	 * Good, made it this far..
	 * Now, the catalog.tar and signature files are loaded
	 * We could check the signature here before unpacking the catalog.tar
	 * file.
	 */

	/*
	 * ==========================================
	 * Send the task script to unpack the catalog
	 * ==========================================
	 */

	E_DEBUG("HERE");
	if (opt_preview == 0 && retval == 0) {
		E_DEBUG("HERE");
		/* TS_catalog_unpack */
		ret = swpl_unpack_catalog_tarfile(swi, vofd, catalog_path, pax_read_command, alt_catalog_root, event_fd);
		if (ret) {
			retval++;
			E_DEBUG2("retval = %d", retval);
		}
	}
	E_DEBUG2("retval = %d", retval);

	E_DEBUG("HERE");
	/*
	 * ==========================================
	 * Send the task script to install the 
	 * session_options file.
	 * ==========================================
	 */

	if (opt_preview == 0 && retval == 0) {
		/* TS_session_options */
		E_DEBUG("HERE");
		ret = write_tar_session_options_file(G,
			swi,
			vofd,
			catalog_path,
			pax_read_command,
			alt_catalog_root,
			event_fd,
			SWBIS_TS_Load_session_options
			);
		if (ret) {
			retval++;
			E_DEBUG2("retval = %d", retval);
		}
	}
	E_DEBUG2("retval = %d", retval);
	
	/*
	 * ==========================================
	 * Send the task script to load the the 
	 * control.sh file.
	 * ==========================================
	 */

	E_DEBUG("HERE");
	if (opt_preview == 0 && retval == 0) {
		/* TS_load_controlsh */
		E_DEBUG("HERE");
		ret = write_tar_controlsh_file(G,
			swi,
			vofd,
			catalog_path,
			pax_read_command,
			alt_catalog_root,
			event_fd,
			SWBIS_TS_Load_control_sh);
		if (ret) {
			retval++;
			E_DEBUG2("retval = %d", retval);
		}
	}


	E_DEBUG("HERE");
	if (opt_preview == 0 && retval == 0) {
		/*
		 * TS_analysis_002
		 */
		E_DEBUG("HERE");
		ret = run_analysis_script(G, swi, vofd, event_fd);
		if (ret) {
			retval++;
			E_DEBUG2("retval = %d", retval);
		}
	}


	/*
	 * ===================================================
	 * Send the INSTALLED file to set the state to transient
	 * ===================================================
	 */

	E_DEBUG("HERE");
	if (opt_preview == 0 && retval == 0) {
		/* TS_load_index_file_90 */
		/*
		 * Audit the control_scripts that must be run
		 */
		E_DEBUG("HERE");
		swpl_audit_execution_scripts(G, swi, swpl_get_script_array());
		
		swpl_update_fileset_state(swi, "*", SW_STATE_TRANSIENT);

		ret = swpl_write_tar_installed_software_index_file(G, swi,
			vofd,
			catalog_path,
			pax_read_command,
			alt_catalog_root,
			event_fd,
			SWBIS_TS_Load_INSTALLED_90);
		if (ret) {
			retval++;
			E_DEBUG2("retval = %d", retval);
		}
	}
	E_DEBUG2("retval = %d", retval);

	/*
	 * =============================================
	 * Send the task script for loading the fileset.
	 * =============================================
	 */
	E_DEBUG("HERE");
	did_fileset = 1;	
	strob_strcpy(newnamebuf, target_path);
	ret = 0;
	if (opt_preview == 0 && retval == 0) {
		/* TS_execution_phase */
		E_DEBUG("HERE");
		ret = construct_execution_phase_script(G, swi,
				 execution_phase_script,
				pax_read_command);

		swicol_set_task_idstring(swi->swicolM, SWBIS_TS_Load_fileset);
		swicol_set_event_fd(swi->swicolM, event_fd);
		ret = swicol_rpsh_task_send_script2(
			swi->swicolM,
			vofd, 
			payload_size,  /* fileset size plus padding */
			".", /* strob_str(newnamebuf), */
			strob_str(execution_phase_script), SWBIS_TS_Load_fileset);
		swicol_set_event_fd(swi->swicolM, -1);
		if (ret == 0)
			/* wait for clear-to-send */
			ret = swicol_rpsh_wait_cts(swi->swicolM, event_fd);
	}
	if (ret) {
		retval++;
		E_DEBUG2("retval=%d", retval);
	}

	if (strlen(swpath_get_pathname(swpath))) {
		/*
		 * Now seek back 512 bytes.
		 * This is not required if the control directoires are
		 * in the package, but since HP-UX packages don't have the
		 * control directories as archive members then this is
		 * required.
		 */
		/*
	 	 * uh-oh we've dived into the storage section
		 * and there are no control directory archive members.
		 * Meaning the first header of an actual storage file
		 * has just been read.  This happens with HP-UX
		 * packages.
		 */
		E_DEBUG("HERE HP style package");
		if (uxfio_lseek(ifd, -512, SEEK_CUR) < 0) {
			SWI_internal_fatal_error();
		}
	}

	/*
	 * =====================================
	 * Loop over Storage section files.
	 * =====================================
	 */
	E_DEBUG2("retval = %d", retval);
	E_DEBUG("HERE");
	taru_ls_verbose_level = -1;
	while (
		xformat_read_header(xformat) > 0 &&
		(!g_signal_flag || (g_signal_flag && *g_signal_flag == 0)) &&
		retval == 0
	) {
		E_DEBUG("HERE: Storage");
		if (xformat_is_end_of_archive(xformat)){
			break;
		}
		optacc_a++;  
		xformat_get_name(xformat, namebuf);
		name = strob_str(namebuf);
		parseret = swpath_parse_path(swpath, name);
		if (parseret < 0) {
			SWBIS_ERROR_IMPL();
			return -55;
		}
		is_catalog = swpath_get_is_catalog(swpath);
		E_DEBUG2("HERE: Storage: Name is %s", name);
		
		/*
		 * Show the parsed path components
		 */
		swpath_shallow_fill_export(swpath_ex, swpath); 

		sw_d_msg(G,"%s", swpath_ex_print(swpath_ex, tmp_swpath_ex, ""));

		/*
		 * The path is obtained from the archive path
		 */
		current_path = swpath_get_pathname(swpath);
		swlib_expand_escapes(NULL, NULL, current_path, current_path_expanded);

		/* An entry in INFO for a archive member of "." has a path of "/" */
		if (strcmp(strob_str(current_path_expanded), ".") == 0) {
			strob_strcpy(current_path_expanded, "/");
		}

		/*
		 * Sanitize and check the filename
		 */
		swpl_sanitize_pathname(current_path);
		swlib_squash_leading_dot_slash(current_path);

		E_DEBUG2("current_path before applying location: %s", current_path);
		/*
		 * apply the location and directory attributes
		 */
		swlib_apply_location(relocated_current_path, current_path, p_location, p_directory);
		E_DEBUG2("current_path after applying location: %s", current_path);
	
		swpl_sanitize_pathname(strob_str(relocated_current_path));
		swlib_squash_leading_dot_slash(strob_str(relocated_current_path));
		l_current_path = strob_str(relocated_current_path);

		if (optacc_a == 1) {
			/*
			 * Look for the current file at the
			 * current position of the infoheader.
			 * This is an optimization path.
			 */
			E_DEBUG2("HERE: Storage: current_path_expanded is [%s]", strob_str(current_path_expanded));
			obj = swheader_get_object_by_tag(infoheader, SW_OBJ_file, strob_str(current_path_expanded));
		} else {
			/*
			 * this is the failsafe code path
			 */
			E_DEBUG("HERE: Storage");
			obj = (char*)NULL;
		}	

		if (obj == (char*)(NULL)) {
			E_DEBUG2("HERE: obj is NULL for %s", name);
			swheader_reset(infoheader);
		}

		/* Now make a test for storage section control directories */
		E_DEBUG2("HERE: Testing if control directory: %s", l_current_path);
	
		if (l_current_path && strlen(l_current_path)) {
			E_DEBUG("HERE: Storage");
			if (obj == (char*)NULL) {
				obj = swheader_get_object_by_tag(infoheader, SW_OBJ_file, strob_str(current_path_expanded));
			}

			if (obj == (char*)NULL) {
				/*
				* This is a hard error, a storage file must have metadata.
				*/
				fprintf(stderr, "%s: File object not found for path=%s\n", swlib_utilname_get(), current_path);
				fprintf(stderr, "%s: This may be a implementation error or an invalid package\n", swlib_utilname_get());
				SWI_internal_error();
				return -58;
			}

			current_file_offset = swheader_get_current_offset(infoheader);
			current_type = swheader_get_attribute(infoheader, "type");
			current_type = swheaderline_get_value(current_type, NULL);
			swheader_set_current_offset(infoheader, current_file_offset);

			if (!current_type) { 
				SWI_internal_error();
				return -59;
			}


			/* Set the flag1 to indicate this software definitiion was
				represented in the storage structure and used */

			swheaderline_set_flag1(obj);
	
			/*
			 * * Here's how to print the file object from the INFO file.
			 * swheaderline_write_debug(obj, STDERR_FILENO);
			 * while((next_attr=swheader_get_next_attribute(infoheader)))
			 * 	swheaderline_write_debug(next_attr, STDERR_FILENO);
			 *  
			 * * Reading the header advances the index pointer, therefore you have
			 * * to seek back or reset and find the object again.
			 *
			 * swheader_reset(infoheader);
			 * obj = swheader_get_object_by_tag(infoheader, SW_OBJ_file, swpath_get_pathname(swpath));
			 */
			
			/*
			 * The metadata in the INFO file trumps the metadata in the
			 * storage file tar header.  For now, just compare and warn if
			 * any are different.  Eventually, in order to be compliant, the
			 * storage section tar header is re-written with the file stats from
			 * the INFO file.
			 */
					
			/*
			 * translate the INFO file attributes to a file_hdr object.
			 */
			ahs_init_header(info_ahs2);
			ret = swheader_fileobject2filehdr(infoheader, ahs_vfile_hdr(info_ahs2));
			if (ret) {
				/*
				* FIXME, handle this error better.
				*/
				SWI_internal_error();
				return -62;
			}


			/*
			 * Restore the offset of the current file object
			 */
			swheader_set_current_offset(infoheader, current_file_offset);

			/*
			 * Find the "md5sum" attribute.
			 */
			filemd5 = swpl_get_attribute(infoheader, "md5sum", (int *)NULL);
			swheader_set_current_offset(infoheader, current_file_offset);

			/*
			 * Find the "is_volatile" attribute.
			 */
			is_volatile = swpl_get_attribute(infoheader, "is_volatile", (int *)NULL);	
			swheader_set_current_offset(infoheader, current_file_offset);

			if (swextopt_is_value_true(is_volatile)) {
				/*
				 * The INFO file had is_volatile set to true.
				 * Apply the volatile file policy here.
				 */
				if (swextopt_is_option_true(SW_E_swbis_install_volatile, opta)) {
					/*
					 * Install volatile file.
					 * Determine the suffix from the user options.
					 */

					newname_suffix = get_opta(opta, SW_E_swbis_volatile_newname);
					if (newname_suffix && strlen(newname_suffix)) {
						strob_strcpy(pathtmp, l_current_path);
						/* swpl_safe_check_pathname(newname_suffix); */
						strob_strcat(pathtmp, newname_suffix);
						l_current_path = strob_str(pathtmp);
					} else {
						/*
						 * no suffix specified for volatile files
						 */
						;
					}
				} else {
					/*
					 * skip this file because its volatile and policy excludes it.
					 * we must move the file pointer, just dump the data to * /dev/null
					 */		
					ret = xformat_copy_pass_md5(xformat,
						swi->nullfdM,
						ifd, md5buf);
					SWLIB_ASSERT(ret >= 0);
					optacc_a = 0;
					continue;
				}
			} else {
				/*
				 * nothing to do if not volatile
				 */
				;
			}

			if (	md5buf &&
				(!filemd5 && *current_type == SW_ITYPE_f)
			) {
				sw_e_msg(G, "md5sum attribute not found: %s\n", current_path);
			}

			/*
			 * Compare the header stats from the storage section archive
			 * member to the stats from the INFO file.
			 */
			if (sanity_compare(xformat_ahs_object(xformat), 
					info_ahs2,
					swpath_get_pathname(swpath), swi->verboseM >= SWC_VERBOSE_5)) {
				cmpret++;
			}

			/*
			 * Impose the stats from the INFO file.
			 */				
			ahs_copy(xformat_ahs_object(xformat), info_ahs2);

			/*
			 * handle INFO files where uid and uname are the same string
			 */
			detect_and_correct_numeric_ids(info_ahs2);

			/*
			 * Set the sanitized, relocated name.
			 */
			ahsStaticSetTarFilename(
				ahs_vfile_hdr(xformat_ahs_object(xformat)),
				l_current_path);

			/*
			 * Set the link name.
			 */
			if (*current_type == SW_ITYPE_h || *current_type == SW_ITYPE_s) {
				current_link_source = swheader_get_attribute(infoheader, SW_A_link_source);
				current_link_source = swheaderline_get_value(current_link_source, NULL);
				swheader_set_current_offset(infoheader, current_file_offset);
				if (!current_link_source) { SWI_internal_error(); return -68; }

				if (strlen(current_link_source) > TARLINKNAMESIZE) {
					fprintf(stderr, "tar link name too long\n");
					SWI_internal_error();
					return -74;
				}
				if (strcmp(target_path, "/") == 0) {
					/* swpl_safe_check_pathname(current_link_source); */
				} else {
					swpl_sanitize_pathname(current_link_source);
				}
				taru_set_new_linkname(xformat->taruM, (struct tar_header *)NULL, current_link_source);
				ahsStaticSetTarLinkname(
					ahs_vfile_hdr(xformat_ahs_object(xformat)),
					 current_link_source);
			} else {
				taru_set_new_linkname(xformat->taruM, (struct tar_header *)NULL, "");
				ahsStaticSetTarLinkname(ahs_vfile_hdr(xformat_ahs_object(xformat)), "");
			}

			if (pstatbytes) {
				off_t meter_len;
				*pstatbytes = 0;
				if (*current_type == SW_ITYPE_f) {
					meter_len = (off_t)(info_ahs2->file_hdrM->c_filesize);
					if (meter_len && meter_len < 512) meter_len = 512;
				} else {
					meter_len = (off_t)0;
				}	
				start_progress_meter(STDOUT_FILENO, l_current_path, meter_len, pstatbytes);
				*swlib_pump_get_ppstatbytes() = pstatbytes;
			}

			/*
			 * Write the tar header to the target for real.
			 */
			if (fileset_fd != vofd) {
				/*
				 * This code path is used in preview mode.
				 */
				xformat_set_ofd(xformat, fileset_fd);
				header_ret = xformat_write_header(xformat);
				xformat_set_ofd(xformat, vofd);
			} else {
				/*
				 * Normal fast path.
				 */
				header_ret = xformat_write_header(xformat);
			}	
	
			if (header_ret < 0) {
				SWI_internal_error();
				return -76;
			}	
			fileset_write_ret += header_ret;
			
			if (*current_type == SW_ITYPE_f) {
				ret = xformat_copy_pass_md5(xformat,
						fileset_fd,
						ifd, md5buf);
			} else {
				ret = 0;
			}

			if (ret < 0) {
				fprintf(stderr, "error sending file data for file %s\n", l_current_path);
				return -77;
			} else {
				if (pstatbytes) {
					*pstatbytes = ret;
				}
				if (*current_type == SW_ITYPE_f) {

					/*
					 * Sanity check
					 */

					if (ret < (int)(info_ahs2->file_hdrM->c_filesize)) {
						SWBIS_ERROR_IMPL();
					}
				}
				fileset_write_ret += ret;
			}

			if (pstatbytes) {
				update_progress_meter(SIGALRM);
				stop_progress_meter();
			}

			if (
				(opt_preview && swi->verboseM >= SWC_VERBOSE_2) ||
				(swi->verboseM >= SWC_VERBOSE_2 && header_ret > 0) ||
				(log_level >= SWC_VERBOSE_2)
			) {
				int do_write_at_level;

				/*
				 * write a long ls listing.
				 */

				/*
				 * Read and decode the tar header that was just written
				 * to the target.
				 */

				taru_read_in_tar_header2(xformat->taruM,
					ahs_vfile_hdr(xformat_ahs_object(xformat)),
					-1,
					strob_str(xformat->taruM->headerM),
					&eoa,
					xformat_get_tarheader_flags(xformat), TARRECORDSIZE);
		
				/*
				 * Print and write the line.
				 */

				do_write_at_level = opt_preview ? 1 : SWC_VERBOSE_3;
				if (log_level >= 2 && (&(G->g_logspec))->logfdM >= 0) {
					do_write_at_level = 1;
				}
		
				if (taru_ls_verbose_level < 0)	
					taru_ls_verbose_level = swpl_determine_tar_listing_verbose_level(swi);	

				taru_print_tar_ls_list(tmp,
					ahs_vfile_hdr(xformat_ahs_object(xformat)),
					taru_ls_verbose_level);

				if (swi->swc_idM == SWC_U_L) {
					/* swlist utility */
					uxfio_write(G->g_t_efd, strob_str(tmp), strob_strlen(tmp));
				} else {
					/* send to logger process */
					/* fprintf(stderr, "%d v=%d JLJL=%s", do_write_at_level, swi->verboseM, strob_str(tmp));
					*/
					swlib_doif_writef(swi->verboseM, do_write_at_level,
						&G->g_logspec, G->g_t_efd, "%s", strob_str(tmp));
				}
			}


			if (md5buf && *current_type == SW_ITYPE_f) {
				if (filemd5 == NULL) {
					sw_e_msg(G, "Warning: md5sum attribute not found in INFO file for %s\n", current_path);
				}
				if (strlen(md5buf) && filemd5) {
					if (strcmp(md5buf, filemd5)) {
						sw_e_msg(G, "md5sum mismatch archive=%s INFO=%s: %s\n", md5buf, filemd5, current_path);
						return -78;
					} else {
						sw_d_msg(G, "%s %s\n", filemd5, current_path);
					}
				}
			}
			if (fileset_write_ret % 512) {
				SWBIS_ERROR_IMPL();
				return -79;
			}
		} else {
			/*
			 * This code path occurs for storage section control directories.
			 */	
			E_DEBUG3("HERE: control directory or empty path: [%s] [%s]", strob_str(namebuf), current_path);
			ret = 0;
		}
		optacc_a = 0;
		if (ret < 0) {
			SWBIS_ERROR_IMPL();
		}
		E_DEBUG("HERE: Storage");
	} /* while */

	E_DEBUG("HERE");
	if (cmpret) {
		fprintf(stderr,
		"swinstall: ==Warning== A mismatch between the INFO file and storage section\n"
		"swinstall: ==Warning== file attributes was detected.  Currently, swinstall\n"
		"swinstall: ==Warning== *does* override all storage section attributes with\n"
		"swinstall: ==Warning== the INFO file attributes. This warning will be removed\n"
		"swinstall: ==Warning== in a future release.\n");
	}

	/* Now make sure that there were no files missing
		from the storage section, the swheaderline flag1 
		is set for all files that were found in the 
		storage section */

	E_DEBUG("HERE");
	if (g_signal_flag == 0 && retval == 0) {
		/* The routine swheaderline_set_flag1() called above
		   tags the file objects as they are installed */
		ret = swpl_assert_all_file_definitions_installed(swi, infoheader);
	} else {
		ret = 0;
	}

	if (ret)
		files_missing_from_storage = 1;

	E_DEBUG("HERE");
	if (opt_preview == 0) {
		if (fileset_write_ret > (payload_size - (uintmax_t)(1024))) {
			/*
			 * This is a bad error.
			 * No room left to send the tar trailer blocks.
			 */
			SWBIS_ERROR_IMPL();
			return -81;	
		}
	}

	/*
	 * Now send the tar trailer blocks and padding that
	 * the remote dd(1) process is exepecting.
	 */

	E_DEBUG("HERE");
	if (swi->verboseM > SWC_VERBOSE_10)
	{
		STROB * xx_tmp2 = strob_open(10);
		STROB * xx_tmp3 = strob_open(10);

		swlib_umaxtostr(payload_size, xx_tmp2);
		swlib_umaxtostr(fileset_write_ret, xx_tmp3);
		fprintf(stderr, "payload size = %s fileset_write_size =%s \n",
			swlib_umaxtostr(payload_size, xx_tmp2),
			swlib_umaxtostr(fileset_write_ret, xx_tmp3));
		fprintf(stderr, "pad amount = %lu\n",  (unsigned long)(payload_size - fileset_write_ret));
		strob_close(xx_tmp2);
		strob_close(xx_tmp3);
	}

	E_DEBUG("HERE");
	if (retval == 0) {
		padamount = (int)(payload_size - fileset_write_ret);
		ret = swlib_pad_amount(vofd, padamount);
		E_DEBUG("HERE");
		if (ret != padamount) {
			if (g_signal_flag == 0) {
				/* this may happen when ctl-c is given by the user */
				SWBIS_ERROR_IMPL();
				fprintf(stderr, "%s: swlib_pad_amount returned [%d]\n", swlib_utilname_get(), padamount);
			}
			retval++;
			E_DEBUG2("retval = %d", retval);
		}
	}

	/*
	 * This waits on the fileset loading task.
	 */
	E_DEBUG("HERE");
	if (opt_preview == 0 && retval == 0) {
		/*
		 * TS_execution_phase
		 */

		E_DEBUG("HERE");
		ret = swicol_rpsh_task_expect(swi->swicolM,
				event_fd,
				SWICOL_TL_6 /*time limit*/);
		if (ret != 0) {
			E_DEBUG("HERE");
			sw_e_msg(G, "error loading fileset\n");
			return -85;
		}

		if (swi->debug_eventsM)
			swicol_show_events_to_fd(swi->swicolM, STDERR_FILENO, -1);

		/*
		 * Now we must read the events stack and update the script
		 * retsults.
		 */
	
		swpl_update_execution_script_results(swi, swi->swicolM, swpl_get_script_array());

	} else {
		E_DEBUG("HERE");
		if (
			/* G->g_to_stdout == 0 && */
			retval == 0 &&
		    	swi->swc_idM == SWC_U_I && /* swinstall utility */
		    	1
		) {
			/*
		 	 * This satisfies the preview task shell
			 * which serves to make the script wait until
			 * it receives this script.
			 */
			/* TS_preview */
			E_DEBUG("HERE at " SWBIS_TS_preview_task);
			ret = swpl_send_nothing_and_wait(swi->swicolM, ofd, event_fd, SWBIS_TS_preview_task , SWICOL_TL_3, SW_SUCCESS);
			E_DEBUG("HERE at " SWBIS_TS_preview_task);
			if (ret) { 
				retval++;
				E_DEBUG2("retval = %d", retval);
			}
		}
	}

	/*
	 * ====================================================
	 * Send the task script to remove the unpacked catalog
	 * ====================================================
	 *
	 * This removes the directory, for example
	 *    var/lib/swbix/catalog/foo/foo/1.1/0/export
	 */

	if (opt_preview == 0 && retval == 0) {
		/* TS_catalog_dir_remove */
		ret = swpl_remove_catalog_directory(swi, vofd, catalog_path, pax_read_command, alt_catalog_root, event_fd);
		if (ret) {
			retval++;
			E_DEBUG2("retval = %d", retval);
		}
	}

	/*
	 * ===================================================
	 * Send the INSTALLED file
	 * ===================================================
	 */

	if (opt_preview == 0 && retval == 0) {
		/* TS_load_index_file_91 */
		swpl_update_fileset_state(swi, "*", SW_STATE_INSTALLED);
		ret = swpl_write_tar_installed_software_index_file(G, swi,
			vofd,
			catalog_path,
			pax_read_command,
			alt_catalog_root,
			event_fd,
			SWBIS_TS_Load_INSTALLED_91);
		if (ret) {
			retval++;
			E_DEBUG2("retval = %d", retval);
		}
	}


	/*
	 * ===================================================
	 * Rename the _INSTALLED to INSTALLED (make it live)
	 * ===================================================
	 */

	if (opt_preview == 0 && retval == 0) {
		swicol_set_task_idstring(swi->swicolM, SWBIS_TS_make_live_INSTALLED);
		strob_sprintf(tmp, 0,
			"cd \"%s\"\n"
			"case \"$?\" in\n"
			"	0)\n"
			"		sw_retval=0\n"
			"		;;\n"
			"	*)\n"
			"		sw_retval=1\n"
			"		;;\n"
			"esac\n"

			"case \"$sw_retval\" in\n"
			"	0)\n"
			"		mv -f "	SW_A__INSTALLED " " SW_A_INSTALLED "\n"
			"		sw_retval=$?\n"
			"		;;\n"
			"esac\n"
			"dd count=1 bs=512 of=/dev/null 2>/dev/null\n"
			,
			swi->swi_pkgM->catalog_entryM);

		ret = swicol_rpsh_task_send_script2(
			swi->swicolM,
			ofd, 
			512, 			/* stdin_file_size */
			target_path, 		/* directory to run in*/
			strob_str(tmp),  	/* the task script */
			SWBIS_TS_make_live_INSTALLED
			);

		if (ret != 0) {
			return -1;
		}

		ret = etar_write_trailer_blocks(NULL, ofd, 1);
		if (ret < 0) {
			SWBIS_ERROR_IMPL();
			SWI_internal_error();
			return -1;
		}

		ret = swicol_rpsh_task_expect(swi->swicolM,
					event_fd,
					SWICOL_TL_5 /*time limit*/);

		if (ret != 0) {
			return -1;
		}
	}

	/*
	 * ===================================================
	 * Done
	 * ===================================================
	 */

	strob_close(shell_lib_buf);
	strob_close(catalogdir);
	strob_close(newnamebuf);
	strob_close(namebuf);
	strob_close(tmp);
	strob_close(current_path_expanded);
	strob_close(tmp_swpath_ex);
	strob_close(resolved_path);
	strob_close(execution_phase_script);
	taru_free_header(file_hdr);
	ahs_close(info_ahs2);
	if (allow_missing_files) {
		return retval;
	} else {
		return retval || files_missing_from_storage;
	}
}

static
int
write_target_install_script(GB * G,
		int ofd, 
		char * fp_targetpath, 
		STROB * control_message, 
		char * sourcepath,
		int delaytime,
		int keep_old_files,
		int nhops,
		int vlv,
		char * pax_read_command_key,
		char * hostname,
		char * blocksize,
		SWI_DISTDATA * distdata,
		char * installed_software_catalog,
		int opt_preview,
		char * sh_dash_s,
		int alt_catalog_root
		)
{
	int retval = 0;
	int ret;
	int is_target_trailing_slash;
	int cmd_verbose = 0;
	char * xx;
	char * pax_read_command;
	char * opt_char_preview;
	char * opt_char_to_stdout;
	char * opt_force_lock;
	char * opt_allow_no_lock;
	char * ignore_scripts;
	STROB * lockfrag_buffer;
	STROB * unlockfrag_buffer;
	STROB * catlockbuffer;
	STROB * buffer_new;
	STROB * tmp;
	STROB * tmpexit;
	STROB * to_devnull;
	STROB * set_vx;
	STROB * subsh, * subsh2;
	STROB * shell_lib_buf;
	char * target_dirname, *source_basename, *target_basename;
	char * isc_relpath;
	char * targetpath;
	char * x_targetpath;
	char umaskbuf[10];
	char * debug_task_shell;

	SWLIB_ASSERT(fp_targetpath != NULL);
	targetpath = strdup(fp_targetpath);

	if (swlib_check_safe_path(targetpath)) { return 2; }
	if (swlib_check_safe_path(sourcepath)) { return 3; }
	
	lockfrag_buffer = strob_open(64);
	unlockfrag_buffer = strob_open(64);
	buffer_new = strob_open(64);
	catlockbuffer = strob_open(64);
	tmp = strob_open(10);
	tmpexit = strob_open(10);
	to_devnull = strob_open(10);
	set_vx = strob_open(10);
	subsh = strob_open(10);
	subsh2 = strob_open(10);
	shell_lib_buf = strob_open(10);

	if (G->g_do_task_shell_debug == 0) {
		debug_task_shell="";
	} else {
		debug_task_shell="x";
	}

	if (swextopt_is_option_true(SW_E_swbis_ignore_scripts, G->optaM)) {
		ignore_scripts="yes";
	} else {
		ignore_scripts="no";
	}

	if (G->g_force_locks) {
		opt_allow_no_lock = "true";
		opt_force_lock = "true";
	} else {
		opt_allow_no_lock = "";
		opt_force_lock = "";
	}

	/*
	 * Run sanity checks on the distribution tag
	 * and product tags and revisions.
	 */

	if (swlib_is_sh_tainted_string(installed_software_catalog)) { return 4; }
	if (swlib_is_sh_tainted_string(distdata->dist_tagM)) { return 5; }
	if (swlib_is_sh_tainted_string(distdata->catalog_bundle_dir1M)) { return 6; }

	/*
	 * Always treat the installed_software catalog path
	 * as being relative to the target_path
	 */

	isc_relpath = swlib_return_relative_path(installed_software_catalog);
	strob_strcpy(catlockbuffer, isc_relpath);
	swlib_unix_dircat(catlockbuffer, distdata->catalog_bundle_dir1M);

	/*
	 * catlockbuffer contents:
	 * <installed_software_catalog>/<bundle_tag>
	 * This is the name that is used as the lock file name
	 */

	if (G->g_to_stdout == 0) {
		opt_char_to_stdout = "";
	} else {
		opt_char_to_stdout = "True";
	}
	
	if (opt_preview == 0) {
		opt_char_preview = "";
	} else {
		opt_char_preview = "True";
	}

	if (vlv <= SWC_VERBOSE_4 && keep_old_files == 0) {
		strob_strcpy(to_devnull, "2>/dev/null");
	}

	if (vlv >= SWC_VERBOSE_7 && keep_old_files == 0) {
		cmd_verbose = SWC_VERBOSE_1;
	}
	
	if (vlv >= SWC_VERBOSE_SWIDB) {
		strob_strcpy(set_vx, "set -vx\n");
	}

	pax_read_command = swc_get_pax_read_command(g_pax_read_commands, pax_read_command_key, 
					cmd_verbose, 
					keep_old_files, DEFAULT_PAX_R);

	swc_print_umask(umaskbuf, sizeof(umaskbuf));

	is_target_trailing_slash = (strlen(targetpath) && 
				targetpath[strlen(targetpath) - 1] == '/');

	sw_d_msg(G, "swc_write_target_copy_script : source_type [%s]\n", strob_str(control_message));

	/*
	 * Squash the trailing slash.
	 */

	swlib_squash_double_slash(targetpath);
	if (strlen(targetpath) > 1) {
		if (targetpath[strlen(targetpath) - 1] == '/' ) {
			targetpath[strlen(targetpath) - 1] = '\0';
		}
	}

	strob_strcpy(tmp, "");	
	swlib_dirname(tmp, targetpath);
	target_dirname = strdup(strob_str(tmp));
	swlib_basename(tmp, sourcepath);
	source_basename = strdup(strob_str(tmp));
	swlib_basename(tmp, targetpath);
	target_basename = strdup(strob_str(tmp));

	/*
	 * Unpack the archive using pax
	 * The source control string was 
	 * SWBIS_SWINSTALL_SOURCE_CTL_DIRECTORY
	 */

	x_targetpath = targetpath;
	
	strob_strcpy(buffer_new, "");

	strob_sprintf(buffer_new, STROB_DO_APPEND,
	"trap '/bin/rm -f ${LOCKPATH}.lock; exit 1' 1 2 15\n"
	"echo " SWBIS_TARGET_CTL_MSG_125 ": " KILL_PID "\n"
	"%s\n" /* Session Begins */
	"%s"
	"opt_allow_no_lock=\"%s\"\n"
	"opt_force_lock=\"%s\"\n"
	"sw_retval=0\n"
	"export PATH\n"
	"export isc_path\n"
	"export swutilname\n"
	"export swbis_ignore_scripts\n"
	"export opt_preview\n"
	"export opt_to_stdout\n"
	"export LOCKPATH\n"
	"export LOCKENTRY\n"
	"swutilname=swinstall\n"
	"LOCKENTRY=$$\n"
	"PATH=`getconf PATH`:$PATH\n"
	"swbis_ignore_scripts=\"%s\"\n"
	"%s"
	"%s\n"			/* shls_bashin from shell_lib.sh */
	"%s\n"			/* shls_false_ from shell_lib.sh */
	"%s\n"			/* lf_ lock routine */
	"%s\n"			/* lf_ lock routine */
	"%s\n"			/* lf_ lock routine */
	"%s\n"			/* lf_ lock routine */
	"opt_preview=\"%s\"\n"
	"opt_to_stdout=\"%s\"\n"
	"LOCKPATH=\"%s\"\n"
	"umask %s\n"
	"blocksize=\"%s\"\n"
	"isc_path=\"%s\"\n"
	"xtarget=\"%s\"\n"
	"wcwd=`pwd`\n"
	"sh_dash_s=\"%s\"\n"
	"debug_task_shell=\"%s\"\n"
	"if test -f \"$xtarget\" -o -p \"$xtarget\" -o -b \"$xtarget\" -o -c \"$xtarget\"; then\n"
			/*
			* error
			* The target must be a directory or be non-existent.
			*/
	"	sw_retval=1\n"
	"	if test \"$sw_retval\" != \"0\"; then\n"
	"		 dd count=1 of=/dev/null 2>/dev/null\n"
	"	fi\n"
	"	%s\n" 				/* SW_SESSION_ENDS: status=1 */
	"	exit $sw_retval\n"
	"fi\n"
	"cd \"$xtarget\" 2>/dev/null\n"
	"sw_retval=$?\n"
	"if test \"$xtarget\" = '.'; then\n"
	"	sw_retval=1\n"
	"else\n"
	"	if test $sw_retval !=  \"0\"; then\n"
	"		if test \"$opt_preview\" != \"True\"; then\n"
	"			mkdir -p \"$xtarget\"\n"
	"			sw_retval=$?\n"
	"		else\n"
	"			xtarget=/\n"
	"			sw_retval=1\n"
	"		fi\n"
	"	else\n"
	"		cd \"$wcwd\"; sw_retval=1;\n"
	"	fi\n"
	"fi\n"
	"if test \"$sw_retval\" = \"0\"; then\n"
	"	%s\n"  /* SOC_CREATED */
	"fi\n"
	"cd \"$xtarget\" 2>/dev/null\n"
	"sw_retval=$?\n"
	"if test -d \"$isc_path\"; then\n"
	"	echo 1>/dev/null\n"
	"else\n"
	"	if test \"$opt_preview\" != \"True\"; then\n"
	"		(umask 027; mkdir -p \"$isc_path\"; exit $?)\n"
	"		sw_retval=$?\n"
	"	fi\n"
	"fi\n"

	/* ---------------- */

	"swexec_status=0\n"
	"export swexec_status\n"
	"case \"$sw_retval\" in\n"
	"	0)\n"
	"		# the value of catpath has no affect\n" 
	"		catpath=\"" SW_UNUSED_CATPATH  "\"\n" 
	"		echo " SWBIS_TARGET_CTL_MSG_128 ": $catpath\n"
	"		;;\n"
	"	*)\n"
	"		swexec_status=1\n"
	"		echo " SWBIS_TARGET_CTL_MSG_508 "\n"
	"		exit 1\n"
	"		;;\n"
	"esac\n"
	"shls_bashin2 \"" SWBIS_TS_uts "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_uts ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"
	"case $sw_retval in 0) ;; *) swexec_status=1; sw_retval=1; ;; esac\n"
	"case $swexec_status in	0) %s ;; *) shls_false_ ;; esac\n" 	/* SW_ANALYSIS_BEGINS */
	"case $sw_retval in 0) ;; *) swexec_status=1; sw_retval=1; ;; esac\n"
	"case \"$sw_retval\" in\n"
	"	0)\n"
	"		;;\n"
	"	*)\n"
	"		swexec_status=1\n" 			/* error! Remove the catalog directory */
	"		sw_retval=1\n"
	"		rmdir \"$catpath\" 2>/dev/null\n"
	"		;;\n"
	"esac\n"
	"case \"$opt_to_stdout\" in\n"
	"True)\n"
	"	;;\n"
	"*)\n"
	"	shls_bashin2 \"" SWBIS_TS_Get_iscs_listing "\"\n"
	"       sw_retval=$?\n"
	"       case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Get_iscs_listing ";; *) shls_false_;; esac\n"
	"       sw_retval=$?\n"
/*----*/
	"	case \"$opt_preview\" in \"\") ;; *) sw_retval=0;; esac\n"


	"       swexec_status=$sw_retval\n"
	"	shls_bashin2 \"" SWBIS_TS_Get_iscs_entry "\"\n"
	"	sw_retval=$?\n"
	"	case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Get_iscs_entry  ";; *) shls_false_;; esac\n"
	"	sw_retval=$?\n"
	"	swexec_status=$sw_retval\n"
	"	shls_bashin2 \"" SWBIS_TS_remove_catalog_entry "\"\n"
	"	sw_retval=$?\n"
	"	case $sw_retval in 0) wwf_did=1; $sh_dash_s " SWBIS_TS_remove_catalog_entry  ";; *) wwf_did=0; shls_false_;; esac\n"
	"	sw_retval=$?\n"
	"	case $sw_retval in\n"
	"		0)\n"
	"			;;\n"
	"		*)\n"
	"			case $wwf_did in\n"
	"				1)\n"
	"				echo \"swinstall: catalog entry removal failed\" 1>&2\n"
	"				sw_retval=0 # ignore this error ??\n"
	"				;;\n"
	"			esac\n"
	"			;;\n"
	"	esac\n"
	"	swexec_status=$sw_retval\n"
	"	;;\n"
	"esac\n"
	"case \"$opt_preview\" in\n"
	"True)\n"
	"	shls_bashin2 \"" SWBIS_TS_preview_task "\"\n"
	"	sw_retval=$?\n"
	"	case $sw_retval in 0) $sh_dash_s " SWBIS_TS_preview_task ";; *) shls_false_;; esac\n"
	"	swexec_status=1\n"
	"	opt_to_stdout=\"True\"\n"
	"	;;\n"
	"esac\n"
	"case \"$opt_to_stdout\" in\n"
	"True)\n"
	"	swexec_status=1\n"
	"	;;\n"
	"*)\n"
	"	;;\n"
	"esac\n"
	"shls_bashin2 \"" SWBIS_TS_make_catalog_dir "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_make_catalog_dir ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"

	"%s\n" /* <<<<----- here's the code to lock a session */

	"shls_bashin2 \"" SWBIS_TS_Load_catalog "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Load_catalog ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"

	"shls_bashin2 \"" SWBIS_TS_Load_signatures "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Load_signatures ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"

	"shls_bashin2 \"" SWBIS_TS_Catalog_unpack "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Catalog_unpack ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"

	"shls_bashin2 \"" SWBIS_TS_Load_session_options "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Load_session_options   ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"
 	
	"shls_bashin2 \"" SWBIS_TS_Load_control_sh "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Load_control_sh  ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"
	
	"shls_bashin2 \"" SWBIS_TS_Analysis_002 "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Analysis_002 ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"
	
	"if [ \"$opt_preview\" = \"\" ]; then\n"
	"	%s\n"							/* SW_ANALYSIS_ENDS */
	"fi\n"
	
	"case $swexec_status in	0) %s ;; *) shls_false_ ;; esac\n" 	/* echo SW_EXECUTION_BEGINS */
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"
	
	"shls_bashin2 \"" SWBIS_TS_Load_INSTALLED_90 "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Load_INSTALLED_90  ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"

	"shls_bashin2 \"" SWBIS_TS_Load_fileset "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Load_fileset   ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"

	"shls_bashin2 \"" SWBIS_TS_Catalog_dir_remove "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Catalog_dir_remove  ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"
	
	"shls_bashin2 \"" SWBIS_TS_Load_INSTALLED_91 "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_Load_INSTALLED_91 ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"
	
	"shls_bashin2 \"" SWBIS_TS_make_live_INSTALLED  "\"\n"
	"sw_retval=$?\n"
	"case $sw_retval in 0) $sh_dash_s " SWBIS_TS_make_live_INSTALLED ";; *) shls_false_;; esac\n"
	"sw_retval=$?\n"
	"swexec_status=$sw_retval\n"

	"%s\n" /* <<<<----- here's the code to unlock a session */

	"case $swexec_status in	0) %s ;; *) shls_false_ ;; esac\n" 	/* SW_EXECUTION_ENDS */
	"case $sw_retval in 0) ;; *) swexec_status=1; sw_retval=1; ;; esac\n"
	"if [ \"$opt_preview\" ]; then\n"
	"	swexec_status=0\n"
	"	sw_retval=0\n"
	"fi\n"
	"sleep %d\n"
	"%s\n" /* SWI_MAIN_SCRIPT_ENDS */
	"%s\n" /* Session Ends */
	"%s\n"
	,
/*_% */	TEVENT(2, vlv, SW_SESSION_BEGINS, ""),
/*_% */	swicol_brace_marks(subsh, "target", 'L', nhops, vlv),
/*_% */	opt_force_lock,
/*_% */	opt_allow_no_lock,
/*_% */	ignore_scripts,
/*_% */	strob_str(set_vx),
/*_% */	shlib_get_function_text_by_name("shls_bashin2", shell_lib_buf, NULL),
/*_% */	shlib_get_function_text_by_name("shls_false_", shell_lib_buf, NULL),
/*_% */	shlib_get_function_text_by_name("lf_make_lockfile_name", shell_lib_buf, NULL),
/*_% */	shlib_get_function_text_by_name("lf_make_lockfile_entry", shell_lib_buf, NULL),
/*_% */	shlib_get_function_text_by_name("lf_test_lock", shell_lib_buf, NULL),
/*_% */	shlib_get_function_text_by_name("lf_remove_lock", shell_lib_buf, NULL),
/*_% */	opt_char_preview,
/*_% */	opt_char_to_stdout,
/*_% */	strob_str(catlockbuffer),
/*_% */	umaskbuf,
/*_% */	blocksize,
/*_% */	isc_relpath,
/*_% */	x_targetpath, 
/*_% */	sh_dash_s,
/*_% */	debug_task_shell,	/* debug_task_shell */	
/*_% */	TEVENT(2, vlv, SW_SESSION_ENDS, "status=$sw_retval"),
/*_% */	TEVENT(2, vlv, SW_SOC_CREATED, x_targetpath),
/*_% */	TEVENT(2, vlv, SW_ANALYSIS_BEGINS, ""),
	swpl_shellfrag_session_lock(lockfrag_buffer, vlv),
/*_% */	TEVENT(2, 1/*verbose level*/, SW_ANALYSIS_ENDS, "status=$sw_retval"),
/*_% */	TEVENT(2, vlv, SW_EXECUTION_BEGINS, ""),
	swpl_shellfrag_session_unlock(unlockfrag_buffer, vlv),
/*_% */	TEVENT(2, vlv, SW_EXECUTION_ENDS, "status=$sw_retval"),
/*_% */	delaytime,
	TEVENT(2, -1, SWI_MAIN_SCRIPT_ENDS, "status=0"),
/*_% */	TEVENT(2, vlv, SW_SESSION_ENDS, "status=$sw_retval"),
/*_% */	swicol_brace_marks(subsh2, "install_target", 'R', nhops, vlv)
	);

	xx = strob_str(buffer_new);
	ret = atomicio((ssize_t (*)(int, void *, size_t))write, ofd, xx, strlen(xx));
	if (ret != (int)strlen(xx)) {
		return 1;
	}
	free(targetpath);
	free(source_basename);
	free(target_dirname);
	free(target_basename);
        if (G->g_target_script_name) {
                swlib_tee_to_file(G->g_target_script_name, -1, xx, -1, 0);
        }

	strob_close(lockfrag_buffer);
	strob_close(unlockfrag_buffer);
	strob_close(catlockbuffer);
	strob_close(tmp);
	strob_close(tmpexit);
	strob_close(set_vx);
	strob_close(to_devnull);
	strob_close(subsh);
	strob_close(subsh2);
	strob_close(shell_lib_buf);
	
	if (ret <= 0) retval = 1;
	sw_d_msg(G, "swc_write_target_copy_script : retval = [%d]\n", retval);
	/*
	 *  retval:
	 *	0 : Ok
	 *	1 : Error
	 */
	return retval;
}

/* ---------------------------------------------------------------- */
/* ------------------ PUBLIC ---- PUBLIC -------------------------- */
/* ----- Routines below are likely called from swinstall.c -------- */
/* ---------------------------------------------------------------- */

int
swinstall_write_target_install_script(GB * G,
		int ofd,
		char * fp_targetpath, 
		STROB * control_message, 
		char * sourcepath,
		int delaytime,
		int keep_old_files,
		int nhops,
		int vlv,
		char * pax_read_command_key,
		char * hostname,
		char * blocksize,
		SWI_DISTDATA * distdata,
		char * installed_software_catalog,
		int opt_preview,
		char * sh_dash_s,
		int alt_catalog_root
		)
{
	return
	write_target_install_script(
		G,
		ofd,
		fp_targetpath, 
		control_message, 
		sourcepath,
		delaytime,
		keep_old_files,
		nhops,
		vlv,
		pax_read_command_key,
		hostname,
		blocksize,
		distdata,
		installed_software_catalog,
		opt_preview,
		sh_dash_s,
		alt_catalog_root);
}

int
swinstall_arfinstall(GB * G, SWI * swi, int ofd, int * g_signal_flag,
	char * target_path, char * catalog_prefix, VPLOB * swspecs,
	int section, int altofd, int opt_preview, char * pax_read_command_key,
	int alt_catalog_root, int event_fd, struct extendedOptions * opta,
	int keep_old_files, uintmax_t * pstatbytes,
	int allow_missing_files)
{
	int ret;
	int format;
	XFORMAT * xformat = swi->xformatM;

	if (swi->swi_pkgM->target_pathM) {
		free(swi->swi_pkgM->target_pathM);
		swi->swi_pkgM->target_pathM = NULL;
	}

	if (swi->swi_pkgM->catalog_entryM) {
		free(swi->swi_pkgM->catalog_entryM);
		swi->swi_pkgM->catalog_entryM = NULL;
	}
	swi->swi_pkgM->target_pathM = strdup(target_path);
	swi->swi_pkgM->catalog_entryM = strdup(catalog_prefix);

	/*
	 * catalog_prefix is nominally:
	 *     var/lib/swbis/catalog/<name>/<version>/<seq_no>
	 * 
	 * target path is the absolute path
	 */
	
	E_DEBUG("HERE");
	if (swi->swc_idM == SWC_U_L) {
		;
	} else {
		if (
			swlib_check_clean_absolute_path(swi->swi_pkgM->target_pathM) ||
			0
		) {
			E_DEBUG2("HERE path is [%s]", swi->swi_pkgM->target_pathM);
			return 10;
		}
	}
	
	E_DEBUG("HERE");
	if (strlen(swi->swi_pkgM->catalog_entryM)) {
		/* Disable the check for zero length strings since this is what
		   happens when swlist uses this routine in list mode */
		if (
			swlib_check_clean_path(swi->swi_pkgM->catalog_entryM) ||
			0
		) {
			return 3;
		}
	}

	E_DEBUG("HERE");
	format = xformat_get_format(xformat);

	swicol_set_targetpath(swi->swicolM, target_path);

	{ /* -+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+ */ 
		/*
		* Scratch test area
		*/
		/*
		STROB * tmp = strob_open(10);
		swextopt_write_session_options(tmp, opta, SWC_U_I);
		fprintf(stderr, "%s", strob_str(tmp));
		strob_close(tmp);
		*/
	} /* -+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+ */ 

	E_DEBUG("HERE");
	if (format == arf_tar || format == arf_ustar) {
		/*
		 * All tar formats
		 */
		ret = arfinstall_tar(G, swi, ofd, g_signal_flag,
				target_path, catalog_prefix, swspecs,
				altofd, opt_preview, pax_read_command_key,
				alt_catalog_root, event_fd, opta,
				keep_old_files, pstatbytes, allow_missing_files);
	} else {
		/*
		 * all other formats
		 */
		fprintf(stderr, "%s: cpio formats currently not supported\n",
			swlib_utilname_get());
		ret = 1;
	}
	return ret;
}
