/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995-1997  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997-1998  Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <syslog.h>
#include "freedom.h"

enum message_type
scan_message (const char *infilename, char *to, const int pass)
{
  FILE *infile;
  unsigned flag = 0;
  int headers = 1, colhead = 0, linenum = 0;
  char line[BUFSIZ], nexthop[BUFSIZ], cutmarks[BUFSIZ], newsgroups[BUFSIZ],
    encryptsub[BUFSIZ], encryptkey[BUFSIZ], latency[BUFSIZ], url[BUFSIZ];
  const char *TO = USE_MIX ? "" : "To: ";

  to[0] = '\0';

  if (!(infile = fopen (infilename, "r")))
    return NON_ANON;
  if (USE_SYSLOG)
    syslog (LOG_DEBUG, "Scanning message");
  while (fgets (line, sizeof (line), infile) && linenum < 2)
    {
      if (!headers)
	{
	  if (linenum > colhead || !blank (line))
	    linenum++;
	  if (linenum == 1 && hdrmarks (line))
	    {
	      headers = 1;
	      colhead = 1;
	    }
	}
      else
	{
	  if (blank (line))
	    headers = 0;
	  else
	    {
	      if (linenum == 0)
		{
		  /* Real headers.  */
		  if (strileft (line, "from") || strileft (line, "reply-to:")
		      || strileft (line, "sender:"))
		    {
		      if (rxfind (line, SOURCE_BLOCK))
			return BLOCKED_SENDER;
		    }
		  else if (strileft (line, "subject:"))
		    {
		      flushleft (line, sizeof ("subject:") - 1);
		      chop (line);
		      if (strieq (line, "help"))
			return HELP_REQ;
		      else if (strieq (line, "remailer-help"))
			return HELP_REQ;
		      else if (strieq (line, "freedom-help"))
			return HELP_REQ;
		      else if (strieq (line, "remailer-conf"))
			return CONF_REQ;
		      else if (strieq (line, "freedom-conf"))
			return CONF_REQ;
		      else if (USE_STATS && strieq (line, "remailer-stats"))
			return STATS_REQ;
		      else if (USE_STATS && strieq (line, "freedom-stats"))
			return STATS_REQ;
		      else if ((ALLOW_PGP || ALLOW_GPG)
			       && strieq (line, "remailer-key"))
			return KEY_REQ;
		      else if ((ALLOW_PGP || ALLOW_GPG)
			       && strieq (line, "freedom-key"))
			return KEY_REQ;
		    }
		}
	      if (strileft (line, "encrypted:"))
		{
		  flushleft (line, sizeof ("encrypted:") - 1);
		  chop (line);
		  if (USE_SYSLOG)
		    syslog (LOG_DEBUG, "%s encrypted message detected", line);
		  if ((ALLOW_PGP || ALLOW_GPG) && strieq (line, "pgp"))
		    return PGP_MESSAGE;
		}
	      /* The Null: directive for bit-bucket mail makes sense only
	         in the ::'ed headers of encrypted cover traffic messages.  */
	      else if (strileft (line, "null:") && colhead)
		{
		  chop (line);
		  if (USE_SYSLOG)
		    syslog (LOG_DEBUG, "%s detected", line);
		  return NULL_MESSAGE;
		}
	      else if (strileft (line, "cutmarks:"))
		{
		  flushleft (line, sizeof ("cutmarks:") - 1);
		  strcpy (cutmarks, line);
		  chop (cutmarks);
		  if (strlen (cutmarks) > 0)
		    flag |= (1 << 8);
		}
	      else if (ALLOW_PGP && ALLOW_WWW && strileft (line, "get-url:"))
		{
		  flushleft (line, sizeof ("get-url:") - 1);
		  strcpy (url, line);
		  chop (url);
		  if (strlen (url) > 0 && pass != 2)
		    flag |= (1 << 7);
		}
	      else if (strileft (line, "encrypt-subject:"))
		{
		  flushleft (line, sizeof ("encrypt-subject:") - 1);
		  strcpy (encryptsub, line);
		  chop (encryptsub);
		  if (strlen (encryptsub) > 0)
		    flag |= (1 << 6);
		}
	      else if (ALLOW_PGP && strileft (line, "encrypt-key:"))
		{
		  flushleft (line, sizeof ("encrypt-key:") - 1);
		  strcpy (encryptkey, line);
		  chop (encryptkey);
		  if (strlen (encryptkey) > 0)
		    flag |= (1 << 5);
		}
	      else if (ALLOW_LTIME && strileft (line, "latent-time:"))
		{
		  flushleft (line, sizeof ("latent-time:") - 1);
		  strcpy (latency, line);
		  chop (latency);
		  if (strlen (latency) > 0)
		    flag |= (1 << 4);
		}
	      else if (USE_MIX && strileft (line, "remix-to:"))
		{
		  flushleft (line, sizeof ("remix-to:") - 1);
		  strcpy (nexthop, line);
		  chop (nexthop);
		  if (strifind (nexthop, REMAILER_ADDR))
		    {
		      /* Don't remix to ourselves.  */
		      strncat (to, TO, BUFSIZ - strlen (to) - 1);
		      strncat (to, line, BUFSIZ - strlen (to) - 1);
		      flag |= (1 << 0);
		    }
		  else if (findline (nexthop, remixlist))
		    /* Next hop is a mixmaster remailer.  */
		    flag |= (1 << 3);
		}
	      else if (ALLOW_PGP && strileft (line, "encrypt-to:"))
		{
		  flushleft (line, sizeof ("encrypt-to:") - 1);
		  strcpy (nexthop, line);
		  chop (nexthop);
		  if (strifind (nexthop, REMAILER_ADDR))
		    {
		      /* Don't repgp to ourselves.  */
		      strncat (to, TO, BUFSIZ - strlen (to) - 1);
		      strncat (to, line, BUFSIZ - strlen (to) - 1);
		      flag |= (1 << 0);
		    }
		  else if (findline (nexthop, repgplist))
		    /* Next hop is a pgprec cpunk remailer.  */
		    flag |= (1 << 2);
		}
	      else if (ALLOW_POST && strileft (line, "anon-post-to:")
		       && flag < (1 << 2))
		{
		  flushleft (line, sizeof ("anon-post-to:") - 1);
		  chop (line);
		  if (!rxfind (line, GROUP_BLOCK))
		    {
		      if (flag & (1 << 1))
			/* Add comma before next newsgroup.  */
			strncat (newsgroups, ",",
				 BUFSIZ - strlen (newsgroups) - 1);
		      else
			{
			  /* Add header keyword before first newsgroup.  */
			  if (USE_MIX)
			    strcpy (newsgroups, "post: ");
			  else
			    strcpy (newsgroups, "Newsgroups: ");
			}
		      strncat (newsgroups, line,
			       BUFSIZ - strlen (newsgroups) - 1);
		      flag |= (1 << 1);
		    }
		}
	      else if (strileft (line, "remail-to:") && flag < (1 << 2))
		{
		  flushleft (line, sizeof ("remail-to:") - 1);
		  if (!rxfind (line, DEST_BLOCK))
		    {
		      strncat (to, TO, BUFSIZ - strlen (to) - 1);
		      strncat (to, line, BUFSIZ - strlen (to) - 1);
		      flag |= (1 << 0);
		    }
		}
	      else if (strileft (line, "anon-to:") && flag < (1 << 2))
		{
		  flushleft (line, sizeof ("anon-to:") - 1);
		  strcpy (nexthop, line);
		  chop (nexthop);
		  if (USE_MIX == 2 && !strifind (nexthop, REMAILER_ADDR)
		      && findline (nexthop, remixlist))
		    flag |= (1 << 3);
		  else if (ALLOW_PGP && !strifind (nexthop, REMAILER_ADDR)
			   && findline (nexthop, repgplist))
		    flag |= (1 << 2);
		  else if (!rxfind (line, DEST_BLOCK))
		    {
		      strncat (to, TO, BUFSIZ - strlen (to) - 1);
		      strncat (to, line, BUFSIZ - strlen (to) - 1);
		      flag |= (1 << 0);
		    }
		}
	      else if (strileft (line, "request-remailing-to:")
		       && flag < (1 << 2))
		{
		  flushleft (line, sizeof ("request-remailing-to:") - 1);
		  strcpy (nexthop, line);
		  chop (nexthop);
		  if (USE_MIX == 2 && !strifind (nexthop, REMAILER_ADDR)
		      && findline (nexthop, remixlist))
		    flag |= (1 << 3);
		  else if (ALLOW_PGP && !strifind (nexthop, REMAILER_ADDR)
			   && findline (nexthop, repgplist))
		    flag |= (1 << 2);
		  else if (!rxfind (line, DEST_BLOCK))
		    {
		      strncat (to, TO, BUFSIZ - strlen (to) - 1);
		      strncat (to, line, BUFSIZ - strlen (to) - 1);
		      flag |= (1 << 0);
		    }
		}
	      else if (strileft (line, "anon-send-to:") && flag < (1 << 2))
		{
		  flushleft (line, sizeof ("anon-send-to:") - 1);
		  strcpy (nexthop, line);
		  chop (nexthop);
		  if (USE_MIX == 2 && !strifind (nexthop, REMAILER_ADDR)
		      && findline (nexthop, remixlist))
		    flag |= (1 << 3);
		  else if (ALLOW_PGP && !strifind (nexthop, REMAILER_ADDR)
			   && findline (nexthop, repgplist))
		    flag |= (1 << 2);
		  else if (!rxfind (line, DEST_BLOCK))
		    {
		      strncat (to, TO, BUFSIZ - strlen (to) - 1);
		      strncat (to, line, BUFSIZ - strlen (to) - 1);
		      flag |= (1 << 0);
		    }
		}
	    }
	}
    }
  fclose (infile);

  if (ALLOW_PGP == 2)
    {
      /* PGP or GPG encryption required.  */
      infile = fopen (infilename, "r");
      /* Read first two lines from infile.  */
      fgets (line, sizeof (line), infile);
      fgets (line, sizeof (line), infile);
      fclose (infile);
      if (!blank (line))
	/* No one-line fake header.  */
	return NON_ANON;
    }

  /* Process headers in order: cutmarks, get-url, encrypt-subject, encrypt-key,
   * latent-time, remix-to.  If these don't exist, then post or remail the
   * message if that is where it's headed; else, return NON_ANON to send it
   * to the spool or bucket.
   */

  if (flag & (1 << 8))
    {
      strcpy (to, cutmarks);
      return CUTMARKS;
    }
  if (flag & (1 << 7))
    {
      strcpy (to, url);
      return URL_REQUEST;
    }
  if (flag & (1 << 6))
    {
      strcpy (to, encryptsub);
      return ENCRYPT_SUB;
    }
  if (flag & (1 << 5))
    {
      strcpy (to, encryptkey);
      return ENCRYPT_KEY;
    }
  if (flag & (1 << 4))
    {
      strcpy (to, latency);
      return LATENT_TIME;
    }
  if (flag & (1 << 3))
    {
      strcpy (to, nexthop);
      return REMIX_MESSAGE;
    }
  if (flag & (1 << 2))
    {
      strcpy (to, nexthop);
      return REPGP_MESSAGE;
    }
  if (flag & (1 << 1))
    {
      strncat (to, newsgroups, BUFSIZ - strlen (to) - 1);
      strncat (to, "\n", BUFSIZ - strlen (to) - 1);
      return ANON_POST;
    }
  if (flag & (1 << 0))
    return ANON_MESSAGE;

  return NON_ANON;
}

void
get_from (const char *infilename, char *from)
{
  /* From the Mixmaster distribution http://www.thur.de/ulf/mix/
   * Modified to use sscanf and check for Reply-To: header; added
   * some comments -Futplex <futplex@pseudonym.com> 95/07/28
   * Modified by hanne@squirrel.owl.de for Freedom 2.3 1998
   */

  FILE *infile;
  char line[BUFSIZ];
  int BestFound = 2, i;
  static char *Header[] =
  {"reply-to:", "from:"};
  from[0] = '\0';

  /* RFC 822 <URL: http://ds.internic.net/rfc/rfc822.txt> recommends
   * that automatic replies be directed to originating address(es)
   * according to the following priority:
   * (0) contents of Reply-To: header, if present
   * (1) contents of From: header, if no Reply-To: header is present
   */

  if (!(infile = fopen (infilename, "r")))
    return;
  if (USE_SYSLOG)
    syslog (LOG_DEBUG, "Looking for reply address");

  /* Read headers one line at a time.  */
  while (fgets (line, sizeof (line), infile))
    {
      /* Empty line delineates the header/body boundary.  */
      if (blank (line))
	break;

      /* Look for an RFC 822 preferred reply header.  */
      for (i = 0; i < BestFound; i++)
	{
	  if (strileft (line, Header[i]))
	    {
	      flushleft (line, strlen (Header[i]));
	      strcpy (from, line);
	      chop (from);
	      BestFound = i;
	    }
	}
      /* Stop looking if we've already found a Reply-To: header.  */
      if (BestFound == 0)
	break;
    }
  fclose (infile);
}

void
remove_headers (FILE * infile, FILE * outfile, enum message_type anon)
{
  fpos_t pos;
  int headers = 1, colhead = 0, addhead = 0, linenum = 0;
  int subject = 0, newsgroups = 0, from = 0, replyto = 0;
  char line[BUFSIZ], tmpline[BUFSIZ];
  enum stats_flag flag;

  fgets (line, sizeof (line), infile);
  rewind (infile);
  if (USE_STATS)
    {
      if (strcmp (line, "Recursive: Cut\n"))
	flag = STATS_MESSAGE;
      if (!strcmp (line, "Recursive: PGP\n"))
	flag = STATS_PGP;
      updatestats (flag);
    }

  while (fgets (line, sizeof (line), infile))
    {
      if (!headers)
	{
	  if (linenum > colhead || !blank (line))
	    linenum++;
	  if (linenum == 1 && hdrmarks (line))
	    {
	      headers = 1;
	      colhead = 1;
	    }
	  else if (linenum == colhead + 1)
	    {
	      if (hashmarks (line))
		addhead = 1;
	      else
		{
		  if (anon == REMIX_MESSAGE || anon == ANON_POST)
		    fputs ("Subject: none\n", outfile), subject++;
		  fputs ("\n", outfile);
		  fputs (line, outfile);
		}
	    }
	  else
	    {
	      if (addhead)
		{
		  /* Read RFC 822 multi-line headers.  */
		  while (1)
		    {
		      fgetpos (infile, &pos);
		      if (fgets (tmpline, sizeof (tmpline), infile)
			  && (tmpline[0] == ' ' || tmpline[0] == '\t'))
			strncat (line, tmpline,
				 sizeof (line) - strlen (line) - 1);
		      else
			break;
		    }
		  fsetpos (infile, &pos);

		  if (blank (line) || ekmarks (line))
		    {
		      if ((anon == REMIX_MESSAGE || anon == ANON_POST
			   || newsgroups) && !subject)
			fputs ("Subject: none\n", outfile), subject++;
		      if (ekmarks (line))
			fputs ("\n", outfile);
		      fputs (line, outfile);
		      addhead = 0;
		    }
		  else if (!rxfind (line, HDRFILTER))
		    {
		      if (strileft (line, "to:") || strileft (line, "cc:")
			  || strileft (line, "bcc:"))
			{
			  if (!rxfind (line, DEST_BLOCK))
			    fputs (line, outfile);
			}
		      else if (strileft (line, "reply-to:"))
			{
			  if (!replyto && !rxfind (line, DEST_BLOCK))
			    fputs (line, outfile), replyto++;
			}
		      else if (strileft (line, "from:"))
			{
			  if (!from && !rxfind (line, DEST_BLOCK))
			    fputs (line, outfile), from++;
			}
		      else if (strileft (line, "newsgroups:"))
			{
			  if (anon != ANON_POST && !newsgroups
			      && !rxfind (line, GROUP_BLOCK))
			    fputs (line, outfile), newsgroups++;
			}
		      else if (strileft (line, "subject:"))
			{
			  if (!subject)
			    fputs (line, outfile), subject++;
			}
		      else
			fputs (line, outfile);
		    }
		}
	      else
		{
		  if (linenum > colhead || !blank (line))
		    fputs (line, outfile);
		}
	    }
	}
      else
	{
	  if (blank (line))
	    headers = 0;
	}
    }
  if ((anon == REMIX_MESSAGE || anon == ANON_POST || newsgroups)
      && (linenum == colhead || addhead) && !subject)
    fputs ("Subject: none\n", outfile), subject++;
}
