/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995-1997  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997-1998  Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <fcntl.h>
#include <time.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/file.h>
#include "freedom.h"

int USE_SYSLOG = 0;
int USE_STATS = 1;
int USE_MIX = 0;
int ALLOW_PGP = 1;
int ALLOW_GPG = 0;
int ALLOW_POST = 1;
int ALLOW_WWW = 1;
int ALLOW_LTIME = 1;
int SIZE_LIMIT = 0;
char remailer_dir[BUFSIZ] = "";
char cur_dir[BUFSIZ];
char MIX_DIR[BUFSIZ] = "";
char QUEUE_DIR[BUFSIZ] = "";
char PGP_DIR[BUFSIZ] = "";
char GPG_DIR[BUFSIZ] = "";
char MAIL_SPOOL[BUFSIZ] = "/dev/null";
char MIXMASTER[BUFSIZ] = "";
char TYPE2_LIST[BUFSIZ] = "";
/* The -t option tells sendmail to expect a To: header in stdin.  */
char SENDMAIL[BUFSIZ] = "/usr/sbin/sendmail -t";
/* The -h option tells inews to expect message headers in stdin.  */
char INEWS[BUFSIZ] = "/usr/bin/inews -h";
char PGP[BUFSIZ] = "/usr/bin/pgp";
char GPG[BUFSIZ] = "/usr/local/bin/gpg";
char WGET[BUFSIZ] = "/usr/bin/wget";
char HELP_FILE[BUFSIZ] = "freedom-help";
char STATS_FILE[BUFSIZ] = "freedom-stats";
char KEY_FILE[BUFSIZ] = "freedom-key";
char PASS_FILE[BUFSIZ] = "passphrase";
char REPGP_KEYS[BUFSIZ] = "repgp-keys.pgp";
char SOURCE_BLOCK[BUFSIZ] = "blocked.source";
char DEST_BLOCK[BUFSIZ] = "blocked.destination";
char GROUP_BLOCK[BUFSIZ] = "blocked.newsgroup";
char URL_BLOCK[BUFSIZ] = "blocked.url";
char HDRFILTER[BUFSIZ] = "headers.del";
char REMAILER_NAME[BUFSIZ] = "Anonymous Remailer";
char ANON_NAME[BUFSIZ] = "Anonymous";
char ORGANIZATION[BUFSIZ] = "Anonymous Posting Service";
char MAIL2NEWS[BUFSIZ] = "mail2news@nym.alias.net";
char REMAILER_ADDR[BUFSIZ] = "";
char ANON_ADDR[BUFSIZ] = "";
char COMPLAINTS[BUFSIZ] = "";
FILE *repgplist = NULL;
FILE *remixlist = NULL;

/* Read word from freedom.conf.  */
#define read_conf(t) \
if (!strncmp(line, #t, sizeof(#t) - 1) && isspace(line[sizeof(#t) - 1]))\
  sscanf(line, "%s %s", junk, t);

/* Read line from freedom.conf, with whitespace.  */
#define read_conf_w(t) \
if (!strncmp(line, #t, sizeof(#t) - 1) && isspace(line[sizeof(#t) - 1])) {\
  foo = line;\
  while (*foo > ' ')\
    /* Scan to first space.  */\
    foo++;\
  while (*foo > '\0' && *foo <= ' ')\
    /* Scan to first non-whitespace. */\
    foo++;\
  strcpy(t, foo);\
  chop(t);\
}

/* Read number from freedom.conf.  */
#define read_conf_i(t) \
if (!strncmp(line, #t, sizeof(#t) - 1) && isspace(line[sizeof(#t) - 1]))\
  sscanf(line, "%s %d", junk, &t);

int
init_remailer (void)
{
  FILE *infile;
  char line[BUFSIZ], junk[BUFSIZ], *foo;

  umask (077);
  remailer_dir[0] = '\0';
  strncat (remailer_dir, REMAILER_DIR, sizeof (remailer_dir) - 1);

  /* Environment variable replaces defined value.  */
  if ((foo = getenv ("FREEDOM")))
    {
      remailer_dir[0] = '\0';
      strncat (remailer_dir, foo, sizeof (remailer_dir) - 1);
    }

  if (!getcwd (cur_dir, sizeof (cur_dir)))
    {
      /* Could not get current dir.  */
      fputs ("Could not get current directory!\n", stderr);
      /* So set cur_dir to remailer_dir.  */
      strcpy (cur_dir, remailer_dir);
    }
  if (cur_dir[strlen (cur_dir) - 1] != '/')
    strcat (cur_dir, "/");

  /* Drop our privileges if remailer_dir != REMAILER_DIR.  */
  if (strcmp (REMAILER_DIR, remailer_dir))
    setuid (getuid ()), setgid (getgid ());

  if (remailer_dir[0] == '\0')
    {
      foo = getenv ("HOME");
      if (foo)
	{
	  /* Try "~/freedom".  */
	  strncat (remailer_dir, foo, sizeof (remailer_dir) - 1);
	  strncat (remailer_dir, "/freedom",
		   sizeof (remailer_dir) - strlen (remailer_dir) - 1);
	}
    }
  if (chdir (remailer_dir))
    fprintf (stderr, "Error changing to directory %s\n", remailer_dir);

  /* Seed the random number generator with time and pid.  */
  srand ((time (NULL) ^ (getpid () | (getpid () << 16))) & RAND_MAX);

  if (!(infile = fopen ("freedom.conf", "r")))
    {
      fputs ("Error: Could not open freedom.conf!\n", stderr);
      return -1;
    }
  else
    {
      while (fgets (line, sizeof (line), infile))
	{
	  if (line[0] != '#' && !blank (line))
	    {
	      if (!strncmp (line, "REMAILER_DIR", sizeof ("REMAILER_DIR") - 1))
		{
		  sscanf (line, "%s %s", junk, remailer_dir);
		  if (chdir (remailer_dir))
		    fprintf (stderr, "Error changing to REMAILER_DIR %s\n",
			     remailer_dir);
		}
	      read_conf_i (USE_SYSLOG);
	      read_conf_i (USE_STATS);
	      read_conf_i (USE_MIX);
	      read_conf_i (ALLOW_PGP);
	      read_conf_i (ALLOW_GPG);
	      read_conf_i (ALLOW_POST);
	      read_conf_i (ALLOW_WWW);
	      read_conf_i (ALLOW_LTIME);
	      read_conf_i (SIZE_LIMIT);
	      read_conf (MIX_DIR);
	      read_conf (QUEUE_DIR);
	      read_conf (PGP_DIR);
	      read_conf (GPG_DIR);
	      read_conf (MAIL_SPOOL);
	      read_conf_w (SENDMAIL);
	      read_conf_w (INEWS);
	      read_conf (MIXMASTER);
	      read_conf (TYPE2_LIST);
	      read_conf (PGP);
	      read_conf (GPG);
	      read_conf (WGET);
	      read_conf (HELP_FILE);
	      read_conf (STATS_FILE);
	      read_conf (KEY_FILE);
	      read_conf (PASS_FILE);
	      read_conf (REPGP_KEYS);
	      read_conf (SOURCE_BLOCK);
	      read_conf (DEST_BLOCK);
	      read_conf (GROUP_BLOCK);
	      read_conf (URL_BLOCK);
	      read_conf (HDRFILTER);
	      read_conf_w (REMAILER_NAME);
	      read_conf_w (ANON_NAME);
	      read_conf_w (ORGANIZATION);
	      read_conf (MAIL2NEWS);
	      read_conf (REMAILER_ADDR);
	      read_conf (ANON_ADDR);
	      read_conf (COMPLAINTS);
	    }
	}
      if (ALLOW_LTIME && QUEUE_DIR[0] == '\0')
	{
	  strcpy (QUEUE_DIR, REMAILER_DIR);
	  strncat (QUEUE_DIR, "/queue",
		   sizeof (QUEUE_DIR) - strlen (QUEUE_DIR) - 1);
	}

      if (USE_SYSLOG)
	openlog (REMAILER_BIN, LOG_PID, SYSLOG_FACILITY);

      if (USE_MIX)
	{
	  if (MIXMASTER[0] == '\0')
	    {
	      strcpy (MIXMASTER, MIX_DIR);
	      strncat (MIXMASTER, "/mixmaster",
		       sizeof (MIXMASTER) - strlen (MIXMASTER) - 1);
	    }
	  if (TYPE2_LIST[0] == '\0')
	    {
	      strcpy (TYPE2_LIST, MIX_DIR);
	      strncat (TYPE2_LIST, "/type2.list",
		       sizeof (TYPE2_LIST) - strlen (TYPE2_LIST) - 1);
	    }
	  if (!(remixlist = fopen (TYPE2_LIST, "r")))
	    {
	      if (USE_SYSLOG)
		syslog (LOG_WARNING, "Can't open %s: %m", TYPE2_LIST);
	      USE_MIX = 0;
	    }
	}

      if (ALLOW_PGP)
	{
	  if (PGP_DIR[0] == '\0')
	    {
	      strcpy (PGP_DIR, REMAILER_DIR);
	      strncat (PGP_DIR, "/.pgp",
		       sizeof (PGP_DIR) - strlen (PGP_DIR) - 1);
	    }
	  /* First try to open ./type1.list.  */
	  if (!(repgplist = fopen ("type1.list", "r")))
	    {
	      /* If that fails, create a new list with 'pgp -kv'. */
	      repgplist = fopen ("type1.list", "w+");
	      run_cmd (PGP, "-kv", REPGP_KEYS, NULL, repgplist, stderr);
	      unlink ("type1.list");
	    }
	}

      if (ALLOW_GPG)
	{
	  if (GPG_DIR[0] == '\0')
	    {
	      strcpy (GPG_DIR, REMAILER_DIR);
	      strncat (GPG_DIR, "/.gnupg",
		       sizeof (GPG_DIR) - strlen (GPG_DIR) - 1);
	    }
	}

      if (ANON_NAME[0] == '\0')
	strcpy (ANON_NAME, REMAILER_NAME);
      if (ANON_ADDR[0] == '\0')
	strcpy (ANON_ADDR, REMAILER_ADDR);
      if (COMPLAINTS[0] == '\0')
	strcpy (COMPLAINTS, REMAILER_ADDR);

      fclose (infile);
      return 0;
    }
}

int
main (int argc, char *argv[])
{
  FILE *infile, *outfile, *p;
  char tempfilename[11], bFilename[12], dFilename[12], eFilename[12], cFilename[12],
    kFilename[12], uFilename[12], line[BUFSIZ];
  char *poolfilename = alloca (strlen (MIX_DIR) + sizeof ("/mailXXXXXX"));
  char *send_to = alloca (BUFSIZ);
  char *message_from = alloca (BUFSIZ);
  int recursing, trimmings = 0;
  int tempfd;
  enum message_type anon;
  struct stat filestat;

  init_remailer ();

  if (argc > 1)
    {
      if (!strcmp (argv[1], "-q"))
	{
	  if (USE_SYSLOG)
	    syslog (LOG_DEBUG, "Running the latent message queue");
	  runlatentqueue ();
	  exit (0);
	}
      else
	{
	  printf ("Freedom Remailer %s\n"
	     "(C) John B. Fleming 1995-1997, Johannes Kroeger 1997-1998\n\n"
		  "Usage: %s\t\t# read mail on stdin and process\n"
		  "       %s -q\t# run queue of latent messages\n",
		  VERSION, REMAILER_BIN, REMAILER_BIN);
	  exit (0);
	}
    }

  /* Make sure we don't overwrite other messages.  */
  sprintf (tempfilename, "mailXXXXXX");
  if ((tempfd = mkstemp (tempfilename)) == -1)
    {
      if (USE_SYSLOG)
	syslog (LOG_ERR, "Can't write new message: %m");
      exit (-1);
    }
  else
    {
      sprintf (bFilename, "%sB", tempfilename);
      sprintf (dFilename, "%sD", tempfilename);
      sprintf (eFilename, "%sE", tempfilename);
      sprintf (cFilename, "%sC", tempfilename);
      sprintf (kFilename, "%sK", tempfilename);
      sprintf (uFilename, "%sU", tempfilename);

      /* Copy message coming in on stdin to a temporary file.  */
      outfile = fdopen (tempfd, "w");
      if (USE_SYSLOG)
	syslog (LOG_DEBUG, "Writing new message to %s", tempfilename);
      while (fgets (line, sizeof (line), stdin))
	fputs (line, outfile);
      fclose (outfile);
    }

  if (SIZE_LIMIT)
    {
      stat (tempfilename, &filestat);
      if (filestat.st_size > SIZE_LIMIT)
	{
	  if (USE_SYSLOG)
	    {
	      syslog (LOG_WARNING, "Message too large: %ld bytes",
		      filestat.st_size);
	      syslog (LOG_DEBUG, "Deleting %s", tempfilename);
	    }
	  unlink (tempfilename);
	  exit (1);
	}
    }

  recursing = 1;
  while (recursing)
    {
      anon = scan_message (tempfilename, send_to, recursing);
      recursing = 0;

      switch (anon)
	{
	case BLOCKED_SENDER:
	  if (USE_SYSLOG)
	    syslog (LOG_NOTICE, "Discarding blocked message");
	  unlink (tempfilename);
	  break;

	case NULL_MESSAGE:
	  if (USE_SYSLOG)
	    syslog (LOG_DEBUG, "Discarding dummy message");
	  unlink (tempfilename);
	  break;

	case PGP_MESSAGE:
	  /* Extract first PGP packet of incoming message to eFilename.  */
	  p = fopen (tempfilename, "r");
	  outfile = fopen (eFilename, "w");
	  while (fgets (line, sizeof (line), p) && !beginpgp (line))
	    continue;
	  do
	    fputs (line, outfile);
	  while (fgets (line, sizeof (line), p) && !endpgp (line));
	  fputs (line, outfile);
	  fclose (outfile);

	  /* Decrypt PGP packet in eFilename to dFilename.  */
	  outfile = fopen (dFilename, "w+");
	  infile = fopen (eFilename, "r");
	  fputs ("Recursive: PGP\n\n", outfile);
	  fflush (outfile);
	  if (USE_SYSLOG)
	    syslog (LOG_DEBUG, "Decrypting PGP message");
	  run_cmd (PGP, "-d", NULL, infile, outfile, NULL);

	  /* PGP 2.6.3in can't decrypt GNUPG messages but returns 0, so we
	     check if dFilename has grown and try to decrypt eFilename with
	     GNUPG if ALLOW_GPG != 0.  */
	  stat (dFilename, &filestat);
	  if (filestat.st_size == 16 && ALLOW_GPG)
	    {
	      if (USE_SYSLOG)
		syslog (LOG_DEBUG, "Decrypting GPG message");
	      rewind (infile);
	      run_cmd (GPG, "-d", NULL, infile, outfile, NULL);
	    }
	  fclose (infile);
	  unlink (eFilename);

	  {
	    fpos_t pos;
	    int ek = 0;

	    /* Check for "Encrypt-Key:" header and "**" marks in dFilename.  */
	    fgetpos (outfile, &pos);
	    rewind (outfile);
	    while (fgets (line, sizeof (line), outfile) && !hdrmarks (line))
	      continue;
	    while (fgets (line, sizeof (line), outfile) && !blank (line))
	      {
		if (strileft (line, "encrypt-key:"))
		  ek = 1;
	      }
	    if (ek == 1)
	      {
		while (fgets (line, sizeof (line), outfile))
		  {
		    if (ekmarks (line))
		      {
			ek = 2;
			break;
		      }
		  }
	      }
	    fflush (outfile);
	    fsetpos (outfile, &pos);

	    /* Append lines after the first PGP packet to decrypted message.
	     * Skip lines before the first "**" line if ek == 1, to prevent
	     * traffic analysis by resending intercepted messages with text
	     * inserted between "-----END PGP MESSAGE-----" and "**" lines.
	     */
	    if (ek == 1)
	      {
		while (fgets (line, sizeof (line), p))
		  {
		    if (ekmarks (line))
		      {
			fputs ("\n", outfile);
			fputs (line, outfile);
			break;
		      }
		  }
	      }
	  }
	  while (fgets (line, sizeof (line), p))
	    fputs (line, outfile);
	  fclose (outfile);

	  fclose (p);
	  rename (dFilename, tempfilename);
	  recursing = 1;
	  break;

	case REMIX_MESSAGE:
	  if (USE_SYSLOG)
	    {
	      syslog (LOG_INFO, "Type 2 destination detected");
	    }
	  outfile = fopen (bFilename, "w");
	  fputs (send_to, outfile);
	  fputs ("\n\n", outfile);
	  fflush (outfile);
	  if ((infile = fopen (tempfilename, "r")))
	    {
	      remove_headers (infile, outfile, anon);
	      fclose (infile);
	    }
	  fclose (outfile);
	  unlink (tempfilename);
	  sprintf (poolfilename, "%s/mailXXXXXX", MIX_DIR);
	  outfile = fdopen (mkstemp (poolfilename), "w");
	  if (USE_SYSLOG)
	    syslog (LOG_INFO, "Reordering message to %s",
		    poolfilename + strlen (MIX_DIR) + 1);
	  fputs ("-----Intermediate Hop-----\n", outfile);
	  fputs (send_to, outfile);
	  fputs ("\nEND\n", outfile);
	  fflush (outfile);
	  if ((infile = fopen (bFilename, "r")))
	    {
	      if (USE_SYSLOG)
		syslog (LOG_DEBUG, "Remixing message to %s", send_to);
	      sprintf (line, "%d", findline (send_to, remixlist));
	      run_cmd (MIXMASTER, "-l", line, infile, outfile, NULL);
	      fclose (infile);
	    }
	  fclose (outfile);
	  unlink (bFilename);
	  break;

	case REPGP_MESSAGE:
	  if (USE_SYSLOG)
	    {
	      syslog (LOG_INFO, "Type 1 destination detected");
	    }
	  outfile = fopen (bFilename, "w");
	  if ((infile = fopen (tempfilename, "r")))
	    {
	      remove_headers (infile, outfile, anon);
	      fclose (infile);
	    }
	  fclose (outfile);
	  unlink (tempfilename);
	  if (USE_MIX)
	    {
	      sprintf (poolfilename, "%s/mailXXXXXX", MIX_DIR);
	      outfile = fdopen (mkstemp (poolfilename), "w");
	      if (USE_SYSLOG)
		syslog (LOG_INFO, "Reordering message to %s",
			poolfilename + strlen (MIX_DIR) + 1);
	      fputs ("-----Final Hop-----\n", outfile);
	      fputs (send_to, outfile);
	      fputs ("\nEND\n\n::\nEncrypted: PGP\n\n", outfile);
	      fflush (outfile);
	      if ((infile = fopen (bFilename, "r")))
		{
		  if (USE_SYSLOG)
		    syslog (LOG_DEBUG, "Encrypting message to %s", send_to);
		  run_cmd (PGP, "-e", send_to, infile, outfile, NULL);
		  fclose (infile);
		}
	      fclose (outfile);
	      unlink (bFilename);
	    }
	  else
	    {
	      if (USE_SYSLOG)
		syslog (LOG_INFO, "Remailing message");
	      sprintf (line, "%s", SENDMAIL);
	      p = popen (line, "w");
	      fprintf (p, "From: %s <%s>\n", ANON_NAME, ANON_ADDR);
	      fprintf (p, DISCLAIMER);
	      fputs ("To: ", p);
	      fputs (send_to, p);
	      fputs ("\n\n::\nEncrypted: PGP\n\n", p);
	      fflush (p);
	      if ((infile = fopen (bFilename, "r")))
		{
		  if (USE_SYSLOG)
		    syslog (LOG_DEBUG, "Encrypting message to %s", send_to);
		  run_cmd (PGP, "-e", send_to, infile, p, NULL);
		  fclose (infile);
		}
	      pclose (p);
	      unlink (bFilename);
	    }
	  break;

	case ANON_MESSAGE:
	  if (USE_SYSLOG)
	    syslog (LOG_INFO, "Anonymous message detected");
	  if (USE_MIX)
	    {
	      sprintf (poolfilename, "%s/mailXXXXXX", MIX_DIR);
	      outfile = fdopen (mkstemp (poolfilename), "w");
	      if (USE_SYSLOG)
		syslog (LOG_INFO, "Reordering message to %s",
			poolfilename + strlen (MIX_DIR) + 1);
	      fputs ("-----Final Hop-----\n", outfile);
	      fputs (send_to, outfile);
	      if (strlen (send_to) > 0 && send_to[strlen (send_to) - 1] != '\n')
		{
		  if (USE_SYSLOG)
		    syslog (LOG_WARNING, "Truncated destination buffer");
		  fputs ("\n", outfile);
		}
	      fputs ("END\n", outfile);
	      fflush (outfile);
	      if ((infile = fopen (tempfilename, "r")))
		{
		  remove_headers (infile, outfile, anon);
		  fclose (infile);
		}
	      fclose (outfile);
	    }
	  else
	    {
	      if (USE_SYSLOG)
		syslog (LOG_INFO, "Remailing message");
	      sprintf (line, "%s", SENDMAIL);
	      p = popen (line, "w");
	      fprintf (p, "From: %s <%s>\n", ANON_NAME, ANON_ADDR);
	      fprintf (p, DISCLAIMER);
	      fputs (send_to, p);
	      if (strlen (send_to) > 0 && send_to[strlen (send_to) - 1] != '\n')
		{
		  if (USE_SYSLOG)
		    syslog (LOG_WARNING, "Truncated destination buffer");
		  fputs ("\n", p);
		}
	      fflush (p);
	      if ((infile = fopen (tempfilename, "r")))
		{
		  remove_headers (infile, p, anon);
		  fclose (infile);
		}
	      pclose (p);
	    }
	  unlink (tempfilename);
	  break;

	case ANON_POST:
	  if (USE_SYSLOG)
	    syslog (LOG_INFO, "Anonymous post detected");
	  if (USE_MIX)
	    {
	      sprintf (poolfilename, "%s/mailXXXXXX", MIX_DIR);
	      outfile = fdopen (mkstemp (poolfilename), "w");
	      if (USE_SYSLOG)
		syslog (LOG_INFO, "Reordering message to %s",
			poolfilename + strlen (MIX_DIR) + 1);
	      fputs ("-----Final Hop-----\n", outfile);
	      fputs (send_to, outfile);
	      if (strlen (send_to) > 0 && send_to[strlen (send_to) - 1] != '\n')
		{
		  if (USE_SYSLOG)
		    syslog (LOG_WARNING, "Truncated destination buffer");
		  fputs ("\n", outfile);
		}
	      fputs ("END\n", outfile);
	      fflush (outfile);
	      if ((infile = fopen (tempfilename, "r")))
		{
		  remove_headers (infile, outfile, anon);
		  fclose (infile);
		}
	      fclose (outfile);
	    }
	  else
	    {
	      if (USE_SYSLOG)
		syslog (LOG_INFO, "Posting message");
	      if (ALLOW_POST == 1)
		sprintf (line, "%s", INEWS);
	      else if (ALLOW_POST == 2)
		sprintf (line, "%s", SENDMAIL);
	      p = popen (line, "w");
	      if (ALLOW_POST == 1)
		fprintf (p, "Organization: %s\n", ORGANIZATION);
	      else if (ALLOW_POST == 2)
		fprintf (p, "To: %s\n", MAIL2NEWS);
	      fprintf (p, "From: %s <%s>\n", ANON_NAME, ANON_ADDR);
	      fprintf (p, DISCLAIMER);
	      fputs (send_to, p);
	      if (strlen (send_to) > 0 && send_to[strlen (send_to) - 1] != '\n')
		{
		  if (USE_SYSLOG)
		    syslog (LOG_WARNING, "Truncated destination buffer");
		  fputs ("\n", p);
		}
	      fflush (p);
	      if ((infile = fopen (tempfilename, "r")))
		{
		  remove_headers (infile, p, anon);
		  fclose (infile);
		}
	      pclose (p);
	    }
	  unlink (tempfilename);
	  break;

	case STATS_REQ:
	  if (USE_SYSLOG)
	    syslog (LOG_INFO, "Statistics were requested");
	  get_from (tempfilename, message_from);
	  if (rxfind (message_from, DEST_BLOCK))
	    {
	      if (USE_SYSLOG)
		syslog (LOG_NOTICE, "Ignoring request");
	    }
	  else
	    {
	      if (USE_SYSLOG)
		syslog (LOG_INFO, "Mailing statistics to %s", message_from);
	      mailstats (message_from);
	    }
	  unlink (tempfilename);
	  break;

	case CONF_REQ:
	  if (USE_SYSLOG)
	    syslog (LOG_INFO, "Abilities were requested");
	  get_from (tempfilename, message_from);
	  if (rxfind (message_from, DEST_BLOCK))
	    {
	      if (USE_SYSLOG)
		syslog (LOG_NOTICE, "Ignoring request");
	    }
	  else
	    {
	      if (USE_SYSLOG)
		syslog (LOG_INFO, "Mailing abilities to %s", message_from);
	      mailconf (message_from);
	    }
	  unlink (tempfilename);
	  break;

	case HELP_REQ:
	  if (USE_SYSLOG)
	    syslog (LOG_INFO, "Help was requested");
	  get_from (tempfilename, message_from);
	  mailfile (HELP_FILE, message_from);
	  unlink (tempfilename);
	  break;

	case KEY_REQ:
	  if (USE_SYSLOG)
	    syslog (LOG_INFO, "Keys were requested");
	  get_from (tempfilename, message_from);
	  mailfile (KEY_FILE, message_from);
	  unlink (tempfilename);
	  break;

	case CUTMARKS:
	  if (USE_SYSLOG)
	    syslog (LOG_DEBUG, "Cutmarks: %s detected", send_to);
	  cut (tempfilename, cFilename, send_to, "cutmarks:");
	  trimmings = 1;
	  break;

	case URL_REQUEST:
	  if (USE_SYSLOG)
	    syslog (LOG_INFO, "URL was requested");
	  append_url (uFilename, tempfilename, send_to);
	  recursing = 2;
	  break;

	case ENCRYPT_SUB:
	  if (USE_SYSLOG)
	    syslog (LOG_DEBUG, "Encrypt-Subject: %s detected", send_to);
	  encrypt_sub (tempfilename, send_to);
	  recursing = 1;
	  break;

	case ENCRYPT_KEY:
	  if (USE_SYSLOG)
	    syslog (LOG_DEBUG, "Encrypt-Key: %s detected", send_to);
	  cut (tempfilename, kFilename, "**", "encrypt-key:");
	  if ((infile = fopen (kFilename, "r")))
	    {
	      outfile = fopen (tempfilename, "a");
	      if (USE_SYSLOG)
		syslog (LOG_DEBUG, "Encrypting message body");
	      run_cmd (PGP, "-c", send_to, infile, outfile, stderr);
	      fclose (outfile);
	      fclose (infile);
	      unlink (kFilename);
	      recursing = 1;
	    }
	  else
	    {
	      if (USE_SYSLOG)
		syslog (LOG_DEBUG, "Discarding message without '**' marks");
	      unlink (tempfilename);
	    }
	  break;

	case LATENT_TIME:
	  if (USE_SYSLOG)
	    syslog (LOG_DEBUG, "Latent-Time: %s detected", send_to);
	  queuelatent (tempfilename, send_to);
	  unlink (tempfilename);
	  break;

	case NON_ANON:
	  if (trimmings)
	    {
	      if (USE_SYSLOG)
		syslog (LOG_DEBUG, "Processing next part of message");
	      rename (cFilename, tempfilename);
	      trimmings = 0;
	      recursing = 1;
	    }
	  else
	    {
	      if (USE_SYSLOG)
		syslog (LOG_INFO, "Non-anonymous message detected");
	      if ((infile = fopen (tempfilename, "r")))
		{
		  fgets (line, sizeof (line), infile);
		  rewind (infile);
		  if (!strcmp (line, "Recursive: Cut\n"))
		    {
		      if (USE_SYSLOG)
			{
			  syslog (LOG_DEBUG, "It's cutmarks leftovers");
			  syslog (LOG_INFO, "Discarding undeliverable message");
			}
		    }
		  else if (!strcmp (line, "Recursive: PGP\n"))
		    {
		      if (USE_SYSLOG)
			{
			  syslog (LOG_DEBUG, "Couldn't decrypt message");
			  syslog (LOG_INFO, "Discarding undeliverable message");
			}
		    }
		  else
		    {
		      if (USE_SYSLOG)
			syslog (LOG_INFO, "Appending message to spool");
		      if ((outfile = fopen (MAIL_SPOOL, "a")))
			{
			  while (fgets (line, sizeof (line), infile))
			    fputs (line, outfile);
			  fclose (outfile);
			}
		    }
		  fclose (infile);
		}
	      unlink (tempfilename);
	    }
	  break;
	}
      if (trimmings)
	recursing = 1;
    }
  exit (0);
}
