/****************************************************************************
  This file is part of the Freedom Remailer.  It is:
  Copyright (C) 1995  John B. Fleming (jfleming@indiana.edu)
  Changes are (C) 1997  Johannes Kroeger (hanne@squirrel.owl.de)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/wait.h>
#include "config.h"

#ifdef REG_EXP_BLOCKING
#include <rxposix.h>
/* #include <rx.h> */ /* #include <rxposix.h> for GNU Rx library */

int rxmatch(const char *string, const char *pattern)
{
  regex_t compiled;

  if (regcomp(&compiled, pattern, REG_EXTENDED | REG_ICASE) != 0)
    return 0;
  return !regexec(&compiled, string, 0, NULL, 0);
}
#endif /* REG_EXP_BLOCKING */

void pgpdecrypt(const char *encrypted, char *decrypted)
{
  pid_t status;
  char pathenv[BUFSIZ], passenv[BUFSIZ];

  sprintf(passenv, "PGPPASS=%s", PASS_PHRASE);
  putenv(passenv);
  sprintf(pathenv, "PGPPATH=%s", PGP_DIR);
  putenv(pathenv);

  if (fork() == 0) {
    execlp(PGP, PGP, "+batchmode", "+verbose=0",
	   "-d", encrypted, "-o", decrypted, NULL);
  } else {
    wait(&status);
  }

  putenv("PGPPASS=");
  putenv("PGPPATH=");
}

void chop(char *string)
{
  char temp[BUFSIZ];
  unsigned int i, j = 0;

  strcpy(temp, string);
  for (i = 0; temp[i] != '\n'; i++) {
    if((j > 0) || !isspace(temp[i])) {
      string[j] = temp[i];
      j++;
    }
  }
  string[j] = '\0';
}

int blocked(const char *address, const char *bfile)
{
  FILE *file;
  char blockFile[BUFSIZ], input[BUFSIZ], temp[BUFSIZ];
  int block = 0;

  sprintf(blockFile, "%s/%s", REMAILER_DIR, bfile);
  file = fopen(blockFile, "r");
  if (file == NULL)
    return 0;

  strcpy(temp, address);
  stringlower(temp);
  while(fgets(input, sizeof(input), file)) {
    stringlower(input);
    chop(input);
    if (input[0] == '#' || strlen (input) == 0)
      continue; /* skip blank or comment lines */
#ifdef REG_EXP_BLOCKING
    if ((rxmatch(temp, input)) && (strlen(input) > 0))
      block = 1;
#else
    if ((strstr(temp, input) != NULL) && (strlen(input) > 0))
      block = 1;
#endif /* REG_EXP_BLOCKING */
  }
  fclose(file);
  return block;
}

void cut(char *origfile, char *cutfile, const char *cutmarks,
	 const char *header, const int fakeheader)
{
  char tempfile[BUFSIZ], input[BUFSIZ], tmp[BUFSIZ];
  FILE *in, *out;
  int cut = 0, headers = 1, linenum = 0;

  sprintf(tempfile, "%sC", origfile);
  rename(origfile, tempfile);

  in = fopen(tempfile, "r");
  if (in == NULL) {
    out = fopen(origfile, "w");
    fclose(out);
    return;
  }
  out = fopen(origfile, "w");

  while (fgets(input, sizeof(input), in)) {
    chop(input);
    if (!headers) {
      if ((linenum > 1) || (strlen(input) > 1))
	linenum++;
      if ((linenum == 1) && (strstr(input, "::") != NULL)) {
	headers = 1;
	fprintf(out, "%s\n", input);
      } else {
	if ((!cut) && (strcmp(input, cutmarks) == 0)) {
	  fclose(out);
	  out = fopen(cutfile, "w");
	  if (fakeheader)
	    fputs("Cut-Recursive: Yes\n\n", out);
	  cut = 1;
	} else if (linenum != 0) {
	  fprintf(out, "%s\n", input);
	}
      }
    } else {
      if (strlen(input) <= 1) {
	headers = 0;
	fputs("\n", out);
      } else {
	strcpy(tmp, input);
	stringlower(tmp);
	if (strncmp(tmp, header, strlen(header)) != 0)
	  fprintf(out, "%s\n", input);
      }
    }
  }
  fclose(in);
  fclose(out);
  unlink(tempfile);
}

void mailhelp(const char *to)
{
  FILE *p, *in;
  char sendmail[BUFSIZ], helpfile[BUFSIZ], input[BUFSIZ];

  sprintf(sendmail, "%s", SENDMAIL);
  sprintf(helpfile, "%s/%s", REMAILER_DIR, HELP_FILE);

  in = fopen(helpfile, "r");
  if (in != NULL) {
    p = (FILE *) popen(sendmail, "w");

    fprintf(p, "From: %s <%s>\nTo: %s\n", REMAILER_NAME, REMAILER_ADDR, to);

    while(fgets(input, sizeof(input), in)) {
      fputs(input, p);
    }
    fclose(in);
    pclose(p);
  }
}

void mailkey(const char *to)
{
  FILE *p, *in;
  char sendmail[BUFSIZ], keyfile[BUFSIZ], input[BUFSIZ];

  sprintf(sendmail, "%s", SENDMAIL);
  sprintf(keyfile, "%s/%s", REMAILER_DIR, PGP_KEY_FILE);

  in = fopen(keyfile, "r");
  if (in != NULL) {
    p = (FILE *) popen(sendmail, "w");

    fprintf(p, "From: %s <%s>\nTo: %s\n", REMAILER_NAME, REMAILER_ADDR,
	    to);

    while(fgets(input, sizeof(input), in)) {
      fputs(input, p);
    }
    fclose(in);
    pclose(p);
  }
}
