#!/usr/common/bin/perl -w

#  This is version 1.1 of Crowds
# 
#  The authors of this software are Mike Reiter and Avi Rubin
#               Copyright (c) 1997 by AT&T.
#  Permission to use, copy, and modify this software without fee is
#  hereby granted, provided that this entire notice is included in all
#  copies of any software which is or includes a copy or modification
#  of this software and in all copies of the supporting documentation
#  for such software.
# 
#  SOME PARTS OF CROWDS MAY BE RESTRICTED UNDER UNITED STATES EXPORT
#  REGULATIONS.
# 
#  THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR
#  IMPLIED WARRANTY.  IN PARTICULAR, NEITHER THE AUTHORS NOR AT&T MAKE
#  ANY REPRESENTATION OR WARRANTY OF ANY KIND CONCERNING THE
#  MERCHANTABILITY OF THIS SOFTWARE OR ITS FITNESS FOR ANY PARTICULAR
#  PURPOSE.

require 5.004;
use strict;
use Socket;
use Carp;
use StreamCipher;
use BlockCipher;
use Random;
use TimeStamp;
use URLParse;
use Sys::Hostname;
use Msg ('EXTERNAL', 'INTERNAL');

#--- My INET address
use vars qw($MY_INETADDR);
*MY_INETADDR = \inet_ntoa(scalar(gethostbyname(hostname())));

#--- Image handling
use vars qw($NO_WAIT); *NO_WAIT = \1;
my %Images;  # Image cache
my %Waiting; # Either undefined, $NO_WAIT (the complete image is in %Images),
             # or contains a list of connnections waiting for the image
use vars qw($SEPARATOR);  # separates returned images
*SEPARATOR = \'RRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRR';
use vars qw($MAX_CONNS);
*MAX_CONNS = \4;

#--- Commit handling
use vars qw($C_NORMAL $C_COMMITTED $C_NEWJOINER);
*C_NORMAL    = \1;  # steady state
*C_COMMITTED = \2;  # a new commit message has been received
*C_NEWJOINER = \3;  # we have just joined and not been committed yet
my $new_commit = $C_NORMAL; # shows our current status
my $commit_time;

#--- Environment status values
use vars qw($S_NULL $S_DROP $S_NEW $S_FORWARD $S_SUBMIT $S_TO_SUBMIT
            $S_HANDLE $S_UNREACHABLE $S_INVALID $S_BAD_PORT $S_CONN_OK);
*S_NULL        = \0;
*S_DROP        = \1;
*S_NEW         = \2;
*S_FORWARD     = \3;
*S_SUBMIT      = \4;
*S_TO_SUBMIT   = \5;
*S_HANDLE      = \6;
*S_UNREACHABLE = \7;
*S_INVALID     = \8;
*S_BAD_PORT    = \9;
*S_CONN_OK     = \10;

#--- Path management
use vars qw($N_SUBMIT); *N_SUBMIT = \'';
my %Next;       # either $N_SUBMIT or address of next path member
my %Translate;  # maps incoming path ids to outgoing path ids

#--- Other global vars
my (%Account, %Lives, %Cipher);
my @Members = ();  #array of current members
my $my_cipher;     #my block cipher (using my password as key)
my $rng;           #my random number generator
my %Params;        #the params specified in the config file

#--- Load information from the config file
load_params();

#--- Join the group
join_group();

#--- Fix my_path_id
my $my_path_id = join("", unpack("H*", $rng->get_rand_string(16)));

#--- Set up my sockets
Msg->new_server(hostname(),
		$Params{jport}+1, \&new_internal_connection, INTERNAL());
Msg->new_server(hostname(),
		$Params{jport}, \&new_browser_connection, EXTERNAL());

#--- Start the event loop
Msg->event_loop(10, \&precomp_rands);

exit();


sub new_internal_connection {
    my ($conn, $client_host, $client_port) = @_;
    my $env = { 'peer_addr' => $client_host,
		'peer_port' => $client_port,
		'client'    => $conn,
	        'status'    => $S_NEW,
	        'browser'   => 0 };

    return (\&handle_internal_connection, $env);
}

sub new_browser_connection {
    my ($conn, $client_host, $client_port) = @_;
    my $env = { 'peer_addr' => $client_host,
		'peer_port' => $client_port,
		'client'    => $conn,
	        'status'    => $S_NEW,
	        'browser'   => 1 };

    if ($new_commit == $C_NEWJOINER) {
	display_new_joiner_message($conn);
	$env->{status} = $S_DROP;
    }

    return (\&handle_browser_connection, $env);
}

sub handle_browser_connection {
    my ($conn, $env, $msg, $err) = @_;

    return if (length($msg) == 0); # Return if browser closed the connection

    if ($env->{status} == $S_NEW) {
	print STDERR "Browser: $msg" if ($Params{verbose});

	my ($url) = ($msg =~ m/(?:GET|POST|HEAD) (\S+)/);
	my ($protocol) = ($url =~ /(\w*):/);
	if ($protocol ne "http") {
	    bad_protocol($conn, $protocol);
	    $env->{status} = $S_DROP;
	    return;
	}    
	parse_url($url, $env);

	if (defined $env->{path} && $env->{path} =~ m|\?crowd\?(\s*)$|) {
	    print STDERR "***   Crowd query   ***\n" if ($Params{verbose});
	    crowd_query($conn, $env->{peer_addr});
	    $env->{status} = $S_DROP;
	    return;
	}
	else {  # Not a crowd query
	    if (in_cache($msg, $conn)) {
		$env->{status} = $S_DROP; # Drop the request; already have it
		return;
	    } elsif ($new_commit == $C_COMMITTED) {
		undef %Next;
		undef %Translate;
		$new_commit = $C_NORMAL;
		display_joining_message($conn);
		$env->{status} = $S_DROP;
		return;
	    } else {
		# New request from user; clean image cache
		undef %Images;
		undef %Waiting;
	    }
	
	    my $direct = ($Params{firewall} ?
			  scalar(@Members) == 0 : scalar(@Members) == 1);
	    unless ($direct || (defined $Next{$my_path_id} &&
				$Next{$my_path_id} eq $N_SUBMIT)) {
		my $req_key = $rng->get_rand_string(128);
		my $resp_key = $rng->get_rand_string(128);
		setup_forward($env, $req_key, $resp_key, $my_path_id);
		handle_proxy_request($env, $req_key, $resp_key, $my_path_id)
		    if ($env->{status} == $S_HANDLE);
	    }

	    if ($env->{status} == $S_FORWARD) {
	        $msg = ($env->{req_stream})->encrypt($msg);
	    } else {  # setup_forward failed
		$Next{$my_path_id} = $N_SUBMIT;
		($url, $env->{headers}) =
		    ($msg =~ m/(?:GET|POST|HEAD) (\S+) (\S+\s*)/);
		setup_submit($env, $url);
		$env->{in_headers} = 1;
		$msg =~ s|http://[^/]*||;
	    }

	    if ($env->{status} != $S_DROP) {
		my $server = $env->{server};
		$server->send_later($msg);
	    }
	}
    } elsif ($env->{status} != $S_DROP) {  # Not the first msg from the browser
	return if ($msg =~ /Proxy-Connection:/);
	return if ($msg =~ /From:/);
	return if ($msg =~ /Referer:/);
	return if (($msg =~ /Cookie:/) && !$Params{cookies});
	return if ($msg =~ /User-Agent:/);
	return if ($msg =~ /Pragma:/);

	if ($env->{status} == $S_FORWARD) {
	    $msg = ($env->{req_stream})->encrypt($msg);
	} elsif ($env->{in_headers}) {
	    $env->{headers} .= $msg;
	    $env->{in_headers} = 0 if ($msg =~ /^\s*$/);
	}
	my $server = $env->{server};
	$server->send_later($msg);
    }
}

sub handle_internal_connection {
    my ($conn, $env, $msg, $err) = @_;

    if (length($msg) == 0 && defined $env->{server}) {
	# Predecessor closed the connection
	$env->{server}->disconnect();
	return;
    }

    if ($env->{status} == $S_NEW) {
	my ($source, $contents) = pop_header($msg);
	return if ($source eq '' || $contents eq '');

	if ($source eq 'blender') {
	    my ($timestamp,$joiner,$account,$tkey,$firewall) =
		split(/,/, $my_cipher->cbc_decrypt($contents));
	    $firewall = 0 unless defined $firewall; #backwards compatibility

	    if (TimeStamp::check_time($timestamp) eq 0) {
		print STDERR "Bad timestamp from blender.\n";
		$conn->disconnect();
		return;
	    }

	    if ($joiner eq "COMMIT") { #got a commit message from blender
		print STDERR "Commit from blender\n" if ($Params{verbose});

		if ($new_commit == $C_NORMAL) {
		    $new_commit = $C_COMMITTED;
		} elsif ($new_commit == $C_NEWJOINER) {
		    $new_commit = $C_NORMAL;
		}

	    } else {
		$tkey = pack("H16", $tkey);
		$Cipher{$joiner} = BlockCipher::new($tkey);
		$Lives{$joiner} = 2;
		$Account{$joiner} = $account;

		unless ($firewall || grep(/$joiner/, @Members) > 0) {
		    push(@Members, $joiner);
		}

		my $member;
		print STDERR "Members:\n";
		foreach $member (@Members) {
		    print STDERR "    $member ($Account{$member})\n";
		}
	    }

	    $conn->disconnect();
	}
	elsif ($source eq 'proxy') {
	    print STDERR "Got a proxy request\n" if ($Params{verbose});

	    chop($contents);
	    $contents =~ /^(.+?):(.+?):(.+?):(.+)/;

	    my $client_id = join(':', $env->{peer_addr}, $1);
	    if ($Cipher{$client_id}) {
		#--- TODO: check $contents for null message?
		my $req_key = $Cipher{$client_id}->cbc_decrypt($2);
		my $resp_key = $Cipher{$client_id}->cbc_decrypt($3);
		$env->{status} = $S_HANDLE;
		handle_proxy_request($env, $req_key, $resp_key, $4);
	    }
	    else {
		$conn->disconnect();
	    }
	}
    } else { # Not the first message

	if ($env->{status} == $S_SUBMIT || $env->{status} == $S_TO_SUBMIT) {
	    $msg = ($env->{req_stream})->decrypt($msg);

	    if ($env->{status} == $S_TO_SUBMIT) {
		my $url;
		($url, $env->{headers}) =
		    ($msg =~ m/(?:GET|POST|HEAD) (\S+) (\S+\s*)/);
		setup_submit($env, $url);
		$env->{in_headers} = 1;
		$msg =~ s|http://[^/]*||;
	    } elsif ($env->{in_headers}) {
		$env->{headers} .= $msg;
		$env->{in_headers} = 0 if ($msg =~ /^\s*$/);
	    }
	}

	if ($env->{status} != $S_DROP) {
	    my $server = $env->{server};
	    $server->send_later($msg);
	}
    }
}


#--- HANDLE_PROXY_REQUEST
#--- This routine decides whether to submit the request to the web
#--- server or to forward it to another proxy.

sub handle_proxy_request {
    my ($env, $req_key, $resp_key, $path_id) = @_;

    unless (defined $Next{$path_id} && $Next{$path_id} eq $N_SUBMIT) {
	while ($env->{status} == $S_HANDLE) {
	    if (!defined $Next{$path_id}) {
		#--- Flip a coin to determine submit vs. forward
		my $coin = $rng->get_rand_int(4);
		print STDERR "Coin flip: $coin\n" if ($Params{verbose});
		last unless ($coin > 0 && scalar(@Members) > 1);
	    }
	    setup_forward($env, $req_key, $resp_key, $path_id);
	}
    }

    if ($env->{status} != $S_FORWARD) {
	$Next{$path_id} = $N_SUBMIT;
	$env->{status} = $S_TO_SUBMIT;
	$env->{req_stream} = StreamCipher::new($req_key);
	$env->{resp_stream} = StreamCipher::new($resp_key);
    }
}


#--- JOIN_GROUP
#--- This contacts the blender and receives the addresses of the members
#--- currently in the group.

sub join_group {

    #--- Set up for authenticating to the group
    if (!$Params{password}) {
        system "stty -echo";
        print "Enter passphrase for account \"$Params{name}\": ";
        chop($Params{password} = <STDIN>);
        print "\n";
        system "stty echo";
    }
    $my_cipher = BlockCipher::new($Params{password});
    my $enc_time = $my_cipher->cbc_encrypt(time());

    #--- Initialize the generator
    my $seed = $Params{password}.`netstat -n`.`ps`;
    $rng = Random::new($seed);
    $rng->generate(1000);

    #--- Connect to the blender
    my $conn = Msg->connect($Params{blender}, $Params{bport},
			    \&handle_blender_response, undef, INTERNAL());
    if (!defined($conn)) {
	die "Cannot connect to blender";
    }

    my $msg = "$enc_time:$Params{name}:$Params{jport}:$Params{firewall}";
    $conn->send_later($msg);
}


sub handle_blender_response {
    my ($conn, $env, $msg, $err) = @_;
    my @Triples;
    my ($tmp, $timestamp, $contents);

    if (length($msg) == 0) {
	logmsg("Error contacting the blender: $err");
	return;
    }

    if ($msg eq '1234567890') {
	logmsg("Blender says bad timestamp.\nContact your sysadmin.");
	return;
    }

    $msg = $my_cipher->cbc_decrypt($msg);
    ($timestamp,$commit_time,$contents) = split(/=/, $msg); 

    if (TimeStamp::check_time($timestamp) eq 0) {
	logmsg("Bad timestamp from blender.\n");
	exit;
    }

    @Triples = split(/;/, $contents);
    foreach $tmp (@Triples) {
        my ($member,$account,$key) = split(/,/, $tmp);

	push @Members, $member;
        $Lives{$member} = 2;
	$Account{$member} = $account;
	my $tkey = pack("H16", $key);
	$Cipher{$member} = BlockCipher::new($tkey);
    }

    print STDERR "Members:\n";
    foreach $tmp (@Members) {
	print STDERR "    $tmp ($Account{$tmp})\n";
    }
 
    $new_commit = $C_NEWJOINER if ($Params{commit});

    $conn->disconnect();
}


#--- SETUP_FORWARD
sub setup_forward {
    my ($env, $req_key, $resp_key, $path_id) = @_;
    my $conn;

    $Translate{$path_id} = join("",unpack("H*",$rng->get_rand_string(16)))
	if (!defined $Translate{$path_id});

    if (!defined $Next{$path_id}) {
	$Next{$path_id} = $Members[$rng->get_rand_int(scalar(@Members))];
	if ($Next{$path_id} eq "$MY_INETADDR:$Params{jport}") {
	    # chose to forward to myself; don't do it but pretend I did
	    delete $Next{$path_id};
	    $env->{status} = $S_HANDLE;
	    return;
	}
    }

    my ($inetaddr, $port) = split(/:/, $Next{$path_id});
    until ($conn = Msg->connect($inetaddr, $port+1,
				\&handle_proxy_reply, $env, INTERNAL())) {
        member_down($inetaddr,$port);
        if ($Params{firewall} ?
	    scalar(@Members) == 0 : scalar(@Members) == 1) {
	    $env->{status} = $S_NULL;  #forward failed
	    return;
	}
	$Next{$path_id} = $Members[$rng->get_rand_int(scalar(@Members))];
	if ($Next{$path_id} eq "$MY_INETADDR:$Params{jport}") {
	    # chose to forward to myself; don't do it but pretend I did
	    delete $Next{$path_id};
	    $env->{status} = $S_HANDLE;
	    return;
	}
        ($inetaddr,$port) = split(/:/, $Next{$path_id});
    }

    print STDERR "Connection to $inetaddr:$port succeeded\n"
	if ($Params{verbose});

    my $key1 = ($Cipher{$Next{$path_id}})->cbc_encrypt($req_key);
    my $key2 = ($Cipher{$Next{$path_id}})->cbc_encrypt($resp_key);
    my $msg = "$Params{jport}:$key1:$key2:$Translate{$path_id}\n";
    $msg = push_header($msg);
    $conn->send_later($msg);

    $env->{server} = $conn;
    if ($env->{browser}) {
	$env->{req_stream} = StreamCipher::new($req_key);
	$env->{resp_stream} = StreamCipher::new($resp_key);
	$env->{parser} = URLParse::new();
    }

    $env->{status} = $S_FORWARD;
}


sub handle_proxy_reply {
    my ($conn, $env, $msg, $err) = @_;
    my (@urls, $url);

    if ($env->{browser}) {
	if (length($msg) == 0) {
	    # Something's wrong; probably the next hop failed
	    $conn->disconnect();
	    if (defined $env->{url} && defined $Waiting{$env->{url}} &&
		$Waiting{$env->{url}} != $NO_WAIT) {
		my $client;
		foreach $client (@{$Waiting{$env->{url}}}) {
		    $client->disconnect();
		}
		delete $Images{$env->{url}};
		$Waiting{$env->{url}} = $NO_WAIT;
	    }
	    delete $env->{url};

	} elsif ($msg =~ s/^$SEPARATOR //) {
	    if (defined $env->{url} && defined $Waiting{$env->{url}} &&
		$Waiting{$env->{url}} != $NO_WAIT) {
		my $client;
		foreach $client (@{$Waiting{$env->{url}}}) {
		    $client->send_now($Images{$env->{url}});
		    $client->disconnect();
		}
		$Waiting{$env->{url}} = $NO_WAIT;
	    }
	    delete $env->{url};
	    delete $env->{parser};

	    if (defined $env->{client}) {
		# Disconnect from the browser
		($env->{client})->disconnect();
		delete $env->{client};
	    }

	    if (($url) = ($msg =~ m/(\S+)/)) {
		$url = pack("H*", $url);
		$url = ($env->{resp_stream})->decrypt($url);
		$env->{url} = $url;
		parse_url($url, $env);
		$Images{$url} = '';
		print STDERR "Now receiving $env->{url}\n"
		    if ($Params{verbose});
	    } else {
		$conn->disconnect();  #DONE: disconnect from path successor
	    }

	} else {  # msg is not empty and is not the SEPARATOR

	    $msg = ($env->{resp_stream})->decrypt($msg);

	    if (exists $env->{client}) {  # Sending directly to the browser
		if (exists $env->{parser}) {
		    @urls = ($env->{parser})->find_urls($env->{host},
							$env->{port},
							$env->{path}, $msg);
		    if (@urls > 0) {
			foreach $url (@urls) {
                            my $url2find = quotemeta($url);
			    if (grep(m|^$url2find$|, @{$env->{waiting}}) == 0) {
				print STDERR "FOUND $url\n"
				    if ($Params{verbose});
				push(@{$env->{waiting}}, $url);
				$Waiting{$url} = [ ]
				    if (!defined $Waiting{$url});
			    }
			}
		    }
		}

		$msg = add_crowd($msg);
		my $client = $env->{client};
		$client->send_now($msg);
	    } else {
		$Images{$env->{url}} .= $msg;
	    }
	}

    } else { # We are not preceded by the browser
	$msg = "$SEPARATOR " if (length($msg) == 0);
        $conn->disconnect() if ($msg =~ m/^$SEPARATOR $/);
	my $client = $env->{client};
	$client->send_later($msg);
    }
}


sub setup_submit {
    my ($env, $url) = @_;

    print STDERR "setup_submit: $url\n" if ($Params{verbose});

    $env->{url} = $url;
    $env->{requested} = [ ];
    $env->{to_request} = [ ];
    $env->{connections} = 1;
    $env->{conns} = { };

    connect_to_server($url, $env); 
    if ($env->{status} == $S_CONN_OK) {
	$env->{parser} = URLParse::new();
	$env->{status} = $S_SUBMIT;
    } elsif ($env->{status} == $S_INVALID) {
	invalid_url($env);
	$env->{status} = $S_DROP;
    } elsif ($env->{status} == $S_BAD_PORT) {
	bad_port($env);
	$env->{status} = $S_DROP;
    } elsif ($env->{status} == $S_UNREACHABLE) {
	unreachable_host($env);
	$env->{status} = $S_DROP;
    }
}


#--- handle_server_reply
#--- Handles replies from web servers
sub handle_server_reply {
    my ($conn, $env, $msg, $err) = @_;
    my (@urls, $url);

    if (length($msg) == 0) {  # Server closed the connection
	$env->{connections}--;
	$url = $env->{conns}->{$conn}->{url};
	print STDERR "Done fetching $url\n" if ($Params{verbose});

	if ($env->{browser}) {
	    if (defined $env->{client}) {
		($env->{client})->disconnect();
		delete $env->{client};
	    } else {
		my $client;
		if (defined $Waiting{$url} && $Waiting{$url} != $NO_WAIT) {
		    foreach $client (@{$Waiting{$url}}) {
			$client->send_now($Images{$url});
			$client->disconnect();
		    }
		    $Waiting{$url} = $NO_WAIT;
		}
	    }
	} else {
	    my $client = $env->{client};
	    unless ($url eq $env->{url}) {
		my $curl = ($env->{resp_stream})->encrypt($url);
		$curl = join('', unpack("H*", $curl));
		$client->send_later("$SEPARATOR $curl");
	    }
	    $msg = $env->{conns}->{$conn}->{contents};
	    $msg = ($env->{resp_stream})->encrypt($msg);
	    $client->send_later($msg);
	    delete $env->{conns}->{$conn};
	    if ($env->{connections} == 0 &&
		scalar(@{$env->{to_request}}) == 0) {
		$client->send_later("$SEPARATOR ");
	    }
	}

	while ($env->{connections} < $MAX_CONNS &&
	       scalar(@{$env->{to_request}}) > 0) {
	    $url = shift @{$env->{to_request}};
	    connect_to_server($url, $env);
	    if ($env->{status} == $S_CONN_OK) {
		$env->{connections}++;
		my $server = $env->{server};
		my $request = "GET $url $env->{headers}";
		$request =~ s|http://[^/]*||;
		$server->send_later($request);
	    }
	}

	delete $env->{parser};

    } else {
	if (defined $env->{parser}) {
	    my @urls = ($env->{parser})->find_urls($env->{host}, $env->{port},
						   $env->{path}, $msg);
	    if (@urls > 0) {
		foreach $url (@urls) {
                    my $url2find = quotemeta($url);
		    if (grep(m|^$url2find$|, @{$env->{requested}}) == 0 &&
			grep(m|^$url2find$|, @{$env->{to_request}}) == 0) {
			print STDERR "FOUND $url\n" if ($Params{verbose});
			push @{$env->{to_request}}, $url;
			if ($env->{browser} && !defined($Waiting{$url})) {
			    $Waiting{$url} = [ ];
			}
		    }
		}
	    }
	}

	if ($env->{browser}) {
	    if (exists $env->{client}) {  # Sending directly to the browser
		$msg = add_crowd($msg);
		my $client = $env->{client};
		$client->send_now($msg);  # send_now to beat the disconnect
	    } else {
		$Images{$env->{conns}->{$conn}->{url}} .= $msg;
	    }
	} else {
	    $env->{conns}->{$conn}->{contents} .= $msg;
	}
    }
}


sub connect_to_server {
    my ($url, $env, $msg) = @_;

    parse_url($url, $env);
    unless ($env->{host}) {
	$env->{status} = $S_INVALID;
	return;
    }

    if ($env->{port} && $env->{port} <= 1024 && $env->{port} != 80) {
	$env->{status} = $S_BAD_PORT;
	return;
    }

    push @{$env->{requested}}, $url;
    my $conn = Msg->connect($env->{host}, $env->{port},
			    \&handle_server_reply, $env, EXTERNAL());
    if (!defined($conn)) {
	$env->{status} = $S_UNREACHABLE;
	return;
    }

    $env->{conns}->{$conn} = { url      => $url,
			       contents => '' };

    $env->{status} = $S_CONN_OK;
    $env->{server} = $conn;
}


#--- parse_url
#--- Parses the given URL and fills out the host, port, and path
#--- components of the given env appropriately.
sub parse_url {
    my ($url, $env) = @_;
    my ($remote_host, $remote_port, $remote_path) =
	($url =~ m!http://([^/:]+)(?::(\d*))?(/\S+)?!);

    #--- Setup connection to target host
    $remote_port = "http" unless ($remote_port);
    if ($remote_port !~ /^\d+$/) {
	$remote_port = (getservbyname($remote_port, "tcp"))[2];
	$remote_port = 80 unless ($remote_port);
    }

    #--- Setup environment for callback
    $env->{host} = $remote_host;
    $env->{port} = $remote_port;
    $env->{path} = $remote_path;    
}


#--- in_cache
#--- Determines if the URL contained in the input message has been received
#--- or will be soon.

sub in_cache {
    my ($msg, $conn) = @_;

    my ($method, $host, $port, $path) = 
	($msg =~ m!(GET|POST|HEAD) http://([^/:]+)(?::(\d*))?(/\S+)?!);

    #TODO: do we need to handle POST?
    if ($method ne "GET") {
	return 0;
    }

    my $url = "http://$host" . 
	      (defined($port) ? ":$port" : "") .
	      (defined($path) ? $path : "");
	
    if (defined $Waiting{$url}) {
	if ($Waiting{$url} == $NO_WAIT) {
	    if (defined $Images{$url}) {
		$conn->send_now($Images{$url});
	    } else {
		$conn->disconnect();
	    }
	} else {
	    push @{$Waiting{$url}}, $conn;
	}
	return 1;
    } else {
	return 0;
    }
}


#--- push_header and pop_header
#--- Routines for adding and removing message headers

sub push_header {
    my ($msg) = @_;
    return 'crowds/1.1:proxy:'.$msg;
}

sub pop_header {
    my ($msg) = @_;
    my ($version, $source);

    if ($msg =~ s|^crowds/||) {
	$msg =~ s|(.+?):||;
	if ($1 eq '1.1') {
	    $msg =~ s|(.+?):||;
	    return ($1, $msg);
	}
	else {
	    print STDERR "Got a V$1 message ... dropping\n";
	    return ('', '');
	}
    }
    else {
	print STDERR "Got an unintelligible message\n";
	return ('', '');
    }
}


#--- PRECOMP_RANDS
#--- Precomputes random numbers for use later
sub precomp_rands {
    $rng->generate(64) if ($rng->size() < 2000);
}


sub bad_protocol{
    my ($conn, $protocol) = @_;

    my $msg = "";
    $msg .= "<HTML>\n";
    $msg .= "<TITLE> Protocol not supported</TITLE>\n";
    $msg .= "<body  bgcolor=\"#FFFFFF\">\n";
    $msg .= "<h1> Crowd Error Message </h1>\n";
    $msg .= "<b> The $protocol protocol is not supported\n";
    $msg .= "by this version of Crowds.</b>";
    $msg .= "</body> </HTML>\n";

    $msg = "Content-length: ".length($msg)."\n\n".$msg;
    $msg = "Content-type: text/html\n".$msg;
    $msg = "HTTP/1.0 200 Document follows\n".$msg;

    $conn->send_later($msg);
}


sub crowd_query {
    my ($conn, $client_addr) = @_;
    my $browser_name = gethostbyaddr(inet_aton($client_addr),AF_INET);

    my $msg = "";
    $msg .= "<HTML>\n";
    $msg .= "<TITLE> Crowd membership page </TITLE>\n";
    $msg .= "<body  bgcolor=\"#FFFFFF\">\n";
    $msg .= " <h1> Welcome to the crowd! </h1>\n";
    $msg .= "<b> Your browser is running on host </b>";
    $msg .= "$browser_name ($client_addr). <P>\n";
    $msg .= "Here are the <i> jondos </i> available: <P>\n";
    $msg .= display_members();
    $msg .= "<P>To begin browsing anonymously, just open a URL.\n";
    $msg .= "<P>If you would rather not, then simply change your\n";
    $msg .= "proxy settings in your browser and try back later.\n";
    $msg .= "<P> <P>\n";
    $msg .= "<A HREF=\"http://www.research.att.com/\">";
    $msg .= "Go to AT&T Research Home Page</A>\n";
    $msg .= "</body></HTML>\n\n";

    $msg = "Content-length: ".length($msg)."\n\n".$msg;
    $msg = "Content-type: text/html\n".$msg;
    $msg = "HTTP/1.0 200 Document follows\n".$msg;

    $conn->send_now($msg);
}


sub display_joining_message{
    my ($client) = @_;

    my $msg = "";
    $msg .= "<HTML>\n";
    $msg .= "<TITLE> Crowd joining in progress </TITLE>\n";
    $msg .= "<body bgcolor=\"#FFFFFF\">\n";
    $msg .= "<h1>New members have joined the crowd</h1>\n";
    $msg .= "The crowd consists of the following jondos:<BR><BR>\n";
    $msg .= display_members();
    $msg .= "<P>To maximally protect your anonymity from other\n";
    $msg .= "crowd members, we recommend that you browse content\n";
    $msg .= "unrelated to what you were previously browsing.";
    $msg .= "</body> </HTML>\n";

    $msg = "Content-length: ".length($msg)."\n\n".$msg;
    $msg = "Content-type: text/html\n".$msg;
    $msg = "HTTP/1.0 200 Document follows\n".$msg;

    $client->send_now($msg);
}


sub display_new_joiner_message {
    my ($client) = @_;

    my $msg = "";
    $msg .= "<HTML>\n";
    $msg .= "<TITLE> New joiner page</TITLE>\n";
    $msg .= "<body  bgcolor=\"#FFFFFF\">\n";
    $msg .= "<h1> Welcome new crowd member!</h1>\n";
    $msg .= "Welcome to the crowd. You should be able to\n";
    $msg .= "browse using the crowd just after $commit_time.\n";
    $msg .= "Until then, to browse (non-anonymously) just\n";
    $msg .= "change the HTTP proxy setting in your browser.\n";
    $msg .= "<P>\n";
    $msg .= "The following jondos are in the crowd:<BR><BR>\n";
    $msg .= display_members();
    $msg .= "</body> </HTML>\n\n";

    $msg = "Content-length: ".length($msg)."\n\n".$msg;
    $msg = "Content-type: text/html\n".$msg;
    $msg = "HTTP/1.0 200 Document follows\n".$msg;

    $client->send_now($msg);
}


sub display_members {
    my ($a_member,$host,$port,$name,$iaddr);
    my $msg = "";

    if ($Params{firewall}) {
	$iaddr = inet_aton($MY_INETADDR) or die "no host: $host";
	$name = gethostbyaddr($iaddr,AF_INET);
	$msg .= "<b>Host</b> $name ($MY_INETADDR)";
	$msg .= " <b>on port</b> $Params{jport}";
	$msg .= " <--- Browser proxy<br>\n";
    }

    foreach $a_member (@Members) {
	($host,$port) = split /:/,$a_member;
	$iaddr = inet_aton($host) or die "no host: $host";
	$name = gethostbyaddr($iaddr,AF_INET);
	$msg .= "<b>Host</b> $name ($host) <b>on port</b> $port";
	$msg .= " ($Account{$a_member})";
	$msg .= " <--- Browser proxy" if ($MY_INETADDR eq $host &&
					  $Params{jport} eq $port);
	$msg .= "<br>\n";
    }

    return $msg;
}


sub invalid_url {
    my ($env) = @_;

    my $msg = "";
    $msg .= "<html>\n";
    $msg .= "<title>Invalid URL</title>\n";
    $msg .= "<body bgcolor=#FFFFFF>\n";
    $msg .= "<h1>Invalid URL</h1>";
    $msg .= "Error: <b>$env->{url}</b> not a valid URL.\n";
    $msg .= "<p>\n";
    $msg .= "If you believe this message to be incorrect or would like\n";
    $msg .= "assistance, please send email to\n";
    $msg .= "<A HREF=mailto:crowds-talk\@research.att.com>";
    $msg .= "crowds-talk\@research.att.com</A>.\n";
    $msg .= "</body> </html>\n";

    $msg = "Content-length: ".length($msg)."\n\n".$msg;
    $msg = "Content-type: text/html\n".$msg;
    $msg = "HTTP/1.0 200 Document follows\n".$msg;

    $msg = ($env->{resp_stream})->encrypt($msg) unless $env->{browser};

    my $client = $env->{client};
    $client->send_later($msg);
}


sub bad_port {
    my ($env) = @_;

    my $msg = "";
    $msg .= "<html>\n";
    $msg .= "<title>Disallowed Port</title>\n";
    $msg .= "<body bgcolor=#FFFFFF>\n";
    $msg .= "<h1>Disallowed Port</h1>";
    $msg .= "Error: port <b>$env->{port}</b> is not allowed.\n";
    $msg .= "<p>\n";
    $msg .= "If you believe this message to be incorrect or would like\n";
    $msg .= "assistance, please send email to\n";
    $msg .= "<A HREF=mailto:crowds-talk\@research.att.com>";
    $msg .= "crowds-talk\@research.att.com</A>.\n";
    $msg .= "</body> </html>\n";

    $msg = "Content-length: ".length($msg)."\n\n".$msg;
    $msg = "Content-type: text/html\n".$msg;
    $msg = "HTTP/1.0 200 Document follows\n".$msg;

    $msg = ($env->{resp_stream})->encrypt($msg) unless $env->{browser};

    my $client = $env->{client};
    $client->send_later($msg);
}


sub unreachable_host {
    my ($env) = @_;

    my $msg = "";
    $msg .= "<html>\n";
    $msg .= "<title>Unreachable host</title>\n";
    $msg .= "<body bgcolor=\"#FFFFFF\">\n";
    $msg .= "<h1>Unreachable host</h1>";
    $msg .= "Error: <b>$env->{host}</b> is presently unreachable.\n";
    $msg .= "Please try again later.\n";
    $msg .= "<p>\n";
    $msg .= "If you believe this message to be incorrect or would like\n";
    $msg .= "assistance, please send email to\n";
    $msg .= "<A HREF=mailto:crowds-talk\@research.att.com>";
    $msg .= "crowds-talk\@research.att.com</A>.\n";
    $msg .= "</body> </html>\n";

    $msg = "Content-length: ".length($msg)."\n\n".$msg;
    $msg = "Content-type: text/html\n".$msg;
    $msg = "HTTP/1.0 200 Document follows\n".$msg;

    $msg = ($env->{resp_stream})->encrypt($msg) unless $env->{browser};

    my $client = $env->{client};
    $client->send_later($msg);
}


sub member_down {
    my ($addr, $port) = @_;
    my $count;
    my $a_member;

    if ($Params{verbose}) {
	print STDERR "Member $addr:$port is down\n";
	print STDERR "All members before: @Members\n";
	print STDERR "All Lives before: " . join(" ", values %Lives) . "\n";
    }

    #--- Iterate on the members of the list until the one in addr:port
    #--- is found. Then use that index to decrement lives or remove
    #--- that member, @members[i]

    for ($count = 0; $count < scalar(@Members); $count += 1) {

        #--- find the member who is gone and set the count index

        if ($Members[$count] eq join ':',$addr,$port) {
            if ($Lives{$Members[$count]} == 0) {
	        #--- Copy the first element in the list to this position
	        #--- and shift off the first value for members and lives.
	        #--- This has the effect of removing the member from the list.
                $Members[$count] = $Members[0];
	        shift @Members;	
	    }
            else {
	        $Lives{$Members[$count]} -= 1;
	    }
	    last;
        }
    }

    if ($Params{verbose}) {
	print STDERR "All members after: @Members\n";
	print STDERR "All Lives after: " . join(" ", values %Lives) . "\n";
    }
}


#--- ADD_CROWD 
#--- This subroutine finds the <title> and adds the word 'Crowd:'
#--- to the beginning of it.  TODO: content-length needs to change,
#--- but for now it seems to work anyway.

sub add_crowd {
    my ($line) = @_;
    $line =~ s/(.*)(\<title\>)(.*)/$1 $2 Crowd: $3/i;
    return $line;
}


sub logmsg {
    print STDERR "$0 $$: @_\n" if ($Params{verbose});
}


sub load_params {
    my $line;
    my $attribute;
    my $value;

    open PARAMS, "<config" or die "can't open config file";
    
    while (defined($line = <PARAMS>)) {
        chop $line;
        ($attribute,$value) = split(/=/, $line);
        $Params{$attribute} = $value;
    }
    $Params{firewall} = 0 unless defined($Params{firewall}); #backwards comp.
    
    close PARAMS;
}
