package Bencher::Scenario::SetOperationModules;

our $DATE = '2017-02-19'; # DATE
our $VERSION = '0.11'; # VERSION

use 5.010001;
use strict;
use warnings;

our $scenario = {
    summary => 'Benchmark Perl set operation (union, intersection, diff, symmetric diff) modules',
    modules => {
        'List::MoreUtils' => {
            version => '0.407', # singleton() is available from this version
        },
    },
    participants => [
        # UNION
        {
            tags => ['op:union'],
            module => 'Array::Utils',
            function => 'unique',
            code_template => '&Array::Utils::unique(<set1>, <set2>)', # we use &func instead of func to defeat prototype which confuses some tools
        },
        {
            tags => ['op:union'],
            module => 'Set::Scalar',
            function => 'union',
            code_template => 'my $s1 = Set::Scalar->new(@{<set1>}); my $s2 = Set::Scalar->new(@{<set2>}); $s1->union($s2)',
        },
        {
            tags => ['op:union'],
            fcall_template => 'List::MoreUtils::PP::uniq(@{<set1>}, @{<set2>})',
        },
        {
            name => 'List::MoreUtils::XS::uniq',
            tags => ['op:union'],
            module => 'List::MoreUtils::XS',
            fcall_template => 'List::MoreUtils::uniq(@{<set1>}, @{<set2>})',
        },
        {
            tags => ['op:union'],
            fcall_template => 'Array::Set::set_union(<set1>, <set2>)',
        },
        {
            tags => ['op:union'],
            module => 'Set::Array',
            function => 'union',
            code_template => 'my $s1 = Set::Array->new(@{<set1>}); my $s2 = Set::Array->new(@{<set2>}); $s1->union($s2)',
        },
        {
            tags => ['op:union'],
            module => 'Array::AsObject',
            function => 'union',
            code_template => 'my $s1 = Array::AsObject->new(@{<set1>}); my $s2 = Array::AsObject->new(@{<set2>}); $s1->union($s2, 1)',
        },
        {
            tags => ['op:union'],
            module => 'Set::Object',
            function => 'union',
            code_template => 'my $s1 = Set::Object->new(@{<set1>}); my $s2 = Set::Object->new(@{<set2>}); $s1->union($s2)',
        },
        {
            tags => ['op:union'],
            module => 'Set::Tiny',
            function => 'union',
            code_template => 'my $s1 = Set::Tiny->new(@{<set1>}); my $s2 = Set::Tiny->new(@{<set2>}); $s1->union($s2)',
        },
        {
            tags => ['op:union'],
            module => 'List::Collection',
            function => 'union',
            code_template => '[List::Collection::union(<set1>, <set2>)]',
        },

        # SYMDIFF
        {
            tags => ['op:symdiff'],
            module => 'Array::Utils',
            function => 'array_diff',
            code_template => '&Array::Utils::array_diff(<set1>, <set2>)', # we use &func instead of func to defeat prototype which confuses some tools
        },
        {
            tags => ['op:symdiff'],
            module => 'Set::Scalar',
            function => 'symmetric_difference',
            code_template => 'my $s1 = Set::Scalar->new(@{<set1>}); my $s2 = Set::Scalar->new(@{<set2>}); $s1->symmetric_difference($s2)',
        },
        # List::MoreUtils' singleton() can do symmetric difference as long as we
        # make sure that set1 and set2 do not contain duplicates (which, since
        # they should be sets, should not)
        {
            tags => ['op:symdiff'],
            fcall_template => 'List::MoreUtils::PP::singleton(@{<set1>}, @{<set2>})',
        },
        {
            name => 'List::MoreUtils::XS::singleton',
            tags => ['op:symdiff'],
            module => 'List::MoreUtils::XS',
            fcall_template => 'List::MoreUtils::singleton(@{<set1>}, @{<set2>})',
        },
        {
            tags => ['op:symdiff'],
            fcall_template => 'Array::Set::set_symdiff(<set1>, <set2>)',
        },
        {
            tags => ['op:symdiff'],
            module => 'Set::Array',
            function => 'symmetric_difference',
            code_template => 'my $s1 = Set::Array->new(@{<set1>}); my $s2 = Set::Array->new(@{<set2>}); $s1->symmetric_difference($s2)',
        },
        # Array::AsObject::symmetric_difference's handling of duplicates is
        # non-standard though, see its doc
        {
            tags => ['op:symdiff'],
            module => 'Array::AsObject',
            function => 'symmetric_difference',
            code_template => 'my $s1 = Array::AsObject->new(@{<set1>}); my $s2 = Array::AsObject->new(@{<set2>}); $s1->symmetric_difference($s2)',
        },
        {
            tags => ['op:symdiff'],
            module => 'Set::Object',
            function => 'symmetric_difference',
            code_template => 'my $s1 = Set::Object->new(@{<set1>}); my $s2 = Set::Object->new(@{<set2>}); $s1->symmetric_difference($s2)',
        },
        {
            tags => ['op:symdiff'],
            module => 'Set::Tiny',
            function => 'symmetric_difference',
            code_template => 'my $s1 = Set::Tiny->new(@{<set1>}); my $s2 = Set::Tiny->new(@{<set2>}); $s1->symmetric_difference($s2)',
        },
        {
            tags => ['op:symdiff'],
            module => 'List::Collection',
            function => 'complement',
            code_template => '[List::Collection::complement(<set1>, <set2>)]',
        },

        # DIFF
        {
            tags => ['op:diff'],
            module => 'Array::Utils',
            function => 'array_minus',
            code_template => '&Array::Utils::array_minus(<set1>, <set2>)', # we use &func instead of func to defeat prototype which confuses some tools
        },
        {
            tags => ['op:diff'],
            module => 'Set::Scalar',
            function => 'difference',
            code_template => 'my $s1 = Set::Scalar->new(@{<set1>}); my $s2 = Set::Scalar->new(@{<set2>}); $s1->difference($s2)',
        },
        {
            tags => ['op:diff'],
            fcall_template => 'Array::Set::set_diff(<set1>, <set2>)',
        },
        {
            tags => ['op:diff'],
            module => 'Set::Array',
            function => 'difference',
            code_template => 'my $s1 = Set::Array->new(@{<set1>}); my $s2 = Set::Array->new(@{<set2>}); $s1->difference($s2)',
        },
        # Array::AsObject::difference's handling of duplicates is non-standard
        # though, see its doc
        {
            tags => ['op:diff'],
            module => 'Array::AsObject',
            function => 'difference',
            code_template => 'my $s1 = Array::AsObject->new(@{<set1>}); my $s2 = Array::AsObject->new(@{<set2>}); $s1->difference($s2)',
        },
        {
            tags => ['op:diff'],
            module => 'Set::Object',
            function => 'difference',
            code_template => 'my $s1 = Set::Object->new(@{<set1>}); my $s2 = Set::Object->new(@{<set2>}); $s1->difference($s2)',
        },
        {
            tags => ['op:diff'],
            module => 'Set::Tiny',
            function => 'difference',
            code_template => 'my $s1 = Set::Tiny->new(@{<set1>}); my $s2 = Set::Tiny->new(@{<set2>}); $s1->difference($s2)',
        },
        {
            tags => ['op:diff'],
            module => 'List::Collection',
            function => 'subtract',
            code_template => '[List::Collection::subtract(<set1>, <set2>)]',
        },

        # INTERSECT
        {
            tags => ['op:intersect'],
            module => 'Array::Utils',
            function => 'intersect',
            code_template => '&Array::Utils::intersect(<set1>, <set2>)', # we use &func instead of func to defeat prototype which confuses some tools
        },
        {
            tags => ['op:intersect'],
            module => 'Set::Scalar',
            function => 'intersection',
            code_template => 'my $s1 = Set::Scalar->new(@{<set1>}); my $s2 = Set::Scalar->new(@{<set2>}); $s1->intersection($s2)',
        },
        # there's no opposite for singleton() yet in List::MoreUtils (as of
        # v0.413).
        {
            tags => ['op:intersect'],
            fcall_template => 'Array::Set::set_intersect(<set1>, <set2>)',
        },
        {
            tags => ['op:intersect'],
            module => 'Set::Array',
            function => 'intersection',
            code_template => 'my $s1 = Set::Array->new(@{<set1>}); my $s2 = Set::Array->new(@{<set2>}); $s1->intersection($s2)',
        },
        {
            tags => ['op:intersect'],
            module => 'Array::AsObject',
            function => 'intersection',
            code_template => 'my $s1 = Array::AsObject->new(@{<set1>}); my $s2 = Array::AsObject->new(@{<set2>}); $s1->intersection($s2, 1)',
        },
        {
            tags => ['op:intersect'],
            module => 'Set::Object',
            function => 'intersection',
            code_template => 'my $s1 = Set::Object->new(@{<set1>}); my $s2 = Set::Object->new(@{<set2>}); $s1->intersection($s2)',
        },
        {
            tags => ['op:intersect'],
            module => 'Set::Tiny',
            function => 'intersection',
            code_template => 'my $s1 = Set::Tiny->new(@{<set1>}); my $s2 = Set::Tiny->new(@{<set2>}); $s1->intersection($s2)',
        },
        {
            tags => ['op:intersect'],
            module => 'List::Collection',
            function => 'intersect',
            code_template => '[List::Collection::intersect(<set1>, <set2>)]',
        },
    ],

    # XXX: add more datasets (larger data, etc)
    datasets => [
        {
            name => 'num10',
            args => {
                set1 => [1..10],
                set2 => [2..11],
            },
        },
        {
            name => 'num100',
            args => {
                set1 => [1..100],
                set2 => [2..101],
            },
        },
        {
            name => 'num1000',
            args => {
                set1 => [1..1000],
                set2 => [2..1001],
            },
            include_by_default => 0,
        },
    ],
};

1;
# ABSTRACT: Benchmark Perl set operation (union, intersection, diff, symmetric diff) modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::SetOperationModules - Benchmark Perl set operation (union, intersection, diff, symmetric diff) modules

=head1 VERSION

This document describes version 0.11 of Bencher::Scenario::SetOperationModules (from Perl distribution Bencher-Scenario-SetOperationModules), released on 2017-02-19.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m SetOperationModules

To run module startup overhead benchmark:

 % bencher --module-startup -m SetOperationModules

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Array::AsObject> 1.02

L<Array::Set> 0.05

L<Array::Utils> 0.5

L<List::Collection> 0.0.4

L<List::MoreUtils> 0.416

L<List::MoreUtils::PP> 0.416

L<Set::Array> 0.30

L<Set::Object> 1.35

L<Set::Scalar> 1.29

L<Set::Tiny> 0.04

=head1 BENCHMARK PARTICIPANTS

=over

=item * Array::Utils::unique (perl_code) [op:union]

Code template:

 &Array::Utils::unique(<set1>, <set2>)



=item * Set::Scalar::union (perl_code) [op:union]

Code template:

 my $s1 = Set::Scalar->new(@{<set1>}); my $s2 = Set::Scalar->new(@{<set2>}); $s1->union($s2)



=item * List::MoreUtils::PP::uniq (perl_code) [op:union]

Function call template:

 List::MoreUtils::PP::uniq(@{<set1>}, @{<set2>})



=item * List::MoreUtils::XS::uniq (perl_code) [op:union]

Function call template:

 List::MoreUtils::uniq(@{<set1>}, @{<set2>})



=item * Array::Set::set_union (perl_code) [op:union]

Function call template:

 Array::Set::set_union(<set1>, <set2>)



=item * Set::Array::union (perl_code) [op:union]

Code template:

 my $s1 = Set::Array->new(@{<set1>}); my $s2 = Set::Array->new(@{<set2>}); $s1->union($s2)



=item * Array::AsObject::union (perl_code) [op:union]

Code template:

 my $s1 = Array::AsObject->new(@{<set1>}); my $s2 = Array::AsObject->new(@{<set2>}); $s1->union($s2, 1)



=item * Set::Object::union (perl_code) [op:union]

Code template:

 my $s1 = Set::Object->new(@{<set1>}); my $s2 = Set::Object->new(@{<set2>}); $s1->union($s2)



=item * Set::Tiny::union (perl_code) [op:union]

Code template:

 my $s1 = Set::Tiny->new(@{<set1>}); my $s2 = Set::Tiny->new(@{<set2>}); $s1->union($s2)



=item * List::Collection::union (perl_code) [op:union]

Code template:

 [List::Collection::union(<set1>, <set2>)]



=item * Array::Utils::array_diff (perl_code) [op:symdiff]

Code template:

 &Array::Utils::array_diff(<set1>, <set2>)



=item * Set::Scalar::symmetric_difference (perl_code) [op:symdiff]

Code template:

 my $s1 = Set::Scalar->new(@{<set1>}); my $s2 = Set::Scalar->new(@{<set2>}); $s1->symmetric_difference($s2)



=item * List::MoreUtils::PP::singleton (perl_code) [op:symdiff]

Function call template:

 List::MoreUtils::PP::singleton(@{<set1>}, @{<set2>})



=item * List::MoreUtils::XS::singleton (perl_code) [op:symdiff]

Function call template:

 List::MoreUtils::singleton(@{<set1>}, @{<set2>})



=item * Array::Set::set_symdiff (perl_code) [op:symdiff]

Function call template:

 Array::Set::set_symdiff(<set1>, <set2>)



=item * Set::Array::symmetric_difference (perl_code) [op:symdiff]

Code template:

 my $s1 = Set::Array->new(@{<set1>}); my $s2 = Set::Array->new(@{<set2>}); $s1->symmetric_difference($s2)



=item * Array::AsObject::symmetric_difference (perl_code) [op:symdiff]

Code template:

 my $s1 = Array::AsObject->new(@{<set1>}); my $s2 = Array::AsObject->new(@{<set2>}); $s1->symmetric_difference($s2)



=item * Set::Object::symmetric_difference (perl_code) [op:symdiff]

Code template:

 my $s1 = Set::Object->new(@{<set1>}); my $s2 = Set::Object->new(@{<set2>}); $s1->symmetric_difference($s2)



=item * Set::Tiny::symmetric_difference (perl_code) [op:symdiff]

Code template:

 my $s1 = Set::Tiny->new(@{<set1>}); my $s2 = Set::Tiny->new(@{<set2>}); $s1->symmetric_difference($s2)



=item * List::Collection::complement (perl_code) [op:symdiff]

Code template:

 [List::Collection::complement(<set1>, <set2>)]



=item * Array::Utils::array_minus (perl_code) [op:diff]

Code template:

 &Array::Utils::array_minus(<set1>, <set2>)



=item * Set::Scalar::difference (perl_code) [op:diff]

Code template:

 my $s1 = Set::Scalar->new(@{<set1>}); my $s2 = Set::Scalar->new(@{<set2>}); $s1->difference($s2)



=item * Array::Set::set_diff (perl_code) [op:diff]

Function call template:

 Array::Set::set_diff(<set1>, <set2>)



=item * Set::Array::difference (perl_code) [op:diff]

Code template:

 my $s1 = Set::Array->new(@{<set1>}); my $s2 = Set::Array->new(@{<set2>}); $s1->difference($s2)



=item * Array::AsObject::difference (perl_code) [op:diff]

Code template:

 my $s1 = Array::AsObject->new(@{<set1>}); my $s2 = Array::AsObject->new(@{<set2>}); $s1->difference($s2)



=item * Set::Object::difference (perl_code) [op:diff]

Code template:

 my $s1 = Set::Object->new(@{<set1>}); my $s2 = Set::Object->new(@{<set2>}); $s1->difference($s2)



=item * Set::Tiny::difference (perl_code) [op:diff]

Code template:

 my $s1 = Set::Tiny->new(@{<set1>}); my $s2 = Set::Tiny->new(@{<set2>}); $s1->difference($s2)



=item * List::Collection::subtract (perl_code) [op:diff]

Code template:

 [List::Collection::subtract(<set1>, <set2>)]



=item * Array::Utils::intersect (perl_code) [op:intersect]

Code template:

 &Array::Utils::intersect(<set1>, <set2>)



=item * Set::Scalar::intersection (perl_code) [op:intersect]

Code template:

 my $s1 = Set::Scalar->new(@{<set1>}); my $s2 = Set::Scalar->new(@{<set2>}); $s1->intersection($s2)



=item * Array::Set::set_intersect (perl_code) [op:intersect]

Function call template:

 Array::Set::set_intersect(<set1>, <set2>)



=item * Set::Array::intersection (perl_code) [op:intersect]

Code template:

 my $s1 = Set::Array->new(@{<set1>}); my $s2 = Set::Array->new(@{<set2>}); $s1->intersection($s2)



=item * Array::AsObject::intersection (perl_code) [op:intersect]

Code template:

 my $s1 = Array::AsObject->new(@{<set1>}); my $s2 = Array::AsObject->new(@{<set2>}); $s1->intersection($s2, 1)



=item * Set::Object::intersection (perl_code) [op:intersect]

Code template:

 my $s1 = Set::Object->new(@{<set1>}); my $s2 = Set::Object->new(@{<set2>}); $s1->intersection($s2)



=item * Set::Tiny::intersection (perl_code) [op:intersect]

Code template:

 my $s1 = Set::Tiny->new(@{<set1>}); my $s2 = Set::Tiny->new(@{<set2>}); $s1->intersection($s2)



=item * List::Collection::intersect (perl_code) [op:intersect]

Code template:

 [List::Collection::intersect(<set1>, <set2>)]



=back

=head1 BENCHMARK DATASETS

=over

=item * num10

=item * num100

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.24.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.5 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m SetOperationModules >>):

 #table1#
 +---------------------------------------+---------+-----------+------------+------------+---------+---------+
 | participant                           | dataset | rate (/s) |  time (ms) | vs_slowest |  errors | samples |
 +---------------------------------------+---------+-----------+------------+------------+---------+---------+
 | Array::AsObject::symmetric_difference | num100  |     112.6 | 8.878      |      1     | 4.3e-07 |      20 |
 | Array::AsObject::intersection         | num100  |     230.2 | 4.343      |      2.044 |   2e-07 |      22 |
 | Array::AsObject::difference           | num100  |     470   | 2.13       |      4.17  | 2.6e-07 |      21 |
 | List::Collection::complement          | num100  |     561   | 1.78       |      4.98  | 9.1e-07 |      20 |
 | Set::Array::intersection              | num100  |     622   | 1.61       |      5.52  | 1.3e-06 |      20 |
 | List::Collection::union               | num100  |     940   | 1.1        |      8.4   | 1.2e-06 |      20 |
 | List::Collection::intersect           | num100  |    1400   | 0.71       |     12     | 1.1e-06 |      20 |
 | Set::Scalar::intersection             | num100  |    2020   | 0.495      |     17.9   | 2.7e-07 |      20 |
 | Set::Scalar::union                    | num100  |    2120   | 0.472      |     18.8   | 3.7e-07 |      20 |
 | Set::Scalar::difference               | num100  |    2230   | 0.449      |     19.8   | 4.2e-07 |      21 |
 | Set::Scalar::symmetric_difference     | num100  |    3010   | 0.332      |     26.7   | 2.1e-07 |      20 |
 | Array::AsObject::symmetric_difference | num10   |    5100   | 0.196      |     45.3   | 4.6e-08 |      27 |
 | List::Collection::complement          | num10   |    8100   | 0.12       |     72     | 6.9e-07 |      20 |
 | Set::Object::symmetric_difference     | num100  |   10000   | 0.1        |     88     | 1.3e-07 |      20 |
 | Set::Object::intersection             | num100  |   11300   | 0.0885     |    100     |   8e-08 |      20 |
 | Array::AsObject::union                | num100  |   11328.9 | 0.0882696  |    100.58  |   0     |      20 |
 | Array::AsObject::intersection         | num10   |   11500   | 0.0872     |    102     | 2.5e-08 |      23 |
 | Set::Scalar::intersection             | num10   |   12000   | 0.087      |    100     | 1.7e-07 |      31 |
 | Set::Scalar::union                    | num10   |   12000   | 0.0835     |    106     | 2.7e-08 |      20 |
 | Set::Scalar::difference               | num10   |   12500   | 0.0802     |    111     | 2.7e-08 |      20 |
 | Set::Array::difference                | num100  |   13200   | 0.076      |    117     | 2.5e-08 |      22 |
 | Set::Array::symmetric_difference      | num100  |   13000   | 0.076      |    120     | 7.8e-08 |      21 |
 | Set::Object::union                    | num100  |   14000   | 0.072      |    120     |   1e-07 |      21 |
 | Set::Object::difference               | num100  |   15000   | 0.069      |    130     | 1.1e-07 |      20 |
 | Set::Tiny::symmetric_difference       | num100  |   16000   | 0.061      |    140     | 1.1e-07 |      20 |
 | List::Collection::intersect           | num10   |   16000   | 0.061      |    150     | 9.7e-08 |      24 |
 | List::Collection::union               | num10   |   17000   | 0.06       |    150     | 4.5e-07 |      20 |
 | Set::Tiny::difference                 | num100  |   17400   | 0.0573     |    155     | 2.6e-08 |      21 |
 | Set::Tiny::union                      | num100  |   18000   | 0.057      |    160     | 1.1e-07 |      20 |
 | Set::Tiny::intersection               | num100  |   18000   | 0.055      |    160     | 1.1e-07 |      20 |
 | Set::Scalar::symmetric_difference     | num10   |   18800   | 0.0532     |    167     | 2.7e-08 |      20 |
 | Set::Array::union                     | num100  |   19100   | 0.0524     |    169     | 2.3e-08 |      28 |
 | Array::Set::set_intersect             | num100  |   21000   | 0.047      |    190     | 1.6e-07 |      27 |
 | Array::Set::set_symdiff               | num100  |   21200   | 0.0471     |    188     | 1.2e-08 |      25 |
 | Array::Set::set_union                 | num100  |   22057.6 | 0.0453358  |    195.83  |   0     |      23 |
 | Array::AsObject::difference           | num10   |   24076.9 | 0.0415336  |    213.758 |   0     |      20 |
 | Array::Utils::array_diff              | num100  |   26000   | 0.038      |    230     | 4.2e-08 |      33 |
 | List::MoreUtils::PP::singleton        | num100  |   27000   | 0.037      |    240     | 1.1e-08 |      30 |
 | Array::Set::set_diff                  | num100  |   28500   | 0.0351     |    253     | 1.2e-08 |      24 |
 | Array::Utils::array_minus             | num100  |   29000   | 0.035      |    250     | 1.9e-07 |      25 |
 | Array::Utils::intersect               | num100  |   29300   | 0.0342     |    260     | 1.3e-08 |      22 |
 | List::Collection::subtract            | num100  |   30400   | 0.0329     |    270     | 1.1e-08 |      30 |
 | List::MoreUtils::PP::uniq             | num100  |   35000   | 0.028      |    320     | 4.4e-08 |      29 |
 | Set::Array::intersection              | num10   |   37000   | 0.027      |    330     | 5.3e-08 |      20 |
 | List::MoreUtils::XS::singleton        | num100  |   43400   | 0.023      |    386     | 6.7e-09 |      20 |
 | List::MoreUtils::XS::uniq             | num100  |   56600   | 0.0177     |    502     | 5.4e-09 |      31 |
 | Set::Object::symmetric_difference     | num10   |   63800   | 0.0157     |    567     | 5.8e-09 |      26 |
 | Set::Array::difference                | num10   |   66800   | 0.015      |    593     | 6.7e-09 |      20 |
 | Set::Array::symmetric_difference      | num10   |   73000   | 0.014      |    650     | 2.5e-08 |      22 |
 | Array::AsObject::union                | num10   |   75000   | 0.013      |    670     | 2.2e-08 |      29 |
 | Set::Object::intersection             | num10   |   94000   | 0.011      |    840     | 1.3e-08 |      20 |
 | Set::Array::union                     | num10   |   95600   | 0.0105     |    848     | 3.1e-09 |      23 |
 | Set::Object::union                    | num10   |  110000   | 0.0089     |   1000     | 1.1e-08 |      27 |
 | Set::Object::difference               | num10   |  120000   | 0.0086     |   1000     | 1.2e-08 |      25 |
 | Set::Tiny::symmetric_difference       | num10   |  130000   | 0.0075     |   1200     |   1e-08 |      20 |
 | Set::Tiny::difference                 | num10   |  140000   | 0.007      |   1300     | 1.3e-08 |      20 |
 | Set::Tiny::union                      | num10   |  150000   | 0.0067     |   1300     | 1.3e-08 |      20 |
 | Set::Tiny::intersection               | num10   |  160000   | 0.0064     |   1400     | 1.3e-08 |      21 |
 | Array::Set::set_symdiff               | num10   |  179258   | 0.00557856 |   1591.47  |   0     |      20 |
 | Array::Set::set_union                 | num10   |  183000   | 0.00547    |   1620     | 1.4e-09 |      27 |
 | Array::Set::set_intersect             | num10   |  184430   | 0.0054221  |   1637.4   |   0     |      20 |
 | List::Collection::subtract            | num10   |  190000   | 0.00526    |   1690     | 1.3e-09 |      32 |
 | Array::Utils::array_diff              | num10   |  220000   | 0.00455    |   1950     | 1.7e-09 |      20 |
 | Array::Set::set_diff                  | num10   |  229376   | 0.00435965 |   2036.43  |   0     |      20 |
 | List::MoreUtils::PP::singleton        | num10   |  243262   | 0.0041108  |   2159.71  |   0     |      20 |
 | Array::Utils::unique                  | num100  |  260000   | 0.0038     |   2300     | 6.7e-09 |      20 |
 | Array::Utils::intersect               | num10   |  263305   | 0.00379788 |   2337.66  |   0     |      20 |
 | Array::Utils::array_minus             | num10   |  265000   | 0.00378    |   2350     | 3.8e-09 |      35 |
 | List::MoreUtils::PP::uniq             | num10   |  315256   | 0.00317203 |   2798.88  |   0     |      20 |
 | List::MoreUtils::XS::singleton        | num10   |  420000   | 0.0024     |   3700     | 2.5e-09 |      20 |
 | List::MoreUtils::XS::uniq             | num10   |  520000   | 0.00192    |   4610     | 8.3e-10 |      20 |
 | Array::Utils::unique                  | num10   |  850000   | 0.0012     |   7500     | 1.2e-09 |      20 |
 +---------------------------------------+---------+-----------+------------+------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m SetOperationModules --module-startup >>):

 #table2#
 +---------------------+------------------------------+--------------------+----------------+-----------+------------------------+------------+---------+---------+
 | participant         | proc_private_dirty_size (MB) | proc_rss_size (MB) | proc_size (MB) | time (ms) | mod_overhead_time (ms) | vs_slowest |  errors | samples |
 +---------------------+------------------------------+--------------------+----------------+-----------+------------------------+------------+---------+---------+
 | Set::Array          | 3                            | 6.5                | 26             |     14    |                  11.45 |       1    | 6.6e-05 |      21 |
 | List::Collection    | 0.93                         | 4.3                | 20             |     14    |                  11.45 |       1    | 2.4e-05 |      20 |
 | Array::AsObject     | 2.2                          | 5.6                | 23             |     14    |                  11.45 |       1    | 2.7e-05 |      20 |
 | Set::Object         | 1                            | 4.4                | 20             |     11    |                   8.45 |       1.3  | 2.1e-05 |      20 |
 | Set::Scalar         | 1.2                          | 4.5                | 20             |     10    |                   7.45 |       1.3  |   4e-05 |      20 |
 | List::MoreUtils     | 1.1                          | 4.5                | 20             |      8.1  |                   5.55 |       1.7  | 2.9e-05 |      20 |
 | List::MoreUtils::PP | 1.6                          | 5.1                | 23             |      5.6  |                   3.05 |       2.5  | 2.4e-05 |      20 |
 | Array::Set          | 2.7                          | 6.1                | 28             |      5.1  |                   2.55 |       2.7  | 2.6e-05 |      20 |
 | Set::Tiny           | 2.3                          | 5.8                | 27             |      3.7  |                   1.15 |       3.7  | 8.5e-06 |      20 |
 | Array::Utils        | 1.9                          | 5.3                | 23             |      3.6  |                   1.05 |       3.9  | 2.9e-05 |      20 |
 | perl -e1 (baseline) | 0.934                        | 4.34               | 20             |      2.55 |                   0    |       5.48 |   2e-06 |      20 |
 +---------------------+------------------------------+--------------------+----------------+-----------+------------------------+------------+---------+---------+


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-SetOperationModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-SetOperationModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-SetOperationModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Benchmark::Featureset::SetOps>

Excluded modules: L<Set::Bag> (expects hashes instead of arrays),
L<Set::SortedArray> (members are sorted).

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017, 2016, 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
