package Bencher::Scenario::Serializers;

our $DATE = '2016-06-30'; # DATE
our $VERSION = '0.13'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark Perl data serialization modules',
    participants => [
        {
            tags => ['json', 'serialize'],
            module => 'JSON::PP',
            function => 'encode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::PP',
            function => 'decode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::Tiny',
            function => 'encode_json',
            code_template => 'JSON::Tiny::encode_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::Tiny',
            function => 'decode_json',
            code_template => 'JSON::Tiny::decode_json(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'Cpanel::JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'Cpanel::JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::MaybeXS',
            function => 'encode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::MaybeXS',
            function => 'decode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Decode::Regexp::from_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'PERLANCAR::JSON::Match::match_json(<data>)',
            include_by_default => 0,
        },
        {
            tags => ['json', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'JSON::Decode::Marpa::from_json(<data>)',
        },
        {
            name => 'Pegex::JSON',
            tags => ['json', 'deserialize'],
            module => 'Pegex::JSON',
            code_template => 'state $obj = Pegex::JSON->new; $obj->load(<data>);',
        },
        {
            tags => ['json', 'serialize'],
            fcall_template => 'JSON::Create::create_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Parse::parse_json(<data>)',
        },

        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Old::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Old::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Syck::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Syck::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::XS::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::XS::Load(<data>)',
        },

        {
            tags => ['binary', 'serialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::freeze(<data>)',
        },
        {
            tags => ['binary', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::thaw(<data>)',
        },

        {
            tags => ['binary', 'serialize'],
            fcall_template => 'Sereal::encode_sereal(<data>)',
        },
        {
            tags => ['binary', 'deserialize'],
            fcall_template => 'Sereal::decode_sereal(<data>)',
        },

        {
            name => 'Data::MessagePack::pack',
            tags => ['binary', 'serialize'],
            module => 'Data::MessagePack',
            function => 'pack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->pack(<data>)',
        },
        {
            name => 'Data::MessagePack::unpack',
            tags => ['binary', 'deserialize'],
            module => 'Data::MessagePack',
            function => 'unpack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->unpack(<data>)',
        },
    ],

    # XXX: add more datasets (larger data, etc)
    datasets => [
        {
            name => 'undef',
            summary => 'undef',
            args => {data=>undef},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A non-Unicode string 1024 characters/bytes long',
            args => {data=>'a' x 1024},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A Unicode string 1024 characters (3072-bytes) long',
            args => {data=>'我爱你爱你一辈子' x 128},
            tags => ['serialize', 'unicode'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>[1..10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_100',
            summary => 'A 100-element array containing ints',
            args => {data=>[1..100]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>[1..1000]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_str1k_10',
            summary => 'A 10-element array containing 1024-characters/bytes-long non-Unicode strings',
            args => {data=>[('a' x 1024) x 10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_ustr1k_10',
            summary => 'A 10-element array containing 1024-characters-long (3072-bytes long) Unicode strings',
            args => {data=>[('我爱你爱你一辈子' x 128) x 10]},
            tags => ['serialize', 'json'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'hash_int_10',
            summary => 'A 10-key hash {1=>0, ..., 10=>0}',
            args => {data=>{map {$_=>0} 1..10}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_100',
            summary => 'A 100-key hash {1=>0, ..., 100=>0}',
            args => {data=>{map {$_=>0} 1..100}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_1000',
            summary => 'A 1000-key hash {1=>0, ..., 1000=>0}',
            args => {data=>{map {$_=>0} 1..1000}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'json:null',
            summary => 'null',
            args => {data=>'null'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:str1k',
            summary => 'A non-Unicode (ASCII) string 1024-characters/bytes long',
            args => {data=>'"' . ('a' x 1024) . '"'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'json:array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..10).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_100',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..100).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>'['.join(',',1..1000).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_str1k_10',
            summary => 'A 10-element array containing 1024-characters/bytes-long non-Unicode strings',
            args => {data=>'['.join(',',(('"'.('a' x 1024).'"') x 10)).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },

        {
            name => 'json:hash_int_10',
            summary => 'A 10-key hash {"1":0, ..., "10":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..10).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_100',
            summary => 'A 100-key hash {"1":0, ..., "100":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..100).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_1000',
            summary => 'A 1000-key hash {"1":0, ..., "1000":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..1000).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
    ],
};

1;
# ABSTRACT: Benchmark Perl data serialization modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::Serializers - Benchmark Perl data serialization modules

=head1 VERSION

This document describes version 0.13 of Bencher::Scenario::Serializers (from Perl distribution Bencher-Scenario-Serializers), released on 2016-06-30.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m Serializers

To run module startup overhead benchmark:

 % bencher --module-startup -m Serializers

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<JSON::PP> 2.27400

L<JSON::Tiny> 0.56

L<JSON::XS> 3.02

L<Cpanel::JSON::XS> 3.0213

L<JSON::MaybeXS> 1.003005

L<JSON::Decode::Regexp> 0.08

L<JSON::Decode::Marpa> 0.02

L<Pegex::JSON> 0.27

L<JSON::Create> 0.21

L<JSON::Parse> 0.41

L<YAML::Old> 1.07

L<YAML::Syck> 1.29

L<YAML::XS> 0.62

L<Storable> 2.53_01

L<Sereal> 3.014

L<Data::MessagePack> 0.50

=head1 BENCHMARK PARTICIPANTS

=over

=item * JSON::PP::encode_json (perl_code) [json, serialize]

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)



=item * JSON::PP::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)



=item * JSON::Tiny::encode_json (perl_code) [json, serialize]

Code template:

 JSON::Tiny::encode_json(<data>)



=item * JSON::Tiny::decode_json (perl_code) [json, deserialize]

Code template:

 JSON::Tiny::decode_json(<data>)



=item * JSON::XS::encode_json (perl_code) [json, serialize]

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * JSON::XS::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * Cpanel::JSON::XS::encode_json (perl_code) [json, serialize]

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * Cpanel::JSON::XS::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * JSON::MaybeXS::encode_json (perl_code) [json, serialize]

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)



=item * JSON::MaybeXS::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)



=item * JSON::Decode::Regexp::from_json (perl_code) [json, deserialize]

Function call template:

 JSON::Decode::Regexp::from_json(<data>)



=item * PERLANCAR::JSON::Match::match_json (perl_code) (not included by default) [json, deserialize, cant_handle_scalar]

Function call template:

 JSON::Decode::Marpa::from_json(<data>)



=item * JSON::Decode::Marpa::from_json (perl_code) [json, deserialize]

Code template:

 state $obj = Pegex::JSON->new; $obj->load(<data>);



=item * Pegex::JSON (perl_code) [json, serialize]

Function call template:

 JSON::Create::create_json(<data>)



=item * JSON::Create::create_json (perl_code) [json, deserialize]

Function call template:

 JSON::Parse::parse_json(<data>)



=item * JSON::Parse::parse_json (perl_code) [yaml, serialize]

Function call template:

 YAML::Old::Dump(<data>)



=item * YAML::Old::Dump (perl_code) [yaml, deserialize]

Function call template:

 YAML::Old::Load(<data>)



=item * YAML::Old::Load (perl_code) [yaml, serialize]

Function call template:

 YAML::Syck::Dump(<data>)



=item * YAML::Syck::Dump (perl_code) [yaml, deserialize]

Function call template:

 YAML::Syck::Load(<data>)



=item * YAML::Syck::Load (perl_code) [yaml, serialize]

Function call template:

 YAML::XS::Dump(<data>)



=item * YAML::XS::Dump (perl_code) [yaml, deserialize]

Function call template:

 YAML::XS::Load(<data>)



=item * YAML::XS::Load (perl_code) [binary, serialize, cant_handle_scalar]

Function call template:

 Storable::freeze(<data>)



=item * Storable::freeze (perl_code) [binary, deserialize, cant_handle_scalar]

Function call template:

 Storable::thaw(<data>)



=item * Storable::thaw (perl_code) [binary, serialize]

Function call template:

 Sereal::encode_sereal(<data>)



=item * Sereal::encode_sereal (perl_code) [binary, deserialize]

Function call template:

 Sereal::decode_sereal(<data>)



=item * Sereal::decode_sereal (perl_code) [binary, serialize]

Code template:

 state $obj = Data::MessagePack->new; $obj->pack(<data>)



=item * Data::MessagePack::pack (perl_code) [binary, deserialize]

Code template:

 state $obj = Data::MessagePack->new; $obj->unpack(<data>)



=item * Data::MessagePack::unpack (perl_code)

L<Data::MessagePack>::unpack



=back

=head1 BENCHMARK DATASETS

=over

=item * undef [serialize]

undef

=item * num [serialize]

A single number (-1.23)

=item * str1k [serialize]

A non-Unicode string 1024 characters/bytes long

=item * str1k [serialize, unicode]

A Unicode string 1024 characters (3072-bytes) long

=item * array_int_10 [serialize]

A 10-element array containing ints

=item * array_int_100 [serialize]

A 100-element array containing ints

=item * array_int_1000 [serialize]

A 1000-element array containing ints

=item * array_str1k_10 [serialize]

A 10-element array containing 1024-characters/bytes-long non-Unicode strings

=item * array_ustr1k_10 [serialize, json]

A 10-element array containing 1024-characters-long (3072-bytes long) Unicode strings

=item * hash_int_10 [serialize]

A 10-key hash {1=>0, ..., 10=>0}

=item * hash_int_100 [serialize]

A 100-key hash {1=>0, ..., 100=>0}

=item * hash_int_1000 [serialize]

A 1000-key hash {1=>0, ..., 1000=>0}

=item * json:null [deserialize]

null

=item * json:num [deserialize]

A single number (-1.23)

=item * json:str1k [deserialize]

A non-Unicode (ASCII) string 1024-characters/bytes long

=item * json:array_int_10 [deserialize]

A 10-element array containing ints

=item * json:array_int_100 [deserialize]

A 10-element array containing ints

=item * json:array_int_1000 [deserialize]

A 1000-element array containing ints

=item * json:array_str1k_10 [deserialize]

A 10-element array containing 1024-characters/bytes-long non-Unicode strings

=item * json:hash_int_10 [deserialize]

A 10-key hash {"1":0, ..., "10":0}

=item * json:hash_int_100 [deserialize]

A 100-key hash {"1":0, ..., "100":0}

=item * json:hash_int_1000 [deserialize]

A 1000-key hash {"1":0, ..., "1000":0}

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.1 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark serializing (C<< bencher -m Serializers --include-participant-tags serialize >>):

 #table1#
 {dataset=>"array_int_10"}
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | YAML::Old::Dump               |      2570 |  389      |        1   | 2.1e-07 |      20 |
 | JSON::PP::encode_json         |     42000 |   24      |       16   |   8e-08 |      20 |
 | JSON::Tiny::encode_json       |     56000 |   18      |       22   | 2.7e-08 |      20 |
 | YAML::Syck::Dump              |     86000 |   12      |       34   | 1.3e-08 |      20 |
 | YAML::XS::Dump                |    100000 |    9.7    |       40   |   6e-08 |      25 |
 | Storable::freeze              |    228000 |    4.39   |       88.8 | 1.6e-09 |      21 |
 | Sereal::encode_sereal         |   1050000 |    0.956  |      408   |   1e-10 |      24 |
 | JSON::MaybeXS::encode_json    |   1200000 |    0.85   |      460   | 1.3e-09 |      20 |
 | Cpanel::JSON::XS::encode_json |   1200000 |    0.84   |      460   | 1.2e-09 |      22 |
 | JSON::XS::encode_json         |   1200000 |    0.8    |      490   | 2.1e-09 |      20 |
 | Data::MessagePack::pack       |   1320000 |    0.756  |      515   | 4.2e-10 |      20 |
 | JSON::Create::create_json     |   1387000 |    0.7207 |      540.3 | 4.6e-11 |      24 |
 +-------------------------------+-----------+-----------+------------+---------+---------+

 #table2#
 {dataset=>"array_int_100"}
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | participant                   | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | YAML::Old::Dump               |       318 |  3.15     |        1   | 1.6e-06 |      20 |
 | JSON::PP::encode_json         |      5600 |  0.18     |       18   | 2.1e-07 |      20 |
 | JSON::Tiny::encode_json       |      9000 |  0.11     |       28   | 1.6e-07 |      20 |
 | YAML::XS::Dump                |     15000 |  0.065    |       48   | 1.1e-07 |      20 |
 | YAML::Syck::Dump              |     16400 |  0.061    |       51.6 | 2.7e-08 |      20 |
 | Storable::freeze              |    100000 |  0.0098   |      320   | 1.3e-08 |      20 |
 | JSON::MaybeXS::encode_json    |    170000 |  0.0057   |      550   | 1.3e-08 |      20 |
 | Cpanel::JSON::XS::encode_json |    180000 |  0.0055   |      570   | 2.7e-08 |      20 |
 | JSON::XS::encode_json         |    180000 |  0.0055   |      570   | 6.5e-09 |      21 |
 | JSON::Create::create_json     |    223800 |  0.004468 |      703.9 | 4.7e-11 |      20 |
 | Sereal::encode_sereal         |    240000 |  0.0042   |      750   | 6.4e-09 |      22 |
 | Data::MessagePack::pack       |    240000 |  0.0042   |      750   | 6.7e-09 |      20 |
 +-------------------------------+-----------+-----------+------------+---------+---------+

 #table3#
 {dataset=>"array_int_1000"}
 +-------------------------------+-----------+------------+------------+---------+---------+
 | participant                   | rate (/s) | time (ms)  | vs_slowest |  errors | samples |
 +-------------------------------+-----------+------------+------------+---------+---------+
 | YAML::Old::Dump               |      32   | 31         |      1     | 7.6e-05 |      20 |
 | JSON::PP::encode_json         |     570   |  1.7       |     18     | 2.2e-06 |      20 |
 | JSON::Tiny::encode_json       |     967   |  1.03      |     30.2   | 4.8e-07 |      20 |
 | YAML::Syck::Dump              |    1700   |  0.61      |     52     | 6.2e-07 |      21 |
 | YAML::XS::Dump                |    1680   |  0.594     |     52.6   | 2.1e-07 |      21 |
 | Storable::freeze              |   14000   |  0.072     |    430     | 4.5e-07 |      20 |
 | JSON::MaybeXS::encode_json    |   19341.3 |  0.0517028 |    604.306 | 3.4e-11 |      20 |
 | JSON::XS::encode_json         |   19442.1 |  0.0514348 |    607.456 | 4.5e-11 |      20 |
 | Cpanel::JSON::XS::encode_json |   19524.3 |  0.0512183 |    610.023 | 3.5e-11 |      23 |
 | JSON::Create::create_json     |   22000   |  0.046     |    680     | 5.3e-08 |      20 |
 | Sereal::encode_sereal         |   25880   |  0.03864   |    808.6   | 1.4e-10 |      26 |
 | Data::MessagePack::pack       |   26300   |  0.0381    |    821     | 1.3e-08 |      20 |
 +-------------------------------+-----------+------------+------------+---------+---------+

 #table4#
 {dataset=>"array_str1k_10"}
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | YAML::Old::Dump               |   2030    |  492      |    1       | 2.7e-07 |      20 |
 | JSON::Tiny::encode_json       |   7159.67 |  139.671  |    3.52582 | 5.7e-11 |      28 |
 | YAML::XS::Dump                |   7300    |  140      |    3.6     | 2.1e-07 |      20 |
 | JSON::PP::encode_json         |  10500    |   95.5    |    5.16    | 2.7e-08 |      20 |
 | YAML::Syck::Dump              |  19571    |   51.0959 |    9.63787 | 4.6e-11 |      22 |
 | JSON::Create::create_json     |  39357    |   25.408  |   19.382   | 9.3e-11 |      20 |
 | JSON::MaybeXS::encode_json    |  44000    |   23      |   22       |   6e-08 |      20 |
 | Cpanel::JSON::XS::encode_json |  45000    |   22      |   22       | 2.7e-08 |      20 |
 | JSON::XS::encode_json         |  45784    |   21.842  |   22.546   | 5.8e-11 |      20 |
 | Storable::freeze              | 170000    |    5.9    |   84       | 1.5e-08 |      20 |
 | Data::MessagePack::pack       | 330000    |    3      |  160       | 6.1e-09 |      24 |
 | Sereal::encode_sereal         | 332000    |    3.01   |  163       | 8.5e-10 |      20 |
 +-------------------------------+-----------+-----------+------------+---------+---------+

 #table5#
 {dataset=>"array_ustr1k_10"}
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | YAML::Old::Dump               |       860 |    1200   |       1    | 1.3e-06 |      20 |
 | JSON::Tiny::encode_json       |      1200 |     830   |       1.4  | 9.1e-07 |      20 |
 | JSON::PP::encode_json         |      1890 |     528   |       2.21 |   2e-07 |      23 |
 | YAML::XS::Dump                |      3100 |     320   |       3.6  | 4.7e-07 |      21 |
 | JSON::XS::encode_json         |      3170 |     315   |       3.7  | 2.1e-07 |      20 |
 | JSON::MaybeXS::encode_json    |      3430 |     291   |       4    | 2.1e-07 |      20 |
 | Cpanel::JSON::XS::encode_json |      3450 |     290   |       4.02 | 2.1e-07 |      20 |
 | YAML::Syck::Dump              |      7300 |     140   |       8.5  | 2.7e-07 |      20 |
 | JSON::Create::create_json     |     14300 |      70   |      16.6  | 2.7e-08 |      20 |
 | Storable::freeze              |    120000 |       8.1 |     140    | 1.2e-08 |      24 |
 | Sereal::encode_sereal         |    180000 |       5.7 |     210    | 6.7e-09 |      20 |
 | Data::MessagePack::pack       |    260000 |       3.9 |     300    | 8.1e-09 |      21 |
 +-------------------------------+-----------+-----------+------------+---------+---------+

 #table6#
 {dataset=>"hash_int_10"}
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | YAML::Old::Dump               |      2000 |    490    |        1   | 1.2e-06 |      20 |
 | JSON::PP::encode_json         |     19000 |     54    |        9.2 | 1.1e-07 |      20 |
 | JSON::Tiny::encode_json       |     29000 |     35    |       14   | 5.1e-08 |      22 |
 | YAML::Syck::Dump              |     52000 |     19    |       25   | 2.7e-08 |      20 |
 | YAML::XS::Dump                |     58000 |     17    |       29   | 2.7e-08 |      20 |
 | Storable::freeze              |    140000 |      7.12 |       69.3 | 3.3e-09 |      20 |
 | Sereal::encode_sereal         |    390000 |      2.6  |      190   | 6.5e-09 |      21 |
 | Data::MessagePack::pack       |    400000 |      2.5  |      200   | 6.8e-09 |      24 |
 | JSON::XS::encode_json         |    480000 |      2.1  |      230   | 2.5e-09 |      20 |
 | JSON::MaybeXS::encode_json    |    480000 |      2.1  |      240   | 3.3e-09 |      20 |
 | Cpanel::JSON::XS::encode_json |    490000 |      2.1  |      240   | 6.5e-09 |      21 |
 | JSON::Create::create_json     |    530000 |      1.9  |      260   | 6.5e-09 |      21 |
 +-------------------------------+-----------+-----------+------------+---------+---------+

 #table7#
 {dataset=>"hash_int_100"}
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | participant                   | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | YAML::Old::Dump               |       245 |     4.08  |        1   | 4.1e-06 |      22 |
 | JSON::PP::encode_json         |      2200 |     0.46  |        8.8 | 6.4e-07 |      20 |
 | JSON::Tiny::encode_json       |      3600 |     0.28  |       15   | 4.7e-07 |      21 |
 | YAML::XS::Dump                |      7100 |     0.14  |       29   | 2.1e-07 |      20 |
 | YAML::Syck::Dump              |      7200 |     0.14  |       29   | 5.9e-07 |      20 |
 | Storable::freeze              |     42000 |     0.024 |      170   | 2.7e-08 |      20 |
 | Sereal::encode_sereal         |     43000 |     0.023 |      180   | 2.7e-08 |      20 |
 | Data::MessagePack::pack       |     46000 |     0.022 |      190   |   5e-08 |      23 |
 | JSON::XS::encode_json         |     53000 |     0.019 |      220   | 2.5e-08 |      22 |
 | JSON::MaybeXS::encode_json    |     54000 |     0.019 |      220   | 3.1e-08 |      23 |
 | Cpanel::JSON::XS::encode_json |     55000 |     0.018 |      220   | 4.6e-08 |      27 |
 | JSON::Create::create_json     |     59000 |     0.017 |      240   | 2.2e-08 |      29 |
 +-------------------------------+-----------+-----------+------------+---------+---------+

 #table8#
 {dataset=>"hash_int_1000"}
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | participant                   | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | YAML::Old::Dump               |        25 |    40     |       1    | 2.5e-05 |      20 |
 | JSON::PP::encode_json         |       225 |     4.44  |       9.01 |   4e-06 |      20 |
 | JSON::Tiny::encode_json       |       370 |     2.7   |      15    |   4e-06 |      20 |
 | YAML::XS::Dump                |       681 |     1.47  |      27.3  | 1.3e-06 |      20 |
 | YAML::Syck::Dump              |       695 |     1.44  |      27.8  | 8.5e-07 |      20 |
 | Sereal::encode_sereal         |      3660 |     0.274 |     146    | 2.1e-07 |      20 |
 | Storable::freeze              |      4200 |     0.24  |     170    | 4.8e-07 |      20 |
 | Data::MessagePack::pack       |      4270 |     0.234 |     171    | 2.1e-07 |      20 |
 | JSON::MaybeXS::encode_json    |      4700 |     0.21  |     190    | 2.1e-07 |      20 |
 | JSON::XS::encode_json         |      4800 |     0.21  |     190    | 1.5e-06 |      21 |
 | Cpanel::JSON::XS::encode_json |      4810 |     0.208 |     192    | 5.2e-08 |      21 |
 | JSON::Create::create_json     |      5100 |     0.2   |     210    | 6.9e-07 |      20 |
 +-------------------------------+-----------+-----------+------------+---------+---------+

 #table9#
 {dataset=>"num"}
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | YAML::Old::Dump               |     13000 |   76      |        1   |   8e-08 |      20 |
 | JSON::Tiny::encode_json       |    140000 |    7.3    |       10   | 2.7e-08 |      20 |
 | YAML::Syck::Dump              |    140000 |    7.1    |       11   | 1.7e-08 |      20 |
 | JSON::PP::encode_json         |    189000 |    5.29   |       14.3 | 1.5e-09 |      26 |
 | YAML::XS::Dump                |    210000 |    4.7    |       16   | 1.3e-08 |      20 |
 | JSON::MaybeXS::encode_json    |   1600000 |    0.61   |      120   | 1.7e-09 |      24 |
 | JSON::XS::encode_json         |   1700000 |    0.59   |      130   | 8.5e-10 |      20 |
 | Cpanel::JSON::XS::encode_json |   1710000 |    0.585  |      129   | 9.8e-11 |      21 |
 | Sereal::encode_sereal         |   2200000 |    0.45   |      170   | 8.4e-10 |      20 |
 | JSON::Create::create_json     |   2308000 |    0.4333 |      174.7 | 3.5e-11 |      20 |
 | Data::MessagePack::pack       |   3360000 |    0.298  |      254   | 5.6e-11 |      28 |
 +-------------------------------+-----------+-----------+------------+---------+---------+

 #table10#
 {dataset=>"str1k"}
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | YAML::Old::Dump               |     13000 |    78     |     1      |   1e-07 |      21 |
 | JSON::Tiny::encode_json       |     58500 |    17.1   |     4.59   | 6.7e-09 |      20 |
 | YAML::XS::Dump                |     62080 |    16.108 |     4.8708 | 4.6e-11 |      20 |
 | JSON::PP::encode_json         |     87000 |    11     |     6.8    | 5.5e-08 |      21 |
 | YAML::Syck::Dump              |    105000 |     9.51  |     8.25   | 8.6e-09 |      27 |
 | JSON::Create::create_json     |    367700 |     2.72  |    28.85   | 4.7e-11 |      20 |
 | Cpanel::JSON::XS::encode_json |    380000 |     2.6   |    30      | 1.1e-08 |      20 |
 | JSON::MaybeXS::encode_json    |    390000 |     2.5   |    31      |   1e-08 |      20 |
 | JSON::XS::encode_json         |    430000 |     2.3   |    33      | 2.5e-09 |      20 |
 | Sereal::encode_sereal         |   1700000 |     0.58  |   130      | 8.5e-10 |      20 |
 | Data::MessagePack::pack       |   2390000 |     0.419 |   187      | 2.2e-10 |      20 |
 +-------------------------------+-----------+-----------+------------+---------+---------+

 #table11#
 {dataset=>"undef"}
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | participant                   | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +-------------------------------+-----------+-----------+------------+---------+---------+
 | YAML::Old::Dump               |     15300 |    65.4   |        1   | 2.7e-08 |      20 |
 | YAML::Syck::Dump              |    180000 |     5.5   |       12   | 6.7e-09 |      20 |
 | JSON::Tiny::encode_json       |    250000 |     4     |       16   |   5e-09 |      20 |
 | JSON::PP::encode_json         |    312000 |     3.21  |       20.4 | 1.7e-09 |      20 |
 | YAML::XS::Dump                |    390000 |     2.6   |       25   | 3.3e-09 |      20 |
 | Sereal::encode_sereal         |   2600000 |     0.39  |      170   | 1.7e-09 |      20 |
 | Data::MessagePack::pack       |   4110000 |     0.243 |      269   | 1.8e-10 |      20 |
 | JSON::XS::encode_json         |   4980000 |     0.201 |      326   | 1.7e-10 |      20 |
 | JSON::MaybeXS::encode_json    |   5050000 |     0.198 |      331   |   1e-10 |      20 |
 | Cpanel::JSON::XS::encode_json |   5080000 |     0.197 |      333   | 9.6e-11 |      36 |
 | JSON::Create::create_json     |   5200000 |     0.192 |      341   | 4.5e-11 |      24 |
 +-------------------------------+-----------+-----------+------------+---------+---------+


Benchmark deserializing (C<< bencher -m Serializers --include-participant-tags deserialize >>):

 #table12#
 {dataset=>"json:array_int_10"}
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | participant                        | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | JSON::Decode::Marpa::from_json     |       970 |  1000     |        1   | 4.1e-06 |      20 |
 | Pegex::JSON                        |      1300 |   800     |        1.3 | 1.1e-06 |      20 |
 | JSON::PP::decode_json              |     18000 |    55.4   |       18.5 | 2.2e-08 |      29 |
 | JSON::Tiny::decode_json            |     26000 |    38     |       27   | 5.3e-08 |      20 |
 | JSON::Decode::Regexp::from_json    |     30900 |    32.4   |       31.7 | 1.3e-08 |      20 |
 | PERLANCAR::JSON::Match::match_json |     66000 |    15     |       67   | 2.9e-08 |      26 |
 | JSON::MaybeXS::decode_json         |   1300000 |     0.74  |     1400   | 1.3e-09 |      20 |
 | Cpanel::JSON::XS::decode_json      |   1400000 |     0.74  |     1400   | 1.2e-09 |      20 |
 | JSON::XS::decode_json              |   1360000 |     0.736 |     1390   | 4.2e-10 |      20 |
 | JSON::Parse::parse_json            |   1400000 |     0.71  |     1400   | 8.3e-10 |      20 |
 +------------------------------------+-----------+-----------+------------+---------+---------+

 #table13#
 {dataset=>"json:array_int_100"}
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | participant                        | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | Pegex::JSON                        |       260 |   3.9     |       1    | 5.7e-06 |      20 |
 | JSON::Decode::Marpa::from_json     |       331 |   3.02    |       1.27 | 2.7e-06 |      20 |
 | JSON::PP::decode_json              |      1970 |   0.507   |       7.6  | 2.1e-07 |      20 |
 | JSON::Decode::Regexp::from_json    |      3900 |   0.26    |      15    | 4.7e-07 |      21 |
 | JSON::Tiny::decode_json            |      4000 |   0.25    |      16    | 5.9e-07 |      20 |
 | PERLANCAR::JSON::Match::match_json |      7900 |   0.13    |      30    | 4.8e-07 |      20 |
 | Cpanel::JSON::XS::decode_json      |    256000 |   0.00391 |     986    | 1.6e-09 |      22 |
 | JSON::MaybeXS::decode_json         |    260000 |   0.0039  |     990    | 6.7e-09 |      20 |
 | JSON::Parse::parse_json            |    257000 |   0.00389 |     989    | 1.7e-09 |      20 |
 | JSON::XS::decode_json              |    270000 |   0.0037  |    1000    | 4.9e-09 |      21 |
 +------------------------------------+-----------+-----------+------------+---------+---------+

 #table14#
 {dataset=>"json:array_int_1000"}
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | participant                        | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | Pegex::JSON                        |        28 | 36        |        1   | 5.3e-05 |      20 |
 | JSON::Decode::Marpa::from_json     |        40 | 25        |        1.4 | 6.3e-05 |      20 |
 | JSON::PP::decode_json              |       180 |  5.5      |        6.5 | 1.1e-05 |      21 |
 | JSON::Decode::Regexp::from_json    |       390 |  2.5      |       14   | 1.2e-05 |      21 |
 | JSON::Tiny::decode_json            |       450 |  2.2      |       16   | 3.8e-06 |      20 |
 | PERLANCAR::JSON::Match::match_json |       800 |  1.3      |       29   | 1.6e-06 |      20 |
 | JSON::Parse::parse_json            |     31000 |  0.032    |     1100   | 5.3e-08 |      20 |
 | JSON::MaybeXS::decode_json         |     31204 |  0.032047 |     1121.5 | 4.6e-11 |      22 |
 | JSON::XS::decode_json              |     32000 |  0.032    |     1100   | 5.3e-08 |      20 |
 | Cpanel::JSON::XS::decode_json      |     32000 |  0.031    |     1200   | 5.3e-08 |      20 |
 +------------------------------------+-----------+-----------+------------+---------+---------+

 #table15#
 {dataset=>"json:array_str1k_10"}
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | participant                        | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | JSON::Decode::Marpa::from_json     |      94.2 |   10.6    |        1   |   1e-05 |      20 |
 | JSON::PP::decode_json              |     130   |    7.4    |        1.4 | 1.3e-05 |      20 |
 | Pegex::JSON                        |     620   |    1.6    |        6.6 | 2.1e-06 |      20 |
 | JSON::Tiny::decode_json            |    1400   |    0.73   |       14   | 8.5e-07 |      20 |
 | JSON::Decode::Regexp::from_json    |    4700   |    0.21   |       50   | 2.5e-07 |      22 |
 | PERLANCAR::JSON::Match::match_json |   15000   |    0.068  |      160   | 1.1e-07 |      20 |
 | JSON::Parse::parse_json            |   35700   |    0.028  |      379   | 1.1e-08 |      30 |
 | Cpanel::JSON::XS::decode_json      |   51000   |    0.0196 |      542   | 5.6e-09 |      28 |
 | JSON::MaybeXS::decode_json         |   51000   |    0.02   |      540   | 2.5e-08 |      22 |
 | JSON::XS::decode_json              |   66700   |    0.015  |      708   | 5.2e-09 |      33 |
 +------------------------------------+-----------+-----------+------------+---------+---------+

 #table16#
 {dataset=>"json:hash_int_10"}
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | participant                        | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | JSON::Decode::Marpa::from_json     |       760 |   1300    |        1   | 1.8e-06 |      21 |
 | Pegex::JSON                        |       930 |   1100    |        1.2 | 1.5e-06 |      20 |
 | JSON::Decode::Regexp::from_json    |      7100 |    140    |        9.4 |   2e-07 |      22 |
 | JSON::PP::decode_json              |      9100 |    110    |       12   | 1.4e-07 |      25 |
 | JSON::Tiny::decode_json            |     14000 |     71    |       18   | 7.8e-08 |      21 |
 | PERLANCAR::JSON::Match::match_json |     42900 |     23.3  |       56.3 |   2e-08 |      20 |
 | Cpanel::JSON::XS::decode_json      |    760000 |      1.3  |     1000   | 1.7e-09 |      20 |
 | JSON::XS::decode_json              |    780000 |      1.3  |     1000   | 1.7e-09 |      20 |
 | JSON::MaybeXS::decode_json         |    780000 |      1.3  |     1000   | 1.7e-09 |      20 |
 | JSON::Parse::parse_json            |    845000 |      1.18 |     1110   | 4.1e-10 |      21 |
 +------------------------------------+-----------+-----------+------------+---------+---------+

 #table17#
 {dataset=>"json:hash_int_100"}
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | participant                        | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | Pegex::JSON                        |       151 |     6.62  |        1   | 5.4e-06 |      20 |
 | JSON::Decode::Marpa::from_json     |       170 |     6     |        1.1 | 6.1e-06 |      20 |
 | JSON::Decode::Regexp::from_json    |       760 |     1.3   |        5   | 3.1e-06 |      20 |
 | JSON::PP::decode_json              |       950 |     1.1   |        6.3 | 1.1e-06 |      21 |
 | JSON::Tiny::decode_json            |      1700 |     0.59  |       11   | 1.5e-06 |      20 |
 | PERLANCAR::JSON::Match::match_json |      4800 |     0.21  |       32   | 1.1e-06 |      20 |
 | JSON::Parse::parse_json            |     88000 |     0.011 |      580   | 1.3e-08 |      21 |
 | Cpanel::JSON::XS::decode_json      |     88000 |     0.011 |      580   | 1.3e-08 |      20 |
 | JSON::MaybeXS::decode_json         |     89000 |     0.011 |      590   | 2.7e-08 |      20 |
 | JSON::XS::decode_json              |     91000 |     0.011 |      600   | 1.3e-08 |      20 |
 +------------------------------------+-----------+-----------+------------+---------+---------+

 #table18#
 {dataset=>"json:hash_int_1000"}
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | participant                        | rate (/s) | time (ms) | vs_slowest |  errors | samples |
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | Pegex::JSON                        |      16   |     62.6  |       1    | 6.2e-05 |      20 |
 | JSON::Decode::Marpa::from_json     |      18.5 |     54.2  |       1.15 | 3.8e-05 |      20 |
 | JSON::Decode::Regexp::from_json    |      72   |     14    |       4.5  | 2.1e-05 |      20 |
 | JSON::PP::decode_json              |      88   |     11.4  |       5.5  | 4.7e-06 |      20 |
 | JSON::Tiny::decode_json            |     173   |      5.77 |      10.9  | 3.4e-06 |      20 |
 | PERLANCAR::JSON::Match::match_json |     440   |      2.3  |      28    | 1.1e-05 |      20 |
 | JSON::Parse::parse_json            |    6700   |      0.15 |     420    | 2.7e-07 |      20 |
 | Cpanel::JSON::XS::decode_json      |    6700   |      0.15 |     420    | 2.1e-07 |      20 |
 | JSON::MaybeXS::decode_json         |    6900   |      0.15 |     430    | 1.6e-07 |      20 |
 | JSON::XS::decode_json              |    7000   |      0.14 |     440    | 2.4e-07 |      24 |
 +------------------------------------+-----------+-----------+------------+---------+---------+

 #table19#
 {dataset=>"json:null"}
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | participant                        | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | Pegex::JSON                        |      2200 |   450     |          1 | 6.8e-07 |      21 |
 | JSON::Tiny::decode_json            |     98000 |    10     |         44 | 1.2e-08 |      23 |
 | JSON::PP::decode_json              |    130000 |     7.9   |         56 | 1.3e-08 |      20 |
 | JSON::Decode::Regexp::from_json    |    251000 |     3.98  |        112 | 1.6e-09 |      21 |
 | PERLANCAR::JSON::Match::match_json |    430000 |     2.3   |        190 | 3.3e-09 |      20 |
 | JSON::MaybeXS::decode_json         |   6000000 |     0.17  |       2700 | 3.1e-10 |      20 |
 | Cpanel::JSON::XS::decode_json      |   6080000 |     0.164 |       2710 | 4.6e-11 |      20 |
 | JSON::XS::decode_json              |   6230000 |     0.161 |       2770 | 4.6e-11 |      20 |
 | JSON::Parse::parse_json            |   7800000 |     0.13  |       3500 | 4.1e-10 |      21 |
 +------------------------------------+-----------+-----------+------------+---------+---------+

 #table20#
 {dataset=>"json:num"}
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | participant                        | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | Pegex::JSON                        |      2320 |    430    |        1   | 4.3e-07 |      20 |
 | JSON::PP::decode_json              |     69000 |     14    |       30   |   2e-08 |      20 |
 | JSON::Tiny::decode_json            |     72800 |     13.7  |       31.3 | 6.7e-09 |      20 |
 | JSON::Decode::Regexp::from_json    |    155000 |      6.47 |       66.6 | 3.2e-09 |      22 |
 | PERLANCAR::JSON::Match::match_json |    250000 |      3.9  |      110   | 6.7e-09 |      20 |
 | JSON::MaybeXS::decode_json         |   1000000 |      0.96 |      450   | 1.7e-09 |      20 |
 | JSON::Parse::parse_json            |   1000000 |      0.96 |      450   | 1.7e-09 |      20 |
 | Cpanel::JSON::XS::decode_json      |   1000000 |      0.96 |      450   | 1.7e-09 |      20 |
 | JSON::XS::decode_json              |   1100000 |      0.95 |      460   | 3.3e-09 |      20 |
 +------------------------------------+-----------+-----------+------------+---------+---------+

 #table21#
 {dataset=>"json:str1k"}
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | participant                        | rate (/s) | time (μs) | vs_slowest |  errors | samples |
 +------------------------------------+-----------+-----------+------------+---------+---------+
 | JSON::PP::decode_json              |      1320 |   760     |       1    | 2.7e-07 |      20 |
 | Pegex::JSON                        |      2000 |   500     |       1.52 | 2.7e-07 |      20 |
 | JSON::Tiny::decode_json            |     12200 |    82.2   |       9.25 | 2.7e-08 |      20 |
 | JSON::Decode::Regexp::from_json    |     42000 |    24     |      32    | 3.3e-08 |      20 |
 | PERLANCAR::JSON::Match::match_json |    120000 |     8.1   |      94    |   1e-08 |      20 |
 | JSON::Parse::parse_json            |    356600 |     2.805 |     271    | 4.6e-11 |      22 |
 | JSON::MaybeXS::decode_json         |    509000 |     1.96  |     387    | 8.3e-10 |      20 |
 | Cpanel::JSON::XS::decode_json      |    525000 |     1.905 |     399.1  | 4.6e-11 |      24 |
 | JSON::XS::decode_json              |    630000 |     1.6   |     480    | 4.1e-09 |      21 |
 +------------------------------------+-----------+-----------+------------+---------+---------+


Benchmark module startup overhead (C<< bencher -m Serializers --module-startup >>):

 #table22#
 +----------------------+-----------+------------------------+------------+-----------+---------+
 | participant          | time (ms) | mod_overhead_time (ms) | vs_slowest |  errors   | samples |
 +----------------------+-----------+------------------------+------------+-----------+---------+
 | JSON::Decode::Marpa  |     130   |                  124.4 |        1   |   0.00087 |      20 |
 | YAML::XS             |      60   |                   54.4 |        2   |   0.0018  |      20 |
 | JSON::PP             |      25   |                   19.4 |        5.4 | 4.2e-05   |      20 |
 | Pegex::JSON          |      24   |                   18.4 |        5.7 |   0.00016 |      20 |
 | JSON::Tiny           |      21   |                   15.4 |        6.3 | 9.9e-05   |      20 |
 | JSON::MaybeXS        |      17   |                   11.4 |        7.7 | 3.8e-05   |      21 |
 | Storable             |      16   |                   10.4 |        8.4 | 8.4e-05   |      20 |
 | Sereal               |      15   |                    9.4 |        8.7 | 6.3e-05   |      21 |
 | YAML::Old            |      15   |                    9.4 |        9.2 |   6e-05   |      20 |
 | YAML::Syck           |      13   |                    7.4 |       10   |   0.00012 |      20 |
 | Cpanel::JSON::XS     |      10   |                    4.4 |       10   |   0.00013 |      20 |
 | JSON::XS             |      13   |                    7.4 |       11   |   7e-05   |      20 |
 | JSON::Parse          |      12   |                    6.4 |       11   | 5.1e-05   |      20 |
 | JSON::Create         |      10   |                    4.4 |       10   |   0.00019 |      20 |
 | JSON::Decode::Regexp |       9.1 |                    3.5 |       15   | 3.3e-05   |      20 |
 | Data::MessagePack    |       8.8 |                    3.2 |       15   | 2.7e-05   |      20 |
 | perl -e1 (baseline)  |       5.6 |                    0   |       24   |   4e-05   |      20 |
 +----------------------+-----------+------------------------+------------+-----------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-Serializers>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-Serializers>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-Serializers>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
