#
#  Copyright 2009-2013 MongoDB, Inc.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#  http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

package MongoDB::Collection;


# ABSTRACT: A MongoDB Collection

use version;
our $VERSION = 'v0.999.998.2'; # TRIAL

use MongoDB::Error;
use MongoDB::WriteConcern;
use MongoDB::_Query;
use MongoDB::Op::_CreateIndexes;
use MongoDB::Op::_Delete;
use MongoDB::Op::_InsertOne;
use MongoDB::Op::_InsertMany;
use MongoDB::Op::_ListIndexes;
use MongoDB::Op::_Update;
use Tie::IxHash;
use Carp 'carp';
use boolean;
use Safe::Isa;
use Scalar::Util qw/blessed reftype/;
use Syntax::Keyword::Junction qw/any/;
use Try::Tiny;
use Moose;
use namespace::clean -except => 'meta';

has _database => (
    is       => 'ro',
    isa      => 'MongoDB::Database',
    required => 1,
);

has _client => (
    is      => 'ro',
    isa     => 'MongoDB::MongoClient',
    lazy    => 1,
    builder => '_build__client',
);

#pod =attr name
#pod
#pod The name of the collection.
#pod
#pod =cut

has name => (
    is       => 'ro',
    isa      => 'Str',
    required => 1,
);

#pod =attr full_name
#pod
#pod The full_name of the collection, including the namespace of the database it's
#pod in.
#pod
#pod =cut

has full_name => (
    is      => 'ro',
    isa     => 'Str',
    lazy    => 1,
    builder => '_build_full_name',
);

#pod =attr read_preference
#pod
#pod A L<MongoDB::ReadPreference> object.  It may be initialized with a string
#pod corresponding to one of the valid read preference modes or a hash reference
#pod that will be coerced into a new MongoDB::ReadPreference object.
#pod
#pod =cut

has read_preference => (
    is       => 'ro',
    isa      => 'ReadPreference',
    required => 1,
    coerce   => 1,
);

#pod =attr write_concern
#pod
#pod A L<MongoDB::WriteConcern> object.  It may be initialized with a hash
#pod reference that will be coerced into a new MongoDB::WriteConcern object.
#pod
#pod =cut

has write_concern => (
    is       => 'ro',
    isa      => 'WriteConcern',
    required => 1,
    coerce   => 1,
);

sub _build__client {
    my ($self) = @_;
    return $self->_database->_client;
}

sub _build_full_name {
    my ($self) = @_;
    my $name    = $self->name;
    my $db_name = $self->_database->name;
    return "${db_name}.${name}";
}


#pod =method clone
#pod
#pod     $coll2 = $coll1->clone( write_concern => { w => 2 } );
#pod
#pod Constructs a copy of the original collection, but allows changing
#pod attributes in the copy.
#pod
#pod =cut

sub clone {
    my ($self, @args) = @_;
    my $class = ref($self);
    if ( @args == 1 && ref( $args[0] ) eq 'HASH' ) {
        return class->new( %$self, %{$args[0]} );
    }

    return $class->new( %$self, @args );
}

#pod =method get_collection ($name)
#pod
#pod     my $collection = $database->get_collection('foo');
#pod
#pod Returns a L<MongoDB::Collection> for the collection called C<$name> within this
#pod collection.
#pod
#pod =cut

#pod =method get_collection
#pod
#pod Collection names can be chained together to access subcollections.  For
#pod instance, the collection C<foo.bar> can be accessed with either:
#pod
#pod     my $collection = $db->get_collection( 'foo' )->get_collection( 'bar' );
#pod
#pod or
#pod
#pod     my $collection = $db->get_collection( 'foo.bar' );
#pod
#pod =cut

sub get_collection {
    my $self = shift @_;
    my $coll = shift @_;

    return $self->_database->get_collection($self->name.'.'.$coll);
}

# utility function to generate an index name by concatenating key/value pairs
sub __to_index_string {
    my $keys = shift;

    my @name;
    if (ref $keys eq 'ARRAY') {
        @name = @$keys;
    }
    elsif (ref $keys eq 'HASH' ) {
        @name = %$keys
    }
    elsif (ref $keys eq 'Tie::IxHash') {
        my @ks = $keys->Keys;
        my @vs = $keys->Values;

        for (my $i=0; $i<$keys->Length; $i++) {
            push @name, $ks[$i];
            push @name, $vs[$i];
        }
    }
    else {
        confess 'expected Tie::IxHash, hash, or array reference for keys';
    }

    return join("_", @name);
}

#pod =method find, query
#pod
#pod     my $cursor = $coll->find( $filter );
#pod     my $cursor = $coll->find( $filter, $options );
#pod
#pod     my $cursor = $collection->find({ i => { '$gt' => 42 } }, {limit => 20});
#pod
#pod Executes a query with the given C<$filter> and returns a C<MongoDB::Cursor> with the results.
#pod C<$filter> can be a hash reference, L<Tie::IxHash>, or array reference (with an
#pod even number of elements).
#pod
#pod The query can be customized using L<MongoDB::Cursor> methods, or with an optional
#pod hash reference of options.
#pod
#pod Valid options include:
#pod
#pod =for :list
#pod * allowPartialResults - get partial results from a mongos if some shards are
#pod   down (instead of throwing an error).
#pod * batchSize – the number of documents to return per batch.
#pod * comment – attaches a comment to the query. If C<$comment> also exists in the
#pod   modifiers document, the comment field overwrites C<$comment>.
#pod * cursorType – indicates the type of cursor to use. It must be one of three
#pod   enumerated values: C<non_tailable> (the default), C<tailable>, and
#pod   C<tailable_await>.
#pod * limit – the maximum number of documents to return.
#pod * maxTimeMS – the maximum amount of time to allow the query to run. If
#pod   C<$maxTimeMS> also exists in the modifiers document, the maxTimeMS field
#pod   overwrites C<$maxTimeMS>.
#pod * modifiers – a hash reference of meta-operators modifying the output or
#pod   behavior of a query.
#pod * noCursorTimeout – if true, prevents the server from timing out a cursor after
#pod   a period of inactivity
#pod * projection - a hash reference defining fields to return. See L<Limit fields
#pod   to
#pod   return|http://docs.mongodb.org/manual/tutorial/project-fields-from-query-results/>
#pod   in the MongoDB documentation for details.
#pod * skip – the number of documents to skip before returning.
#pod * sort – a L<Tie::IxHash> or array reference of key value pairs defining the
#pod   order in which to return matching documents. If C<$orderby> also exists
#pod    * in the modifiers document, the sort field overwrites C<$orderby>.
#pod
#pod See also core documentation on querying:
#pod L<http://docs.mongodb.org/manual/core/read/>.
#pod
#pod The C<query> method is a legacy alias for C<find>.
#pod
#pod =cut

sub find {
    my ( $self, $filter, $opts ) = @_;

    $opts ||= {};
    $opts->{sort} = delete $opts->{sort_by} if $opts->{sort_by};

    my $query = MongoDB::_Query->new(
        db_name         => $self->_database->name,
        coll_name       => $self->name,
        client          => $self->_client,
        read_preference => $self->read_preference,
        filter          => $filter,
        %$opts,
    );

    return MongoDB::Cursor->new( query => $query );
}

#pod =method find_one($query, $fields?, $options?)
#pod
#pod     my $object = $collection->find_one({ name => 'Resi' });
#pod     my $object = $collection->find_one({ name => 'Resi' }, { name => 1, age => 1});
#pod     my $object = $collection->find_one({ name => 'Resi' }, {}, {max_time_ms => 100});
#pod
#pod Executes the given C<$query> and returns the first object matching it.
#pod C<$query> can be a hash reference, L<Tie::IxHash>, or array reference (with an
#pod even number of elements).  If C<$fields> is specified, the resulting document
#pod will only include the fields given (and the C<_id> field) which can cut down on
#pod wire traffic. If C<$options> is specified, the cursor will be set with the contained options.
#pod
#pod =cut

sub find_one {
    my ($self, $query, $fields, $options) = @_;
    $query ||= {};
    $fields ||= {};
    $options ||= {};

    my $cursor = $self->find($query)->limit(-1)->fields($fields);

    for my $key (keys %$options) {

        if (!MongoDB::Cursor->can($key)) {
            confess("$key is not a known method in MongoDB::Cursor");
        }
        $cursor->$key($options->{$key});
    }

    return $cursor->next;
}

#pod =method insert ($object, $options?)
#pod
#pod     my $id1 = $coll->insert({ name => 'mongo', type => 'database' });
#pod     my $id2 = $coll->insert({ name => 'mongo', type => 'database' }, {safe => 1});
#pod
#pod Inserts the given C<$object> into the database and returns it's id
#pod value. C<$object> can be a hash reference, a reference to an array with an
#pod even number of elements, or a L<Tie::IxHash>.  The id is the C<_id> value
#pod specified in the data or a L<MongoDB::OID>.
#pod
#pod The optional C<$options> parameter can be used to specify if this is a safe
#pod insert.  A safe insert will check with the database if the insert succeeded and
#pod croak if it did not.  You can also check if the insert succeeded by doing an
#pod unsafe insert, then calling L<MongoDB::Database/"last_error($options?)">.
#pod
#pod See also core documentation on insert: L<http://docs.mongodb.org/manual/core/create/>.
#pod
#pod =cut

sub insert {
    my ( $self, $document, $opts ) = @_;

    unless ( $opts->{'no_ids'} ) {
        $self->_add_oids( [$document] );
    }

    my $op = MongoDB::Op::_InsertOne->new(
        db_name       => $self->_database->name,
        coll_name     => $self->name,
        document      => $document,
        write_concern => $self->_dynamic_write_concern($opts),
    );

    my $result = $self->_client->send_write_op($op);

    return $result->inserted_id;
}

sub _add_oids {
    my ($self, $docs) = @_;
    my @ids;

    for my $d ( @$docs ) {
        my $type = reftype($d);
        my $found_id;
        if (ref($d) eq 'Tie::IxHash') {
            $found_id = $d->FETCH('_id');
            unless ( defined $found_id ) {
                $d->Unshift( '_id', $found_id = MongoDB::OID->new );
            }
        }
        elsif ($type eq 'ARRAY') {
            # search for an _id or prepend one
            for my $i ( 0 .. (@$d/2 - 1) ) {
                if ( $d->[2*$i] eq '_id' ) {
                    $found_id = $d->[2*$i+1];
                    last;
                }
            }
            unless (defined $found_id) {
                unshift @$d, '_id', $found_id = MongoDB::OID->new;
            }
        }
        elsif ($type eq 'HASH') {
            # hash or IxHash
            $found_id = $d->{_id};
            unless ( defined $found_id ) {
                $found_id = MongoDB::OID->new;
                $d->{_id} = $found_id;
            }
        }
        else {
            $type = 'scalar' unless $type;
            Carp::croak("unhandled type $type")
        }
        push @ids, $found_id;
    }

    return \@ids;
}

#pod =method batch_insert (\@array, $options)
#pod
#pod     my @ids = $collection->batch_insert([{name => "Joe"}, {name => "Fred"}, {name => "Sam"}]);
#pod
#pod Inserts each of the documents in the array into the database and returns an
#pod array of their _id fields.
#pod
#pod The optional C<$options> parameter can be used to specify if this is a safe
#pod insert.  A safe insert will check with the database if the insert succeeded and
#pod croak if it did not. You can also check if the inserts succeeded by doing an
#pod unsafe batch insert, then calling L<MongoDB::Database/"last_error($options?)">.
#pod
#pod =cut

sub batch_insert {
    my ( $self, $documents, $opts ) = @_;

    confess 'not an array reference' unless ref $documents eq 'ARRAY';

    unless ( $opts->{'no_ids'} ) {
        $self->_add_oids($documents);
    }

    my $op = MongoDB::Op::_InsertMany->new(
        db_name       => $self->_database->name,
        coll_name     => $self->name,
        documents     => $documents,
        write_concern => $self->_dynamic_write_concern($opts),
    );

    my $result = $self->_client->send_write_op($op);

    return @{ $result->inserted_ids };
}

sub _legacy_index_insert {
    my ($self, $doc, $options) = @_;

    my $wc = $self->_dynamic_write_concern( $options );
    my $result = $self->_client->send_insert($self->full_name, $doc, $wc, undef, 0);

    $result->assert;

    return 1;
}

#pod =method update (\%criteria, \%object, \%options?)
#pod
#pod     $collection->update({'x' => 3}, {'$inc' => {'count' => -1} }, {"upsert" => 1, "multiple" => 1});
#pod
#pod Updates an existing C<$object> matching C<$criteria> in the database.
#pod
#pod Returns 1 unless the C<safe> option is set. If C<safe> is set, this will return
#pod a hash of information about the update, including number of documents updated
#pod (C<n>).  If C<safe> is set and the update fails, C<update> will croak. You can
#pod also check if the update succeeded by doing an unsafe update, then calling
#pod L<MongoDB::Database/"last_error($options?)">.
#pod
#pod C<update> can take a hash reference of options.  The options currently supported
#pod are:
#pod
#pod =over
#pod
#pod =item C<upsert>
#pod If no object matching C<$criteria> is found, C<$object> will be inserted.
#pod
#pod =item C<multiple|multi>
#pod All of the documents that match C<$criteria> will be updated, not just
#pod the first document found. (Only available with database version 1.1.3 and
#pod newer.)  An error will be throw if both C<multiple> and C<multi> exist
#pod and their boolean values differ.
#pod
#pod =item C<safe>
#pod If the update fails and safe is set, the update will croak.
#pod
#pod =back
#pod
#pod See also core documentation on update: L<http://docs.mongodb.org/manual/core/update/>.
#pod
#pod =cut

sub update {
    my ( $self, $query, $object, $opts ) = @_;

    if ( exists $opts->{multiple} ) {
        if ( exists( $opts->{multi} ) && !!$opts->{multi} ne !!$opts->{multiple} ) {
            MongoDB::Error->throw(
                "can't use conflicting values of 'multiple' and 'multi' in 'update'");
        }
        $opts->{multi} = delete $opts->{multiple};
    }

    my $op = MongoDB::Op::_Update->new(
        db_name       => $self->_database->name,
        coll_name     => $self->name,
        filter        => $query,
        update        => $object,
        multi         => $opts->{multi},
        upsert        => $opts->{upsert},
        write_concern => $self->_dynamic_write_concern($opts),
    );

    my $result = $self->_client->send_write_op( $op );

    # emulate key fields of legacy GLE result
    return {
        ok => 1,
        n => $result->matched_count,
        ( $result->upserted_id ? ( upserted => $result->upserted_id ) : () ),
    };
}

#pod =method find_and_modify
#pod
#pod     my $result = $collection->find_and_modify( { query => { ... }, update => { ... } } );
#pod
#pod Perform an atomic update. C<find_and_modify> guarantees that nothing else will come along
#pod and change the queried documents before the update is performed.
#pod
#pod Returns the old version of the document, unless C<new => 1> is specified. If no documents
#pod match the query, it returns nothing.
#pod
#pod =cut

sub find_and_modify {
    my ( $self, $opts ) = @_;

    my $conn = $self->_client;
    my $db   = $self->_database;

    my $result;
    try {
        $result = $db->run_command( [ findAndModify => $self->name, %$opts ] )
    }
    catch {
        die $_ unless $_ eq 'No matching object found';
    };

    return $result->{value} if $result;
    return;
}


#pod =method aggregate
#pod
#pod     my $result = $collection->aggregate( [ ... ] );
#pod
#pod Run a query using the MongoDB 2.2+ aggregation framework. The first argument is an array-ref of
#pod aggregation pipeline operators.
#pod
#pod The type of return value from C<aggregate> depends on how you use it.
#pod
#pod =over 4
#pod
#pod =item * By default, the aggregation framework returns a document with an embedded array of results, and
#pod the C<aggregate> method returns a reference to that array.
#pod
#pod =item * MongoDB 2.6+ supports returning cursors from aggregation queries, allowing you to bypass
#pod the 16MB size limit of documents. If you specifiy a C<cursor> option, the C<aggregate> method
#pod will return a L<MongoDB::QueryResult> object which can be iterated in the normal fashion.
#pod
#pod     my $cursor = $collection->aggregate( [ ... ], { cursor => 1 } );
#pod
#pod Specifying a C<cursor> option will cause an error on versions of MongoDB below 2.6.
#pod
#pod The C<cursor> option may also have some useful options of its own. Currently, the only one
#pod is C<batchSize>, which allows you to control how frequently the cursor must go back to the
#pod database for more documents.
#pod
#pod     my $cursor = $collection->aggregate( [ ... ], { cursor => { batchSize => 10 } } );
#pod
#pod =item * MongoDB 2.6+ supports an C<explain> option to aggregation queries to retrieve data
#pod about how the server will process a query pipeline.
#pod
#pod     my $result = $collection->aggregate( [ ... ], { explain => 1 } );
#pod
#pod In this case, C<aggregate> will return a document (not an array) containing the explanation
#pod structure.
#pod
#pod =item * Finally, MongoDB 2.6+ will return an empty results array if the C<$out> pipeline operator is used to
#pod write aggregation results directly to a collection. Create a new C<Collection> object to
#pod query the result collection.
#pod
#pod =back
#pod
#pod See L<Aggregation|http://docs.mongodb.org/manual/aggregation/> in the MongoDB manual
#pod for more information on how to construct aggregation queries.
#pod
#pod =cut

sub aggregate {
    my ( $self, $pipeline, $opts ) = @_;
    $opts = ref $opts eq 'HASH' ? $opts : { };

    my $db   = $self->_database;

    if ( exists $opts->{cursor} ) {
        $opts->{cursor} = { } unless ref $opts->{cursor} eq 'HASH';
    }

    # explain requires a boolean
    if ( exists $opts->{explain} ) {
        $opts->{explain} = $opts->{explain} ? true : false;
    }

    my @command = ( aggregate => $self->name, pipeline => $pipeline, %$opts );
    my ($last_op) = keys %{$pipeline->[-1]};
    my $read_pref = $last_op eq '$out' ? undef : $self->read_preference;

    my $op = MongoDB::Op::_Command->new(
        db_name => $db->name,
        query => \@command,
        ( $read_pref ? ( read_preference => $read_pref ) : () )
    );

    my $result = $self->_client->send_read_op( $op );
    my $response = $result->result;

    # if we got a cursor option then we need to construct a wonky cursor
    # object on our end and populate it with the first batch, since
    # commands can't actually return cursors.
    if ( exists $opts->{cursor} ) {
        unless ( exists $response->{cursor} ) {
            die "no cursor returned from aggregation";
        }

        my $qr = MongoDB::QueryResult->new(
            _client => $self->_client,
            address => $result->address,
            cursor  => $response->{cursor},
        );

        return $qr;
    }

    # return the whole response document if they want an explain
    if ( $opts->{explain} ) {
        return $response;
    }

    # TODO: handle errors?

    return $response->{result};
}

#pod =method parallel_scan($max_cursors)
#pod
#pod     my @query_results = $collection->parallel_scan(10);
#pod
#pod Scan the collection in parallel. The argument is the maximum number of
#pod L<MongoDB::QueryResult> objects to return and must be a positive integer between 1
#pod and 10,000.
#pod
#pod As long as the collection is not modified during scanning, each document will
#pod appear only once in one of the cursors' result sets.
#pod
#pod Only iteration methods may be called on parallel scan cursors.
#pod
#pod If an error occurs, an exception will be thrown.
#pod
#pod =cut

sub parallel_scan {
    my ( $self, $num_cursors, $opts ) = @_;
    unless (defined $num_cursors && $num_cursors == int($num_cursors)
        && $num_cursors > 0 && $num_cursors <= 10000
    ) {
        Carp::croak( "first argument to parallel_scan must be a positive integer between 1 and 10000" )
    }
    $opts = ref $opts eq 'HASH' ? $opts : { };

    my $db   = $self->_database;

    my @command = ( parallelCollectionScan => $self->name, numCursors => $num_cursors );

    my $op = MongoDB::Op::_Command->new(
        db_name         => $db->name,
        query           => \@command,
        read_preference => $self->read_preference,
    );

    my $result = $self->_client->send_read_op( $op );
    my $response = $result->result;

    Carp::croak("No cursors returned")
        unless $response->{cursors} && ref $response->{cursors} eq 'ARRAY';

    my @cursors;
    for my $c ( map { $_->{cursor} } @{$response->{cursors}} ) {
        my $qr = MongoDB::QueryResult->new(
            _client => $self->_client,
            address => $result->address,
            cursor  => $c,
        );
        push @cursors, $qr;
    }

    return @cursors;
}

#pod =method rename ("newcollectionname")
#pod
#pod     my $newcollection = $collection->rename("mynewcollection");
#pod
#pod Renames the collection.  It expects that the new name is currently not in use.
#pod
#pod Returns the new collection.  If a collection already exists with that new collection name this will
#pod die.
#pod
#pod =cut

sub rename {
    my ($self, $collectionname) = @_;

    my $conn = $self->_client;
    my $database = $conn->get_database( 'admin' );
    my $fullname = $self->full_name;

    my ($db, @collection_bits) = split(/\./, $fullname);
    my $collection = join('.', @collection_bits);
    my $obj = $database->run_command([ 'renameCollection' => "$db.$collection", 'to' => "$db.$collectionname" ]);

    return $conn->get_database( $db )->get_collection( $collectionname );
}

#pod =method remove ($query?, $options?)
#pod
#pod     $collection->remove({ answer => { '$ne' => 42 } });
#pod
#pod Removes all objects matching the given C<$query> from the database. If no
#pod parameters are given, removes all objects from the collection (but does not
#pod delete indexes, as C<MongoDB::Collection::drop> does).
#pod
#pod Returns 1 unless the C<safe> option is set.  If C<safe> is set and the remove
#pod succeeds, C<remove> will return a hash of information about the remove,
#pod including how many documents were removed (C<n>).  If the remove fails and
#pod C<safe> is set, C<remove> will croak.  You can also check if the remove
#pod succeeded by doing an unsafe remove, then calling
#pod L<MongoDB::Database/"last_error($options?)">.
#pod
#pod C<remove> can take a hash reference of options.  The options currently supported
#pod are
#pod
#pod =over
#pod
#pod =item C<just_one>
#pod Only one matching document to be removed.
#pod
#pod =item C<safe>
#pod If the update fails and safe is set, this function will croak.
#pod
#pod =back
#pod
#pod See also core documentation on remove: L<http://docs.mongodb.org/manual/core/delete/>.
#pod
#pod =cut


sub remove {
    my ($self, $query, $opts) = @_;
    confess "optional argument to remove must be a hash reference"
        if defined $opts && ref $opts ne 'HASH';

    my $op = MongoDB::Op::_Delete->new(
        db_name       => $self->_database->name,
        coll_name     => $self->name,
        filter        => $query,
        just_one      => !! $opts->{just_one},
        write_concern => $self->_dynamic_write_concern($opts),
    );

    my $result = $self->_client->send_write_op( $op );

    # emulate key fields of legacy GLE result
    return {
        ok => 1,
        n => $result->deleted_count,
    };
}

#pod =method ensure_index
#pod
#pod     $collection->ensure_index( $keys );
#pod     $collection->ensure_index( $keys, $options );
#pod     $collection->ensure_index(["foo" => 1, "bar" => -1], { unique => 1 });
#pod
#pod Makes sure the given C<$keys> of this collection are indexed. C<$keys> can be
#pod an array reference, hash reference, or C<Tie::IxHash>.  Array references or
#pod C<Tie::IxHash> is preferred for multi-key indexes, so that the keys are in the
#pod correct order.  1 creates an ascending index, -1 creates a descending index.
#pod
#pod If an optional C<$options> argument is provided, those options are passed
#pod through to the database to modify index creation.  Typical options include:
#pod
#pod =for :list
#pod * background – build the index in the background
#pod * name – a name for the index; one will be generated if not provided
#pod * unique – if true, inserting duplicates will fail
#pod
#pod See the MongoDB L<index documentation|http://docs.mongodb.org/manual/indexes/>
#pod for more information on indexing and index options.
#pod
#pod Returns true on success and throws an exception on failure.
#pod
#pod Note: index creation can take longer than the network timeout, resulting
#pod in an exception.  If this is a concern, consider setting the C<background>
#pod option.
#pod
#pod =cut

sub ensure_index {
    my ( $self, $keys, $opts ) = @_;
    MongoDB::Error->throw("ensure_index options must be a hash reference")
      if $opts && !ref($opts) eq 'HASH';

    $keys = Tie::IxHash->new(@$keys) if ref $keys eq 'ARRAY';
    $opts = $self->_clean_index_options( $opts, $keys );

    # always use safe write concern for index creation
    my $wc =
        $self->write_concern->is_safe
      ? $self->write_concern
      : MongoDB::WriteConcern->new;

    my $op = MongoDB::Op::_CreateIndexes->new(
        db_name       => $self->_database->name,
        coll_name     => $self->name,
        indexes       => [ { key => $keys, %$opts } ],
        write_concern => $wc,
    );

    $self->_client->send_write_op($op);

    return 1;
}

#pod =method save($doc, $options)
#pod
#pod     $collection->save({"author" => "joe"});
#pod     my $post = $collection->find_one;
#pod
#pod     $post->{author} = {"name" => "joe", "id" => 123, "phone" => "555-5555"};
#pod
#pod     $collection->save( $post );
#pod     $collection->save( $post, { safe => 1 } )
#pod
#pod Inserts a document into the database if it does not have an _id field, upserts
#pod it if it does have an _id field.
#pod
#pod The return types for this function are a bit of a mess, as it will return the
#pod _id if a new document was inserted, 1 if an upsert occurred, and croak if the
#pod safe option was set and an error occurred.  You can also check if the save
#pod succeeded by doing an unsafe save, then calling
#pod L<MongoDB::Database/"last_error($options?)">.
#pod
#pod =cut

sub save {
    my ($self, $doc, $options) = @_;

    if (exists $doc->{"_id"}) {

        if (!$options || !ref $options eq 'HASH') {
            $options = {"upsert" => boolean::true};
        }
        else {
            $options->{'upsert'} = boolean::true;
        }

        return $self->update({"_id" => $doc->{"_id"}}, $doc, $options);
    }
    else {
        return $self->insert($doc, $options);
    }
}


#pod =method count($query?)
#pod
#pod     my $n_objects = $collection->count({ name => 'Bob' });
#pod
#pod Counts the number of objects in this collection that match the given C<$query>.
#pod If no query is given, the total number of objects in the collection is returned.
#pod
#pod =cut

sub count {
    my ($self, $query, $options) = @_;
    $query ||= {};
    $options ||= {};

    my $cursor = $self->find($query);

    for my $key (keys %$options) {

        if (!MongoDB::Cursor->can($key)) {
            confess("$key is not a known method in MongoDB::Cursor");
        }
        $cursor->$key($options->{$key});
    }

    return $cursor->count;
}


#pod =method validate
#pod
#pod     $collection->validate;
#pod
#pod Asks the server to validate this collection.
#pod Returns a hash of the form:
#pod
#pod     {
#pod         'ok' => '1',
#pod         'ns' => 'foo.bar',
#pod         'result' => info
#pod     }
#pod
#pod where C<info> is a string of information
#pod about the collection.
#pod
#pod =cut

sub validate {
    my ($self, $scan_data) = @_;
    $scan_data = 0 unless defined $scan_data;
    my $obj = $self->_database->run_command({ validate => $self->name });
}


#pod =method drop_indexes
#pod
#pod     $collection->drop_indexes;
#pod
#pod Removes all indexes from this collection.
#pod
#pod =cut

sub drop_indexes {
    my ($self) = @_;
    return $self->drop_index('*');
}

#pod =method drop_index ($index_name)
#pod
#pod     $collection->drop_index('foo_1');
#pod
#pod Removes an index called C<$index_name> from this collection.
#pod Use C<MongoDB::Collection::get_indexes> to find the index name.
#pod
#pod =cut

sub drop_index {
    my ($self, $index_name) = @_;
    return $self->_database->run_command([
        dropIndexes => $self->name,
        index => $index_name,
    ]);
}

#pod =method get_indexes
#pod
#pod     my @indexes = $collection->get_indexes;
#pod
#pod Returns a list of all indexes of this collection.
#pod Each index contains C<ns>, C<name>, and C<key>
#pod fields of the form:
#pod
#pod     {
#pod         'ns' => 'db_name.collection_name',
#pod         'name' => 'index_name',
#pod         'key' => {
#pod             'key1' => dir1,
#pod             'key2' => dir2,
#pod             ...
#pod             'keyN' => dirN
#pod         }
#pod     }
#pod
#pod where C<dirX> is 1 or -1, depending on if the
#pod index is ascending or descending on that key.
#pod
#pod =cut

sub get_indexes {
    my ($self) = @_;

    my $op = MongoDB::Op::_ListIndexes->new(
        db_name    => $self->_database->name,
        coll_name  => $self->name,
        client     => $self->_client,
        bson_codec => $self->_client,
    );

    my $res = $self->_client->send_read_op($op);

    return $res->all;
}

#pod =method drop
#pod
#pod     $collection->drop;
#pod
#pod Deletes a collection as well as all of its indexes.
#pod
#pod =cut

sub drop {
    my ($self) = @_;
    try {
        $self->_database->run_command({ drop => $self->name });
    }
    catch {
        die $_ unless /ns not found/;
    };
    return;
}

#pod =method initialize_ordered_bulk_op, ordered_bulk
#pod
#pod     my $bulk = $collection->initialize_ordered_bulk_op;
#pod     $bulk->insert( $doc1 );
#pod     $bulk->insert( $doc2 );
#pod     ...
#pod     my $result = $bulk->execute;
#pod
#pod Returns a L<MongoDB::BulkWrite> object to group write operations into fewer network
#pod round-trips.  This method creates an B<ordered> operation, where operations halt after
#pod the first error. See L<MongoDB::BulkWrite> for more details.
#pod
#pod The method C<ordered_bulk> may be used as an alias for C<initialize_ordered_bulk_op>.
#pod
#pod =cut

sub initialize_ordered_bulk_op {
    my ($self) = @_;
    return MongoDB::BulkWrite->new( collection => $self, ordered => 1 );
}

#pod =method initialize_unordered_bulk_op, unordered_bulk
#pod
#pod This method works just like L</initialize_ordered_bulk_op> except that the order that
#pod operations are sent to the database is not guaranteed and errors do not halt processing.
#pod See L<MongoDB::BulkWrite> for more details.
#pod
#pod The method C<unordered_bulk> may be used as an alias for C<initialize_unordered_bulk_op>.
#pod
#pod =cut

sub initialize_unordered_bulk_op {
    my ($self) = @_;
    return MongoDB::BulkWrite->new( collection => $self, ordered => 0 );
}

sub _dynamic_write_concern {
    my ( $self, $opts ) = @_;
    if ( !exists( $opts->{safe} ) || $opts->{safe} ) {
        return $self->write_concern;
    }
    else {
        return MongoDB::WriteConcern->new( w => 0 );
    }
}

# old API allowed some snake_case options; some options must
# be turned into booleans
sub _clean_index_options {
    my ( $self, $orig, $keys ) = @_;

    # copy the original so we don't modify it
    my $opts = { $orig ? %$orig : () };

    # add name if not provided
    $opts->{name} = __to_index_string($keys)
      unless defined $opts->{name};

    # safe is no more
    delete $opts->{safe} if exists $opts->{safe};

    # convert snake case
    if ( exists $opts->{drop_dups} ) {
        $opts->{dropDups} = delete $opts->{drop_dups};
    }

    # convert snake case and turn into an integer
    if ( exists $opts->{expire_after_seconds} ) {
        $opts->{expireAfterSeconds} = int( delete $opts->{expire_after_seconds} );
    }

    # convert some things to booleans
    for my $k (qw/unique background sparse dropDups/) {
        next unless exists $opts->{$k};
        $opts->{$k} = boolean( $opts->{$k} );
    }

    return $opts;
}

BEGIN {
    # aliases
    no warnings 'once';
    *query = \&find;
    *ordered_bulk = \&initialize_ordered_bulk_op;
    *unordered_bulk = \&initialize_unordered_bulk_op;
}

__PACKAGE__->meta->make_immutable;

1;

=pod

=encoding UTF-8

=head1 NAME

MongoDB::Collection - A MongoDB Collection

=head1 VERSION

version v0.999.998.2

=head1 SYNOPSIS

    # get a Collection via the Database object
    my $coll = $db->get_collection("people");

    # insert a document
    $coll->insert( { name => "John Doe", age => 42 } );

    # find a single document
    my $doc = $coll->find_one( { name => "John Doe" } )

    # Get a MongoDB::Cursor for a query
    my $cursor = $coll->find( { age => 42 } );

=head1 DESCRIPTION

This class models a MongoDB collection and provides an API for interacting
with it.

Generally, you never construct one of these directly with C<new>.  Instead, you
call C<get_collection> on a L<MongoDB::Database> object.

=head1 ATTRIBUTES

=head2 name

The name of the collection.

=head2 full_name

The full_name of the collection, including the namespace of the database it's
in.

=head2 read_preference

A L<MongoDB::ReadPreference> object.  It may be initialized with a string
corresponding to one of the valid read preference modes or a hash reference
that will be coerced into a new MongoDB::ReadPreference object.

=head2 write_concern

A L<MongoDB::WriteConcern> object.  It may be initialized with a hash
reference that will be coerced into a new MongoDB::WriteConcern object.

=head1 METHODS

=head2 clone

    $coll2 = $coll1->clone( write_concern => { w => 2 } );

Constructs a copy of the original collection, but allows changing
attributes in the copy.

=head2 get_collection ($name)

    my $collection = $database->get_collection('foo');

Returns a L<MongoDB::Collection> for the collection called C<$name> within this
collection.

=head2 get_collection

Collection names can be chained together to access subcollections.  For
instance, the collection C<foo.bar> can be accessed with either:

    my $collection = $db->get_collection( 'foo' )->get_collection( 'bar' );

or

    my $collection = $db->get_collection( 'foo.bar' );

=head2 find, query

    my $cursor = $coll->find( $filter );
    my $cursor = $coll->find( $filter, $options );

    my $cursor = $collection->find({ i => { '$gt' => 42 } }, {limit => 20});

Executes a query with the given C<$filter> and returns a C<MongoDB::Cursor> with the results.
C<$filter> can be a hash reference, L<Tie::IxHash>, or array reference (with an
even number of elements).

The query can be customized using L<MongoDB::Cursor> methods, or with an optional
hash reference of options.

Valid options include:

=over 4

=item *

allowPartialResults - get partial results from a mongos if some shards are down (instead of throwing an error).

=item *

batchSize – the number of documents to return per batch.

=item *

comment – attaches a comment to the query. If C<$comment> also exists in the modifiers document, the comment field overwrites C<$comment>.

=item *

cursorType – indicates the type of cursor to use. It must be one of three enumerated values: C<non_tailable> (the default), C<tailable>, and C<tailable_await>.

=item *

limit – the maximum number of documents to return.

=item *

maxTimeMS – the maximum amount of time to allow the query to run. If C<$maxTimeMS> also exists in the modifiers document, the maxTimeMS field overwrites C<$maxTimeMS>.

=item *

modifiers – a hash reference of meta-operators modifying the output or behavior of a query.

=item *

noCursorTimeout – if true, prevents the server from timing out a cursor after a period of inactivity

=item *

projection - a hash reference defining fields to return. See L<Limit fields to return|http://docs.mongodb.org/manual/tutorial/project-fields-from-query-results/> in the MongoDB documentation for details.

=item *

skip – the number of documents to skip before returning.

=item *

sort – a L<Tie::IxHash> or array reference of key value pairs defining the order in which to return matching documents. If C<$orderby> also exists * in the modifiers document, the sort field overwrites C<$orderby>.

=back

See also core documentation on querying:
L<http://docs.mongodb.org/manual/core/read/>.

The C<query> method is a legacy alias for C<find>.

=head2 find_one($query, $fields?, $options?)

    my $object = $collection->find_one({ name => 'Resi' });
    my $object = $collection->find_one({ name => 'Resi' }, { name => 1, age => 1});
    my $object = $collection->find_one({ name => 'Resi' }, {}, {max_time_ms => 100});

Executes the given C<$query> and returns the first object matching it.
C<$query> can be a hash reference, L<Tie::IxHash>, or array reference (with an
even number of elements).  If C<$fields> is specified, the resulting document
will only include the fields given (and the C<_id> field) which can cut down on
wire traffic. If C<$options> is specified, the cursor will be set with the contained options.

=head2 insert ($object, $options?)

    my $id1 = $coll->insert({ name => 'mongo', type => 'database' });
    my $id2 = $coll->insert({ name => 'mongo', type => 'database' }, {safe => 1});

Inserts the given C<$object> into the database and returns it's id
value. C<$object> can be a hash reference, a reference to an array with an
even number of elements, or a L<Tie::IxHash>.  The id is the C<_id> value
specified in the data or a L<MongoDB::OID>.

The optional C<$options> parameter can be used to specify if this is a safe
insert.  A safe insert will check with the database if the insert succeeded and
croak if it did not.  You can also check if the insert succeeded by doing an
unsafe insert, then calling L<MongoDB::Database/"last_error($options?)">.

See also core documentation on insert: L<http://docs.mongodb.org/manual/core/create/>.

=head2 batch_insert (\@array, $options)

    my @ids = $collection->batch_insert([{name => "Joe"}, {name => "Fred"}, {name => "Sam"}]);

Inserts each of the documents in the array into the database and returns an
array of their _id fields.

The optional C<$options> parameter can be used to specify if this is a safe
insert.  A safe insert will check with the database if the insert succeeded and
croak if it did not. You can also check if the inserts succeeded by doing an
unsafe batch insert, then calling L<MongoDB::Database/"last_error($options?)">.

=head2 update (\%criteria, \%object, \%options?)

    $collection->update({'x' => 3}, {'$inc' => {'count' => -1} }, {"upsert" => 1, "multiple" => 1});

Updates an existing C<$object> matching C<$criteria> in the database.

Returns 1 unless the C<safe> option is set. If C<safe> is set, this will return
a hash of information about the update, including number of documents updated
(C<n>).  If C<safe> is set and the update fails, C<update> will croak. You can
also check if the update succeeded by doing an unsafe update, then calling
L<MongoDB::Database/"last_error($options?)">.

C<update> can take a hash reference of options.  The options currently supported
are:

=over

=item C<upsert>
If no object matching C<$criteria> is found, C<$object> will be inserted.

=item C<multiple|multi>
All of the documents that match C<$criteria> will be updated, not just
the first document found. (Only available with database version 1.1.3 and
newer.)  An error will be throw if both C<multiple> and C<multi> exist
and their boolean values differ.

=item C<safe>
If the update fails and safe is set, the update will croak.

=back

See also core documentation on update: L<http://docs.mongodb.org/manual/core/update/>.

=head2 find_and_modify

    my $result = $collection->find_and_modify( { query => { ... }, update => { ... } } );

Perform an atomic update. C<find_and_modify> guarantees that nothing else will come along
and change the queried documents before the update is performed.

Returns the old version of the document, unless C<new => 1> is specified. If no documents
match the query, it returns nothing.

=head2 aggregate

    my $result = $collection->aggregate( [ ... ] );

Run a query using the MongoDB 2.2+ aggregation framework. The first argument is an array-ref of
aggregation pipeline operators.

The type of return value from C<aggregate> depends on how you use it.

=over 4

=item * By default, the aggregation framework returns a document with an embedded array of results, and
the C<aggregate> method returns a reference to that array.

=item * MongoDB 2.6+ supports returning cursors from aggregation queries, allowing you to bypass
the 16MB size limit of documents. If you specifiy a C<cursor> option, the C<aggregate> method
will return a L<MongoDB::QueryResult> object which can be iterated in the normal fashion.

    my $cursor = $collection->aggregate( [ ... ], { cursor => 1 } );

Specifying a C<cursor> option will cause an error on versions of MongoDB below 2.6.

The C<cursor> option may also have some useful options of its own. Currently, the only one
is C<batchSize>, which allows you to control how frequently the cursor must go back to the
database for more documents.

    my $cursor = $collection->aggregate( [ ... ], { cursor => { batchSize => 10 } } );

=item * MongoDB 2.6+ supports an C<explain> option to aggregation queries to retrieve data
about how the server will process a query pipeline.

    my $result = $collection->aggregate( [ ... ], { explain => 1 } );

In this case, C<aggregate> will return a document (not an array) containing the explanation
structure.

=item * Finally, MongoDB 2.6+ will return an empty results array if the C<$out> pipeline operator is used to
write aggregation results directly to a collection. Create a new C<Collection> object to
query the result collection.

=back

See L<Aggregation|http://docs.mongodb.org/manual/aggregation/> in the MongoDB manual
for more information on how to construct aggregation queries.

=head2 parallel_scan($max_cursors)

    my @query_results = $collection->parallel_scan(10);

Scan the collection in parallel. The argument is the maximum number of
L<MongoDB::QueryResult> objects to return and must be a positive integer between 1
and 10,000.

As long as the collection is not modified during scanning, each document will
appear only once in one of the cursors' result sets.

Only iteration methods may be called on parallel scan cursors.

If an error occurs, an exception will be thrown.

=head2 rename ("newcollectionname")

    my $newcollection = $collection->rename("mynewcollection");

Renames the collection.  It expects that the new name is currently not in use.

Returns the new collection.  If a collection already exists with that new collection name this will
die.

=head2 remove ($query?, $options?)

    $collection->remove({ answer => { '$ne' => 42 } });

Removes all objects matching the given C<$query> from the database. If no
parameters are given, removes all objects from the collection (but does not
delete indexes, as C<MongoDB::Collection::drop> does).

Returns 1 unless the C<safe> option is set.  If C<safe> is set and the remove
succeeds, C<remove> will return a hash of information about the remove,
including how many documents were removed (C<n>).  If the remove fails and
C<safe> is set, C<remove> will croak.  You can also check if the remove
succeeded by doing an unsafe remove, then calling
L<MongoDB::Database/"last_error($options?)">.

C<remove> can take a hash reference of options.  The options currently supported
are

=over

=item C<just_one>
Only one matching document to be removed.

=item C<safe>
If the update fails and safe is set, this function will croak.

=back

See also core documentation on remove: L<http://docs.mongodb.org/manual/core/delete/>.

=head2 ensure_index

    $collection->ensure_index( $keys );
    $collection->ensure_index( $keys, $options );
    $collection->ensure_index(["foo" => 1, "bar" => -1], { unique => 1 });

Makes sure the given C<$keys> of this collection are indexed. C<$keys> can be
an array reference, hash reference, or C<Tie::IxHash>.  Array references or
C<Tie::IxHash> is preferred for multi-key indexes, so that the keys are in the
correct order.  1 creates an ascending index, -1 creates a descending index.

If an optional C<$options> argument is provided, those options are passed
through to the database to modify index creation.  Typical options include:

=over 4

=item *

background – build the index in the background

=item *

name – a name for the index; one will be generated if not provided

=item *

unique – if true, inserting duplicates will fail

=back

See the MongoDB L<index documentation|http://docs.mongodb.org/manual/indexes/>
for more information on indexing and index options.

Returns true on success and throws an exception on failure.

Note: index creation can take longer than the network timeout, resulting
in an exception.  If this is a concern, consider setting the C<background>
option.

=head2 save($doc, $options)

    $collection->save({"author" => "joe"});
    my $post = $collection->find_one;

    $post->{author} = {"name" => "joe", "id" => 123, "phone" => "555-5555"};

    $collection->save( $post );
    $collection->save( $post, { safe => 1 } )

Inserts a document into the database if it does not have an _id field, upserts
it if it does have an _id field.

The return types for this function are a bit of a mess, as it will return the
_id if a new document was inserted, 1 if an upsert occurred, and croak if the
safe option was set and an error occurred.  You can also check if the save
succeeded by doing an unsafe save, then calling
L<MongoDB::Database/"last_error($options?)">.

=head2 count($query?)

    my $n_objects = $collection->count({ name => 'Bob' });

Counts the number of objects in this collection that match the given C<$query>.
If no query is given, the total number of objects in the collection is returned.

=head2 validate

    $collection->validate;

Asks the server to validate this collection.
Returns a hash of the form:

    {
        'ok' => '1',
        'ns' => 'foo.bar',
        'result' => info
    }

where C<info> is a string of information
about the collection.

=head2 drop_indexes

    $collection->drop_indexes;

Removes all indexes from this collection.

=head2 drop_index ($index_name)

    $collection->drop_index('foo_1');

Removes an index called C<$index_name> from this collection.
Use C<MongoDB::Collection::get_indexes> to find the index name.

=head2 get_indexes

    my @indexes = $collection->get_indexes;

Returns a list of all indexes of this collection.
Each index contains C<ns>, C<name>, and C<key>
fields of the form:

    {
        'ns' => 'db_name.collection_name',
        'name' => 'index_name',
        'key' => {
            'key1' => dir1,
            'key2' => dir2,
            ...
            'keyN' => dirN
        }
    }

where C<dirX> is 1 or -1, depending on if the
index is ascending or descending on that key.

=head2 drop

    $collection->drop;

Deletes a collection as well as all of its indexes.

=head2 initialize_ordered_bulk_op, ordered_bulk

    my $bulk = $collection->initialize_ordered_bulk_op;
    $bulk->insert( $doc1 );
    $bulk->insert( $doc2 );
    ...
    my $result = $bulk->execute;

Returns a L<MongoDB::BulkWrite> object to group write operations into fewer network
round-trips.  This method creates an B<ordered> operation, where operations halt after
the first error. See L<MongoDB::BulkWrite> for more details.

The method C<ordered_bulk> may be used as an alias for C<initialize_ordered_bulk_op>.

=head2 initialize_unordered_bulk_op, unordered_bulk

This method works just like L</initialize_ordered_bulk_op> except that the order that
operations are sent to the database is not guaranteed and errors do not halt processing.
See L<MongoDB::BulkWrite> for more details.

The method C<unordered_bulk> may be used as an alias for C<initialize_unordered_bulk_op>.

=head1 AUTHORS

=over 4

=item *

David Golden <david@mongodb.com>

=item *

Mike Friedman <friedo@mongodb.com>

=item *

Kristina Chodorow <kristina@mongodb.com>

=item *

Florian Ragwitz <rafl@debian.org>

=back

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2015 by MongoDB, Inc..

This is free software, licensed under:

  The Apache License, Version 2.0, January 2004

=cut

__END__


# vim: ts=4 sts=4 sw=4 et:
