# Copyright 2011 Kevin Ryde

# Finance-Quote-More is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as published
# by the Free Software Foundation; either version 3, or (at your option) any
# later version.
#
# Finance-Quote-More is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

package Finance::Quote::ATHEX;
use 5.004;
use strict;
use Carp;

# uncomment this to run the ### lines
use Devel::Comments;

our $VERSION = 11;

# eg. http://www.ase.gr/content/en/marketdata/stocks/prices/Share_SearchResults.asp?share=HTO
#
use constant ATHEX_LAST30_BASE_URL =>
  'http://www.ase.gr/content/en/marketdata/stocks/prices/Share_SearchResults.asp?share=';

sub methods {
  return (athex => \&athex_quotes);
}
sub labels {
  return (athex => [ qw(date isodate name
                        open high low last
                        method source success errormsg
                      ) ]);
}

sub athex_quotes {
  my ($fq, @symbol_list) = @_;
  my $ua = $fq->user_agent;
  my %quotes;

  foreach my $symbol (@symbol_list) {
    my $url = ATHEX_LAST30_BASE_URL . URI::Escape::uri_escape($symbol);

    require HTTP::Request;
    my $req = HTTP::Request->new ('GET', $url);
    $ua->prepare_request ($req);
    $req->accept_decodable; # we have decoded_content() below
    $req->user_agent ("Finance::Quote::ATHEX/$VERSION " . $req->user_agent);
    ### req: $req->as_string

    my $resp = $ua->request ($req);
    resp_to_quotes ($fq, $resp, \%quotes, $symbol);
  }
  return wantarray() ? %quotes : \%quotes;
}

sub resp_to_quotes {
  my ($fq, $resp, $quotes, $symbol) = @_;
  ### ATHEX resp_to_quotes() ...

  $quotes->{$symbol,'method'}  = 'athex';
  $quotes->{$symbol,'source'}  = __PACKAGE__;
  $quotes->{$symbol,'success'} = 0;

  if (! $resp->is_success) {
    $quotes->{$symbol,'errormsg'} = $resp->status_line;
    return;
  }
  my $content = $resp->decoded_content (raise_error => 1, charset => 'none');

  # message in page if bad symbol
  if ($content =~ /Your search didn't return any results/) {
    ### unknown symbol ...
    $quotes->{$symbol,'errormsg'} = 'Unknown symbol';
    return;
  }

  unless ($content
          =~ m{Share Closing Prices: ([A-Z]+)[^-]*-[^>]*>([^<]+)</a>}) {
    $quotes->{$symbol,'errormsg'} = 'Oops, last30 name not matched';
    return;
  }
  my $name = $2;
  ### raw name: $name

  # Some names on the english pages have greek 8859-7 capitals, mung those
  # to plain ascii.  This tr generated by devel/athex-tr.pl.
  #
  $name =~ tr{\xB6\xB8\xB9\xBA\xBF\xC1\xC2\xC3\xC4\xC5\xC6\xC7\xC9\xCA\xCB\xCC\xCD\xCE\xD0\xD1\xD3\xD4\xD9\xDC\xDD\xDF\xE1\xE2\xE3\xE4\xE5\xE6\xE9\xEA\xEB\xEC\xED\xF0\xF1\xF2\xF3\xF4\xF9\xFA\xFE}
             {AEHIOABGDEZHIKLMNXPRSTOaeiabgdeziklmnprsstoio};
  ### tr name: $name

  require HTML::TableExtract;
  my $te = HTML::TableExtract->new
    (headers => ['Date', 'Open', 'Max', 'Min', 'Price', 'Volume' ]);
  $te->parse($content);
  if (! $te->tables) {
    $quotes->{$symbol,'errormsg'} = 'Oops, last30 table not matched';
    return;
  }

  my $rows = $te->rows;
  ### row: $rows->[0]
  if (! $rows->[0]) {
    $quotes->{$symbol,'errormsg'} = 'No trades';
    return;
  }

  my ($date, $open, $high, $low, $close, $volume) = @{$rows->[0]};
  my $prev;
  if ($rows->[1]) {
    $prev = $rows->[1]->[4];
  }

  $volume =~ s/,//g;  # remove comma thousands separators

  $quotes->{$symbol,'method'}  = 'athex';
  $quotes->{$symbol,'source'}  = __PACKAGE__;
  $quotes->{$symbol,'success'} = 1;
  $quotes->{$symbol,'name'}    = $name;
  $quotes->{$symbol,'open'}    = $open;
  $quotes->{$symbol,'high'}    = $high;
  $quotes->{$symbol,'low'}     = $low;
  $quotes->{$symbol,'last'}    = $close; # today's close
  $quotes->{$symbol,'close'}   = $prev;  # previous close
  $quotes->{$symbol,'volume'}  = $volume;
  $fq->store_date($quotes, $symbol, {eurodate => $date});
}


1;
__END__

=head1 NAME

Finance::Quote::ATHEX - download futures quotes from ATHEX

=head1 SYNOPSIS

 use Finance::Quote;
 my $fq = Finance::Quote->new ('ATHEX');
 my %quotes = $fq->fetch('athex', 'CLZ11');

=head1 DESCRIPTION

This module downloads share prices from the Athens Stock Exchange,

=over 4

http://www.ase.gr/

=back

Using the English "last 30 days" pages such as "HTO" for Hellenic Telecom,

=over 4

http://www.ase.gr/content/en/marketdata/stocks/prices/Share_SearchResults.asp?share=HTO

=back

=head1 FIELDS

The following standard fields are available

=for Finance_Quote_Grab fields flowed standard

    date isodate name currency
    open high low last volume close
    method source success errormsg

For reference, some of the English names in the web pages have ISO-8859-7
Greek characters such as 0xC2 Beta for "B" in "BANK".  They're
transliterated to their apparent ASCII intention where possible.

=head1 SEE ALSO

L<Finance::Quote>, L<LWP>

ATHEX web site http://www.ase.gr

=head1 HOME PAGE

http://user42.tuxfamily.org/finance-quote-grab/index.html

=head1 LICENCE

Copyright 2011 Kevin Ryde

Finance-Quote-Grab is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3, or (at your option) any later
version.

Finance-Quote-Grab is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
more details.

You should have received a copy of the GNU General Public License along with
Finance-Quote-Grab; see the file F<COPYING>.  If not, see
<http://www.gnu.org/licenses/>.

=cut
