use strict;
use warnings;
use Test::More;
use File::Find;
use Parse::Distname;

my $cpan_mirror;
BEGIN {
  $cpan_mirror = $ENV{TEST_CPAN_MIRROR}
    or plan skip_all => "requires \$ENV{TEST_CPAN_MIRROR} to test";

  -d "$cpan_mirror/authors/id/"
    or plan skip_all => "\$ENV{TEST_CPAN_MIRROR} seems not a CPAN mirror";

  eval "use CPAN::DistnameInfo 0.12; 1"
    or plan skip_all => "requires CPAN::DistnameInfo to compare";
}

my %KnownDiff;
{
  local $/ = "";
  while(<DATA>) {
    s/(?:\015\012|\015|\012)/\n/sg;
    s/\A#[^\n]*\n//sg;
    my ($file, $diff) = split "\n", $_, 2;
    chomp $file; chomp $diff;
    $file =~ s/\s*\(.+$//; # remove year info
    $KnownDiff{$file} = $diff;
  }
}

find({
  wanted => sub {
    my $file = $File::Find::name;
    return unless -f $file;
    return unless $file =~ /$Parse::Distname::SUFFRE/i;
    my $cpan_path = $file;
    $cpan_path =~ s|^.+authors/id/||;

    my %cd = CPAN::DistnameInfo->new($cpan_path)->properties;
    my %pd = Parse::Distname->new($cpan_path)->properties;

    # ignore spammers
    return if $pd{cpanid} =~ /^(?:DOCRIVERS|FULLHQ|ULTRAHD)$/;

    my $ok = 1;
    my $diff = '';
    for my $key (sort keys %cd) {
      if (defined $cd{$key}) {
        if (defined $pd{$key}) {
          if ($cd{$key} eq $pd{$key}) {
            # OK
            $diff .= <<"DIFF";
   "$key": "$cd{$key}",
DIFF
          } else {
            $diff .= <<"DIFF";
-  "$key": "$cd{$key}",
+  "$key": "$pd{$key}",
DIFF
            $ok = 0;
          }
        } else {
          $diff .= <<"DIFF";
-  "$key": "$cd{$key}",
+  "$key": null,
DIFF
          $ok = 0;
        }
      }
      elsif (defined $pd{$key}) {
        $diff .= <<"DIFF";
-  "$key": null,
+  "$key": "$pd{$key}",
DIFF
        $ok = 0;
      }
      else {
        # both are undef; OK
        $diff .= <<"DIFF";
   "$key": null,
DIFF
      }
    }

    $diff =~ s/(?:\015\012|\015|\012)/\n/gs;
    chomp $diff;

    if ($ok) {
        pass $file;
    } else {
        SKIP: {
            skip "not portable", 1 if $^O eq 'MSWin32' and $cpan_path =~ /\.\\/;
            if (!$KnownDiff{$cpan_path} or $diff ne $KnownDiff{$cpan_path}) {
                my $mtime = (stat($file))[9];
                my $year = (localtime($mtime))[5] + 1900;
                print STDERR "$cpan_path ($year)\n$diff\n\n";
                fail $file;
            } else {
                pass $file;
            }
        }
    }
  },
  no_chdir => 1,
}, "$cpan_mirror/authors/id/");

done_testing;

__DATA__
D/DP/DPCHRIST/Math-TriangularNumbers-r0_03.tar.gz (2005)
   "cpanid": "DPCHRIST",
-  "dist": "Math-TriangularNumbers-r",
+  "dist": "Math-TriangularNumbers",
   "distvname": "Math-TriangularNumbers-r0_03",
   "extension": "tar.gz",
   "filename": "Math-TriangularNumbers-r0_03.tar.gz",
   "maturity": "released",
   "pathname": "D/DP/DPCHRIST/Math-TriangularNumbers-r0_03.tar.gz",
-  "version": "0_03",
+  "version": "r0_03",

D/DP/DPCHRIST/Math-TriangularNumbers-r0_02.tar.gz (2005)
   "cpanid": "DPCHRIST",
-  "dist": "Math-TriangularNumbers-r",
+  "dist": "Math-TriangularNumbers",
   "distvname": "Math-TriangularNumbers-r0_02",
   "extension": "tar.gz",
   "filename": "Math-TriangularNumbers-r0_02.tar.gz",
   "maturity": "released",
   "pathname": "D/DP/DPCHRIST/Math-TriangularNumbers-r0_02.tar.gz",
-  "version": "0_02",
+  "version": "r0_02",

H/HO/HOLCAPEK/0.1.tar.gz (2017)
   "cpanid": "HOLCAPEK",
-  "dist": "0.1",
+  "dist": "",
   "distvname": "0.1",
   "extension": "tar.gz",
   "filename": "0.1.tar.gz",
   "maturity": "released",
   "pathname": "H/HO/HOLCAPEK/0.1.tar.gz",
-  "version": null,
+  "version": "0.1",

H/HW/HWAT/0.02.tar.gz (2014)
   "cpanid": "HWAT",
-  "dist": "0.02",
+  "dist": "",
   "distvname": "0.02",
   "extension": "tar.gz",
   "filename": "0.02.tar.gz",
   "maturity": "released",
   "pathname": "H/HW/HWAT/0.02.tar.gz",
-  "version": null,
+  "version": "0.02",

H/HA/HALLORAN/PRIMERVIEWv3.zip (2017)
   "cpanid": "HALLORAN",
-  "dist": "PRIMERVIEWv",
+  "dist": "PRIMERVIEW",
   "distvname": "PRIMERVIEWv3",
   "extension": "zip",
   "filename": "PRIMERVIEWv3.zip",
   "maturity": "released",
   "pathname": "H/HA/HALLORAN/PRIMERVIEWv3.zip",
-  "version": "3",
+  "version": "v3",

H/HA/HALLORAN/fastQ_brewv1.0.3.zip (2017)
   "cpanid": "HALLORAN",
-  "dist": "fastQ_brewv",
+  "dist": "fastQ_brew",
   "distvname": "fastQ_brewv1.0.3",
   "extension": "zip",
   "filename": "fastQ_brewv1.0.3.zip",
   "maturity": "released",
   "pathname": "H/HA/HALLORAN/fastQ_brewv1.0.3.zip",
-  "version": "1.0.3",
+  "version": "v1.0.3",

H/HA/HARPREET/XMS-MotifSetv1.0.tar.gz (2010)
   "cpanid": "HARPREET",
-  "dist": "XMS-MotifSetv",
+  "dist": "XMS-MotifSet",
   "distvname": "XMS-MotifSetv1.0",
   "extension": "tar.gz",
   "filename": "XMS-MotifSetv1.0.tar.gz",
   "maturity": "released",
   "pathname": "H/HA/HARPREET/XMS-MotifSetv1.0.tar.gz",
-  "version": "1.0",
+  "version": "v1.0",

O/OP/OPSVIEW/Opsview-Performanceparsing-1-TRIAL.tar.gz (2013)
   "cpanid": "OPSVIEW",
-  "dist": "Opsview-Performanceparsing-1-TRIAL",
+  "dist": "Opsview-Performanceparsing",
   "distvname": "Opsview-Performanceparsing-1-TRIAL",
   "extension": "tar.gz",
   "filename": "Opsview-Performanceparsing-1-TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "O/OP/OPSVIEW/Opsview-Performanceparsing-1-TRIAL.tar.gz",
-  "version": null,
+  "version": "1-TRIAL",

O/ON/ONEONETWO/Array-KeepGrepped-2-TRIAL.tar.gz (2013)
   "cpanid": "ONEONETWO",
-  "dist": "Array-KeepGrepped-2-TRIAL",
+  "dist": "Array-KeepGrepped",
   "distvname": "Array-KeepGrepped-2-TRIAL",
   "extension": "tar.gz",
   "filename": "Array-KeepGrepped-2-TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "O/ON/ONEONETWO/Array-KeepGrepped-2-TRIAL.tar.gz",
-  "version": null,
+  "version": "2-TRIAL",

C/CA/CAFFIEND/font_ft2_0.1.0.tgz (2002)
   "cpanid": "CAFFIEND",
-  "dist": "font_ft",
+  "dist": "font_ft2",
   "distvname": "font_ft2_0.1.0",
   "extension": "tgz",
   "filename": "font_ft2_0.1.0.tgz",
   "maturity": "released",
   "pathname": "C/CA/CAFFIEND/font_ft2_0.1.0.tgz",
-  "version": "2_0.1.0",
+  "version": "0.1.0",

C/CO/COLINO/Allegro-0.02_0-ppm-MSWin32-x86-multi-thread-5.6.tar.gz
   "cpanid": "COLINO",
-  "dist": "Allegro-0.02_0-ppm-MSWin32-x86-multi-thread",
+  "dist": "Allegro",
   "distvname": "Allegro-0.02_0-ppm-MSWin32-x86-multi-thread-5.6",
   "extension": "tar.gz",
   "filename": "Allegro-0.02_0-ppm-MSWin32-x86-multi-thread-5.6.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "C/CO/COLINO/Allegro-0.02_0-ppm-MSWin32-x86-multi-thread-5.6.tar.gz",
-  "version": "5.6",
+  "version": "0.02_0-ppm-MSWin32-x86-multi-thread-5.6",

C/CO/COLINO/Allegro-0.02_0-ppm-MSWin32-x86-multi-thread-5.8.tar.gz (2004)
   "cpanid": "COLINO",
-  "dist": "Allegro-0.02_0-ppm-MSWin32-x86-multi-thread",
+  "dist": "Allegro",
   "distvname": "Allegro-0.02_0-ppm-MSWin32-x86-multi-thread-5.8",
   "extension": "tar.gz",
   "filename": "Allegro-0.02_0-ppm-MSWin32-x86-multi-thread-5.8.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "C/CO/COLINO/Allegro-0.02_0-ppm-MSWin32-x86-multi-thread-5.8.tar.gz",
-  "version": "5.8",
+  "version": "0.02_0-ppm-MSWin32-x86-multi-thread-5.8",

C/CY/CYCLES/ocharles-Test-Magpie-v0.04-2-g9a8e649.tar.gz (2010)
   "cpanid": "CYCLES",
-  "dist": "ocharles-Test-Magpie-v0.04",
+  "dist": "ocharles-Test-Magpie",
   "distvname": "ocharles-Test-Magpie-v0.04-2-g9a8e649",
   "extension": "tar.gz",
   "filename": "ocharles-Test-Magpie-v0.04-2-g9a8e649.tar.gz",
   "maturity": "released",
   "pathname": "C/CY/CYCLES/ocharles-Test-Magpie-v0.04-2-g9a8e649.tar.gz",
-  "version": "2-g9a8e649",
+  "version": "v0.04-2-g9a8e649",

C/CD/CDONLEY/netldapapi-140-winnt.zip (1998)
   "cpanid": "CDONLEY",
-  "dist": "netldapapi-140-winnt",
+  "dist": "netldapapi",
   "distvname": "netldapapi-140-winnt",
   "extension": "zip",
   "filename": "netldapapi-140-winnt.zip",
   "maturity": "released",
   "pathname": "C/CD/CDONLEY/netldapapi-140-winnt.zip",
-  "version": null,
+  "version": "140-winnt",

C/CD/CDONLEY/netldapapi-139a-winnt.zip (1997)
   "cpanid": "CDONLEY",
-  "dist": "netldapapi-139a-winnt",
+  "dist": "netldapapi",
   "distvname": "netldapapi-139a-winnt",
   "extension": "zip",
   "filename": "netldapapi-139a-winnt.zip",
   "maturity": "released",
   "pathname": "C/CD/CDONLEY/netldapapi-139a-winnt.zip",
-  "version": null,
+  "version": "139a-winnt",

C/CD/CDRAKE/Crypt-MatrixSSL3.tar.gz (2012)
   "cpanid": "CDRAKE",
-  "dist": "Crypt",
+  "dist": "Crypt-MatrixSSL",
   "distvname": "Crypt-MatrixSSL3",
   "extension": "tar.gz",
   "filename": "Crypt-MatrixSSL3.tar.gz",
   "maturity": "released",
   "pathname": "C/CD/CDRAKE/Crypt-MatrixSSL3.tar.gz",
-  "version": "MatrixSSL3",
+  "version": "3",

C/CR/CRLCU/v0.1.tar.gz (2016)
   "cpanid": "CRLCU",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.1",
   "extension": "tar.gz",
   "filename": "v0.1.tar.gz",
   "maturity": "released",
   "pathname": "C/CR/CRLCU/v0.1.tar.gz",
-  "version": "0.1",
+  "version": "v0.1",

C/CN/CNANDOR/DProf-19990108-bin-1-Mac.tgz (1999)
   "cpanid": "CNANDOR",
-  "dist": "DProf-19990108-bin-1-Mac",
+  "dist": "DProf",
   "distvname": "DProf-19990108-bin-1-Mac",
   "extension": "tgz",
   "filename": "DProf-19990108-bin-1-Mac.tgz",
   "maturity": "released",
   "pathname": "C/CN/CNANDOR/DProf-19990108-bin-1-Mac.tgz",
-  "version": null,
+  "version": "19990108-bin-1-Mac",

C/CN/CNANDOR/glue.19990522.tar.gz (1999)
   "cpanid": "CNANDOR",
   "dist": "glue",
   "distvname": "glue.19990522",
   "extension": "tar.gz",
   "filename": "glue.19990522.tar.gz",
   "maturity": "released",
   "pathname": "C/CN/CNANDOR/glue.19990522.tar.gz",
-  "version": ".19990522",
+  "version": "19990522",

C/CN/CNANDOR/glue.19990622.tar.gz (1999)
   "cpanid": "CNANDOR",
   "dist": "glue",
   "distvname": "glue.19990622",
   "extension": "tar.gz",
   "filename": "glue.19990622.tar.gz",
   "maturity": "released",
   "pathname": "C/CN/CNANDOR/glue.19990622.tar.gz",
-  "version": ".19990622",
+  "version": "19990622",

C/CS/CSANDEEP/0.1.1.tar.gz (2018)
   "cpanid": "CSANDEEP",
-  "dist": "0.1.1",
+  "dist": "",
   "distvname": "0.1.1",
   "extension": "tar.gz",
   "filename": "0.1.1.tar.gz",
   "maturity": "released",
   "pathname": "C/CS/CSANDEEP/0.1.1.tar.gz",
-  "version": null,
+  "version": "0.1.1",

T/TQ/TQISJIM/BerkeleyDB_Locks_0_0_1.tar.gz (2003)
   "cpanid": "TQISJIM",
   "dist": "BerkeleyDB_Locks",
   "distvname": "BerkeleyDB_Locks_0_0_1",
   "extension": "tar.gz",
   "filename": "BerkeleyDB_Locks_0_0_1.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "T/TQ/TQISJIM/BerkeleyDB_Locks_0_0_1.tar.gz",
   "version": "0_0_1",

T/TE/TEODESIAN/v1.25.tar.gz (2018)
   "cpanid": "TEODESIAN",
-  "dist": "v",
+  "dist": "",
   "distvname": "v1.25",
   "extension": "tar.gz",
   "filename": "v1.25.tar.gz",
   "maturity": "released",
   "pathname": "T/TE/TEODESIAN/v1.25.tar.gz",
-  "version": "1.25",
+  "version": "v1.25",

T/TI/TIMA/0.21.zip (2003)
   "cpanid": "TIMA",
-  "dist": "0.21",
+  "dist": "",
   "distvname": "0.21",
   "extension": "zip",
   "filename": "0.21.zip",
   "maturity": "released",
   "pathname": "T/TI/TIMA/0.21.zip",
-  "version": null,
+  "version": "0.21",

T/TI/TIMB/perl5.004_04-m2.tar.gz (1998)
   "cpanid": "TIMB",
   "dist": "perl",
   "distvname": "perl5.004_04-m2",
   "extension": "tar.gz",
   "filename": "perl5.004_04-m2.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "T/TI/TIMB/perl5.004_04-m2.tar.gz",
   "version": "5.004_04-m2",

T/TI/TIMB/perl5.004_04-m4.tar.gz (1998)
   "cpanid": "TIMB",
   "dist": "perl",
   "distvname": "perl5.004_04-m4",
   "extension": "tar.gz",
   "filename": "perl5.004_04-m4.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "T/TI/TIMB/perl5.004_04-m4.tar.gz",
   "version": "5.004_04-m4",

T/TI/TIMB/perl5.004_04-m3.tar.gz (1998)
   "cpanid": "TIMB",
   "dist": "perl",
   "distvname": "perl5.004_04-m3",
   "extension": "tar.gz",
   "filename": "perl5.004_04-m3.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "T/TI/TIMB/perl5.004_04-m3.tar.gz",
   "version": "5.004_04-m3",

T/TU/TURNERJW/LoadHtml.5_0.tar.gz (2001)
   "cpanid": "TURNERJW",
   "dist": "LoadHtml",
   "distvname": "LoadHtml.5_0",
   "extension": "tar.gz",
   "filename": "LoadHtml.5_0.tar.gz",
   "maturity": "released",
   "pathname": "T/TU/TURNERJW/LoadHtml.5_0.tar.gz",
-  "version": ".5_0",
+  "version": "5_0",

T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.31.09-Nov_16-09_00.tgz (2009)
   "cpanid": "TURNERMM",
-  "dist": "HTML-WikiConverter-DokuWikiFCK-0.31.09-Nov_16",
+  "dist": "HTML-WikiConverter-DokuWikiFCK",
   "distvname": "HTML-WikiConverter-DokuWikiFCK-0.31.09-Nov_16-09_00",
   "extension": "tgz",
   "filename": "HTML-WikiConverter-DokuWikiFCK-0.31.09-Nov_16-09_00.tgz",
   "maturity": "released",
   "pathname": "T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.31.09-Nov_16-09_00.tgz",
-  "version": "09_00",
+  "version": "0.31.09-Nov_16-09_00",

T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.32.10-Feb_09-12_16.tgz (2010)
   "cpanid": "TURNERMM",
-  "dist": "HTML-WikiConverter-DokuWikiFCK-0.32.10-Feb_09",
+  "dist": "HTML-WikiConverter-DokuWikiFCK",
   "distvname": "HTML-WikiConverter-DokuWikiFCK-0.32.10-Feb_09-12_16",
   "extension": "tgz",
   "filename": "HTML-WikiConverter-DokuWikiFCK-0.32.10-Feb_09-12_16.tgz",
   "maturity": "released",
   "pathname": "T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.32.10-Feb_09-12_16.tgz",
-  "version": "12_16",
+  "version": "0.32.10-Feb_09-12_16",

T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.30.09-Oct_08-07_09.tgz (2009)
   "cpanid": "TURNERMM",
-  "dist": "HTML-WikiConverter-DokuWikiFCK-0.30.09-Oct_08",
+  "dist": "HTML-WikiConverter-DokuWikiFCK",
   "distvname": "HTML-WikiConverter-DokuWikiFCK-0.30.09-Oct_08-07_09",
   "extension": "tgz",
   "filename": "HTML-WikiConverter-DokuWikiFCK-0.30.09-Oct_08-07_09.tgz",
   "maturity": "released",
   "pathname": "T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.30.09-Oct_08-07_09.tgz",
-  "version": "07_09",
+  "version": "0.30.09-Oct_08-07_09",

T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.32_Feb-26-2013.tgz (2013)
   "cpanid": "TURNERMM",
-  "dist": "HTML-WikiConverter-DokuWikiFCK-0.32_Feb",
+  "dist": "HTML-WikiConverter-DokuWikiFCK",
   "distvname": "HTML-WikiConverter-DokuWikiFCK-0.32_Feb-26-2013",
   "extension": "tgz",
   "filename": "HTML-WikiConverter-DokuWikiFCK-0.32_Feb-26-2013.tgz",
   "maturity": "released",
   "pathname": "T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.32_Feb-26-2013.tgz",
-  "version": "262013",
+  "version": "0.32_Feb-26-2013",

T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.32_Feb-27-2013.tgz (2013)
   "cpanid": "TURNERMM",
-  "dist": "HTML-WikiConverter-DokuWikiFCK-0.32_Feb",
+  "dist": "HTML-WikiConverter-DokuWikiFCK",
   "distvname": "HTML-WikiConverter-DokuWikiFCK-0.32_Feb-27-2013",
   "extension": "tgz",
   "filename": "HTML-WikiConverter-DokuWikiFCK-0.32_Feb-27-2013.tgz",
   "maturity": "released",
   "pathname": "T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.32_Feb-27-2013.tgz",
-  "version": "272013",
+  "version": "0.32_Feb-27-2013",

T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.32.10-Sep_09-08_10.tgz (2010)
   "cpanid": "TURNERMM",
-  "dist": "HTML-WikiConverter-DokuWikiFCK-0.32.10-Sep_09",
+  "dist": "HTML-WikiConverter-DokuWikiFCK",
   "distvname": "HTML-WikiConverter-DokuWikiFCK-0.32.10-Sep_09-08_10",
   "extension": "tgz",
   "filename": "HTML-WikiConverter-DokuWikiFCK-0.32.10-Sep_09-08_10.tgz",
   "maturity": "released",
   "pathname": "T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.32.10-Sep_09-08_10.tgz",
-  "version": "08_10",
+  "version": "0.32.10-Sep_09-08_10",

T/TZ/TZADIKV/-withoutworldwriteables.tar.gz (2010)
   "cpanid": "TZADIKV",
-  "dist": "-withoutworldwriteables",
+  "dist": "",
   "distvname": "-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "T/TZ/TZADIKV/-withoutworldwriteables.tar.gz",
   "version": null,

Y/YA/YAKEX/Software-License-Boost_1_0-v0.0.1.tar.gz (2013)
   "cpanid": "YAKEX",
-  "dist": "Software-License-Boost",
+  "dist": "Software-License-Boost_1_0",
   "distvname": "Software-License-Boost_1_0-v0.0.1",
   "extension": "tar.gz",
   "filename": "Software-License-Boost_1_0-v0.0.1.tar.gz",
   "maturity": "released",
   "pathname": "Y/YA/YAKEX/Software-License-Boost_1_0-v0.0.1.tar.gz",
-  "version": "1_0-v0.0.1",
+  "version": "v0.0.1",

I/IL/ILYAZ/MathPari_a3.tar.gz (1995)
   "cpanid": "ILYAZ",
-  "dist": "MathPari_a",
+  "dist": "MathPari",
   "distvname": "MathPari_a3",
   "extension": "tar.gz",
   "filename": "MathPari_a3.tar.gz",
   "maturity": "released",
   "pathname": "I/IL/ILYAZ/MathPari_a3.tar.gz",
-  "version": "3",
+  "version": "a3",

I/IL/ILYAZ/os2/perl5os2.patch.zip (1995)
   "cpanid": "ILYAZ",
-  "dist": "perl5os",
+  "dist": "perl5os2.patch",
   "distvname": "perl5os2.patch",
   "extension": "zip",
   "filename": "os2/perl5os2.patch.zip",
   "maturity": "released",
   "pathname": "I/IL/ILYAZ/os2/perl5os2.patch.zip",
-  "version": "2.patch",
+  "version": null,

I/IL/ILYAZ/os2/perl5n.os2.patch.zip (1995)
   "cpanid": "ILYAZ",
-  "dist": "perl5n.os",
+  "dist": "perl5n.os2.patch",
   "distvname": "perl5n.os2.patch",
   "extension": "zip",
   "filename": "os2/perl5n.os2.patch.zip",
   "maturity": "released",
   "pathname": "I/IL/ILYAZ/os2/perl5n.os2.patch.zip",
-  "version": "2.patch",
+  "version": null,

I/IL/ILYAZ/os2/perlos2.fix1.zip (1996)
   "cpanid": "ILYAZ",
-  "dist": "perlos",
+  "dist": "perlos2.fix",
   "distvname": "perlos2.fix1",
   "extension": "zip",
   "filename": "os2/perlos2.fix1.zip",
   "maturity": "released",
   "pathname": "I/IL/ILYAZ/os2/perlos2.fix1.zip",
-  "version": "2.fix1",
+  "version": "1",

I/IL/ILYAZ/os2/perl5.00301.os2.zip (1996)
   "cpanid": "ILYAZ",
   "dist": "perl",
   "distvname": "perl5.00301.os2",
   "extension": "zip",
   "filename": "os2/perl5.00301.os2.zip",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "I/IL/ILYAZ/os2/perl5.00301.os2.zip",
   "version": "5.00301.os2",

I/IL/ILYAZ/os2/perl.500301.os2.bin.zip (1996)
   "cpanid": "ILYAZ",
   "dist": "perl",
   "distvname": "perl.500301.os2.bin",
   "extension": "zip",
   "filename": "os2/perl.500301.os2.bin.zip",
   "maturity": "released",
   "pathname": "I/IL/ILYAZ/os2/perl.500301.os2.bin.zip",
-  "version": ".500301.os2.bin",
+  "version": "500301.os2.bin",

I/IL/ILYAZ/os2/perl.500201.os2.bin.zip (1996)
   "cpanid": "ILYAZ",
   "dist": "perl",
   "distvname": "perl.500201.os2.bin",
   "extension": "zip",
   "filename": "os2/perl.500201.os2.bin.zip",
   "maturity": "released",
   "pathname": "I/IL/ILYAZ/os2/perl.500201.os2.bin.zip",
-  "version": ".500201.os2.bin",
+  "version": "500201.os2.bin",

I/IL/ILYAZ/os2/tk/perltk_os2_402_patches.zip (1997)
   "cpanid": "ILYAZ",
-  "dist": "perltk_os",
+  "dist": "perltk_os2_402_patches",
   "distvname": "perltk_os2_402_patches",
   "extension": "zip",
   "filename": "os2/tk/perltk_os2_402_patches.zip",
   "maturity": "released",
   "pathname": "I/IL/ILYAZ/os2/tk/perltk_os2_402_patches.zip",
-  "version": "2_402_patches",
+  "version": null,

I/IL/ILYAZ/modules/MathPari_a3.tar.gz (1995)
   "cpanid": "ILYAZ",
-  "dist": "MathPari_a",
+  "dist": "MathPari",
   "distvname": "MathPari_a3",
   "extension": "tar.gz",
   "filename": "modules/MathPari_a3.tar.gz",
   "maturity": "released",
   "pathname": "I/IL/ILYAZ/modules/MathPari_a3.tar.gz",
-  "version": "3",
+  "version": "a3",

I/IN/INFRACANI/Mail-OpenDKIM-4201-patched.tar.gz (2015)
   "cpanid": "INFRACANI",
-  "dist": "Mail-OpenDKIM-4201-patched",
+  "dist": "Mail-OpenDKIM",
   "distvname": "Mail-OpenDKIM-4201-patched",
   "extension": "tar.gz",
   "filename": "Mail-OpenDKIM-4201-patched.tar.gz",
   "maturity": "released",
   "pathname": "I/IN/INFRACANI/Mail-OpenDKIM-4201-patched.tar.gz",
-  "version": null,
+  "version": "4201-patched",

I/IC/ICIBIN/bins-Message-Passing-Output-MongoDB-1eb79aa.tar.gz (2012)
   "cpanid": "ICIBIN",
-  "dist": "bins-Message-Passing-Output-MongoDB-1eb79aa",
+  "dist": "bins-Message-Passing-Output-MongoDB",
   "distvname": "bins-Message-Passing-Output-MongoDB-1eb79aa",
   "extension": "tar.gz",
   "filename": "bins-Message-Passing-Output-MongoDB-1eb79aa.tar.gz",
   "maturity": "released",
   "pathname": "I/IC/ICIBIN/bins-Message-Passing-Output-MongoDB-1eb79aa.tar.gz",
-  "version": null,
+  "version": "1eb79aa",

I/II/IIS/v1.0.5.tar.gz (2015)
   "cpanid": "IIS",
-  "dist": "v",
+  "dist": "",
   "distvname": "v1.0.5",
   "extension": "tar.gz",
   "filename": "v1.0.5.tar.gz",
   "maturity": "released",
   "pathname": "I/II/IIS/v1.0.5.tar.gz",
-  "version": "1.0.5",
+  "version": "v1.0.5",

I/IK/IKRUGLOV/v0.6.tar.gz (2014)
   "cpanid": "IKRUGLOV",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.6",
   "extension": "tar.gz",
   "filename": "v0.6.tar.gz",
   "maturity": "released",
   "pathname": "I/IK/IKRUGLOV/v0.6.tar.gz",
-  "version": "0.6",
+  "version": "v0.6",

I/IS/ISTEEL/HTMLTMPL.130.tar.gz (2001)
   "cpanid": "ISTEEL",
   "dist": "HTMLTMPL",
   "distvname": "HTMLTMPL.130",
   "extension": "tar.gz",
   "filename": "HTMLTMPL.130.tar.gz",
   "maturity": "released",
   "pathname": "I/IS/ISTEEL/HTMLTMPL.130.tar.gz",
-  "version": ".130",
+  "version": "130",

I/IS/ISTEEL/HTMLTMPL.131.tar.gz (2001)
   "cpanid": "ISTEEL",
   "dist": "HTMLTMPL",
   "distvname": "HTMLTMPL.131",
   "extension": "tar.gz",
   "filename": "HTMLTMPL.131.tar.gz",
   "maturity": "released",
   "pathname": "I/IS/ISTEEL/HTMLTMPL.131.tar.gz",
-  "version": ".131",
+  "version": "131",

F/FE/FELIPE/IO-Framed-0.03_TRIAL1.tar.gz (2017)
   "cpanid": "FELIPE",
   "dist": "IO-Framed",
   "distvname": "IO-Framed-0.03_TRIAL1",
   "extension": "tar.gz",
   "filename": "IO-Framed-0.03_TRIAL1.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "F/FE/FELIPE/IO-Framed-0.03_TRIAL1.tar.gz",
   "version": "0.03_TRIAL1",

F/FE/FELIPE/Crypt-Perl-0.24_TRIAL1.tar.gz (2018)
   "cpanid": "FELIPE",
   "dist": "Crypt-Perl",
   "distvname": "Crypt-Perl-0.24_TRIAL1",
   "extension": "tar.gz",
   "filename": "Crypt-Perl-0.24_TRIAL1.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "F/FE/FELIPE/Crypt-Perl-0.24_TRIAL1.tar.gz",
   "version": "0.24_TRIAL1",

F/FE/FELIPE/IO-Framed-0.03_TRIAL3.tar.gz (2017)
   "cpanid": "FELIPE",
   "dist": "IO-Framed",
   "distvname": "IO-Framed-0.03_TRIAL3",
   "extension": "tar.gz",
   "filename": "IO-Framed-0.03_TRIAL3.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "F/FE/FELIPE/IO-Framed-0.03_TRIAL3.tar.gz",
   "version": "0.03_TRIAL3",

F/FE/FELIPE/IO-Framed-0.03_TRIAL2.tar.gz (2017)
   "cpanid": "FELIPE",
   "dist": "IO-Framed",
   "distvname": "IO-Framed-0.03_TRIAL2",
   "extension": "tar.gz",
   "filename": "IO-Framed-0.03_TRIAL2.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "F/FE/FELIPE/IO-Framed-0.03_TRIAL2.tar.gz",
   "version": "0.03_TRIAL2",

F/FE/FELIPE/Crypt-Perl-0.24_TRIAL2.tar.gz (2018)
   "cpanid": "FELIPE",
   "dist": "Crypt-Perl",
   "distvname": "Crypt-Perl-0.24_TRIAL2",
   "extension": "tar.gz",
   "filename": "Crypt-Perl-0.24_TRIAL2.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "F/FE/FELIPE/Crypt-Perl-0.24_TRIAL2.tar.gz",
   "version": "0.24_TRIAL2",

F/FE/FELIPE/IO-Framed-0.02_TRIAL.tar.gz (2017)
   "cpanid": "FELIPE",
   "dist": "IO-Framed",
   "distvname": "IO-Framed-0.02_TRIAL",
   "extension": "tar.gz",
   "filename": "IO-Framed-0.02_TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "F/FE/FELIPE/IO-Framed-0.02_TRIAL.tar.gz",
   "version": "0.02_TRIAL",

F/FE/FERNANDES/155.zip (2007)
   "cpanid": "FERNANDES",
-  "dist": "155",
+  "dist": "",
   "distvname": "155",
   "extension": "zip",
   "filename": "155.zip",
   "maturity": "released",
   "pathname": "F/FE/FERNANDES/155.zip",
-  "version": null,
+  "version": "155",

F/FI/FIBO/PNI_PROJECT.20091111.zip (2009)
   "cpanid": "FIBO",
   "dist": "PNI_PROJECT",
   "distvname": "PNI_PROJECT.20091111",
   "extension": "zip",
   "filename": "PNI_PROJECT.20091111.zip",
   "maturity": "released",
   "pathname": "F/FI/FIBO/PNI_PROJECT.20091111.zip",
-  "version": ".20091111",
+  "version": "20091111",

F/FI/FIBO/PNI.20100406.zip (2010)
   "cpanid": "FIBO",
   "dist": "PNI",
   "distvname": "PNI.20100406",
   "extension": "zip",
   "filename": "PNI.20100406.zip",
   "maturity": "released",
   "pathname": "F/FI/FIBO/PNI.20100406.zip",
-  "version": ".20100406",
+  "version": "20100406",

F/FI/FIBO/PNIProject.20091027.zip (2009)
   "cpanid": "FIBO",
   "dist": "PNIProject",
   "distvname": "PNIProject.20091027",
   "extension": "zip",
   "filename": "PNIProject.20091027.zip",
   "maturity": "released",
   "pathname": "F/FI/FIBO/PNIProject.20091027.zip",
-  "version": ".20091027",
+  "version": "20091027",

F/FI/FIBO/PNIProject.20091027.zip (2009)
   "cpanid": "FIBO",
   "dist": "PNIProject",
   "distvname": "PNIProject.20091027",
   "extension": "zip",
   "filename": "PNIProject.20091027.zip",
   "maturity": "released",
   "pathname": "F/FI/FIBO/PNIProject.20091027.zip",
-  "version": ".20091027",
+  "version": "20091027",

Z/ZT/ZTURK/woda-uk-3-402-pm.tar.gz (1999)
   "cpanid": "ZTURK",
-  "dist": "woda-uk-3-402-pm",
+  "dist": "woda-uk",
   "distvname": "woda-uk-3-402-pm",
   "extension": "tar.gz",
   "filename": "woda-uk-3-402-pm.tar.gz",
   "maturity": "released",
   "pathname": "Z/ZT/ZTURK/woda-uk-3-402-pm.tar.gz",
-  "version": null,
+  "version": "3-402-pm",

P/PH/PHLUKS/v1.0.tar.gz (2016)
   "cpanid": "PHLUKS",
-  "dist": "v",
+  "dist": "",
   "distvname": "v1.0",
   "extension": "tar.gz",
   "filename": "v1.0.tar.gz",
   "maturity": "released",
   "pathname": "P/PH/PHLUKS/v1.0.tar.gz",
-  "version": "1.0",
+  "version": "v1.0",

P/PH/PHAM/v0.06.tar.gz (2016)
   "cpanid": "PHAM",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.06",
   "extension": "tar.gz",
   "filename": "v0.06.tar.gz",
   "maturity": "released",
   "pathname": "P/PH/PHAM/v0.06.tar.gz",
-  "version": "0.06",
+  "version": "v0.06",

P/PH/PHAM/v0.05.tar.gz (2015)
   "cpanid": "PHAM",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.05",
   "extension": "tar.gz",
   "filename": "v0.05.tar.gz",
   "maturity": "released",
   "pathname": "P/PH/PHAM/v0.05.tar.gz",
-  "version": "0.05",
+  "version": "v0.05",

P/PT/PTILL/mod_perl-1.11_1-apache-1.3b7dev-bin-bindist1-i386-win95-vc5.zip (1998)
   "cpanid": "PTILL",
-  "dist": "mod_perl-1.11_1-apache",
+  "dist": "mod_perl",
   "distvname": "mod_perl-1.11_1-apache-1.3b7dev-bin-bindist1-i386-win95-vc5",
   "extension": "zip",
   "filename": "mod_perl-1.11_1-apache-1.3b7dev-bin-bindist1-i386-win95-vc5.zip",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "P/PT/PTILL/mod_perl-1.11_1-apache-1.3b7dev-bin-bindist1-i386-win95-vc5.zip",
-  "version": "1.3b7dev-bin-bindist1-i386-win95-vc5",
+  "version": "1.11_1-apache-1.3b7dev-bin-bindist1-i386-win95-vc5",

P/PW/PWBENNETT/Net-IPAddress-Util-2.003_TRIAL.tar.gz (2013)
   "cpanid": "PWBENNETT",
   "dist": "Net-IPAddress-Util",
   "distvname": "Net-IPAddress-Util-2.003_TRIAL",
   "extension": "tar.gz",
   "filename": "Net-IPAddress-Util-2.003_TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "P/PW/PWBENNETT/Net-IPAddress-Util-2.003_TRIAL.tar.gz",
   "version": "2.003_TRIAL",

P/PW/PWBENNETT/Net-IPAddress-Util-2.001_TRIAL.tar.gz (2013)
   "cpanid": "PWBENNETT",
   "dist": "Net-IPAddress-Util",
   "distvname": "Net-IPAddress-Util-2.001_TRIAL",
   "extension": "tar.gz",
   "filename": "Net-IPAddress-Util-2.001_TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "P/PW/PWBENNETT/Net-IPAddress-Util-2.001_TRIAL.tar.gz",
   "version": "2.001_TRIAL",

P/PW/PWBENNETT/Net-IPAddress-Util-2.002_TRIAL.tar.gz (2013)
   "cpanid": "PWBENNETT",
   "dist": "Net-IPAddress-Util",
   "distvname": "Net-IPAddress-Util-2.002_TRIAL",
   "extension": "tar.gz",
   "filename": "Net-IPAddress-Util-2.002_TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "P/PW/PWBENNETT/Net-IPAddress-Util-2.002_TRIAL.tar.gz",
   "version": "2.002_TRIAL",

P/PW/PWBENNETT/Net-IPAddress-Util-2.000_TRIAL.tar.gz (2013)
   "cpanid": "PWBENNETT",
   "dist": "Net-IPAddress-Util",
   "distvname": "Net-IPAddress-Util-2.000_TRIAL",
   "extension": "tar.gz",
   "filename": "Net-IPAddress-Util-2.000_TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "P/PW/PWBENNETT/Net-IPAddress-Util-2.000_TRIAL.tar.gz",
   "version": "2.000_TRIAL",

P/PW/PWBENNETT/Net-IPAddress-Util-2.004_TRIAL.tar.gz (2013)
   "cpanid": "PWBENNETT",
   "dist": "Net-IPAddress-Util",
   "distvname": "Net-IPAddress-Util-2.004_TRIAL",
   "extension": "tar.gz",
   "filename": "Net-IPAddress-Util-2.004_TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "P/PW/PWBENNETT/Net-IPAddress-Util-2.004_TRIAL.tar.gz",
   "version": "2.004_TRIAL",

P/PF/PFEIFFER/makepp-121020-test.tgz (2012)
   "cpanid": "PFEIFFER",
-  "dist": "makepp-121020-test",
+  "dist": "makepp",
   "distvname": "makepp-121020-test",
   "extension": "tgz",
   "filename": "makepp-121020-test.tgz",
   "maturity": "released",
   "pathname": "P/PF/PFEIFFER/makepp-121020-test.tgz",
-  "version": null,
+  "version": "121020-test",

P/PF/PFEIFFER/makepp-121021-test.tgz (2012)
   "cpanid": "PFEIFFER",
-  "dist": "makepp-121021-test",
+  "dist": "makepp",
   "distvname": "makepp-121021-test",
   "extension": "tgz",
   "filename": "makepp-121021-test.tgz",
   "maturity": "released",
   "pathname": "P/PF/PFEIFFER/makepp-121021-test.tgz",
-  "version": null,
+  "version": "121021-test",

P/PF/PFEIFFER/makepp-121023-test.tgz (2012)
   "cpanid": "PFEIFFER",
-  "dist": "makepp-121023-test",
+  "dist": "makepp",
   "distvname": "makepp-121023-test",
   "extension": "tgz",
   "filename": "makepp-121023-test.tgz",
   "maturity": "released",
   "pathname": "P/PF/PFEIFFER/makepp-121023-test.tgz",
-  "version": null,
+  "version": "121023-test",

P/PF/PFEIFFER/makepp-cvs-100210test.tgz (2010)
   "cpanid": "PFEIFFER",
-  "dist": "makepp-cvs-100210test",
+  "dist": "makepp-cvs",
   "distvname": "makepp-cvs-100210test",
   "extension": "tgz",
   "filename": "makepp-cvs-100210test.tgz",
   "maturity": "released",
   "pathname": "P/PF/PFEIFFER/makepp-cvs-100210test.tgz",
-  "version": null,
+  "version": "100210test",

P/PE/PERLISH/Abjad-withoutworldwriteables.tar.gz (2011)
   "cpanid": "PERLISH",
-  "dist": "Abjad-withoutworldwriteables",
+  "dist": "Abjad",
   "distvname": "Abjad-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "Abjad-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "P/PE/PERLISH/Abjad-withoutworldwriteables.tar.gz",
   "version": null,

P/PL/PLOCKABY/v1.0.0.tar.gz (2014)
   "cpanid": "PLOCKABY",
-  "dist": "v",
+  "dist": "",
   "distvname": "v1.0.0",
   "extension": "tar.gz",
   "filename": "v1.0.0.tar.gz",
   "maturity": "released",
   "pathname": "P/PL/PLOCKABY/v1.0.0.tar.gz",
-  "version": "1.0.0",
+  "version": "v1.0.0",

P/PU/PUSHTAEV/DBIx-Class-Factory-0.01_TRIAL.tar.gz (2015)
   "cpanid": "PUSHTAEV",
   "dist": "DBIx-Class-Factory",
   "distvname": "DBIx-Class-Factory-0.01_TRIAL",
   "extension": "tar.gz",
   "filename": "DBIx-Class-Factory-0.01_TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "P/PU/PUSHTAEV/DBIx-Class-Factory-0.01_TRIAL.tar.gz",
   "version": "0.01_TRIAL",

P/PU/PUSHTAEV/DBIx-Class-Factory-TRIAL.tar.gz (2015)
   "cpanid": "PUSHTAEV",
-  "dist": "DBIx-Class-Factory-TRIAL",
+  "dist": "DBIx-Class-Factory",
   "distvname": "DBIx-Class-Factory-TRIAL",
   "extension": "tar.gz",
   "filename": "DBIx-Class-Factory-TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "P/PU/PUSHTAEV/DBIx-Class-Factory-TRIAL.tar.gz",
-  "version": null,
+  "version": "TRIAL",

U/UL/ULIZAMA/CGI-Application-Muto-withoutworldwriteables.tar.gz (2010)
   "cpanid": "ULIZAMA",
-  "dist": "CGI-Application-Muto-withoutworldwriteables",
+  "dist": "CGI-Application-Muto",
   "distvname": "CGI-Application-Muto-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "CGI-Application-Muto-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "U/UL/ULIZAMA/CGI-Application-Muto-withoutworldwriteables.tar.gz",
   "version": null,

U/UG/UGEXE/Perl6/v0.1.30.tar.gz (2017)
   "cpanid": "UGEXE",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.1.30",
   "extension": "tar.gz",
   "filename": "Perl6/v0.1.30.tar.gz",
   "maturity": "released",
   "pathname": "U/UG/UGEXE/Perl6/v0.1.30.tar.gz",
-  "version": "0.1.30",
+  "version": "v0.1.30",

M/MO/MORGOTHII/multiscriptsx.03b.zip (2005)
   "cpanid": "MORGOTHII",
-  "dist": "multiscriptsx.03b",
+  "dist": "multiscriptsx",
   "distvname": "multiscriptsx.03b",
   "extension": "zip",
   "filename": "multiscriptsx.03b.zip",
   "maturity": "released",
   "pathname": "M/MO/MORGOTHII/multiscriptsx.03b.zip",
-  "version": null,
+  "version": "03b",

M/MP/MPERRY/Config-INI-Reader-Encrypted2.tar.gz (2015)
   "cpanid": "MPERRY",
-  "dist": "Config-INI-Reader",
+  "dist": "Config-INI-Reader-Encrypted",
   "distvname": "Config-INI-Reader-Encrypted2",
   "extension": "tar.gz",
   "filename": "Config-INI-Reader-Encrypted2.tar.gz",
   "maturity": "released",
   "pathname": "M/MP/MPERRY/Config-INI-Reader-Encrypted2.tar.gz",
-  "version": "Encrypted2",
+  "version": "2",

M/MA/MARTINB/Xforms4Perl.3.tgz (1996)
   "cpanid": "MARTINB",
   "dist": "Xforms4Perl",
   "distvname": "Xforms4Perl.3",
   "extension": "tgz",
   "filename": "Xforms4Perl.3.tgz",
   "maturity": "released",
   "pathname": "M/MA/MARTINB/Xforms4Perl.3.tgz",
-  "version": ".3",
+  "version": "3",

M/MA/MARTINB/Xforms4Perl.6.tgz (1997)
   "cpanid": "MARTINB",
   "dist": "Xforms4Perl",
   "distvname": "Xforms4Perl.6",
   "extension": "tgz",
   "filename": "Xforms4Perl.6.tgz",
   "maturity": "released",
   "pathname": "M/MA/MARTINB/Xforms4Perl.6.tgz",
-  "version": ".6",
+  "version": "6",

M/MA/MARTINB/Xforms4Perl.5.tgz (1997)
   "cpanid": "MARTINB",
   "dist": "Xforms4Perl",
   "distvname": "Xforms4Perl.5",
   "extension": "tgz",
   "filename": "Xforms4Perl.5.tgz",
   "maturity": "released",
   "pathname": "M/MA/MARTINB/Xforms4Perl.5.tgz",
-  "version": ".5",
+  "version": "5",

M/MA/MAXDB/DBD-MaxDB-7_5_00_26.tar.gz (2005)
   "cpanid": "MAXDB",
   "dist": "DBD-MaxDB",
   "distvname": "DBD-MaxDB-7_5_00_26",
   "extension": "tar.gz",
   "filename": "DBD-MaxDB-7_5_00_26.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "M/MA/MAXDB/DBD-MaxDB-7_5_00_26.tar.gz",
   "version": "7_5_00_26",

M/MA/MARSAB/Bundle-FinalTest3.tar.gz (2011)
   "cpanid": "MARSAB",
-  "dist": "Bundle",
+  "dist": "Bundle-FinalTest",
   "distvname": "Bundle-FinalTest3",
   "extension": "tar.gz",
   "filename": "Bundle-FinalTest3.tar.gz",
   "maturity": "released",
   "pathname": "M/MA/MARSAB/Bundle-FinalTest3.tar.gz",
-  "version": "FinalTest3",
+  "version": "3",

M/MA/MARSAB/Bundle-FinalTest5.tar.gz (2011)
   "cpanid": "MARSAB",
-  "dist": "Bundle",
+  "dist": "Bundle-FinalTest",
   "distvname": "Bundle-FinalTest5",
   "extension": "tar.gz",
   "filename": "Bundle-FinalTest5.tar.gz",
   "maturity": "released",
   "pathname": "M/MA/MARSAB/Bundle-FinalTest5.tar.gz",
-  "version": "FinalTest5",
+  "version": "5",

M/MA/MARSAB/Bundle-FinalTest4.tar.gz (2011)
   "cpanid": "MARSAB",
-  "dist": "Bundle",
+  "dist": "Bundle-FinalTest",
   "distvname": "Bundle-FinalTest4",
   "extension": "tar.gz",
   "filename": "Bundle-FinalTest4.tar.gz",
   "maturity": "released",
   "pathname": "M/MA/MARSAB/Bundle-FinalTest4.tar.gz",
-  "version": "FinalTest4",
+  "version": "4",

M/MA/MARSAB/Test123.pm.tar.gz (2011)
   "cpanid": "MARSAB",
-  "dist": "Test",
+  "dist": "Test123",
   "distvname": "Test123.pm",
   "extension": "tar.gz",
   "filename": "Test123.pm.tar.gz",
   "maturity": "released",
   "pathname": "M/MA/MARSAB/Test123.pm.tar.gz",
-  "version": "123.pm",
+  "version": null,

M/MA/MARSAB/Bundle-FinalTest2.tar.gz (2011)
   "cpanid": "MARSAB",
-  "dist": "Bundle",
+  "dist": "Bundle-FinalTest",
   "distvname": "Bundle-FinalTest2",
   "extension": "tar.gz",
   "filename": "Bundle-FinalTest2.tar.gz",
   "maturity": "released",
   "pathname": "M/MA/MARSAB/Bundle-FinalTest2.tar.gz",
-  "version": "FinalTest2",
+  "version": "2",

M/MA/MARCEL/-0.01.tar.gz (2008)
   "cpanid": "MARCEL",
-  "dist": "-0.01",
+  "dist": "",
   "distvname": "-0.01",
   "extension": "tar.gz",
   "filename": "-0.01.tar.gz",
   "maturity": "released",
   "pathname": "M/MA/MARCEL/-0.01.tar.gz",
-  "version": null,
+  "version": "0.01",

M/ML/MLARUE/0.1_02.zip (2014)
   "cpanid": "MLARUE",
-  "dist": "0.1_02",
+  "dist": "",
   "distvname": "0.1_02",
   "extension": "zip",
   "filename": "0.1_02.zip",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "M/ML/MLARUE/0.1_02.zip",
-  "version": null,
+  "version": "0.1_02",

M/MM/MMUSGROVE/v0.09.tar.gz (2014)
   "cpanid": "MMUSGROVE",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.09",
   "extension": "tar.gz",
   "filename": "v0.09.tar.gz",
   "maturity": "released",
   "pathname": "M/MM/MMUSGROVE/v0.09.tar.gz",
-  "version": "0.09",
+  "version": "v0.09",

M/MM/MMATHEWS/HTML-Entities-Latin2.pm-0.01.tar.gz (2004)
   "cpanid": "MMATHEWS",
-  "dist": "HTML-Entities-Latin",
+  "dist": "HTML-Entities-Latin2",
   "distvname": "HTML-Entities-Latin2.pm-0.01",
   "extension": "tar.gz",
   "filename": "HTML-Entities-Latin2.pm-0.01.tar.gz",
   "maturity": "released",
   "pathname": "M/MM/MMATHEWS/HTML-Entities-Latin2.pm-0.01.tar.gz",
-  "version": "2.pm-0.01",
+  "version": "0.01",

M/MI/MIKEH/mvend203b.tar.gz (1997)
   "cpanid": "MIKEH",
-  "dist": "mvend203b",
+  "dist": "mvend",
   "distvname": "mvend203b",
   "extension": "tar.gz",
   "filename": "mvend203b.tar.gz",
   "maturity": "released",
   "pathname": "M/MI/MIKEH/mvend203b.tar.gz",
-  "version": null,
+  "version": "203b",

M/MI/MIKEB/Spreadsheet-XLSX-0.14_TRIAL.tar.gz (2015)
   "cpanid": "MIKEB",
   "dist": "Spreadsheet-XLSX",
   "distvname": "Spreadsheet-XLSX-0.14_TRIAL",
   "extension": "tar.gz",
   "filename": "Spreadsheet-XLSX-0.14_TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "M/MI/MIKEB/Spreadsheet-XLSX-0.14_TRIAL.tar.gz",
   "version": "0.14_TRIAL",

M/ME/MEWILCOX/apache.authnetldap.018.tar.gz (2000)
   "cpanid": "MEWILCOX",
   "dist": "apache.authnetldap",
   "distvname": "apache.authnetldap.018",
   "extension": "tar.gz",
   "filename": "apache.authnetldap.018.tar.gz",
   "maturity": "released",
   "pathname": "M/ME/MEWILCOX/apache.authnetldap.018.tar.gz",
-  "version": ".018",
+  "version": "018",

M/ME/MEWILCOX/apache.authnetldap.015.tar.gz (2000)
   "cpanid": "MEWILCOX",
   "dist": "apache.authnetldap",
   "distvname": "apache.authnetldap.015",
   "extension": "tar.gz",
   "filename": "apache.authnetldap.015.tar.gz",
   "maturity": "released",
   "pathname": "M/ME/MEWILCOX/apache.authnetldap.015.tar.gz",
-  "version": ".015",
+  "version": "015",

M/ME/MEWILCOX/apache.authznetldap.01.tar.gz (2000)
   "cpanid": "MEWILCOX",
   "dist": "apache.authznetldap",
   "distvname": "apache.authznetldap.01",
   "extension": "tar.gz",
   "filename": "apache.authznetldap.01.tar.gz",
   "maturity": "released",
   "pathname": "M/ME/MEWILCOX/apache.authznetldap.01.tar.gz",
-  "version": ".01",
+  "version": "01",

M/ME/MEWILCOX/apache.authnetldap.016.tar.gz (2000)
   "cpanid": "MEWILCOX",
   "dist": "apache.authnetldap",
   "distvname": "apache.authnetldap.016",
   "extension": "tar.gz",
   "filename": "apache.authnetldap.016.tar.gz",
   "maturity": "released",
   "pathname": "M/ME/MEWILCOX/apache.authnetldap.016.tar.gz",
-  "version": ".016",
+  "version": "016",

M/ME/MEWILCOX/apache.authznetldap.02.tar.gz (2000)
   "cpanid": "MEWILCOX",
   "dist": "apache.authznetldap",
   "distvname": "apache.authznetldap.02",
   "extension": "tar.gz",
   "filename": "apache.authznetldap.02.tar.gz",
   "maturity": "released",
   "pathname": "M/ME/MEWILCOX/apache.authznetldap.02.tar.gz",
-  "version": ".02",
+  "version": "02",

M/ME/MEWILCOX/apache.authznetldap.03.tar.gz (2000)
   "cpanid": "MEWILCOX",
   "dist": "apache.authznetldap",
   "distvname": "apache.authznetldap.03",
   "extension": "tar.gz",
   "filename": "apache.authznetldap.03.tar.gz",
   "maturity": "released",
   "pathname": "M/ME/MEWILCOX/apache.authznetldap.03.tar.gz",
-  "version": ".03",
+  "version": "03",

M/MG/MGV/Gruntmaster-Daemon-5999-TRIAL.tar.gz (2014)
   "cpanid": "MGV",
-  "dist": "Gruntmaster-Daemon-5999-TRIAL",
+  "dist": "Gruntmaster-Daemon",
   "distvname": "Gruntmaster-Daemon-5999-TRIAL",
   "extension": "tar.gz",
   "filename": "Gruntmaster-Daemon-5999-TRIAL.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "M/MG/MGV/Gruntmaster-Daemon-5999-TRIAL.tar.gz",
-  "version": null,
+  "version": "5999-TRIAL",

M/MS/MSPERL/Angle-Omega-withoutworldwriteables.tar.gz (2011)
   "cpanid": "MSPERL",
-  "dist": "Angle-Omega-withoutworldwriteables",
+  "dist": "Angle-Omega",
   "distvname": "Angle-Omega-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "Angle-Omega-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "M/MS/MSPERL/Angle-Omega-withoutworldwriteables.tar.gz",
   "version": null,

M/MS/MSCROGGIN/dbgui_216b.tar.gz (1999)
   "cpanid": "MSCROGGIN",
-  "dist": "dbgui_216b",
+  "dist": "dbgui",
   "distvname": "dbgui_216b",
   "extension": "tar.gz",
   "filename": "dbgui_216b.tar.gz",
   "maturity": "released",
   "pathname": "M/MS/MSCROGGIN/dbgui_216b.tar.gz",
-  "version": null,
+  "version": "216b",

M/MS/MSCROGGIN/dbgui_217b.tar.gz (1999)
   "cpanid": "MSCROGGIN",
-  "dist": "dbgui_217b",
+  "dist": "dbgui",
   "distvname": "dbgui_217b",
   "extension": "tar.gz",
   "filename": "dbgui_217b.tar.gz",
   "maturity": "released",
   "pathname": "M/MS/MSCROGGIN/dbgui_217b.tar.gz",
-  "version": null,
+  "version": "217b",

M/MH/MHCRNL/perl_create_fileV03.zip (2017)
   "cpanid": "MHCRNL",
-  "dist": "perl_create_fileV",
+  "dist": "perl_create_file",
   "distvname": "perl_create_fileV03",
   "extension": "zip",
   "filename": "perl_create_fileV03.zip",
   "maturity": "released",
   "pathname": "M/MH/MHCRNL/perl_create_fileV03.zip",
-  "version": "03",
+  "version": "V03",

M/MH/MHCRNL/CNPclass1.pm.tar.gz (2015)
   "cpanid": "MHCRNL",
-  "dist": "CNPclass",
+  "dist": "CNPclass1",
   "distvname": "CNPclass1.pm",
   "extension": "tar.gz",
   "filename": "CNPclass1.pm.tar.gz",
   "maturity": "released",
   "pathname": "M/MH/MHCRNL/CNPclass1.pm.tar.gz",
-  "version": "1.pm",
+  "version": null,

M/MN/MNIKHIL/0.01.tar.gz (2013)
   "cpanid": "MNIKHIL",
-  "dist": "0.01",
+  "dist": "",
   "distvname": "0.01",
   "extension": "tar.gz",
   "filename": "0.01.tar.gz",
   "maturity": "released",
   "pathname": "M/MN/MNIKHIL/0.01.tar.gz",
-  "version": null,
+  "version": "0.01",

S/SN/SNEEX/HTML_Month.v6a.zip (2002)
   "cpanid": "SNEEX",
-  "dist": "HTML_Month.v6a",
+  "dist": "HTML_Month",
   "distvname": "HTML_Month.v6a",
   "extension": "zip",
   "filename": "HTML_Month.v6a.zip",
   "maturity": "released",
   "pathname": "S/SN/SNEEX/HTML_Month.v6a.zip",
-  "version": null,
+  "version": "v6a",

S/SZ/SZECK/zfilter24b.tar.gz (1997)
   "cpanid": "SZECK",
-  "dist": "zfilter24b",
+  "dist": "zfilter",
   "distvname": "zfilter24b",
   "extension": "tar.gz",
   "filename": "zfilter24b.tar.gz",
   "maturity": "released",
   "pathname": "S/SZ/SZECK/zfilter24b.tar.gz",
-  "version": null,
+  "version": "24b",

S/ST/STEFANOS/Net-SMTP_auth-SSL-withoutworldwriteables.tar.gz (2011)
   "cpanid": "STEFANOS",
-  "dist": "Net-SMTP_auth-SSL-withoutworldwriteables",
+  "dist": "Net-SMTP_auth-SSL",
   "distvname": "Net-SMTP_auth-SSL-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "Net-SMTP_auth-SSL-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "S/ST/STEFANOS/Net-SMTP_auth-SSL-withoutworldwriteables.tar.gz",
   "version": null,

S/SH/SHERZODR/CGI-Session3.tar.gz (2002)
   "cpanid": "SHERZODR",
-  "dist": "CGI",
+  "dist": "CGI-Session",
   "distvname": "CGI-Session3",
   "extension": "tar.gz",
   "filename": "CGI-Session3.tar.gz",
   "maturity": "released",
   "pathname": "S/SH/SHERZODR/CGI-Session3.tar.gz",
-  "version": "Session3",
+  "version": "3",

S/SC/SCHIECHEO/v0.2.4.tar.gz (2015)
   "cpanid": "SCHIECHEO",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.2.4",
   "extension": "tar.gz",
   "filename": "v0.2.4.tar.gz",
   "maturity": "released",
   "pathname": "S/SC/SCHIECHEO/v0.2.4.tar.gz",
-  "version": "0.2.4",
+  "version": "v0.2.4",

S/SC/SCESANO/0.1.tar.gz (2016)
   "cpanid": "SCESANO",
-  "dist": "0.1",
+  "dist": "",
   "distvname": "0.1",
   "extension": "tar.gz",
   "filename": "0.1.tar.gz",
   "maturity": "released",
   "pathname": "S/SC/SCESANO/0.1.tar.gz",
-  "version": null,
+  "version": "0.1",

S/SB/SBALA/jperl_beta_r1.tar.gz (1998)
   "cpanid": "SBALA",
-  "dist": "jperl_beta_r",
+  "dist": "jperl_beta",
   "distvname": "jperl_beta_r1",
   "extension": "tar.gz",
   "filename": "jperl_beta_r1.tar.gz",
   "maturity": "released",
   "pathname": "S/SB/SBALA/jperl_beta_r1.tar.gz",
-  "version": "1",
+  "version": "r1",

S/SE/SEKIMURA/0.05.tar.gz (2013)
   "cpanid": "SEKIMURA",
-  "dist": "0.05",
+  "dist": "",
   "distvname": "0.05",
   "extension": "tar.gz",
   "filename": "0.05.tar.gz",
   "maturity": "released",
   "pathname": "S/SE/SEKIMURA/0.05.tar.gz",
-  "version": null,
+  "version": "0.05",

S/SE/SETITESUK/setitesuk-Test--Data--Structures-v0.02-0-g2b1e961.tar.gz (2010)
   "cpanid": "SETITESUK",
-  "dist": "setitesuk-Test--Data--Structures-v0.02",
+  "dist": "setitesuk-Test--Data--Structures",
   "distvname": "setitesuk-Test--Data--Structures-v0.02-0-g2b1e961",
   "extension": "tar.gz",
   "filename": "setitesuk-Test--Data--Structures-v0.02-0-g2b1e961.tar.gz",
   "maturity": "released",
   "pathname": "S/SE/SETITESUK/setitesuk-Test--Data--Structures-v0.02-0-g2b1e961.tar.gz",
-  "version": "0-g2b1e961",
+  "version": "v0.02-0-g2b1e961",

S/SI/SIMATIKA/subclustv1_0.tar.gz (2004)
   "cpanid": "SIMATIKA",
-  "dist": "subclustv",
+  "dist": "subclust",
   "distvname": "subclustv1_0",
   "extension": "tar.gz",
   "filename": "subclustv1_0.tar.gz",
   "maturity": "released",
   "pathname": "S/SI/SIMATIKA/subclustv1_0.tar.gz",
-  "version": "1_0",
+  "version": "v1_0",

S/SU/SUMMER/Logic3.pm-1.03.tar.gz (1999)
   "cpanid": "SUMMER",
-  "dist": "Logic",
+  "dist": "Logic3",
   "distvname": "Logic3.pm-1.03",
   "extension": "tar.gz",
   "filename": "Logic3.pm-1.03.tar.gz",
   "maturity": "released",
   "pathname": "S/SU/SUMMER/Logic3.pm-1.03.tar.gz",
-  "version": "3.pm-1.03",
+  "version": "1.03",

S/SA/SARGIE/sequin.01.zip (2000)
   "cpanid": "SARGIE",
   "dist": "sequin",
   "distvname": "sequin.01",
   "extension": "zip",
   "filename": "sequin.01.zip",
   "maturity": "released",
   "pathname": "S/SA/SARGIE/sequin.01.zip",
-  "version": ".01",
+  "version": "01",

S/SA/SARGIE/sequin.02.zip (2000)
   "cpanid": "SARGIE",
   "dist": "sequin",
   "distvname": "sequin.02",
   "extension": "zip",
   "filename": "sequin.02.zip",
   "maturity": "released",
   "pathname": "S/SA/SARGIE/sequin.02.zip",
-  "version": ".02",
+  "version": "02",

S/SO/SONNY/DBIx-Class-InflateColumn-S3.tar.gz (2011)
   "cpanid": "SONNY",
-  "dist": "DBIx-Class-InflateColumn",
+  "dist": "DBIx-Class-InflateColumn-S3",
   "distvname": "DBIx-Class-InflateColumn-S3",
   "extension": "tar.gz",
   "filename": "DBIx-Class-InflateColumn-S3.tar.gz",
   "maturity": "released",
   "pathname": "S/SO/SONNY/DBIx-Class-InflateColumn-S3.tar.gz",
-  "version": "S3",
+  "version": null,

S/SO/SOKOLOV/HTML-Macro.1-20.tgz (2002)
   "cpanid": "SOKOLOV",
-  "dist": "HTML-Macro.1",
+  "dist": "HTML-Macro",
   "distvname": "HTML-Macro.1-20",
   "extension": "tgz",
   "filename": "HTML-Macro.1-20.tgz",
   "maturity": "released",
   "pathname": "S/SO/SOKOLOV/HTML-Macro.1-20.tgz",
-  "version": "20",
+  "version": "1-20",

S/SO/SOKOLOV/HTML-Macro.1-14.tgz (2002)
   "cpanid": "SOKOLOV",
-  "dist": "HTML-Macro.1",
+  "dist": "HTML-Macro",
   "distvname": "HTML-Macro.1-14",
   "extension": "tgz",
   "filename": "HTML-Macro.1-14.tgz",
   "maturity": "released",
   "pathname": "S/SO/SOKOLOV/HTML-Macro.1-14.tgz",
-  "version": "14",
+  "version": "1-14",

S/SO/SOKOLOV/HTML-Macro.1-19.tgz (2002)
   "cpanid": "SOKOLOV",
-  "dist": "HTML-Macro.1",
+  "dist": "HTML-Macro",
   "distvname": "HTML-Macro.1-19",
   "extension": "tgz",
   "filename": "HTML-Macro.1-19.tgz",
   "maturity": "released",
   "pathname": "S/SO/SOKOLOV/HTML-Macro.1-19.tgz",
-  "version": "19",
+  "version": "1-19",

S/SO/SOKOLOV/HTML-Macro.1-13.tgz (2001)
   "cpanid": "SOKOLOV",
-  "dist": "HTML-Macro.1",
+  "dist": "HTML-Macro",
   "distvname": "HTML-Macro.1-13",
   "extension": "tgz",
   "filename": "HTML-Macro.1-13.tgz",
   "maturity": "released",
   "pathname": "S/SO/SOKOLOV/HTML-Macro.1-13.tgz",
-  "version": "13",
+  "version": "1-13",

S/SO/SOKOLOV/HTML-Macro.1-18.tgz (2002)
   "cpanid": "SOKOLOV",
-  "dist": "HTML-Macro.1",
+  "dist": "HTML-Macro",
   "distvname": "HTML-Macro.1-18",
   "extension": "tgz",
   "filename": "HTML-Macro.1-18.tgz",
   "maturity": "released",
   "pathname": "S/SO/SOKOLOV/HTML-Macro.1-18.tgz",
-  "version": "18",
+  "version": "1-18",

S/SO/SOKOLOV/HTML-Macro.1-15.tgz (2002)
   "cpanid": "SOKOLOV",
-  "dist": "HTML-Macro.1",
+  "dist": "HTML-Macro",
   "distvname": "HTML-Macro.1-15",
   "extension": "tgz",
   "filename": "HTML-Macro.1-15.tgz",
   "maturity": "released",
   "pathname": "S/SO/SOKOLOV/HTML-Macro.1-15.tgz",
-  "version": "15",
+  "version": "1-15",

S/SO/SOKOLOV/HTML-Macro.1-17.tgz (2002)
   "cpanid": "SOKOLOV",
-  "dist": "HTML-Macro.1",
+  "dist": "HTML-Macro",
   "distvname": "HTML-Macro.1-17",
   "extension": "tgz",
   "filename": "HTML-Macro.1-17.tgz",
   "maturity": "released",
   "pathname": "S/SO/SOKOLOV/HTML-Macro.1-17.tgz",
-  "version": "17",
+  "version": "1-17",

B/BB/BBARKER/0.01.tar.gz (2015)
   "cpanid": "BBARKER",
-  "dist": "0.01",
+  "dist": "",
   "distvname": "0.01",
   "extension": "tar.gz",
   "filename": "0.01.tar.gz",
   "maturity": "released",
   "pathname": "B/BB/BBARKER/0.01.tar.gz",
-  "version": null,
+  "version": "0.01",

B/BD/BDFOY/weblint++-1.15.tar.gz (2007)
   "cpanid": "BDFOY",
-  "dist": "weblint",
+  "dist": "weblint++",
   "distvname": "weblint++-1.15",
   "extension": "tar.gz",
   "filename": "weblint++-1.15.tar.gz",
   "maturity": "released",
   "pathname": "B/BD/BDFOY/weblint++-1.15.tar.gz",
-  "version": "++-1.15",
+  "version": "1.15",

B/BD/BDFOY/weblint++-1.14.tar.gz (2002)
   "cpanid": "BDFOY",
-  "dist": "weblint",
+  "dist": "weblint++",
   "distvname": "weblint++-1.14",
   "extension": "tar.gz",
   "filename": "weblint++-1.14.tar.gz",
   "maturity": "released",
   "pathname": "B/BD/BDFOY/weblint++-1.14.tar.gz",
-  "version": "++-1.14",
+  "version": "1.14",

B/BM/BMIDD/perl5.004_02-AlphaNTPreComp.tar.gz (1997)
   "cpanid": "BMIDD",
   "dist": "perl",
   "distvname": "perl5.004_02-AlphaNTPreComp",
   "extension": "tar.gz",
   "filename": "perl5.004_02-AlphaNTPreComp.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "B/BM/BMIDD/perl5.004_02-AlphaNTPreComp.tar.gz",
   "version": "5.004_02-AlphaNTPreComp",

E/EJ/EJORDAN/v0.2.1.tar.gz (2014)
   "cpanid": "EJORDAN",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.2.1",
   "extension": "tar.gz",
   "filename": "v0.2.1.tar.gz",
   "maturity": "released",
   "pathname": "E/EJ/EJORDAN/v0.2.1.tar.gz",
-  "version": "0.2.1",
+  "version": "v0.2.1",

E/ER/ERWANMAS/v0.10.zip (2018)
   "cpanid": "ERWANMAS",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.10",
   "extension": "zip",
   "filename": "v0.10.zip",
   "maturity": "released",
   "pathname": "E/ER/ERWANMAS/v0.10.zip",
-  "version": "0.10",
+  "version": "v0.10",

E/EL/ELCAMLOST/0.01.tar.gz (2016)
   "cpanid": "ELCAMLOST",
-  "dist": "0.01",
+  "dist": "",
   "distvname": "0.01",
   "extension": "tar.gz",
   "filename": "0.01.tar.gz",
   "maturity": "released",
   "pathname": "E/EL/ELCAMLOST/0.01.tar.gz",
-  "version": null,
+  "version": "0.01",

E/ES/ESTRABD/FLAT-Legacy-FA.1.tgz (2006)
   "cpanid": "ESTRABD",
   "dist": "FLAT-Legacy-FA",
   "distvname": "FLAT-Legacy-FA.1",
   "extension": "tgz",
   "filename": "FLAT-Legacy-FA.1.tgz",
   "maturity": "released",
   "pathname": "E/ES/ESTRABD/FLAT-Legacy-FA.1.tgz",
-  "version": ".1",
+  "version": "1",

E/ES/ESTRABD/0.01.tar.gz (2015)
   "cpanid": "ESTRABD",
-  "dist": "0.01",
+  "dist": "",
   "distvname": "0.01",
   "extension": "tar.gz",
   "filename": "0.01.tar.gz",
   "maturity": "released",
   "pathname": "E/ES/ESTRABD/0.01.tar.gz",
-  "version": null,
+  "version": "0.01",

E/EH/EHOOD/perlSGML.1995Dec08.tar.gz (1995)
   "cpanid": "EHOOD",
-  "dist": "perlSGML.1995Dec",
+  "dist": "perlSGML",
   "distvname": "perlSGML.1995Dec08",
   "extension": "tar.gz",
   "filename": "perlSGML.1995Dec08.tar.gz",
   "maturity": "released",
   "pathname": "E/EH/EHOOD/perlSGML.1995Dec08.tar.gz",
-  "version": "08",
+  "version": "1995Dec08",

E/EH/EHOOD/perlSGML.1997Sep18.tar.gz (1997)
   "cpanid": "EHOOD",
-  "dist": "perlSGML.1997Sep",
+  "dist": "perlSGML",
   "distvname": "perlSGML.1997Sep18",
   "extension": "tar.gz",
   "filename": "perlSGML.1997Sep18.tar.gz",
   "maturity": "released",
   "pathname": "E/EH/EHOOD/perlSGML.1997Sep18.tar.gz",
-  "version": "18",
+  "version": "1997Sep18",

E/EH/EHSAN/Date-Jalali2.tar.gz (2012)
   "cpanid": "EHSAN",
-  "dist": "Date",
+  "dist": "Date-Jalali",
   "distvname": "Date-Jalali2",
   "extension": "tar.gz",
   "filename": "Date-Jalali2.tar.gz",
   "maturity": "released",
   "pathname": "E/EH/EHSAN/Date-Jalali2.tar.gz",
-  "version": "Jalali2",
+  "version": "2",

E/EG/EGOR/Catalyst-Action-SubDomain-withoutworldwriteables.tar.gz (2009)
   "cpanid": "EGOR",
-  "dist": "Catalyst-Action-SubDomain-withoutworldwriteables",
+  "dist": "Catalyst-Action-SubDomain",
   "distvname": "Catalyst-Action-SubDomain-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "Catalyst-Action-SubDomain-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "E/EG/EGOR/Catalyst-Action-SubDomain-withoutworldwriteables.tar.gz",
   "version": null,

E/EV/EVANZS/PDF-CreateSimple2.tar.gz (2005)
   "cpanid": "EVANZS",
-  "dist": "PDF",
+  "dist": "PDF-CreateSimple",
   "distvname": "PDF-CreateSimple2",
   "extension": "tar.gz",
   "filename": "PDF-CreateSimple2.tar.gz",
   "maturity": "released",
   "pathname": "E/EV/EVANZS/PDF-CreateSimple2.tar.gz",
-  "version": "CreateSimple2",
+  "version": "2",

E/EF/EFORGEOT/HTML-WikiConverter-Txt2tags-withoutworldwriteables.tar.gz (2013)
   "cpanid": "EFORGEOT",
-  "dist": "HTML-WikiConverter-Txt2tags-withoutworldwriteables",
+  "dist": "HTML-WikiConverter-Txt2tags",
   "distvname": "HTML-WikiConverter-Txt2tags-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "HTML-WikiConverter-Txt2tags-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "E/EF/EFORGEOT/HTML-WikiConverter-Txt2tags-withoutworldwriteables.tar.gz",
   "version": null,

G/GU/GUIMARD/0.16.tar.gz (2016)
   "cpanid": "GUIMARD",
-  "dist": "0.16",
+  "dist": "",
   "distvname": "0.16",
   "extension": "tar.gz",
   "filename": "0.16.tar.gz",
   "maturity": "released",
   "pathname": "G/GU/GUIMARD/0.16.tar.gz",
-  "version": null,
+  "version": "0.16",

G/GF/GFRANKS/JSON-API-v1.0.3-1.tar.gz (2015)
   "cpanid": "GFRANKS",
-  "dist": "JSON-API-v1.0.3",
+  "dist": "JSON-API",
   "distvname": "JSON-API-v1.0.3-1",
   "extension": "tar.gz",
   "filename": "JSON-API-v1.0.3-1.tar.gz",
   "maturity": "released",
   "pathname": "G/GF/GFRANKS/JSON-API-v1.0.3-1.tar.gz",
-  "version": "1",
+  "version": "v1.0.3-1",

G/GS/GSLONDON/parsenative.070425_1.tar.gz (2007)
   "cpanid": "GSLONDON",
   "dist": "parsenative",
   "distvname": "parsenative.070425_1",
   "extension": "tar.gz",
   "filename": "parsenative.070425_1.tar.gz",
   "maturity": "released",
   "pathname": "G/GS/GSLONDON/parsenative.070425_1.tar.gz",
-  "version": ".070425_1",
+  "version": "070425_1",

G/GS/GSLONDON/iperl.20040205.html.tar.gz (2004)
   "cpanid": "GSLONDON",
   "dist": "iperl",
   "distvname": "iperl.20040205.html",
   "extension": "tar.gz",
   "filename": "iperl.20040205.html.tar.gz",
   "maturity": "released",
   "pathname": "G/GS/GSLONDON/iperl.20040205.html.tar.gz",
-  "version": ".20040205.html",
+  "version": "20040205.html",

G/GS/GSLONDON/iperl.20040205.zip (2004)
   "cpanid": "GSLONDON",
   "dist": "iperl",
   "distvname": "iperl.20040205",
   "extension": "zip",
   "filename": "iperl.20040205.zip",
   "maturity": "released",
   "pathname": "G/GS/GSLONDON/iperl.20040205.zip",
-  "version": ".20040205",
+  "version": "20040205",

G/GS/GSAR/perl5.00401-bindist-bc.tar.gz (1997)
   "cpanid": "GSAR",
   "dist": "perl",
   "distvname": "perl5.00401-bindist-bc",
   "extension": "tar.gz",
   "filename": "perl5.00401-bindist-bc.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "G/GS/GSAR/perl5.00401-bindist-bc.tar.gz",
   "version": "5.00401-bindist-bc",

G/GS/GSAR/perl5.00401-bindist02-bc.tar.gz (1997)
   "cpanid": "GSAR",
   "dist": "perl",
   "distvname": "perl5.00401-bindist02-bc",
   "extension": "tar.gz",
   "filename": "perl5.00401-bindist02-bc.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "G/GS/GSAR/perl5.00401-bindist02-bc.tar.gz",
   "version": "5.00401-bindist02-bc",

G/GS/GSM/SNMP-1.8.1-bin-x86-ASPerl5.005_03-515.zip (1999)
   "cpanid": "GSM",
-  "dist": "SNMP-1.8.1-bin-x86-ASPerl5.005_03",
+  "dist": "SNMP",
   "distvname": "SNMP-1.8.1-bin-x86-ASPerl5.005_03-515",
   "extension": "zip",
   "filename": "SNMP-1.8.1-bin-x86-ASPerl5.005_03-515.zip",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "G/GS/GSM/SNMP-1.8.1-bin-x86-ASPerl5.005_03-515.zip",
-  "version": "515",
+  "version": "1.8.1-bin-x86-ASPerl5.005_03-515",

G/GS/GSM/SNMP-3.1.0-bin-x86-ASPerl5.005_03-515.zip (2000)
   "cpanid": "GSM",
-  "dist": "SNMP-3.1.0-bin-x86-ASPerl5.005_03",
+  "dist": "SNMP",
   "distvname": "SNMP-3.1.0-bin-x86-ASPerl5.005_03-515",
   "extension": "zip",
   "filename": "SNMP-3.1.0-bin-x86-ASPerl5.005_03-515.zip",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "G/GS/GSM/SNMP-3.1.0-bin-x86-ASPerl5.005_03-515.zip",
-  "version": "515",
+  "version": "3.1.0-bin-x86-ASPerl5.005_03-515",

G/GL/GLITCHMR/v1.2.4.tar.gz (2012)
   "cpanid": "GLITCHMR",
-  "dist": "v",
+  "dist": "",
   "distvname": "v1.2.4",
   "extension": "tar.gz",
   "filename": "v1.2.4.tar.gz",
   "maturity": "released",
   "pathname": "G/GL/GLITCHMR/v1.2.4.tar.gz",
-  "version": "1.2.4",
+  "version": "v1.2.4",

J/JG/JGOFF/parrot-0_0_7.tgz (2002)
   "cpanid": "JGOFF",
   "dist": "parrot",
   "distvname": "parrot-0_0_7",
   "extension": "tgz",
   "filename": "parrot-0_0_7.tgz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "J/JG/JGOFF/parrot-0_0_7.tgz",
   "version": "0_0_7",

J/JG/JGNI/Locale-CLDR-v0.26.6-TRIAL-1.tar.gz (2014)
   "cpanid": "JGNI",
-  "dist": "Locale-CLDR-v0.26.6",
+  "dist": "Locale-CLDR",
   "distvname": "Locale-CLDR-v0.26.6-TRIAL-1",
   "extension": "tar.gz",
   "filename": "Locale-CLDR-v0.26.6-TRIAL-1.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "J/JG/JGNI/Locale-CLDR-v0.26.6-TRIAL-1.tar.gz",
-  "version": "TRIAL1",
+  "version": "v0.26.6-TRIAL-1",

J/JG/JGNI/v0.0.1.zip (2014)
   "cpanid": "JGNI",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.0.1",
   "extension": "zip",
   "filename": "v0.0.1.zip",
   "maturity": "released",
   "pathname": "J/JG/JGNI/v0.0.1.zip",
-  "version": "0.0.1",
+  "version": "v0.0.1",

J/JG/JGNI/Locale-CLDR-v0.26.10-1.tar.gz (2015)
   "cpanid": "JGNI",
-  "dist": "Locale-CLDR-v0.26.10",
+  "dist": "Locale-CLDR",
   "distvname": "Locale-CLDR-v0.26.10-1",
   "extension": "tar.gz",
   "filename": "Locale-CLDR-v0.26.10-1.tar.gz",
   "maturity": "released",
   "pathname": "J/JG/JGNI/Locale-CLDR-v0.26.10-1.tar.gz",
-  "version": "1",
+  "version": "v0.26.10-1",

J/JG/JGNI/Locale-CLDR-v0.25.5-TRIAL-1.tar.gz (2014)
   "cpanid": "JGNI",
-  "dist": "Locale-CLDR-v0.25.5",
+  "dist": "Locale-CLDR",
   "distvname": "Locale-CLDR-v0.25.5-TRIAL-1",
   "extension": "tar.gz",
   "filename": "Locale-CLDR-v0.25.5-TRIAL-1.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "J/JG/JGNI/Locale-CLDR-v0.25.5-TRIAL-1.tar.gz",
-  "version": "TRIAL1",
+  "version": "v0.25.5-TRIAL-1",

J/JG/JGNI/Locale-CLDR-v0.26.6-TRIAL-2.tar.gz (2014)
   "cpanid": "JGNI",
-  "dist": "Locale-CLDR-v0.26.6",
+  "dist": "Locale-CLDR",
   "distvname": "Locale-CLDR-v0.26.6-TRIAL-2",
   "extension": "tar.gz",
   "filename": "Locale-CLDR-v0.26.6-TRIAL-2.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "J/JG/JGNI/Locale-CLDR-v0.26.6-TRIAL-2.tar.gz",
-  "version": "TRIAL2",
+  "version": "v0.26.6-TRIAL-2",

J/JG/JGNI/Locale-CLDR-v0.25.5-TRIAL-0.tar.gz (2014)
   "cpanid": "JGNI",
-  "dist": "Locale-CLDR-v0.25.5",
+  "dist": "Locale-CLDR",
   "distvname": "Locale-CLDR-v0.25.5-TRIAL-0",
   "extension": "tar.gz",
   "filename": "Locale-CLDR-v0.25.5-TRIAL-0.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "J/JG/JGNI/Locale-CLDR-v0.25.5-TRIAL-0.tar.gz",
-  "version": "TRIAL0",
+  "version": "v0.25.5-TRIAL-0",

J/JU/JULVR/code-unifdef+-0.005.001.tar.gz (2017)
   "cpanid": "JULVR",
-  "dist": "code-unifdef",
+  "dist": "code-unifdef+",
   "distvname": "code-unifdef+-0.005.001",
   "extension": "tar.gz",
   "filename": "code-unifdef+-0.005.001.tar.gz",
   "maturity": "released",
   "pathname": "J/JU/JULVR/code-unifdef+-0.005.001.tar.gz",
-  "version": "+-0.005.001",
+  "version": "0.005.001",

J/JU/JULVR/Parse/Preprocessor/code-unifdef+-0.005.002.tar.gz (2017)
   "cpanid": "JULVR",
-  "dist": "code-unifdef",
+  "dist": "code-unifdef+",
   "distvname": "code-unifdef+-0.005.002",
   "extension": "tar.gz",
   "filename": "Parse/Preprocessor/code-unifdef+-0.005.002.tar.gz",
   "maturity": "released",
   "pathname": "J/JU/JULVR/Parse/Preprocessor/code-unifdef+-0.005.002.tar.gz",
-  "version": "+-0.005.002",
+  "version": "0.005.002",

J/JW/JWHITE/SlideMap_1_2_1.tar.gz (2005)
   "cpanid": "JWHITE",
   "dist": "SlideMap",
   "distvname": "SlideMap_1_2_1",
   "extension": "tar.gz",
   "filename": "SlideMap_1_2_1.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "J/JW/JWHITE/SlideMap_1_2_1.tar.gz",
   "version": "1_2_1",

J/JW/JWHITE/SlideMap_1_2_2.tar.gz (2005)
   "cpanid": "JWHITE",
   "dist": "SlideMap",
   "distvname": "SlideMap_1_2_2",
   "extension": "tar.gz",
   "filename": "SlideMap_1_2_2.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "J/JW/JWHITE/SlideMap_1_2_2.tar.gz",
   "version": "1_2_2",

J/JW/JWIED/HTML-EP-MSWin32.tar.gz (2001)
   "cpanid": "JWIED",
-  "dist": "HTML-EP",
+  "dist": "HTML-EP-MSWin32",
   "distvname": "HTML-EP-MSWin32",
   "extension": "tar.gz",
   "filename": "HTML-EP-MSWin32.tar.gz",
   "maturity": "released",
   "pathname": "J/JW/JWIED/HTML-EP-MSWin32.tar.gz",
-  "version": "MSWin32",
+  "version": null,

J/JO/JONG/BioV2.0.tar.gz (1998)
   "cpanid": "JONG",
-  "dist": "BioV",
+  "dist": "Bio",
   "distvname": "BioV2.0",
   "extension": "tar.gz",
   "filename": "BioV2.0.tar.gz",
   "maturity": "released",
   "pathname": "J/JO/JONG/BioV2.0.tar.gz",
-  "version": "2.0",
+  "version": "V2.0",

J/JO/JOCASA/futils.02Jun96.zip (1996)
   "cpanid": "JOCASA",
-  "dist": "futils.02Jun",
+  "dist": "futils",
   "distvname": "futils.02Jun96",
   "extension": "zip",
   "filename": "futils.02Jun96.zip",
   "maturity": "released",
   "pathname": "J/JO/JOCASA/futils.02Jun96.zip",
-  "version": "96",
+  "version": "02Jun96",

J/JP/JPRIT/ExtUtils-ExCxx-0.02-5_00456.tar.gz (1998)
   "cpanid": "JPRIT",
   "dist": "ExtUtils-ExCxx",
   "distvname": "ExtUtils-ExCxx-0.02-5_00456",
   "extension": "tar.gz",
   "filename": "ExtUtils-ExCxx-0.02-5_00456.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "J/JP/JPRIT/ExtUtils-ExCxx-0.02-5_00456.tar.gz",
   "version": "0.02-5_00456",

J/JP/JPRIT/ExtUtils-ExCxx-0.04-5_00457.tar.gz (1998)
   "cpanid": "JPRIT",
   "dist": "ExtUtils-ExCxx",
   "distvname": "ExtUtils-ExCxx-0.04-5_00457",
   "extension": "tar.gz",
   "filename": "ExtUtils-ExCxx-0.04-5_00457.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "J/JP/JPRIT/ExtUtils-ExCxx-0.04-5_00457.tar.gz",
   "version": "0.04-5_00457",

J/JP/JPRIT/ExtUtils-ExCxx-0.03-5_00456.tar.gz (1998)
   "cpanid": "JPRIT",
   "dist": "ExtUtils-ExCxx",
   "distvname": "ExtUtils-ExCxx-0.03-5_00456",
   "extension": "tar.gz",
   "filename": "ExtUtils-ExCxx-0.03-5_00456.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "J/JP/JPRIT/ExtUtils-ExCxx-0.03-5_00456.tar.gz",
   "version": "0.03-5_00456",

J/JP/JPRIT/ExtUtils-ExCxx-0.01-5_00456.tar.gz (1998)
   "cpanid": "JPRIT",
   "dist": "ExtUtils-ExCxx",
   "distvname": "ExtUtils-ExCxx-0.01-5_00456",
   "extension": "tar.gz",
   "filename": "ExtUtils-ExCxx-0.01-5_00456.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "J/JP/JPRIT/ExtUtils-ExCxx-0.01-5_00456.tar.gz",
   "version": "0.01-5_00456",

J/JP/JPIERCE/0.11.tgz (2002)
   "cpanid": "JPIERCE",
-  "dist": "0.11",
+  "dist": "",
   "distvname": "0.11",
   "extension": "tgz",
   "filename": "0.11.tgz",
   "maturity": "released",
   "pathname": "J/JP/JPIERCE/0.11.tgz",
-  "version": null,
+  "version": "0.11",

J/JB/JBAKER/perl-5.005_02+apache1.3.3+modperl-1.16-bin-bindist1-i386-win32-vc5.zip (1998)
   "cpanid": "JBAKER",
-  "dist": "perl-5.005_02+apache1.3.3+modperl",
+  "dist": "perl",
   "distvname": "perl-5.005_02+apache1.3.3+modperl-1.16-bin-bindist1-i386-win32-vc5",
   "extension": "zip",
   "filename": "perl-5.005_02+apache1.3.3+modperl-1.16-bin-bindist1-i386-win32-vc5.zip",
   "maturity": "released",
   "pathname": "J/JB/JBAKER/perl-5.005_02+apache1.3.3+modperl-1.16-bin-bindist1-i386-win32-vc5.zip",
-  "version": "1.16-bin-bindist1-i386-win32-vc5",
+  "version": "5.005_02+apache1.3.3+modperl-1.16-bin-bindist1-i386-win32-vc5",

J/JK/JKING/thrift/Thrift-v0.11.0-1.tar.gz (2017)
   "cpanid": "JKING",
-  "dist": "Thrift-v0.11.0",
+  "dist": "Thrift",
   "distvname": "Thrift-v0.11.0-1",
   "extension": "tar.gz",
   "filename": "thrift/Thrift-v0.11.0-1.tar.gz",
   "maturity": "released",
   "pathname": "J/JK/JKING/thrift/Thrift-v0.11.0-1.tar.gz",
-  "version": "1",
+  "version": "v0.11.0-1",

J/JC/JCROCHON/Country-Codes-withoutworldwriteables.tar.gz (2013)
   "cpanid": "JCROCHON",
-  "dist": "Country-Codes-withoutworldwriteables",
+  "dist": "Country-Codes",
   "distvname": "Country-Codes-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "Country-Codes-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "J/JC/JCROCHON/Country-Codes-withoutworldwriteables.tar.gz",
   "version": null,

J/JR/JRUBIN/FT817COMM-0.9.0_18-1.tar.gz (2014)
   "cpanid": "JRUBIN",
-  "dist": "FT817COMM-0.9.0_18",
+  "dist": "FT817COMM",
   "distvname": "FT817COMM-0.9.0_18-1",
   "extension": "tar.gz",
   "filename": "FT817COMM-0.9.0_18-1.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "J/JR/JRUBIN/FT817COMM-0.9.0_18-1.tar.gz",
-  "version": "1",
+  "version": "0.9.0_18-1",

J/JR/JROCKWAY/__-0.01.tar.gz (2008)
   "cpanid": "JROCKWAY",
-  "dist": "__-0.01",
+  "dist": "__",
   "distvname": "__-0.01",
   "extension": "tar.gz",
   "filename": "__-0.01.tar.gz",
   "maturity": "released",
   "pathname": "J/JR/JROCKWAY/__-0.01.tar.gz",
-  "version": null,
+  "version": "0.01",

J/JA/JACOBY/v1.33.tar.gz (2016)
   "cpanid": "JACOBY",
-  "dist": "v",
+  "dist": "",
   "distvname": "v1.33",
   "extension": "tar.gz",
   "filename": "v1.33.tar.gz",
   "maturity": "released",
   "pathname": "J/JA/JACOBY/v1.33.tar.gz",
-  "version": "1.33",
+  "version": "v1.33",

J/JA/JALOPEURA/perl_HaikuKits-0.01_20110820-135621.tar.gz (2011)
   "cpanid": "JALOPEURA",
-  "dist": "perl_HaikuKits-0.01_20110820",
+  "dist": "perl_HaikuKits",
   "distvname": "perl_HaikuKits-0.01_20110820-135621",
   "extension": "tar.gz",
   "filename": "perl_HaikuKits-0.01_20110820-135621.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "J/JA/JALOPEURA/perl_HaikuKits-0.01_20110820-135621.tar.gz",
-  "version": "135621",
+  "version": "0.01_20110820-135621",

J/JA/JAMESR/Dancer-Plugin-SiteMap-withoutworldwriteables.tar.gz (2010)
   "cpanid": "JAMESR",
-  "dist": "Dancer-Plugin-SiteMap-withoutworldwriteables",
+  "dist": "Dancer-Plugin-SiteMap",
   "distvname": "Dancer-Plugin-SiteMap-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "Dancer-Plugin-SiteMap-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "J/JA/JAMESR/Dancer-Plugin-SiteMap-withoutworldwriteables.tar.gz",
   "version": null,

J/JS/JSTOWE/Perl6/v0.0.1.tar.gz (2016)
   "cpanid": "JSTOWE",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.0.1",
   "extension": "tar.gz",
   "filename": "Perl6/v0.0.1.tar.gz",
   "maturity": "released",
   "pathname": "J/JS/JSTOWE/Perl6/v0.0.1.tar.gz",
-  "version": "0.0.1",
+  "version": "v0.0.1",

J/JS/JSTOWE/Perl6/v0.0.1.tar.gzEuclideanRhythm-0.0.1.tar.gz (2016)
   "cpanid": "JSTOWE",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.0.1.tar.gzEuclideanRhythm-0.0.1",
   "extension": "tar.gz",
   "filename": "Perl6/v0.0.1.tar.gzEuclideanRhythm-0.0.1.tar.gz",
   "maturity": "released",
   "pathname": "J/JS/JSTOWE/Perl6/v0.0.1.tar.gzEuclideanRhythm-0.0.1.tar.gz",
-  "version": "0.0.1.tar.gzEuclideanRhythm-0.0.1",
+  "version": "v0.0.1.tar.gzEuclideanRhythm-0.0.1",

J/JI/JIMI/.\Tk-Markdown-withoutworldwriteables.tar.gz (2013)
   "cpanid": "JIMI",
-  "dist": ".\Tk-Markdown-withoutworldwriteables",
+  "dist": "Tk-Markdown",
-  "distvname": ".\Tk-Markdown-withoutworldwriteables",
+  "distvname": "Tk-Markdown-withoutworldwriteables",
   "extension": "tar.gz",
-  "filename": ".\Tk-Markdown-withoutworldwriteables.tar.gz",
+  "filename": "Tk-Markdown-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "J/JI/JIMI/.\Tk-Markdown-withoutworldwriteables.tar.gz",
   "version": null,

J/JI/JIPIPAYO/0.02.tar.gz (2014)
   "cpanid": "JIPIPAYO",
-  "dist": "0.02",
+  "dist": "",
   "distvname": "0.02",
   "extension": "tar.gz",
   "filename": "0.02.tar.gz",
   "maturity": "released",
   "pathname": "J/JI/JIPIPAYO/0.02.tar.gz",
-  "version": null,
+  "version": "0.02",

L/LO/LOVELLE/v0.01.zip (2015)
   "cpanid": "LOVELLE",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.01",
   "extension": "zip",
   "filename": "v0.01.zip",
   "maturity": "released",
   "pathname": "L/LO/LOVELLE/v0.01.zip",
-  "version": "0.01",
+  "version": "v0.01",

L/LB/LBROCARD/perl5.005_03-MAINT22213.tar.gz (2004)
   "cpanid": "LBROCARD",
   "dist": "perl",
   "distvname": "perl5.005_03-MAINT22213",
   "extension": "tar.gz",
   "filename": "perl5.005_03-MAINT22213.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "L/LB/LBROCARD/perl5.005_03-MAINT22213.tar.gz",
   "version": "5.005_03-MAINT22213",

L/LB/LBROCARD/perl5.005_03-MAINT21792.tar.gz (2003)
   "cpanid": "LBROCARD",
   "dist": "perl",
   "distvname": "perl5.005_03-MAINT21792",
   "extension": "tar.gz",
   "filename": "perl5.005_03-MAINT21792.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "L/LB/LBROCARD/perl5.005_03-MAINT21792.tar.gz",
   "version": "5.005_03-MAINT21792",

L/LB/LBROCARD/perl5.005_03-MAINT22178.tar.gz (2004)
   "cpanid": "LBROCARD",
   "dist": "perl",
   "distvname": "perl5.005_03-MAINT22178",
   "extension": "tar.gz",
   "filename": "perl5.005_03-MAINT22178.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "L/LB/LBROCARD/perl5.005_03-MAINT22178.tar.gz",
   "version": "5.005_03-MAINT22178",

L/LM/LMOLNAR/perl5.00503-bin-1-dos-djgpp.zip (1999)
   "cpanid": "LMOLNAR",
   "dist": "perl",
   "distvname": "perl5.00503-bin-1-dos-djgpp",
   "extension": "zip",
   "filename": "perl5.00503-bin-1-dos-djgpp.zip",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "L/LM/LMOLNAR/perl5.00503-bin-1-dos-djgpp.zip",
   "version": "5.00503-bin-1-dos-djgpp",

L/LT/LTBOOTS/test/genomics.01.tar.gz (2005)
   "cpanid": "LTBOOTS",
   "dist": "genomics",
   "distvname": "genomics.01",
   "extension": "tar.gz",
   "filename": "test/genomics.01.tar.gz",
   "maturity": "released",
   "pathname": "L/LT/LTBOOTS/test/genomics.01.tar.gz",
-  "version": ".01",
+  "version": "01",

L/LG/LGODDARD/Tk-Wizard-2.144_win32_gzip1_3_11.tar.gz (2009)
   "cpanid": "LGODDARD",
   "dist": "Tk-Wizard",
   "distvname": "Tk-Wizard-2.144_win32_gzip1_3_11",
   "extension": "tar.gz",
   "filename": "Tk-Wizard-2.144_win32_gzip1_3_11.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "L/LG/LGODDARD/Tk-Wizard-2.144_win32_gzip1_3_11.tar.gz",
   "version": "2.144_win32_gzip1_3_11",

L/LD/LDS/CGI3.pm-3.01.tar.gz (2000)
   "cpanid": "LDS",
-  "dist": "CGI",
+  "dist": "CGI3",
   "distvname": "CGI3.pm-3.01",
   "extension": "tar.gz",
   "filename": "CGI3.pm-3.01.tar.gz",
   "maturity": "released",
   "pathname": "L/LD/LDS/CGI3.pm-3.01.tar.gz",
-  "version": "3.pm-3.01",
+  "version": "3.01",

R/RH/RHARMAN/v0.8.tar.gz (2018)
   "cpanid": "RHARMAN",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.8",
   "extension": "tar.gz",
   "filename": "v0.8.tar.gz",
   "maturity": "released",
   "pathname": "R/RH/RHARMAN/v0.8.tar.gz",
-  "version": "0.8",
+  "version": "v0.8",

R/RE/RENEEB/-0.01.tar.gz (2008)
   "cpanid": "RENEEB",
-  "dist": "-0.01",
+  "dist": "",
   "distvname": "-0.01",
   "extension": "tar.gz",
   "filename": "-0.01.tar.gz",
   "maturity": "released",
   "pathname": "R/RE/RENEEB/-0.01.tar.gz",
-  "version": null,
+  "version": "0.01",

N/NA/NATHANL/Spreadsheet-WriteExcel-WebPivot2.tar.gz (2005)
   "cpanid": "NATHANL",
-  "dist": "Spreadsheet-WriteExcel",
+  "dist": "Spreadsheet-WriteExcel-WebPivot",
   "distvname": "Spreadsheet-WriteExcel-WebPivot2",
   "extension": "tar.gz",
   "filename": "Spreadsheet-WriteExcel-WebPivot2.tar.gz",
   "maturity": "released",
   "pathname": "N/NA/NATHANL/Spreadsheet-WriteExcel-WebPivot2.tar.gz",
-  "version": "WebPivot2",
+  "version": "2",

N/NO/NOBJAS/0.0.3.tar.gz (2009)
   "cpanid": "NOBJAS",
-  "dist": "0.0.3",
+  "dist": "",
   "distvname": "0.0.3",
   "extension": "tar.gz",
   "filename": "0.0.3.tar.gz",
   "maturity": "released",
   "pathname": "N/NO/NOBJAS/0.0.3.tar.gz",
-  "version": null,
+  "version": "0.0.3",

N/NI/NICZERO/niczero-mojar-79e7a4a.tar.gz (2013)
   "cpanid": "NICZERO",
-  "dist": "niczero-mojar-79e7a4a",
+  "dist": "niczero-mojar",
   "distvname": "niczero-mojar-79e7a4a",
   "extension": "tar.gz",
   "filename": "niczero-mojar-79e7a4a.tar.gz",
   "maturity": "released",
   "pathname": "N/NI/NICZERO/niczero-mojar-79e7a4a.tar.gz",
-  "version": null,
+  "version": "79e7a4a",

N/NI/NI-S/perl5.003_02a.tar.gz (1996)
   "cpanid": "NI-S",
   "dist": "perl",
   "distvname": "perl5.003_02a",
   "extension": "tar.gz",
   "filename": "perl5.003_02a.tar.gz",
-  "maturity": "developer",
+  "maturity": "released",
   "pathname": "N/NI/NI-S/perl5.003_02a.tar.gz",
   "version": "5.003_02a",

A/AR/ARJUNS/Dao-Map-Helper-withoutworldwriteables.tar.gz (2011)
   "cpanid": "ARJUNS",
-  "dist": "Dao-Map-Helper-withoutworldwriteables",
+  "dist": "Dao-Map-Helper",
   "distvname": "Dao-Map-Helper-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "Dao-Map-Helper-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "A/AR/ARJUNS/Dao-Map-Helper-withoutworldwriteables.tar.gz",
   "version": null,

A/AR/AREIBENS/PDF-APIx-Layout-0.00_001.MSWin32-5.0.ppm.zip (2005)
   "cpanid": "AREIBENS",
-  "dist": "PDF-APIx-Layout-0.00_001.MSWin32",
+  "dist": "PDF-APIx-Layout",
   "distvname": "PDF-APIx-Layout-0.00_001.MSWin32-5.0.ppm",
   "extension": "zip",
   "filename": "PDF-APIx-Layout-0.00_001.MSWin32-5.0.ppm.zip",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "A/AR/AREIBENS/PDF-APIx-Layout-0.00_001.MSWin32-5.0.ppm.zip",
-  "version": "5.0.ppm",
+  "version": "0.00_001.MSWin32-5.0.ppm",

A/AR/AREIBENS/PDF-Maki-0.00_04.MSWin32-4.0.ppm.zip (2005)
   "cpanid": "AREIBENS",
-  "dist": "PDF-Maki-0.00_04.MSWin32",
+  "dist": "PDF-Maki",
   "distvname": "PDF-Maki-0.00_04.MSWin32-4.0.ppm",
   "extension": "zip",
   "filename": "PDF-Maki-0.00_04.MSWin32-4.0.ppm.zip",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "A/AR/AREIBENS/PDF-Maki-0.00_04.MSWin32-4.0.ppm.zip",
-  "version": "4.0.ppm",
+  "version": "0.00_04.MSWin32-4.0.ppm",

A/AR/ARUN/1.6.tar.gz (2016)
   "cpanid": "ARUN",
-  "dist": "1.6",
+  "dist": "",
   "distvname": "1.6",
   "extension": "tar.gz",
   "filename": "1.6.tar.gz",
   "maturity": "released",
   "pathname": "A/AR/ARUN/1.6.tar.gz",
-  "version": null,
+  "version": "1.6",

A/AU/AUTOLIFE/Mojo-UserAgent_t2.tar.gz (2016)
   "cpanid": "AUTOLIFE",
-  "dist": "Mojo",
+  "dist": "Mojo-UserAgent",
   "distvname": "Mojo-UserAgent_t2",
   "extension": "tar.gz",
   "filename": "Mojo-UserAgent_t2.tar.gz",
   "maturity": "released",
   "pathname": "A/AU/AUTOLIFE/Mojo-UserAgent_t2.tar.gz",
-  "version": "UserAgent_t2",
+  "version": "t2",

A/AD/ADARSHTP/-withoutworldwriteables.tar.gz (2010)
   "cpanid": "ADARSHTP",
-  "dist": "-withoutworldwriteables",
+  "dist": "",
   "distvname": "-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "A/AD/ADARSHTP/-withoutworldwriteables.tar.gz",
   "version": null,

A/AP/APEIRON/-0.01.tar.gz (2011)
   "cpanid": "APEIRON",
-  "dist": "-0.01",
+  "dist": "",
   "distvname": "-0.01",
   "extension": "tar.gz",
   "filename": "-0.01.tar.gz",
   "maturity": "released",
   "pathname": "A/AP/APEIRON/-0.01.tar.gz",
-  "version": null,
+  "version": "0.01",

A/AQ/AQUMSIEH/ZooZv0.9a.zip (2004)
   "cpanid": "AQUMSIEH",
-  "dist": "ZooZv",
+  "dist": "ZooZ",
   "distvname": "ZooZv0.9a",
   "extension": "zip",
   "filename": "ZooZv0.9a.zip",
   "maturity": "released",
   "pathname": "A/AQ/AQUMSIEH/ZooZv0.9a.zip",
-  "version": "0.9a",
+  "version": "v0.9a",

A/AQ/AQUMSIEH/ZooZv1.0-RC1.tar.gz (2004)
   "cpanid": "AQUMSIEH",
-  "dist": "ZooZv",
+  "dist": "ZooZ",
   "distvname": "ZooZv1.0-RC1",
   "extension": "tar.gz",
   "filename": "ZooZv1.0-RC1.tar.gz",
   "maturity": "released",
   "pathname": "A/AQ/AQUMSIEH/ZooZv1.0-RC1.tar.gz",
-  "version": "1.0-RC1",
+  "version": "v1.0-RC1",

A/AH/AHERNIT/Class-Injection-withoutworldwriteables.tar.gz (2010)
   "cpanid": "AHERNIT",
-  "dist": "Class-Injection-withoutworldwriteables",
+  "dist": "Class-Injection",
   "distvname": "Class-Injection-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "Class-Injection-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "A/AH/AHERNIT/Class-Injection-withoutworldwriteables.tar.gz",
   "version": null,

A/AN/ANANTHBV/OpenXML-Properties-withoutworldwriteables.tar.gz (2012)
   "cpanid": "ANANTHBV",
-  "dist": "OpenXML-Properties-withoutworldwriteables",
+  "dist": "OpenXML-Properties",
   "distvname": "OpenXML-Properties-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "OpenXML-Properties-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "A/AN/ANANTHBV/OpenXML-Properties-withoutworldwriteables.tar.gz",
   "version": null,

A/AN/ANDK/Memo-bindist-any-bin-2-archname-compiler.tar.gz (1998)
   "cpanid": "ANDK",
-  "dist": "Memo-bindist-any-bin-2-archname-compiler",
+  "dist": "Memo-bindist-any-bin",
   "distvname": "Memo-bindist-any-bin-2-archname-compiler",
   "extension": "tar.gz",
   "filename": "Memo-bindist-any-bin-2-archname-compiler.tar.gz",
   "maturity": "released",
   "pathname": "A/AN/ANDK/Memo-bindist-any-bin-2-archname-compiler.tar.gz",
-  "version": null,
+  "version": "2-archname-compiler",

A/AN/ANDK/Memo-readme-bin-01-processor-os-compiler.tar.gz (1998)
   "cpanid": "ANDK",
-  "dist": "Memo-readme-bin-01-processor-os-compiler",
+  "dist": "Memo-readme-bin",
   "distvname": "Memo-readme-bin-01-processor-os-compiler",
   "extension": "tar.gz",
   "filename": "Memo-readme-bin-01-processor-os-compiler.tar.gz",
   "maturity": "released",
   "pathname": "A/AN/ANDK/Memo-readme-bin-01-processor-os-compiler.tar.gz",
-  "version": null,
+  "version": "01-processor-os-compiler",

A/AN/ANDYD/mc_units-5.004_70-01.tar.gz (1998)
   "cpanid": "ANDYD",
-  "dist": "mc_units-5.004_70",
+  "dist": "mc_units",
   "distvname": "mc_units-5.004_70-01",
   "extension": "tar.gz",
   "filename": "mc_units-5.004_70-01.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "A/AN/ANDYD/mc_units-5.004_70-01.tar.gz",
-  "version": "01",
+  "version": "5.004_70-01",

A/AN/ANDYD/5.003_07-2.U.tar.gz (1996)
   "cpanid": "ANDYD",
-  "dist": "5.003_07-2.U",
+  "dist": "",
   "distvname": "5.003_07-2.U",
   "extension": "tar.gz",
   "filename": "5.003_07-2.U.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "A/AN/ANDYD/5.003_07-2.U.tar.gz",
-  "version": null,
+  "version": "5.003_07-2.U",

A/AN/ANDYD/mc_units-5.005_00-01.tar.gz (1998)
   "cpanid": "ANDYD",
-  "dist": "mc_units-5.005_00",
+  "dist": "mc_units",
   "distvname": "mc_units-5.005_00-01",
   "extension": "tar.gz",
   "filename": "mc_units-5.005_00-01.tar.gz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "A/AN/ANDYD/mc_units-5.005_00-01.tar.gz",
-  "version": "01",
+  "version": "5.005_00-01",

A/AN/ANDYJONES/v0.50.tar.gz (2015)
   "cpanid": "ANDYJONES",
-  "dist": "v",
+  "dist": "",
   "distvname": "v0.50",
   "extension": "tar.gz",
   "filename": "v0.50.tar.gz",
   "maturity": "released",
   "pathname": "A/AN/ANDYJONES/v0.50.tar.gz",
-  "version": "0.50",
+  "version": "v0.50",

A/AT/ATTILA/0.1.0.tar.gz (2017)
   "cpanid": "ATTILA",
-  "dist": "0.1.0",
+  "dist": "",
   "distvname": "0.1.0",
   "extension": "tar.gz",
   "filename": "0.1.0.tar.gz",
   "maturity": "released",
   "pathname": "A/AT/ATTILA/0.1.0.tar.gz",
-  "version": null,
+  "version": "0.1.0",

A/AL/ALLEN/0.03.tar.gz (2016)
   "cpanid": "ALLEN",
-  "dist": "0.03",
+  "dist": "",
   "distvname": "0.03",
   "extension": "tar.gz",
   "filename": "0.03.tar.gz",
   "maturity": "released",
   "pathname": "A/AL/ALLEN/0.03.tar.gz",
-  "version": null,
+  "version": "0.03",

A/AL/ALFIE/MySQL--Replication-5953e9a.tar.gz (2011)
   "cpanid": "ALFIE",
-  "dist": "MySQL--Replication-5953e9a",
+  "dist": "MySQL--Replication",
   "distvname": "MySQL--Replication-5953e9a",
   "extension": "tar.gz",
   "filename": "MySQL--Replication-5953e9a.tar.gz",
   "maturity": "released",
   "pathname": "A/AL/ALFIE/MySQL--Replication-5953e9a.tar.gz",
-  "version": null,
+  "version": "5953e9a",

A/AB/ABERNDT/Win32-Snarl-withoutworldwriteables.tar.gz (2010)
   "cpanid": "ABERNDT",
-  "dist": "Win32-Snarl-withoutworldwriteables",
+  "dist": "Win32-Snarl",
   "distvname": "Win32-Snarl-withoutworldwriteables",
   "extension": "tar.gz",
   "filename": "Win32-Snarl-withoutworldwriteables.tar.gz",
   "maturity": "released",
   "pathname": "A/AB/ABERNDT/Win32-Snarl-withoutworldwriteables.tar.gz",
   "version": null,

A/AC/ACCARDO/1.65.tar.gz (2015)
   "cpanid": "ACCARDO",
-  "dist": "1.65",
+  "dist": "",
   "distvname": "1.65",
   "extension": "tar.gz",
   "filename": "1.65.tar.gz",
   "maturity": "released",
   "pathname": "A/AC/ACCARDO/1.65.tar.gz",
-  "version": null,
+  "version": "1.65",

W/WA/WANGQ/1.0.2.tar.gz (2016)
   "cpanid": "WANGQ",
-  "dist": "1.0.2",
+  "dist": "",
   "distvname": "1.0.2",
   "extension": "tar.gz",
   "filename": "1.0.2.tar.gz",
   "maturity": "released",
   "pathname": "W/WA/WANGQ/1.0.2.tar.gz",
-  "version": null,
+  "version": "1.0.2",

K/KI/KIAVASH/0.05.tar.gz (2017)
   "cpanid": "KIAVASH",
-  "dist": "0.05",
+  "dist": "",
   "distvname": "0.05",
   "extension": "tar.gz",
   "filename": "0.05.tar.gz",
   "maturity": "released",
   "pathname": "K/KI/KIAVASH/0.05.tar.gz",
-  "version": null,
+  "version": "0.05",

C/CB/CBAIL/perl5_003.tar-gz (1996)
   "cpanid": "CBAIL",
-  "dist": null,
+  "dist": "perl",
   "filename": "perl5_003.tar-gz",
   "maturity": "released",
   "pathname": "C/CB/CBAIL/perl5_003.tar-gz",
-  "version": null,
+  "version": "5_003",

L/LM/LMOLNAR/perl542b.zip (1997)
   "cpanid": "LMOLNAR",
-  "dist": "perl542b",
+  "dist": "perl",
   "distvname": "perl542b",
   "extension": "zip",
   "filename": "perl542b.zip",
   "maturity": "released",
   "pathname": "L/LM/LMOLNAR/perl542b.zip",
-  "version": null,
+  "version": "542b",

H/HM/HMBRAND/mc_units-20060409.tbz (2006)
   "cpanid": "HMBRAND",
-  "dist": null,
+  "dist": "mc_units",
   "filename": "mc_units-20060409.tbz",
   "maturity": "released",
   "pathname": "H/HM/HMBRAND/mc_units-20060409.tbz",
-  "version": null,
+  "version": "20060409",

H/HM/HMBRAND/mc_units-20061116.tbz (2006)
   "cpanid": "HMBRAND",
-  "dist": null,
+  "dist": "mc_units",
   "filename": "mc_units-20061116.tbz",
   "maturity": "released",
   "pathname": "H/HM/HMBRAND/mc_units-20061116.tbz",
-  "version": null,
+  "version": "20061116",

H/HM/HMBRAND/DBD-CSV-0.24_02.tbz (2009)
   "cpanid": "HMBRAND",
-  "dist": null,
+  "dist": "DBD-CSV",
   "filename": "DBD-CSV-0.24_02.tbz",
-  "maturity": "released",
+  "maturity": "developer",
   "pathname": "H/HM/HMBRAND/DBD-CSV-0.24_02.tbz",
-  "version": null,
+  "version": "0.24_02",

H/HM/HMBRAND/mc_units-20050201.tbz (2005)
   "cpanid": "HMBRAND",
-  "dist": null,
+  "dist": "mc_units",
   "filename": "mc_units-20050201.tbz",
   "maturity": "released",
   "pathname": "H/HM/HMBRAND/mc_units-20050201.tbz",
-  "version": null,
+  "version": "20050201",

C/CO/COVINGTON/Log-Reproducible-v0.12.1_a.tar_gz (2014)
   "cpanid": "COVINGTON",
-  "dist": null,
+  "dist": "Log-Reproducible",
   "filename": "Log-Reproducible-v0.12.1_a.tar_gz",
   "maturity": "released",
   "pathname": "C/CO/COVINGTON/Log-Reproducible-v0.12.1_a.tar_gz",
-  "version": null,
+  "version": "v0.12.1_a",

C/CB/CBAIL/perl5_002_01.tar-gz (1996)
   "cpanid": "CBAIL",
-  "dist": null,
+  "dist": "perl",
   "filename": "perl5_002_01.tar-gz",
   "maturity": "released",
   "pathname": "C/CB/CBAIL/perl5_002_01.tar-gz",
-  "version": null,
+  "version": "5_002_01",

C/CB/CBAIL/perl5_003_01.tar-gz (1996)
   "cpanid": "CBAIL",
-  "dist": null,
+  "dist": "perl",
   "filename": "perl5_003_01.tar-gz",
   "maturity": "released",
   "pathname": "C/CB/CBAIL/perl5_003_01.tar-gz",
-  "version": null,
+  "version": "5_003_01",

P/PF/PFAUT/vms-logical-0_4.tar-gz (2009)
   "cpanid": "PFAUT",
-  "dist": null,
+  "dist": "vms-logical",
   "filename": "vms-logical-0_4.tar-gz",
   "maturity": "released",
   "pathname": "P/PF/PFAUT/vms-logical-0_4.tar-gz",
-  "version": null,
+  "version": "0_4",

