;;; $Id: irchat-handle.el,v 1.13 1994/08/02 12:51:15 tmo Exp $
;;;
;;; see file irchat-copyright.el for change log and copyright in(eval-wfo

(eval-when-compile
  (require 'irchat-inlines)
  (require 'irchat-filter))

(defun irchat-handle-error-msg (prefix rest)
  (message "IRC error: %s" rest))


(defun irchat-handle-channel-msg (prefix rest)
  (if (string= prefix irchat-nickname)
      (progn
	(setq irchat-current-channel rest)
	(setq irchat-channel-indicator
	      (if (string= rest "0")
		  "No channel"
		(format "Channel %s" rest)))))
  (if (string= rest "0")
      (insert (format "%s%s has left this channel\n" 
		      irchat-change-prefix prefix))
    (if (not irchat-ignore-changes)
	(insert (format "%s%s has joined this channel\n" 
			irchat-change-prefix prefix)))
    (irchat-change-nick-of prefix prefix)))


(defun irchat-handle-nick-msg (prefix rest)
  (irchat-change-nick-of prefix rest)
  (put (intern rest) 'chnl (get (intern prefix) 'chnl))
  (put (intern prefix) 'chnl nil)
  (if (string= prefix irchat-nickname)
      (setq irchat-nickname rest))
  (insert (format "%s%s is now known as %s" irchat-change-prefix prefix rest))
  (newline))


(defun irchat-handle-notice-msg (prefix rest)
  (if (and irchat-shorten-kills
	   (string-match 
	    "Notice[: -]*Received KILL message for \\([^.]*\\)\\. From \\([^ ]*\\) Path: \\([^ ]*\\) ?\\(.*\\)"
	    rest))
      (let ((killed (matching-substring rest 1))
	    (killer (matching-substring rest 2))
	    (reason (matching-substring rest 4))
	    (buf (current-buffer)))
	(set-buffer irchat-KILLS-buffer)
	(goto-char (point-max))
	(insert (format "%s\n" rest))
	(set-buffer buf)
	(insert (format "%s%s KILLed %s %s\n" irchat-notice-prefix
			killer killed (if (= (length reason) 0)
					  "-No reason supplied-"
					reason))))

    (if (or (not irchat-ignore-extra-notices)
	    (not prefix)
	    (not (string-match "as being away" rest)))
	(if prefix
	    (cond 
	     ((string-match "\\(.*\\)" rest) ; prefixed clt-a-notice
	      (irchat-ctl-a-notice prefix rest))
	     ((and irchat-ignore-fakes
		   (string-match ".*Notice.*Fake:.*" rest))
	      t)
	     ((string-match ".*Notice -- \\(.*\\)" rest) ; not a clt-a, but notice
	      (insert (format "%s%s: %s\n" irchat-notice-prefix
			      prefix (matching-substring rest 1))))
	     (t ; else send user a private message
	      (irchat-handle-privmsg-msg prefix rest)))
	  (progn
	    ;; no prefix
	    (string-match "^\\([^ ]*\\) :\\(.*\\)" rest)
	    (insert (format "%s%s" irchat-notice-prefix 
			    (matching-substring rest 2)))
	    (newline))))))


(defun irchat-handle-ping-msg (prefix rest)
  (irchat-send "PONG yourself")
  (irchat-maybe-poll))


(defun irchat-handle-pong-msg (prefix rest)
  ())


(defun irchat-handle-privmsg-msg (prefix rest)
  (if (and prefix
	   (memq (intern prefix) irchat-kill-nickname)
	   (irchat-msg-from-ignored prefix rest))
      nil
    (if (and (string-match "\007" rest) irchat-beep-on-bells)
	(beep t))
    (string-match "^\\([^ ]+\\) :\\(.*\\)" rest)
    (let ((chnl (matching-substring rest 1))
	  (temp (matching-substring rest 2))
	  (case-fold-search t))
      (if (string-match "\\(.*\\)" temp)
	  (setq temp (irchat-ctl-a-msg prefix temp)))
      (if (not (string= temp ""))
	  (progn
	    (insert
	     (cond
	      ((string-ci-equal chnl irchat-nickname)
	       (format irchat-format-string1 prefix))

	      ((string-ci-equal chnl (or irchat-current-channel ""))
	       (if (irchat-user-on-this-channel prefix chnl)
		   (format irchat-format-string2 prefix)
		 (format irchat-format-string4 prefix)))

	      (t ;; channel we are joined (not current)
	       (if (irchat-user-on-this-channel prefix chnl) 
		   (format irchat-format-string3 prefix chnl)
		 (format irchat-format-string5 prefix chnl)))))

	    (insert " " temp)
	    (or (irchat-get-buffer-window (current-buffer))
		(not (string-ci-equal chnl irchat-nickname))
		(message "IRCHAT: A private message has arrived from %s" 
			 prefix))
	    ;; only private messages to us get time-stamp
	    (if (and (string-equal "A" irchat-away-indicator) 
		     (string-ci-equal chnl irchat-nickname))
		(insert (format " (%s)" (current-time-string))))
	    (newline))))))


(defun irchat-handle-wall-msg (prefix rest)
  "Handle the WALL message."
  (insert (format "%s%s %s" 
		  irchat-broadcast-prefix
		  (if prefix (concat "from " prefix) "")
		  rest))
  (newline))


(defun irchat-handle-wallops-msg (prefix rest)
  "Handle the WALLOPS message."
  (if irchat-show-wallops
      (insert (format "%s%s: %s\n" irchat-wallops-prefix 
		      (if prefix prefix "UNKNOWN") rest)))
  (let ((buf (current-buffer)))
    (set-buffer irchat-WALLOPS-buffer)
    (goto-char (point-max))
    (insert (format "%s%s %s\n" irchat-wallops-prefix 
		    (if prefix (concat "from " prefix) "") rest))
    (set-buffer buf)))


(defun irchat-handle-quit-msg (prefix rest)
  "Handle the QUIT message."
  (if (not irchat-ignore-changes)
      (insert (format "%s%s has left IRC %s\n" irchat-change-prefix prefix
		      (if (= 0 (length rest))
			  ""
			(format "(%s)" rest)))))
  (irchat-change-nick-of prefix nil))


(defun irchat-handle-topic-msg (prefix rest)
  "Handle the TOPIC message."
  (if (string-match "\\([^ :]*\\)[: ]*\\(.*\\)" rest)
      (let ((chnl (matching-substring rest 1))
	    (topic (matching-substring rest 2)))
	(if (not irchat-ignore-changes)
	    (progn
	      (if (string-equal chnl irchat-current-channel)
		  (insert (format "%sNew topic on this channel set by %s: %s"
				  irchat-change-prefix prefix topic))
		(insert (format "%sNew topic on channel %s set by %s: %s"
				irchat-change-prefix chnl prefix topic)))
	      (newline))))
    (message "IRCHAT: Strange TOPIC")))


(defun irchat-handle-mode-msg (prefix rest)
  "Handle the MODE message."
  (if (not irchat-ignore-changes)
      (let ((chnl " ") (str ""))
	(if (or (and (string-match "\\([^ ]*\\) \\(.*\\)" rest)
		     (setq chnl (matching-substring rest 1)
			   str (matching-substring rest 2)
			   chnl (if (string-equal chnl irchat-current-channel)
				    " "
				  chnl)))
		(and (string-match " \\(.*\\)" rest) 
		     (setq str (matching-substring rest 1))))
	    (insert (format "%sNew mode%sset by %s: %s\n" irchat-change-prefix
			    (if (string= chnl " ")
				chnl
			      (concat " for " chnl " "))
			    prefix str))
	  (message "IRCHAT: Strange MODE")))))


(defun irchat-handle-kick-msg (prefix rest)
  "Handle the KICK message."
  (if (string-match "^\\([^ ]*\\) \\([^ ]*\\)" rest)
      (let ((match1 (matching-substring rest 1))
	    (match2 (matching-substring rest 2)))
	(if (string= match2 irchat-nickname)
	  (progn
	    (insert (format "%sYou were kicked off channel %s by %s.\n"
			    irchat-change-prefix
			    match1
			    prefix))
	    (setq 
	     irchat-current-channels (string-list-ci-delete 
				      irchat-current-channel
				      irchat-current-channels)
	     irchat-current-channel (car irchat-current-channels)
	     irchat-channel-indicator (if irchat-current-channel
					  (format "Channel %s" 
						  irchat-current-channel)
					"No channel"))
	    (irchat-remove-from-thischannel irchat-nickname 
					    match1))
	  (insert (format "%s%s has kicked %s out%s\n" irchat-change-prefix 
			  prefix match2
			  (if (string= (or irchat-current-channel "")
				       match1)
			      ""
			    (format " from channel %s" match1))))))
    (message "IRCHAT: Strange KICK.")))


(defun irchat-handle-invite-msg (prefix rest)
  (if (string-match " :\\([^ ]+\\)" rest)
      (let ((chnl (matching-substring rest 1)))
	(insert (format "*** %s invites you to channel %s\n" prefix chnl))
	(setq irchat-invited-channel chnl))
    (message "IRCHAT: Strange INVITE")))


(defun irchat-handle-kill-msg (prefix rest)
  (if (string-match "[^ ]+ +:\\(.*\\)" rest)
      (let ((path (matching-substring rest 1)))
	(insert (format 
		 "*** IRCHAT: You were killed by %s. Path: %s. RIP" 
		 prefix path))
	(newline))
    (message "IRCHAT: strange KILL"))
  (setq irchat-channel-indicator "No channel"))


(defun irchat-handle-join-msg (prefix rest) ; kmk, 14101990
  (if (string-match "\\([^ ]*\\)\ .*" rest)
      (setq rest (matching-substring rest 1))) ;; throw away the channel mode
  (if (string= prefix irchat-nickname)
      (progn
	(setq irchat-current-channel rest)
	(setq irchat-current-channels
	      (cons irchat-current-channel irchat-current-channels))
 	(setq irchat-channel-indicator
	      (format "Channel %s" rest)))
    (irchat-add-to-channel prefix rest))
  (if (not irchat-ignore-changes)
      (insert (format 
	       "%s%s (%s) has joined this channel%s\n" 
	       irchat-change-prefix prefix irchat-userathost
	       (if (string= (or irchat-current-channel "") rest) ""
		 (format " (%s)" rest)))))
  (irchat-change-nick-of prefix prefix))


(defun irchat-handle-part-msg (prefix rest) ; kmk, 14101990
  (if (string= prefix irchat-nickname)
      (setq irchat-current-channels
	    (string-list-ci-delete rest irchat-current-channels)
	    irchat-current-channel (car irchat-current-channels)
	    irchat-channel-indicator (if irchat-current-channel
					 (format "Channel %s" 
						 irchat-current-channel)
				       "No channel")))
  (if (not irchat-ignore-changes)
      (insert (format 
	       "%s%s has left this channel%s\n"
	       irchat-change-prefix prefix
	       (if (string= (or irchat-current-channel "") rest) ""
		 (format " (%s)" rest)))))
  (irchat-remove-from-thischannel prefix rest)
  (irchat-change-nick-of prefix prefix))

;;;
;;; eof
;;;
