/*
 * Copyright(c) 1992 Bell Communications Research, Inc. (Bellcore)
 *                        All rights reserved
 * Permission to use, copy, modify and distribute this material for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of Bellcore not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * Bellcore.
 *
 * BELLCORE MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL BELLCORE OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES RELAT-
 * ING TO THE SOFTWARE.
 *
 * MatrixWidget Author: Andrew Wason, Bellcore, aw@bae.bellcore.com
 */

#include <Xm/Xm.h>
#include <Xm/ScrollBar.h>
#include <Xbae/MatrixP.h>
#include <Xbae/Shadow.h>
#include <Xbae/Draw.h>
#include <Xbae/ScrollMgr.h>
#include <Xbae/Actions.h>
#include <Xbae/Utils.h>
#include <Xbae/Clip.h>

/*
 * Public.c created by Andrew Lister (7 August, 1995)
 */

/*
 * Public interface to set_cell method
 */
void
XbaeMatrixSetCell(w, row, column, value )
Widget w;
int row;
int column;
const String value;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the set_cell method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_cell)
	((XbaeMatrixWidget) w, row, column, value, True);
}


/*
 * Public interface to edit_cell method
 */
void
XbaeMatrixEditCell(w, row, column)
Widget w;
int row, column;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the edit_cell method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.edit_cell)
	((XbaeMatrixWidget) w, row, column);

    XmProcessTraversal(TextChild(((XbaeMatrixWidget) w)), XmTRAVERSE_CURRENT);
}

/*
 * Public interface to select_cell method
 */
void
XbaeMatrixSelectCell(w, row, column)
Widget w;
int row, column;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the select_cell method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.select_cell)
	((XbaeMatrixWidget) w, row, column);
}

/*
 * Public interface to select_row method
 */
void
XbaeMatrixSelectRow(w, row)
Widget w;
int row;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the select_row method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.select_row)
	((XbaeMatrixWidget) w, row);
}

/*
 * Public interface to select_column method
 */
void
XbaeMatrixSelectColumn(w, column)
Widget w;
int column;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the select_column method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.select_column)
	((XbaeMatrixWidget) w, column);
}

/*
 * Public interface to deselect_all method
 */
void
XbaeMatrixDeselectAll(w)
Widget w;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
     * Call the deselect_all method
     */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.deselect_all)
	((XbaeMatrixWidget) w);
}

/*
 * Public interface to select_all method
 */
void
XbaeMatrixSelectAll(w)
Widget w;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
     * Call the deselect_all method
     */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.select_all)
	((XbaeMatrixWidget) w);
}

/*
 * Public interface to deselect_cell method
 */
void
XbaeMatrixDeselectCell(w, row, column)
Widget w;
int row;
int column;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
     * Call the deselect_cell method
     */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.deselect_cell)
	((XbaeMatrixWidget) w, row, column);
}

/*
 * Public interface to deselect_row method
 */
void
XbaeMatrixDeselectRow(w, row)
Widget w;
int row;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the deselect_row method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.deselect_row)
	((XbaeMatrixWidget) w, row);
}

/*
 * Public interface to deselect_column method
 */
void
XbaeMatrixDeselectColumn(w, column)
Widget w;
int column;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the deselect_column method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.deselect_column)
	((XbaeMatrixWidget) w, column);
}

/*
 * Public interface to get_cell method
 */
String
XbaeMatrixGetCell(w, row, column)
Widget w;
int row, column;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the get_cell method
   */
    return (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.get_cell)
	((XbaeMatrixWidget) w, row, column);
}

/*
 * Public interface to commit_edit method
 */
Boolean
#if NeedFunctionPrototypes
XbaeMatrixCommitEdit(Widget w, Boolean unmap)
#else
XbaeMatrixCommitEdit(w, unmap)
Widget w;
Boolean unmap;
#endif
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the commit_edit method
   */
    return (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.commit_edit)
	((XbaeMatrixWidget) w, unmap);
}

/*
 * Public interface to cancel_edit method
 */
void
#if NeedFunctionPrototypes
XbaeMatrixCancelEdit(Widget w, Boolean unmap)
#else
XbaeMatrixCancelEdit(w, unmap)
Widget w;
Boolean unmap;
#endif
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the cancel_edit method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.cancel_edit)
	((XbaeMatrixWidget) w, unmap);
}

/*
 * Public interface to add_rows method
 */
void
XbaeMatrixAddRows(w, position, rows, labels, colors, num_rows)
Widget w;
int position;
String *rows;
String *labels;
Pixel *colors;
int num_rows;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the add_rows method
   */
    ( *( ( XbaeMatrixWidgetClass )XtClass( w ) )->matrix_class.add_rows )
	( ( XbaeMatrixWidget ) w, position, rows, labels, colors, NULL,
	  num_rows );
}

/*
 * Public interface to delete_rows method
 */
void
XbaeMatrixDeleteRows(w, position, num_rows)
Widget w;
int position;
int num_rows;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
     * Call the delete_rows method
     */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.delete_rows)
	((XbaeMatrixWidget) w, position, num_rows);
}

/*
 * Public interface to add_columns method
 */
void
XbaeMatrixAddColumns(w, position, columns, labels, widths, max_lengths,
		     alignments, label_alignments, colors, num_columns)
Widget w;
int position;
String *columns;
String *labels;
short *widths;
int *max_lengths;
unsigned char *alignments;
unsigned char *label_alignments;
Pixel *colors;
int num_columns;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the add_columns method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.add_columns)
	((XbaeMatrixWidget) w, position, columns, labels, widths,
	 max_lengths, alignments, label_alignments, colors, NULL, num_columns);
}

/*
 * Public interface to delete_columns method
 */
void
XbaeMatrixDeleteColumns(w, position, num_columns)
Widget w;
int position;
int num_columns;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
     * Call the delete_columns method
     */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.delete_columns)
	((XbaeMatrixWidget) w, position, num_columns);
}

/*
 * Public interface to set_row_colors method
 */
void
XbaeMatrixSetRowColors(w, position, colors, num_colors)
Widget w;
int position;
Pixel *colors;
int num_colors;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the set_row_colors method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_row_colors)
	((XbaeMatrixWidget) w, position, colors, num_colors, False);
}

/*
 * Public interface to set_column_colors method
 */
void
XbaeMatrixSetColumnColors(w, position, colors, num_colors)
Widget w;
int position;
Pixel *colors;
int num_colors;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the set_column_colors method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_column_colors)
	((XbaeMatrixWidget) w, position, colors, num_colors, False);
}

/*
 * Public interface to set_cell_color method
 */
void
XbaeMatrixSetCellColor(w, row, column, color)
Widget w;
int row;
int column;
Pixel color;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the set_cell_color method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_cell_color)
	((XbaeMatrixWidget) w, row, column, color, False);
}

/*
 * Public interface to set_row_colors method
 */
void
XbaeMatrixSetRowBackgrounds(w, position, colors, num_colors)
Widget w;
int position;
Pixel *colors;
int num_colors;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the set_row_colors method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_row_colors)
	((XbaeMatrixWidget) w, position, colors, num_colors, True);
}

/*
 * Public interface to set_column_colors method
 */
void
XbaeMatrixSetColumnBackgrounds(w, position, colors, num_colors)
Widget w;
int position;
Pixel *colors;
int num_colors;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the set_column_colors method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_column_colors)
	((XbaeMatrixWidget) w, position, colors, num_colors, True);
}

/*
 * Public interface to set_cell_color method
 */
void
XbaeMatrixSetCellBackground(w, row, column, color)
Widget w;
int row;
int column;
Pixel color;
{
    /*
     * Make sure w is a Matrix or a subclass
     */
    XtCheckSubclass(w, xbaeMatrixWidgetClass, NULL);

    /*
   * Call the set_cell_color method
   */
    (*((XbaeMatrixWidgetClass) XtClass(w))->matrix_class.set_cell_color)
	((XbaeMatrixWidget) w, row, column, color, True);
}

/*
 * Help the user know what row & column he is in given an x & y (via an event).
 * Return True on success, False on failure.
 */
int
XbaeMatrixGetEventRowColumn(w, event, row, column)
Widget w;
XEvent * event;
int *row;
int *column;
{
    XbaeMatrixWidget mw;
    int x, y;
    CellType cell;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return (False);

    /* Convert the event to the correct XY for the matrix widget. */
    mw = (XbaeMatrixWidget) w;
    if (!xbaeEventToXY(mw, event, &x, &y, &cell))
	return (False);

    /* Convert the point to a row,column. If it does not pick a valid cell,
       then return. */
    if (!xbaeXYToRowCol(mw, &x, &y, row, column, cell))
	return (False);

    return (True);
}


/*
 * Help the programmer to know what row & column we are currently at.
 * Set the row & column to -1 if bad widget.  Maybe the program will core. :)
 */
void
XbaeMatrixGetCurrentCell(w, row, column)
Widget w;
int *row;
int *column;
{
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	*row = *column = -1;
    else
    {
	mw = (XbaeMatrixWidget)w;
	*row = mw->matrix.current_row;
	*column = mw->matrix.current_column;
    }
}


/*
 * Allow the programmer to call the Expose method directly if he feels
 * that it is really needed.
 */
void
XbaeMatrixRefresh(w)
Widget w;
{
    XbaeMatrixWidget mw = ( XbaeMatrixWidget )w;

    if (XtIsSubclass(w, xbaeMatrixWidgetClass) && XtIsRealized((Widget)mw) &&
	(0 == mw->matrix.disable_redisplay))
    {
	/*
	 * Generate expose events on Matrix and Clip to force the redrawing.
	 */
	XClearArea(XtDisplay(mw), XtWindow(mw),
		   0, 0,
		   0 /*Full Width*/, 0 /*Full Height*/,
		   True);
	XbaeClipRedraw(ClipChild(mw));
    }
}

/*
 *  XbaeMatrixVisibleRows()
 *
 *  This routine returns the number of rows that are visible in the matrix.
 *
 *  D. Craig Wilson  5-MAY-1995
 *      - Cloned from the local "xbaeAdjustTopRow(mw)" routine.
 */
int
XbaeMatrixVisibleRows( w )
Widget w;
{
    XbaeMatrixWidget    matrix = ( XbaeMatrixWidget ) w;

    int rows_visible = VISIBLE_HEIGHT(matrix) / ROW_HEIGHT(matrix);

    /*
     *  If we have less than one full row visible, then count it as a full row.
     */
    if ( rows_visible == 0 )
        rows_visible = 1;

    /*
     *  rows_visible might be inaccurate since Clip may not have been resized.
     *  Test this routine and see if we need to call XbaeMatrixRefresh() to
     *  ensure accuracy.
     */
    else if ( rows_visible > matrix->matrix.rows )
        rows_visible = matrix->matrix.rows;

    return ( rows_visible );

} /* XbaeMatrixVisibleRows */



/*
 *  XbaeMatrixVisibleColumns()
 *
 *  This routine returns the number of columns that are visible in the matrix.
 *
 *  D. Craig Wilson  5-MAY-1995
 *      - Cloned from the local "xbaeAdjustTopRow(mw)" routine.
 */
int
XbaeMatrixVisibleColumns ( w )
Widget w;
{
    XbaeMatrixWidget    matrix = ( XbaeMatrixWidget )w;

    int left_column;
    int right_column;

    xbaeGetVisibleColumns(matrix, &left_column, &right_column);

    return ( right_column - left_column + 1 );

} /* XbaeMatrixVisibleColumns */

/*
 * Get per-cell user data
 */
XtPointer
XbaeMatrixGetCellUserData(w, row, column)
Widget w;
int row;
int column;
{
    XbaeMatrixWidget mw;
    XtPointer data;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return NULL;

    mw = (XbaeMatrixWidget) w;
    if (mw->matrix.cell_user_data)
	data = mw->matrix.cell_user_data[row][column];
    else
	return NULL;

    return data;
}

/*
 * Set per-cell user data
 */
void
XbaeMatrixSetCellUserData(w, row, column, data)
Widget w;
int row;
int column;
XtPointer data;
{
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;

    mw = (XbaeMatrixWidget) w;
    if (! mw->matrix.cell_user_data)
    {
	XtPointer **copy;
	register int i;

	copy = (XtPointer **) XtMalloc(mw->matrix.rows * sizeof(XtPointer*));

	for (i = 0; i < mw->matrix.rows; i++)
	    copy[i] = (XtPointer*) XtCalloc(mw->matrix.columns, sizeof(XtPointer));

	mw->matrix.cell_user_data = copy;
    }

    mw->matrix.cell_user_data[row][column] = data;
    return;
}

/*
 * Get per-row user data
 */
XtPointer
XbaeMatrixGetRowUserData(w, row)
Widget w;
int row;
{
    XbaeMatrixWidget mw;
    XtPointer data;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return NULL;

    mw = (XbaeMatrixWidget) w;
    if (mw->matrix.row_user_data)
	data = mw->matrix.row_user_data[row];
    else
	return NULL;

    return data;
}


/*
 * Set per-row user data
 */
void
XbaeMatrixSetRowUserData(w, row, data)
Widget w;
int row;
XtPointer data;
{
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;

    mw = (XbaeMatrixWidget) w;
    if (! mw->matrix.row_user_data)
    {
	XtPointer *copy;

	copy = (XtPointer *) XtCalloc(mw->matrix.rows, sizeof(XtPointer));

	mw->matrix.row_user_data = copy;
    }

    mw->matrix.row_user_data[row]= data;
    return;
}

/*
 * Get per-column user data
 */
XtPointer
XbaeMatrixGetColumnUserData(w, column)
Widget w;
int column;
{
    XbaeMatrixWidget mw;
    XtPointer data;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return NULL;

    mw = (XbaeMatrixWidget) w;
    if (mw->matrix.column_user_data)
	data = mw->matrix.column_user_data[column];
    else
	return NULL;

    return data;
}

/*
 * Set per-column user data
 */
void
XbaeMatrixSetColumnUserData(w, column, data)
Widget w;
int column;
XtPointer data;
{
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;

    mw = (XbaeMatrixWidget) w;
    if (! mw->matrix.column_user_data)
    {
	XtPointer *copy;

	copy = (XtPointer *) XtCalloc(mw->matrix.columns, sizeof(XtPointer));

	mw->matrix.column_user_data = copy;
    }

    mw->matrix.column_user_data[column]= data;
    return;
}

#if CELL_WIDGETS
/*
 * Set per-cell widget
 */
void
XbaeMatrixSetCellWidget(w, row, column, widget)
Widget w;
int row;
int column;
Widget widget;
{
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;

    mw = (XbaeMatrixWidget) w;

    if (! mw->matrix.cell_widgets)
    {
	Widget **copy;
	register int i;

	copy = (Widget **) XtMalloc(mw->matrix.rows * sizeof(Widget*));

	for (i = 0; i < mw->matrix.rows; i++)
	    copy[i] = (Widget*) XtCalloc(mw->matrix.columns, sizeof(Widget));

	mw->matrix.cell_widgets = copy;
    }

    mw->matrix.cell_widgets[row][column] = widget;
    return;
}
#endif

Boolean
XbaeMatrixIsRowSelected( w, row )
Widget w;
int row;
{
    int col;
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return False;

    mw = (XbaeMatrixWidget) w;

    if( row < 0 || row >= mw->matrix.rows )
    {
	XtAppContext appcontext = XtWidgetToApplicationContext( w );
	XtAppError( appcontext,
		    "Invalid row passed to XbaeMatrixIsRowSelected()" );
	return False;
    }

    if( !mw->matrix.selected_cells )
	return False;

    /*
     * Check all the cells in the row
     */
    for( col = 0 ; col < mw->matrix.columns ; col++ )
	if( ! mw->matrix.selected_cells[row][col] )
	    return False;

    /*
     * Return success
     */
    return True;
}

Boolean
XbaeMatrixIsColumnSelected( w, col )
Widget w;
int col;
{
    int row;
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return False;

    mw = (XbaeMatrixWidget) w;

    if( col < 0 || col >= mw->matrix.columns )
    {
	XtAppContext appcontext = XtWidgetToApplicationContext( w );
	XtAppError( appcontext,
		    "Invalid column passed to XbaeMatrixIsColumnSelected()" );
	return False;
    }

    if( !mw->matrix.selected_cells )
	return False;

    /*
     * Check all the cells in the row
     */
    for( row = 0 ; row < mw->matrix.rows ; row++ )
	if( ! mw->matrix.selected_cells[row][col] )
	    return False;

    /*
     * Return success
     */
    return True;
}

Boolean
XbaeMatrixIsCellSelected( w, row, column )
Widget w;
int row;
int column;
{
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return False;

    mw = (XbaeMatrixWidget) w;

    if( column < 0 || column >= mw->matrix.columns || row < 0 ||
	row >= mw->matrix.rows)
    {
	XtAppContext appcontext = XtWidgetToApplicationContext( w );
	XtAppError(
	    appcontext,
	    "Invalid coordinates passed to XbaeMatrixIsCellSelected()" );
	return False;
    }

    if( !mw->matrix.selected_cells )
	return False;

    if( ! mw->matrix.selected_cells[row][column] )
	return False;

    /*
     * Return success
     */
    return True;
}

int
XbaeMatrixFirstSelectedRow( w )
Widget w;
{
    int i;
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return -1;

    mw = (XbaeMatrixWidget) w;

    if( !mw->matrix.selected_cells )
	return -1;

    /*
     * Linear search for first selected
     */
    for( i = 0 ; i < mw->matrix.rows ; i++ )
	if( XbaeMatrixIsRowSelected( w, i ) )
	    return i;
    /*
     * No selection - return an invalid row id
     */
    return -1;
}

int
XbaeMatrixFirstSelectedColumn( w )
Widget w;
{
    int i;
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return -1;

    mw = (XbaeMatrixWidget) w;

    if( !mw->matrix.selected_cells )
	return -1;

    /*
     * Linear search for first selected
     */
    for( i = 0 ; i < mw->matrix.columns ; i++ )
	if( XbaeMatrixIsColumnSelected( w, i ) )
	    return i;
    /*
     * No selection - return an invalid row id
     */
    return -1;
}

void
XbaeMatrixFirstSelectedCell( w, row, column )
Widget w;
int *row;
int *column;
{
    int i, j;
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
    {
	*row = *column = -1;
	return;
    }

    mw = (XbaeMatrixWidget) w;

    if( !mw->matrix.selected_cells )
    {
	*row = -1;
	*column = -1;
	return;
    }

    for( i = 0; i < mw->matrix.rows; i++ )
	for( j = 0; j < mw->matrix.columns; j++ )
	    if( mw->matrix.selected_cells[ i ][ j ] )
	    {
		*row = i;
		*column = j;
		return;
	    }
    *row = *column = -1;
}

int
XbaeMatrixYToRow( w, y )
Widget w;
int y;
{
    int value = 0, dummy;
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return -1;

    mw = (XbaeMatrixWidget) w;

    /*
     * Do we need to worry about the scrollbar ?
     */
    if( XtIsManaged(VertScrollChild(mw)) )
    {
	/*
	 * Get scrollbar values
	 */
	XmScrollBarGetValues( VertScrollChild(mw), &value, &dummy,
			      &dummy, &dummy );
    }

    /*
     * Return YtoRow() calculation
     */
    return YtoRow(mw,y) + value - 1;
}

int
XbaeMatrixXToCol( w, x )
Widget w;
int x;
{
    int value = 0, dummy;
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return -1;

    mw = (XbaeMatrixWidget) w;

    /*
     * Do we need to worry about the scrollbar ?
     */
    if( XtIsManaged(HorizScrollChild(mw)) )
    {
	/*
	 * Get scrollbar values
	 */
	XmScrollBarGetValues( HorizScrollChild(mw), &value, &dummy,
			      &dummy, &dummy );
    }

    /*
     * Return XtoCol() calculation
     */
    return xbaeXtoCol(mw,(x+value));
}

int
XbaeMatrixGetNumSelected( w )
Widget w;
{
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return 0;

    mw = (XbaeMatrixWidget) w;

    return mw->matrix.num_selected_cells;    
}


int
XbaeMatrixNumColumns( w )
Widget w;
{
    XbaeMatrixWidget    matrix = ( XbaeMatrixWidget )w;
    return matrix->matrix.columns;
}


int
XbaeMatrixNumRows( w )
Widget w;
{
    XbaeMatrixWidget    matrix = ( XbaeMatrixWidget )w;
    return matrix->matrix.rows;
}


void
XbaeMatrixUnhighlightAll( w )
Widget w;
{
    XbaeMatrixWidget mw;
    int lc, rc, tr, br, k, m;
    Boolean didRow, *didColumn, setMask = False;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;
    mw = (XbaeMatrixWidget) w;
    
    if( !mw->matrix.highlighted_cells )
	return;

    xbaeGetVisibleCells(mw, &tr, &br, &lc, &rc);

    mw->matrix.highlight_location = UnhighlightAll;

    didColumn = (Boolean*) XtCalloc(mw->matrix.columns, sizeof(Boolean));
    for (k = 0; k < mw->matrix.rows; k++)
    {
	didRow = False;
	for (m = 0; m < mw->matrix.columns; m++)
	{
	    /*
	     * If the cell is visible
	     */
	    if ( ( (k < (int)mw->matrix.fixed_rows) ||
		   (k >= TRAILING_VERT_ORIGIN(mw)) ||
		   (k >= tr && k <= br) ) &&
		 ( (m < (int)mw->matrix.fixed_columns) ||
		   (m >= TRAILING_HORIZ_ORIGIN(mw)) ||
		   (m >= lc && m <= rc) ) )
	    {
		if (!didRow &&
		    (XmGRID_ROW_SHADOW == mw->matrix.grid_type) &&
		    (HighlightRow & mw->matrix.highlighted_cells[k][0]))
		{
		    xbaeDrawRowShadow(mw, k, GRID_REDRAW_HIGHLIGHT);
		    didRow = True;
		}

		if (!didColumn[m] &&
		    (XmGRID_COLUMN_SHADOW == mw->matrix.grid_type) &&
		    (HighlightColumn & mw->matrix.highlighted_cells[0][m]))
		{
		    didColumn[m] = True;
		    xbaeDrawColumnShadow(mw, m, GRID_REDRAW_HIGHLIGHT);
		}
		    
		if (!setMask && (k >= TRAILING_VERT_ORIGIN(mw)))
		{
		    setMask = True;
		    xbaeSetClipMask(mw, CLIP_TRAILING_FIXED_ROWS);
		}

		if (mw->matrix.highlighted_cells[k][m])
		    xbaeDrawCell(mw, k, m);
	    }

	    mw->matrix.highlighted_cells[k][m] = HighlightNone;
	}
    }

    XtFree((XtPointer) didColumn);
    
    if (setMask)
	xbaeSetClipMask(mw, CLIP_NONE);

    mw->matrix.highlight_location = HighlightNone;
}


void
XbaeMatrixHighlightCell( w, row, column )
Widget w;
int row;
int column;
{
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;
    mw = (XbaeMatrixWidget) w;

    if (row >= mw->matrix.rows || row < 0 ||
	column >= mw->matrix.columns || column < 0)
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext((Widget) mw),
	    "highlightCell", "badIndex", "XbaeMatrix",
	    "XbaeMatrix: Row or column parameter out of bounds for HighlightCell.",
	    NULL, 0);
	return;
    }

    /*
     * Scroll the cell onto the screen
     */
    if (mw->matrix.scroll_select)
	xbaeMakeCellVisible(mw, row, column);

    if( !mw->matrix.highlighted_cells )
	xbaeCopyHighlightedCells(mw);

    /*
     * Establish location -- must be after scroll,
     * otherwise may not redraw properly
     */
    mw->matrix.highlight_location = HighlightCell;

    /*
     * If the cell is not already highlighted
     */
    if (!(mw->matrix.highlighted_cells[row][column] & HighlightCell))
    {
	mw->matrix.highlighted_cells[row][column] |= HighlightCell;

	/*
	 * Only redraw if cell is visible
	 */
	if (xbaeIsCellVisible(mw, row, column))
	{
	    int x, y;
	    Window win;

	    /*
	     * Get the correct window
	     */
	    win = IS_FIXED(mw, row, column) ?
		XtWindow(mw) : 	XtWindow(ClipChild(mw)) ;

	    /*
	     * Convert row,column coordinates to the
	     * coordinates relative to the correct window
	     */
	    xbaeRowColToXY(mw, row, column, &x, &y);

	    _XmDrawHighlight( XtDisplay(mw), win, 
			      mw->manager.highlight_GC,
			      x + mw->matrix.cell_shadow_thickness,
			      y + mw->matrix.cell_shadow_thickness,
			      COLUMN_WIDTH(mw, column) -
			      (2 * mw->matrix.cell_shadow_thickness),
			      ROW_HEIGHT(mw) -
			      (2 * mw->matrix.cell_shadow_thickness),
			      mw->matrix.cell_highlight_thickness,
			      LineSolid);
	}
    }
    mw->matrix.highlight_location = HighlightNone;
}


void
XbaeMatrixUnhighlightCell( w, row, column )
Widget w;
int row;
int column;
{
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;
    mw = (XbaeMatrixWidget) w;

    if (row >= mw->matrix.rows || row < 0 ||
	column >= mw->matrix.columns || column < 0)
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext((Widget) mw),
	    "unhighlightCell", "badIndex", "XbaeMatrix",
	    "XbaeMatrix: Row or column parameter out of bounds for UnhighlightCell.",
	    NULL, 0);
	return;
    }

    if( !mw->matrix.highlighted_cells )
	return;
    
    mw->matrix.highlight_location = UnhighlightCell;
    
    if (xbaeIsCellVisible(mw, row, column))
    {
	if (row >= TRAILING_VERT_ORIGIN(mw))
	    xbaeSetClipMask(mw, CLIP_TRAILING_FIXED_ROWS);

	xbaeDrawCell(mw, row, column);
	
	if (row >= TRAILING_VERT_ORIGIN(mw))
	    xbaeSetClipMask(mw, CLIP_NONE);
    }
    
    mw->matrix.highlighted_cells[row][column] &= ~HighlightCell;

    mw->matrix.highlight_location = HighlightNone;
}


void
XbaeMatrixHighlightRow( w, row )
Widget w;
int row;
{
    XbaeMatrixWidget mw;
    int j, lc, rc;
    Boolean visible;
    unsigned char highlight;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;
    mw = (XbaeMatrixWidget) w;

    if (row >= mw->matrix.rows || row < 0)
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext((Widget) mw),
	    "highlightRow", "badIndex", "XbaeMatrix",
	    "XbaeMatrix: Row out of bounds for HighlightRow.",
	    NULL, 0);
	return;
    }

    if( !mw->matrix.highlighted_cells )
	xbaeCopyHighlightedCells(mw);

    /*
     * Scroll the row onto the screen
     */
    if (mw->matrix.scroll_select)
	xbaeMakeRowVisible(mw, row);

    /*
     * Establish location -- must be after scroll,
     * otherwise may not redraw properly
     */
    mw->matrix.highlight_location = HighlightRow;

    /*
     * For each cell in the row, if the cell is not already highlighted,
     * highlight it and redraw it if it is visible
     */
     
    visible = xbaeIsRowVisible(mw, row);
    xbaeGetVisibleColumns(mw, &lc, &rc);
    highlight = ((XmGRID_ROW_SHADOW == mw->matrix.grid_type) ?
		 HighlightRow : HighlightOther);

    if (row >= TRAILING_VERT_ORIGIN(mw))
	xbaeSetClipMask(mw, CLIP_TRAILING_FIXED_ROWS);

    for (j = 0; j < mw->matrix.columns; j++)
    {
	if (!(mw->matrix.highlighted_cells[row][j] & highlight))
	{
	    mw->matrix.highlighted_cells[row][j] |= highlight;
	    	
	    if ((XmGRID_ROW_SHADOW != mw->matrix.grid_type) &&
		(visible &&
		 ((j >= lc && j <= rc) ||
		  (j < (int)mw->matrix.fixed_columns) ||
		  (j >= TRAILING_HORIZ_ORIGIN(mw)))))
	    {
		xbaeDrawCell(mw, row, j);
	    }
	}
    }

    /*
     * Draw row highlight if necessary
     */
    if (visible && (XmGRID_ROW_SHADOW == mw->matrix.grid_type))
	xbaeDrawRowShadow(mw, row, GRID_REDRAW_HIGHLIGHT);

    if (row >= TRAILING_VERT_ORIGIN(mw))
	xbaeSetClipMask(mw, CLIP_NONE);

    mw->matrix.highlight_location = HighlightNone;
}


void
XbaeMatrixUnhighlightRow( w, row )
Widget w;
int row;
{
    XbaeMatrixWidget mw;
    int j, lc, rc;
    Boolean visible;
    unsigned char highlight;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;
    mw = (XbaeMatrixWidget) w;

    if (row >= mw->matrix.rows || row < 0)
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext((Widget) mw),
	    "highlightRow", "badIndex", "XbaeMatrix",
	    "XbaeMatrix: Row out of bounds for UnhighlightRow.",
	    NULL, 0);
	return;
    }

    if( !mw->matrix.highlighted_cells )
	return;
    
    mw->matrix.highlight_location = UnhighlightRow;

    visible = xbaeIsRowVisible(mw, row);

    if (visible && (XmGRID_ROW_SHADOW == mw->matrix.grid_type))
	xbaeDrawRowShadow(mw, row, GRID_REDRAW_HIGHLIGHT);

    /*
     * Establish any necessary clipping for redrawing the cells
     */
    if	(row < mw->matrix.fixed_rows)
	xbaeSetClipMask(mw, CLIP_FIXED_ROWS);
    else if (row >= TRAILING_VERT_ORIGIN(mw))
	xbaeSetClipMask(mw, CLIP_TRAILING_FIXED_ROWS);
    
    /*
     * For each cell in the row, if the cell is highlighted,
     * unhighlight it and redraw it if it is visible
     */
     
    xbaeGetVisibleColumns(mw, &lc, &rc);
    highlight = ((XmGRID_ROW_SHADOW == mw->matrix.grid_type) ?
		 HighlightRow : HighlightOther);
    for (j = 0; j < mw->matrix.columns; j++)
    {
	if (mw->matrix.highlighted_cells[row][j] & highlight)
	{
	    if (visible &&
		((j >= lc && j <= rc) ||
		 (j < (int)mw->matrix.fixed_columns) ||
		 (j >= TRAILING_HORIZ_ORIGIN(mw))))
	    {
		xbaeDrawCell(mw, row, j);
	    }
	    mw->matrix.highlighted_cells[row][j] &= ~highlight;
	}
    }

    if	((row < mw->matrix.fixed_rows) ||
	 (row >= TRAILING_VERT_ORIGIN(mw)))
	xbaeSetClipMask(mw, CLIP_NONE);

    mw->matrix.highlight_location = HighlightNone;
}


void
XbaeMatrixHighlightColumn( w, column )
Widget w;
int column;
{
    XbaeMatrixWidget mw;
    int j, tr, br;
    Boolean visible, once = False;
    unsigned char highlight;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;
    mw = (XbaeMatrixWidget) w;

    if (column >= mw->matrix.columns || column < 0)
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext((Widget) mw),
	    "highlightColumn", "badIndex", "XbaeMatrix",
	    "XbaeMatrix: Column out of bounds for HighlightColumn.",
	    NULL, 0);
	return;
    }
    
    if( !mw->matrix.highlighted_cells )
	xbaeCopyHighlightedCells(mw);
    /*
     * Scroll the row onto the screen
     */
    if (mw->matrix.scroll_select)
	xbaeMakeColumnVisible(mw, column);

    /*
     * Establish location -- must be after scroll,
     * otherwise may not redraw properly
     */
    mw->matrix.highlight_location = HighlightColumn;
    
    if	(column < mw->matrix.fixed_columns)
	xbaeSetClipMask(mw, CLIP_FIXED_COLUMNS | CLIP_VISIBLE_HEIGHT);
    else if (column >= TRAILING_HORIZ_ORIGIN(mw))
	xbaeSetClipMask(mw, CLIP_TRAILING_FIXED_COLUMNS | CLIP_VISIBLE_HEIGHT);
    
    /*
     * For each cell in the column, if the cell is not already highlighted,
     * highlight it and redraw it if it is visible
     */
     
    visible = xbaeIsColumnVisible(mw, column);
    xbaeGetVisibleRows(mw, &tr, &br);
    highlight =	((XmGRID_COLUMN_SHADOW == mw->matrix.grid_type) ?
		 HighlightColumn : HighlightOther);
    for (j = 0; j < mw->matrix.rows; j++)
    {
	if (!(mw->matrix.highlighted_cells[j][column] & highlight))
	{
	    mw->matrix.highlighted_cells[j][column] |= highlight;
	    
	    if ((XmGRID_COLUMN_SHADOW != mw->matrix.grid_type) &&
		(visible &&
		 ((j >= tr && j <= br) ||
		  (j < (int)mw->matrix.fixed_rows) ||
		  (j >= TRAILING_VERT_ORIGIN(mw)))))
	    {
		if ((j >= TRAILING_VERT_ORIGIN(mw)) && (! once))
		{
		    once = True;
		    xbaeSetClipMask(mw, CLIP_TRAILING_FIXED_ROWS);
		}
		
		xbaeDrawCell(mw, j, column);
	    }
	}
    }

    /*
     * May need to redraw the shadow to get the highlight
     */
    if (visible && (XmGRID_COLUMN_SHADOW == mw->matrix.grid_type))
	xbaeDrawColumnShadow(mw, column, GRID_REDRAW_HIGHLIGHT);
    
    if	(once || (column < mw->matrix.fixed_columns) ||
	 (column >= TRAILING_HORIZ_ORIGIN(mw)))
	xbaeSetClipMask(mw, CLIP_NONE);
    
    mw->matrix.highlight_location = HighlightNone;
}


void
XbaeMatrixUnhighlightColumn( w, column )
Widget w;
int column;
{
    XbaeMatrixWidget mw;
    int j, tr, br;
    unsigned int clip_reason = 0;
    Boolean visible, once = False;
    unsigned char highlight;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;
    mw = (XbaeMatrixWidget) w;

    if (column >= mw->matrix.columns || column < 0)
    {
	XtAppWarningMsg(
	    XtWidgetToApplicationContext((Widget) mw),
	    "highlightColumn", "badIndex", "XbaeMatrix",
	    "XbaeMatrix: Column out of bounds for UnhighlightColumn.",
	    NULL, 0);
	return;
    }

    if( !mw->matrix.highlighted_cells )
	return;
    
    mw->matrix.highlight_location = UnhighlightColumn;

    visible = xbaeIsColumnVisible(mw, column);

    if (visible && XmGRID_COLUMN_SHADOW == mw->matrix.grid_type)
	xbaeDrawColumnShadow(mw, column, GRID_REDRAW_HIGHLIGHT);

    /*
     * Establish any necessary clipping for redrawing the cells
     */
    if	(column < mw->matrix.fixed_columns)
	clip_reason = CLIP_FIXED_COLUMNS;
    else if (column >= TRAILING_HORIZ_ORIGIN(mw))
	clip_reason = CLIP_TRAILING_FIXED_COLUMNS;

    if (clip_reason)
	xbaeSetClipMask(mw, clip_reason | CLIP_VISIBLE_HEIGHT);

    /*
     * For each cell in the row, if the cell is highlighted,
     * unhighlight it and redraw it if it is visible.
     */
    xbaeGetVisibleRows(mw, &tr, &br);
    highlight = ((XmGRID_COLUMN_SHADOW == mw->matrix.grid_type) ?
		 HighlightColumn : HighlightOther);
    for (j = 0; j < mw->matrix.rows; j++)
    {
	if (mw->matrix.highlighted_cells[j][column] & highlight)
	{
	    if (visible &&
		((j >= tr && j <= br) ||
		 (j < (int)mw->matrix.fixed_rows) ||
		 (j >= TRAILING_VERT_ORIGIN(mw))))
	    {
		if ((j >= TRAILING_VERT_ORIGIN(mw)) && (! once))
		{
		    once = True;
		    xbaeSetClipMask(mw, clip_reason | CLIP_TRAILING_FIXED_ROWS);
		}
		xbaeDrawCell(mw, j, column);
	    }
	    mw->matrix.highlighted_cells[j][column] &= ~highlight;
	}
    }

    if	(once || clip_reason)
	xbaeSetClipMask(mw, CLIP_NONE);

    mw->matrix.highlight_location = HighlightNone;
}


void
XbaeMatrixDisableRedisplay( w )
Widget w;
{
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;
    mw = (XbaeMatrixWidget) w;

    mw->matrix.disable_redisplay++ ;
}


void
#if NeedFunctionPrototypes
XbaeMatrixEnableRedisplay(Widget w, Boolean redisplay)
#else
XbaeMatrixEnableRedisplay(w, redisplay)
Widget w;
Boolean redisplay;
#endif
{
    XbaeMatrixWidget mw;

    if (!XtIsSubclass(w, xbaeMatrixWidgetClass))
	return;
    mw = (XbaeMatrixWidget) w;

    if (mw->matrix.disable_redisplay)
	mw->matrix.disable_redisplay--;

#undef FORCE_REDISPLAY_IF_TRUE
#ifndef FORCE_REDISPLAY_IF_TRUE
    if (redisplay && (0 == mw->matrix.disable_redisplay))
	XbaeMatrixRefresh( w );
#else
    if (redisplay)
    {
	mw->matrix.disable_redisplay = 0;
	XbaeMatrixRefresh( w );
    }
#endif
}
