/*
 * Copyright(c) 1992 Bell Communications Research, Inc. (Bellcore)
 *                        All rights reserved
 * Permission to use, copy, modify and distribute this material for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of Bellcore not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * Bellcore.
 *
 * BELLCORE MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL BELLCORE OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES RELAT-
 * ING TO THE SOFTWARE.
 */

#include <Xm/Xm.h>
#include <Xm/XmP.h>
#if XmVersion > 1001
#include <Xm/DrawP.h>
#endif
#include <Xbae/MatrixP.h>
#include <Xbae/Utils.h>
#include <Xbae/Draw.h>

static void DrawCellShadow P(( XbaeMatrixWidget mw, Window win, int row,
			       int column, int x, int y, Boolean clipped ));
static void xbaeDrawCellString P(( XbaeMatrixWidget mw, int row, int column,
			    String string, Pixel bg, Pixel fg ));
#if CELL_WIDGETS
static void xbaeDrawCellWidget P(( XbaeMatrixWidget mw, int row, int column,
				   Widget widget, Pixel bg, Pixel fg ));
#endif
static void xbaeDrawCellPixmap P(( XbaeMatrixWidget mw, int row, int column,
				   Pixmap pixmap, Pixmap mask,
				   int width, int height, Pixel bg, Pixel fg,
				   int depth ));

static void
#if NeedFunctionPrototypes
DrawCellShadow(XbaeMatrixWidget mw, Window win, int row, int column, int x,
	       int y, Boolean clipped)
#else
DrawCellShadow(mw, win, row, column, x, y, clipped)
XbaeMatrixWidget mw;
Window win;
int row;
int column;
int x;
int y;
Boolean clipped;
#endif
{
    /*
     * Surround the cell with a shadow.
     */
    if (mw->matrix.cell_shadow_thickness)
    {
	unsigned char shadow = mw->matrix.cell_shadow_types ?
	    mw->matrix.cell_shadow_types[row][column] :
	    mw->matrix.cell_shadow_type;
	
	if (clipped)
	{
	    switch (mw->matrix.grid_type)
	    {
	    case XmGRID_NONE:
	    case XmGRID_ROW_SHADOW:
	    case XmGRID_COLUMN_SHADOW:
		break;
	    case XmGRID_LINE:
		DRAW_SHADOW(XtDisplay(mw), win,
			    mw->matrix.cell_grid_line_gc,
			    mw->matrix.cell_grid_line_gc,
			    mw->matrix.cell_shadow_thickness, x, y,
			    COLUMN_WIDTH(mw, column), ROW_HEIGHT(mw),
			    shadow);
		break;
	    case XmGRID_SHADOW_OUT:
		DRAW_SHADOW(XtDisplay(mw), win,
			    mw->matrix.cell_bottom_shadow_clip_gc,
			    mw->matrix.cell_top_shadow_clip_gc,
			    mw->matrix.cell_shadow_thickness, x, y,
			    COLUMN_WIDTH(mw, column), ROW_HEIGHT(mw),
			    shadow);
		break;
	    case XmGRID_SHADOW_IN:
		DRAW_SHADOW(XtDisplay(mw), win,
			    mw->matrix.cell_top_shadow_clip_gc,
			    mw->matrix.cell_bottom_shadow_clip_gc,
			    mw->matrix.cell_shadow_thickness, x, y,
			    COLUMN_WIDTH(mw, column), ROW_HEIGHT(mw),
			    shadow);
		break;
	    }
	}
	else
	{
	    switch (mw->matrix.grid_type)
	    {
	    case XmGRID_ROW_SHADOW:
	    case XmGRID_COLUMN_SHADOW:
	    case XmGRID_NONE:
		break;
	    case XmGRID_LINE:
		DRAW_SHADOW(XtDisplay(mw), win,
			    mw->matrix.grid_line_gc,
			    mw->matrix.grid_line_gc,
			    mw->matrix.cell_shadow_thickness,
			    x, y,
			    COLUMN_WIDTH(mw, column),
			    ROW_HEIGHT(mw),
			    shadow);
		break;
	    case XmGRID_SHADOW_OUT:
		DRAW_SHADOW(XtDisplay(mw), win,
			    mw->manager.bottom_shadow_GC,
			    mw->manager.top_shadow_GC,
			    mw->matrix.cell_shadow_thickness,
			    x, y,
			    COLUMN_WIDTH(mw, column),
			    ROW_HEIGHT(mw),
			    shadow);
		break;
	    case XmGRID_SHADOW_IN:
		DRAW_SHADOW(XtDisplay(mw), win,
			    mw->manager.top_shadow_GC,
			    mw->manager.bottom_shadow_GC,
			    mw->matrix.cell_shadow_thickness,
			    x, y,
			    COLUMN_WIDTH(mw, column),
			    ROW_HEIGHT(mw),
			    shadow);
		break;
	    }
	}    
    }
}

/*
 * Draw a fixed or non-fixed cell. The coordinates are calculated relative
 * to the correct window and pixmap is copied to that window.
 */
static void
xbaeDrawCellPixmap(mw, row, column, pixmap, mask, width, height, bg, fg, depth)
XbaeMatrixWidget mw;
int row;
int column;
Pixmap pixmap;
Pixmap mask;
int width;
int height;
Pixel bg;
Pixel fg;
int depth;
{
    int x, y;
    int src_x = 0, src_y, dest_x, dest_y;
    int copy_width, copy_height;
    int cell_height = ROW_HEIGHT(mw); /*- TEXT_HEIGHT_OFFSET(mw) * 2;*/
    int cell_width = COLUMN_WIDTH(mw, column) - TEXT_WIDTH_OFFSET(mw) * 2;
    unsigned char alignment = mw->matrix.column_alignments ?
	mw->matrix.column_alignments[ column ] : XmALIGNMENT_BEGINNING;
    Display *display = XtDisplay( mw );
    GC gc;
    Window win = CELL_WINDOW(mw, row, column);
    Boolean clipped = ((row < (int)mw->matrix.fixed_rows) ||
		       (row >= TRAILING_VERT_ORIGIN(mw))) &&
	(column >= (int)mw->matrix.fixed_columns) &&
	(column < TRAILING_HORIZ_ORIGIN(mw));

    if (!win)
	return;

    /*
     * Convert the row/column to the coordinates relative to the correct
     * window
     */
    xbaeRowColToXY(mw, row, column, &x, &y);
    dest_x = x + TEXT_WIDTH_OFFSET(mw);

    gc = (clipped ? mw->matrix.pixmap_clip_gc : mw->matrix.pixmap_gc);

    XSetForeground( display, gc, bg);
    XFillRectangle( display, win, gc, x, y,
		    COLUMN_WIDTH(mw, column), ROW_HEIGHT(mw) );

    XSetForeground(display, gc, fg);
    XSetBackground(display, gc, bg);

    /*
     * Adjust the x and y drawing destinations as appropriate.  First the
     * y value....
     */
    dest_y = y; /* + TEXT_HEIGHT_OFFSET(mw);*/
    if( height > cell_height )
    {
	/* Adjust the starting location in the src image */
	src_y = ( height - cell_height ) / 2;
	copy_height = cell_height;
    }
    else
    {
	/* Adjust the destination point */
	src_y = 0;
	dest_y += ( ( cell_height - height ) / 2 );
	copy_height = height;
    }

    /*
     * Adjust the x value, paying attention to the columnAlignment
     */
    if( width > cell_width )
	copy_width = cell_width;
    else
	copy_width = width;
    
    switch( alignment )
    {
    case XmALIGNMENT_BEGINNING:
	src_x = 0;
	break;
    case XmALIGNMENT_CENTER:
	if( width > cell_width )
	    src_x = ( width - cell_width ) / 2;
	else
	{
	    src_x = 0;
	    dest_x += ( ( cell_width - width ) / 2 );
	}
	break;
    case XmALIGNMENT_END:	
	if( width > cell_width )
	    src_x = width - cell_width;
	else
	{
	    src_x = 0;
	    dest_x = x + COLUMN_WIDTH( mw, column ) - TEXT_WIDTH_OFFSET( mw ) -
		width;
	}
	break;
    }

    /*
     * Draw the pixmap.  Clip it, if necessary
     */
    if( pixmap )
    {
	if( depth > 1 )		/* A pixmap using xpm */
	{
	    if( mask )
	    {
		XSetClipMask( display, gc, mask );

		XSetClipOrigin( display, gc, dest_x - src_x, dest_y - src_y );
	    }
	    XCopyArea( display, pixmap, win, gc, src_x, src_y, copy_width,
		       copy_height, dest_x, dest_y );
	    if( mask )
		XSetClipMask( display, gc, None );
	}
	else			/* A plain old bitmap */
	    XCopyPlane( display, pixmap, win, gc, src_x, src_y, copy_width,
			copy_height, dest_x, dest_y, 1L );
    }
    
    DrawCellShadow( mw, win, row, column, x, y, clipped );
}

/*
 * Draw a fixed or non-fixed cell. The coordinates are calculated relative
 * to the correct window and the cell is drawn in that window.
 */
static void
xbaeDrawCellString(mw, row, column, string, bg, fg)
XbaeMatrixWidget mw;
int row, column;
String string;
Pixel bg, fg;
{
    int x, y;
    GC gc;
    Window win = CELL_WINDOW(mw, row, column);
    Boolean clipped = ((row < (int)mw->matrix.fixed_rows) ||
		       (row >= TRAILING_VERT_ORIGIN(mw))) &&
	(column >= (int)mw->matrix.fixed_columns) &&
	(column < TRAILING_HORIZ_ORIGIN(mw));
    Dimension thick = (2 * mw->matrix.cell_shadow_thickness);
    Dimension column_width = COLUMN_WIDTH(mw, column);
    Dimension row_height = ROW_HEIGHT(mw);
    Dimension width = column_width;
    Dimension height = row_height;
    Boolean selected = mw->matrix.selected_cells ?
	mw->matrix.selected_cells[row][column] : False;
    String str = string;
    
    if (!win)
	return;

    /*
     * Convert the row/column to the coordinates relative to the correct
     * window
     */
    xbaeRowColToXY(mw, row, column, &x, &y);

    /*
     * Make sure y coordinate is valid 
     */
    if ( (win == XtWindow(mw)) &&
	 ( (y > ( CLIP_VERT_VISIBLE_SPACE(mw) + ROW_LABEL_OFFSET(mw) - 1) ) ||
	   (y < ROW_LABEL_OFFSET(mw)) ) )
	return;

    gc = (clipped ? mw->matrix.draw_clip_gc : mw->matrix.draw_gc);
    XSetForeground(XtDisplay(mw), gc, bg);

    /*
     * If we are merely unhighlighting a cell
     */
    if (mw->matrix.highlighted_cells && 0xF0 & mw->matrix.highlight_location)
    {
	/*
	 * If this cell is not explicitly highlighted or we are not
	 * explicitly unhighlighting the cell or we are unhighlighting
	 * the row or column when we are in grid row or column mode,
	 * then erase the highlight. Erasing the highlight ensures that
	 * the cell's background color gets properly redrawn.
	 *
	 * This handles the case where grid is row or column shadow,
	 * and hence a highlight has been drawn around the entire row
	 * or column, but we also have explicitly highlighted a particular
	 * cell.
	 */

	if ( !(HighlightCell & mw->matrix.highlighted_cells[row][column]) ||
	     (UnhighlightCell & mw->matrix.highlight_location) ||
	     ( (UnhighlightRow == mw->matrix.highlight_location) &&
	       (XmGRID_ROW_SHADOW == mw->matrix.grid_type) ) ||
	     ( (UnhighlightColumn == mw->matrix.highlight_location) &&
	       (XmGRID_COLUMN_SHADOW == mw->matrix.grid_type) ) )
	{
	    Position hx = x, hy = y;

	    if ((XmGRID_ROW_SHADOW != mw->matrix.grid_type) ||
		!(UnhighlightRow & mw->matrix.highlight_location))
		width -= thick;
	    else
	    {
		if (0 != column)
		{
		    hx -= mw->matrix.cell_shadow_thickness;
		    if ((mw->matrix.columns - 1) == column)
		    {
			if ( NEED_HORIZ_FILL( mw ) )
			{
			    /* damn picky, but it works */
			    width -= FILL_HORIZ_WIDTH(mw);  
			    /* oops prevent wrap around */
			    if (width > column_width)
				width = column_width;
			}
			else
			    width -= mw->matrix.cell_shadow_thickness;
		    }
		}
		else
		    width -= mw->matrix.cell_shadow_thickness;
	    }

	    if ((XmGRID_COLUMN_SHADOW != mw->matrix.grid_type) ||
		!(UnhighlightColumn & mw->matrix.highlight_location))
		height -= thick;
	    else
	    {
		if (0 != row)
		{
		    hy -= mw->matrix.cell_shadow_thickness;
		    if ((mw->matrix.rows - 1) == row)
		    {
			if ( NEED_VERT_FILL( mw ) )
			{
			    Dimension fvh = FILL_VERT_HEIGHT(mw);
			    if (fvh < mw->matrix.cell_shadow_thickness)
				height -= (mw->matrix.cell_shadow_thickness - fvh);
			    else
				height -= fvh;
			    if (height > row_height)
				/* oops prevent wrap around */
				height = row_height;
			}
			else
			    height -= mw->matrix.cell_shadow_thickness;
		    }
		}
		else
		    height -= mw->matrix.cell_shadow_thickness;
	    }

	    _XmDrawHighlight( XtDisplay(mw), win, gc,
			      hx + mw->matrix.cell_shadow_thickness,
			      hy + mw->matrix.cell_shadow_thickness,
			      width, height,
			      mw->matrix.cell_highlight_thickness,
			      LineSolid);

	    /*
	     * If we are unhighlightling all and this cell was explicitly
	     * highlighted and we are in grid row or column shadow, we need
	     * to explicitly erase the highlight
	     */
	    if ((UnhighlightAll == mw->matrix.highlight_location) &&
		((XmGRID_ROW_SHADOW == mw->matrix.grid_type) ||
		 (XmGRID_COLUMN_SHADOW == mw->matrix.grid_type)))
	    {
		_XmDrawHighlight( XtDisplay(mw), win, gc,
				  x + mw->matrix.cell_shadow_thickness,
				  y + mw->matrix.cell_shadow_thickness,
				  column_width - thick,
				  row_height - thick,
				  mw->matrix.cell_highlight_thickness,
				  LineSolid);
	    }
	}

	/*
	 * If this cell is explicitly highlighted and we are not
	 * explicitly unhighlighting it, draw the highlight.
	 */
	if ((HighlightCell & mw->matrix.highlighted_cells[row][column]) &&
	    !(UnhighlightCell & mw->matrix.highlight_location))
	{
	    _XmDrawHighlight( XtDisplay(mw), win,
			      clipped ? mw->matrix.highlight_clip_gc :
			      mw->manager.highlight_GC,
			      x + mw->matrix.cell_shadow_thickness,
			      y + mw->matrix.cell_shadow_thickness,
			      column_width - thick,
			      row_height - thick,
			      mw->matrix.cell_highlight_thickness,
			      LineSolid);
	}

	return;
    }

    /*
     * Or we are in the process of explicitly highlighting cells
     */
    else if (mw->matrix.highlighted_cells &&
	     (0x0F & mw->matrix.highlight_location) &&
	     ((HighlightCell | HighlightOther) &
	      mw->matrix.highlighted_cells[row][column]))
    {
	_XmDrawHighlight( XtDisplay(mw), win,
			  clipped ? mw->matrix.highlight_clip_gc :
			  mw->manager.highlight_GC,
			  x + mw->matrix.cell_shadow_thickness,
			  y + mw->matrix.cell_shadow_thickness,
			  column_width - thick,
			  row_height - thick,
			  mw->matrix.cell_highlight_thickness,
			  LineSolid);
	return;
    }

    /*
     * Fill the cell's background if it can be done
     * without duplicating work below
     */
    if ((XtWindow(mw) != win) ||
	( !((XmGRID_COLUMN_SHADOW == mw->matrix.grid_type) &&
	    ((mw->matrix.rows - 1) == row) &&  NEED_VERT_FILL( mw )) &&
	  !((XmGRID_ROW_SHADOW == mw->matrix.grid_type) &&
	    ((mw->matrix.columns - 1) == column) && NEED_HORIZ_FILL( mw )) ) )
	XFillRectangle( XtDisplay(mw), win, gc, x, y,
			column_width, row_height );

    /*
     * If we need to fill the rest of the space, do so
     */
    if ((XmGRID_COLUMN_SHADOW == mw->matrix.grid_type) &&
	((mw->matrix.rows - 1) == row) && 
	NEED_VERT_FILL( mw ))
    {
	height = mw->core.height - HORIZ_SB_SPACE(mw) -
	    mw->manager.shadow_thickness - y -
	    ((XtWindow(mw) != win) ? FIXED_ROW_LABEL_OFFSET(mw) : 0);

	XFillRectangle( XtDisplay(mw), XtWindow(mw), gc,
			x + ((XtWindow(mw) != win) ?
			     FIXED_COLUMN_LABEL_OFFSET(mw) : 0),
			y + ((XtWindow(mw) != win) ?
			     FIXED_ROW_LABEL_OFFSET(mw) : 0),
			column_width, height );
    }
    else if ((XmGRID_ROW_SHADOW == mw->matrix.grid_type) &&
	     ((mw->matrix.columns - 1) == column) &&
	     NEED_HORIZ_FILL( mw ))
    {
	width = mw->core.width - VERT_SB_SPACE(mw) -
	    mw->manager.shadow_thickness - x -
	    ((XtWindow(mw) != win) ? FIXED_COLUMN_LABEL_OFFSET(mw) : 0);

	XFillRectangle( XtDisplay(mw), XtWindow(mw), gc,
			x + ((XtWindow(mw) != win) ?
			     FIXED_COLUMN_LABEL_OFFSET(mw) : 0),
			y + ((XtWindow(mw) != win) ?
			     FIXED_ROW_LABEL_OFFSET(mw) : 0),
			width, row_height );
    }

    /*
     * Draw the string in the cell.
     */

    xbaeDrawString(mw, win, str, strlen(str),
		   x + TEXT_X_OFFSET(mw), y + TEXT_Y_OFFSET(mw),
		   mw->matrix.column_widths[column],
		   mw->matrix.column_alignments ?
		   mw->matrix.column_alignments[column] :
		   XmALIGNMENT_BEGINNING, selected,
		   False, clipped, False, False, fg);

    /*
     * Draw a highlight around the cell if it was highlighted because
     * of an explicit cell highlight call or because it was highlighted
     * due to a row or column highlight call
     */
    if ( mw->matrix.highlighted_cells && (HighlightCell | HighlightOther) &
	 mw->matrix.highlighted_cells[row][column])
    {
	_XmDrawHighlight( XtDisplay(mw), win,
			  clipped ? mw->matrix.highlight_clip_gc :
			  mw->manager.highlight_GC,
			  x + mw->matrix.cell_shadow_thickness,
			  y + mw->matrix.cell_shadow_thickness,
			  column_width - thick,
			  row_height - thick,
			  mw->matrix.cell_highlight_thickness,
			  LineSolid);
    }
        
    DrawCellShadow( mw, win, row, column, x, y, clipped );
}
#if CELL_WIDGETS
/*
 * Draw a user defined widget in the cell
 */
static void
xbaeDrawCellWidget(mw, row, column, widget, bg, fg)
XbaeMatrixWidget mw;
int row, column;
Widget widget;
Pixel bg, fg;
{
    int x, y;
    GC gc;
    Window win = CELL_WINDOW(mw, row, column);
    Boolean clipped = ((row < (int)mw->matrix.fixed_rows) ||
		       (row >= TRAILING_VERT_ORIGIN(mw))) &&
	(column >= (int)mw->matrix.fixed_columns) &&
	(column < TRAILING_HORIZ_ORIGIN(mw));

    if (!win)
	return;

    /*
     * Convert the row/column to the coordinates relative to the correct
     * window
     */
    xbaeRowColToXY(mw, row, column, &x, &y);

    gc = (clipped ? mw->matrix.draw_clip_gc : mw->matrix.draw_gc);
    XSetForeground(XtDisplay(mw), gc, bg);
    XFillRectangle( XtDisplay(mw), win, gc, x, y,
		    COLUMN_WIDTH(mw, column), ROW_HEIGHT(mw) );

    /*
     * Draw the widget in the cell.
     */
    XtMoveWidget(widget,
		 x + mw->matrix.cell_shadow_thickness +
		 mw->matrix.cell_highlight_thickness,
		 y + mw->matrix.cell_shadow_thickness +
		 mw->matrix.cell_highlight_thickness);

    DrawCellShadow( mw, win, row, column, x, y, clipped );
}
#endif

/*
 * Width in pixels of a character in a given font
 */
#define charWidth(fs,c) ((fs)->per_char\
                         ? (fs)->per_char[(unsigned char)(c) -\
					 (fs)->min_char_or_byte2].width\
                         : (fs)->min_bounds.width)


/*
 * Draw a string with specified attributes. We want to avoid having to
 * use a GC clip_mask, so we clip by characters. This complicates the code.
 */
void
#if NeedFunctionPrototypes
xbaeDrawString( XbaeMatrixWidget mw, Window win, String string, int length,
		int x, int y, int maxlen, unsigned char alignment,
		Boolean highlight, Boolean bold, Boolean clip,
		Boolean rowLabel, Boolean colLabel,
		Pixel color )
#else
xbaeDrawString( mw, win, string, length, x, y, maxlen, alignment, highlight,
		bold, clip, rowLabel, colLabel, color )
XbaeMatrixWidget mw;
Window win;
String string;
int length;
int x;
int y;
int maxlen;
unsigned char alignment;
Boolean highlight;
Boolean bold;
Boolean clip;
Boolean rowLabel;
Boolean colLabel;
Pixel color;
#endif
{
    int start, width, maxwidth;
    GC gc;
    XFontStruct	*font;

    if( rowLabel || colLabel )
	font = mw->matrix.label_font;
    else
	font = mw->matrix.font;
    /*
     * Initialize starting character in string
     */
    start = 0;

    if( !rowLabel )
	maxwidth = maxlen * FONT_WIDTH(mw);
    else
	maxwidth = maxlen * LABEL_MAX_WIDTH(mw);

    width = XTextWidth(font, string, length);

    /*
     * If the width of the string is greater than the width of this cell,
     * we need to clip. We don't want to use the server to clip because
     * it is slow, so we truncate characters if we exceed a cells pixel width.
     */
    if (width > maxwidth)
    {
	switch (alignment)
	{

	case XmALIGNMENT_CENTER:
	{
	    int startx = x;
	    int endx = x + maxwidth - 1;
	    int newendx;

	    /*
	     * Figure out our x for the centered string.  Then loop and chop
	     * characters off the front until we are within the cell.
	     * Adjust x, the starting character and the length of the string
	     * for each char.
	     */
	    x += maxwidth / 2 - width / 2;
	    while (x < startx)
	    {
		int cw = charWidth(font, string[start]);

		x += cw;
		width -= cw;
		length--;
		start++;
	    }

	    /*
	     * Now figure out the end x of the string.  Then loop and chop
	     * characters off the end until we are within the cell.
	     */
	    newendx = x + width - 1;
	    while (newendx > endx)
	    {
		newendx -= charWidth(font, string[start + length - 1]);
		length--;
	    }

	    break;
	}

	case XmALIGNMENT_END:
	{

	    /*
	     * Figure out our x for the right justified string.
	     * Then loop and chop characters off the front until we fit.
	     * Adjust x for each char lopped off. Also adjust the starting
	     * character and length of the string for each char.
	     */
	    x += maxwidth - width;
	    while (width > maxwidth)
	    {
		int cw = charWidth(font, string[start]);

		width -= cw;
		x += cw;
		length--;
		start++;
	    }
	    break;
	}

	case XmALIGNMENT_BEGINNING:
	default:
	    /*
	     * Leave x alone, but chop characters off the end until we fit
	     */
	    while (width > maxwidth)
	    {
		width -= charWidth(font, string[length - 1]);
		length--;
	    }
	    break;
	}
    }

    /*
     * We fit inside our cell, so just compute the x of the start of our string
     */
    else
    {
	switch (alignment)
	{

	case XmALIGNMENT_CENTER:
	    x += maxwidth / 2 - width / 2;
	    break;

	case XmALIGNMENT_END:
	    x += maxwidth - width;
	    break;

	case XmALIGNMENT_BEGINNING:
	default:
	    /*
	     * Leave x alone
	     */
	    break;
	}
    }

    /*
     * Figure out which GC to use
     */
    if(!rowLabel && !colLabel)
	if(!clip)
	    gc = mw->matrix.draw_gc;
	else
	    gc = mw->matrix.draw_clip_gc;
    else
	if (!clip)
	    gc = mw->matrix.label_gc;
	else
	    gc = mw->matrix.label_clip_gc;

    /*
     * Don't worry, XSetForeground is smart about avoiding unnecessary
     * protocol requests.
     */
    XSetForeground(XtDisplay(mw), gc, color);

    /*
     * Now draw the string at x starting at char 'start' and of length 'length'
     */
#ifdef NEED_WCHAR
    if (TWO_BYTE_FONT(mw))
	XDrawString16(XtDisplay(mw), win, gc, x, y, &string[start], length);
    else
#endif
	XDrawString(XtDisplay(mw), win, gc, x, y, &string[start], length);

    /*
     * If bold is on, draw the string again offset by 1 pixel (overstrike)
     */
    if (bold)
#ifdef NEED_WCHAR
	if (TWO_BYTE_FONT(mw))
	    XDrawString16(XtDisplay(mw), win, gc, x - 1, y,
			  &string[start], length);
	else
#endif
	    XDrawString(XtDisplay(mw), win, gc, x - 1, y,
			&string[start], length);
}

void
xbaeComputeCellColors(mw, row, column, fg, bg)
XbaeMatrixWidget mw;
int row, column;
Pixel *fg, *bg;
{
    Boolean alt = (mw->matrix.alt_row_count &&
		   row >= (int)mw->matrix.fixed_rows) ? (
		       ( (row - (int)mw->matrix.fixed_rows) /
			 mw->matrix.alt_row_count) % 2 ) : False;
    
    /*
     * Compute the background and foreground colours of the cell
     */
    if (mw->matrix.selected_cells && mw->matrix.selected_cells[row][column])
	if( mw->matrix.reverse_select )
	    if( mw->matrix.colors && mw->matrix.colors[ row ][ column ])
		*bg = mw->matrix.colors[ row ][ column ];
	    else
		*bg = mw->manager.foreground;
	else
	    *bg = mw->matrix.selected_background;
    else if (mw->matrix.cell_background &&
	     mw->matrix.cell_background[row][column] !=
	     mw->core.background_pixel )
	*bg = mw->matrix.cell_background[row][column];
    else
    {
	if (alt)
	    *bg = mw->matrix.odd_row_background;
	else
	    *bg = mw->matrix.even_row_background;
    }

    if (mw->matrix.selected_cells && mw->matrix.selected_cells[row][column])
	if( mw->matrix.reverse_select )
	    if (mw->matrix.cell_background && 
		mw->matrix.cell_background[row][column] !=
		mw->manager.foreground)
		*fg = mw->matrix.cell_background[row][column];
	    else
		*fg = mw->core.background_pixel;
	else
	    *fg = mw->matrix.selected_foreground;
    else if( mw->matrix.colors /*&& mw->matrix.colors[ row ][ column ] */)
	*fg = mw->matrix.colors[ row ][ column ];
    else
	*fg = mw->manager.foreground;    
}

void
xbaeDrawCell(mw, row, column)
XbaeMatrixWidget mw;
int row, column;
{
    Pixel bg, fg;
    String string;

    xbaeComputeCellColors(mw, row, column, &fg, &bg);

#if CELL_WIDGETS
    if (mw->matrix.cell_widgets[row][column])
	xbaeDrawCellWidget(mw, row, column,
			   mw->matrix.cell_widgets[row][column], bg, fg );
    else
#endif

    if( !mw->matrix.draw_cell_callback )
    {	
	string = mw->matrix.cells ? mw->matrix.cells[row][column] : "";
	xbaeDrawCellString(mw, row, column, string, bg, fg);
    }
    else
    {
	Pixmap pixmap;
	Pixmap mask;
	XbaeCellType type;
	int width, height;
	int depth;
	
	type = xbaeGetDrawCellValue( mw, row, column, &string, &pixmap,
				     &mask, &width, &height, &bg, &fg, &depth);
	if ( type == XbaeString)
	    xbaeDrawCellString(mw, row, column, string, bg, fg);
	else if( type == XbaePixmap )
	    xbaeDrawCellPixmap(mw, row, column, pixmap, mask, width,
			       height, bg, fg, depth);
    }
}

XbaeCellType
xbaeGetDrawCellValue(mw, row, column, string, pixmap, mask, width,
		     height, bg, fg, depth)
XbaeMatrixWidget mw;
int row;
int column;
String *string;
Pixmap *pixmap;
Pixmap *mask;
int *width, *height;
Pixel *bg, *fg;
int *depth;
{
    XbaeMatrixDrawCellCallbackStruct cbd;

    cbd.reason = XbaeDrawCellReason;
    cbd.row = row;
    cbd.column = column;
    cbd.width = COLUMN_WIDTH(mw, column) - TEXT_WIDTH_OFFSET(mw) * 2;
    cbd.height = ROW_HEIGHT(mw) - TEXT_HEIGHT_OFFSET(mw) * 2;
    cbd.type = XbaeString;
    cbd.string = "";
    cbd.pixmap = (Pixmap)NULL;
    cbd.mask = (Pixmap)NULL;
    cbd.foreground = *fg;
    cbd.background = *bg;

    XtCallCallbackList((Widget)mw, mw->matrix.draw_cell_callback,
		       (XtPointer) &cbd);

    *pixmap = cbd.pixmap;
    *mask = cbd.mask;
    *string = cbd.string ? cbd.string : ""; /* Handle NULL strings */
    *fg = cbd.foreground;
    *bg = cbd.background;
    *width = cbd.width;
    *height = cbd.height;
    
    if (cbd.type == XbaePixmap)
    {
	if( *mask == XmUNSPECIFIED_PIXMAP || *mask == BadPixmap )
	    cbd.mask = 0;

	if( *pixmap == XmUNSPECIFIED_PIXMAP || *pixmap == BadPixmap )
	{
	    XtAppWarningMsg(
		XtWidgetToApplicationContext((Widget)mw),
		"drawCellCallback", "Pixmap", "XbaeMatrix",
		"XbaeMatrix: Bad pixmap passed from drawCellCallback",
		NULL, 0 );
	    cbd.type = XbaeString;
	    *string = "";
	}
	else
	{
	    Window root_return;
	    int x_return, y_return;
	    unsigned int width_return, height_return;
	    unsigned int border_width_return;
	    unsigned int depth_return;

	    if( XGetGeometry( XtDisplay( mw ), *pixmap, &root_return,
			      &x_return, &y_return, &width_return,
			      &height_return, &border_width_return,
			      &depth_return ) )
	    {
		*width = width_return;
		*height = height_return;
		*depth = depth_return;
	    }
	}
    }
    return (cbd.type);
}

/*
 * Convert a row/column cell position to the x/y of its upper left corner
 * wrt the window it will be drawn in (either the matrix window for
 * fixed cells, or the clip window for non-fixed).
 */
void
xbaeRowColToXY(mw, row, column, x, y )
XbaeMatrixWidget mw;
int row;
int column;
int *x;
int *y;
{
    /*
     * If we are in a fixed cell, calculate x/y relative to Matrixs
     * window (take into account labels etc)
     */
    if (IS_FIXED(mw, row, column))
    {
	/*
	 * Ignore horiz_origin if we are in a fixed column
	 */
	if( (column < (int)mw->matrix.fixed_columns) )
	    *x = COLUMN_LABEL_OFFSET(mw) + COLUMN_POSITION(mw, column);
	else if (column >= TRAILING_HORIZ_ORIGIN(mw))
	{
	    int m;
	    *x = TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw);
	    for (m = TRAILING_HORIZ_ORIGIN(mw); m < column; m++)
		*x += COLUMN_WIDTH(mw, m);
	}
	else
	    *x = COLUMN_LABEL_OFFSET(mw) +
		COLUMN_POSITION(mw, column) - HORIZ_ORIGIN(mw);

	/*
	 * Ignore vert_origin if we are in a fixed row
	 */
	if( (row < (int)mw->matrix.fixed_rows) )
	    *y = ROW_LABEL_OFFSET(mw) + ROW_HEIGHT(mw) * row;
	else if (row >= TRAILING_VERT_ORIGIN(mw))
	{
	    int m;
	    *y = TRAILING_FIXED_ROW_LABEL_OFFSET(mw);
	    for (m = TRAILING_VERT_ORIGIN(mw); m < row; m++)
		*y += ROW_HEIGHT(mw);
	}
	else
	    *y = ROW_LABEL_OFFSET(mw) + ROW_HEIGHT(mw) *
		(row - VERT_ORIGIN(mw));
    }

    /*
     * If we are not fixed we must account for fixed rows/columns
     * and scrolling origins.
     */
    else
    {
	*x = (COLUMN_POSITION(mw, column) -
	      COLUMN_POSITION(mw, mw->matrix.fixed_columns)) -
	    HORIZ_ORIGIN(mw);
	*y = ROW_HEIGHT(mw) * ((row - (int) mw->matrix.fixed_rows) -
			       VERT_ORIGIN(mw));
    }
}

/*
 * Draw the column label for the specified column.  Handles labels in
 * fixed and non-fixed columns.
 */
void
xbaeDrawColumnLabel(mw, column)
XbaeMatrixWidget mw;
int column;
{
    String label;
    int x, y, i;

    if (mw->matrix.column_labels[column][0] == '\0')
	return;

    /*
     * If the column label is in a fixed column, we don't need to account
     * for the horiz_origin
     */
    if (column < (int)mw->matrix.fixed_columns)
	x = COLUMN_LABEL_OFFSET(mw) + COLUMN_POSITION(mw, column) +
	    TEXT_X_OFFSET(mw);
    else if (column >= TRAILING_HORIZ_ORIGIN(mw))
	x = TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw) +
	    COLUMN_POSITION(mw, column) -
	    COLUMN_POSITION(mw, TRAILING_HORIZ_ORIGIN(mw)) +
	    TEXT_X_OFFSET(mw);
    else
	x = COLUMN_LABEL_OFFSET(mw) + (COLUMN_POSITION(mw, column) -
				       HORIZ_ORIGIN(mw)) + TEXT_X_OFFSET(mw);

    /*
     * Set our y to the baseline of the first line in this column
     */
    y = mw->matrix.label_font->max_bounds.ascent +
	mw->matrix.cell_shadow_thickness +
	mw->matrix.cell_highlight_thickness +
	mw->matrix.cell_margin_height +
	(mw->matrix.column_label_maxlines -
	 mw->matrix.column_label_lines[column].lines) * LABEL_HEIGHT(mw) +
	HORIZ_SB_OFFSET(mw);

    label = mw->matrix.column_labels[column];
    for (i = 0; i < mw->matrix.column_label_lines[column].lines; i++)
    {
	xbaeDrawString(mw, XtWindow(mw),
		       label, mw->matrix.column_label_lines[column].lengths[i],
		       x, y, mw->matrix.column_widths[column],
		       mw->matrix.column_label_alignments ?
		       mw->matrix.column_label_alignments[column] :
		       XmALIGNMENT_BEGINNING, False, mw->matrix.bold_labels,
		       (column >= (int)mw->matrix.fixed_columns) &&
		       (column < TRAILING_HORIZ_ORIGIN(mw)), False, True,
		       mw->matrix.column_label_color);
	y += LABEL_HEIGHT(mw);
	label += mw->matrix.column_label_lines[column].lengths[i] + 1;
    }
}

/*
 * Draw the row label for the specified row. Handles labels in fixed and
 * non-fixed rows.
 */
void
xbaeDrawRowLabel(mw, row)
XbaeMatrixWidget mw;
int row;
{
    int y;

    if (mw->matrix.row_labels[row][0] == '\0')
	return;

    /*
     * If the row label is in a fixed row we don't need to account
     * for the vert_origin
     */
    if (row < (int)mw->matrix.fixed_rows)
	y = ROW_LABEL_OFFSET(mw) + ROW_HEIGHT(mw) * row + TEXT_Y_OFFSET(mw);
    else if (row >= TRAILING_VERT_ORIGIN(mw))
	y = TRAILING_FIXED_ROW_LABEL_OFFSET(mw) +
	    ROW_HEIGHT(mw) * (row - TRAILING_VERT_ORIGIN(mw)) + TEXT_Y_OFFSET(mw);
    else
	y = ROW_LABEL_OFFSET(mw) + ROW_HEIGHT(mw) * (row - VERT_ORIGIN(mw)) +
	    LABEL_Y_OFFSET(mw);

    xbaeDrawString(mw, XtWindow(mw),
		   mw->matrix.row_labels[row],
		   strlen(mw->matrix.row_labels[row]),
		   TEXT_X_OFFSET(mw) + VERT_SB_OFFSET(mw), y,
		   mw->matrix.row_label_width,
		   mw->matrix.row_label_alignment, False,
		   mw->matrix.bold_labels, False, True, False,
		   mw->matrix.row_label_color);
}
