/*
 * Copyright(c) 1992 Bell Communications Research, Inc. (Bellcore)
 *                        All rights reserved
 * Permission to use, copy, modify and distribute this material for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of Bellcore not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * Bellcore.
 *
 * BELLCORE MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL BELLCORE OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES RELAT-
 * ING TO THE SOFTWARE.
 */

#include <stdio.h>
#include <Xm/Xm.h>
#include <Wc/WcCreate.h>
#include <Xbae/Matrix.h>

/*
 * Creates a matrix and two option menus.  One menu selects the type of
 * selection desired (row, column or cell).  The other selects the mode
 * (exclusive - only one item at a time, or add - multiple items
 * can be selected).
 * The File menu can be used to load the matrix with dummy data.
 * The Edit menu can be used to copy, clear or delete the currently
 * selected item, or to deselect all selections.
 */

#define RCO(name, func)		WcRegisterConstructor(app, name, func)
#define RCN(name, class)	WcRegisterClassName(app, name, class)
#define RCP(name, class)	WcRegisterClassPtr(app, name, class)

extern void MriRegisterMotif();


typedef enum {
    NoSelection,
    CellSelection,
    RowSelection,
    ColumnSelection
} SelectionType;

typedef enum {
    AddMode,
    ExclusiveMode
} SelectionMode;

typedef struct _SelectionStruct {
    int row, column;
    SelectionType type;
    SelectionMode mode;
    Boolean selected;
    Widget matrix;
} *SelectionPtr, SelectionStruct;

SelectionPtr GetSelectionFromWidget();
void SelectCB(), NewMatrixCB(), RegisterSelectionWidgetCB();
void SetSelectionTypeCB(), SetSelectionModeCB();
void DeleteCB(), CopyCB(), ClearCB(), LoadCB(), DeselectCB();

void
main(argc, argv)
int argc;
char *argv[];
{
    Widget toplevel;
    XtAppContext app;

    toplevel = XtAppInitialize(&app, "Select",
			       NULL, 0,
			       &argc, argv,
			       NULL,
			       NULL, 0);

    MriRegisterMotif(app);
    
    RCN("XbaeMatrix", xbaeMatrixWidgetClass);
    RCP("xbaeMatrixWidgetClass", xbaeMatrixWidgetClass);

    WcRegisterCallback(app, "NewMatrixCB", NewMatrixCB, NULL);
    WcRegisterCallback(app, "RegisterSelectionWidgetCB",
		       RegisterSelectionWidgetCB, NULL);
    WcRegisterCallback(app, "SelectCB", SelectCB, NULL);
    WcRegisterCallback(app, "SetSelectionTypeCB", SetSelectionTypeCB, NULL);
    WcRegisterCallback(app, "SetSelectionModeCB", SetSelectionModeCB, NULL);
    WcRegisterCallback(app, "DeleteCB", DeleteCB, NULL);
    WcRegisterCallback(app, "CopyCB", CopyCB, NULL);
    WcRegisterCallback(app, "ClearCB", ClearCB, NULL);
    WcRegisterCallback(app, "LoadCB", LoadCB, NULL);
    WcRegisterCallback(app, "DeselectCB", DeselectCB, NULL);

    WcWidgetCreation(toplevel);

    XtRealizeWidget(toplevel);
    XtAppMainLoop(app);
}

/*
 * Create our internal matrix object and associate it with the widget
 */
/* ARGSUSED */
void
NewMatrixCB(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    SelectionPtr selection = XtNew(SelectionStruct);
    selection->type = NoSelection;
    selection->selected = False;
    selection->matrix = w;

    XtVaSetValues(w,
		  XmNuserData, selection,
		  NULL);
}

/*
 * Associate the selection object with a widget
 */
/* ARGSUSED */
void
RegisterSelectionWidgetCB(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    SelectionPtr selection;
    Widget matrix;

    if ((matrix = WcFullNameToWidget(w, (String)client_data)) == NULL)
	return;

    selection = GetSelectionFromWidget(matrix);
    XtVaSetValues(w,
		  XmNuserData, selection,
		  NULL);
}

/*
 * Callback when a cell is clicked on
 */
/* ARGSUSED */
void
SelectCB(w, client_data, call_data)
Widget w;
XtPointer client_data;
XbaeMatrixSelectCellCallbackStruct *call_data;
{
    SelectionPtr selection = GetSelectionFromWidget(w);

    if (selection->mode == ExclusiveMode && selection->selected)
	switch (selection->type) {
	case CellSelection:
	    XbaeMatrixDeselectCell(w, selection->row, selection->column);
	    break;
	case RowSelection:
	    XbaeMatrixDeselectRow(w, selection->row);
	    break;
	case ColumnSelection:
	    XbaeMatrixDeselectColumn(w, selection->column);
	    break;
	}

    selection->row = call_data->row;
    selection->column = call_data->column;

    switch (selection->type) {
    case CellSelection:
	XbaeMatrixSelectCell(w, selection->row, selection->column);
	break;
    case RowSelection:
	XbaeMatrixSelectRow(w, selection->row);
	break;
    case ColumnSelection:
	XbaeMatrixSelectColumn(w, selection->column);
	break;
    }

    selection->selected = True;
}

/* ARGSUSED */
void
SetSelectionTypeCB(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    SelectionPtr selection = GetSelectionFromWidget(w);

    XbaeMatrixDeselectAll(selection->matrix);
    selection->selected = False;

    switch (((String)client_data)[0]) {
    case '1':
	selection->type = CellSelection;
	break;
    case '2':
	selection->type = RowSelection;
	break;
    case '3':
	selection->type = ColumnSelection;
	break;
    default:
	XtAppWarningMsg(XtWidgetToApplicationContext(w),
			"setSelectionTypeCB", "badParam", "Select",
			"Bad param passed to SetSelectionTypeCB",
			(String *) NULL, (Cardinal *) NULL);
	break;
    }
}

/* ARGSUSED */
void
SetSelectionModeCB(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    SelectionPtr selection = GetSelectionFromWidget(w);

    XbaeMatrixDeselectAll(selection->matrix);
    selection->selected = False;

    switch (((String)client_data)[0]) {
    case '0':
	selection->mode = ExclusiveMode;
	break;
    case '1':
	selection->mode = AddMode;
	break;
    default:
	XtAppWarningMsg(XtWidgetToApplicationContext(w),
			"setSelectionModeCB", "badParam", "Select",
			"Bad param passed to SetSelectionModeCB",
			(String *) NULL, (Cardinal *) NULL);
	break;
    }
}

/*
 * Delete the selected row or column.
 */
/* ARGSUSED */
void
DeleteCB(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    Widget matrix;
    SelectionPtr selection;

    if ((matrix = WcFullNameToWidget(w, (String)client_data)) == NULL)
	return;

    selection = GetSelectionFromWidget(matrix);

    if (!selection->selected)
	return;

    /*
     * We cop out and only delete the last row/column selected
     */
    switch (selection->type) {
    case RowSelection:
	XbaeMatrixDeleteRows(matrix, selection->row, 1);
	break;
    case ColumnSelection:
	XbaeMatrixDeleteColumns(matrix, selection->column, 1);
	break;
    }
}

/*
 * Clear the selected row, column or cell.
 */
/* ARGSUSED */
void
ClearCB(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    Widget matrix;
    int rows, columns;
    int i;
    SelectionPtr selection;

    if ((matrix = WcFullNameToWidget(w, (String)client_data)) == NULL)
	return;

    selection = GetSelectionFromWidget(matrix);

    if (!selection->selected)
	return;

    XtVaGetValues(matrix,
		  XmNcolumns, &columns,
		  XmNrows, &rows,
		  NULL);

    /*
     * We only clear the last selection made
     */
    switch (selection->type) {
    case CellSelection:
	XbaeMatrixSetCell(matrix, selection->row, selection->column, "");
	break;
    case RowSelection:
	for (i = 0; i < columns; i++)
	    XbaeMatrixSetCell(matrix, selection->row, i, "");
	break;
    case ColumnSelection:
	for (i = 0; i < rows; i++)
	    XbaeMatrixSetCell(matrix, i, selection->column, "");
	break;
    }
}
/*
 * Copy the selected row or column. Place the copy below it.
 */
/* ARGSUSED */
void
CopyCB(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    Widget matrix;
    int rows;
    String **cells;
    String *data;
    String label, *row_labels, *column_labels;
    short *widths, width;
    int *lengths, length;
    int i;
    SelectionPtr selection;

    if ((matrix = WcFullNameToWidget(w, (String)client_data)) == NULL)
	return;

    selection = GetSelectionFromWidget(matrix);

    if (!selection->selected)
	return;

    XtVaGetValues(matrix,
		  XmNcells, &cells,
		  XmNrows, &rows,
		  XmNcolumnWidths, &widths,
		  XmNcolumnMaxLengths, &lengths,
		  XmNrowLabels, &row_labels,
		  XmNcolumnLabels, &column_labels,
		  NULL);

    /*
     * Using data retrieved via GetValues in an Xt/widget function
     * without copying it locally is dangerous.
     * The below is just for demo purposes.
     * We only copy the last selection made
     */
    switch (selection->type) {
    case RowSelection:
	label = row_labels ? row_labels[selection->row] : NULL;
	XbaeMatrixAddRows(matrix, selection->row, cells[selection->row],
			  &label, NULL, 1);
	break;
    case ColumnSelection:
	data = (String *)XtMalloc(rows * sizeof(String));
	for (i = 0; i < rows; i++)
	    data[i] = cells[i][selection->column];
	/* We could get colors, alignments etc and copy those too,
	 * but I am lazy */
	width = widths[selection->column];
	length = lengths ?
	    lengths[selection->column] : widths[selection->column];
	label = column_labels ? column_labels[selection->column] : NULL;
	XbaeMatrixAddColumns(matrix, selection->column, data,
			     &label, &width, &length, NULL, NULL, NULL, 1);
	XtFree((XtPointer)data);
	break;
    }
}

/* ARGSUSED */
void
DeselectCB(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    Widget matrix;
    SelectionPtr selection;

    if ((matrix = WcFullNameToWidget(w, (String)client_data)) == NULL)
	return;

    selection = GetSelectionFromWidget(matrix);

    XbaeMatrixDeselectAll(matrix);
    selection->selected = False;
}

/* ARGSUSED */
void
LoadCB(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    int i, j;
    int rows, columns;
    char buf[BUFSIZ];
    String **cells;
    Widget matrix;

    if ((matrix = WcFullNameToWidget(w, client_data)) == NULL)
	return;

    XtVaGetValues(matrix,
		  XmNrows, &rows,
		  XmNcolumns, &columns,
		  NULL);

    cells = (String **) XtMalloc(sizeof(String *) * rows);
    for (i = 0; i < rows; i++) {
	cells[i] = (String *) XtMalloc(sizeof(String) * columns);
	for (j = 0; j < columns; j++) {
	    sprintf(buf, "r%dc%d", i, j);
	    cells[i][j] = XtNewString(buf);
	}
    }

    XtVaSetValues(matrix,
		  XmNcells, cells,
		  NULL);

    for (i = 0; i < rows; i++) {
	for (j = 0; j < columns; j++)
	    XtFree((XtPointer)cells[i][j]);
	XtFree((XtPointer)cells[i]);
    }
    XtFree((XtPointer)cells);
}

SelectionPtr
GetSelectionFromWidget(w)
Widget w;
{
    SelectionPtr selection = NULL;

    XtVaGetValues(w,
		  XmNuserData, &selection,
		  NULL);
    if (selection == NULL)
	XtAppErrorMsg(XtWidgetToApplicationContext(w),
		      "getSelectionFromWidget", "notRegistered", "Select",
		      "Widget not registered",
		      NULL, 0);
    return selection;
}
