# ro-tag.tcl:  implement a read-only tag for text widgets.  Characters tagged
#  with the tag name 'readonly' will not be editable.  For the most part this
#  is accomplished by modifying the routine tkTextInsert to check for the
#  tag before inserting any characters.  Bindings that result in deletion
#  of characters have to be handled separately, so these bindings are
#  redefined.

###     ro-tag.tcl : implement read-only text tags for text widgets.
###     Copyright (C) 1995 Norman Danner
### 
###     This program is free software; you can redistribute it and/or modify
###     it under the terms of the GNU General Public License as published by
###     the Free Software Foundation; either version 2 of the License, or
###     (at your option) any later version.
### 
###     This program is distributed in the hope that it will be useful,
###     but WITHOUT ANY WARRANTY; without even the implied warranty of
###     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
###     GNU General Public License for more details.
### 
###     You should have received a copy of the GNU General Public License
###     along with this program; if not, write to the Free Software
###     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

if {$tk_version >= 4.0} {

  bind Text <Delete> {
    if {[%W tag nextrange sel 1.0 end] != ""} {
      if {[ROtextReadOnlyTag %W sel.first sel.last]} \
        {return} {%W delete sel.first sel.last}
    } else {
      if {[ROtextReadOnlyTag %W insert]} {return}
      %W delete insert
      %W see insert
    }
  }
  bind Text <BackSpace> {
    if {[%W tag nextrange sel 1.0 end] != ""} {
      if {[ROtextReadOnlyTag %W sel.first sel.last]} \
        {return} {%W delete sel.first sel.last}
    } else {
      if {[ROtextReadOnlyTag %W insert-1c]} {return}
      %W delete insert-1c
      %W see insert
    }
  }
  
  # Additional emacs-like bindings:
  
  if !$tk_strictMotif {
      bind Text <Control-d> {
  	if {![ROtextReadOnlyTag %W insert]} {%W delete insert}
      }
      bind Text <Control-k> {
  	if [%W compare insert == {insert lineend}] {
  	    if {![ROtextReadOnlyTag %W insert]} {%W delete insert}
  	} else {
  	    %W delete insert [ROtextNextReadOnly %W insert]
  	}
      }
      bind Text <Control-o> {
        if {![ROtextReadOnlyTag %W insert]} {
  	%W insert insert \n
  	%W mark set insert insert-1c
        }
      }
      bind Text <Control-t> {
  	tkTextTranspose %W
      }
      bind Text <Meta-d> {
          if {[ROtextReadOnlyTag %W insert {insert wordend}]} {
            %W delete insert [ROtextNextReadOnly %W insert]
          } else {
  	  %W delete insert {insert wordend}
  	}
      }
      bind Text <Meta-BackSpace> {
        	if {[ROtextReadOnlyTag %W {insert -1c wordstart} insert]} {
            %W delete [ROtextPrevReadOnly %W {insert-1c}]
          } else {
            %W delete {insert -1c wordstart} insert
          }
      }
  
      # A few additional bindings of my own.
  
      bind Text <Control-h> {
  	if {([%W compare insert != 1.0]) && \
              (![ROtextReadOnlyTag %W insert-1c])} {
  	    %W delete insert-1c
  	    %W see insert
  	}
      }
  }
  
  # tkTextInsert --
  # Insert a string into a text at the point of the insertion cursor.
  # If there is a selection in the text, and it covers the point of the
  # insertion cursor, then delete the selection before inserting.
  #
  # Arguments:
  # w -		The text window in which to insert the string
  # s -		The string to insert (usually just a single character)
  
  proc tkTextInsert {w s} {
      if {($s == "") || ([$w cget -state] == "disabled")} {
  	return
      }
      if {[ROtextReadOnlyTag $w insert] && [ROtextReadOnlyTag $w insert-1c] && \
          ([$w index insert] != 1.0)} {return}
      catch {
  	if {[$w compare sel.first <= insert]
  		&& [$w compare sel.last >= insert]} {
  	    $w delete sel.first sel.last
  	}
      }
      $w insert insert $s
      $w see insert
  }
  

} else {
  
  source $tk_library/tk.tcl
  source $tk_library/text.tcl
  
  proc tk_textBackspace w {
      if {[ROtextReadOnlyTag $w insert-1c]} {return}
      $w delete insert-1c insert
  }

  bind Text <Any-KeyPress> {
    if {[ROtextReadOnlyTag %W insert] && [ROtextReadOnlyTag %W insert-1c] && \
        ([%W index insert] != 1.0) && ([%W compare [%W index insert] == end])} \
        {return}
    if {"%A" != ""} {
	%W insert insert %A
	if {[ROtextReadOnlyTag %W insert-1c]} {%W tag remove readonly insert-1c}
	%W yview -pickplace insert
    }
  }
  bind Text <Return> {
    if {[ROtextReadOnlyTag %W insert]} return
    %W insert insert \n; %W yview -pickplace insert
  }
  bind Text <Control-d> {
    if {[ROtextReadOnlyTag %W sel.first sel.last]} \
      {return} {%W delete sel.first sel.last}
  }
  bind Text <Control-v> {
      if {[ROtextReadOnlyTag %W insert]} return
      %W insert insert [selection get]
      %W yview -pickplace insert
  }

}

proc ROtextReadOnlyTag {w i1 {i2 ""}} {
  
  if {$i2 == ""} {
    if {[lsearch -exact [$w tag names $i1] readonly] != -1} \
      {return 1} {return 0}
  } else {
    for {set i 0} {[$w compare "${i1}+${i}c" < $i2]} {incr i} {
      if {[lsearch -exact [$w tag names $i1+${i}c] readonly] != -1} {return 1}
    }
    return 0
  }
  
}

proc ROtextNextReadOnly {w i} {
  
  if {![ROtextReadOnlyTag $w $i "$i lineend"]} {return "$i lineend"}
  for {set j 0} {[$w compare "${i}+${j}c" <= "${i} lineend"]} {incr j} {
    if {[ROtextReadOnlyTag $w "${i}+${j}c"]} {
      return [$w index ${i}+${j}c]
    }
  }
  
}

proc ROtextPrevReadOnly {w i} {
  
  if {![ROtextReadOnlyTag $w "$i linestart" $i]} {return "$i linestart"}
  for {set j 0} {[$w compare "${i}-${j}c" >= "${i} linestart"]} {incr j} {
    if {[ROtextReadOnlyTag $w "${i}-${j}c"]} {
      return [$w index ${i}-${j}c]
    }
  }
  
}

# A do-nothing proc--call it to force auto-loading of this file.

proc ROinit {} {}

