/*
 * bltGraph.h --
 *
 * Copyright 1991-1994 by AT&T Bell Laboratories.
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the
 * names of AT&T Bell Laboratories any of their entities not be used
 * in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * AT&T disclaims all warranties with regard to this software, including
 * all implied warranties of merchantability and fitness.  In no event
 * shall AT&T be liable for any special, indirect or consequential
 * damages or any damages whatsoever resulting from loss of use, data
 * or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or
 * performance of this software.
 *
 * Graph widget created by Sani Nassif and George Howlett.
 */

#ifndef _GRAPH_H
#define _GRAPH_H

#include <math.h>
#ifdef HAVE_FLOAT_H
#include <float.h>
#endif
#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

/* math.h */

/*
 * ----------------------------------------------------------------------
 *
 * 	MAX_DBL_VALUE and MIN_DBL_VALUE:
 *
 *	There values are the largest and small values representable
 * 	on the graph or barchart.  They are used for the following
 *
 *	1. when searching for smallest or largest value of a set,
 *	    initialize current min/max to this.
 *	2. represent tag coordinates "Inf" and "-Inf".
 *	3. uninitialized floating point values.
 *
 * ----------------------------------------------------------------------
 */
#ifdef HAVE_FLOAT_H
#define MAX_DBL_VALUE 	DBL_MAX
#define MIN_DBL_VALUE	DBL_MIN
#else
/* Don't want to include __infinity (definition of HUGE_VAL (SC1.x)) */
#ifdef sun
#define MAX_DBL_VALUE	1.7976931348623157E+308
#define MIN_DBL_VALUE 	2.2250738585072014E-308
#else
/* Start to go into punt formation. */
#ifdef HUGE_VAL
#define MAX_DBL_VALUE   HUGE_VAL
#define MIN_DBL_VALUE   -HUGE_VAL
#else
#ifdef HUGE
#define MAX_DBL_VALUE	HUGE
#define MIN_DBL_VALUE	-HUGE
#else
/* Punt: Assume something simple and relatively small */
#define MAX_DBL_VALUE	3.40282347E+38
#define MIN_DBL_VALUE 	1.17549435E-38
#endif /*HUGE*/
#endif /*HUGE_VAL*/
#endif /*sun*/
#endif /*HAVE_FLOAT_H*/

#undef MIN
#define MIN(a,b)	(((a)<(b))?(a):(b))

#undef MAX
#define MAX(a,b)	(((a)>(b))?(a):(b))

/*
 * ------------------------------------------------------------------------
 *
 * 	Replacement macros for math library functions, fabs, abs, rint,
 *	and exp10.
 *
 * 	Although many of these routines may be in the math library, they
 *	aren't used in libtcl.a or libtk.a.  This makes it impossible to
 *	load BLT library as a shared object without requiring that the
 *	math library be a shared object too.  Avoid the problem by
 *	replacing the "exotic" math routines with macros.
 *
 * ------------------------------------------------------------------------
 */

#undef FABS
#define FABS(x)		(((x)<0.0)?(-(x)):(x))

#undef ABS
#define ABS(x)		(((x)<0)?(-(x)):(x))

#undef ROUND
#define ROUND(x)        ((int)((x) + (((x)<0.0) ? -0.5 : 0.5)))

#undef EXP10
#define EXP10(x)	(pow(10.0,(x)))

#ifndef M_PI
#define M_PI    	3.14159265358979323846
#endif /* M_PI */

#ifndef M_SQRT2
#define M_SQRT2		1.41421356237309504880
#endif /* M_SQRT1_2 */

#ifndef M_SQRT1_2
#define M_SQRT1_2	0.70710678118654752440
#endif /* M_SQRT1_2 */

#ifndef SHRT_MAX
#define SHRT_MAX	0x7FFF
#endif /* SHRT_MAX */

#include "bltList.h"

#define TRUE 1
#define FALSE 0

/*
 * Mask values used to selectively enable entries in the
 * configuration specs:
 */

#define XYGRAPH_MASK	TK_CONFIG_USER_BIT
#define BARCHART_MASK	TK_CONFIG_USER_BIT << 1
#define CONTOUR_MASK	TK_CONFIG_USER_BIT << 2
#define ALL_MASK	(XYGRAPH_MASK | BARCHART_MASK)

/*
 * NUMDIGITS specifies the number of digits accuracy used when
 * outputing data (e.g. axis tick labels, etc)
 */
#define NUMDIGITS	12

#define PADX		2	/* Padding between labels/titles */
#define PADY    	2	/* Padding between labels */
#define TEXTHEIGHT(f) 	((f)->ascent + (f)->descent)

#define DEF_POSITION -SHRT_MAX	/* Indicates that no position was specified */

/*
 * -------------------------------------------------------------------
 * 	Graph component structure definitions
 * -------------------------------------------------------------------
 */

typedef struct Graph Graph;
typedef struct GraphAxis GraphAxis;
typedef struct GraphLegend GraphLegend;
typedef struct GraphPostScript GraphPostScript;
typedef struct GraphCrosshairs GraphCrosshairs;

/*
 * -------------------------------------------------------------------
 *
 * 	GraphClassType --
 *
 *	Enumerates the different types of graphs this program
 *	produces. Currently, only one this of graph is allowed
 *	on a single axis.
 *
 * -------------------------------------------------------------------
 */
typedef enum {
    GRAPH, BARCHART, CONTOUR
} GraphClassType;

/*
 * -------------------------------------------------------------------
 *
 * 	TextAttr --
 *
 * 	Represents a convenience structure to hold text attributes
 *	which determine how a text string is to be displayed on the
 *	window, or drawn with PostScript commands.  The alternative
 *	is to have drawing and printing routines with more parameters.
 * 	This seems like a more efficient and less cumbersome way.
 *
 * -------------------------------------------------------------------
 */
typedef struct {
    char *text;			/* Text string to be displayed  */
    XFontStruct *fontPtr;	/* Font to use for string */
    XColor *bgColorPtr;		/* Background color of string */
    XColor *fgColorPtr;		/* Foreground color of string */
    Tk_Anchor anchor;		/* Anchor type: used to position the string */
    double theta;		/* Rotation of text in degrees. */
    GC gc;			/* Graphics context to use when displaying the
				 * string on the window */
} TextAttr;

/*
 * -------------------------------------------------------------------
 *
 * 	AxisTypes --
 *
 *	Enumerated type representing the types of axes
 *
 * -------------------------------------------------------------------
 */
typedef enum AxisTypes {
    X1, Y1, X2, Y2
} AxisType;

#define AXIS_X1_FLAG		(1<<X1)
#define AXIS_Y1_FLAG		(1<<Y1)
#define AXIS_X2_FLAG		(1<<X2)
#define AXIS_Y2_FLAG		(1<<Y2)
#define AXIS_X_MASK		(AXIS_X1_FLAG|AXIS_X2_FLAG)
#define AXIS_Y_MASK		(AXIS_Y1_FLAG|AXIS_Y2_FLAG)
#define AXIS_TYPE_MASK		(AXIS_Y_MASK|AXIS_X_MASK)
#define AXIS_LIMIT_USER_BIT	0x10
#define AXIS_LIMIT_MIN_MASK	(AXIS_LIMIT_USER_BIT << LMIN)
#define AXIS_LIMIT_MAX_MASK	(AXIS_LIMIT_USER_BIT << LMAX)
#define AXIS_LIMIT_MASK		(AXIS_LIMIT_MIN_MASK | AXIS_LIMIT_MAX_MASK)
#define AXIS_MIN_AUTOSCALE(a) 	(((a)->flags & AXIS_LIMIT_MIN_MASK)==0)
#define AXIS_MAX_AUTOSCALE(a) 	(((a)->flags & AXIS_LIMIT_MAX_MASK)==0)

/*
 * -------------------------------------------------------------------
 *
 * 	GraphAxis --
 *
 * 	Structure contains options controlling how the axis will be
 * 	displayed.
 *
 * -------------------------------------------------------------------
 */

typedef void (AxisDisplayProc) _ANSI_ARGS_((Graph *, GraphAxis *, TextAttr *));
typedef void (AxisPrintProc) _ANSI_ARGS_((Graph *, GraphAxis *, TextAttr *));
typedef void (AxisLayoutProc) _ANSI_ARGS_((Graph *, GraphAxis *));
typedef void (AxisDestroyProc) _ANSI_ARGS_((Graph *, GraphAxis *));

struct GraphAxis {
    unsigned int flags;		/* Axis flags */
    int logScale;		/* If non-zero, scale values logarithmically */
    int mapped;			/* If non-zero, display the axis */

    AxisDisplayProc *displayProc;
    AxisPrintProc *printProc;
    AxisLayoutProc *layoutProc;
    AxisDestroyProc *destroyProc;
};

/*
 * -------------------------------------------------------------------
 *
 * 	GraphLegend --
 *
 * 	Contains information specific to how the legend will be
 *	displayed.
 *
 * -------------------------------------------------------------------
 */

typedef void (LegendDisplayProc) _ANSI_ARGS_((Graph *));
typedef void (LegendPrintProc) _ANSI_ARGS_((Graph *));
typedef void (LegendDestroyProc) _ANSI_ARGS_((Graph *));
typedef int (LegendGeometryProc) _ANSI_ARGS_((Graph *));

struct GraphLegend {
    int mapped;			/* Requested state of the legend, If non-zero,
				 * legend is displayed */
    int width, height;		/* Dimensions of the legend */
    XPoint anchorPos;		/* Window coordinates of legend positioning
				 * point. Used in conjunction with the anchor
				 * to determine the location of the legend. If
				 * x or y are DEF_POSITION the legend is set
				 * in the right margin */
    int useDefault;		/* Draw the legend in the default location */

    LegendDisplayProc *displayProc;
    LegendPrintProc *printProc;
    LegendDestroyProc *destroyProc;
    LegendGeometryProc *geomProc;
};

/*
 * -------------------------------------------------------------------
 *
 * 	GraphPostScript --
 *
 * 	Structure contains information specific to the outputing of
 *	PostScript commands to print the graph.
 *
 * -------------------------------------------------------------------
 */

typedef int (PostScriptConfigureProc) _ANSI_ARGS_((Graph *, int, char **));
typedef int (PostScriptPrintProc) _ANSI_ARGS_((Graph *, int, char **));
typedef void (PostScriptDestroyProc) _ANSI_ARGS_((Graph *));

struct GraphPostScript {
    PostScriptConfigureProc *configProc;
    PostScriptPrintProc *printProc;
    PostScriptDestroyProc *destroyProc;
};

/*
 * -------------------------------------------------------------------
 *
 * 	GraphCrosshairs --
 *
 *	Contains the line segments positions and graphics context used
 *	to simulate crosshairs (by XORing) on the graph.
 *
 * -------------------------------------------------------------------
 */
typedef void (CrosshairsToggleProc) _ANSI_ARGS_((Graph *));
typedef void (CrosshairsUpdateProc) _ANSI_ARGS_((Graph *));
typedef void (CrosshairsConfigProc) _ANSI_ARGS_((Graph *));
typedef void (CrosshairsDestroyProc) _ANSI_ARGS_((Graph *));

struct GraphCrosshairs {

    CrosshairsToggleProc *toggleProc;	/* Toggle visiblity of crosshairs */
    CrosshairsUpdateProc *updateProc;	/* Update lengths of hairs */
    CrosshairsConfigProc *configProc;	/* Configure GC */
    CrosshairsDestroyProc *destroyProc;	/* Release X resources */


};

/*
 * -------------------------------------------------------------------
 *
 * 	Graph --
 *
 *	Top level structure containing everything pertaining to
 *	the graph.
 *
 * -------------------------------------------------------------------
 */
struct Graph {
    Tk_Window tkwin;		/* Window that embodies the graph.  NULL means
				 * that the window has been destroyed but the
				 * data structures haven't yet been cleaned
				 * up. */
    Pixmap canvas;		/* Pixmap for double buffering output */
    Display *display;		/* Display containing widget;  needed, among
				 * other things, to release resources after
				 * tkwin has already gone away. */
    char *pathName;		/* Pathname of the widget. Is saved here in
				 * case the widget is destroyed and tkwin
				 * become unavailable for querying */
    Tcl_Interp *interp;		/* Interpreter associated with graph widget */
    GraphClassType type;	/* Type: either GRAPH or BARCHART */
    unsigned int flags;		/* Flags;  see below for definitions. */
    int reqWidth, reqHeight;	/* Requested size of graph window */
    int width, height;		/* Size of graph window or PostScript page */

    int halo;			/* Maximum distance allowed between points
				 * when searching for a point */

    GraphPostScript *postscript;/* PostScript options: see bltGrPS.c */
    GraphLegend *legendPtr;	/* Legend information: see bltGrLegd.c */
    GraphAxis *axisArr[4];	/* Coordinate axis info: see bltGrAxis.c */
    GraphCrosshairs *crosshairs;/* Crosshairs information: see bltGrHairs.c */

    char *scratchPtr;		/* Utility space for building strings. Points
				 * to buffer of BUFSIZ bytes on the stack.
				 * Currently used to create PostScript output
				 * for the "postscript" command. */

    int leftMargin;		/* Requested sizes for margins surrounding */
    int rightMargin;		/* the plotting area. If non-zero, the */
    int topMargin;		/* requested margin is used, otherwise the */
    int bottomMargin;		/* computed margin sizes are used. */

    XPoint origin;		/* Origin of the graph */
    XPoint extreme;		/* Outer limit of graph */

    Tk_3DBorder border;		/* 3-D border used to delineate the plot
				 * surface and outer edge of window */
    GC intGC;			/* Interior region (plotting surface) GC */
    int intBWidth;		/* Width of interior 3-D border. */
    int intRelief;		/* 3-d effect: TK_RELIEF_RAISED etc. */

    GC extGC;			/* Graphics context for margin. Includes
				 * margin background */
    GC extBgGC;			/* Background GC for margin areas */
    int extBWidth;		/* Width the exterior border */
    int extRelief;		/* Relief of the exterior border */

    XColor *intBgColorPtr;	/* Color of plot surface */
    double barWidth;		/* Scale factor for the width of bar */
    double avgSymSize;		/* Average size of a symbol */

    char *title;		/* Graph title */

    XFontStruct *fontPtr;	/* Font for title and axis labels */
    XColor *fgColorPtr;		/* Foreground color of title and axis labels */
    Cursor cursor;

    Tcl_HashTable elemTable;	/* Hash table containing all elements created,
				 * not just those currently displayed. */
    Blt_LinkedList elemList;	/* Display list of elements */
    Tcl_HashTable tagTable;	/* Hash table of tags */
    Blt_LinkedList tagList;	/* Display list of tags */
    int nextTagId;		/* Tracks next tag identifier available */

};

/*
 * Flag bits for graphs:
 *
 * REDRAW_PENDING:		Non-zero means a DoWhenIdle handler has
 *				already been queued to redraw this window.
 *
 * REDRAW_ALL:		        Non-zero means that exterior region of the
 *				graph, in addition to the interior (plotting
 *				surface) needs to be redrawn. The possible
 *				reasons are: 1) an axis configuration changed
 *				2) an axis limit changed 3) titles have changed
 *				4) window was resized.
 *
 * LAYOUT_PENDING:		Non-zero means that a graph configuration
 *				has changed (element, tag, axis, legend, etc)
 *				and the layout of the graph (position of the
 *				graph in the window) needs to be recalculated.
 *
 * LAYOUT_ALL:		        Non-zero indicates that the layout of the axes
 *				and all elements and tags need to be
 *				recalculated.  Otherwise, the layout routines
 *				will recompute the layout of only those tags
 *				and elements that have changed.
 *
 */
#define REDRAW_PENDING		(1<<0)
#define REDRAW_ALL		(1<<1)
#define LAYOUT_PENDING	        (1<<2)
#define LAYOUT_ALL		(1<<3)

/*
 * ---------------------- Forward declarations ------------------------
 */
extern Pixmap Blt_CreateTextBitmap _ANSI_ARGS_((Display * display,
	Drawable draw, XFontStruct *fontPtr, char *textStr, double theta,
	unsigned int *bmWPtr, unsigned int *bmHPtr));
extern void Blt_DrawText _ANSI_ARGS_((Display * display, Drawable draw,
	char *text, TextAttr *attrPtr, int x, int y));
extern void Blt_EventuallyRedraw _ANSI_ARGS_((Graph *graphPtr));
extern void Blt_GetBoundingBox _ANSI_ARGS_((unsigned int width,
	unsigned int height, double theta, unsigned int *widthPtr,
	unsigned int *heightPtr, XPoint *pointArr));
extern Pixmap Blt_RotateBitmap _ANSI_ARGS_((Display * display, Drawable draw,
	GC gc, Pixmap bitmap, unsigned int width, unsigned int height,
	double theta, unsigned int *rotWPtr, unsigned int *rotHPtr));
extern void Blt_ComputeAxes _ANSI_ARGS_((Graph *graphPtr));

extern int Blt_Transform _ANSI_ARGS_((GraphAxis *axis, double value));
extern double Blt_InvTransform _ANSI_ARGS_((GraphAxis *axis, int coord));
extern XPoint Blt_TransformPt _ANSI_ARGS_((Graph *graphPtr, double x, double y,
	unsigned int axisFlags));
extern int Blt_TransformDist _ANSI_ARGS_((GraphAxis *axis, double value));

extern void Blt_StencilBitmap _ANSI_ARGS_((Display * display, Drawable draw,
	GC gc, Pixmap bitmap, int x, int y, unsigned int width,
	unsigned int height));
extern unsigned int Blt_TextStringWidth _ANSI_ARGS_((XFontStruct *fontPtr,
	char *text));
extern XPoint Blt_TranslateBoxCoords _ANSI_ARGS_((int x, int y,
	unsigned int width, unsigned int height, Tk_Anchor anchor));
extern XPoint Blt_TranslateTextCoords _ANSI_ARGS_((XFontStruct *fontPtr,
	char *text, int x, int y, Tk_Anchor anchor));

extern void Blt_PrintLine _ANSI_ARGS_((Graph *graphPtr, XPoint *pointArr,
	int numPoints));
extern void Blt_PrintBitmap _ANSI_ARGS_((Graph *graphPtr, Pixmap bitmap, int x,
	int y, unsigned int width, unsigned int height));
extern void Blt_SetLineAttributes _ANSI_ARGS_((Graph *graphPtr,
	XColor *colorPtr, int lineWidth, int lineDashes));
extern void Blt_3DRectangleToPostScript _ANSI_ARGS_((Graph *graphPtr,
	Tk_3DBorder border, int x, int y, unsigned int width,
	unsigned int height, int borderWidth, int relief));
extern void Blt_BackgroundToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XColor *colorPtr));
extern void Blt_BitmapToPostScript _ANSI_ARGS_((Graph *graphPtr, Pixmap bitmap,
	int x, int y, unsigned int width, unsigned int height, double theta,
	XColor *bgColorPtr));
extern void Blt_FontToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XFontStruct *fontPtr));
extern void Blt_ForegroundToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XColor *colorPtr));
extern void Blt_LineDashesToPostScript _ANSI_ARGS_((Graph *graphPtr,
	int lineDashes));
extern void Blt_LineWidthToPostScript _ANSI_ARGS_((Graph *graphPtr,
	int lineWidth));
extern void Blt_LinesToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XPoint *pointArr, int numPoints));
extern void Blt_PolygonToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XPoint *pointArr, int numPoints));
extern void Blt_Print3DRectangle _ANSI_ARGS_((Graph *graphPtr,
	Tk_3DBorder border, int x, int y, unsigned int width,
	unsigned int height, int borderWidth, int relief));
extern void Blt_RectangleToPostScript _ANSI_ARGS_((Graph *graphPtr,
	int x, int y, unsigned int width, unsigned int height));
extern void Blt_RectanglesToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XRectangle *rectArr, int numRects));
extern void Blt_SegmentsToPostScript _ANSI_ARGS_((Graph *graphPtr,
	XSegment *segArr, int numSegs));
extern void Blt_StippleToPostScript _ANSI_ARGS_((Graph *graphPtr,
	Pixmap bitmap, unsigned int width, unsigned int height, int fgOrBg));
extern void Blt_TextToPostScript _ANSI_ARGS_((Graph *graphPtr, char *text,
	TextAttr *attrPtr, int x, int y));

/*
 * ---------------------- Global declarations ------------------------
 */
extern double Blt_negInfinity, Blt_posInfinity;

#endif /* _GRAPH_H */
