/*
 *=============================================================================
 *                                tSippInt.h
 *-----------------------------------------------------------------------------
 * Internal, global declarations for Tcl SIPP interface.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1995 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippInt.h,v 5.15 1996/12/09 07:04:05 markd Exp $
 *=============================================================================
 */

#ifndef TSIPPINT_H
#define TSIPPINT_H

#include <stdio.h>
#include <math.h>
#ifdef HAVE_STRING_H
#   include <string.h>
#else
#   include <strings.h>
#endif
#ifdef HAVE_MEMORY_H
#   include <memory.h>
#endif
#ifdef HAVE_VALUES_H
#   include <values.h>
#endif
#include <time.h>
#include <sys/types.h>

#ifdef STDC_HEADERS
#   include <stdlib.h>
#endif

char *
ctime ();

#include <tclExtend.h>
#include "tSipp.h"
#include "../libsipp/sipp.h"
#include "../libsipp/shaders.h"
#include "../libsipp/primitives.h"
#include "../libsipp/sipp_bitmap.h"
#include "smalloc.h"
#include "xalloca.h"

/*
 * Forward and external struct declarations.
 */
struct tSippStorageClass_t;
struct tSippGlob_t;
struct tSippOutputParms_t;

/*
 *-----------------------------------------------------------------------------
 * Image Store class table declaration.  See tSippIStore.c for full
 * documention on its use and the value of the fields.
 *-----------------------------------------------------------------------------
 */

/*
 * Maximum number of image storage class that are currently supported.
 */
#define TSIPP_MAX_STORAGE_CLASSES  4

/*
 * Supported scanning direction.
 */
typedef enum {
    TSIPP_SCAN_RANDOM,
    TSIPP_SCAN_TOP_DOWN,
    TSIPP_SCAN_BOTTOM_UP
} scanDirection_t;

/*
 * Storage class identification function. 
 */
typedef int
classIdent_t _ANSI_ARGS_((struct tSippGlob_t *tSippGlobPtr,
                          char               *handle));

/*
 * Function called at the start of output.
 */
typedef void_pt
outputStart_t _ANSI_ARGS_((struct tSippGlob_t        *tSippGlobPtr,
                           struct tSippOutputParms_t *outputParmsPtr,
                           char                      *handle,
                           char                     **comments));

/*
 * Function called to output a scan line.
 */
typedef bool
outputLine_t _ANSI_ARGS_((struct tSippGlob_t  *tSippGlobPtr,
                          void_pt              clientData,
                          int                  y,
                          u_char              *rowPtr));

/*
 * Function called to output a bitmap.
 */
typedef bool
outputBitMap_t _ANSI_ARGS_((struct tSippGlob_t  *tSippGlobPtr,
                            void_pt              clientData,
                            Sipp_bitmap         *bitMapPtr));

/*
 * Function called at the end of output.
 */
typedef bool
outputEnd_t _ANSI_ARGS_((struct tSippGlob_t         *tSippGlobPtr,
                         struct tSippOutputParms_t  *outputParmsPtr,
                         void_pt                     clientData));

/*
 * Function called to copy an image.
 */
typedef bool
copyImage_t _ANSI_ARGS_((struct tSippGlob_t          *tSippGlobPtr,
                         char                        *srcHandle,
                         char                        *destHandle,
                         struct tSippStorageClass_t  *destClassPtr,
                         bool                         clear));

/*
 * Function called to set an image comment.
 */
typedef bool
commentPut_t _ANSI_ARGS_((struct tSippGlob_t  *tSippGlobPtr,
                          char                *handle,
                          char                *name,
                          char                *value));

/*
 * Function called to get an image comment.
 */
typedef bool
commentGet_t _ANSI_ARGS_((struct tSippGlob_t  *tSippGlobPtr,
                          char                *handle,
                          char                *name,
                          char               **valuePtr));


/*
 * Function called to get a pointer to the image comment argv.
 */
typedef bool
commentGetVec_t _ANSI_ARGS_((struct tSippGlob_t   *tSippGlobPtr,
                             char                *handle,
                             char               ***arvgPtr));

/*
 * Function called to delete an image comment or all image comments.
 */
typedef bool
commentDelete_t _ANSI_ARGS_((struct tSippGlob_t  *tSippGlobPtr,
                             char                *handle,
                             char                *name));



/*
 * Function called to free all image comment.
 */
typedef bool
commentFree_t _ANSI_ARGS_((struct tSippGlob_t  *tSippGlobPtr,
                           char                *handle));

/*
 * Structure used to describe an image store class.  See tSippIStore.c
 * for field descriptions.
 */
typedef struct tSippStorageClass_t {
    char             *handlePrefix;
    int               prefixSize;
    classIdent_t     *identFunc;
    scanDirection_t   scanDirection;
    bool              bitMapOptimal;

    outputStart_t    *outputStart;
    outputLine_t     *outputLine;
    outputBitMap_t   *outputBitMap;
    outputEnd_t      *outputEnd;
    copyImage_t      *copyImage;
    commentPut_t     *commentPut;
    commentGet_t     *commentGet;
    commentGetVec_t  *commentGetVec;
    commentDelete_t  *commentDelete;
    
} tSippStorageClass_t;

/*
 *-----------------------------------------------------------------------------
 * TSIPP global object.  Contains all TSIPP data global to the process.
 * Note that the SIPP library has its own process-global table.
 *-----------------------------------------------------------------------------
 */

/*
 * Type of function used to cleanup all active objects on modules that
 * are optional.
 */
typedef void
tSippCleanUp_t _ANSI_ARGS_((struct tSippGlob_t  *tSippGlobPtr));


/*
 * Type of function used to implement subcommands in several places.
 */
typedef int
tSippSubCmd_t _ANSI_ARGS_((struct tSippGlob_t *tSippGlobPtr,
                           int                 argc,
                           char              **argv));

/*
 * Global Tcl SIPP structure.  It is the client data to all commands.
 */
typedef struct tSippGlob_t {
    Tcl_Interp    *interp;           /* Tcl interpreter pointer.             */
    bool           showBackFaces;    /* Render or draw backfacing polygons.  */
    bool           rendering;        /* Rendering in progress.               */
    Color          backgroundColor;  /* Current background color.            */
    Color          lineColor;        /* Color for lines in RLE LINE render   */

    /*
     * Routine called during rendering at each update period.  Under Tk, this
     * handles event updates and signals.  When not using Tk, this just handles
     * signals.  Delayed error is used when an error occurs while handling a
     * signal during rendering.  Its an error message to return at the end.
     */
    Update_func  *updateProc;
    char         *delayedError;

    /*
     * Table of image storage classes that images can be rendered to.
     * Terminated by a NULL entry.
     */
    tSippStorageClass_t  *storageClassPtrs [TSIPP_MAX_STORAGE_CLASSES + 1];

    /*
     * Handle tables for the various tables we keep.
     */
    void_pt        surfaceTblPtr;    /* Surface table.            */
    void_pt        objectTblPtr;     /* Object table.             */
    Tcl_HashTable  objectXRefTbl;    /* Object cross ref table.   */
    void_pt        shaderTblPtr;     /* Shader table.             */
    void_pt        lightTblPtr;      /* Light table.              */
    void_pt        cameraTblPtr;     /* Camera table.             */
    void_pt        rleTblPtr;        /* RLE file table.           */
    void_pt        pbmTblPtr;        /* PBMPlus file table.       */
    void_pt        pixMapTblPtr;     /* Pixmap table.             */
    void_pt        tiffTblPtr;       /* TIFF file table.          */

    /*
     * Cleanup functions for optional modules.
     */
    tSippCleanUp_t  *rleCleanUpProc;  /* Cleanup RLE file table */

    /*
     * Main window ptr for tksipp.
     */
    void_pt        tkMainWindow;

    /*
     * SippColor optional subcommands.
     */
    tSippSubCmd_t *sippColorToXColor;
    tSippSubCmd_t *xColorToSippColor;

    /*
     * Floating point exception and matherr async handler and error infomation.
     * The error number and signal number of the error that occured.
     */
    Tcl_AsyncHandler  mathErrorHandler;
    int               mathErrorNum;
    int               mathSignal;
    
} tSippGlob_t;


/*
 * Used to set mode field when the mode is unknown 24 bit image.
 */
#define MODE_UNKNOWN24 255

/*
 * Information describing the image.
 */
typedef struct {
    /*
     * This describes the image being rendered and is saved with certain
     * types of image storage.
     */
    unsigned   xSize;                /* Size of image to output.      */
    unsigned   ySize;
    unsigned   mode;                 /* FLAT, GOURAUD, PHONG, LINE,   */
                                     /* MODE_UNKNOWN24.               */
    unsigned   overSampling;         /* Oversampling factor.          */
    int        field;                /* BOTH, ODD or EVEN scan lines. */
    u_char     backgroundColor [3];  /* Current background color.     */
    u_char     lineColor [3];        /* Color for lines in LINE mode. */

} tSippImageData_t;

/*
 * Type containing data about an image that is going to be rendered or copied
 * as well as other attributes that influence rendering. This information is
 * passed to the image store functions.
 */
typedef struct tSippOutputParms_t {
    tSippGlob_t      *tSippGlobPtr;   /* TSIPP globals.                      */
    tSippImageData_t  imgData;        /* Image data.                         */

    char            **argv;           /* Render command, if available.       */
    int               update;         /* Update period. -1 is not specified. */
    bool              clear;          /* Clear image before rendering.       */
    bool              bitMapOutput;   /* TRUE if a bitmap will be outputted. */
    scanDirection_t   scanDirection;  /* Scanning direction to use.          */
} tSippOutputParms_t;

/*
 * Color indicies for the scanlines that are passed to the output object's
 * renderLine functions.
 */
#define TSIPP_RED    0
#define TSIPP_GREEN  1
#define TSIPP_BLUE   2

/*
 * Type used in initializing Tcl commands.
 */
typedef struct {
    char        *name;
    Tcl_CmdProc *proc;
} tSippTclCmdTbl_t;

/*
 * Type used to store a translated handle list.  A static array is used
 * unless the list is bigger, then it goes dynamic.
 */
#define HANDLE_LIST_STATIC_SIZE 16

typedef struct {
    void   **ptr;
    int      len;
    void    *staticArray [HANDLE_LIST_STATIC_SIZE];
} handleList_t;

/*
 * Macros to do string compares.  They pre-check the first character before
 * checking of the strings are equal.
 */

#define STREQU(str1, str2) \
    (((str1) [0] == (str2) [0]) && (strcmp (str1, str2) == 0))
#define STRNEQU(str1, str2, cnt) \
    (((str1) [0] == (str2) [0]) && (strncmp (str1, str2, cnt) == 0))

/*
 * Macro to add bytes to a pointer, no matter what the type.  Always returns
 * a void *.
 */
#define PTR_ADD(ptr, incr) ((void *) (((char *) ptr) + incr))

/*
 * Macros to handle dealing with numbers that may have encountered a round
 * off error:
 *   FZERO - Is the number zero, with in the tolerance.
 *   FZERO_ROUND - Round the number to zero, if its in the tolerance of zero.
 *   FEQUAL - Is the number equal, within the tolerance.
 *
 */
#define TOLERANCE 1e-13
#define FZERO(f) ((f) < TOLERANCE && (f) > -TOLERANCE)
#define FZERO_ROUND(f) (FZERO(f) ? 0.0 : (f))
#define FEQUAL(f1, f2) ((((f2) - TOLERANCE) < (f1)) && \
                        ((f1) < (((f2) + TOLERANCE))))

/*
 * Functions supplied by Tcl, but not exported.
 */
extern void
panic ();

/*
 * tSippUtil.c externals.
 */
extern bool
TSippConvertFraction _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                  char         *strBuf,
                                  double       *valuePtr));

extern bool
TSippConvertUnsignedDbl _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                     char         *strBuf,
                                     double       *valuePtr));

extern bool
TSippConvertPosUnsigned _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                     char         *strBuf,
                                     unsigned     *valuePtr));

extern bool
TSippConvertColor _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                               char         *listStr,
                               Color        *colorPtr));

extern void
TSippFormatColor _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                              Color         color,
                              char         *listStr));

extern void
TSippFormatIntColor _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                 u_char       *color,
                                 char         *listStr));

extern bool
TSippConvertOpacity _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                 char         *listStr,
                                 Color        *opacityPtr));

extern bool
TSippConvertAngleRad _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                  char         *strBuf,
                                  double       *valuePtr));

extern bool
TSippConvertAngleDeg _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                  char         *strBuf,
                                  double       *valuePtr));

extern bool
TSippConvert2DPoint _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                 char         *listStr,
                                 double       *xCoordPtr,
                                 double       *yCoordPtr));

extern bool
TSippConvertVertex _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                char         *listStr,
                                Vector       *vertexPtr));

extern bool
TSippConvertMatrix _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                char         *listStr,
                                Transf_mat   *matrixPtr));

extern char *
TSippFormatMatrix _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                               Transf_mat   *matrixPtr));

extern bool
TSippHandleListConvert _ANSI_ARGS_((tSippGlob_t     *tSippGlobPtr,
                                    void            *handleTblPtr,
                                    char            *listPtr,
                                    handleList_t    *handleListPtr,
                                    handleList_t    *handleEntryListPtr,
                                    char          ***handleListArgvPtr));

extern void
TSippHandleListFree _ANSI_ARGS_((handleList_t  *handleListPtr));

extern void
TSippInitCmds _ANSI_ARGS_((tSippGlob_t       *tSippGlobPtr,
                           tSippTclCmdTbl_t  *cmdTablePtr));

extern bool
TSippParseTextureMapping _ANSI_ARGS_((tSippGlob_t   *tSippGlobPtr,
                                      char          *textureStr,
                                      int           *texturePtr,
                                      int           *invalidList));

extern int
TSippNotWhileRendering _ANSI_ARGS_((Tcl_Interp *interp));

extern int
TSippAtEOF _ANSI_ARGS_((Tcl_Interp  *interp,
                        char        *handle));

extern int
TSippNotReadable _ANSI_ARGS_((Tcl_Interp *interp,
                              char       *handle));

extern int
TSippNotWritable _ANSI_ARGS_((Tcl_Interp *interp,
                              char       *handle));

extern void
TSippAddHist _ANSI_ARGS_((char   ***commentsPtr,
                          char    **argv));

extern int
TSippCallCmd ();

/*
 * tSippComment.c externals.
 */
extern void
TSippCommentPut _ANSI_ARGS_((char  ***commentsPtr,
                             char    *name,
                             char    *value));

extern char *
TSippCommentGet _ANSI_ARGS_((char  **comments,
                             char    *name));

extern void
TSippCommentDelete _ANSI_ARGS_((char  ***commentsPtr,
                                char    *name));

extern char **
TSippCommentCopy _ANSI_ARGS_((char  **comments));

extern void
TSippCommentMerge _ANSI_ARGS_((char   **oldComments,
                               char  ***newCommentsPtr));

extern void
TSippCommentFree _ANSI_ARGS_((char  ***commentsPtr));

void
TSippCommentInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
 * tSippIStore.c externals.
 */
extern void
TSippAddStorageClass _ANSI_ARGS_((tSippGlob_t          *tSippGlobPtr,
                                  tSippStorageClass_t  *storageClassPtr));

extern tSippStorageClass_t *
TSippFindStorageClass _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                                   char        *handle));

/*
  *tSippError.c externals.
 */
int
TSippErrorHandlingInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippBezier.c externals.
 */
extern void
TSippBezierInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippPoly.c externals.
 */
extern Surface *
TSippSurfaceHandleToPtr _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                     char         *handle));

extern bool
TSippSurfaceListConvert _ANSI_ARGS_((tSippGlob_t     *tSippGlobPtr,
                                     char            *listPtr,
                                     handleList_t    *handleListPtr,
                                     char          ***handleListArgvPtr));

extern void
TSippPolyInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

extern void
TSippPolyCleanUp _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippObj.c externals.
 */
extern void
TSippBindObjectToHandle _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                    Object        *objectPtr));

extern Object *
TSippObjectHandleToPtr _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                    char         *handle));

extern bool
TSippObjectToFirstHandle _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr,
                                      Object      *objectPtr));

extern void
TSippObjectInit _ANSI_ARGS_((tSippGlob_t  *tclSippGlobP));

extern void
TSippObjectCleanUp _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippLight.c externals.
 */
extern void
TSippLightInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

extern void
TSippLightCleanUp _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippCamera.c externals.
 */
extern void
TSippCameraInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

extern void
TSippCameraCleanUp _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippShader.c externals.
 */
extern Shader *
TSippShaderHandleToPtr _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr,
                                    char         *handle,
                                    void        **surfDescHdrPtrPtr));

extern void
TSippShaderInit _ANSI_ARGS_((tSippGlob_t *tclSippGlobP));

extern void
TSippShaderCleanUp _ANSI_ARGS_((tSippGlob_t *tSippGlobPtr));

/*
  *tSippPrim.c externals.
 */
extern void
TSippPrimInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippPBM.c externals.
 */
extern void
TSippPBMInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

extern void
TSippPBMCleanUp _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippRLE.c externals.
 */
extern void
TSippRLEInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippRender.c externals.
 */
extern void
TSippHandleRenderingError _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

extern void
TSippRenderInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippTk.c externals.
 */
extern void
TkSippUpdate _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippPixMap.c externals.
 */
extern void
TSippPixMapInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

extern void
TSippPixMapCleanUp _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippGeom.c externals.
 */
extern void
TSippGeomInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippShadows.c externals.
 */
extern void
TSippShadowsInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

/*
  *tSippPhoto.c externals.
 */
extern void
TSippPhotoInit _ANSI_ARGS_((tSippGlob_t  *tSippGlobPtr));

#endif
