;;; -*-Scheme-*-
;;;
;;; $Id: macros.sc,v 1.7 1993/03/31 00:28:11 cph Exp $
;;;
;;; Copyright (c) 1993 Massachusetts Institute of Technology
;;;
;;; This material was developed by the Scheme project at the
;;; Massachusetts Institute of Technology, Department of Electrical
;;; Engineering and Computer Science.  Permission to copy this
;;; software, to redistribute it, and to use it for any purpose is
;;; granted, subject to the following restrictions and understandings.
;;;
;;; 1. Any copy made of this software must include this copyright
;;; notice in full.
;;;
;;; 2. Users of this software agree to make their best efforts (a) to
;;; return to the MIT Scheme project any improvements or extensions
;;; that they make, so that these may be included in future releases;
;;; and (b) to inform MIT of noteworthy uses of this software.
;;;
;;; 3. All materials developed as a consequence of the use of this
;;; software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4. MIT has made no warrantee or representation that the operation
;;; of this software will be error-free, and MIT is under no
;;; obligation to provide any services, by way of maintenance, update,
;;; or otherwise.
;;;
;;; 5. In conjunction with products arising from the use of this
;;; material, there shall be no use of the name of the Massachusetts
;;; Institute of Technology nor of any adaptation thereof in any
;;; advertising, promotional, or sales literature without prior
;;; written consent from MIT in each case.

;;;; Macros for Scheme Object System

(module sosmacro
  (top-level transform:define-class
	     transform:define-generic-procedure
	     transform:define-method))
(include "mitutil.sch")

(eval-when (eval load)
  (include "macros.sch"))

(define (transform:define-class name superclasses . slot-arguments)
  (if (not (or (symbol? name)
	       (and (pair? name)
		    (symbol? (car name)))))
      (serror 'DEFINE-CLASS "Malformed class name:" name))
  (if (not (list? superclasses))
      (serror 'DEFINE-CLASS "Malformed class superclasses:" superclasses))
  (let ((name+options (if (pair? name) name (list name))))
    (let ((options (cdr name+options))
	  (args
	   `(',(car name+options)
	     (LIST ,@superclasses)
	     (LIST
	      ,@(map
		 (lambda (arg)
		   (cond ((symbol? arg)
			  `',arg)
			 ((and (pair? arg)
			       (symbol? (car arg))
			       (list? (cdr arg)))
			  `(LIST ',(car arg)
				 ,@(let loop ((plist (cdr arg)))
				     (cond ((null? plist)
					    '())
					   ((and (symbol? (car plist))
						 (pair? (cdr plist)))
					    (cons* `',(car plist)
						   (cadr plist)
						   (loop (cddr plist))))
					   (else
					    (serror 'DEFINE-CLASS
						    "Malformed slot argument:"
						    arg))))))
			 (else
			  (serror 'DEFINE-CLASS
				  "Malformed slot argument:" arg))))
		 slot-arguments)))))
      (if (not (and (list? options)
		    (for-all? options
		      (lambda (option)
			(or (symbol? option)
			    (and (pair? option)
				 (symbol? (car option))
				 (list? (cdr option))))))))
	  (serror 'DEFINE-CLASS "Malformed class options:" options))
      (let ((metaclass '<CLASS>))
	(for-each (lambda (option)
		    (let ((name (if (pair? option) (car option) option)))
		      (case name
			((METACLASS)
			 (if (not (and (pair? option)
				       (pair? (cdr option))
				       (null? (cddr option))))
			     (serror 'DEFINE-CLASS
				     "Malformed class option:" option))
			 (set! metaclass (cadr option)))
			(else
			 (serror 'DEFINE-CLASS
				 "Unknown class option:" name)))))
		  options)
	`(DEFINE ,(car name+options)
	   ,(if (eq? metaclass '<CLASS>)
		`(MAKE-CLASS ,@args)
		`(MAKE-INSTANCE ,metaclass ,@args)))))))

(define (transform:define-generic-procedure name lambda-list . body)
  (if (not (symbol? name))
      (serror 'DEFINE-GENERIC-PROCEDURE
	      "Malformed generic procedure name:" name))
  (let ((initial-method? (not (null? body))))
    (call-with-values
	(lambda ()
	  (parse-lambda-list lambda-list
			     initial-method?
			     'DEFINE-GENERIC-PROCEDURE))
      (lambda (required optional rest)
	(let ((min-arity (length required)))
	  `(BEGIN
	     (DEFINE ,name
	       (MAKE-GENERIC-PROCEDURE ',name
				       ,min-arity
				       ,(and (not rest)
					     (+ min-arity (length optional)))))
	     ,@(if initial-method?
		   `((DEFINE-METHOD ,name ,lambda-list ,@body))
		   '())))))))

(define (transform:define-method name lambda-list . body)
  (if (not (symbol? name))
      (serror 'DEFINE-METHOD "Malformed method name:" name))
  (call-with-values
      (lambda () (parse-lambda-list lambda-list #t 'DEFINE-METHOD))
    (lambda (required optional rest)
      (call-with-values (lambda () (extract-required-specializers required))
	(lambda (required specializers)
	  `(ADD-METHOD ,name
	     (MAKE-METHOD (LIST ,@specializers)
	       ,(make-named-lambda name
				   (cons 'CALL-NEXT-METHOD required)
				   optional
				   rest
				   (cons 'CALL-NEXT-METHOD body)))))))))

(define (transform:method lambda-list . body)
  (call-with-values (lambda () (parse-lambda-list lambda-list #t 'METHOD))
    (lambda (required optional rest)
      (call-with-values (lambda () (extract-required-specializers required))
	(lambda (required specializers)
	  `(MAKE-METHOD (LIST ,@specializers)
	     ,(make-named-lambda #f
				 (cons 'CALL-NEXT-METHOD required)
				 optional
				 rest
				 (cons 'CALL-NEXT-METHOD body))))))))

(define (extract-required-specializers required)
  (let loop ((required required) (names '()) (specializers '()))
    (cond ((null? required)
	   (values (reverse! names)
		   (reverse! (let loop ((specializers specializers))
			       (if (and (not (null? specializers))
					(eq? '<OBJECT> (car specializers))
					(not (null? (cdr specializers))))
				   (loop (cdr specializers))
				   specializers)))))
	  ((pair? (car required))
	   (loop (cdr required)
		 (cons (caar required) names)
		 (cons (cadar required) specializers)))
	  (else
	   (loop (cdr required)
		 (cons (car required) names)
		 (cons '<OBJECT> specializers))))))

(define (parse-lambda-list lambda-list allow-specializers? specform)
  (let loop ((lambda-list lambda-list) (required '()))
    (cond ((null? lambda-list)
	   (values (reverse required) '() #f))
	  ((pair? lambda-list)
	   (cond ((or (symbol? (car lambda-list))
		      (and allow-specializers?
			   (pair? (car lambda-list))
			   (symbol? (caar lambda-list))
			   (pair? (cdar lambda-list))
			   (null? (cddar lambda-list))))
		  (loop (cdr lambda-list) (cons (car lambda-list) required)))
		 (else
		  (serror specform
			  "Illegal parameter list element:"
			  (car lambda-list)))))
	  ((symbol? lambda-list)
	   (values (reverse required) '() lambda-list))
	  (else
	   (serror specform "Illegal parameter list tail:" lambda-list)))))

(define (make-named-lambda name required optional rest body)
  `(LAMBDA (,@required . ,(or rest '())) ,@body))