#!perl

#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
#   file: t/manifest-read.t
#
#   Copyright © 2015 Van de Bugger
#
#   This file is part of perl-Dist-Zilla-Plugin-Manifest-Read.
#
#   perl-Dist-Zilla-Plugin-Manifest-Read is free software: you can redistribute it and/or modify it
#   under the terms of the GNU General Public License as published by the Free Software Foundation,
#   either version 3 of the License, or (at your option) any later version.
#
#   perl-Dist-Zilla-Plugin-Manifest-Read is distributed in the hope that it will be useful, but
#   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
#   PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License along with
#   perl-Dist-Zilla-Plugin-Manifest-Read. If not, see <http://www.gnu.org/licenses/>.
#
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

use strict;
use warnings;
use lib 't';

use POSIX qw{ locale_h };
use Test::Deep qw{ re cmp_deeply };
use Test::More;
use Test::Routine::Util;

# `AutoPrereqs` hints:
use Dist::Zilla::Plugin::Templates qw{};
use Dist::Zilla::Tester::DieHard v0.4.0 qw{};
    # ^ We do not use `Dist::Zilla::Tester::DieHard` directly. This requirement added because
    #   `Dist::Zilla::Tester::DieHard` v0.4.0 have enabled diagnostic warnings, which may (I hope)
    #   help to debug a strange test failure on MSWin32 machines.

#   Some tests check error messages, which expected to be in English.
setlocale( LC_ALL, 'C' )
    or diag "*** Can't set \"C\" locale, some tests may fail! ***";

my $role     = 'ManifestReadTester';
my $aborting = "Aborting...\n";

my %include = ( # Files to include into distribution.
    # filename             # content
    q{README}           => 'Dummy v0.003',          # Name without slashes.
    q{lib/Dummy.pm}     => 'package Dummy; 1;',     # Name with slash.
    q{filename-0}       => '0',                     # Ordinary filenames.
    q{filename-1}       => '1',
    q{filename-2}       => '2',
    q{filename-3}       => '3',
    q{filename-4}       => '4',
    q{filename-5}       => '5',
    q{filename-6}       => '6',
    q{filename-7}       => '7',
    q{filename-8}       => '8',
    q{filename-9}       => '9',
    q{filename-a}       => 'a',
    q{filename-b}       => 'b',
    q{filename-c}       => 'c',
    q{filename-d}       => 'd',
    q{filename-e}       => 'e',
    q{filename-f}       => 'f',
    q{file name 0}      => ' 0',                    # Filenames with spaces.
    q{file name 1}      => ' 1',
    q{file name 2}      => ' 2',
    q{file name 3}      => ' 3',
    q{file'name'0}      => '\'0',                   # Filenames with apostrophes.
    q{file'name'1}      => '\'1',
    q{file'name'2}      => '\'2',
    q{file'name'3}      => '\'3',
    q{file\name\0}      => '\\0',                   # Filenames with backslashes.
    q{file\name\1},     => '\\1',
    q{file\name\2},     => '\\2',
    q{file\name\3},     => '\\3',
);
my %exclude = ( # Source files to exclude from distribution.
    # filename             # content
    q{filename_0}       => '_0',
    q{filename 1}       => '_1',
    q{filename'2}       => '_2',
    q{filename\3}       => '_3',
);

# --------------------------------------------------------------------------------------------------

plan tests => 5;

run_tests 'Successful build', $role => {
    manifest => 'manifest.txt',
    files => {
        'manifest.txt' => [

            "# Comment line",
            "       # Comment line may have leading spaces",
            "\t# Comment line may have leading tabs",
            "# Comment line may have trailing spaces   ",
            "# Comment line may have trailing tabs\t\t\t",
            "   # Comment line may have leading and trailing spaces   ",

            "",                                 # Empty line.
            "    ",                             # Space-only line.
            " \t ",                             # Whitespace-only line.

            "README",                           # Filename without slashes.
            "lib/Dummy.pm",                     # Filename with slash.

            "filename-0",                       # Just unquoted filename.
            " \t filename-1",                   # … + leading whitespace.
            "filename-2 \t ",                   # … + trailing wwhitespace.
            " \t filename-3 \t ",               # … + leading and trailing whitespace.

            "filename-4    #",                  # … + hash marker.
            " filename-5   # \t ",
            "filename-6    #\tcomment",
            " filename-7   # comment\t",

            "\tfilename-8  +",                  # … + plus delimiter.
            "filename-9    +  ",
            "\tfilename-a  + comment",
            "filename-b    + comment  ",

            "\t'filename-c'\t",                 # Non-funny names can be quoted.
            "'filename-d'    #  ",
            "\t'filename-e'  +",
            "'filename-f'    + yep ",

            "'file name 0'",                    # Filenames with spaces require quoting.
            " 'file name 1'  #",
            "'file name 2'   + ",
            " 'file name 3'  + ",

            "file'name'0",                      # Filenames with apostrophes may be unquoted
            "'file\\'name\\'1'   #",            # or quoted (with escaping the quote marks).
            "\tfile'name'2       + ",
            "\t'file\\'name\\'3' + cmnt",

            "file\\name\\0",                    # Filenames with backslashes may be unquoted
            "\t'file\\\\name\\\\1' #",          # or quoted (with
            "'file\\name\\2'       + ",         # or without escaping).
            "\tfile\\name\\3       + ",

            "filename_0     -",                 # These files should not appear in distro.
            " 'filename 1'  -\t",
            " filename'2    - comment",
            "'filename\\3'  -\tcomment  ",

            "lib/           / This is a directory.",

        ],
        %include,
        %exclude,
    },
    expected => {
        files => {
            %include,
            map( { $_ => undef } keys( %exclude ) ),
                # ^ `undef` means these files should not be in distro.
        },
    },
};

run_tests 'Syntax error', $role => {
    manifest => 'Manifest.lst',
    files => {
        'Manifest.lst' => [
            'lib/Assa.pm',
            q{'File'name +},        # Invalid filename.
            q{'File'name' -},       # Invalid filename.
            q{Filename ?},          # `?` is a bad marker
            q{lib/Assa.pm},         # File already listed.
            q{'lib/Assa.pm'},       # File already listed — quoting does not matter.
        ],
    },
    expected => {
        exception => $aborting,
        messages => [
            'Syntax error at Manifest.lst line 2.',
            'Syntax error at Manifest.lst line 3.',
            'Syntax error at Manifest.lst line 4.',
            'lib/Assa.pm at Manifest.lst line 5',
            '    also listed at Manifest.lst line 1.',
            'lib/Assa.pm at Manifest.lst line 6',
            '    also listed at Manifest.lst line 1.',
            'Manifest.lst:',
            '    1: lib/Assa.pm',
            '       ^^^ The file also listed at line 5 ^^^',
            '       ^^^ The file also listed at line 6 ^^^',
            '    2: \'File\'name +',
            '       ^^^ Syntax error ^^^',
            '    3: \'File\'name\' -',
            '       ^^^ Syntax error ^^^',
            '    4: Filename ?',
            '       ^^^ Syntax error ^^^',
            '    5: lib/Assa.pm',
            '       ^^^ The file also listed at line 1 ^^^',
            '    6: \'lib/Assa.pm\'',
            '       ^^^ The file also listed at line 1 ^^^',
        ],
    },
};

run_tests 'Bad files', $role => {
    files => {
        'MANIFEST' => [
            'file1',        # Nonexistent files with various markers.
            'file2 #',
            'file3 +',
            'file4 -',
            '..',           # A directory must be used with marker `/`.
            'README /',     # This marker should be used only with directories.
        ],
        'README' => 'Read me...',
    },
    expected => {
        exception => $aborting,
        messages => [
            'file1 does not exist at MANIFEST line 1.',
            'file2 does not exist at MANIFEST line 2.',
            'file3 does not exist at MANIFEST line 3.',
            'file4 does not exist at MANIFEST line 4.',
            '.. is not a plain file at MANIFEST line 5.',
            'README is not a directory at MANIFEST line 6.',
        ],
    },
};

#   `Manifest::Read` is a `FileFinder`.
run_tests 'File finder', $role => {
    files => {
        'MANIFEST' => [
            'MANIFEST       -',
            'lib/Dummy.pm   #',
            'README         +',
        ],
        'lib/Dummy.pm'      => 'package Dummy {{$dist->version}}; 1;',
        'README'            => [
            '{{$dist->name}} {{$dist->version}}',
            '{{$dist->abstract}}',
        ],
    },
    plugins => [
        'Manifest::Read',
        [ 'Templates' => {
            #   `Templates` plugin has `templates` option, which accepts `FileFinder`.
            #   By specifying `Manifest::Read` we declare all the manifested files are templates.
            'templates' => 'Manifest::Read',
        } ],
    ],
    expected => {
        files => {
            #   In both files Perl fragments should be evaluated.
            'lib/Dummy.pm'  => 'package Dummy 0.003; 1;',
            'README'        => [
                'Dummy 0.003',
                'Dummy abstract',
            ],
            'MANIFEST'      => undef,
        },
    },
};

run_tests 'No manifest', $role => {
    expected => {
        exception => $aborting,
        messages => [
            re( qr{^.*?/MANIFEST: No such file or directory} ),
        ],
    },
};

done_testing;

exit( 0 );

# end of file #
