package Data::Timeline::SVK;

use strict;
use warnings;
use DateTime::Format::DateParse;


our $VERSION = '0.01';


use base qw(Data::Timeline::Builder);


__PACKAGE__->mk_scalar_accessors(qw(base_dir));


sub create {
    my $self = shift;

    my $timeline = $self->make_timeline;

    chdir $self->base_dir or
        die sprintf "can't chdir to %s: $!\n", $self->base_dir;

    open my $fh, '-|', "svk log" or die "can't read from pipe 'svk log': $!\n";
    while (<$fh>) {
        next unless /^(r\d+:\s*\w+)\s*\|\s*(20.*)/;
        $timeline->entries_push($self->make_entry(
            description => $1,
            timestamp   => DateTime::Format::DateParse->parse_datetime($2),
            type        => 'svk',
        ));
    }
    close $fh or die "can't close pipe to 'svk log': $!\n";
    $timeline;
}


1;


__END__



=head1 NAME

Data::Timeline::SVK - Builds a timeline from an 'svk log'


=head1 SYNOPSIS

    my $svk_timeline = Data::Timeline::SVK->new(
        base_dir => $svk_base_dir
    )->create;

=head1 DESCRIPTION

This class parses the output of C<svk log> in a given directory and creates a
timeline with entries corresponding to commits. Only the revision number and
the committer name are recorded in the entry descriptions.

The timeline entries generated by this builder have the type C<svk>.

This class manually parses the output of C<svk log>; it would probably be
better to use classes provided by the L<SVK> distribution, but because of its
dependency on L<SVN::Core>, the L<SVK> distribution is a bit difficult to
install.

Data::Timeline::SVK inherits from L<Data::Timeline::Builder>.

=head1 METHODS

=over 4

=item base_dir

    my $value = $obj->base_dir;
    $obj->base_dir($value);

A basic getter/setter method. If called without an argument, it returns the
value. If called with a single argument, it sets the value.

=item base_dir_clear

    $obj->base_dir_clear;

Clears the value.

=item clear_base_dir

    $obj->clear_base_dir;

Clears the value.

=item base_dir

The directory from which the C<svk log> command is issued.

=item create

Start parsing the output of C<svk log> and return a L<Data::Timeline> object
with the commit information.

=back

=head1 TAGS

If you talk about this module in blogs, on del.icio.us or anywhere else,
please use the C<datatimelinesvk> tag.

=head1 BUGS AND LIMITATIONS

No bugs have been reported.

Please report any bugs or feature requests to
C<<bug-data-timeline-svk@rt.cpan.org>>, or through the web interface at
L<http://rt.cpan.org>.

=head1 INSTALLATION

See perlmodinstall for information and options on installing Perl modules.

=head1 AVAILABILITY

The latest version of this module is available from the Comprehensive Perl
Archive Network (CPAN). Visit <http://www.perl.com/CPAN/> to find a CPAN
site near you. Or see <http://www.perl.com/CPAN/authors/id/M/MA/MARCEL/>.

=head1 AUTHOR

Marcel GrE<uuml>nauer, C<< <marcel@cpan.org> >>

=head1 COPYRIGHT AND LICENSE

Copyright 2007 by Marcel GrE<uuml>nauer

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.


=cut

