#!/usr/bin/env perl
use strict;
use warnings;
use Test::More;
use Test::Exception;
use lib 'lib';
use Catan::Game;
use Data::Dumper;
use List::Util 'sum';
use Catan::Map;

my $tiles = Catan::Map->_starter;
open my $log_fh, '>>', \my $log;
ok my $game = Catan::Game->new({log => $log_fh}), 'constructor';
is $game->{max_players}, 4, '4 max players by default';

# setup
ok $game->action('PA'), 'add player 1';
ok $game->action('PA'), 'add player 2';
ok $game->action('PA'), 'add player 3';
ok $game->action('PA'), 'add player 4';
ok exists $game->action('PA')->[0]{CH}, 'exceeds max players chat msg';
ok $game->random_map, 'define random map';
ok $game->starter_map, 'define starter map';
ok $game->action('MD', $tiles), 'define custom map';
ok $game->action('PE'), 'end setup phase';
ok exists $game->action('XxX')->[0]{CH}, 'incorrect action code msg';

# deployment
is $game->robber->location, $game->map->tiles->{"-2,0"}, 'robber has been deployed to desert tile';
ok $game->action('BS', { player => 1, intersection => [[0,-2],[-1,-1],[-1,-2]]}), 'deploy settlement';
ok $game->action('BR', { player => 1, path => [[[0,-2],[-1,-1],[-1,-2]],[[0,-2],[0,-1],[-1,-1]]]}), 'deploy road';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('BS', { player => 2, intersection => [[0,2],[-1,3],[-1,2]]}), 'deploy settlement';
ok $game->action('BR', { player => 2, path => [[[0,2],[-1,3],[-1,2]],[[0,1],[0,2],[-1,2]]]}), 'deploy road';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('BS', { player => 3, intersection => [[-2,1],[-2,2],[-3,2]]}), 'deploy settlement';
ok $game->action('BR', { player => 3, path => [[[-2,1],[-2,2],[-3,2]],[[-2,2],[-3,3],[-3,2]]]}), 'deploy road';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('BS', { player => 4, intersection => [[2,-2],[2,-1],[1,-1]]}), 'deploy settlement';
ok $game->action('BR', { player => 4, path => [[[2,-2],[2,-1],[1,-1]],[[2,-1],[1,-1],[1,0]]]}), 'deploy road';
ok $game->action('TE', { player => 4}), 'end turn';
ok $game->action('BS', { player => 4, intersection => [[2,0],[2,-1],[1,0]]}), 'deploy settlement';
ok $game->action('BR', { player => 4, path => [[[2,0],[2,-1],[1,0]],[[2,-1],[1,-1],[1,0]]]}), 'deploy road';
ok $game->action('TE', { player => 4}), 'end turn';
ok $game->action('BS', { player => 3, intersection => [[-1,0],[-1,1],[-2,1]]}), 'deploy settlement';
ok exists $game->action('BS', { player => 3, intersection => [[-1,0],[-1,1],[-2,1]]})->[0]{CH}, 'deploy settlement in same spot msg';
ok $game->action('BR', { player => 3, path => [[[-1,0],[-1,1],[-2,1]],[[0,0],[-1,1],[-1,0]]]}), 'deploy road';
ok $game->action('BS', { player => 2, intersection => [[1,1],[0,2],[0,1]]})->[0]{CH}, 'deploy settlement out of turn msg';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('BS', { player => 2, intersection => [[1,1],[0,2],[0,1]]}), 'deploy settlement';
ok $game->action('BR', { player => 2, path => [[[1,1],[0,2],[0,1]],[[0,1],[0,2],[-1,2]]]}), 'deploy road';
ok $game->action('TE', { player => 2}), 'end turn';
ok exists $game->action('TE', { player => 1})->[0]{CH}, 'end turn unfinished business msg';
ok $game->action('BS', { player => 1, intersection => [[1,-2],[0,-1],[0,-2]]}), 'deploy settlement';
ok $game->action('BR', { player => 1, path => [[[1,-2],[0,-1],[0,-2]],[[0,-2],[0,-1],[-1,-1]]]}), 'deploy road';
ok $game->action('TE', { player => 1}), 'end turn';
cmp_ok grep(2 == $_->victory_points_count, @{$game->players}),'==', 4, 'all players have 2 victory points at deployment end';

print $log;
# play
# round 1
ok $game->action('DR', { player => 1, result => 4}), 'roll dice';
my $p1_total_resources = sum values %{$game->player->resources};
my $bank_total_resources = sum values %{$game->bank->resources};
ok $game->action('BR', { player => 1, path => [[[0,-2],[0,-1],[-1,-1]],[[0,-1],[-1,0],[-1,-1]]]}), 'build road';
ok sum(values %{$game->player->resources}) - $p1_total_resources == -2, 'building a road removed 2 resources';
ok sum(values %{$game->bank->resources}) - $bank_total_resources == 2, 'the bank gained 2 resources';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 6}), 'roll dice';
ok $game->action('TB', { player => 2, resources => {2 => {L=>-2, B=>1}} }), 'trade 2 lumber for a brick';
ok $game->action('BR', { player => 2, path => [[[1,0],[1,1],[0,1]],[[1,1],[0,2],[0,1]]]}), 'build road';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 8}), 'roll dice';
ok $game->action('BR', { player => 3, path => [[[0,-1],[0,0],[-1,0]],[[0,0],[-1,1],[-1,0]]]}), 'build road';
ok $game->action('TO', { player => 3, resources=> {3=>{B=>-1,L=>1},4=>{B=>1,L=>-1}}, uuid => 1}), 'trade offer';
ok $game->action('TA', { player => 4, uuid => 1}), 'trade accept';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 4}), 'roll dice';
ok $game->action('BR', { player => 4, path => [[[2,-1],[2,0],[1,0]],[[2,0],[1,1],[1,0]]]}), 'build road';
ok $game->action('TE', { player => 4}), 'end turn';

# round 2
ok $game->action('DR', { player => 1, result => 11}), 'roll dice';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 7}), 'roll dice';
ok $game->action('RM', { player => 2, tile => [1,0]}), 'move robber';
ok $game->action('RR', { player => 2, target_player => 4, code => 'G'}), 'robber rob';
ok $game->action('BD', { player => 2, type => 'YP'}), 'build development card';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 5}), 'roll dice';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 9}), 'roll dice';
ok $game->action('BR', { player => 4, path => [[[1,0],[1,1],[0,1]],[[2,0],[1,1],[1,0]]]}), 'build road';
ok $game->action('TE', { player => 4}), 'end turn';

# round 3
ok $game->action('DR', { player => 1, result => 6}), 'roll dice';
ok $game->action('TO', { player => 1, resources=> {1=>{L=>-1,G=>1},4=>{L=>1,G=>-1}}, uuid => 2}), 'trade offer';
ok $game->action('TA', { player => 4, uuid => 2}), 'trade accept';
ok $game->action('BS', { player => 1, intersection => [[0,-1],[-1,0],[-1,-1]]}), 'build settlement';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('PD', { player => 2, type => 'YP' }), 'play year of plenty';
ok $game->action('YP', { player => 2, resources => {2 =>{G=>1, B=>1}} }), 'collect year of plenty resources';
ok $game->action('DR', { player => 2, result => 11}), 'roll dice';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 12}), 'roll dice';
ok $game->action('BR', { player => 3, path => [[[0,0],[-1,1],[-1,0]],[[0,0],[0,1],[-1,1]]]}), 'build road';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 5}), 'roll dice';
ok $game->action('BR', { player => 4, path => [[[1,0],[1,1],[0,1]],[[1,0],[0,1],[0,0]]]}), 'build road';
ok $game->action('TE', { player => 4}), 'end turn';

# round 4
ok $game->action('DR', { player => 1, result => 4}), 'roll dice';
ok $game->action('TO', { player => 1, resources=> {1=>{L=>1,B=>-1},2=>{L=>-1,B=>1}}, uuid => 3}), 'trade offer';
ok $game->action('TC', { player => 1, uuid => 3}), 'trade cancel';
ok $game->action('TO', { player => 1, resources=> {1=>{L=>1,B=>-1,O=>-1},2=>{O=>1,L=>-1,B=>1}}, uuid => 4}), 'trade offer';
ok $game->action('TA', { player => 2, uuid => 4}), 'trade accept';
ok $game->action('BR', { player => 1, path => [[[0,-1],[0,0],[-1,0]],[[0,-1],[-1,0],[-1,-1]]]}), 'build road';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 7}), 'roll dice';
ok $game->action('RM', { player => 2, tile => [1,-1]}), 'move robber';
ok exists $game->action('RR', { player => 2, target_player => 4, code => 'B'})->[0]->{CH}, 'robber steal resource they dont have';
ok $game->action('RR', { player => 2, target_player => 4, code => 'W'}), 'robber steal';
ok $game->action('BR', { player => 2, path => [[[0,1],[0,2],[-1,2]],[[0,1],[-1,2],[-1,1]]]}), 'build road';
ok $game->action('BD', { player => 2, type => 'MO'}), 'build development card';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 10}), 'roll dice';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 5}), 'roll dice';
ok $game->action('TE', { player => 4}), 'end turn';

# round 5
ok $game->action('DR', { player => 1, result => 6}), 'roll dice';
ok $game->action('TO', { player => 1, resources=> {1=>{O=>-1,G=>1},4=>{G=>-1,O=>1}}, uuid => 5}), 'trade offer';
ok $game->action('TA', { player => 4, uuid => 5}), 'trade accept';
ok exists $game->action('BC', { player => 1, intersection => [[0,-1],[-1,0],[-1,-1]]})->[0]{CH}, 'build city too few resources';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 2}), 'roll dice';
ok $game->action('BS', { player => 2, intersection => [[0,1],[-1,2],[-1,1]]}), 'build settlement';
ok $game->action('PD', { player => 2, type => 'MO'}), 'play monopoly';
ok $game->action('MO', { player => 2, resource_code => 'O'}), 'declare monopoly on Ore';
ok $game->action('TB', { player => 2, resources => {2 =>{O=>-4, B=>1}}}), 'trade 4 ore to the bank';
ok $game->action('BR', { player => 2, path => [[[0,1],[-1,2],[-1,1]],[[-1,1],[-1,2],[-2,2]]]}), 'build road';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 8}), 'roll dice';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 5}), 'roll dice';
ok $game->action('TO', { player => 4, resources=> {4=>{O=>1,W=>-1},3=>{W=>1,O=>-1}}, uuid => 6}), 'trade offer';
ok $game->action('TA', { player => 3, uuid => 6}), 'trade accept';
ok $game->action('BD', { player => 4, type => 'RB'}), 'build development card';
ok $game->action('TE', { player => 4}), 'end turn';

# round 6
ok $game->action('DR', { player => 1, result => 6}), 'roll dice';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 2}), 'roll dice';
ok $game->action('BC', { player => 2, intersection => [[0,1],[-1,2],[-1,1]]}), 'build city';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 3}), 'roll dice';
ok $game->action('BR', { player => 3, path => [[[-2,2],[-3,3],[-3,2]],[[-2,2],[-2,3],[-3,3]]]}), 'build road';
ok $game->action('TE', { player => 3}), 'end turn';

my $p4_total_resources = sum values %{$game->player->resources};
ok $game->action('PD', { player => 4, type => 'RB' }), 'play road building';
ok $game->action('BR', { player => 4, path => [[[1,-1],[1,0],[0,0]],[[1,0],[0,1],[0,0]]]}), 'build road';
ok $game->action('BR', { player => 4, path => [[[2,-1],[1,0],[1,-1]],[[1,-1],[1,0],[0,0]]]}), 'build road';
ok sum(values %{$game->player->resources}) - $p4_total_resources == 0, 'building a road with roadbuilding did not use resources';

ok $game->action('TE', { player => 1}), 'end turn';
ok exists $game->action('TE', { player => 4})->[0]{CH}, 'end turn before rolling dice msg';
ok $game->action('DR', { player => 4, result => 7}), 'roll dice';
#ok $game->action('CR', { player => 1, resources => {1 => {O=>-3, L=>-1}}}), 'concede resources';
#ok exists $game->action('CR', { player => 1, resources => {1 => {O=>-3, L=>-1}}})->[0]{CH}, 'concede resources twice';
#ok $game->action('CR', { player => 2, resources => {2 => {W=>-2}}}), 'concede some resources';
#ok $game->action('CR', { player => 3, resources => {3 => {W=>-3,O=>-1}}}), 'concede some resources';
#ok $game->action('CR', { player => 2, resources => {2 => {L=>-3}}}), 'concede remaining resources';
ok $game->action('RM', { player => 4, tile => [0,2]}), 'move robber';
ok $game->action('RR', { player => 4, target_player => 2, code => 'L'}), 'robber steal';
ok $game->action('TE', { player => 4}), 'end turn';

# round 7
ok $game->action('DR', { player => 1, result => 4}), 'roll dice';
ok $game->action('BR', { player => 1, path => [[[1,-1],[0,0],[0,-1]],[[0,-1],[0,0],[-1,0]]]}), 'build road';
ok $game->action('BS', { player => 1, intersection => [[1,-1],[0,0],[0,-1]]}), 'build settlement';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 2}), 'roll dice';
ok $game->action('TO', { player => 2, resources=> {2=>{B=>1,W=>1,G=>-2},3=>{B=>-1,G=>2,W=>-1}}, uuid => 7}), 'trade offer';
ok $game->action('TA', { player => 3, uuid => 7}), 'trade accept';
ok $game->action('BR', { player => 2, path => [[[-1,2],[-2,3],[-2,2]],[[-1,1],[-1,2],[-2,2]]]}), 'build road';
ok $game->action('BD', { player => 2, type => 'RB'}), 'build development card';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 7}), 'roll dice';
ok $game->action('RM', { player => 3, tile => [0,-1]}), 'move robber';
ok $game->action('RR', { player => 3, target_player => 1, code => 'L'}), 'robber steal';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 5}), 'roll dice';
ok $game->action('TE', { player => 4}), 'end turn';

# round 8
ok $game->action('DR', { player => 1, result => 11}), 'roll dice';
ok $game->action('TO', { player => 1, resources=> {1=>{L=>-1,O=>1,G=>1},3=>{L=>1,G=>-1,O=>-1}}, uuid => 8}), 'trade offer';
ok $game->action('TA', { player => 3, uuid => 8}), 'trade accept';
ok $game->action('BC', { player => 1, intersection => [[1,-1],[0,0],[0,-1]]}), 'build city';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 6}), 'roll dice';
ok $game->action('PD', { player => 2, type => 'RB' }), 'play road building';
ok $game->action('BR', { player => 2, path => [[[-1,2],[-2,3],[-2,2]],[[-1,2],[-1,3],[-2,3]]]}), 'build road';
ok $game->action('BR', { player => 2, path => [[[0,2],[-1,3],[-1,2]],[[-1,2],[-1,3],[-2,3]]]}), 'build road';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 8}), 'roll dice';
ok $game->action('BS', { player => 3, intersection => [[-2,2],[-2,3],[-3,3]]}), 'build settlement';
ok $game->action('BR', { player => 3, path => [[[-2,2],[-2,3],[-3,3]],[[-1,2],[-2,3],[-2,2]]]}), 'build road';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 5}), 'roll dice';
ok $game->action('TO', { player => 4, resources=> {3=>{B=>1,W=>-1},4=>{B=>-1,W=>1}}, uuid => 8}), 'trade offer';
ok $game->action('TA', { player => 3, uuid => 8}), 'trade accept';
ok $game->action('BS', { player => 4, intersection => [[1,0],[0,1],[0,0]]}), 'build settlement';
ok $game->action('TE', { player => 4}), 'end turn';

# round 9
ok $game->action('DR', { player => 1, result => 7}), 'roll dice';
ok $game->action('RM', { player => 1, tile => [0,-2]}), 'move robber';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 6}), 'roll dice';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 11}), 'roll dice';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 5}), 'roll dice';
ok $game->action('BR', { player => 4, path => [[[2,-2],[1,-1],[1,-2]],[[2,-2],[2,-1],[1,-1]]]}), 'build road';
ok $game->action('TE', { player => 4}), 'end turn';

# round 10
ok $game->action('DR', { player => 1, result => 11}), 'roll dice';
ok $game->action('BC', { player => 1, intersection => [[0,-1],[-1,0],[-1,-1]]}), 'build city';
ok $game->action('TB', { player => 1, resources => {1 => {G=>1, B=>-4}} }), 'trade 4 brick for a grain';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 2}), 'roll dice';
ok $game->action('TB', { player => 2, resources => {2 => {B=>1, L=>-2}} }), 'trade 2 lumber for a brick';
ok $game->action('BR', { player => 2, path => [[[0,2],[-1,3],[-1,2]],[[0,2],[0,3],[-1,3]]]}), 'build road';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 11}), 'roll dice';
ok $game->action('TO', { player => 3, resources=> {1=>{L=>-1,B=>-1,O=>4},3=>{L=>1,B=>1,O=>-4}}, uuid => 2}), 'trade offer';
ok $game->action('TA', { player => 1, uuid => 2}), 'trade accept';
ok $game->action('BR', { player => 3, path => [[[-1,0],[-1,1],[-2,1]],[[-1,0],[-2,1],[-2,0]]]}), 'build road';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 6}), 'roll dice';
ok $game->action('TE', { player => 4}), 'end turn';

# round 11
ok $game->action('DR', { player => 1, result => 12}), 'roll dice';
ok $game->action('BC', { player => 1, intersection => [[1,-2],[0,-1],[0,-2]]}), 'build city';
ok $game->action('BC', { player => 1, intersection => [[0,-2],[-1,-1],[-1,-2]]}), 'build city';
ok $game->action('BD', { player => 1, type => 'KN'}), 'build development card';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('TB', { player => 2, resources => {2 => {B=>1, L=>-2}} }), 'trade 2 lumber for a brick';
ok $game->action('DR', { player => 2, result => 12}), 'roll dice';
ok $game->action('BR', { player => 2, path => [[[1,2],[0,3],[0,2]],[[0,2],[0,3],[-1,3]]]}), 'build road';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 12}), 'roll dice';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 6}), 'roll dice';
ok $game->action('TE', { player => 4}), 'end turn';

# round 12
ok $game->action('DR', { player => 1, result => 11}), 'roll dice';
ok $game->action('PD', { player => 1, type => 'KN'}), 'play knight';
ok $game->action('RM', { player => 1, tile => [-1,2]}), 'move robber';
ok $game->action('RR', { player => 1, target_player => 2, code => 'G'}), 'robber rob';
ok $game->action('BD', { player => 1, type => 'KN'}), 'build development card';
ok $game->action('BD', { player => 1, type => 'KN'}), 'build development card';
ok $game->action('BD', { player => 1, type => 'VP'}), 'build development card';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 9}), 'roll dice';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 11}), 'roll dice';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 2}), 'roll dice';
ok $game->action('TE', { player => 4}), 'end turn';

# round 13
ok $game->action('DR', { player => 1, result => 2}), 'roll dice';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 2}), 'roll dice';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 2}), 'roll dice';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 2}), 'roll dice';
ok $game->action('TE', { player => 4}), 'end turn';

# round 14
ok $game->action('PD', { player => 1, type => 'KN'}), 'play knight';
ok $game->action('RM', { player => 1, tile => [1,-2]}), 'move robber';
ok $game->action('DR', { player => 1, result => 12}), 'roll dice';
ok $game->action('TE', { player => 1}), 'end turn';
ok $game->action('DR', { player => 2, result => 12}), 'roll dice';
ok $game->action('TE', { player => 2}), 'end turn';
ok $game->action('DR', { player => 3, result => 12}), 'roll dice';
ok $game->action('TE', { player => 3}), 'end turn';
ok $game->action('DR', { player => 4, result => 12}), 'roll dice';
ok $game->action('TE', { player => 4}), 'end turn';

# round 15
ok my $endgame = $game->action('PD', { player => 1, type => 'KN'}), 'play knight';

# end
ok exists $endgame->[-3]{GO}, 'game over event';
ok exists $endgame->[-2]{PE}, 'phase end play';
ok exists $endgame->[-1]{PS}, 'phase start end';

done_testing;
#print Dumper({$_->name => $_->summary}) for @{$game->players};
