# NAME

Venus - OO Library

# ABSTRACT

OO Standard Library for Perl 5

# VERSION

0.04

# SYNOPSIS

    package main;

    use Venus qw(
      catch
      error
      raise
    );

    # error handling
    my ($error, $result) = catch {
      error;
    };

    # boolean keywords
    if ($result and $result eq false) {
      true;
    }

    # raise exceptions
    if (false) {
      raise 'MyApp::Error';
    }

    # and much more!
    true ne false;

# DESCRIPTION

This library provides an object-orientation framework and extendible standard
library for Perl 5, built on top of [Moo](https://metacpan.org/pod/Moo) with classes which wrap most native
Perl data types. Venus has a simple modular architecture, robust library of
classes and methods, supports pure-Perl autoboxing, advanced exception
handling, "true" and "false" keywords, package introspection, command-line
options parsing, and more. This package will always automatically exports
`true` and `false` keyword functions (unless existing routines of the same
name already exist in the calling package), otherwise exports keyword functions
as requested at import. This library requires Perl `5.18+`.

# FUNCTIONS

This package provides the following functions:

## catch

    catch(CodeRef $block) (Error, Any)

The catch function executes the code block trapping errors and returning the
caught exception in scalar context, and also returning the result as a second
argument in list context.

_Since `0.01`_

- catch example 1

        package main;

        use Venus 'catch';

        my $error = catch {die};

        $error; # 'Died at ...'

- catch example 2

        package main;

        use Venus 'catch';

        my ($error, $result) = catch {error};

        $error; # Venus::Error

- catch example 3

        package main;

        use Venus 'catch';

        my ($error, $result) = catch {true};

        $result; # 1

## error

    error(Maybe[HashRef] $args) (Error)

The error function throws a [Venus::Error](https://metacpan.org/pod/Venus%3A%3AError) exception object using the
exception object arguments provided.

_Since `0.01`_

- error example 1

        package main;

        use Venus 'error';

        my $error = error;

- error example 2

        package main;

        use Venus 'error';

        my $error = error {
          message => 'Something failed!',
        };

## false

    false() (Bool)

The false function returns a falsy boolean value which is designed to be
practically indistinguishable from the conventional numerical `0` value.

_Since `0.01`_

- false example 1

        package main;

        use Venus;

        my $false = false;

- false example 2

        package main;

        use Venus;

        my $true = !false;

## raise

    raise(Str $class | Tuple[Str, Str] $class, Maybe[HashRef] $args) (Error)

The raise function generates and throws a named exception object derived from
[Venus::Error](https://metacpan.org/pod/Venus%3A%3AError), or provided base class, using the exception object arguments
provided.

_Since `0.01`_

- raise example 1

        package main;

        use Venus 'raise';

        my $error = raise 'MyApp::Error';

- raise example 2

        package main;

        use Venus 'raise';

        my $error = raise ['MyApp::Error', 'Venus::Error'];

- raise example 3

        package main;

        use Venus 'raise';

        my $error = raise ['MyApp::Error', 'Venus::Error'], {
          message => 'Something failed!',
        };

## true

    true() (Bool)

The true function returns a truthy boolean value which is designed to be
practically indistinguishable from the conventional numerical `1` value.

_Since `0.01`_

- true example 1

        package main;

        use Venus;

        my $true = true;

- true example 2

        package main;

        use Venus;

        my $false = !true;

# FEATURES

This package provides the following features:

- standard-library

    This library provides a Perl object-oriented standard library with value
    classes and consistently named methods.

    **example 1**

        package main;

        use Venus::Array;

        my $array = Venus::Array->new([1..4]);

        # $array->all(sub{ $_ > 0 });
        # $array->any(sub{ $_ > 0 });
        # $array->each(sub{ $_ > 0 });
        # $array->grep(sub{ $_ > 0 });
        # $array->map(sub{ $_ > 0 });
        # $array->none(sub{ $_ < 0 });
        # $array->one(sub{ $_ == 0 });
        # $array->random;

        use Venus::Hash;

        my $hash = Venus::Hash->new({1..8});

        # $hash->all(sub{ $_ > 0 });
        # $hash->any(sub{ $_ > 0 });
        # $hash->each(sub{ $_ > 0 });
        # $hash->grep(sub{ $_ > 0 });
        # $hash->map(sub{ $_ > 0 });
        # $hash->none(sub{ $_ < 0 });
        # $hash->one(sub{ $_ == 0 });
        # $hash->random;

        $array->count == $hash->count;

- value-classes

    This library provides value classes which wrap native Perl data types and
    provides methods for operating their values.

    **example 1**

        package main;

        use Venus::Array;

        my $array = Venus::Array->new;

    **example 2**

        package main;

        use Venus::Boolean;

        my $boolean = Venus::Boolean->new;

    **example 3**

        package main;

        use Venus::Code;

        my $code = Venus::Code->new;

    **example 4**

        package main;

        use Venus::Float;

        my $float = Venus::Float->new;

    **example 5**

        package main;

        use Venus::Hash;

        my $hash = Venus::Hash->new;

    **example 6**

        package main;

        use Venus::Number;

        my $number = Venus::Number->new;

    **example 7**

        package main;

        use Venus::Regexp;

        my $regexp = Venus::Regexp->new;

    **example 8**

        package main;

        use Venus::Scalar;

        my $scalar = Venus::Scalar->new;

    **example 9**

        package main;

        use Venus::String;

        my $string = Venus::String->new;

    **example 10**

        package main;

        use Venus::Undef;

        my $undef = Venus::Undef->new;

- builtin-autoboxing

    This library provides opt-in pure Perl autoboxing allowing you to chain methods
    calls across objects and values.

    **example 1**

        package main;

        use Venus::String;

        my $string = Venus::String->new('hello, world');

        $string->box->split(', ')->join(' ')->titlecase->unbox->get; # Hello World

- utility-classes

    This library provides serveral essential utility classes for performing common
    programming tasks.

    **example 1**

        package main;

        use Venus::Args;

        my $args = Venus::Args->new;

    **example 2**

        package main;

        use Venus::Box;

        my $box = Venus::Box->new;

    **example 3**

        package main;

        use Venus::Data;

        my $docs = Venus::Data->new->docs;

    **example 4**

        package main;

        use Venus::Date;

        my $date = Venus::Date->new;

    **example 5**

        package main;

        use Venus::Error;

        my $error = Venus::Error->new;

    **example 6**

        package main;

        use Venus::Json;

        my $json = Venus::Json->new;

    **example 7**

        package main;

        use Venus::Name;

        my $name = Venus::Name->new;

    **example 8**

        package main;

        use Venus::Opts;

        my $opts = Venus::Opts->new;

    **example 9**

        package main;

        use Venus::Path;

        my $path = Venus::Path->new;

    **example 10**

        package main;

        use Venus::Data;

        my $text = Venus::Data->new->text;

    **example 11**

        package main;

        use Venus::Space;

        my $space = Venus::Space->new;

    **example 12**

        package main;

        use Venus::Throw;

        my $throw = Venus::Throw->new;

    **example 13**

        package main;

        use Venus::Try;

        my $try = Venus::Try->new;

    **example 14**

        package main;

        use Venus::Type;

        my $type = Venus::Type->new;

    **example 15**

        package main;

        use Venus::Vars;

        my $vars = Venus::Vars->new;

- package-reflection

    This library provides a package reflection class, [Venus::Space](https://metacpan.org/pod/Venus%3A%3ASpace), which can be
    used to perform meta-programming on package spaces.

    **example 1**

        package main;

        use Venus::Space;

        my $space = Venus::Space->new('Venus');

        $space->do('tryload')->routines;

- exception-handling

    This library provides a framework for raising, i.e. generating and throwing,
    exception objects and catching them.

    **example 1**

        package MyApp;

        use Venus::Class;

        with 'Venus::Role::Throwable';
        with 'Venus::Role::Catchable';

        sub execute {
          my ($self) = @_;

          $self->throw->error;
        }

        package main;

        my $myapp = MyApp->new;

        my $error = $myapp->catch('execute');

        # $error->isa('MyApp::Error');

- composable-standards

    This library provides a library of composable roles which can be used to extend
    core behaviors to custom objects.

    **example 1**

        package MyApp;

        use Venus::Class;

        with 'Venus::Role::Dumpable';
        with 'Venus::Role::Stashable';

        package main;

        my $myapp = MyApp->new;

        $myapp->stash(greeting => 'hello world');

        $myapp->dump('stash');

        # '{"greeting" => "hello world"}'

- pluggable-library

    This library provides a mechanism for extending the standard library, i.e.
    value classes, using plugins which can be automatically discovered and invoked.
    (no monkey-patching necessary)

    **example 1**

        package Venus::String::Plugin::Base64;

        sub new {
          return bless {};
        }

        sub execute {
          my ($self, $string, @args) = @_;

          require MIME::Base64;

          return MIME::Base64::encode_base64($string->value);
        }

        package main;

        use Venus::String;

        my $string = Venus::String->new('hello, world');

        $string->base64;

- template-system

    This library provides a minimalistic templating system.

    **example 1**

        package main;

        use Venus::Template;

        my $template = Venus::Template->new(q(
          {{ if user.name }}
          Welcome, {{ user.name }}!
          {{ else user.name }}
          Welcome, friend!
          {{ end user.name }}
        ));

        $template->render;

# AUTHORS

Cpanery, `cpanery@cpan.org`

# LICENSE

Copyright (C) 2021, Cpanery

Read the ["license"](https://github.com/cpanery/venus/blob/master/LICENSE) file.

# PROJECT

[https://venus.cpanery.com](https://venus.cpanery.com)

[https://github.com/cpanery/venus/wiki](https://github.com/cpanery/venus/wiki)

[https://github.com/cpanery/venus/issues](https://github.com/cpanery/venus/issues)

[https://cpanery.com](https://cpanery.com)
