/* LinkedList.c -- implementation of singly-linked list

	THIS SOFTWARE FITS THE DESCRIPTION IN THE U.S. COPYRIGHT ACT OF A
	"UNITED STATES GOVERNMENT WORK".  IT WAS WRITTEN AS A PART OF THE
	AUTHOR'S OFFICIAL DUTIES AS A GOVERNMENT EMPLOYEE.  THIS MEANS IT
	CANNOT BE COPYRIGHTED.  THIS SOFTWARE IS FREELY AVAILABLE TO THE
	PUBLIC FOR USE WITHOUT A COPYRIGHT NOTICE, AND THERE ARE NO
	RESTRICTIONS ON ITS USE, NOW OR SUBSEQUENTLY.

Author:
	K. E. Gorlen
	Bg. 12A, Rm. 2033
	Computer Systems Laboratory
	Division of Computer Research and Technology
	National Institutes of Health
	Bethesda, Maryland 20892
	Phone: (301) 496-1111
	uucp: uunet!nih-csl!keith
	Internet: keith@alw.nih.gov
	October, 1985

Function:
	
LinkedLists are ordered by the sequence in which objects are added and removed
from them.  Object elements are accessible by index.

$Log:	LinkedList.c,v $
 * Revision 2.204  89/10/07  23:20:11  keith
 * Pre-release
 * 
 * Revision 2.203  89/08/12  23:42:05  keith
 * Pre-release
 * Move nilLink to class Link
 * Use Link::isListEnd() test.
 * Implement addAfter().
 * 
 * Revision 2.202.1.8  89/08/01  15:08:13  keith
 * Change linkCastdown(Object*) to non-virtual inline.
 * 
 * Revision 2.202.1.7  89/07/30  20:41:00  keith
 * Add linkCastdown() functions.
 * 
 * Revision 2.202.1.6  89/07/11  12:01:01  keith
 * Add static member nilLink.
 * Overload add(), addFirst(), addLast(), and remove() for
 * class Link& arguments.
 * 
 * Revision 2.202.1.5  89/07/10  21:56:06  keith
 * Add copy constructors to fix yacc stack overflows.
 * 
 * Revision 2.202.1.4  89/07/08  19:11:57  keith
 * Add initialization of virtual base Object to readFrom() constructors
 * 
 * Revision 2.202.1.3  89/07/02  11:23:47  keith
 * Fix doNext to check for nilLink instead of nil.
 * 
 * Revision 2.202.1.2  89/07/01  23:44:20  keith
 * Make nilLink a pointer to a new LinkOb instead of
 * Object::nil castdown to a Link*, which won't work
 * under MI.
 * 
 * Revision 2.202.1.1  89/07/01  21:54:46  keith
 * Base revision for R2.00 MI version
 * 
 * Revision 2.202  89/06/22  20:54:41  keith
 * Base revision for AT&T C++ R2.0 release (Cycle 20)
 * 
 * Revision 2.201.1.3  89/06/22  10:14:53  keith
 * Remove unnecessary copy constructors.
 * 
 * Revision 2.201.1.2  89/06/20  23:12:35  keith
 * Make operator[]() return Object* instead of Object*&.
 * Make at() shouldNotImplement.
 * Add explicit base names to constructor initializer lists.
 * Change 2nd argument of DO to class name.
 * 
 * Revision 2.201.1.1  89/06/13  22:57:47  keith
 * Base revision for Cycle 16.1.
 * 
 * Revision 2.201  89/05/12  11:18:05  keith
 * Release for R2.0 Beta test.
 * 
 * Revision 2.200.1.4  89/05/12  10:59:26  keith
 * Revised Object I/O.
 * 
 * Revision 2.200.1.3  89/05/05  09:58:35  keith
 * Overload reference-returning member functions (e.g.
 * operator[](int) and at(int) for const instances.
 * 
 * Revision 2.200.1.2  89/05/03  23:08:51  keith
 * Utilize abstract classes.
 * 
 * Revision 2.200.1.1  89/04/24  17:16:13  keith
 * Working revision for R2.0 Beta 6++
 * 
 * Revision 2.200  89/04/17  23:29:20  keith
 * Base revision for R2.0 Beta 6.
 * 
 * Revision 2.121  89/02/16  11:07:03  keith
 * Base revision for C++ R1.2.1 compatible version.
 * 
*/

#include "LinkedList.h"
#include "nihclIO.h"

#define	THIS	LinkedList
#define	BASE	SeqCltn
#define BASE_CLASSES BASE::desc()
#define MEMBER_CLASSES
#define VIRTUAL_BASE_CLASSES

DEFINE_CLASS(LinkedList,1,"$Header: LinkedList.c,v 2.204 89/10/07 23:20:11 keith Stab $",NULL,NULL);

extern const int NIHCL_DBLLNK,NIHCL_CLTNEMPTY,NIHCL_MISSINGLNK,NIHCL_OBNOTFOUND,NIHCL_INDEXRANGE;

/*
Derived classes must override these definitions of linkCastdown() when
adding objects to a LinkedList that are multiply derived from class
Link.
*/

Link& LinkedList::linkCastdown(Object& p) const
{
	return Link::castdown(p);
}

LinkedList::LinkedList()
{
	firstLink = lastLink = Link::nilLink;
	count = 0;
}

#ifndef BUG_TOOBIG
// yacc stack overflow

LinkedList::LinkedList(const LinkedList& l)
{
	firstLink = l.firstLink;
	lastLink = l.lastLink;
	count = l.count;
}

#endif

bool LinkedList::operator==(const LinkedList& ll) const
{
	if (count != ll.count) return NO;
	else {
		register Link* p = firstLink;
		register Link* q = ll.firstLink;
		while (!p->isListEnd()) {
			if (!(p->isEqual(*q))) return NO;
			p = p->nextLink();
			q = q->nextLink();
		}
	}
	return YES;
}

Object* LinkedList::add(Link& ob)
{
	register Link* lnk = &ob;
	if (lnk->nextLink() != NULL) errDblLnk("add",*lnk);
	if (count == 0) firstLink = lnk;
	else lastLink->nextLink(lnk);
	lastLink = lnk;
	lnk->nextLink(Link::nilLink);
	count++;
	return &ob;
}

Object* LinkedList::add(Object& ob)
{
	assertArgClass(ob,*Link::desc(),"add");
	return add(Link::castdown(ob));
}

Object* LinkedList::addAfter(Link& afterLink, Link& newLink)
{
	if (newLink.nextLink() != NULL) errDblLnk("addAfter",newLink);
	if (afterLink.nextLink() == NULL || count == 0)
		setError(NIHCL_MISSINGLNK,DEFAULT,afterLink.className(),this,className(),
			afterLink.className(),&afterLink,newLink.className(),&newLink);
	newLink.nextLink(afterLink.nextLink());
	afterLink.nextLink(&newLink);
	if (lastLink == &afterLink) lastLink = &newLink;
	count++;
	return &newLink;
}

Object* LinkedList::addAfter(Object& afterLink, Object& newLink)
{
	assertArgClass(afterLink,*Link::desc(),"add");
	assertArgClass(newLink,*Link::desc(),"add");
	return addAfter(Link::castdown(afterLink), Link::castdown(newLink));
}

Collection& LinkedList::addContentsTo(Collection& cltn) const
{
	register Link* p = firstLink;
	while (!p->isListEnd()) {
		register Link* t = linkCastdown(p->copy());
		cltn.add(*t);
		p = p->nextLink();
	}
	return cltn;
}

Object* LinkedList::addFirst(Link& ob)
{
	assertArgClass(ob,*Link::desc(),"addFirst");
	if (count == 0) return add(ob);
	else {
		register Link* lnk = &ob;
		if (lnk->nextLink() != NULL) errDblLnk("addFirst",*lnk);
		lnk->nextLink(firstLink);
		firstLink = lnk;
		count++;
		return &ob;
	}
}

Object* LinkedList::addFirst(Object& ob)
{
	assertArgClass(ob,*Link::desc(),"addFirst");
	return addFirst(Link::castdown(ob));
}

Object* LinkedList::addLast(Link& ob) { return add(ob); }

Object* LinkedList::addLast(Object& ob) { return add(ob); }

Object* LinkedList::operator[](int i)
/*
Note that we can't return an Object*& because of MI: a Link* may need
to be adjusted to become an Object*.
*/
{
	if ((unsigned)i >= count) indexRangeErr();
	if (i==0) return firstLink;
	else {
		register Link* p = firstLink;
		for (register int j=i-1; j>0; j--) p = p->nextLink();
		return p->next;
	}
}

const Object *const LinkedList::operator[](int i) const
{
	if ((unsigned)i >= count) indexRangeErr();
	if (i==0) return firstLink;
	else {
		register Link* p = firstLink;
		for (register int j=i-1; j>0; j--) p = p->nextLink();
		return p->next;
	}
}

Object*& LinkedList::at(int)
// Can't implement at() because of MI -- see operator[]().
{
	shouldNotImplement("at");
	return (Object*&)nil;
}

const Object *const& LinkedList::at(int) const
// Can't implement at() because of MI -- see operator[]().
{
	shouldNotImplement("at");
	return (Object*&)nil;
}

void LinkedList::atAllPut(Object&) { shouldNotImplement("atAllPut"); }

void LinkedList::deepenShallowCopy()
{
	BASE::deepenShallowCopy();
	register Link* p = firstLink;
	firstLink = lastLink = Link::nilLink;
	count = 0;
	while (!p->isListEnd()) {
		add(*(p->deepCopy()));
		p = p->nextLink();
	}
}

Object* LinkedList::first() const
{
	if (count==0) errEmpty("first");
	else return firstLink;
}

unsigned LinkedList::hash() const
{
	register unsigned h = count;
	register Link* p = firstLink;
	while (!p->isListEnd()) {
		h^= p->hash();
		p = p->nextLink();
	}
	return h;
}

bool LinkedList::includes(const Object& ob) const
{
	return (indexOf(ob) != -1);
}

int LinkedList::indexOf(const Object& ob) const
{
	register int i = 0;
	register Link* p = firstLink;
	while (!p->isListEnd()) {
		if (p->isEqual(ob)) return i;
		p = p->nextLink();
		i++;
	}
	return -1;
}

int LinkedList::indexOfSubCollection(const SeqCltn& /*cltn*/, int /*start*/) const
{	shouldNotImplement("indexOfSubCollection"); return 0;	}

bool LinkedList::isEmpty() const	{ return count==0; }

bool LinkedList::isEqual(const Object& a) const
{
	return a.isSpecies(classDesc) && *this==castdown(a);
}

const Class* LinkedList::species() const { return &classDesc; }

Object* LinkedList::last() const
{
	if (count==0) errEmpty("last");
	else return lastLink;
}

Object* LinkedList::doNext(Iterator& pos) const
{
	if (pos.ptr == 0) {
		if (count == 0) return 0;
		else {
			pos.ptr = firstLink;
			pos.index = 1;
			return firstLink;
		}
	}
else	if (pos.ptr == lastLink) return 0;
else		{
		pos.ptr = linkCastdown(pos.ptr)->nextLink();
		pos.index++;
		return pos.ptr;
	}
}

unsigned LinkedList::occurrencesOf(const Object& ob) const
{
	register unsigned n=0;
	register Link* p = firstLink;
	while (!p->isListEnd()) {
		if (p->isEqual(ob)) n++;
		p = p->nextLink();
	}
	return n;
}

Object* LinkedList::remove(const Link& ob)
{
	if (count==0) errEmpty("remove");
	register Link* lnk = &(Link&)ob;
	if (lnk == firstLink) {
		firstLink = lnk->nextLink();
		if (lnk == lastLink) lastLink = firstLink;
		goto wrapup;
	}
	else {
		register Link* p = firstLink;
		while (!p->isListEnd()) {
			if (lnk == p->nextLink()) {
				p->nextLink(lnk->nextLink());
				if (lnk == lastLink) lastLink = p;
				goto wrapup;
			}
			p = p->nextLink();
		}
		errNotFound("remove",ob);
	}
wrapup:
	lnk->nextLink(NULL);
	count--;
	return lnk;
}

Object* LinkedList::remove(const Object& ob)
{
	assertArgClass(ob,*Link::desc(),"remove");
	return remove(Link::castdown(ob));
}

void LinkedList::removeAll()
{
	while (count) remove(*firstLink);
}

Object* LinkedList::removeFirst()	{ return remove(*firstLink); }

Object* LinkedList::removeLast()	{ return remove(*lastLink); }
	
void LinkedList::replaceFrom(int /*start*/, int /*stop*/, const SeqCltn& /*replacement*/, int /*startAt*/)
{
	shouldNotImplement("replaceFrom");
}

void LinkedList::reSize(unsigned newSize) {}

unsigned LinkedList::size() const	{ return count; }
	
LinkedList::LinkedList(OIOin& strm)
:
#ifdef MI
	Object(strm),
#endif
	BASE(strm)
{
	firstLink = lastLink = Link::nilLink;
	count = 0;
	unsigned n;
	strm >> n;
	while (n--) add(*Object::readFrom(strm));
}

void LinkedList::storer(OIOout& strm) const
{
	BASE::storer(strm);
	strm << size();
	DO(*this,Object,ob) ob->storeOn(strm); OD
}

LinkedList::LinkedList(OIOifd& fd)
:
#ifdef MI
	Object(fd),
#endif
	BASE(fd)
{
	firstLink = lastLink = Link::nilLink;
	count = 0;
	unsigned n;
	fd >> n;
	while (n--) add(*Object::readFrom(fd));
}

void LinkedList::storer(OIOofd& fd) const
{
	BASE::storer(fd);
	fd << size();
	DO(*this,Object,ob) ob->storeOn(fd); OD
}

void LinkedList::errDblLnk(const char* fn, const Link& lnk) const
{
	setError(NIHCL_DBLLNK,DEFAULT,lnk.className(),className(),this,className(),fn,lnk.className(),&lnk);
}

void LinkedList::errEmpty(const char* fn) const
{
	setError(NIHCL_CLTNEMPTY,DEFAULT,this,className(),fn);
}

void LinkedList::errNotFound(const char* fn, const Object& ob) const
{
	setError(NIHCL_OBNOTFOUND,DEFAULT,this,className(),fn,ob.className(),&ob);
}
