/*
** Copyright 1989, 1993 by Matthias Boldt
**			   and several students at the 
**			   Institute for Computergraphics
**			   University Rostock (Germany)
**
** Permission to use, copy, and distribute for non-commercial purposes,
** is hereby granted without fee, providing that the above copyright
** notice appear in all copies and that both the copyright notice and this
** permission notice appear in supporting documentation.
** The software may be modified for your own purposes, but modified versions
** may not be distributed.
** This software is provided "as is" without any expressed or implied warranty.
**
**
*/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <values.h>

#ifdef __TURBOC__
  #include <getopt.c>
#endif

# define boolean unsigned char
# define true    1
# define false   0


/************************************************************************/
/*                                                                      */
/*        Globale Typ- und Variablenvereinbarungen fuer den             */
/*                          Scanline-Algorithmus                        */
/*                     und die Schattierungsverfahren                   */
/*                                                                      */
/*                        Matthias Boldt 27.2.1993                      */
/************************************************************************/


#define max_length_texfield 500
#define maxzwpuf            32
#define max_light_source    32


#define pi              3.14159265

typedef double MATRIX[4][4];

typedef struct color {
  unsigned int r, g, b;
} color;

typedef struct pointI {
  int x, y, z;
} pointI;

typedef struct pointR3 {
  double x, y, z;
} pointR3;

typedef struct pointR4 {
  double x, y, z, w;
} pointR4;


typedef pointR3 Vektor;

typedef struct zwischenpuffer {
  int anz, farbind, specn;
  pointI vektor[maxzwpuf];
  pointR3 normale[maxzwpuf];
} zwischenpuffer;

typedef enum {
  konst, gour, phong, modif2, noshade, randomsh
} schattierung;

typedef struct textur {
  int     texid;
  char    dname[50];
  color   cfeld[5];
  pointR3 p;
  int     i1, i2, i3, i4;
} textur;

typedef color texfeld[max_length_texfield];

typedef struct light_source {
  double  r,g,b;
  pointR3 l,h;
  double  i;
} light_source; 

/**************************************************************************/
/*      die allgemeinen Datenstrukturen                                   */
/**************************************************************************/

typedef struct attrib {
  int           pol;
  double        kd,ks;
  textur        *tex;
  struct attrib *next;
} attrib;


typedef struct edge {
  pointI  apkt;              /* aktueller Anfangspunkt            */
  pointI  epkt;              /* Endpunkt                          */
  double mx, mz;             /* Anstieg vom Anfangs- zum Endpunkt */
  double ax, az;             /* Hilfsvariablen fuer akt. Schnittp.*/
  long   ay;
  pointR3 mv;                /* "Anstieg" der Normale             */
  pointR3 anor;              /* aktuelle und Anfangsnormale       */
  pointR3 enor;              /* Endnormale                        */
  int     Nay;               /* Vorgaengerpunkt zum Anfangspunkt  */
  struct edge *prev;         /* vorhergehende Kante in der Liste  */
  struct edge *next;         /* nachfolgende Kante in der Liste   */
} edge;


typedef struct Polygon {
  edge              *k;     /* sortierte Kantenliste des Polygons */
  pointR3            v;     /* Gesamtnormale des Polygons         */
  int             ymax,     /* maximale y-Koordinate des Polygons */
                  ymin;     /* minimale y-Koordinate des Polygons */
  struct Polygon *prev;     /* vorhergehendes Polygon in Liste    */
  struct Polygon *next;     /* naechstes Polygon in Liste         */
  attrib         *att;
  color          col;
} Polygon;

typedef struct list_Poly {
  Polygon *list;            /* sortierte Polygonliste            */
  Polygon *pointer;         /* Zeiger auf aktuelles Polygon      */
} list_Poly;


typedef struct spanne {
  int xbeg,                 /* Anfangs und Endpunkt der Spanne   */
      xend, zbeg, zend;
  pointR3       norbeg;     /* Anfangsnormale der Spanne         */
  pointR3       norend;     /* Endnormale der Spanne             */
  color         col;
  attrib        *att;
  struct spanne *next;      /* naechste Spanne der Liste         */
  struct spanne *prev;      /* vorhergehende Spanne der Liste    */
} spanne;

/**************************************************************************/
/*         die globalen Parameter der Prozeduren                          */
/**************************************************************************/


static  char *infile = NULL,
	     *outfile = NULL;
static FILE *A, *B;                    /* Ein- und Ausgabedateien        */
static boolean comments;
static edge *start_edge, *akt_edge;
static color back;                     /* Hintergrundfarbe               */
static int   x_dim,y_dim;              /* x- und y-Ausdehnung der Szene  */
                                       /* Viewport 0..x_dim-1 0..y_dim-1 */
                                       /* Vektor der Punktlichtquellen   */
static light_source lights[max_light_source];
static int   anz_lights;
static attrib *attr_list; 


/**************************************************************************/
/*         allgemeine Prozeduren und Makros                               */
/**************************************************************************/

static void setPixel(color pix)
{
 unsigned char c;

 if (pix.r > 255)
   c=255;
 else
   c=(unsigned char)pix.r;
 fputc(c,B);
 if (pix.g > 255)
   c=255;
 else
   c=(unsigned char)pix.g;
 fputc(c,B);
 if (pix.b > 255)
   c=255;
 else
   c=(unsigned char)pix.b;
 fputc(c,B);
}


#define BOGEN(a) ((a) * 0.017453)

static double betrag(pointR3 v)
{
  return sqrt(v.x * v.x + v.y * v.y + v.z * v.z);
}


static void normalise(pointR3 *v)
{
  double b;

  b = sqrt(v->x * v->x + v->y * v->y + v->z * v->z);
  if (b == 0)
    return;
  v->x /= b;
  v->y /= b;
  v->z /= b;
}


static double scalar_p(pointR3 v1,pointR3 v2)
{
  return (v1.x * v2.x + v1.y * v2.y + v1.z * v2.z);
}


static void addpoint(pointR3 p1,double l,pointR3 v,pointR3 *p2)
{
  p2->x = p1.x + v.x * l;
  p2->y = p1.y + v.y * l;
  p2->z = p1.z + v.z * l;
}

static void vectorproduct(pointR3 a,pointR3 b,pointR3 *erg)
{
  double x,y,z; 
    x = a.y * b.z - b.y * a.z;
    y = a.z * b.x - b.z * a.x;
    z = a.x * b.y - b.x * a.y;
    erg->x = x;
    erg->y = y;
    erg->z = z;
}

static void transform_point(pointR4 p,MATRIX mx,pointR4 *erg)
{
    erg->x = p.x*mx[0][0] + p.y*mx[1][0] + p.z*mx[2][0] + p.w*mx[3][0] ;
    erg->y = p.x*mx[0][1] + p.y*mx[1][1] + p.z*mx[2][1] + p.w*mx[3][1] ;
    erg->z = p.x*mx[0][2] + p.y*mx[1][2] + p.z*mx[2][2] + p.w*mx[3][2] ;
    erg->w = p.x*mx[0][3] + p.y*mx[1][3] + p.z*mx[2][3] + p.w*mx[3][3] ;
}

#define pointR3_to_R4(a,b)  (b).x=(a).x;(b).y=(a).y;(b).z=(a).z;(b).w=1.0;

#define pointR4_to_R3(a,b)  (b).x=(a).x/(a).w;(b).y=(a).y/(a).w;(b).z=(a).z/(a).w;

static void matrix_multiply(MATRIX m1,MATRIX m2,MATRIX m3)
{
int	i, j ;

    for ( i = 0 ; i < 4 ; i++ ) {
	for ( j = 0 ; j < 4 ; j++ ) {
	    m3[i][j] = m1[i][0]*m2[0][j] +
		       m1[i][1]*m2[1][j] +
		       m1[i][2]*m2[2][j] +
		       m1[i][3]*m2[3][j] ;
	}
    }
}



/********************************************************************/
/* Prozeduren fuer Texturerzeugung und -mapping bei der Darstellung */
/********************************************************************/

/* Texturwertberechnung fuer einen Oberflaechenpunkt */
/* bei einem solid-texturing fuer "Holz"             */ 
static int holz(pointI p,pointR3 o,int n, int deltar)
{
  double r, konst;
  int index;

  if (deltar == 0) return(0);
  konst = 0.1 * (p.y - o.y);
  r = sqrt((p.x - o.x) * (p.x - o.x) + (p.z - o.z) * (p.z - o.z));
  r = deltar / 3.0 * sin(konst) * (deltar / 3.0) * cos(konst) + r;
  r /= deltar;
  index = (int)floor(r) % n;
  return(index);
} 




/*---------------------------------------------------------------------*/

/* Texturwerte fuer eine Spanne bestimmen */
static void texturiere(pointR3 n1,pointR3 n2,pointI p1,pointI p2,pointI p3,
                       int first,textur *zt,texfeld tf)
{
  int     n, deltar, x, xa;
  pointR3 o;
  pointI  punkt;
  double  z, t1, t2, an, hx, mx, mz, ax, az, ex, ez;

  if (zt == NULL) return;
  switch (zt->texid) 
  {
     case 0:
         /* solid-texturing ---> wood */
         /* get the data from the texture-structure */ 
         o = zt->p;
         n = zt->i1;
         deltar = zt->i2;  
         n1.y = 0.0;
         n2.y = 0.0;
         if (first == 1)
           {
             ax = p1.x;
             az = p1.z;
             ex = p2.x;
             ez = p2.z;
           }
         else
           {
             ax = p2.x;
             az = p2.z;
             ex = p3.x;
             ez = p3.z;
           };
         mx = p2.x;
         mz = p2.z;
         mx -= p1.x;
         ex -= p1.x;
         hx = p3.x-p1.x;
         /* determine values for the interpolation */
         if (mx == 0) 
           {
	     t2 = 0.0;
	     an = (ez - az)/ex;
           } 
         else
           {
	     t1 = (mz - p1.z) / mx;
	     t2 = (ez - p1.z - t1 * hx) / (hx * (hx - mx));
	     t1 -= t2 * mx;
	     an = t1 + t2;
	     t2 = 2 * t2;
           };
         /* Interpolation und Texturberechnung */
         z = az;
         x = (int)floor(ax);
         xa = 0;
         ex += ax;
         while (x <= ex) 
           {
             punkt.x = x;
             punkt.z = (int)floor(z);
             tf[xa] = zt->cfeld[holz(punkt,o,n,deltar)];
             z += an;
             an += t2;
             x++;
             xa++;
           }
     break;
     case 1:
       /* blank case */
     break;
  } /*case*/
}  




/*******************************************************************/
/*         Schattierung und Ausgabe der Punkte                     */
/*******************************************************************/

static int last_x;


static void set_point(int i,int j,double r_inten,double g_inten,double b_inten,
                                  int r_specin,int g_specin,int b_specin,
                                  boolean texturen,texfeld colarray,color farb, 
                                  boolean facet)
{
  int   x;
  color pixel;

  if (i <= last_x) return;               /* Punkt wurde bereits ausgegeben (in einer anderen Farbe)             */
  if ((i >= x_dim) || (i < 0)) return;   /* Fehler ---> Punkt liegt ausserhalb des Viewports                    */
  if (i > (last_x + 1))                  /* Punkte bis zu i muessen noch mit Hintergrundfarbe ausgegeben werden */
     for (x=(last_x + 1); x < i; x++) setPixel(back);
  if (! facet)
    { 
      if (texturen)
        {
          pixel.r=(int)floor(colarray[j].r * r_inten) + r_specin;
          pixel.g=(int)floor(colarray[j].g * g_inten) + g_specin;
          pixel.b=(int)floor(colarray[j].b * b_inten) + b_specin;
        }
      else
        {
          pixel.r=(int)floor(farb.r * r_inten) + r_specin;
          pixel.g=(int)floor(farb.g * g_inten) + g_specin;
          pixel.b=(int)floor(farb.b * b_inten) + b_specin;
        };
      setPixel(pixel);
    }
  else
    setPixel(farb);
  last_x=i;
}



static void konst_sh(pointI pbeg,pointI pend,pointR3 nb,
                     double kd,double ks,int specn,color farb,
                     texfeld colarray,boolean texturen,boolean specular)
{
  double r_inten, b_inten, g_inten, sp, spa; 
  int r_specin, b_specin, g_specin;
  int i, j, k;

  r_inten = 0;
  g_inten = 0;
  b_inten = 0;
  r_specin = 0; 
  g_specin = 0; 
  b_specin = 0; 
  normalise(&nb);
  for (i=0; i<anz_lights; i++)
    {
      if ((sp = scalar_p(lights[i].l,nb) * lights[i].i) < 0) sp = 0;
      r_inten += sp * lights[i].r;
      g_inten += sp * lights[i].g;
      b_inten += sp * lights[i].b;
      if (specular)
        { 
          spa = scalar_p(lights[i].h,nb);
          sp = spa; 
          if (sp > 0)
            for (j=2; j<=specn; j++) sp *= spa;
          else 
            sp = 0;
          sp *= lights[i].i * 255;
          r_specin += (int)floor(sp * lights[i].r);
          g_specin += (int)floor(sp * lights[i].g);
          b_specin += (int)floor(sp * lights[i].b);
        };
    };
  r_inten *= kd;
  g_inten *= kd;
  b_inten *= kd; 
  if (specular)
    {
      r_specin = (int)floor(r_specin * ks);
      g_specin = (int)floor(g_specin * ks);
      b_specin = (int)floor(b_specin * ks);
    }
  else
    {
      r_specin = 0;
      b_specin = 0;
      g_specin = 0;   
    }; 
  k = 0;
  i = pbeg.x;
  j = pend.x;
  while (i <= j) 
    {
      set_point(i,k, r_inten, g_inten, b_inten, 
                     r_specin, g_specin, b_specin, 
                     texturen, colarray, farb, false);
      i++;
      k++;
    };
}



static void modif2_sh(pointI pbeg,pointI pend,pointR3 nb,pointR3 ne,
                      double kd,double ks,int specn,color farb,
                      texfeld colarray,boolean texturen,boolean specular)
{
  double im, t1, t2, an, xe, xm, ib, ie, spia, 
         spim, spie, sp;
  double r_an, g_an, b_an, r_t2, g_t2, b_t2,
         r_span, g_span, b_span, r_spt2, g_spt2, b_spt2;
  double r_inten, b_inten, g_inten, r_sp, g_sp, b_sp;
  double r_b, g_b, b_b, r_m, g_m, b_m, r_e, g_e, b_e;  
  int    r_specin, b_specin, g_specin;
  int    dx, i, j, k;
  pointR3 nm;

  r_sp = g_sp = b_sp = r_inten = g_inten = b_inten = 0; 
  r_an = g_an = b_an = r_t2 = g_t2 = b_t2 = 0;
  r_span = g_span = b_span = r_spt2 = g_spt2 = b_spt2 = 0;
  r_specin = g_specin = b_specin = 0;
  dx = pend.x - pbeg.x;
  if (dx < 1) dx = 1;
  xe = dx;
  xm = xe / 2;
  normalise(&nb);
  normalise(&ne);
  nm.x = nb.x + ne.x;
  nm.y = nb.y + ne.y;
  nm.z = nb.z + ne.z;
  normalise(&nm);
  for (i=0; i<anz_lights; i++)
    {
      if ((ib = scalar_p(lights[i].l,nb) * lights[i].i * kd) < 0) ib = 0;
      if ((ie = scalar_p(lights[i].l,ne) * lights[i].i * kd) < 0) ie = 0;
      if ((im = scalar_p(lights[i].l,nm) * lights[i].i * kd) < 0) im = 0;
      r_inten += ib * lights[i].r;
      g_inten += ib * lights[i].g;
      b_inten += ib * lights[i].b;
      t1 = (im - ib) / xm;
      t2 = (ie - ib - t1 * xe) / (xe * (xe - xm));
      t1 -= t2 * xm;
      an = t1 + t2;
      t2 = 2 * t2;
      r_an += an * lights[i].r; 
      g_an += an * lights[i].g; 
      b_an += an * lights[i].b; 
      r_t2 += t2 * lights[i].r;
      g_t2 += t2 * lights[i].g;
      b_t2 += t2 * lights[i].b;
      if (specular) 
        {
          spia = scalar_p(lights[i].h,nb);
          spim = scalar_p(lights[i].h,nm);
          spie = scalar_p(lights[i].h,ne);
          sp = spia; 
          if (sp > 0)
            for (j=2; j<=specn; j++) sp *= spia;
          else 
            sp = 0;
          sp *= lights[i].i * 255 * ks;
          r_sp += r_b = sp * lights[i].r;
          g_sp += g_b = sp * lights[i].g;
          b_sp += b_b = sp * lights[i].b;

          sp = spim; 
          if (sp > 0)
            for (j=2; j<=specn; j++) sp *= spim;
          else 
            sp = 0;
          sp *= lights[i].i * 255 * ks;
          r_m = sp * lights[i].r;
          g_m = sp * lights[i].g;
          b_m = sp * lights[i].b;

          sp = spie; 
          if (sp > 0)
            for (j=2; j<=specn; j++) sp *= spie;
          else 
            sp = 0;
          sp *= lights[i].i * 255 * ks;
          r_e = sp * lights[i].r;
          g_e = sp * lights[i].g;
          b_e = sp * lights[i].b;
           
          t1 = (r_m - r_b) / xm;
          t2 = (r_e - r_b - t1 * xe) / (xe * (xe - xm));
          t1 -= t2 * xm;
          r_span += t1 + t2;
          r_spt2 += 2 * t2;

          t1 = (g_m - g_b) / xm;
          t2 = (g_e - g_b - t1 * xe) / (xe * (xe - xm));
          t1 -= t2 * xm;
          g_span += t1 + t2;
          g_spt2 += 2 * t2;

          t1 = (b_m - b_b) / xm;
          t2 = (b_e - b_b - t1 * xe) / (xe * (xe - xm));
          t1 -= t2 * xm;
          b_span += t1 + t2;
          b_spt2 += 2 * t2;
        };
    }; 
  k = 0;  
  i = pbeg.x;
  j = pend.x;
  while (i <= j) 
    {     
      if (specular) 
        {
          r_specin = (int)floor(r_sp);
          g_specin = (int)floor(g_sp);
          b_specin = (int)floor(b_sp);
        };
      if (r_inten < 0) r_inten = 0;
      if (g_inten < 0) g_inten = 0;
      if (b_inten < 0) b_inten = 0;
      if (r_specin < 0) r_specin = 0;
      if (g_specin < 0) g_specin = 0;
      if (b_specin < 0) b_specin = 0;
      set_point(i,k, r_inten, g_inten, b_inten, 
                     r_specin, g_specin, b_specin, 
                     texturen, colarray, farb, false);
      if (specular) 
        {
          r_sp += r_span;
          r_span  += r_spt2;
          g_sp += g_span;
          g_span  += g_spt2;
          b_sp += b_span;
          b_span  += b_spt2;
        };
      r_inten += r_an;
      r_an  += r_t2;
      g_inten += g_an;
      g_an  += g_t2;
      b_inten += b_an;
      b_an  += b_t2;
      i++;
      k++;
    };
}



static void gouraud_sh(pointI pbeg,pointI pend,pointR3 nb,pointR3 ne,
                       double kd,double ks,int specn,color farb,
                       texfeld colarray,boolean texturen,boolean specular)
{

  double an, ib, ie, spia, spie, sp;
  double r_an, g_an, b_an, r_span, g_span, b_span;
  double r_inten, b_inten, g_inten, r_sp, g_sp, b_sp;
  double r_b, g_b, b_b, r_e, g_e, b_e;  
  int    r_specin, b_specin, g_specin;
  int    dx, i, j, k;

  r_sp = g_sp = b_sp = r_inten = g_inten = b_inten = 0; 
  r_an = g_an = b_an = 0;
  r_span = g_span = b_span = 0;
  r_specin = g_specin = b_specin = 0;
  dx = pend.x - pbeg.x;
  if (dx < 1) dx = 1;
  normalise(&nb);
  normalise(&ne);
  for (i=0; i<anz_lights; i++)
    {
      if ((ib = scalar_p(lights[i].l,nb) * lights[i].i * kd) < 0) ib = 0;
      if ((ie = scalar_p(lights[i].l,ne) * lights[i].i * kd) < 0) ie = 0;
      r_inten += ib * lights[i].r;
      g_inten += ib * lights[i].g;
      b_inten += ib * lights[i].b;
      an = (ie - ib) / dx;
      r_an += an * lights[i].r; 
      g_an += an * lights[i].g; 
      b_an += an * lights[i].b; 
      if (specular) 
        {
          spia = scalar_p(lights[i].h,nb);
          spie = scalar_p(lights[i].h,ne);
          sp = spia; 
          if (sp > 0)
            for (j=2; j<=specn; j++) sp *= spia;
          else 
            sp = 0;
          sp *= lights[i].i * 255 * ks;
          r_sp += r_b = sp * lights[i].r;
          g_sp += g_b = sp * lights[i].g;
          b_sp += b_b = sp * lights[i].b;

          sp = spie; 
          if (sp > 0)
            for (j=2; j<=specn; j++) sp *= spie;
          else 
            sp = 0;
          sp *= lights[i].i * 255 * ks;
          r_e = sp * lights[i].r;
          g_e = sp * lights[i].g;
          b_e = sp * lights[i].b;
           
          an = (r_e - r_b) / dx;
          r_span += an;

          an = (g_e - g_b) / dx;
          g_span += an;

          an = (b_e - b_b) / dx;
          b_span += an;

        };
    };
  k = 0;   
  i = pbeg.x;
  j = pend.x;
  while (i <= j) 
    {
      if (specular) 
        {
          r_specin = (int)floor(r_sp);
          g_specin = (int)floor(g_sp);
          b_specin = (int)floor(b_sp);
        };
      if (r_inten < 0) r_inten = 0;
      if (g_inten < 0) g_inten = 0;
      if (b_inten < 0) b_inten = 0;
      if (r_specin < 0) r_specin = 0;
      if (g_specin < 0) g_specin = 0;
      if (b_specin < 0) b_specin = 0;
      set_point(i,k, r_inten, g_inten, b_inten, 
                     r_specin, g_specin, b_specin, 
                     texturen, colarray, farb, false);
      if (specular) 
        {
          r_sp += r_span;
          g_sp += g_span;
          b_sp += b_span;
        };
      r_inten += r_an;
      g_inten += g_an;
      b_inten += b_an;
      i++;
      k++;
    };
}




static void phong_sh(pointI pbeg,pointI pend,pointR3 nb,pointR3 ne,
                     double kd,double ks,int specn,color farb,
                     texfeld colarray,boolean texturen,boolean specular,
                     boolean htest)
{
  boolean h_test;
  double sp,spa,t,r_inten,g_inten,b_inten;
  int    r_specin, g_specin, b_specin;
  pointR3 nm,dn;
  int dx, i, j, k, l, m;

  /* t = test-value for H-test (0.8 is a good estimate) */
  t = 0.8;
  dx = pend.x - pbeg.x;
  normalise(&ne);
  normalise(&nb);
  if (dx < 1) 
    {
      dn.x = 0.0;
      dn.y = 0.0;
      dn.z = 0.0;
      dx = 1;
    } 
  else 
    {
      dn.x = (ne.x - nb.x) / dx;
      dn.y = (ne.y - nb.y) / dx;
      dn.z = (ne.z - nb.z) / dx;
    };
  if (htest) 
    {  
      nm.x = nb.x + ne.x;
      nm.y = nb.y + ne.y;
      nm.z = nb.z + ne.z;
      normalise(&nm);
      /* h-test */
      h_test = false;
      for (i=0; i<anz_lights; i++)
        {
          if (scalar_p(lights[i].h,nb) >= t) 
            {
              h_test = true;
              break;
            };
          if (scalar_p(lights[i].h,nm) >= t) 
            {
              h_test = true;
              break;
            };
          if (scalar_p(lights[i].h,ne) >= t) 
            {
              h_test = true;
              break;
            };
        };
      if (! h_test)
        {
         gouraud_sh(pbeg,pend,nb,ne,kd,ks,specn,farb,colarray,texturen,true);
         return;
        };
    };
  m = 0;
  i = pbeg.x;
  j = pend.x;
  while (i <= j) 
    {
      r_inten = 0;
      g_inten = 0;
      b_inten = 0;
      r_specin = 0; 
      g_specin = 0; 
      b_specin = 0; 
      for (k=0; k<anz_lights; k++)
        {
          if ((sp = scalar_p(lights[k].l,nb) * lights[k].i) < 0) sp = 0;
          r_inten += sp * lights[k].r;
          g_inten += sp * lights[k].g;
          b_inten += sp * lights[k].b;
          if (specular)
            { 
              spa = scalar_p(lights[k].h,nb);
              sp = spa; 
              if (sp > 0)
                for (l=2; l<=specn; l++) sp *= spa;
              else 
                sp = 0;
              sp *= lights[k].i * 255;
              r_specin += (int)floor(sp * lights[k].r);
              g_specin += (int)floor(sp * lights[k].g);
              b_specin += (int)floor(sp * lights[k].b);
            };
        };
      r_inten *= kd;
      g_inten *= kd;
      b_inten *= kd; 
      if (specular)
        {
          r_specin = (int)floor(r_specin * ks);
          g_specin = (int)floor(g_specin * ks);
          b_specin = (int)floor(b_specin * ks);
        }
      else
        {
          r_specin = 0;
          b_specin = 0;
          g_specin = 0;   
        };  
      set_point(i,m, r_inten, g_inten, b_inten, 
                     r_specin, g_specin, b_specin, 
                     texturen, colarray, farb, false);
      /* build next normal*/
      nb.x += dn.x;
      nb.y += dn.y;
      nb.z += dn.z;
      normalise(&nb);
      i++;
      m++;
    };
} 



static void output_spans(int ay,spanne *listsp,schattierung shading,
                         boolean specular,boolean texturen,boolean htest)
{
  spanne  *hilf;
  pointR3 nb, ne;
  color   farb;
  int     specn;
  pointI  pbeg, pend, phelp;
  texfeld colarray;
  textur  *tex;
  int     i,j,x;
  double  kd,ks;
  double  dumd = 0;
  int     dumi = 0;
  boolean got_tex_points;
  boolean real_tex;


  hilf = listsp;
  last_x = -1;
  if (hilf != NULL) 
      /* fill the beginning of the scanline with background-colour */
      if ((hilf->xbeg > 0) && (hilf->xbeg < x_dim))
        { 
         for (x=0; x < hilf->xbeg; x++) 
           {
            setPixel(back);
           }
         last_x=hilf->xbeg - 1;
        };
  while (hilf != NULL) 
    {
      pbeg.x = hilf->xbeg;
      pbeg.y = ay;
      pbeg.z = hilf->zbeg;
      pend.x = hilf->xend;
      pend.y = ay;
      pend.z = hilf->zend; 
      /* get the surface-attributs of the current polygon */
      kd     = hilf->att->kd; 
      ks     = hilf->att->ks;  
      farb   = hilf->col;
      specn  = hilf->att->pol;
      tex    = hilf->att->tex;
      /* do the texturing */
      nb = hilf->norbeg;
      ne = hilf->norend;
      if (texturen && (tex != NULL))
        {
          got_tex_points = false;
          if (hilf->next != NULL)
            if ((hilf->att == hilf->next->att) &&
                (pend.x == hilf->next->xbeg) &&
                (pend.z == hilf->next->zbeg))
                {
                  phelp.x=hilf->next->xend;
                  phelp.z=hilf->next->zend;
                  texturiere(nb,ne,pbeg,pend,phelp,1,tex,colarray);
                  got_tex_points = true;
                };
          if ((hilf->prev != NULL) && !got_tex_points)
            if ((hilf->att == hilf->prev->att) &&
                (pbeg.x == hilf->prev->xend) &&
                (pbeg.z == hilf->prev->zend))
                {
                  phelp.x=hilf->prev->xbeg;
                  phelp.z=hilf->prev->zbeg;
                  texturiere(nb,ne,phelp,pbeg,pend,2,tex,colarray);
                  got_tex_points = true;
                };
          if (!got_tex_points)
            texturiere(nb,ne,pbeg,pbeg,pend,2,tex,colarray);
          real_tex = true;
        }
      else 
        real_tex = false;
      switch (shading) 
        {
         case noshade:
           i = pbeg.x;
           j = pend.x;
           while (i <= j)
            {        
             set_point(i,i,dumd,dumd,dumd,dumi,dumi,dumi,false,colarray,farb,true);
             i++;
            };
           break;
         case konst:
           if (texturen)
             konst_sh(pbeg,pend,nb,kd,ks,specn,farb,colarray,real_tex,specular);
           else
             {
               i = pbeg.x;
               j = pend.x;
               while (i <= j)
                 {        
                   set_point(i,i,dumd,dumd,dumd,dumi,dumi,dumi,false,colarray,farb,true);
                   i++;
                 };
             };
           break;
         case randomsh:   
         case modif2:
           modif2_sh(pbeg,pend,nb,ne,kd,ks,specn,farb,colarray,real_tex,specular);
           break;
         case gour:
           gouraud_sh(pbeg,pend,nb,ne,kd,ks,specn,farb,colarray,real_tex,specular);
           break;
         case phong:
           if (htest) specular=true;
           phong_sh(pbeg,pend,nb,ne,kd,ks,specn,farb,colarray,real_tex,specular,htest);
           break;
        };
      hilf = hilf->next;
    }  /* while */
  /* fill the remainder of the scanline with background-colour */
  if (last_x < x_dim) 
    for (x=(last_x + 1); x < x_dim; x++) setPixel(back);
}


/**************************************************************************/
/*         die Prozeduren zur Verwaltung der Liste der Spannen            */
/**************************************************************************/

static spanne * new_spans(void)
{
  return(NULL);
}


static spanne * clear_spans(spanne *l)
{
  spanne *h;

  if (l == NULL) return(NULL);
  h = l;
  while (h->next != NULL) 
    {
      h = h->next;
      free(h->prev);
    }
  free(h);
  return(NULL);
}


static void del_span(spanne *weg,spanne **listsp)
{
  spanne *hi;

  hi = weg;
  if (weg == *listsp) 
    {
      *listsp = (*listsp)->next;
      if (*listsp != NULL) (*listsp)->prev = NULL;
    } 
  else 
    {
      if (weg->next == NULL) 
        weg->prev->next = NULL;
      else 
        {
          weg->prev->next = weg->next;
          weg->next->prev = weg->prev;
        };
    };
  free(hi);
}


static void add_span(spanne *ein,spanne **listsp)
{
  spanne *hilf;

  ein->next = ein->prev = NULL;
  hilf = *listsp;
  if (hilf == NULL) 
    {
       *listsp = ein;
       return;
    };
  while (hilf->next != NULL && hilf->xbeg <= ein->xbeg)
    hilf = hilf->next;
  if (hilf->next == NULL && hilf->xbeg <= ein->xbeg) 
    {
      hilf->next = ein;
      ein->prev = hilf;
      return;
    };
  ein->next = hilf;
  if (hilf == *listsp)
    {
      hilf->prev = ein;
      *listsp = ein;
      return;
    };
  ein->prev = hilf->prev;
  hilf->prev->next = ein;
  hilf->prev = ein;
}



/**************************************************************************/
/*       die Prozeduren zur Verwaltung der Kantenlisten der Polygone      */
/**************************************************************************/

/* alle Kanten der Liste l loeschen */
static void del_all_edges(edge **l)
{
  edge *h, *h1;

  if (*l == NULL) return;
  h = *l;
  while (h != NULL) 
    {
      h1 = h;
      h = h->next;
      free(h1);
    };
  *l = NULL;
}


/* Einsortierung von e in die Liste l --> absteigend in y */
static void add_edge(edge *e,edge **l)
{
  edge *h;

  e->next = e->prev = NULL;
  h = *l;
  if (h == NULL) 
    {
      /* Kantenliste ist noch leer */
      *l = e;
      return;
    };
  /* Einsortieren der Kante in die Liste (absteigend in y) */
  while ((h->next != NULL) && (h->apkt.y > e->apkt.y)) h = h->next;
  if ((h->next == NULL) && (h->apkt.y > e->apkt.y)) 
    {
      /* Kante ans Ende der Liste anhaengen */
      h->next = e;
      e->prev = h;
      return;
    };
  if (h == *l) 
    {
      /* Kante an den Anfang der Liste haengen */
      e->next = *l;
      (*l)->prev = e;
      *l = e;
    } 
  else 
    {
      /* Kante 'e' vor Kante 'h' in Liste einsortieren */
      e->next = h;
      e->prev = h->prev;
      h->prev = e;
      e->prev->next = e;
    };
}


/* Listenzeiger akt_edge auf erste Kante setzen */
static void set_first_edge(edge *e)
{
  start_edge = e;
  akt_edge = NULL;
}


/* Zeiger auf aktuelle Kante wird geliefert */
static boolean get_edge(edge **e)
{
  if (akt_edge == NULL) 
    {
      *e = start_edge;
      akt_edge = *e;
      return(*e != NULL);
    };
  *e = akt_edge->next;
  if (*e != NULL) akt_edge = *e;
  return(*e != NULL);
}


/* aktuelle Kante wird geloescht */
static void del_edge(Polygon *p)
{
  edge *h;

  h = akt_edge;
  if (h == NULL) return;
  if (h == start_edge) 
    {
      /* erste Kante loeschen */
      start_edge = h->next;
      p->k = start_edge;
      akt_edge = NULL;
      /* get_edge nimmt beim naechsten Aufruf die erste Kante */
      if (start_edge != NULL) start_edge->prev = NULL;
      free(h);
      return;
    };
  if (h->next == NULL) 
    {
      /* letzte Kante loeschen */
      akt_edge = h->prev;
      if (akt_edge != NULL) akt_edge->next = NULL;
      free(h);
      return;
    } 
  else 
    {
      /* Kante aus der Mitte loeschen */
      akt_edge = akt_edge->prev;   /* get_edge() nimmt die naechste Kante */
      akt_edge->next = h->next;
      akt_edge->next->prev = akt_edge;
      free(h);
    };
}


/**************************************************************************/
/*          die Prozeduren zur Verwaltung der Polygonlisten               */
/**************************************************************************/

static void new_List(list_Poly *l)
{
  l->list = NULL;
  l->pointer = NULL;
}


static void set_first_Poly(list_Poly *l)
{
  l->pointer = l->list;
}


static boolean last_Poly(list_Poly l)
{
  if (l.list != NULL)
    return (l.pointer->next == NULL);
  else
    return true;
}


static boolean set_next_Poly(list_Poly *l)
{
  if (l->list != NULL) 
    {
      if (l->pointer->next == NULL)
        return false;
      else 
        {
          l->pointer = l->pointer->next;
          return true;
        }
    } 
  else
    return false;
}


static Polygon *get_Poly(list_Poly l)
{
  return (l.pointer);
}


static boolean Poly_aktiv(list_Poly l,int y)
{
  if (l.pointer != NULL)
    return ((l.pointer->ymin <= y) && (l.pointer->ymax >= y));
  else
    return false;
}


static boolean Poly_unused(list_Poly l,int y)
{
  if (l.pointer != NULL)
    return ((l.pointer->ymin > y) && (l.pointer->ymax > y));
  else
    return false;
}


static void del_Poly(list_Poly *l)
{
  Polygon *h;

  if (l->pointer == NULL) return;
  h = l->pointer;
  if (l->pointer == l->list) 
    {
      /* Polygon am Anfang der Liste */
      l->pointer = l->pointer->next;
      l->list = l->pointer;
      if (l->pointer != NULL) l->pointer->prev = NULL;
    } 
  else if (l->pointer->next == NULL) 
    {
      /* Polygon am Ende der Liste */
      l->pointer = l->pointer->prev;
      l->pointer->next = NULL;
    } 
  else 
    {
      /* Polygon innerhalb der Liste */
      l->pointer->next->prev = l->pointer->prev;
      l->pointer->prev->next = l->pointer->next;
      l->pointer = l->pointer->next;
    };
  del_all_edges(&h->k);
  free(h);
}


static void add_Poly(Polygon *p,list_Poly *l)
{
  Polygon *h;

  p->next = p->prev = NULL;
  if (l->list == NULL) 
    {
      l->list = p;
      l->pointer = p;
      return;
    };
  h = l->list;
  while (h->ymax > p->ymax && h->next != NULL) h = h->next;
  if (h->next == NULL && h->ymax >= p->ymax) 
    {
      /* Polygon ans Ende der Liste haengen */
      h->next = p;
      p->prev = h;
      p->next = NULL;
      return;
    };
  /* Polygon 'p' vor 'h' einsortieren */
  if (h == l->list) 
    {
      /* Polygon an den Anfang der Liste haengen */
      h->prev = p;
      p->next = h;
      p->prev = NULL;
      l->list = p;
      return;
    };
  /* Polygon in die Liste einsortieren */
  p->next = h;
  p->prev = h->prev;
  h->prev = p;
  p->prev->next = p;
}


static void move_Poly(list_Poly *l1,list_Poly *l2)
{
  Polygon *h;

  /* Polygon aus Liste l1 entfernen */
  if (l1->pointer == NULL) return;
  h = l1->pointer;
  if (l1->pointer == l1->list) 
    {
      /* Polygon am Anfang der Liste */
      l1->pointer = l1->pointer->next;
      l1->list = l1->pointer;
      if (l1->pointer != NULL) l1->pointer->prev = NULL;
    } 
  else if (l1->pointer->next == NULL) 
    {
      /* Polygon am Ende der Liste */
      l1->pointer = l1->pointer->prev;
      l1->pointer->next = NULL;
    } 
  else 
    {
      /* Polygon innerhalb der Liste */
      l1->pointer->next->prev = l1->pointer->prev;
      l1->pointer->prev->next = l1->pointer->next;
      l1->pointer = l1->pointer->next;
    };
  /* Polygon in Liste l2 einfuegen */
  add_Poly(h, l2);
}


#define maxint          MAXINT


static void read_NFF_back_face_culling(boolean inverse,
				       boolean backfaceculling, 
                                       schattierung shading, 
                                       boolean specular, 
                                       boolean texturen, 
                                       int *maxy, 
                                       int *miny, 
                                       list_Poly *lp)
{
  pointR3 proj,gn;
  pointR4 pR4_1,pR4_2;
  double s,d,dy,h2,sx,sy,vx,vy,winkel,kd,ks;
  Polygon *p;
  edge *se;
  boolean backface,parallel;
  pointR3 pkt[160],nor[160];
  char   line[128],line1[128];
  char   dum,test,test1;
  int    anz,i,j,k;
  long   primi,got;
  color  fa;
  pointR3 p1,p2,p3,p4,p5,up;
  MATRIX H,M,P,T;
  attrib *last_att,*att;
  textur *tex; 

  *maxy = 0;
  *miny = maxint;

  primi=0;
  got=0; 
  anz_lights=0;
  attr_list=last_att=NULL;
  if (comments) printf("\nreading NFF-file : %s",infile);
  while (!feof(A))
    { 
      fscanf(A, "%[^\n]s", line);
      if (!feof(A)) fgetc(A);      /*  CR entfernen */
      sscanf(line, "%c", &test);
      switch (test)
        {
          case '#' : /* comment-line               */
	     if (comments) printf("\n comment-line : %s",line);
          break;
          case 'v' : /* viewing vectors and angles */
	     if (comments) printf("\n viewing vectors and angles");
	     /* Berechnung der Transformationsmatrix T fuer WC-->DC */
	     fscanf(A, "from %lg %lg %lg%*[^\n]", &p1.x, &p1.y, &p1.z);
             if (!feof(A)) fgetc(A);
	     if (comments) printf("\n     eye-point: %g %g %g",p1.x,p1.y,p1.z);
	     fscanf(A, "at %lg %lg %lg%*[^\n]", &p2.x, &p2.y, &p2.z);
             if (!feof(A)) fgetc(A);
	     if (comments) printf("\n     lock at: %g %g %g",p2.x,p2.y,p2.z);
	     p3.x = p1.x - p2.x;
	     p3.y = p1.y - p2.y;
	     p3.z = p1.z - p2.z;
	     proj.x = p2.x - p1.x;
	     proj.y = p2.y - p1.y;
	     proj.z = p2.z - p1.z;
             normalise(&proj);
	     d = betrag(p3);
	     normalise(&p3);
	     if (comments) printf("\n     direction: %g %g %g", p3.x, p3.y, p3.z);
	     fscanf(A, "up %lg %lg %lg%*[^\n]", &up.x, &up.y, &up.z);
             normalise(&up);
             if (!feof(A)) fgetc(A);
	     if (comments) printf("\n     lock-up vector: %g %g %g",up.x,up.y,up.z);
	     fscanf(A, "angle %lg%*[^\n]", &winkel);
             if (!feof(A)) fgetc(A);
	     if (comments) printf("\n     angle: %g ",winkel);
	     fscanf(A, "%[^\n]s", line1);                   /* ignored    */
             if (!feof(A)) fgetc(A);
	     fscanf(A, "resolution %d %d%*[^\n]", &x_dim, &y_dim);
	     if (!feof(A)) fgetc(A);
	     x_dim /=10;
	     x_dim *=10;
	     y_dim /=10;
	     y_dim *=10;
	     if (comments) printf("\n     resolution: %d %d ",x_dim,y_dim);
             /* Fehlerbehandlung, wenn p3 == up */
             if (fabs(scalar_p(p3,up)) == 1)
               {
                 printf("\n ERROR: view-up-vector and projektion-vektor are equal!\n");
               };
	     /* Drehung & Translation und Wandlung Rechtssystem --> Linkssystem */
	     vectorproduct(up,p3,&p4);
	     normalise(&p4);
	     vectorproduct(p3,p4,&p5);
	     normalise(&p5);
             /* Verschiebematrix p2 (lock at) --> (0,0,0);(neuer Ursprung) */
	     P[0][0] = 1;     P[0][1] = 0;     P[0][2] = 0;     P[0][3] = 0;
	     P[1][0] = 0;     P[1][1] = 1;     P[1][2] = 0;     P[1][3] = 0;
	     P[2][0] = 0;     P[2][1] = 0;     P[2][2] = 1;     P[2][3] = 0;
	     P[3][0] = -p2.x; P[3][1] = -p2.y; P[3][2] = -p2.z; P[3][3] = 1;

             /* Rotationsmatrix (zu neuem Koordinatensystem) */
	     M[0][0] = p4.x;  M[0][1] = p5.x;  M[0][2] = p3.x;  M[0][3] = 0;
	     M[1][0] = p4.y;  M[1][1] = p5.y;  M[1][2] = p3.y;  M[1][3] = 0;
	     M[2][0] = p4.z;  M[2][1] = p5.z;  M[2][2] = p3.z;  M[2][3] = 0;
	     M[3][0] = 0;     M[3][1] = 0;     M[3][2] = 0;     M[3][3] = 1;
	     matrix_multiply(P,M,T);

             /* Verschiebematrix (Augpunkt --> Ursprung) */
	     P[0][0] = 1;     P[0][1] = 0;     P[0][2] = 0;     P[0][3] = 0;
	     P[1][0] = 0;     P[1][1] = 1;     P[1][2] = 0;     P[1][3] = 0;
	     P[2][0] = 0;     P[2][1] = 0;     P[2][2] = 1;     P[2][3] = 0;
	     P[3][0] = 0;     P[3][1] = 0;     P[3][2] = -d;    P[3][3] = 1;
	     matrix_multiply(T,P,M);

             /* rechts-System --> links-System */
	     P[0][0] = 1;     P[0][1] = 0;     P[0][2] = 0;     P[0][3] = 0;
	     P[1][0] = 0;     P[1][1] = 1;     P[1][2] = 0;     P[1][3] = 0;
	     P[2][0] = 0;     P[2][1] = 0;     P[2][2] = -1;    P[2][3] = 0;
	     P[3][0] = 0;     P[3][1] = 0;     P[3][2] = 0;     P[3][3] = 1;
	     matrix_multiply(M,P,H);
	     /* Uebergang zu NDC */
             winkel /= 2;
	     s = d * tan(BOGEN(winkel));
	     /* perspektivische Transformation ^Newman/Sproull S. 358 */
	     P[0][0] = 1;     P[0][1] = 0;     P[0][2] = 0;     P[0][3] = 0;
	     P[1][0] = 0;     P[1][1] = 1;     P[1][2] = 0;     P[1][3] = 0;
	     P[2][0] = 0;     P[2][1] = 0;     P[2][2] = 2*s/d; P[2][3] = s/d;
	     P[3][0] = 0;     P[3][1] = 0;     P[3][2] = (-2)*s;P[3][3] = 0;
	     /* Zusammensetzen der Transformationsmatrix T=M*P */
	     matrix_multiply(H,P,T);
	     /* Parameter fuer Uebergang zu Geraetekoordinaten */
	     sx = (double)(x_dim / 2);
	     sy = (double)(y_dim / 2);
	     vx = sx;
	     vy = sy;
	  break;
	  case 'l' : /* positional light location  */
             anz = sscanf(line, "l %lg %lg %lg %lg %lg %lg %lg%*[^\n]", &p4.x, &p4.y, &p4.z, &lights[anz_lights].i,
                                 &lights[anz_lights].r, &lights[anz_lights].g, &lights[anz_lights].b);
             if (anz < 4) lights[anz_lights].i = 1.0;
             if (anz < 7)
               {  
                lights[anz_lights].r = 1.0; 
                lights[anz_lights].g = 1.0; 
                lights[anz_lights].b = 1.0; 
               };
	     /* Vektor in Richtung Lichtquelle bilden --> paralleles Licht */
	     p4.x -= p2.x;
	     p4.y -= p2.y;
	     p4.z -= p2.z;
	     lights[anz_lights].l = p4;
	     normalise(&lights[anz_lights].l);
	     lights[anz_lights].h.x = lights[anz_lights].l.x + p3.x;
	     lights[anz_lights].h.y = lights[anz_lights].l.y + p3.y;
	     lights[anz_lights].h.z = lights[anz_lights].l.z + p3.z;
	     normalise(&lights[anz_lights].h);
             anz_lights++;
	     if (comments) printf("\n positional light location");
          break;
          case 'b' : /* background color           */
             if (comments) printf("\n background color");
	     sscanf(line, "b %lg %lg %lg%*[^\n]", &p4.x, &p4.y, &p4.z);
	     back.r = floor(p4.x * 255);
	     back.g = floor(p4.y * 255);
	     back.b = floor(p4.z * 255);
          break;
          case 't' : /* texture properties */
             if (comments) printf("\n reading a texture-description");
             fscanf(A, "%[^\n]s", line);
             sscanf(line, "%c", &test1);
             if (!feof(A)) fgetc(A);              /*  CR entfernen */
             while ((!feof(A)) && (test1 != 'e'))
               { 
                 switch (test1)
                 {
                   case 'w':  /* a wooden texture was spezified */
                     if ((tex = (textur *)malloc(sizeof(textur))) == NULL) exit(1);
                     sscanf(line,"w %lg %lg %lg %d %d",
                                  &tex->p.x,&tex->p.y,&tex->p.z,
                                  &tex->i2,&tex->i1);
                     /* transform midpoint */
                     pointR3_to_R4(tex->p,pR4_1);
                     transform_point(pR4_1,T,&pR4_2);
                     pointR4_to_R3(pR4_2,tex->p);
                     tex->p.x = tex->p.x * vx + sx;
                     tex->p.y = tex->p.y * vy + sy;
                     tex->p.z = (tex->p.z +1) * vx;
                     for (k=0; k<tex->i1; k++)
                       {
                          /* read wooden colours */
	                  fscanf(A, "%lg %lg %lg%*[^\n]",&p4.x,&p4.y,&p4.z); 
                          tex->cfeld[k].r = (int)floor(p4.x * 255),
                          tex->cfeld[k].g = (int)floor(p4.y * 255);
                          tex->cfeld[k].b = (int)floor(p4.z * 255);
                          if (!feof(A)) fgetc(A);
                       };
                     last_att->tex = tex;
                   break;
                   case 'b':;  /* a bumb-mapping was spezified */
                   case 'p':;  /* a 2D-pattern-mapping was spezified */
                 };
                 fscanf(A, "%[^\n]s", line);
                 if (!feof(A)) fgetc(A);          /*  CR entfernen */
                 sscanf(line, "%c", &test1);
               };
          break;   
          case 'f' : /* object material properties */
             if (comments) printf("\n object material properties");
	     sscanf(line, "f %lg %lg %lg %lg %lg %lg%*[^\n]",&p4.x,&p4.y,&p4.z,&kd,&ks,&h2);
	     if ((att = (attrib *)malloc(sizeof(attrib))) == NULL) exit(1);
	     fa.r = (int)floor(p4.x * 255);
	     fa.g = (int)floor(p4.y * 255);
	     fa.b = (int)floor(p4.z * 255); 
	     att->pol  = floor(h2);
             att->kd   = kd;
             att->ks   = ks;
             att->tex  = NULL;
             att->next = NULL; 
             if (last_att == NULL) 
               {
                 attr_list = att;
                 last_att = att;
               }
             else
               {
                 last_att->next = att;
                 last_att = att; 
               };  
          break;
          case 'c' : /* cone or cylinder primitive */
             primi++;
	     for (i=0; i<2; i++)
	       {
		 fscanf(A, "%[^\n]s",line);
		 if (!feof(A)) fgetc(A);
	       };
          break;
          case 's' : /* sphere primitive           */
	     primi++;
          break;
          case 'p' : /* ??? */
          primi++;
          test1=' ';
          sscanf(line, "%c%c", &dum, &test1);
          switch (test1)
            {
              case ' ' : /* polygon primitive              */
		    sscanf(line, "p %d", &anz);
		    for (i=0; i<anz; i++)
		      {
	                fscanf(A, "%lg %lg %lg%*[^\n]", 
                               &(pkt[i].x),&(pkt[i].y),&(pkt[i].z));
                        if (!feof(A)) fgetc(A);
		      };
                    p4 = pkt[1];
                    p5 = pkt[2];
                    p4.x -= pkt[0].x;
                    p4.y -= pkt[0].y;
                    p4.z -= pkt[0].z;
                    p5.x -= pkt[1].x;
                    p5.y -= pkt[1].y;
                    p5.z -= pkt[1].z;
                    vectorproduct(p5,p4,&gn);
		    if (inverse)
                      {
                        gn.x = 0 - gn.x;
                        gn.y = 0 - gn.y;
                        gn.z = 0 - gn.z;
                      };
                    if ((gn.x == 0) && (gn.y == 0) && (gn.z == 0))
                      {
                        if (comments) printf("\n warning: bad polygon, ignored");
                        break;   
                      }; 
                    normalise(&gn);
		    for (i=0; i<anz; i++) nor[i] = gn;
              case 'p' : /* polygonal patch primitive      */
                    if (test1 == 'p')
                      {
		        sscanf(line, "pp %d", &anz);
		        for (i=0; i<anz; i++)
		          {
	                    fscanf(A, "%lg %lg %lg %lg %lg %lg%*[^\n]", 
                                   &(pkt[i].x),&(pkt[i].y),&(pkt[i].z),
                                   &(nor[i].x),&(nor[i].y),&(nor[i].z));
                            if (!feof(A)) fgetc(A);
                            if (inverse) 
                              {
                                nor[i].x = 0 - nor[i].x;
                                nor[i].y = 0 - nor[i].y;
                                nor[i].z = 0 - nor[i].z;
                              };
		          };
                        p4 = pkt[1];
                        p5 = pkt[2];
                        p4.x -= pkt[0].x;
                        p4.y -= pkt[0].y;
                        p4.z -= pkt[0].z;
                        p5.x -= pkt[1].x;
                        p5.y -= pkt[1].y;
                        p5.z -= pkt[1].z;
                        vectorproduct(p5,p4,&gn);
		        if (inverse)
                          {
                            gn.x = 0 - gn.x;
                            gn.y = 0 - gn.y;
                            gn.z = 0 - gn.z;
                          };
                        if ((gn.x == 0) && (gn.y == 0) && (gn.z == 0))
                          {
                            if (comments) printf("\n warning: bad polygon, ignored");
                            break;   
                          }; 
                        normalise(&gn);
                      };
                    /* transforming vertices */
                    for (i=0; i<anz; i++)
                      {
                        pointR3_to_R4(pkt[i],pR4_1);
                        transform_point(pR4_1,T,&pR4_2);
                        pointR4_to_R3(pR4_2,pkt[i]);
                        pkt[i].x = pkt[i].x * vx + sx;
                        pkt[i].y = pkt[i].y * vy + sy;
                        pkt[i].z = (pkt[i].z +1) * vx;
                      };
                    backface = false;
                    /* Datenstruktur des Polygons erzeugen */
                    p = (Polygon *)malloc(sizeof(Polygon));
                    if (p == NULL) exit(1);
                    p->k = NULL;   /* sortierte Kantenliste des Polygons */
                    p->ymax = (int)floor(pkt[0].y + 0.5); /* maximale y-Koordinate des Polygons */
                    p->ymin = (int)floor(pkt[0].y + 0.5); /* minimale y-Koordinate des Polygons */
                    /* bei shading == konst --> hier Schattierung durchfuehren */
                    if ((shading == konst) && (!texturen))
                      {
                        double r_inten, b_inten, g_inten, sp, spa; 
                        int r_specin, b_specin, g_specin;
                        int i, j;

                        r_inten = 0;
                        g_inten = 0;
                        b_inten = 0;
                        r_specin = 0; 
                        g_specin = 0; 
                        b_specin = 0; 
                        for (i=0; i<anz_lights; i++)
                          {
                            if ((sp = scalar_p(lights[i].l,gn) * lights[i].i) < 0) sp = 0;
                            r_inten += sp * lights[i].r;
                            g_inten += sp * lights[i].g;
                            b_inten += sp * lights[i].b;
                            if (specular)
                              { 
                                spa = scalar_p(lights[i].h,gn);
                                sp = spa; 
                                if (sp > 0)
                                  for (j=2; j<=last_att->pol; j++) sp *= spa;
                                else 
                                  sp = 0;
                                sp *= lights[i].i * 255;
                                r_specin += (int)floor(sp * lights[i].r);
                                g_specin += (int)floor(sp * lights[i].g);
                                b_specin += (int)floor(sp * lights[i].b);
                              };
                          };
                        r_inten *= kd;
                        g_inten *= kd;
                        b_inten *= kd; 
                        if (specular)
                          {
                            p->col.r = (int)floor(fa.r * r_inten + r_specin * ks);
                            p->col.g = (int)floor(fa.g * g_inten + g_specin * ks);
                            p->col.b = (int)floor(fa.b * b_inten + b_specin * ks);
                          }
                        else
                          {
                            p->col.r = (int)floor(fa.r * r_inten); 
                            p->col.g = (int)floor(fa.g * g_inten); 
                            p->col.b = (int)floor(fa.b * b_inten); 
                          };  
                      }
                    else      
                      p->col = fa;
                    if (shading == randomsh)
                      {
                        p->col.r = rand() % 255;
                        p->col.g = rand() % 255;
                        p->col.b = rand() % 255;  
                      };
                    p->att = last_att;
                    /* Kanten des Polygons erzeugen und einsortieren */
                    i = 1;
                    while (i < anz && !backface)
                      {
	                if (backfaceculling)
                        if (scalar_p(proj, gn) > 0)
	                  {
	                    backface = true;
	                    break;
	                  };
	                if (p->ymax < pkt[i - 1].y) p->ymax = (int)floor(pkt[i - 1].y + 0.5);
	                if (p->ymin > pkt[i - 1].y) p->ymin = (int)floor(pkt[i - 1].y + 0.5);
	                se = (edge *)malloc(sizeof(edge));
                        if (se == NULL) exit(1);
	                if (pkt[i - 1].y > pkt[i].y ||
	                    (pkt[i - 1].y == pkt[i].y && pkt[i - 1].x < pkt[i].x))
	                  {
	                    se->ax = se->apkt.x = (int)floor(pkt[i - 1].x + 0.5);
	                    se->az = se->apkt.z = (int)floor(pkt[i - 1].z + 0.5);
	                    se->ay = se->apkt.y = (int)floor(pkt[i - 1].y + 0.5);
	                    se->epkt.x = (int)floor(pkt[i].x + 0.5);
	                    se->epkt.y = (int)floor(pkt[i].y + 0.5);
	                    se->epkt.z = (int)floor(pkt[i].z + 0.5);
	                    se->anor = nor[i-1];
	                    se->enor = nor[i];
	                    if (i != 1)
		                se->Nay = (int)floor(pkt[i - 2].y + 0.5); 
	                    else
		                se->Nay = (int)floor(pkt[anz - 1].y + 0.5);
	                    if (se->apkt.y == se->epkt.y)
	                      parallel = true;
	                    else
                              {
                                parallel = false;
	                        dy = (double)se->apkt.y - (double)se->epkt.y;
	                        se->mx = ((double)se->apkt.x - (double)se->epkt.x) / dy;
	                        se->mz = ((double)se->apkt.z - (double)se->epkt.z) / dy;
	                        se->mv.x = (se->anor.x - se->enor.x) / dy;
	                        se->mv.y = (se->anor.y - se->enor.y) / dy;
	                        se->mv.z = (se->anor.z - se->enor.z) / dy;
                              };
	                  }
                        else
                          { 
	                    se->ax = se->apkt.x = (int)floor(pkt[i].x + 0.5);
	                    se->az = se->apkt.z = (int)floor(pkt[i].z + 0.5);
	                    se->ay = se->apkt.y = (int)floor(pkt[i].y + 0.5);
	                    se->epkt.x = (int)floor(pkt[i - 1].x + 0.5);
	                    se->epkt.y = (int)floor(pkt[i - 1].y + 0.5);
	                    se->epkt.z = (int)floor(pkt[i - 1].z + 0.5);
	                    se->anor = nor[i];
	                    se->enor = nor[i - 1];
	                    if (i != 1)
	                      {
		                if (i < anz - 1)
		                   se->Nay = (int)floor(pkt[i + 1].y + 0.5);
		                else
		                   se->Nay = (int)floor(pkt[0].y + 0.5);
	                      }
	                    else
		              se->Nay = (int)floor(pkt[2].y + 0.5);
	                    if (se->apkt.y == se->epkt.y)
	                      parallel = true;
	                    else
                              {
                                parallel = false;
	                        dy = (double)se->apkt.y - (double)se->epkt.y;
	                        se->mx = ((double)se->apkt.x - (double)se->epkt.x) / dy;
	                        se->mz = ((double)se->apkt.z - (double)se->epkt.z) / dy;
	                        se->mv.x = (se->anor.x - se->enor.x) / dy;
	                        se->mv.y = (se->anor.y - se->enor.y) / dy;
	                        se->mv.z = (se->anor.z - se->enor.z) / dy;
                              };
	                  };
	                if (parallel)
                          free(se);
                        else
                          add_edge(se, &(p->k));
	                i++;
                      };
                    se = (edge *)malloc(sizeof(edge));
                    if (se == NULL) exit(1);
                    if (pkt[anz - 1].y > pkt[0].y ||
	                (pkt[anz - 1].y == pkt[0].y && pkt[anz - 1].x < pkt[0].x))
                      {
	                se->ax = se->apkt.x = (int)floor(pkt[anz - 1].x + 0.5);
	                se->az = se->apkt.z = (int)floor(pkt[anz - 1].z + 0.5);
	                se->ay = se->apkt.y = (int)floor(pkt[anz - 1].y + 0.5);
	                se->epkt.x = (int)floor(pkt[0].x + 0.5);
	                se->epkt.y = (int)floor(pkt[0].y + 0.5);
	                se->epkt.z = (int)floor(pkt[0].z + 0.5);
	                se->anor = nor[anz - 1];
	                se->enor = nor[0];
	                se->Nay = (int)floor(pkt[anz - 2].y + 0.5);
	                if (se->apkt.y == se->epkt.y)
	                  parallel = true;
	                else
                          {
                            parallel = false;
	                    dy = (double)se->apkt.y - (double)se->epkt.y;
	                    se->mx = ((double)se->apkt.x - (double)se->epkt.x) / dy;
	                    se->mz = ((double)se->apkt.z - (double)se->epkt.z) / dy;
	                    se->mv.x = (se->anor.x - se->enor.x) / dy;
	                    se->mv.y = (se->anor.y - se->enor.y) / dy;
	                    se->mv.z = (se->anor.z - se->enor.z) / dy;
                          };
                      }
                    else
                      {
	                se->ax = se->apkt.x = (int)floor(pkt[0].x + 0.5);
	                se->az = se->apkt.z = (int)floor(pkt[0].z + 0.5);
	                se->ay = se->apkt.y = (int)floor(pkt[0].y + 0.5);
	                se->epkt.x = (int)floor(pkt[anz - 1].x + 0.5);
	                se->epkt.y = (int)floor(pkt[anz - 1].y + 0.5);
	                se->epkt.z = (int)floor(pkt[anz - 1].z + 0.5);
	                se->anor = nor[0];
	                se->enor = nor[anz - 1];
	                se->Nay = (int)floor(pkt[1].y + 0.5);
	                if (se->apkt.y == se->epkt.y)
	                  parallel = true;
	                else
                          {
                            parallel = false;
	                    dy = (double)se->apkt.y - (double)se->epkt.y;
	                    se->mx = ((double)se->apkt.x - (double)se->epkt.x) / dy;
	                    se->mz = ((double)se->apkt.z - (double)se->epkt.z) / dy;
	                    se->mv.x = (se->anor.x - se->enor.x) / dy;
	                    se->mv.y = (se->anor.y - se->enor.y) / dy;
	                    se->mv.z = (se->anor.z - se->enor.z) / dy;
                          }; 
                      };
                    if (parallel)
                       free(se);
                    else
                       add_edge(se, &(p->k));
                    if (p->ymax < pkt[anz - 1].y) p->ymax = (int)floor(pkt[anz - 1].y + 0.5);
                    if (p->ymin > pkt[anz - 1].y) p->ymin = (int)floor(pkt[anz - 1].y + 0.5);
                    /* Polygon einsortieren, wenn keine Rueckflaeche */
                    if ((! backfaceculling) || (!backface))
                      { 
                        p->v = gn;
                        if (*maxy < p->ymax) *maxy = p->ymax;
                        if (*miny > p->ymin) *miny = p->ymin;
                        add_Poly(p, lp);
                        got++;
                      }
                    else
                      {
	                /* sinnlose Datenstrukturen entfernen */
	                del_all_edges(&p->k);
	                free(p);
                      };
              break;
	      case 'b' : /* bicubic Bezier patch primitive */
                    for (i=0; i<4; i++)
                      for (j=0; j<4; j++)
                        {
			  fscanf(A,"%[^\n]s",line);
                          if (!feof(A)) fgetc(A);
			};
		    primi++;
              break;
            };
          break;
        };
    };
  if (comments) printf("\n\n%ld primitives entered 'scanline', but only %ld left\n", primi, got);
}

#undef maxint



/**************************************************************************/
/*             die Prozeduren des Scanline-Algorithmus                    */
/**************************************************************************/


static void einsort(pointI a,pointR3 norm,zwischenpuffer *puffer)
{
  int l, i, m, n;

  if (puffer->anz == 0) 
    {
      puffer->vektor[0] = a;
      puffer->normale[0] = norm;
      puffer->anz = 1;
      return;
    };
  l = puffer->anz;
  n = puffer->anz / 2;
  if (n == 0) n = 1;
  m = n;
  do 
    {
      if (a.x > puffer->vektor[m - 1].x) 
        {
          if (puffer->anz == 1) 
            {
	      puffer->anz = 2;
	      puffer->vektor[1] = a;
	      puffer->normale[1] = norm;
            } 
          else 
            {
	      if (a.x <= puffer->vektor[m].x) 
                {
	          for (i = puffer->anz; i > m; i--) 
                    {
	              puffer->vektor[i] = puffer->vektor[i - 1];
	              puffer->normale[i] = puffer->normale[i - 1];
	            };
	          puffer->vektor[m] = a;
	          puffer->normale[m] = norm;
	          puffer->anz++;
	        } 
              else 
                {
	          if (n > 1) n /= 2;
	          m += n;
	          if (m > puffer->anz) 
                    {
	              puffer->anz++;
	              puffer->vektor[puffer->anz - 1] = a;
	              puffer->normale[puffer->anz - 1] = norm;
	            };
	        };
            };
        } 
      else 
        {
          if (n > 1) n /= 2;
          m -= n;
          if (m < 1) 
            {
	      for (i = puffer->anz; i >= 1; i--) 
                {
	          puffer->vektor[i] = puffer->vektor[i - 1];
	          puffer->normale[i] = puffer->normale[i - 1];
	        };
	      puffer->anz++;
	      puffer->vektor[0] = a;
	      puffer->normale[0] = norm;
            };
        };
    } 
  while (puffer->anz <= l);
}


static void extrema(edge *e,zwischenpuffer *puffer)
{
/* !!!!!!!!!!!!!!!!!!!!!!!! < oder <= ??????????????????!!!!!!!!!!! */
  if ((e->Nay <= e->apkt.y) && (e->epkt.y <= e->apkt.y))   
    einsort(e->apkt, e->anor, puffer);
}


static void common(edge *e,int akty,zwischenpuffer *puffer)
{
  pointI  p;
  pointR3 nrm;

  if (akty == e->epkt.y)
    {
      p = e->epkt;
      nrm = e->enor;
    }
  else
    {
      while ((e->ay - 1) > akty)
        {
          /* e->ax und e->az fuer 1. Schnitt bestimmen */
          e->ax -= e->mx;
          e->az -= e->mz;
          e->anor.x -= e->mv.x;
          e->anor.y -= e->mv.y;
          e->anor.z -= e->mv.z;
          (e->ay)--;
        };
      p.y = e->ay = akty;
      p.x = (int)floor((e->ax -= e->mx) + 0.5);
      p.z = (int)floor((e->az -= e->mz) + 0.5);
      nrm.x = e->anor.x -= e->mv.x;
      nrm.y = e->anor.y -= e->mv.y;
      nrm.z = e->anor.z -= e->mv.z;
     };
  normalise(&nrm);
  einsort(p, nrm, puffer);
}



static void make_points(Polygon *p,int akty,zwischenpuffer *puffer)
{
  edge *e;

  /* parallele Kanten koennen hier schon nicht mehr auftreten */
  puffer->anz = 0;
  set_first_edge(p->k);
  while (get_edge(&e))
    {
      /* alle relevanten Kanten untersucht ? */ 
      if (e->apkt.y < akty) break;
      if (e->epkt.y > akty) 
        {
         /* Kante wird nicht mehr benoetigt */
         del_edge(p);
         continue;
        }; 
      if (e->apkt.y == akty)
         extrema(e, puffer);
      else
         common(e, akty, puffer);
      if (e->epkt.y >= akty) del_edge(p); 
    };
/*
if (puffer->anz != 2) printf("\npuffer.anz: %d",puffer->anz); 
*/  
}


#define maxint          MAXINT



static void subdivide_spans(spanne *z1,spanne *z2,spanne **listsp)
{
  int x;
  double tz1, tz2, b1, b2, dx1, dx2, hilf, m1, m2;
  pointR3 dn1, dn2, null;
  spanne *hi, *hi1, *hi2, *neu;
  int aktspanne;

  /* Entfernen von z1 aus listsp --> hi1 */
  hi1 = z1;
  if (z1 == *listsp) 
    {
      *listsp = (*listsp)->next;
      if (*listsp != NULL) (*listsp)->prev = NULL;
    } 
  else 
    {
      if (z1->next == NULL)
        z1->prev->next = NULL;
      else 
        {
         z1->prev->next = z1->next;
         z1->next->prev = z1->prev;
        };
    };

  /* Entfernen von z2 aus listsp --> hi2 */
  hi2 = z2;
  if (z2 == *listsp) 
    {
      *listsp = (*listsp)->next;
      if (*listsp != NULL) (*listsp)->prev = NULL;
    } 
  else 
    {
      if (z2->next == NULL)
        z2->prev->next = NULL;
      else 
        {
          z2->prev->next = z2->next;
          z2->next->prev = z2->prev;
        };
    };

  /* Entfernung pathologischer Faelle */
  /* hi1 oder hi2 ueberdecken nur einen Punkt */
  if ((hi1->xbeg == hi1->xend) || (hi2->xbeg == hi2->xend))
   {
    if (hi2->xbeg != hi2->xend)
      {
       /* hi2 ausgeben */
       add_span(hi2,listsp);
       free(hi1);
      }
    else if (hi1->xbeg != hi1->xend)
      {
       /* hi1 ausgeben */
       add_span(hi1,listsp);
       free(hi2);
      }
    else
      {
       /* beide Spannen nur ein Punkt und hintereinander */
       if (hi1->zbeg > hi2->zbeg)
         {
          /* hi2 ausgeben */ 
          add_span(hi2,listsp);
          free(hi1);
         }
       else
         {
          /* hi1 ausgeben */
          add_span(hi1,listsp);
          free(hi2);
         };   
      };
    return; /* alles getan */
   };


  /* Berechnung der Geradengleichungen der Spannen */
  m1 = hi1->zbeg - hi1->zend;
  hilf = hi1->xbeg - hi1->xend;
  m1 /= hilf;
  hilf = m1 * (double)hi1->xbeg;
  b1 = (double)hi1->zbeg - hilf;
  m2 = hi2->zbeg - hi2->zend;
  hilf = hi2->xbeg - hi2->xend;
  m2 /= hilf;
  hilf = m2 * (double)hi2->xbeg;
  b2 = (double)hi2->zbeg - hilf;

  /* Festlegen von Spanne1 und Spanne2 */
  if (hi1->xbeg == hi2->xbeg) 
      if ((hi1->zbeg > hi2->zbeg) || ((hi1->zbeg == hi2->zbeg) && (m1 > m2)))
        {
          hi = hi1;
          hi1 = hi2;
          hi2 = hi;
          hilf = m1;
          m1 = m2;
          m2 = hilf;
          hilf = b1;
          b1 = b2;
          b2 = hilf;
        };


  /* hi2 liegt vollstaendig hinter hi1 */
  if ((hi1->xbeg <= hi2->xbeg) && (hi2->xend <= hi1->xend) &&
      (floor(m1 * hi2->xbeg + b1 + 0.5) < hi2->zbeg) &&  
      (floor(m1 * hi2->xend + b1 + 0.5) < hi2->zend))
      {
        /* hi1 ausgeben */
        add_span(hi1,listsp);
        free(hi2);
        return; /* alles getan */
      };
  /* hi1 liegt vollstaendig hinter hi2 */
  if ((hi2->xbeg <= hi1->xbeg) && (hi1->xend <= hi2->xend) &&
      (floor(m2 * hi1->xbeg + b2 + 0.5) < hi1->zbeg) &&  
      (floor(m2 * hi1->xend + b2 + 0.5) < hi1->zend))
      {
        /* hi2 ausgeben */
        add_span(hi2,listsp);
        free(hi1); 
        return; /* alles getan */
      };


  dx1 = hi1->xend - hi1->xbeg;
  dx2 = hi2->xend - hi2->xbeg;
  null.x = 0.0;
  null.y = 0.0;
  null.z = 0.0;
  if (dx1 == 0)
    dn1 = null;
  else 
    {
      dn1.x = (hi1->norend.x - hi1->norbeg.x) / dx1;
      dn1.y = (hi1->norend.y - hi1->norbeg.y) / dx1;
      dn1.z = (hi1->norend.z - hi1->norbeg.z) / dx1;
    };
  if (dx2 == 0)
    dn2 = null;
  else 
    {
      dn2.x = (hi2->norend.x - hi2->norbeg.x) / dx2;
      dn2.y = (hi2->norend.y - hi2->norbeg.y) / dx2;
      dn2.z = (hi2->norend.z - hi2->norbeg.z) / dx2;
    };

  /* Vergleich der Spannen */

  /* erste Teilspanne extrahieren */
  if ((neu = (spanne *)malloc(sizeof(spanne))) == NULL) exit(1);
  neu->xbeg = hi1->xbeg;
  neu->zbeg = hi1->zbeg;
  neu->norbeg = hi1->norbeg;
  neu->col = hi1->col;
  neu->att = hi1->att;
  /* bis Anfang von hi2 vorgehen */
  if (neu->xbeg != hi2->xbeg)
    {  
      tz1 = m1 * hi2->xbeg + b1;
      neu->zend = (int)floor(tz1 + 0.5);
      neu->xend = hi2->xbeg;
      dx1 = neu->xend - neu->xbeg;
      addpoint(hi1->norbeg, dx1, dn1, &neu->norend);
      add_span(neu, listsp);
    } 
  else
    { 
      free(neu);
      tz1 = (double)hi1->zbeg;      
    };   

  /* zweite Teilspanne extrahieren */
  if ((neu = (spanne *)malloc(sizeof(spanne))) == NULL) exit(1);
  if ((tz1 < hi2->zbeg) ||
      ((hi1->xbeg == hi2->xbeg) && (hi1->zbeg == hi2->zbeg)))
    {
      aktspanne = 1;
      neu->xbeg = hi2->xbeg;
      neu->zbeg = (int)floor(tz1 + 0.5);
      dx1 = neu->xbeg - hi1->xbeg;
      addpoint(hi1->norbeg, dx1, dn1, &neu->norbeg);
      neu->col = hi1->col;
      neu->att = hi1->att;
    } 
  else 
    {
      aktspanne = 2;
      neu->xbeg = hi2->xbeg;
      neu->zbeg = hi2->zbeg;
      neu->norbeg = hi2->norbeg;
      neu->col = hi2->col;
      neu->att = hi2->att;
    };
  tz2 = hi2->zbeg;
  x = hi2->xbeg;
  while (true) 
    {
      if (x >= hi2->xend) 
        {
          if (aktspanne == 2) 
            {
	      neu->zend = hi2->zend;
	      neu->xend = x;
	      neu->norend = hi2->norend;
	      add_span(neu, listsp);
	      if ((neu = (spanne *)malloc(sizeof(spanne))) == NULL) exit(1);
	      neu->xbeg = x;
	      neu->zbeg = (int)floor(tz1 + 0.5);
	      dx1 = neu->xbeg - hi1->xbeg;
	      addpoint(hi1->norbeg, dx1, dn1, &neu->norbeg);
	      neu->att = hi1->att;
	      neu->col = hi1->col;
              neu->xend = hi1->xend;
              neu->zend = hi1->zend;
              neu->norend = hi1->norend;
              add_span(neu, listsp);
              break;
            } 
          else 
            {
	      neu->zend = hi1->zend;
	      neu->xend = hi1->xend;
	      neu->norend = hi1->norend;
	      add_span(neu, listsp);
              break;
            };
        };
      if (x >= hi1->xend) 
        {
          if (aktspanne == 1) 
            {
	      neu->zend = hi1->zend;
	      neu->xend = x;
	      neu->norend = hi1->norend;
	      add_span(neu, listsp);
	      if ((neu = (spanne *)malloc(sizeof(spanne))) == NULL) exit(1);
	      neu->xbeg = x;
	      neu->zbeg = (int)floor(tz2 + 0.5);
	      dx2 = neu->xbeg - hi2->xbeg;
	      addpoint(hi2->norbeg, dx2, dn2, &neu->norbeg);
	      neu->att = hi2->att;
	      neu->col = hi2->col;
              neu->xend = hi2->xend;
              neu->zend = hi2->zend;
              neu->norend = hi2->norend;
              add_span(neu, listsp);
              break;
            } 
          else 
            {
	      neu->zend = hi2->zend;
	      neu->xend = hi2->xend;
	      neu->norend = hi2->norend;
	      add_span(neu, listsp);
              break;
            };
        };
      if ((aktspanne == 1) && (tz2 < tz1)) 
            {
	      /* aktspanne tauschen und neue Spanne initialisieren */
	      aktspanne = 2;
	      neu->zend = (int)floor(tz1 + 0.5);
	      neu->xend = x;
	      dx1 = neu->xend - hi1->xbeg;
	      addpoint(hi1->norbeg, dx1, dn1, &neu->norend);
	      add_span(neu, listsp);
	      if ((neu = (spanne *)malloc(sizeof(spanne))) == NULL) exit(1);
	      neu->xbeg = x;
	      neu->zbeg = (int)floor(tz2 + 0.5);
	      dx2 = neu->xbeg - hi2->xbeg;
	      addpoint(hi2->norbeg, dx2, dn2, &neu->norbeg);
	      neu->att = hi2->att;
	      neu->col = hi2->col;  
              continue;
            }
      else if ((aktspanne == 2) && (tz1 < tz2))
            {
	      /* aktspanne tauschen und neue Spanne initialisieren */
	      aktspanne = 1;
	      neu->zend = (int)floor(tz2 + 0.5);
	      neu->xend = x;
	      dx2 = neu->xend - hi2->xbeg;
	      addpoint(hi2->norbeg, dx2, dn2, &neu->norend);
	      add_span(neu, listsp);
	      if ((neu = (spanne *)malloc(sizeof(spanne))) == NULL) exit(1);
	      neu->xbeg = x;
	      neu->zbeg = (int)floor(tz1 + 0.5);
	      dx1 = neu->xbeg - hi1->xbeg;
	      addpoint(hi1->norbeg, dx1, dn1, &neu->norbeg);
	      neu->att = hi1->att;
	      neu->col = hi1->col;
              continue; 
            };    
      x++;
      tz1 += m1;
      tz2 += m2;
    };  /* while */
  
  /* hi1 und hi2 loeschen */
  free(hi1);
  free(hi2);
}

#undef maxint


static void compare_spans(spanne **listsp)
{
  /* Vergleich der Abschnitte -------> Liste der Spannen */
  spanne *hi, *hilf;

  if (*listsp == NULL) return;
  hilf = *listsp;
  while (hilf != NULL && hilf->next != NULL) 
    {
      if (hilf->next->xbeg == hilf->next->xend &&
  	 (hilf->xend == hilf->next->xbeg || hilf->xbeg == hilf->next->xend)) 
        {
          del_span(hilf->next, listsp);
          continue;
        };
      if (hilf->xbeg == hilf->xend &&
	 (hilf->next->xbeg == hilf->xend || hilf->next->xend == hilf->xend)) 
        {
          hi = hilf->next;
          del_span(hilf, listsp);
          hilf = hi;
          continue;
        };
      if (hilf->xend <= hilf->next->xbeg) 
        {
          hilf = hilf->next;
          continue;
        };
      if (hilf != *listsp)
        hi = hilf->prev;
      else
        hi = NULL;
      subdivide_spans(hilf, hilf->next, listsp);
      if (hi == NULL)
        hilf = *listsp;
      else
        hilf = hi;
      hilf = *listsp;
    };
}


static void make_spans(zwischenpuffer puffer,Polygon *zeiger,spanne **listsp)
{
  /*        Eintragung der Fuellabschnitte eines Polygons           */
  /*              in die Liste der Fuellabschnitte                  */
  int k;
  pointI a, b;
  int xa, xb, za, zb;
  pointR3 nb, ne, c;
  spanne *ein;

  k = 1;
  while (k < puffer.anz) 
    {
      a = puffer.vektor[k - 1];
      b = puffer.vektor[k];
      nb = puffer.normale[k - 1];
      ne = puffer.normale[k];
      if (a.x > b.x) 
        {
          xa = b.x;
          xb = a.x;
          za = b.z;
          zb = a.z;
          c = nb;
          nb = ne;
          ne = c;
        } 
      else 
        {
          xa = a.x;
          xb = b.x;
          za = a.z;
          zb = b.z;
        };
      if ((ein = (spanne *)malloc(sizeof(spanne))) == NULL) exit(1);
      ein->xbeg = xa;
      ein->xend = xb;
      ein->zbeg = za;
      ein->zend = zb;
      ein->norbeg = nb;
      ein->norend = ne;
      ein->att = zeiger->att;
      ein->col = zeiger->col;
      add_span(ein, listsp);
      k += 2;
    };
}




static void print_spans(spanne *listsp)
{
  spanne *h;

  h = listsp;
  while (h != NULL)
    {
       if (h->xbeg > h->xend) printf("\n  x:%d %d; z:%d %d",h->xbeg,h->xend,h->zbeg,h->zend);
       h = h->next;
    };
}



/**************************************************************************/
/*          die Hauptprozedur des Scanline-Algorithmus                    */
/**************************************************************************/

static void SCAN(boolean backfaceculling, 
                 schattierung shading, 
                 boolean specular, 
                 boolean texturen, 
                 boolean htest,
		 boolean inverse)
{
  spanne         *listsp;    /* Liste der Spannen            */
  list_Poly      lap;        /* Liste der aktiven Polygone   */
  list_Poly      lip;        /* Liste der inaktiven Polygone */
  int            akty;
  int            maxy, miny;
  boolean        ende;
  zwischenpuffer puffer;
  color          dimension;
  int            x,y;
  attrib         *att;

  listsp=new_spans();
  new_List(&lap);
  new_List(&lip);
  x_dim = 400;
  y_dim = 400;
  back.r=10;
  back.g=10;
  back.b=100;
  read_NFF_back_face_culling(inverse, backfaceculling, shading, specular, texturen, &maxy, &miny, &lip);
  if (comments) printf("\nY-range:%d %d",maxy,miny);
  x_dim /=10;
  y_dim /=10;
  dimension.r=x_dim;
  dimension.g=y_dim;
  dimension.b=24;
  x_dim *=10;
  y_dim *=10;
  setPixel(dimension);
  if (miny < 0) miny = 0;
  if (miny > y_dim-1) miny = y_dim-1;
  if (maxy < 0) maxy = 0;
  if (maxy > y_dim-1) maxy = y_dim-1;
  if (miny > maxy) 
    {                    
      maxy = y_dim+1;
      miny = y_dim;
    };

  for (y=y_dim-1; y > maxy; y--)
    {
      for (x=0; x < x_dim; x++) setPixel(back);
    };
  akty = maxy;

  /* Entfernung nicht sichtbarer Polygone aus der Liste der inaktiven Polygone */
  set_first_Poly(&lip);
  ende = false;
  do
   {
     if (Poly_unused(lip, akty))
       {
	 del_Poly(&lip);
	 if (last_Poly(lip)) ende = true;
       }
     else ende = !set_next_Poly(&lip);
   }
  while (!ende);

  /* Hauptschleife des Algorithmus */
  while ((akty >= miny) && (akty < y_dim))
    {
       if (comments) printf("\n processing line %d -->",akty);
       /* entfernen der inaktiven Polygone aus Liste der aktiven Polygone */
       set_first_Poly(&lap);
       ende = false;
       do
	 {
	  if (!Poly_aktiv(lap, akty))
	    {
	     del_Poly(&lap);
	     if (last_Poly(lap)) ende = true;
	    }
	  else ende = !set_next_Poly(&lap);
	 }
       while (!ende);
       /* Uebernahme der aktiven Polygone aus Liste der inaktiven Polygone */
       set_first_Poly(&lip);
       while (Poly_aktiv(lip, akty)) move_Poly(&lip, &lap);
       /* Erstellung der Liste der Fuellabschnitte */
       listsp = clear_spans(listsp);
       set_first_Poly(&lap);
       if (lap.list != NULL)
         do
	   {
	     make_points(get_Poly(lap), akty, &puffer);
	     make_spans(puffer, get_Poly(lap), &listsp);
  	   }
         while (set_next_Poly(&lap));
       /* Vergleich der Fuellabschnitte --> Liste der Spannen */
       compare_spans(&listsp);
/*
print_spans(listsp);
*/
       /* Ausgabe der Liste der Spannen */
       output_spans(akty, listsp, shading, specular, texturen, htest);
       if (comments) printf(" ready");
       akty--;
    };


  /* entfernen aller Datenstrukturen des Algorithmus' */
  listsp = clear_spans(listsp);
  set_first_Poly(&lap);
  do
     del_Poly(&lap);
  while (!last_Poly(lap));
  set_first_Poly(&lip);
  do
     del_Poly(&lip);
  while (!last_Poly(lip));
  while (attr_list != NULL)
    {
      att = attr_list;
      attr_list = attr_list->next;
      free(att);
    }; 
  del_Poly(&lap);
  for (y=(miny - 1); y >= 0; y--)
    for (x=0; x < x_dim; x++)
      setPixel(back);
}




void main (int argc,char *argv[])
{
  schattierung shading;
  boolean      specular,texturen,htest,backfaceculling,inverse,Filter;
  extern char *  optarg;
  extern int optind;
  extern int opterr;
  int     opt;


   if (argc < 2 )
     {
       printf("usage: scanline [<infile> <outfile>] [-c] [-s] [-t] [-b] [-i] [-z] [-m <method>]\n\n");
       printf("                <infile>  ---> '*.nff'-type\n");
       printf("                <outfile> ---> '*.pic'-type\n");
       printf("                -c ---> enable comments\n");
       printf("                -s ---> enable calculation of specular reflections\n");
       printf("                -t ---> enable calculation of textur (if defined)\n");
       printf("                -b ---> enable backfaceculling\n");
       printf("                -i ---> inverse surface-normals\n");
       printf("                -z ---> filter-behavior\n");
       printf("                -m <method> ---> adjust shading-method ('modif' is default)\n");
       printf("                shading-<method>'s:  'facet'   --> constant-shading\n");
       printf("                                     'gouraud' --> Gouraud-shading\n");     
       printf("                                     'modif'   --> modified Gouraud-shading\n");     
       printf("                                     'phong'   --> Phong-shading\n");     
       printf("                                     'phong_h' --> Phong-shading with H-test\n");     
       printf("                                     'non'     --> without shading\n");
       printf("                                     'random'  --> each polygon in another color\n");  
       exit(1);
     };

   /* default arguments */
   shading = modif2;
   specular = false;
   texturen = false;
   inverse = false;
   htest = false;
   backfaceculling = false;
   comments = false;
   Filter = false;

   infile = (char *)argv[1];
   outfile = (char *)argv[2];
   /* get arguments */
   optind = 1; 
   /* scanning for the first "-" */
   while ((optind < argc) && (strncmp(argv[optind],"-",1) != 0)) optind++;
   opterr = 0;
   while ((opt = getopt(argc,argv,"m:csztbi")) != EOF) 
     {
       switch (opt) 
        {
           case 'i':
                   inverse = true;
                   break ;
           case 'c':
                   comments = true;
                   break ;  
           case 's':
                   specular = true;
                   break ;  
           case 't':
                   texturen = true;
                   break ;  
           case 'z':
                   Filter = true;
                   comments = false;
                   break;
           case 'b':
                   backfaceculling = true;
                   break ;  
           case 'm':
                   if (strncmp(optarg,"facet",5) == 0)
                     {
                       shading = konst;
                       break;
                     }
                   else if (strncmp(optarg,"gouraud",7) == 0)
                     {
                       shading = gour;
                       break;
                     }     
                   else if (strncmp(optarg,"modif",5) == 0)
                     {
                       shading = modif2;
                       break;
                     }     
                   else if (strncmp(optarg,"phong_h",7) == 0)
                     {
                       shading = phong;
                       htest = true;
                       break;
                     }     
                   else if (strncmp(optarg,"phong",5) == 0)
                     {
                       shading = phong;
                       break;
                     }     
                   else if (strncmp(optarg,"non",3) == 0)
                     {
                       shading = noshade;
                       htest = false;
                       specular = false;
                       texturen = false;  
                       break;
                     }     
                   else if (strncmp(optarg,"random",6) == 0)
                     {
                       shading = randomsh;
                       texturen = false;  
                       break;
                     }     
	   case '?':
                   printf("usage: scanline [<infile> <outfile>] [-c] [-s] [-t] [-b] [-i] [-z] [-m <method>]\n\n");
                   printf("                <infile>  ---> '*.nff'-type\n");
                   printf("                <outfile> ---> '*.pic'-type\n");
                   printf("                -c ---> enable comments\n");
                   printf("                -s ---> enable calculation of specular reflections\n");
                   printf("                -t ---> enable calculation of textur (if defined)\n");
                   printf("                -b ---> enable backfaceculling\n");
                   printf("                -i ---> inverse surface-normals\n");
                   printf("                -z ---> filter-behavior\n");
                   printf("                -m <method> ---> adjust shading-method ('modif' is default)\n");
                   printf("                shading-<method>'s:  'facet'   --> constant-shading\n");
                   printf("                                     'gouraud' --> Gouraud-shading\n");     
                   printf("                                     'modif'   --> modified Gouraud-shading\n");     
                   printf("                                     'phong'   --> Phong-shading\n");     
                   printf("                                     'phong_h' --> Phong-shading with H-test\n");     
                   printf("                                     'non'     --> without shading\n");  
                   printf("                                     'random'  --> each polygon with another color\n");  
                   exit(1);
	 };
      };

   if (Filter)
     {
       comments = false;
       A = stdin;
       B = stdout;
     }
   else
     {
       A = fopen(infile,"rt");
       if (A == NULL)
         {
           printf("error: reading inputfile '%s' (doesn't exist)\n",infile);
           exit(1);
         };
       B = fopen(outfile,"wb");
     };

   SCAN(backfaceculling,shading,specular,texturen,htest,inverse);

   if (!Filter)
     {
       fclose(A);
       fclose(B);
     };

   if (comments) printf("\noutput-file '%s' created\n",outfile);
   exit(0);
}
