/*****************************************************************************/
/* module opr.c								     */
/*									     */
/* Author: Rene Lutz							     */
/*	   Labo Image							     */
/*	   Computing Science Center					     */
/*	   University of Geneva, Switzerland				     */
/* Date:   September 1990						     */
/* Copyright (c) A. Jacot-Descombes, T. Pun, C. Pellegrini, Uni. of Geneva   */
/* (This copyright notice should appear).				     */
/*									     */
/*****************************************************************************/

/*******************************************************************/
/*								   */
/*                                                                 */
/*	      Methode de segmentation de regions recursive	   */
/*		Algorithme de Ohlander, Price et Reddy		   */
/*								   */
/*		 Version 1.0, destinee au LaboImage	       	   */
/*								   */
/*******************************************************************/

#include <suntool/sunview.h>
#include <suntool/panel.h>
#include <suntool/canvas.h>
#include <suntool/scrollbar.h>
#include <pixrect/pixrect_hs.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <stdio.h>
#include <math.h>

#include "define.h"
#include "structure.h"
#include "global.h"

extern char *paneltabs[];
extern char *mastertabs[];
extern char *labeltabs[];
extern char *filetabs[];
extern char *canvastabs[];
extern char *oprmailtabs[];
extern Frame frame;

/********************************************************

		     macros et constantes

********************************************************/


#define MIN(a,b) ((a) < (b)) ? (a) : (b)
#define MIN3(a,b,c) (((a) < (b)) ? (((a) < (c)) ? (a) : (c)) : (((b) < (c)) ? (b) : (c)))
#define MAX(a,b) ((a) > (b)) ? (a) : (b)
#define MAX3(a,b,c) (((a) > (b)) ? (((a) > (c)) ? (a) : (c)) : (((b) > (c)) ? (b) : (c)))


/********************************************************

		declaration des structures

********************************************************/

struct pile_masque
{
    unsigned char *masque ;
    int nb_region ;
    struct pile_masque *next ;
};

struct min_max_list
{
    int val ;			     /* position dans l'histogramme */
    int mmax ;			     /*  1 s'il s'agit d'un maximum */
    struct min_max_list *next ;	
};

struct list_pic
{
    int pic;				/* position du pic = maxima */
    int l, r;		  /* minimas gauche & droit associes au pic */
    int surface ;			/* surface approximative */
    double kurtosis ;			      /* valeur statistique */
    float pente;				 /* pente moyenne   */
    int indice;				/* numero du plan image */
    struct list_pic *next;
};

struct precede	   /* stockage du pic et de son ordre de precedence */
{
    int pic;				/* position du pic = maxima */
    int l, r;		  /* minimas gauche & droit associes au pic */
    int priorite;			    /* niveau de precedence */
    float pente;				 /* pente moyenne   */
    double kurtosis ;			      /* valeur statistique */
    int surface ;			/* surface approximative */
    int indice ;			/* numero du plan image */

};


/********************************************************

    procedures externes au module, lien avec LaboImage

********************************************************/


extern void hproc_Ohlander_Price_Reddy_rgb();

extern ouverture() ;
extern fermeture() ;
extern Menu     mcurs, mcoord ;
extern int	write_master();
extern int	write_erreur();
extern void	fromto();
extern int	statis() ;
extern void	information() ;
extern int Confirm();


/********************************************************

	procedures du module de segmentation

********************************************************/


/*
	procedures de lien avec LaboImage
*/

caddr_t		init_main_opr_frame() ;
static void	proc_quit_opr() ;


/*	
	procedures de gestions des objets SunView
*/

static void opr_frame_init() ;
static void opr_panel_init() ;
static void opr_canvas_init() ;
static void init_masque_canvas() ;
static void close_masque_canvas() ;
static void show_masque() ;
static void show_pixrect() ;
static void paint_region() ;
static int verbose_on() ;
static int step_on() ;
static int masque_on() ;
static int smooth_on() ;
static void masque_proc() ;
static void start_proc() ;
static void wait_proc() ;
static void change_button_label() ;


static void line();
static void text() ;
static void clear_screen() ;
static void write_message() ;
static void draw_histogram_axes() ;
static void plot_histograms() ;
static void draw_peek_location() ;
static void display_histograms() ;

static void allocate_data_space() ;
static void init_val() ;
static void convert() ;
static void init_proc() ;

static int lire_filergb() ;
static void ecrire_fichier() ;
static caddr_t proc_couleur_rgb() ;

/*
    procedures de segmentation
*/

static void compute_histograms() ;
int *opr_mov_avg() ;
static void smooth_histograms() ;
static void fix_histogram_maximum() ;
struct min_max_list *opr_extrema() ;

struct list_pic *compute_peeks() ;
struct precede  extract_best_peek() ;

static void region_segmentation() ;
static void O_P_R() ;
static void store_proc() ;
static void load_proc() ;


/********************************************************

		variables globales

********************************************************/


/*
	Lien avec LaboImage
*/

extern int flag_break, flag_creer, flag_exec ;

Menu main_opr_menu;

int flag_opr_win_open = FALSE;
int flag_in_opr = FALSE;
int flag_conv, flag_autoconvert ;


/*
	objets SunView
*/

static Frame	    opr_frame, masque_frame ;
static Panel	    opr_panel ;
static Canvas	    opr_canvas, masque_canvas ;
static Pixwin	    *cpw, *ppw, *fpw, *masque_pw ;
static Panel_item   step_mode_item, verbose_mode_item, smooth_item,
		    threshold_slider, number_min,
		    next_button, message_item, masque_item,
		    store_button, load_button ;


/*
	table de couleur
*/

static unsigned char red[256] ;
static unsigned char green[256] ;
static unsigned char blue[256] ;


/*
	variables de calcul
*/


/* positions <x,y> des axes des histogrammes */

int x_pos[9] = {45, 275, 505, 45, 275, 505, 45, 275, 505} ; 
int y_pos[9] = {100, 100, 100, 225, 225, 225, 350, 350, 350} ;

/* valeurs max et min de l'intervale et val. max de l'hist. */

int hmax[9] ;
int val_min[9] ;
int val_max[9] ;

/* poiteurs sur images et histogrammes, type (lettre) */

int *data[9] ;
int *histo[9] ;
char type[9] ;

int taille_image ;
int width, height ;

/* image des regions, couleur et nb de pts minimums */

unsigned char *region_image;
int colour, nombre_min, ti ;
struct list_pic *liste_de_pics ;


/*******************************************************************/
/*								   */
/*		 Procedures de lien avec LaboImage		   */
/*								   */
/*******************************************************************/


/*******************************************************************/
/*								   */
/* nom      : init_main_opr_frame				   */
/*								   */
/* fonction : procedure appellee par le menu de LaboImage qui      */
/*            initialise et ouvre l'outil                          */
/*                                                                 */
/* entrees  : Menu	m                                          */
/*            Menu_item	mi                                         */
/*                                                                 */
/* globales : int flag_bother, flag_help, flag_in_opr		   */
/*	      Frame opr_frame					   */
/*	      Menu	main_opr_menu				   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : proc_quit_opr, hproc_Ohlander_Price_Reddy_rgb,	   */ 
/*	      Ohlander_Price_Reddy_rgbopr_frame_init		   */
/*	      opr_panel_init, opr_canvas_init			   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

caddr_t init_main_opr_frame(m,mi)
Menu	m ;
Menu_item mi ;
{
	static void proc_quit_opr();

	if (flag_bother)
	{
		hproc_Ohlander_Price_Reddy_rgb();
		return;
	}
	if (flag_help) 
		hproc_Ohlander_Price_Reddy_rgb();
	flag_opr_win_open = TRUE;
	if (flag_in_opr) 
		return ;
	flag_in_opr = TRUE;


	opr_frame_init() ;
	main_opr_menu = (Menu) window_get(opr_frame, WIN_MENU, 0);
	menu_set(main_opr_menu,
			MENU_REMOVE, 1,
			MENU_ACTION_ITEM, "quit*", proc_quit_opr,
			0);
	window_set(opr_frame, WIN_MENU, main_opr_menu,0);
	opr_panel_init() ;
	opr_canvas_init() ;
	window_fit(opr_frame) ;
	window_set(opr_frame, WIN_SHOW, TRUE, 0) ;
    
}


/*******************************************************************/
/*								   */
/* nom      : proc_quit_opr				           */
/*								   */
/* fonction : termine la session avec et quitte l'outil	           */
/*                                                                 */
/* entrees  : ---	                                           */
/*                                                                 */
/* globales :  int flag_opr_win_open, flag_in_opr		   */
/*             Frame opr_frame.					   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : Confirm()						   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
proc_quit_opr()
{
	if (Confirm("quit Tool ?") == 1)
	{
		window_destroy(opr_frame);
		flag_opr_win_open = FALSE;
		flag_in_opr = FALSE;
	}
}


/*******************************************************************/
/*								   */
/*		Procedures de gestions d'objets SunView		   */
/*								   */
/*******************************************************************/



/*******************************************************************/
/*								   */
/* nom      : opr_frame_init					   */
/*								   */
/* fonction : initialisartion du cadre de la fenetre		   */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : Frame opr_frame					   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
opr_frame_init()
{
    opr_frame = window_create  (frame, FRAME,
			    FRAME_LABEL, labeltabs[8],
			    WIN_X, 412, 
			    WIN_Y, 406, 
			    FRAME_SHOW_LABEL, TRUE, 
			    0) ;

}


/*******************************************************************/
/*								   */
/* nom      : opr_panel_init					   */
/*								   */
/* fonction : creation du panneau d'application			   */
/*                                                                 */
/* entrees  : ---				                   */
/*                                                                 */
/* globales : Frame opr_frame; Panel opr_panel;			   */
/*	      Panel_item verbose/smooth/masque/step_mode_item,     */
/*			 next_button, message_item ,		   */
/*			 threshold_slider, number_min ;		   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
opr_panel_init()
{
    opr_panel = window_create  (opr_frame, PANEL,
			    WIN_SHOW, TRUE,
			    0) ;

    verbose_mode_item = panel_create_item(opr_panel, PANEL_CYCLE,
			    PANEL_LABEL_X, ATTR_COL(1),
			    PANEL_LABEL_Y, ATTR_ROW(0),
			    PANEL_LABEL_STRING, paneltabs[600],
			    PANEL_CHOICE_STRINGS, paneltabs[601],
			      paneltabs[602], 0,
			    PANEL_LABEL_BOLD, TRUE,
			    0) ;

    step_mode_item = panel_create_item(opr_panel, PANEL_CYCLE,
			    PANEL_LABEL_X, ATTR_COL(18),
			    PANEL_LABEL_Y, ATTR_ROW(0),
			    PANEL_LABEL_STRING, paneltabs[603],
			    PANEL_CHOICE_STRINGS, paneltabs[602],
			      paneltabs[601], 0,
			    PANEL_LABEL_BOLD, TRUE,
			    0) ;

    smooth_item = panel_create_item(opr_panel, PANEL_CYCLE,
			    PANEL_LABEL_X, ATTR_COL(40),
			    PANEL_LABEL_Y, ATTR_ROW(0),
			    PANEL_LABEL_STRING, paneltabs[604],
			    PANEL_CHOICE_STRINGS, paneltabs[601],
			      paneltabs[602], 0,
			    PANEL_LABEL_BOLD, TRUE,
			    0) ;

    masque_item = panel_create_item(opr_panel, PANEL_CYCLE,
			    PANEL_LABEL_X, ATTR_COL(60),
			    PANEL_LABEL_Y, ATTR_ROW(0),
			    PANEL_LABEL_STRING, paneltabs[605],
			    PANEL_CHOICE_STRINGS, paneltabs[602],
			      paneltabs[601], 0,
			    PANEL_LABEL_BOLD, TRUE,
			    PANEL_NOTIFY_PROC, masque_proc, 
			    0) ;

    next_button = panel_create_item(opr_panel, PANEL_BUTTON,
			    PANEL_LABEL_X, ATTR_COL(82),
			    PANEL_LABEL_Y, ATTR_ROW(0),
			    PANEL_LABEL_BOLD, TRUE,
			    PANEL_LABEL_IMAGE,
			    panel_button_image(opr_panel,paneltabs[606],6,0),
			    PANEL_NOTIFY_PROC, start_proc, 
			    0) ; 

    store_button = panel_create_item(opr_panel, PANEL_BUTTON,
			    PANEL_LABEL_X, ATTR_COL(82),
			    PANEL_LABEL_Y, ATTR_ROW(1),
			    PANEL_LABEL_BOLD, TRUE,
			    PANEL_LABEL_IMAGE,
			    panel_button_image(opr_panel,paneltabs[607],6,0),
			    PANEL_NOTIFY_PROC, store_proc,
			    0) ; 

    load_button = panel_create_item(opr_panel, PANEL_BUTTON,
			    PANEL_LABEL_X, ATTR_COL(82),
			    PANEL_LABEL_Y, ATTR_ROW(2),
			    PANEL_LABEL_BOLD, TRUE,
			    PANEL_LABEL_IMAGE,
			    panel_button_image(opr_panel,paneltabs[608],6,0),
			    PANEL_NOTIFY_PROC, load_proc,
			    0) ; 		    

    threshold_slider = panel_create_item(opr_panel, PANEL_SLIDER,
			    PANEL_LABEL_X, ATTR_COL(1),
			    PANEL_LABEL_Y, ATTR_ROW(1),
			    PANEL_LABEL_BOLD, TRUE,
                            PANEL_LABEL_STRING, paneltabs[609],
			    PANEL_MIN_VALUE, 1,
			    PANEL_MAX_VALUE, 50,
			    PANEL_SLIDER_WIDTH, 200,
			    PANEL_VALUE, 5,
			    PANEL_NOTIFY_LEVEL, PANEL_ALL,
			    0) ;

    number_min = panel_create_item(opr_panel, PANEL_TEXT,
			    PANEL_LABEL_X, ATTR_COL(60),
			    PANEL_LABEL_Y, ATTR_ROW(1),
			    PANEL_LABEL_BOLD, TRUE,
                            PANEL_LABEL_STRING, paneltabs[610],
			    PANEL_VALUE_DISPLAY_LENGTH, 5,
			    PANEL_VALUE, "1000",
			    0) ;

    message_item = panel_create_item(opr_panel, PANEL_TEXT,
			    PANEL_LABEL_X, ATTR_COL(1),
			    PANEL_LABEL_Y, ATTR_ROW(2),
			    PANEL_VALUE_DISPLAY_LENGTH, 70,
			    PANEL_LABEL_STRING, "---> ",
			    PANEL_LABEL_BOLD, TRUE,
			    0) ;
    window_fit_height(opr_panel) ;
}


/*******************************************************************/
/*								   */
/* nom      : opr_canvas_init					   */
/*								   */
/* fonction : initialise le canevas et les couleurs		   */
/*                                                                 */
/* entrees  : ---				                   */
/*                                                                 */
/* globales : Pixwin *ppw, *cpw, *fpw ;				   */
/*	      Frame opr_frame; Canvas opr_canvas; Panel opr_panel  */
/*	      unsigned char red[], green[], blue[]                 */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
opr_canvas_init()
{
    
    opr_canvas = window_create (opr_frame, CANVAS,
			    WIN_BELOW, opr_panel, 
			    WIN_WIDTH, 730, 
			    WIN_HEIGHT, 400, 
			    0) ;

    red[2] = 255 ;
    green[2] = 255 ;
    blue[2] = 100 ;

    red[3] = 210 ;
    green[3] = 30 ;
    blue[3] = 0 ;
    red[4] = 15 ;
    green[4] = 210 ;
    blue[4] = 35 ;
    red[5] = 16 ;
    green[5] = 138 ;
    blue[5] = 213 ;

    red[6] = 162  ;
    green[6] = 111 ;
    blue[6] = 109 ;
    red[7] = 123 ;
    green[7] = 162 ;
    blue[7] = 109 ;
    red[8] = 109 ;
    green[8] = 116 ;
    blue[8] = 162 ;

    red[9] = 220 ;
    green[9] = 156 ;
    blue[9] = 248 ;
    red[10] = 255 ;
    green[10] = 255 ;
    blue[10] = 255 ;
    red[11] = 160 ;
    green[11] = 160 ;
    blue[11] = 160  ;
    red[255] = 255 ;
    green[255] = 255 ;
    blue[255] = 255 ;
   

    cpw = (Pixwin *) canvas_pixwin(opr_canvas) ;
    pw_setcmsname(cpw, "CANVAS") ;
    pw_putcolormap(cpw, 0, 256, red, green, blue) ;
    ppw = (Pixwin *) window_get(opr_panel, WIN_PIXWIN) ;
    pw_setcmsname(ppw, "PANEL") ;
    pw_putcolormap(ppw, 0, 256, red, green, blue) ;
    fpw = (Pixwin *) window_get(opr_frame, WIN_PIXWIN) ;
    pw_setcmsname(fpw, "FRAME") ;
    pw_putcolormap(fpw, 0, 256, red, green, blue) ;
    
}


/*******************************************************************/
/*								   */
/* nom      : init_masque_canvas				   */
/*								   */
/* fonction : initialise la fentre contenant le masque		   */
/*                                                                 */
/* entrees  :							   */
/*                                                                 */
/* globales : opr_frame, masque_frame, masque_pw, masque_canvas    */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : write_message					   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
init_masque_canvas()
{
    Scrollbar barvert, barhoriz ;

    masque_frame = window_create(opr_frame, FRAME,
			    FRAME_LABEL, labeltabs[9],
			    WIN_X, -320, 
			    WIN_Y, 0, 
			    WIN_SHOW, TRUE,
			    0) ;

    masque_canvas = window_create (masque_frame, CANVAS,
                         CANVAS_AUTO_EXPAND, FALSE,
                         CANVAS_AUTO_SHRINK, FALSE,
                         CANVAS_AUTO_CLEAR, TRUE,
                         CANVAS_FIXED_IMAGE, FALSE,
			 CANVAS_RETAINED, TRUE, 
			 WIN_VERTICAL_SCROLLBAR, barvert = scrollbar_create(0),
			 WIN_HORIZONTAL_SCROLLBAR, barhoriz = scrollbar_create(0),
                         WIN_WIDTH, 300,     
                         WIN_HEIGHT, 300,
                         0);

    window_fit(masque_frame) ;

    scrollbar_paint_clear (barvert);
    scrollbar_paint_clear (barhoriz); 
 

    masque_pw = (Pixwin *) canvas_pixwin(masque_canvas) ;
    pw_setcmsname(masque_pw, "MASQUECANVAS") ;
}


/*******************************************************************/
/*								   */
/* nom      : close_masque_canvas		         	   */
/*								   */
/* fonction : ferme la fenetre du masque			   */
/*                                                                 */
/* entrees  :							   */
/*                                                                 */
/* globales : masque_frame					   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
close_masque_canvas()
{
	window_set(masque_frame, FRAME_NO_CONFIRM, TRUE, 0) ;
        window_destroy(masque_frame) ;	
}

/*******************************************************************/
/*								   */
/* nom      : show_masque					   */
/*								   */
/* fonction : affiche le masque dans la fenetre speciale	   */
/*                                                                 */
/* entrees  : char *mask					   */
/*                                                                 */
/* globales : masque_pw, taille_image				   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/


static void
show_masque(mask)
unsigned char *mask ;
{
    unsigned char *d ;
    register i=0 ;
    struct pixrect *pix_masque ;

	if ((pix_masque = mem_create (width,height,8)) == NULL)
	{
	    write_message(oprmailtabs[0]) ;
	    return ;
	} 
   
	d = (unsigned char *) mpr_d(pix_masque)-> md_image ;

	while (i < taille_image)
	{
            if(*mask) 
		*d = 255 ;
            else *d = 0 ;

		d++ ; 
		mask++; 
		i++ ; 
	}
  
    pw_rop(masque_pw, 0, 0, width, height, PIX_SRC, pix_masque, 0, 0) ;
    pr_close(pix_masque) ;
}


/*******************************************************************/
/*								   */
/* nom      : show_pixrect					   */
/*								   */
/* fonction : affiche une image dans le pixrect ouvert 		   */
/*                                                                 */
/* entrees  : unsigned char *image				   */
/*                                                                 */
/* globales : int taille_image, width, height 	  	           */
/*	      Pixwin *masque_pw					   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : write_message				           */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
show_pixrect(image) 
unsigned char *image ;
{
    unsigned char *d;
    register i=0 ;
    struct pixrect *pix ;
	
	if ((pix = mem_create (width,height,8)) == NULL)
	{
	    write_message(oprmailtabs[0]) ;
	    return ;
	} 

	d = (unsigned char *) mpr_d(pix)-> md_image ;

	while (i < taille_image)
	{
		*d = *image;
	        if (*d == 0) *d = 255 ;
		d++ ; 
                image++ ;
		i++ ; 
	}

    for (i=0 ; i < 256 ; i++)
	red[i] = green[i] = blue[i] = i ;

    pw_putcolormap(masque_pw, 0, 256, red, green, blue) ;
  
    pw_rop(masque_pw, 0, 0, width, height, PIX_SRC, pix, 0, 0) ;
    pr_close(pix) ;
}


/********************************************************************/
/*								    */
/* Nom	    : paint_region					    */
/*								    */
/* Fonction : ecriture d'une region a travers le pochoir Masque	    */
/*								    */
/* Entree   : unsigned char *Masque : Masque pochoir			    */
/*								    */
/* Global   : unsigned char *region_image : image des regions	    */
/*	      int colour = couleur des regions, taille_image        */
/*								    */
/* Routine  : ---						    */
/*								    */
/* Sortie   : ---						    */
/*								    */
/* Made by Rene W. LUTZ 24.01.1990				    */
/********************************************************************/

static void
paint_region (Masque)
unsigned char *Masque;
{
    unsigned char *tmp ;
    register int i ;

    /* ecriture de l'image a travers le masque */

    colour = (random() % 256) ;

    tmp = region_image ;
    for (i = 0; i < taille_image; i++)
    {
	if (*Masque) 
	    *tmp = colour;
	Masque++;
	tmp++;
    } 
} 


/*******************************************************************/
/*								   */
/* nom      : masque_on						   */
/*								   */
/* fonction : determine si le masque est affiche		   */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : Panel_item masque_item				   */
/*								   */
/* return   : TRUE si "ON", "FALSE" si "OFF"			   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static int
masque_on()
{
    int	index ;

    index = (int) panel_get_value(masque_item) ;

    if (index == 0)
	return(FALSE) ;
    else 
	return(TRUE);
}


/*******************************************************************/
/*								   */
/* nom      : smooth_on						   */
/*								   */
/* fonction : determine si le lissage du masque doit etre fait	   */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : Panel_item smooth_item				   */
/*								   */
/* return   : TRUE si "ON", "FALSE" si "OFF"			   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static int
smooth_on()
{
    int	index ;

    index = (int) panel_get_value(smooth_item) ;

    if (index == 0)
	return(TRUE) ;
    else 
	return(FALSE);
}


/*******************************************************************/
/*								   */
/* nom      : verbose_on					   */
/*								   */
/* fonction : determine si le mode verbeux est actif		   */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : Panel_item verbose_mode_item			   */
/*								   */
/* return   : TRUE si "ON", "FALSE" si "OFF"			   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static int
verbose_on()
{
    int	index ;

    index = (int) panel_get_value(verbose_mode_item) ;

    if (index == 0)
	return(TRUE) ;
    else 
	return(FALSE);
}


/*******************************************************************/
/*								   */
/* nom      : step_on						   */
/*								   */
/* fonction : determine si le mode pas a pas est actif		   */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : Panel_item step_mode_item				   */
/*								   */
/* return   : TRUE si "ON", "FALSE" si "OFF"			   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static int
step_on()
{
    int	index ;

    index = (int) panel_get_value(step_mode_item) ;

    if (index == 0)
	return(FALSE) ;
    else 
	return(TRUE);
}


/*******************************************************************/
/*								   */
/* nom      : masque_proc					   */
/*								   */
/* fonction : ouvre la fenetre du masque ou la ferme		   */
/*                                                                 */
/* entrees  :							   */
/*                                                                 */
/* globales :							   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : masque_on, init_masque_canvas,			   */
/*	      close_masque_canvas				   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
masque_proc()
{
    if (masque_on())
	init_masque_canvas() ;	
    else
	close_masque_canvas() ;
}


/*******************************************************************/
/*								   */
/* nom      : start_proc					   */
/*								   */
/* fonction : lance la segmentation par region			   */
/*                                                                 */
/* entrees  : ---						   */
/*                                                                 */
/* globales : ---						   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : region_segmentation				   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
start_proc()
{
	region_segmentation() ;
}

/********************************************************************/
/*								    */
/* Nom	    : wait_proc						    */
/*								    */
/* Fonction : attend que l'utilisateur presse <gauche>		    */
/*								    */
/* Entree   : ---						    */
/*								    */
/* Global   : Frame opr_frame					    */
/*								    */
/* Routine  : ---						    */
/*								    */
/* Sortie   : ---						    */
/*								    */
/* Made by Rene W. LUTZ 24.01.1990				    */
/********************************************************************/

static void
wait_proc()
{
    int test_mouse = 0 ;

    write_message(oprmailtabs[1]);
    while (!test_mouse)
        test_mouse = (int) window_get(opr_frame, WIN_EVENT_STATE, MS_LEFT) ;
	write_message(oprmailtabs[2]) ;
    
}


/*******************************************************************/
/*								   */
/* nom      : change_button_label				   */
/*								   */
/* fonction : ecrit le message "str" sur le bouton		   */
/*                                                                 */
/* entrees  : char *str                                            */
/*                                                                 */
/* globales : Panel_item next_button ; Panel opr_panel		   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
change_button_label(str)
char *str ;
{
	panel_set(next_button, PANEL_LABEL_IMAGE,
	panel_button_image(opr_panel, str, 6 , 0), 0);
}





/*******************************************************************/
/*								   */
/* nom      : load_proc					           */
/*								   */
/* fonction : charge un plan image RGB			           */
/*                                                                 */
/* entrees  : ---	                                           */
/*                                                                 */
/* globales : int *data[]					   */
/*	      int flag_break, flag_conv, flag_autoconvert	   */
/*	      dir_image[], dir_desc[], index_image[]		   */
/*	      int height, width, taille_image			   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : fromto(), interruption(), statis()		   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
load_proc()
{
    register i ;

	if ( (data[0] != NULL) && ( (flag_conv == TRUE) || (flag_autoconvert == TRUE)))
	{
	    cfree(data[0]) ;cfree(data[1]) ;cfree(data[2]) ;
	}

	    fromto(FROM, RGB) ;
	    if (flag_break)
	    {
		interruption();
		return;
	    }

            sprintf (buf,mastertabs[134], index_image[0],
		     index_image[2],index_image[4]);
            write_master (buf);

	switch (dir_desc[index_image[0]].type)
	{
		case -1:
		case 0:
		case 1:	flag_conv = TRUE;
			data[0] = (int *) conv_int (index_image[0]);
			data[1] = (int *) conv_int (index_image[2]);
			data[2] = (int *) conv_int (index_image[4]);
			break;
		case 2: flag_conv = FALSE;
			data[0] = (int *) dir_image[index_image[0]].image;
			data[1] = (int *) dir_image[index_image[2]].image;
			data[2] = (int *) dir_image[index_image[4]].image;
			break;
		case 3:
		case 4:
		case 5: flag_autoconvert = TRUE;

			autoconvert_integer(index_image[0]);
			data[0] = (int *) buffer_image[0].image;
			buffer_image[0].image = NULL ;

 /* bricolage de buffer_image*/
			autoconvert_integer(index_image[2]);
			data[1] = (int *) buffer_image[0].image;
			buffer_image[0].image = NULL ;
 /* bricolage de buffer_image*/

			autoconvert_integer(index_image[4]);
			data[2] = (int *) buffer_image[0].image;
			buffer_image[0].image = NULL ;

			break;
	}

	height = dir_desc[index_image[1]].nligne;
	width = dir_desc[index_image[1]].ncolonne;
	taille_image = width*height ;
	
	/* allocation pour les plans images */
	
	for (i = 3 ; i <= 8 ; i++)
		data[i] = (int *) calloc (taille_image, sizeof(int));
	for (i = 0 ; i < 9 ; i++) 
	{
		val_min[i] = 10000 ;
		val_max[i] = -1 ;
	}

	type[0] = 'R' ; type[1] = 'G' ; type[2] = 'B' ;
	type[3] = 'Y' ; type[4] = 'I' ; type[5] = 'Q' ;
	type[6] = 'H' ; type[7] = 'L' ; type[8] = 'S' ;

	/* conversion des RGB --> YIQ et HLS calcul des min et max */

	convert(data[0],data[1],data[2],data[3],data[4],data[5],data[6],data[7],data[8]) ;

	/* allocation pour les histogrammes */

	for (i=0; i<=8 ; i++)
		histo[i] = (int *) calloc (val_max[i] - val_min[i] + 1, sizeof(int));

}

/*******************************************************************/
/*								   */
/* nom      : store_proc				           */
/*								   */
/* fonction : sauve le plan des regions dans un plan image         */
/*                                                                 */
/* entrees  : ---                                                  */
/*                                                                 */
/* globales : int flag_break,					   */
/*	      dir_image[], dir_desc[], index_image[]		   */
/*	      unsigned char *region_image			   */
/*								   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : fromto(), interruption(), statis()		   */
/*								   */
/* Made by Rene W. LUTZ 19.09.1990				   */
/*******************************************************************/

static void
store_proc()
{
	fromto(TO, DEFAUT) ;
	if (flag_break) 
	{
		interruption() ;
		return;
	}

	if (dir_image[index_image[1]].image != NULL) 
		free (dir_image[index_image[1]]);

	dir_image[index_image[1]].image =  (unsigned char *) region_image ;
	dir_desc[index_image[1]] = dir_desc[index_image[0]];
	dir_desc[index_image[1]].type = 0;

	statis (dir_image[index_image[1]].image,
			dir_desc[index_image[1]].type, 
			dir_desc[index_image[1]].nligne,
			dir_desc[index_image[1]].ncolonne,
			&(dir_desc[index_image[1]].mmin),
			&(dir_desc[index_image[1]].mmax), 
			&(dir_desc[index_image[1]].mu),
			&(dir_desc[index_image[1]].ecart));
}



/*******************************************************************/
/*								   */
/*        Procedures de gestion du canevas des histogrammes	   */
/*								   */
/*******************************************************************/



/*******************************************************************/
/*								   */
/* nom      : line						   */
/*								   */
/* fonction : dessine une ligne					   */
/*                                                                 */
/* entrees  : int x1,y1,x2,y2 positions				   */
/*            color : couleur dans la table                        */
/*                                                                 */
/* globales : pw						   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990			           */
/*******************************************************************/

static void
line(x1,y1,x2,y2,color)
int x1,y1,x2,y2,color ;
{
    pw_vector(cpw, x1, y1, x2, y2, PIX_SRC | PIX_COLOR(color), 1) ;
}


/*******************************************************************/
/*								   */
/* nom      : text						   */
/*								   */
/* fonction : affiche un texte sur le canevas			   */
/*                                                                 */
/* entrees  : int x,y (position)				   */
/*            char *  (chaine de caracteres                        */
/*            color   (couleur)                                    */
/*                                                                 */
/* globales : pw						   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
text(x,y,s, color)
int x,y ;
char *s;
int color ;
{
    pw_text(cpw, x, y, PIX_SRC | PIX_COLOR(color), 0, s) ;
}


/*******************************************************************/
/*								   */
/* nom      : clear_screen  					   */
/*								   */
/* fonction : efface les histogrammes				   */
/*                                                                 */
/* entrees  :							   */
/*                                                                 */
/* globales : cpw, opr_canvas				           */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
clear_screen()
{
	int xdim, ydim ;
	
	xdim = (int) window_get(opr_canvas, WIN_WIDTH) ;
	ydim = (int) window_get(opr_canvas, WIN_HEIGHT) ;

	pw_writebackground(cpw, 0, 0, xdim, ydim, PIX_SRC | PIX_COLOR(0)) ;
}


/*******************************************************************/
/*								   */
/* nom      : write_message				           */
/*								   */
/* fonction : ecrit le message "str" dans le panneau principal     */
/*                                                                 */
/* entrees  : char *str                                            */
/*                                                                 */
/* globales : Panel_item message_item;				   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
write_message(str) 
char *str ;
{
    if (verbose_on())
	panel_set(message_item, PANEL_VALUE, str, 0);
}


/*******************************************************************/
/*								   */
/* nom      : draw_histogram_axes				   */
/*								   */
/* fonction : dessin des reperes des histogrammes		   */
/*                                                                 */
/* entrees  : int posx, posy, hmax, mmin, mmax, color ;            */
/*            char c ;                                             */
/*                                                                 */
/* globales : Pixwin *pw					   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : text, line					   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
draw_histogram_axes(posx, posy, hmax, mmin, mmax, c, color) 
int posx, posy, hmax, mmin, mmax, color ;
char c ;
{
    char s[30] ;

    line(posx - 4, posy + 1, posx + 205, posy + 1, color) ;
    line(posx - 1, posy, posx - 1, posy - 80,color) ;
    line(posx - 4, posy - 20, posx - 1, posy - 20, color) ;
    line(posx - 4, posy - 40, posx - 1, posy - 40, color) ;
    line(posx - 4, posy - 60, posx - 1, posy - 60, color) ;
    line(posx - 4, posy - 80, posx - 1, posy - 80, color) ;

    sprintf(s,"%4d", hmax) ;
    text(posx - 31, posy - 75, s, color) ;
    sprintf(s,"%3d <= %c <= %3d", mmin, c, mmax) ;
    text(posx + 25, posy + 22, s, color) ;
 
}


/*******************************************************************/
/*								   */
/* nom      : plot_histograms					   */
/*								   */
/* fonction : dessin des histogrammes				   */
/*                                                                 */
/* entrees  : int *histogram, posx, posy, hmax, mmin, mmax, color; */
/*                                                                 */
/* globales : Pixwin *pw					   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : line, 						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
plot_histograms(histogram, posx, posy, hmax, mmin, mmax, color)
int *histogram, posx, posy, hmax, mmin, mmax, color ;
{
    register int i=0, length, haut, xx;
    float height = (float) 80.0/hmax ;
    
    length = (mmax - mmin + 1) ;
    height = (float) 80.0/hmax ;

    if (length < 205)
    {
	while (i < length)
	{
	    haut = (int) rint ((double) (*histogram * height)) ;
	    line(posx + i, posy, posx + i, posy - haut, color) ;
	    histogram++ ;
	    i++ ;
	}
    }
    else
    {
	float dist = (float) length/205.0 ;
	while (i < length)
	{
	    haut = (int) rint ((double) (*histogram * height)) ;
	    xx = (int) rint ((double) (i/dist)) ;
	    line(posx + xx, posy, posx + xx, posy - haut, color) ;
	    histogram++ ;
	    i++ ;
	}

    }
}


/*******************************************************************/
/*								   */
/* nom      : draw_peek_location				   */
/*								   */
/* fonction : dessine un rectangle et une bare qui montre le pic   */
/*                                                                 */
/* entrees  : int pic, l, r, i                                     */
/*                                                                 */
/* globales : x_pos[9], y_pos[9], val_max[9], val_min[9]	   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : line()						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
draw_peek_location(pic,l,r,i)
int pic, l, r, i ;
{ 
    int right, left, peek ;

    if (val_max[i] < 205)
    {
	line(x_pos[i] + l, y_pos[i] + 1, x_pos[i] + r , y_pos[i] + 1, 2) ;
	line(x_pos[i] + l, y_pos[i] - 80 , x_pos[i] + r, y_pos[i] -80, 2) ;
	line(x_pos[i] + l, y_pos[i] + 1, x_pos[i] + l, y_pos[i] -80, 2) ;
	line(x_pos[i] + r, y_pos[i] + 1, x_pos[i] + r, y_pos[i] -80, 2) ;
	line(x_pos[i] + pic, y_pos[i] -70, x_pos[i] + pic, y_pos[i] -80, 2) ;
    }
    else
    {
	float dist = (float) (val_max[i]-val_min[i] + 1)/205.0 ;

	right = (int) rint ((double) (r/dist)) ;
	left = (int) rint ((double) (l/dist)) ;
	peek = (int) rint ((double) (pic/dist)) ;
	line(x_pos[i] + left, y_pos[i] + 1, x_pos[i] + right, y_pos[i] + 1, 2) ;
	line(x_pos[i] + left, y_pos[i] - 80 , x_pos[i] + right, y_pos[i] -80, 2) ;
	line(x_pos[i] + left, y_pos[i] + 1, x_pos[i] + left, y_pos[i] -80, 2) ;
	line(x_pos[i] + right, y_pos[i] + 1, x_pos[i] + right, y_pos[i] -80, 2) ;
	line(x_pos[i] + peek, y_pos[i] -70, x_pos[i] + peek, y_pos[i] -80, 2) ;
    }
    
}


/*******************************************************************/
/*								   */
/* nom      : display_histograms				   */
/*								   */
/* fonction : dessine les histogrammes et les axes		   */
/*                                                                 */
/* entrees  :						           */
/*                                                                 */
/* globales :							   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : draw_histogram_axes, plot_histograms 		   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
display_histograms()
{
    register i ;

    clear_screen() ;

    for (i=0 ; i < 9 ; i++)
    {
	plot_histograms(histo[i], x_pos[i], y_pos[i], hmax[i], val_min[i],
	val_max[i], i + 3) ;
	draw_histogram_axes(x_pos[i], y_pos[i], hmax[i] , val_min[i],
	val_max[i],type[i], 255) ; 
    }
}



/*******************************************************************/
/*								   */
/*		Procedures de calcul et segmentation		   */
/*								   */
/*******************************************************************/


/*******************************************************************/
/*								   */
/* nom      : convert						   */
/*								   */
/* fonction : converti les plan RGB en plan YIQ HLS		   */
/*                                                                 */
/* entrees  : int *r,*g,*b,*y,*i,*q,*h,*l,*s ;                     */
/*                                                                 */
/* globales : int *val_min[9], val_max[9] ;		           */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
convert(r,g,b,y,i,q,h,l,s)
int *r,*g,*b,*y,*i,*q,*h,*l,*s ;
{
    register int  count = 0 ;
    register int mini, maxi, moins, plus ;
    float saturation, hue, lightness ;
    register float rc, bc, gc ;
    
    val_max[0] = dir_desc[index_image[0]].mmax ;
    val_min[0] = dir_desc[index_image[0]].mmin ;
    val_max[1] = dir_desc[index_image[2]].mmax ;
    val_min[1] = dir_desc[index_image[2]].mmin ;
    val_max[2] = dir_desc[index_image[4]].mmax ;
    val_min[2] = dir_desc[index_image[4]].mmin ;

    write_message(oprmailtabs[3]) ;

    while(count < taille_image)
    {
/*	    val_min[0] = MIN(*r, val_min[0]) ;
	    val_max[0] = MAX(*r, val_max[0]) ;
	    val_min[1] = MIN(*g, val_min[1]) ;
	    val_max[1] = MAX(*g, val_max[1]) ;
	    val_min[2] = MIN(*b, val_min[2]) ;
	    val_max[2] = MAX(*b, val_max[2]) ;*/	

	*y = (int) rint ((double) *r * 0.3 + *g * 0.59 + *b * 0.11) ;
	*i = (int) rint ((double) *r * 0.6 - *g * 0.28 - *b * 0.32) ;
	*q = (int) rint ((double) *r * 0.21 + *g * 0.52 + *b * 0.31);

/* espace de Ohta et al. ne donne pas de bons resultats */

/*
	*y = (int) rint ((double) (*r + *g + *b) * 0.33333333) ;
	*i = (int) rint ((double) (*r - *b) * 0.5) ;
	*q = (int) rint ((double) ((2 * *g) - *r -*b) * 0.25);
*/

	mini = MIN3(*r, *g, *b) ;
	maxi = MAX3(*r, *g, *b) ;
	moins = maxi - mini ;
	plus = maxi + mini ;
	lightness = (float) (plus / 2.0) ;
	*l = (int) floor ((double) lightness) ;

	if (maxi == mini)
	{
	    *s = 0 ;
	    *h = -1 ;
	}
	else
	{
	    if (lightness <= 128.0) 
		saturation = (float) moins/plus*256.0 ;
		saturation = (float) moins/(512.0 - plus)*256.0 ;

	    *s = (int) rint ((double) saturation) ;

	    rc = (float) (maxi - *r)/moins ;
	    gc = (float) (maxi - *g)/moins ;
	    bc = (float) (maxi - *b)/moins ;

	    if (*r == maxi) hue = bc - gc ;
	    else if (*g == maxi) hue = 2.0 + rc - bc ;
	    else hue = 4.0 + gc - rc ;

	    *h = (int) rint((double) hue * 60) ;
	    if (*h < 0) 
		*h = *h + 360 ;
	}

	    val_max[6] = MAX(*h, val_max[6]) ;
	    if (*h > 0)
	    val_min[6] = MIN(*h, val_min[6]) ;
	    val_max[7] = MAX(*l, val_max[7]) ;
	    val_min[7] = MIN(*l, val_min[7]) ;
	    val_max[8] = MAX(*s, val_max[8]) ;
	    val_min[8] = MIN(*s, val_min[8]) ;

	    val_min[3] = MIN(*y, val_min[3]) ;
	    val_max[3] = MAX(*y, val_max[3]) ;
	    val_min[4] = MIN(*i, val_min[4]) ;
	    val_max[4] = MAX(*i, val_max[4]) ;
	    val_min[5] = MIN(*q, val_min[5]) ;
	    val_max[5] = MAX(*q, val_max[5]) ;

	
	r++; g++; b++ ;
	y++; i++; q++ ;
	h++; l++; s++ ;
	count++ ;
    }
    write_message(oprmailtabs[4]) ;

}


/*******************************************************************/
/*								   */
/*		Procedures de calcul et segmentation		   */
/*								   */
/*******************************************************************/



/*******************************************************************/
/*								   */
/* nom      : compute_histograms				   */
/*								   */
/* fonction : calcul des histogrammes avec masque		   */
/*                                                                 */
/* entrees  : unsigned char *masque			           */
/*                                                                 */
/* globales : int val_min[9], val_max[9]			   */
/*	      *histo[9], *data[9].				   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : ---						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
compute_histograms(masque) 
unsigned char *masque ;
{
    register t = 0;
    int *im[9], i, j, taille, *tmp ;

    for (i=0 ; i<9 ; i++)
    {
	im[i] = data[i] ;
        taille = val_max[i] - val_min[i] + 1 ;
        j=0 ;

        tmp = histo[i] ;
        while(j++ <taille)
	    *tmp++ = 0 ;
    }

	while (t < taille_image)
	{
	    if (*masque)
	    {
		(*(histo[0] + (*im[0] - val_min[0])))++ ;
		(*(histo[1] + (*im[1] - val_min[1])))++ ;
		(*(histo[2] + (*im[2] - val_min[2])))++ ;
		(*(histo[3] + (*im[3] - val_min[3])))++ ;
		(*(histo[4] + (*im[4] - val_min[4])))++ ;
		(*(histo[5] + (*im[5] - val_min[5])))++ ;
		(*(histo[6] + (*im[6] - val_min[6])))++ ;
		(*(histo[7] + (*im[7] - val_min[7])))++ ;
		(*(histo[8] + (*im[8] - val_min[8])))++ ;
	    }
	    t++ ;
	    masque++;
	    im[0]++ ; im[1]++ ; im[2]++ ;
	    im[3]++ ; im[4]++ ; im[5]++ ;
	    im[6]++ ; im[7]++ ; im[8]++ ;
	}
}


/********************************************************************/
/*								    */
/* Nom : opr_mov_avg						    */
/* Fonction : 1D moving average operator			    */
/* Entree : Data    vector of data				    */
/*	    NData   nr data					    */
/*	    WSize   window size					    */
/* Global : -							    */
/* routine : -							    */
/* Sortie : AvgData filtered vector				    */
/*								    */
/* Made by Thierry PUN 13.12.1988				    */
/********************************************************************/

int *opr_mov_avg (Data, NData, WSize)

int *Data;
int NData, WSize;
{
    int i, ii, kk, ind;
    int WSm, WSp;
    int *AvgData, *ad, *dat;
    float val;

    if (WSize > 1)
    {
	AvgData = (int *) calloc (NData, sizeof (*AvgData));

        WSm = (int) rint (floor((double)WSize / 2.0));
	WSp = (int) rint (floor(((double)WSize - 1.0) / 2.0));
	ad = AvgData;

	for (ii = 0; ii < NData; ii ++)
	{
	    val = 0.0;
	    for (kk = ii - WSm; kk <= ii + WSp; kk ++)
	    {
		if (kk < 0) 
		    ind = 0;
		else if (kk >= NData) 
		    ind = NData - 1;
		else 
		    ind = kk;
		dat = Data;
		for (i = 0; i < ind; i ++)
		    dat ++;
		val = val + (float) *dat;
	    }
	    *ad = (int) rint(val / (float) WSize);
	    if (ii != (NData - 1)) ad ++;
	}
    } 
    else AvgData = Data;

    return (AvgData);
}


/********************************************************************/
/*								    */
/* Nom	    : smooth_histograms					    */
/*								    */
/* Fonction : appelle le lissage des histogrammes		    */
/*								    */
/* Entree   : int size : taille du masque de lissage		    */
/*								    */
/* Global   : int *histo[9], val_min[9], val_max[9]		    */
/*								    */
/* Routine  : ---						    */
/*								    */
/* Sortie   : ---						    */
/*								    */
/* Made by Rene W. LUTZ 24.01.1990				    */
/********************************************************************/

static void
smooth_histograms(size)
int size ;
{
    register int i ;

    for (i = 0 ; i < 9 ; i++)
	histo[i] = opr_mov_avg(histo[i], val_max[i] - val_min[i] + 1 , size) ;

}


/*******************************************************************/
/*								   */
/* nom      : fix_histogram_maximum				   */
/*								   */
/* fonction : calcul du maximum des histogrammes		   */
/*                                                                 */
/* entrees  : ---						   */
/*                                                                 */
/* globales : int val_min[9], val_max[9], *histo[9]		   */
/*								   */
/* return   : ---						   */
/*								   */
/* routines : MAX						   */
/*								   */
/* Made by Rene W. LUTZ 24.01.1990				   */
/*******************************************************************/

static void
fix_histogram_maximum()
{
    register int i,j,length, *tmp ;
    register int wrk ;
    
    for (i = 0 ; i < 9 ; i++)
    {
	j = 0 ;
	wrk = 0 ;
	tmp = histo[i] ;
	length = val_max[i] - val_min[i] + 1 ;

	while (j++ < length)
	{
	    wrk = MAX(wrk, *tmp) ;
	    tmp++ ;	
	}

	hmax[i] = wrk ;
    }
}


/********************************************************************/
/*								    */
/* nom      : second_derivativ					    */
/*								    */
/* fonction : calcule de la seconde derivee des histogrammes	    */
/*								    */
/* entree   : int *histo, size					    */
/*								    */
/* Global   : ---						    */
/*	      							    */
/* Routine  :							    */
/*	      							    */
/* Sortie   : int *tmp derivee seconde de l'histogramme		    */
/*								    */
/* Made by Rene LUTZ 07.02.90					    */
/********************************************************************/

static int
*second_derivativ(histo, size)
int *histo, size ;
{
     int *tmp, *deriv, 
     i = 0 ;

     deriv = (int *) calloc (size, sizeof (*deriv));

     tmp = deriv ;

     while(i < size)
     {
	   if (i++ == 0) 
		*deriv = 0 ;
	   else
	   {
		*deriv = *histo++ ;
		*deriv -= 2 * *histo++ ;
		*deriv++ += *histo-- ;
	   }
     }
     return(tmp) ;
}


/********************************************************************/
/*								    */
/* nom      : opr_extrema					    */
/*								    */
/* fonction : Finds extrema of a serie of data			    */
/*								    */
/* entree   : Data    vector of data				    */
/*	      NData   nr data					    */
/*								    */
/* Global   : ---						    */
/*	      							    */
/* Routine  : ---						    */
/*	      							    */
/* Sortie   : NExtr   nr extrema found				    */
/*	      PosExtr contains position of extrema		    */
/*								    */
/* Made by Thierry PUN 13.12.1988 addapted by Rene LUTZ 30.01.90    */
/********************************************************************/

struct min_max_list *opr_extrema(Data, NData, NExtr, Max)
int *Data;
int NData;
int *NExtr;
int Max ;
{
    int ii;
    int left, midl, rite;
    int *dat;
    struct min_max_list *mm, *tmp, *pt ;

    dat = Data;

    for (ii = 0 ; ii < NData ; ii++)
    {
	midl = *dat ;

	if (ii == 0) 
	    left = 0;
	else 
	    left = *(dat - 1);

	if (ii == (NData - 1)) 
	    rite = 0;
	else 
	    rite = *(dat + 1);

	dat++ ;
	    
        if (ii==0)
	{
	    mm = (struct min_max_list *) calloc (1,sizeof(*mm));
	    mm->val = 0;
            if (midl >= rite)
		mm->mmax = 1 ;
	    else
		mm->mmax = 0 ;
	    mm->next = NULL ;

	    tmp = mm ;
	    *NExtr = 1 ;
	}

	else if ( ((left > midl) && (rite > midl))  ||	        /* \/ */
	( (left < midl) && (rite < midl))  ||	                /* /\ */
	( (fabs(left-midl) > 5 ) && (rite == midl)) ||	/* >- */
	( (fabs(rite-midl) > 5 ) && (left == midl)))         /* -< */
	{
	    if(tmp->val != ii - 1)
	    {
		pt = (struct min_max_list *) calloc (1, sizeof(*pt));
		pt->val = ii ;
		if ((midl >= rite) && (midl >= left))
		    pt->mmax = 1 ;
		else
		    pt->mmax = 0 ;
		pt->next = NULL ;

		tmp->next = pt ;
		pt = NULL ;
		tmp = tmp->next ;
		(*NExtr)++;
	    }
	}
    }

    return (mm);
}



/*******************************************************************/
/*								   */
/*		Procedures de gestions des pics			   */
/*								   */
/*******************************************************************/


/********************************************************************/
/*								    */
/* Nom      : min_max_curve					    */
/*								    */
/* Fonction : construction de la liste des pics en analysant	    */
/*	      les min et maxima de la coubure de la seconde derivee */
/*								    */
/* Entree   : int *deriv : ptr sur la derive seconde		    */
/*	      int size : taille de la derivee seconde		    */
/*	      int *histo : ptr sur l'histogramme analyse 	    */
/*								    */
/* Global   : ---						    */
/*								    */
/* Routine  : ---						    */
/*								    */
/* Sortie  : struct list_pic *first_pic : ptr sur liste de pics	    */
/*								    */
/* Made by Rene W. LUTZ 07.02.1990				    */
/********************************************************************/


static void
min_max_curve (deriv, size, histo, indice)

int *deriv, size, *histo;
int indice ;
{		
    int i, *lval, *rval, *val;

	int nb_points ;
    int	dleft, dmidd, dright;
    int lpos, mpos, rpos ;
    float sum ;
     

    struct list_pic *lpic, *lpic_tmp, *cur_old, *cur, *tmp_pic ;
    int dist;
    double somme, terme, somme2, somme4 ;
    double moyenne, N ;

    i = 0 ; 
    while(i < size)
    {
        dleft = *deriv++ ;
        i++ ;
	while ((i < size) && (dleft < *deriv))
	{
	    dleft = *deriv++ ;
	    i++ ;
	}

	lpos = i - 1 ;
	dmidd = *deriv++ ;
	i++ ;

	while ((i < size) && (dmidd > *deriv))
	{
	    dmidd = *deriv++ ;
	    i++ ;	    
	}

	dright = *deriv++ ;
	mpos = i - 1 ;
	i++ ;

	while ((i < size) && (dright < *deriv))
	{
	    dright = *deriv++ ;
	    i++ ;
	}

	rpos = i - 1 ;
	dist = rpos - lpos + 1 ;

	if ((i < size) && (dist > 3))
	{
/*
	    mesure du pic par calcul du kurtosis
*/
	    somme = 0 ;
	    N = 0 ;
	
	    nb_points = 0 ;

	    for(i = 0 ; i < size ; i++)
		nb_points += *(histo + i) ;	    

	    for (i = lpos ; i <= rpos ; i++)
	    {
		somme += (double) *(histo + i)*i ;
		N += *(histo + i) ;
	    }

	    if ((N > nombre_min) && ((nb_points - N) > nombre_min))
	    {
		val = histo + mpos ;
		lval = histo + lpos ;
		rval = histo + rpos ;

		lpic = (struct list_pic *) calloc (1,sizeof(*lpic));
		lpic->pic = mpos;
		lpic->l = lpos;
		lpic->r = rpos;
		lpic->pente = (float) ((*val - *lval)/(mpos - lpos) + (*val - *rval)/(rpos - mpos)/dist) ;
		lpic->surface = (int) rint (N) ;
		lpic->indice = indice ;

		moyenne = somme/N ;
		somme4 = 0.0 ;
		somme2 = 0.0 ;

		for (i = lpos ; i <= rpos ; i++)
		{
			terme = (double) (i - moyenne) ;
			terme *= terme ;
			somme4 += (double) *(histo + i)*terme*terme ;
			somme2 += (double) *(histo + i)*terme ;
		}

		lpic->kurtosis = somme4*N/(somme2*somme2) ;
		lpic->next = NULL;

	/* construction de la liste de pics */

		if (liste_de_pics == NULL)
                    liste_de_pics = lpic ;
		else
		{
			cur_old = liste_de_pics ;
			cur = liste_de_pics->next ;
			while((cur != NULL) && (cur_old->pente >= lpic->pente))
			{
				cur_old = cur ;
				cur = cur->next ;
			}
			if (cur == NULL) 
			    cur_old->next = lpic ;
			else
			{
				lpic->next = cur ;
				cur_old->next = lpic ;
			}
		}/*else */
	    } /* if */
	}
    }
} 


/********************************************************************/
/*								    */
/* Nom	    : extract_best_peek					    */
/*								    */
/* Fonction : recherche le meilleur pic				    */
/*								    */
/* Entree   : struct list_pic *first_pic : liste de pics	    */
/*	      int *histo : histogramme du plan courant		    */
/*								    */
/* Routine  : ---						    */
/*								    */
/* Global   : ---						    */
/*								    */
/* Sortie   : struct precede preced : le best pic & sa priorite	    */
/*								    */
/* Made by Rene W. LUTZ 21.09.1990				    */
/********************************************************************/

struct precede 
extract_preced (pic)
struct list_pic *pic;
{		   
    struct precede best ;    /* le meilleur pic avec sa precedence */
    int precedence;

    while (pic != NULL)		/* parcours de la liste des pics */
    {

	/* teste avec un peu de tolerance */

	if((pic->pente > best.pente) ||
	   (((pic->pente - best.pente) > 0.2) && (pic->kurtosis -
	   best.kurtosis > 0.5)))
	{
            best.pic = pic->pic ;
            best.l = pic->l ;
            best.r = pic->r ;
	    best.kurtosis = pic->kurtosis ;
	    best.priorite = precedence;
	    best.pente = pic->pente ;
            best.indice = pic->indice ;
	    best.surface = pic->surface ;	
	}

	pic = pic->next ;

    } /* if pic <> NULL */
    return (best);
}



/*******************************************************************/
/*								   */
/*		   Procedures de segmentation			   */
/*								   */
/*******************************************************************/



/********************************************************************/
/*								    */
/* Nom	    : region_segmentation				    */
/*								    */
/* Fonction : segmentation des regions par OPR			    */
/*								    */
/* Entree   : ---						    */
/*								    */
/* Global   : unsigned char *region_image,			    */
/*	      int colour = couleur des regions		            */
/*	      int nombre_min = nb de pts minimum pour une seg.	    */
/*	      int taille_image.			    		    */
/*								    */
/* Routine  : O_P_R, show_pixrect				    */
/*	      write_message					    */
/*								    */
/* Sortie   : ---						    */
/*								    */
/* Made by Rene W. LUTZ 21.09.1990				    */
/********************************************************************/

static void
region_segmentation()
{
    register int i, verbose ;

    register int	nb_pts_region, nb_pts_reste,
			lseg, rseg,	/* bornes gauche et droite */
			pic,		/* pic retenu */
			*p,		/* pointeur sur l'image de base */
			indice;		/* numero du plan de base */

    int *deriv,		/* liste des derivees secondes */
	 size ;		/* taille de la liste */

    unsigned char	*region_mask,		/* resultat */
			*tmp_region,		/* region extraite */
			*background_mask,	/* reste de l'image */
			*tmp_background,	/* idem */
			*wrk,			/* var. de travail */
			*masque ;		/* devinez ... */

    struct precede best_pic;			/* meilleur pic */
    struct pile_masque *first, *el, *tmp_el ;   /* gestion de la pile de
masques */
    colour = 2;		    /* nb de region, pour ecrire dans region_image */

    region_image = (unsigned char *) calloc (taille_image, sizeof(*region_image));
    masque = (unsigned char *) calloc (taille_image, sizeof(*masque));

    wrk = masque ;
    i = 0 ;
    while (i++ < taille_image)
        *wrk++ = 1 ;

    first = (struct pile_masque *) calloc(1, sizeof(*first)) ;
    first->masque = masque ;
    first->nb_region = taille_image ;
    first->next = NULL ;

    nombre_min = atoi((char *) panel_get(number_min, PANEL_VALUE)) ;

    verbose = verbose_on() ;

    write_message(oprmailtabs[5]) ;

    while (first != NULL) 
    {
/***********************************************************

	Calcul des histogrammes et traitements

***********************************************************/

	masque = first->masque ;
        compute_histograms(masque) ;

	smooth_histograms((int) panel_get(threshold_slider, PANEL_VALUE)) ;
	fix_histogram_maximum() ;
        if (verbose)
	    display_histograms() ;

	if (masque_on())
	    show_masque(masque) ;

	if (liste_de_pics != NULL)
		cfree(liste_de_pics) ;
	liste_de_pics = NULL ;

	/* traitement des 9 histogrammes et contruction de liste_de_pics */

	for (i = 0 ; i < 9 ; i++)
	{
	    size = val_max[i] - val_min[i] + 1 ;
	    deriv = second_derivativ(histo[i], size) ;
	    deriv = opr_mov_avg(deriv, size, 5) ;
	    min_max_curve(deriv, size, histo[i], i) ;
	}

/***********************************************************

		extraction du meilleur pic

***********************************************************/

	if (liste_de_pics != NULL)
	{						  
	    best_pic = extract_preced(liste_de_pics);
	    pic = best_pic.pic + val_min[best_pic.indice] ;
            lseg = best_pic.l + val_min[best_pic.indice] ;	/* lseg = seuil gauche */
	    rseg = best_pic.r + val_min[best_pic.indice] ;	/* rseg = seuil droit */

	    printf("gauche : %d, pic : %d, droit : %d, plan : \"%c\", \npente : %1.6f , kurtosis : %1.6f \n\n",
	    lseg, pic, rseg, type[best_pic.indice], best_pic.pente,
	    best_pic.kurtosis );


/************************************************************************

    creation des masques r_m : region segmentee, b_m : reste a segmenter

************************************************************************/

	    background_mask = (unsigned char *) calloc (taille_image, sizeof
	    (*background_mask));

 	    region_mask = (unsigned char *) calloc (taille_image, sizeof
	    (*region_mask)) ;

            tmp_region = region_mask ;
            tmp_background = background_mask ;
	    p = data[best_pic.indice] ;

	    for (i = 0; i < taille_image; i ++)
	    {
		if (*masque)
		{
	            if ((*p >= lseg) && (*p <= rseg))
		    { 
			*background_mask = 0; /* ne pas resegmenter */
                        *region_mask = 1 ;

		    }
		    else
                    {
			*background_mask = 1; /* ce qu'il reste a segmenter */
			*region_mask = 0;	  /* en dehors de la region */
		    }
		}
                else
		{
		   *background_mask = 0 ;
                   *region_mask = 0 ;
		}
	
		background_mask++;
		region_mask++;
                masque++ ;
		p++;

	    } /* for */

	/* code (a redefinir pour utiliser la morphologie bin. */

/*
            if (smooth_on())
	    {
		    *smooth_mask = tmp_region ;
		    ouverture(256,256,smooth_mask,1) ;
                    fermeture(256,256,smoth_mask,1) ;
	    }
*/

	/* localisation us pic sur les histogrammes */

            if (verbose)
		draw_peek_location(best_pic.pic,best_pic.l,best_pic.r,best_pic.indice) ;

	/* dessin de la region dans l'image resultat */

	    paint_region(tmp_region) ;

	/* attente active eventuelle */

	    if (step_on())
                wait_proc() ;

	/* triaitement de la pile de masques */

	    first->masque = tmp_background ;

	    el = (struct pile_masque *) calloc(1, sizeof(*el)) ;
	    el->masque = tmp_region ;
	    el->next = first ;
	    first = el ;

	}   /* if pic acceptable */
	else
	    first = first->next;


    } /* while */

    write_message(oprmailtabs[6]) ;

    if (masque_on())
	show_pixrect(region_image) ;
}

