;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-sector.el -- Sector simulator, mob calc for Gnu Emacs Empire Tool (GEET)
;; Version:	 0.5
;; RCS:		 $Header: emp-sector.el,v 1.10 90/12/12 10:54:15 darrylo Exp $
;; Description:	 Various and sundry useful routines to calculate mobility
;;		 and to simulate a sector.
;; Author:	 Darryl Okahata (darrylo%hpnmd@hpcea.hp.com)
;; Created:	 Wed Oct 17 17:53:22 1990
;; Modified:     Mon Dec 10 21:30:52 1990 (Darryl Okahata) darrylo@hpsrdmo
;; Language:	 Emacs-Lisp
;; Package:	 N/A
;;
;; HISTORY 
;; 25-Jan-1991		Lynn Slater x2048	
;;    split out sector initialization calls to emp-const
;; Tue Nov 20, 1990     Darryl Okahata
;;    Sectors with zero pop will no longer spontaneously give birth.
;; 11-Nov-1990		Darryl Okahata
;;    Sector simulator now works with redes sectors.  Displays work %
;; Wed Nov  7, 1990	Darryl Okahata
;;    Added sector simulator.
;;    Added new variable `empire-prompt-for-dyna-mobility' (if non-nil, will
;;        ask for initial mobility when doing a dyna-mob).
;;    Made simulation be in empire-data-mode
;; 6-Nov-1990		Lynn Slater	
;;    added display of dest sector to dyna-mob-display.
;;    Made display be compatable in format with other map displays
;;    will not move more than starting amount
;;    will not divide by zero when moving on roads
;;    will increment destination sectors
;; TABLE OF CONTENTS
;;   empire-display-plague-chance -- Display, in the minibuffer, the estimated chance of plague for the
;;   dynamic-mobility-show-used-left -- Place the map buffer into dynamic-mobility-show-mode and place the
;;   dynamic-mobility-show-amount-can-move -- Place the map buffer into dynamic-mobility-show-mode and place the
;;   dynamic-mobility-show-mode -- Enter dynamic-mobility-show-mode.  This mode saves and replaces the
;;   dyna-mob-quit -- Exit dyna-mobility-show-mode.  Restore the old mode.  Move the cursor
;;   dyna-mob-insert-move-and-exit -- Insert a move command into the empire shell buffer and exit
;;   dyna-mob-move-j -- Dyna move in direction j
;;   dyna-mob-move-u -- Dyna move in direction u
;;   dyna-mob-move-y -- Dyna move in direction y
;;   dyna-mob-move-g -- Dyna move in direction g
;;   dyna-mob-move-b -- Dyna move in direction b
;;   dyna-mob-move-n -- Dyna move in direction n
;;   dyna-mob-move -- Dyna move in direction DIR.
;;   dyna-mob-unmove -- Undo the effects of the last move.
;;   dyna-mob-next-display-mode -- Cycle to the next display mode.
;;   dyna-mob-display -- Display, in the minibuffer, the amount of mobility used/left, or the
;;   empire-simulate-sector -- Simulate the sector at X,Y for up to MAX-ITER iterations, displaying
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; emp-sector.el -- Dynamic mobility routines for the GNU Emacs Empire tool.
;; Copyright (C) 1990 Darryl Okahata (darrylo%hpnmd@hpcea.hp.com)
;; 
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; ***** IMPORTANT NOTE *****
;;
;; The mobility calculation routines rely on a number of constants which
;; are compiled into BSD Empire.  These constants are not visible to the
;; user (they cannot be displayed using any Empire command), and so a
;; copy of these constants have been placed into this file.  The
;; constants contained within will work for a vanilla version of BSD
;; Empire 1.1, but will not work for a version of Empire for which they
;; have been modified (or for a version that uses different algorithms).
;;
;; Also note that the routines contained in this file use floating-point
;; calculations (the range of integers in GNU Emacs is too small to be
;; used), and these calculations are done using the equivalent of
;; single-precision floating-point.  As Empire does its calculations
;; using double-precision floating-point, small differences will
;; sometimes exist between Empire and the routines in this file.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; The "core" mobility calculation routines are `calc-commodity-weight-float'
;; and `calc-mob-to-sector'.  An example of how they are used can be
;; found in the routine `calc-required-mobility'.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; Note that  "(provide 'emp-sector)" is done at the END of this file.

(require 'cl)
(require 'emp-db)

;;
;; Here, we use float.el instead of Dave Gillespie's excellent GNU
;; Emacs Calc routines because float.el is significantly faster.
;;
(if (not (fboundp 'float-to-string))
  (progn
    ;;
    ;; We try loading our own faster, yet compatible, version of float.el.
    ;; If this fails, we load the normal one.
    ;;
    (condition-case nil
      (load "emp-float")
      (load "float")	;; float.el does not provide anything
    )
  )
)


;;	***** NOTE: Some initializations occur at the end of this file!


(defvar sector-constants nil
  "A list of sector types, and attributes associated with sector types.
This variable is set by the function `define-sector', and should not be
set in any other way."
)
;;(setq sector-constants nil)		;;; for debugging

(defvar commodity-constants nil
  "A list of commodity types, and attributes associated with
commodities.  This variable is set by the function `define-commodity',
and should not be set in any other way."
)
;;(setq commodity-constants nil)		;;; for debugging


(defvar sector-type-strings nil
  "A list of legal sector designations, as strings.  This variable is
set as a side-effect of `define-sector'.")


;;
;; Bitfield constants, used to determine the attributes of a sector:
;;
(defconst sector-produces-nothing	0)
(defconst sector-produces-uranium	1)
(defconst sector-produces-happiness	2)
(defconst sector-produces-guns		4)
(defconst sector-produces-shells	8)
(defconst sector-produces-iron		16)
(defconst sector-produces-dust		32)
(defconst sector-produces-food		64)
(defconst sector-produces-oil		128)
(defconst sector-produces-lcm		256)
(defconst sector-produces-hcm		512)
(defconst sector-produces-tech		1024)
(defconst sector-produces-research	2048)
(defconst sector-produces-education	4096)
(defconst sector-produces-food		8192)
(defconst sector-produces-bar		16384)
(defconst sector-produces-petrol	32768)

;;
;; indices into the commodity packaging array:
;;
(defconst empire-normal-packaging	0)
(defconst empire-warehouse-packaging	1)
(defconst empire-urban-packaging	2)	;; not used
(defconst empire-bank-packaging		3)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar empire-efficiency-float-array 
  (let (array i)
    (if (not empire-batch-play)
	(message "Initializing emp-sector efficiency vector ..."))
    (setq array (make-vector 101 nil))
    (dotimes (i 101)
      (aset array i (f i))
    )
    (if (not empire-batch-play)
	(message ""))
    array
  )
  "Precomputed vector of integer efficiency (0-100) to floats.
Precomputing the efficiency values makes calculating mobility faster, as
this is just one less int that has to be converted into a float.")

;;
;; The following macros and variables are defined here for efficiency.
;;
(defmacro get-efficiency-float (eff)
  (`
   (aref empire-efficiency-float-array (, eff))
  )
)

(if (not (boundp '_f-minus-one))
  (setq _f-minus-one (f -1))
)

(if (not (boundp '_f100))
  (setq _f100 (f 100))
)

(if (not (boundp '_f500))
  (setq _f500 (f 500))
)

(if (not (boundp '_f1/500))
  (setq _f1/500 (f/ _f1 _f500))
)

(if (not (boundp '_f-point-99))
  (setq _f-point-99 (string-to-float "0.99"))
)

(if (not (boundp '_f-point-01))
  (setq _f-point-01 (string-to-float "0.01"))
)

(if (not (boundp '_f-point-5))
  (setq _f-point-5 (string-to-float "0.5"))
)

(if (not (boundp '_f32768))
  (setq _f32768 (f 32768))
)

(if (not (boundp '_f5))
  (setq _f5 (f 5))
)

(if (not (boundp '_f3))
  (setq _f3 (f 3))
)

(if (not (boundp '_f2))
  (setq _f2 (f 2))
)

(if (not (boundp '_f10000))
  (setq _f10000 (f 10000))
)

(if (not (boundp '_f40))
  (setq _f40 (f 40))
)

(if (not (boundp '_f999))
  (setq _f999 (f 999))
)


(defun define-sector (des produces move-cost packaging-index)
  "Define a sector type and the associated attributes.  If the sector
type already exists in the database, this routine simply replaces the
existing values; if the sector type does not exist, it is inserted into
the database.  Note that this routine scales move-cost by 100."
  (let (item)
    (if (and sector-constants
	     (setq item (assoc des sector-constants))
	)
      (progn
	;; item exists -- replace existing values
	(setcdr item (vector produces move-cost
			     (f* (f move-cost) _f100) packaging-index))
      )
      (progn
	;; item does not exist -- insert new sector type definition
	(setq sector-constants (cons (cons des (vector produces move-cost
						       (f (* move-cost 100))
						       packaging-index))
				     sector-constants))
      )
    )
    ;;
    ;; Also update sector-type-strings (this provides a quick test to
    ;; see if a sector designation is valid).
    ;;
    (if (or (not sector-type-strings)
	    (not (assoc des sector-type-strings))
	)
      (setq sector-type-strings (cons (list des) sector-type-strings))
    )
  )
)


(defmacro sector-info (sector thing)
  "For the sector type given by SECTOR, return the attribute given by
THING."
  (let ( (temp-sector (make-symbol "sector"))
	 (temp-thing (make-symbol "thing"))
	 (temp-item (make-symbol "item"))
       )
    (`
     (let ( (, temp-item)
	    ( (, temp-sector) (, sector) )
	    ( (, temp-thing) (, thing) )
	  )
       (if (setq (, temp-item) (assoc (, temp-sector) sector-constants))
	 (cond
	  ( (eq (, temp-thing) 'produces) (aref (cdr (, temp-item)) 0) )
	  ( (eq (, temp-thing) 'move-cost) (aref (cdr (, temp-item)) 1) )
	  ( (eq (, temp-thing) 'move-cost-float) (aref (cdr (, temp-item)) 2) )
	  ( (eq (, temp-thing) 'packaging-index) (aref (cdr (, temp-item)) 3) )
	 )
       )
     )
    )
  )
)


(defun define-commodity (name lbs packaging)
  "Define a commodity and its assocated attributes.  If the commodity
already exists in the database, this routine simply replaces the
existing values; if the commodity does not exist, it is inserted into
the database."
  (let (item packaging-float lbs-float)
    ;;
    ;; Make a copy of packaging, so that we don't destroy the original
    ;;
    (setq packaging-float (copy-sequence packaging))
    (dotimes (i (length packaging-float))
      ;;
      ;; Convert packging info to float notation
      ;;
      (aset packaging-float i (f (aref packaging i)))
    )
    (setq lbs-float (f lbs))		;;; convert lbs to float notation
    ;;
    (if (and commodity-constants
	     (setq item (assoc name commodity-constants))
	)
      (progn
	(setcdr item (vector lbs packaging packaging-float))
      )
      (progn
	(setq commodity-constants (cons (cons name (vector lbs lbs-float
							   packaging
							   packaging-float))
					commodity-constants))
      )
    )
  )
)


(defmacro commodity-info (commodity thing)
  "For the commodity given by COMMODITY, return the attribute given by
THING."
  (let ( (temp-commodity (make-symbol "commodity"))
	 (temp-thing (make-symbol "thing"))
	 (temp-item (make-symbol "item"))
       )
    (`
     (let ( (, temp-item)
	    ( (, temp-commodity) (, commodity) )
	    ( (, temp-thing) (, thing) )
	  )
       (if (setq (, temp-item) (assoc (, temp-commodity) commodity-constants))
	 (cond
	  ( (eq (, temp-thing) 'lbs) (aref (cdr (, temp-item)) 0) )
	  ( (eq (, temp-thing) 'lbs-float) (aref (cdr (, temp-item)) 1) )
	  ( (eq (, temp-thing) 'packaging) (aref (cdr (, temp-item)) 2) )
	  ( (eq (, temp-thing) 'packaging-float) (aref (cdr (, temp-item)) 3) )
	 )
       )
     )
    )
  )
)


(defmacro dyna-mob-sanity-check (des eff)
  "Perform sanity checks on DES and EFF. In particular, if DES is `nil',
assume that it's a wilderness sector (but, if DES is non-nil, signal an
error if it is not a known sector type).  DES is the sector designation
in string form.  If EFF is `nil', assume that the efficiency is zero."
  (`
   (progn
     ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
     ;;
     ;; Sanity checks
     ;;
     (if (not (, des))
       (setq (, des) "-")		;;; Assume it's a wilderness.
					;;; Assume the user knows what he is
					;;; doing.
     )
     (if (not (, eff))
       (setq (, eff) 0))		;; Assume it's not efficent
     (if (not (assoc (, des) sector-type-strings))
       (error "What kind of sector is `%s'?" (, des)))
     ;;
     ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   )
  )
)

(defun calc-commodity-weight-float (item amount sector-des sector-efficiency)
  "Given an ITEM, AMOUNT, starting SECTOR-DES, and starting
SECTOR-EFFICIENCY, calculate the transport \"weight\".  This weight is
normally passed to `calc-mob-to-sector' to calculate the mobility needed
to move into a sector.  Note that SECTOR-DES and SECTOR-EFFICIENCY are
the des and eff of the sector from which the move is *started* (it is
the transporting sector).  ITEM is a symbol representing the name of the
commodity to move; it is not a string.

The weight is returned as a floating-point number."
  (let (packing lbs weight)
    (dyna-mob-sanity-check sector-des sector-efficiency)
    (setq lbs (commodity-info item 'lbs-float))
    (setq packing (aref (commodity-info item 'packaging-float)
			(sector-info sector-des 'packaging-index)))
    (if (< sector-efficiency 60)
      (setq packing _f1)
    )
    (setq weight (f/ (f* amount lbs) packing))
  )
)


(defun calc-mob-to-sector-float (weight new-des new-eff)
  "From WEIGHT, NEW-DES, and NEW-EFF, calculate the mobility needed to
move into a new sector.  WEIGHT is the \"weight\" of the items being
transported, as calculated by `calc-commodity-weight-float'.  NEW-DES and
NEW-EFF are the des and eff of the sector into which the items are being
moved.  NEW-DES is the des of the sector in string form.

The mobility required is returned as a floating-point number.  If it is
illegal to move into the sector, `-1' (in float form) is returned."
  (let (cost move-cost)
    (dyna-mob-sanity-check new-des new-eff)
    ;;
    ;; cost = (((move-cost * 100) - new-eff) * weight) / 500
    ;;
    ;; Below, note that move-cost is already scaled by 100.
    ;; Also note that we divide by 500 by multiplying by 1/500
    ;; (it's faster).
    ;;
    (if (f> (setq move-cost(sector-info new-des 'move-cost-float)) _f0)
      (setq cost (f* (f* (f- move-cost (get-efficiency-float new-eff))
			 weight) _f1/500))
      _f-minus-one				;;; can't move here
    )
  )
)

(defun calc-required-mobility-float (item amount start-x start-y directions)
  "Calculate the mobility needed to move ITEM, amount AMOUNT, from the
sector given by START-X and START-Y, over the path given by DIRECTIONS.
ITEM is a symbol representing the commodity to move, and DIRECTIONS is a
direction string containing one or more of [juygbn].

The mobility is returned as a floating-point number.  If the path is
illegal (goes through sea sectors, etc.), `-1' (in float form) is
returned.  Note that \"unknown\" sectors are treated as 0% wildernesses.
"
  (let (des eff weight dir (x start-x) (y start-y) mob incremental-mob)
    (catch 'error
      (setq mob _f0)
      (setq des (recall start-x start-y (position-of 'des)))
      (if (not (setq eff (recall start-x start-y (position-of 'eff))))
	(setq eff 0))
      (if (or (not des)
	      (not (assoc des sector-type-strings))
	  )
	(progn
	  ;; no way are we going to move this
	  (setq mob _f-minus-one)
	  (throw 'error nil)
	)
      )
      ;;
      ;; We only need to calculate the weight once
      ;;
      (setq weight (calc-commodity-weight-float item (f amount) des eff))
      (while (not (string= directions ""))
	(setq dir (string-to-char directions))
	(cond
	 ( (eq dir ?j)
	   (setq x (+ x 2)) )
	 ( (eq dir ?u)
	   (setq x (1+ x))
	   (setq y (1- y)) )
	 ( (eq dir ?y)
	   (setq x (1- x))
	   (setq y (1- y)) )
	 ( (eq dir ?g)
	   (setq x (- x 2)) )
	 ( (eq dir ?b)
	   (setq x (1- x))
	   (setq y (1+ y)) )
	 ( (eq dir ?n)
	   (setq x (1+ x))
	   (setq y (1+ y)) )
	)
	(setq des (recall x y (position-of 'des)))
	(setq eff (recall x y (position-of 'eff)))
	(setq incremental-mob (calc-mob-to-sector-float weight des eff))
	(if (f< incremental-mob _f0)
	  (progn
	    (setq mob _f-minus-one)
	    (throw 'error nil)
	  )
	)
	(setq mob (f+ mob incremental-mob))
	(setq directions (substring directions 1))
      )
    )
    mob
  )
)


(defun calc-required-mobility (item amount start-x start-y directions)
  "Calculate the mobility needed to move ITEM, amount AMOUNT, from the
sector given by START-X and START-Y, over the path given by DIRECTIONS.
ITEM is a symbol representing the commodity to move, and DIRECTIONS is a
direction string containing one or more of [juygbn].

If the path is illegal (goes through sea sectors, etc.), `-1' is
returned.  Note that \"unknown\" sectors are treated as 0%
wildernesses."
  (let (required-mobility)
    (if (f>= (setq required-mobility
		   (calc-required-mobility-float item amount
						 start-x start-y
						 directions))
	     _f0)
      (progn
	(setq required-mobility (fint (f+ required-mobility _f-point-99)))
      )
      -1
    )
  )
)


(defun calc-movable-amount (item start-x start-y directions
				 &optional mob-leave quick-estimate)
  "Returns the int amount of ITEM that can be moved from START-X START-Y
over the path given by DIRECTIONS.  Return as large an integer that can be
moved and still have the originating sector's mobility be larger than the
optional mob-leave (defaults to 0).  ITEM is a symbol representing the
commodity to move, and DIRECTIONS is a direction string containing one or
more of [juygbn].

If the path is illegal (goes through sea sectors, etc.), `-1' is returned.
Note that \"unknown\" sectors are treated as 0% wildernesses.
"
  (let (amount-can-move
	mob-float
	start-sector-mobility
	mob-limit
	(amount-in-start-sector
	 (recall start-x start-y (dynamic-position-of item)))
	)
    (if (not mob-leave)
      (setq mob-leave 0))
    (setq mob-limit (- (recall start-x start-y (position-of 'mob)) mob-leave))
    (setq mob-float (calc-required-mobility-float item 10 start-x start-y
						  directions))
    (if (f>= mob-float _f0)
      (progn
	(if (f> mob-float _f0)
	  (progn
	    (setq start-sector-mobility (recall start-x start-y
						(position-of 'mob)))
	    (if (> start-sector-mobility mob-limit)
		;; limit the amount of mobility we can use
	      (setq start-sector-mobility mob-limit)
	    )
	    (if quick-estimate
	      (setq amount-can-move (/ (* start-sector-mobility
					  10)
				       (fint (f+ mob-float _f1))))
	      (setq amount-can-move (fint (f/ (f* (f start-sector-mobility)
						  _f10)
					      mob-float)))
	    )
	  )
	  (setq amount-can-move amount-in-start-sector)
	)
      )
      (progn
	(setq amount-can-move -1)
      )
    )
  )
)



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Calculate chance of plague
;;
;; These routines are not the fastest -- they can probably be optimized
;; (first, we get things working; then we go back and make them faster).
;;

(defun empire-calc-plague-chance-float (civ mil uw iron oil eff mob)
  "Given the basic list of commodities, calculate the chance of plague
as a float."
  (let (f-numerator f-denominator)
    ;;
    ;;		    civ + mil + uw    /  iron + oil		  \
    ;; numerator = ---------------- * | ------------ + tech + 100 |
    ;;		         999	      \	     10		          /
    ;;
    (setq f-numerator (f* (f/ (f+ (f+ (f civ) (f mil)) (f uw))
			      _f999)
			  (f+ (f+ (f/ (f+ (f iron) (f oil)) _f10)
				  (f empire-technology-level)) _f100)))
    ;;
    ;; denominator = eff + mob + 100 + research
    ;;
    (setq f-denominator (f+ (f+ (f+ (f eff) (f mob)) _f100)
			    (f empire-research-level)))
    (f/ f-numerator f-denominator)
  )
)

(defun empire-calc-plague-chance (x y)
  "Calculate the approximate chance of plague in sector X,Y.  An
integer number in the range 0 to 100 is returned.  A negative number
is returned if the chance cannot be calculated."
  (let (civ mil uw iron oil eff mob)
    (setq civ (recall x y (position-of 'civ))
	  mil (recall x y (position-of 'mil))
	  uw (recall x y (position-of 'uw))
	  iron (recall x y (position-of 'iron))
	  oil (recall x y (position-of 'oil))
	  eff (recall x y (position-of 'eff))
	  mob (recall x y (position-of 'mob))
	  )
    (if (and civ mil uw iron oil eff mob)
      (progn
	(fint (empire-calc-plague-chance-float civ mil uw iron oil eff mob))
      )
      (progn
	-1
      )
    )
  )
)

(defun empire-display-plague-chance (x y)
  "Display, in the minibuffer, the estimated chance of plague for the
sector at X,Y.  If this function is run interactively, the chance for
the current map sector is displayed."
  (interactive (let ( (sect (get-map-sector)) )
		 (list (car sect) (cdr sect))
		 )
	       )
  (let (chance)
    (setq chance (empire-calc-plague-chance x y))
    (message "Estimated chance of plague = %s%% (%s eff %s at %s,%s)"
	     chance
	     (recall x y (position-of 'eff)) (recall x y (position-of 'des))
	     x y)
  )
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; None of the following variables are user-settable.

(defvar dyna-mob-display-mode 0
  "The number of the mode in which the calculated mobility is to be
displayed.  Currently, the following modes have been implemented:
	0 Display mob used/left and the starting mob
	1 Display the maximum amount of the item that can be moved.
")

(defconst dyna-mob-max-display-modes 2
  "The maximum number of display modes.  This number is used to cycle
through the list of display modes.")

(defvar dyna-mob-start-x 0
  "The X coordinate of the transporting sector.")

(defvar dyna-mob-start-y 0
  "The Y coordinate of the transporting sector.")

(defvar dyna-mob-start-mobility 0
  "The \"initial\" mobility of the starting sector, as an integer.  It
is stored here so that we don't have to constantly refer to the sector
database.  Note that this \"initial\" mobility may be different from the
actual amount in a sector, if the user wants to see how much of an item
can be moved given a certain amount of mobility.")

(defvar dyna-mob-start-mobility-float 0
  "The \"initial\" mobility of the sector, as a floating-point number.
This number is precalculated and stored here so that we don't have to
constantly make expensive calls to convert ints to floats.")

(defvar dyna-mob-item nil
  "This is the item being moved, as a symbol.")

(defvar dyna-mob-amount 0
  "The amount of the item being moved, as an integer.")

(defvar dyna-mob-amount-float _f0
  "The amount of the item being moved, as an floating-point number.
This number is precalculated and stored here so that we don't have to
constantly make expensive calls to convert ints to floats.")

(defvar dyna-mob-xypath nil
  "A list of previous X,Y and mobility used at that X,Y location.  This
list is maintained by `dyna-mob-move' and `dyna-mob-unmove'.")

(defvar dyna-mob-current-x 0
  "The current X position of the cursor.")

(defvar dyna-mob-current-y 0
  "The current Y position of the cursor.")

(defvar dyna-mob-weight _f0
  "The \"weight\" of the items being moved, as a floating-point number.")

(defvar dyna-mob-current-mob _f0
  "The amount of mobility used in reaching the current cursor position,
as a floating-point number.")

(defvar dyna-mob-old-mode-name nil
  "The mode-name of the map buffer, before dyna-mob-mode was activated.")

(defvar dyna-mob-old-minor-mode-alist nil
  "The minor-mode-alist of the map buffer, before dyna-mob-mode was
activated.")

(defvar dyna-mob-old-map nil
  "The old local keymap of the map buffer, before dyna-mob-mode was
activated.")

(defvar dyna-mob-keymap nil
  "The local keymap used by dyna-mob-mode.")


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Define the local dyna-mob-keymap
;;
(progn
  (setq dyna-mob-keymap (make-keymap))
  (suppress-keymap dyna-mob-keymap t)

  (define-key dyna-mob-keymap help-character 'help-for-empire-dyna-mob)

  (define-key dyna-mob-keymap "\C-?" 'dyna-mob-unmove)
  (if (not (equal help-char ?\C-h))
    (define-key dyna-mob-keymap "\C-h" 'dyna-mob-unmove)
  )

  (define-key dyna-mob-keymap "j" 'dyna-mob-move-j)
  (define-key dyna-mob-keymap "u" 'dyna-mob-move-u)
  (define-key dyna-mob-keymap "y" 'dyna-mob-move-y)
  (define-key dyna-mob-keymap "g" 'dyna-mob-move-g)
  (define-key dyna-mob-keymap "b" 'dyna-mob-move-b)
  (define-key dyna-mob-keymap "n" 'dyna-mob-move-n)

  (define-key dyna-mob-keymap "J" 'dyna-mob-move-j)
  (define-key dyna-mob-keymap "U" 'dyna-mob-move-u)
  (define-key dyna-mob-keymap "Y" 'dyna-mob-move-y)
  (define-key dyna-mob-keymap "G" 'dyna-mob-move-g)
  (define-key dyna-mob-keymap "B" 'dyna-mob-move-b)
  (define-key dyna-mob-keymap "N" 'dyna-mob-move-n)

  (define-key dyna-mob-keymap "q" 'dyna-mob-quit)
  (define-key dyna-mob-keymap "h" 'dyna-mob-quit)
  (define-key dyna-mob-keymap "x" 'dyna-mob-quit)

  (define-key dyna-mob-keymap " " 'dyna-mob-next-display-mode)
  (define-key dyna-mob-keymap "\C-m" 'dyna-mob-display)
  (define-key dyna-mob-keymap "\C-i" 'dyna-mob-display)
  (define-key dyna-mob-keymap "." 'display-map-sector-info)

  (define-key dyna-mob-keymap "!" 'dyna-mob-insert-move-and-exit)

  (define-key dyna-mob-keymap "?" 'help-for-empire-dyna-mob)
)


(defun dynamic-mobility-show-used-left ()
  "Place the map buffer into dynamic-mobility-show-mode and place the
display into the mode where the amount of mobility used/left is
displayed."
  (interactive)
  (setq dyna-mob-display-mode 0)
  (call-interactively 'dynamic-mobility-show-mode)
)


(defun dynamic-mobility-show-amount-can-move ()
  "Place the map buffer into dynamic-mobility-show-mode and place the
display into the mode where the amount of mobility used/left is
displayed."
  (interactive)
  (setq dyna-mob-display-mode 1)
  (call-interactively 'dynamic-mobility-show-mode)
)


(make-help-screen help-for-empire-dyna-mob
		  "j u y g b n . ! [Space] [Return] [TAB]"
		  "You have discovered the empire dynamic mobility calculator.
You can use the following commands from here:

j u y g b n
J U Y G B N
	Move the map cursor in the indicated direction and show the
	amount of mobility needed, relative to the starting sector, to
	enter the new sector.

h q x	Exit dynamic-mobility-show-mode.  The cursor is moved back to
	the starting sector.

[Space] Toggle between showing the amount of mobility used/left and the
	total amount that can be moved.

.	Show the sector info of the sector under the cursor.

[Return]
[TAB]
	Redisplay the amount of mobility used/left or the total amount
	that can be moved.

!	Insert into the empire shell buffer, but do not execute, a move
	command that contains the path currently traversed.  Also, exit
	dynamic-mobility-show-mode, but do not move the cursor to the
	original location; leave it where it is.

"
	dyna-mob-keymap)


(defun dynamic-mobility-show-mode (start-x start-y initial-mobility argp)
  "Enter dynamic-mobility-show-mode.  This mode saves and replaces the
existing mode in the map buffer.  In this mode, you are prompted for an
item, an amount to move from the current map sector, and the starting
mobility (the default is the amount of mobility left in the sector), and
the mobility used/left is displayed as you move the cursor over the map.
Pressing the space bar toggles between displaying mobility used/left and
displaying the amount that can be moved given the available mobility.

If you want to see how much of an item can be moved given a particular
amount of mobility, enter that amount when prompted for the mobility.

If given a univeral arg (\C-u), will ask for the mobility it is allowed to
consume. Othewise will use all mobility from sector.

The full dynamic-mobility-show-mode key bindings are:
\\{dyna-mob-keymap}"
  (interactive (let ( (sect (get-map-sector)) )
		 (list (car sect) (cdr sect) nil current-prefix-arg)
		 )
	       )
  (let (info des eff)
    (pop-to-buffer empire-map-buffer)
    (show-pt (cons start-x start-y) t)
    (setq info (empire-prompting-read-items start-x start-y "dyna move"
					    empire-complete-commodity-list))
    (setq dyna-mob-item (nth 0 info))
    (if (<= (setq dyna-mob-amount (nth 1 info)) 0)
      (error "You must move at least 1 %s" dyna-mob-item))
    (setq dyna-mob-amount-float (f dyna-mob-amount))
    (setq des (recall start-x start-y (position-of 'des)))
    (if (or (not des)
	    (not (assoc des sector-type-strings))
	)
      (if des
	(error "What kind of sector is `%s'?" des)
	(error "What kind of sector is this?")
      )
    )
    (setq eff (recall start-x start-y (position-of 'eff)))
    (setq dyna-mob-weight (calc-commodity-weight-float (read dyna-mob-item)
						       dyna-mob-amount-float
						       des eff))
    (setq dyna-mob-current-x start-x)
    (setq dyna-mob-current-y start-y)
    (setq dyna-mob-start-x start-x)
    (setq dyna-mob-start-y start-y)
    (if (not initial-mobility)
      (progn
	;;
	;; If the initial-mobility is not given, prompt the user for it.
	;;
	(setq initial-mobility (recall start-x start-y (position-of 'mob)))
	(if argp
	  (setq initial-mobility (empire-read-number-from-minibuffer
				  (format "Max mobility to use up [%s]? "
					  initial-mobility)
				  nil initial-mobility)))
	)
    )
;;    (if (<= initial-mobility 0)
;;      (error "The initial mobility must be greater than zero."))
    (setq dyna-mob-start-mobility initial-mobility)
    (setq dyna-mob-start-mobility-float (f dyna-mob-start-mobility))
    (setq dyna-mob-old-map (current-local-map))
    (setq dyna-mob-current-mob _f0)
    (setq dyna-mob-xypath nil)
    (setq dyna-mob-old-mode-name mode-name)
    (setq dyna-mob-old-minor-mode-alist minor-mode-alist)
    (setq mode-name "Dynamic Mobility Mode")
    (setq minor-mode-alist nil)
    (use-local-map dyna-mob-keymap)
    (dyna-mob-display)
  )
)


(defun dyna-mob-quit (&optional no-restore)
  "Exit dyna-mobility-show-mode.  Restore the old mode.  Move the cursor
to the starting location, if NO-RESTORE is `nil'."
  (interactive)
  (let ()
    (if dyna-mob-old-map
      (progn
	(use-local-map dyna-mob-old-map)
      )
    )
    (setq mode-name dyna-mob-old-mode-name)
    (setq minor-mode-alist dyna-mob-old-minor-mode-alist)
    ;;
    ;; go back to the original sector
    ;;
    (if (not no-restore)
      (show-pt (cons dyna-mob-start-x dyna-mob-start-y) t))
    (message "Dyna move done")
  )
)


(defun dyna-mob-get-path-string ()
  "Get the string form representation of the path over which the user
has moved."
  (let (node (path "") )
    (dolist (node dyna-mob-xypath)
      (setq path (format "%s%s" (nth 3 node) path))
    )
    path
  )
)


(defun dyna-mob-insert-move-and-exit ()
  "Insert a move command into the empire shell buffer and exit
dynamic-mobility-show-mode.  The amount to move is determined by the
current display mode."
  (interactive)
  (let ( (path (dyna-mob-get-path-string))
	 amount
	 amount-in-sector
	 dest
	 mob-used
       )
    (if (not (string= path ""))
      (progn
	 (if (eq dyna-mob-display-mode 0)
	   (progn
	     ;;
	     ;; The display is currently showing the amount of mobility
	     ;; left/used.  We therefore insert the amount that the user
	     ;; specified to move.
	     ;;
	     (setq amount dyna-mob-amount)
	     (setq mob-used (1+ (fint dyna-mob-current-mob)))
	   )
	   (progn
	     ;;
	     ;; The display is currently showing the amount of the item that
	     ;; can be moved.  We therefore insert the amount that can be
	     ;; moved, and the amount that the user originally specified.
	     ;;
	     (setq amount
		   (if (f> dyna-mob-current-mob _f0)
		     (min dyna-mob-amount
			  (fint (f/ (f* dyna-mob-start-mobility-float
					dyna-mob-amount-float)
				    dyna-mob-current-mob)))
		     dyna-mob-amount))
	     (setq mob-used (1+ (fint (f/ (f* (f amount)
					      dyna-mob-current-mob)
					  dyna-mob-amount-float))))
	   )
	 )
	 (move-to-empire-shell-prompt)
	 (visible-insert-in-empire-buffer (format "mov %s %s,%s %s %sh"
						  dyna-mob-item
						  dyna-mob-start-x
						  dyna-mob-start-y
						  amount path)
					  nil t)
	 (decriment-sector dyna-mob-start-x dyna-mob-start-y
			        (dynamic-position-of (read dyna-mob-item))
				amount)
	 (decriment-sector dyna-mob-start-x dyna-mob-start-y
			        (position-of mob)
				mob-used)

	 ;;
	 ;; Do not let mobility go negative
	 ;;
	 (if (< (recall dyna-mob-start-x dyna-mob-start-y (position-of mob))
		0)
	   (record dyna-mob-start-x dyna-mob-start-y (position-of mob) 0)
	 )

	 (setq dest (car (last (path-to-r path (cons dyna-mob-start-x
						     dyna-mob-start-y)))))
	 (incriment-sector (car dest) (cdr dest)
			        (dynamic-position-of (read dyna-mob-item))
				amount)
      )
    )
    (dyna-mob-quit t)
  )
)


(defun dyna-mob-move-j ()
  "Dyna move in direction j"
  (interactive)
  (dyna-mob-move 'j))


(defun dyna-mob-move-u ()
  "Dyna move in direction u"
  (interactive)
  (dyna-mob-move 'u))


(defun dyna-mob-move-y ()
  "Dyna move in direction y"
  (interactive)
  (dyna-mob-move 'y))


(defun dyna-mob-move-g ()
  "Dyna move in direction g"
  (interactive)
  (dyna-mob-move 'g))


(defun dyna-mob-move-b ()
  "Dyna move in direction b"
  (interactive)
  (dyna-mob-move 'b))


(defun dyna-mob-move-n ()
  "Dyna move in direction n"
  (interactive)
  (dyna-mob-move 'n))

(defun dyna-mob-move (dir)
  "Dyna move in direction DIR."
  (interactive)
  (let (new-x new-y new-des new-eff incremental-mob new-mob)
    (cond
     ( (eq dir 'j)
       (setq new-x (+ dyna-mob-current-x 2))
       (setq new-y dyna-mob-current-y) )
     ( (eq dir 'u)
       (setq new-x (1+ dyna-mob-current-x))
       (setq new-y (1- dyna-mob-current-y)) )
     ( (eq dir 'y)
       (setq new-x (1- dyna-mob-current-x))
       (setq new-y (1- dyna-mob-current-y)) )
     ( (eq dir 'g)
       (setq new-x (- dyna-mob-current-x 2))
       (setq new-y dyna-mob-current-y) )
     ( (eq dir 'b)
       (setq new-x (1- dyna-mob-current-x))
       (setq new-y (1+ dyna-mob-current-y)) )
     ( (eq dir 'n)
       (setq new-x (1+ dyna-mob-current-x))
       (setq new-y (1+ dyna-mob-current-y)) )
     ( t
       (error "huh?"))
    )
    (setq new-des (recall new-x new-y (position-of 'des)))
    (setq new-eff (recall new-x new-y (position-of 'eff)))
    (setq incremental-mob (calc-mob-to-sector-float dyna-mob-weight
						    new-des new-eff))
    (if (f< incremental-mob _f0)
      (error "You can't go there.")
    )
    (setq new-mob (f+ dyna-mob-current-mob incremental-mob))
    (setq dyna-mob-xypath (cons
			   (list dyna-mob-current-x
				 dyna-mob-current-y
				 dyna-mob-current-mob
				 dir)
			   dyna-mob-xypath
			  ))
    (setq dyna-mob-current-x new-x)
    (setq dyna-mob-current-y new-y)
    (setq dyna-mob-current-mob new-mob)
    (show-pt (cons dyna-mob-current-x dyna-mob-current-y) t)
    (dyna-mob-display)
  )
)

(defun dyna-mob-unmove ()
  "Undo the effects of the last move."
  (interactive)
  (let (last x y mob)
    (if (not dyna-mob-xypath)
      (error "You can't backup any farther.")
    )
    (setq last (car dyna-mob-xypath))
    (setq x (nth 0 last))
    (setq y (nth 1 last))
    (setq mob (nth 2 last))
    (setq dyna-mob-current-x x)
    (setq dyna-mob-current-y y)
    (setq dyna-mob-current-mob mob)
    (setq dyna-mob-xypath (cdr dyna-mob-xypath))
    (show-pt (cons x y) t)
    (dyna-mob-display)
  )
)


(defun dyna-mob-next-display-mode ()
  "Cycle to the next display mode."
  (interactive)
  (let ()
    (setq dyna-mob-display-mode (mod (1+ dyna-mob-display-mode)
				     dyna-mob-max-display-modes))
    (dyna-mob-display)
  )
)


(defun dyna-mob-get-rounded-mob-string (mob)
  "Given a floating-point value of mobility, MOB, round it off to the
tenths place, and return it as a string."
  (let (units decimal)
    ;;
    ;; The strange things we do for speed ....
    ;;
    ;; Note that `fint' is expensive.
    ;;
    (if (f< mob _f0)
      (progn
	(setq mob (format "%s" (fint (f- (f* mob _f10) _f1/2))))
      )
      (progn
	(setq mob (format "%s" (fint (f+ (f* mob _f10) _f1/2))))
      )
    )
    (setq units (substring mob 0 -1))
    (setq decimal (substring mob -1 nil))
    (if (or (string= units "")
	    (string= units "-")
	)
      (progn
	(setq mob (concat units "0." decimal))
      )
      (progn
	(setq mob (concat units "." decimal))
      )
    )
  )
)

(defun dyna-mob-display ()
  "Display, in the minibuffer, the amount of mobility used/left, or the
amount that can be currently moved."
  (interactive)
  (let (header trailer mob units decimal start-mob start-des start-eff
	       x y stop-des stop-eff)
    ;;
    ;; We go through these expensive display routines only if there is
    ;; no pending input.
    ;;
    (if (not (input-pending-p))
      (progn
	(setq x (map-x)
	      y (map-y)
	      stop-des (recall x y (position-of 'des))
	      stop-eff (recall x y (position-of 'eff)))
	(setq start-des (recall dyna-mob-start-x dyna-mob-start-y
				(position-of 'des)))
	(setq start-eff (recall dyna-mob-start-x dyna-mob-start-y
				(position-of 'eff)))
	(setq start-mob dyna-mob-start-mobility)
	(setq header (format "Moving %s %s from %s,%s (%s %s%%) to %s,%s (%s %s%%)"
			     dyna-mob-amount dyna-mob-item
			     dyna-mob-start-x dyna-mob-start-y
			     start-des start-eff
			     x y
			     stop-des stop-eff
			     ))
	(cond
	 ( (eq dyna-mob-display-mode 0)	;;; Mobility left/used
	   (setq mob (f- dyna-mob-start-mobility-float dyna-mob-current-mob))
	   (setq trailer (format "mobility used/left = %s/%s (%s)"
				 (dyna-mob-get-rounded-mob-string
				  dyna-mob-current-mob)
				 (dyna-mob-get-rounded-mob-string mob)
				 dyna-mob-start-mobility))
	   )
	 ( (eq dyna-mob-display-mode 1)	;;; Amount can move
	   (if (f= dyna-mob-current-mob _f0)
	     (setq decimal "any amount")
	     (setq decimal (fint (f/ (f* dyna-mob-start-mobility-float
					 dyna-mob-amount-float)
				     dyna-mob-current-mob)))
	   )
	   (setq trailer (format "can move %s" decimal))
	   )
	 ( t
	   (error "Unknown display mode")
	   )
	)
	(message "%s: %s" header trailer)
      )
    )
  )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; The sector simulator ...
;;


(defvar emp-sector-x 0
  "The X coordinate of the sector being simulated.")

(defvar emp-sector-y 0
  "The Y coordinate of the sector being simulated.")

(defvar emp-sector-des nil
  "The current designation of the sector being simulated.")

(defvar emp-sector-newdes nil
  "The new designation of the sector being simulated.")

(defvar emp-sector-eff 0
  "The current efficiency of the sector being simulated.")

(defvar emp-sector-food 10000
  "The current amount of food of the sector being simulated.")

(defvar emp-sector-civ 0
  "The current amount of civilians of the sector being simulated.")

(defvar emp-sector-mil 0
  "The current amount of military of the sector being simulated.")

(defvar emp-sector-uw 0
  "The current amount of uncompensated workers of the sector being simulated.")

(defvar emp-sector-mob 0
  "The current amount of mobility of the sector being simulated.")

(defvar emp-sector-work 100
  "The current level of work efficiency of the sector being simulated.")

(defvar emp-sector-loyalty 100
  "The current level of loyalty of the sector being simulated.")

(defvar emp-sector-money-used 0
  "The amount of money spent in the current simulation interation.")

(defvar emp-sector-total-money-used 0
  "The total amount of money spent in the current simulation.")

(defvar emp-sector-food-grown 0
  "The amount of emergency food rations grown in the current simulation
iteration.  This is nonzero only if there is insufficient food to feed
the people in the sector.")

(defvar emp-sector-food-eaten 0
  "The amount of food eaten in the current simulation iteration.")

(defvar emp-sector-starved 0
  "The amount of people that starved in the current simulation iteration.")

(defvar emp-sector-civ-growth-limited nil
  "Is non-nil, if the civilian population growth was limited for some
reason (not enough food, population exceeds sector capacity, etc.)")

(defvar emp-sector-uw-growth-limited nil
  "Is non-nil, if the uncompensated worker population growth was limited
for some reason (not enough food, population exceeds sector capacity,
etc.)")

(defvar emp-sector-iteration 0
  "The iteration number of the current simulation iteration.  Iteration
zero contains the initial starting values.  Iterations 1 and up are the
actual sector simulation iterations.")

(defvar emp-tech-level-float _f0
  "The current tech level of the country, as a float.")

(defvar emp-research-level-float _f0
  "The current research level of the country, as a float.")

(defvar emp-edu-level-float _f0
  "The current education level of the country, as a float.")

(defvar emp-happiness-level-float _f0
  "The current happiness level of the country, as a float.")

(defvar emp-sector-etu-per-update-float (f 8)
  "The number of empire time units per update, as a float.")

(defvar emp-sector-eatrate (string-to-float "0.0010")
  "The rate at which people eat food, as a float.  This value is the
result of `F/P', where F is the amount of food eaten by P people in one
ETU.")

(defvar emp-sector-babyeat (string-to-float "0.0120")
  "The rate at which babies eat food, as a float.  This value is the
result of `F/P', where F is the amount of food eaten by P babies in one
ETU.")

(defvar emp-sector-fgrate (string-to-float "0.0012")
  "The rate at which food is grown, as a float.  This value is the
result of `F/E', where F is the amount of food grown by an `E' fertility
sector in one ETU.")

(defvar emp-sector-fcrate (string-to-float "0.0013")
  "The rate at which food is harvested by people, as a float.  This
value is the result of `F/P', where F is the amount of food harvested
by P people in one ETU.")

(defvar emp-sector-obrate (string-to-float "0.005")
  "The rate at which civilians make babies, as a float.  This value is
the result of `B/P', where B is the number of babies born to P civilians
in one ETU.")

(defvar emp-sector-uwbrate (string-to-float "0.0025")
  "The rate at which uncompensated workers make babies, as a float.
This value is the result of `B/P', where B is the number of babies born
to P uncompensated workers in one ETU.")

(defvar emp-simulation-setup-hook nil
  "If non-nil, this is the function to call just before a simulation is
done.  If given, this function should perform any necessary
initializations.  This function may also want to display information to
the user, such as the initial starting values (but this is not
required).")

(defvar emp-simulation-hook 'emp-sector-end-simulation-p
  "This is the function to call after each iteration, and it must be
specified.  This function controls how long the simulation runs.  If
this function returns non-nil, the simulation stops, and
`simulate-sector' will return this non-nil value back to the caller.  If
nil is returned, the simulation will continue.  This function may also
want to display the current values of civ, uw, food, etc..")

(defvar emp-simulation-finish-hook nil
  "If non-nil, this is the function to call when a simulation has
completed.  The purpose of this optional function to perform any
necessary cleanup (for whatever was initialized by
`emp-simulation-setup-hook') and to summarize the results of the
simulation, if necessary.)")


(defmacro empire-random ()
  "This macro should expand into the function that should be called to
return a random integer in the range of [0 .. 32768]."
  '(random)
)

(defun empire-chance (val-float)
  "Generate a random number in the range `[ 0 .. )'.  If the generated
number is greater than VAL-FLOAT, return `1', as a float; else return a
float `0'."
  (let (roll)
    (setq roll (mod (empire-random) 32767))
    (if (f>= (f* val-float _f32768) (f roll))
      _f1
      _f0
    )
  )
)


(defun empire-roundavg (val-float)
  "Round out VAL-FLOAT to the next higher or lower integer, and return
this integer.  The chance of rounding upwards depends on the fractional
part of VAL-FLOAT."
  (let (flr)
    (setq flr (ftrunc val-float))
    (if (f< flr _f0)
      (setq flr (f- flr (empire-chance (f- flr val-float))))
      (setq flr (f+ flr (empire-chance (f- val-float flr))))
    )
    (fint flr)
  )
)


(defun get-workforce (civ mil uw work)
  "Get the workforce, as an integer, available to do work."
  (let (workforce)
;    ;;
;    ;; First, we play games with loyalty
;    ;;
;    (setq pct (f+ (f/ (f- empire-tech-level-float _f40) _f40)
;		  (f/ empire-edu-level-float _f3)))
;    (if (and (f< emp-happiness-level-float pct)
;	     (not (f= (chance (f/ (f- pct emp-happiness-level-float)
;				  _f5)) _f0))
;	)
;      (progn
;	
;      )
;    )
    ;;
    ;;		   (civ * work)   2 * mil
    ;;		   ------------ + ------- + uw
    ;;			100	     5
    ;; workforce = ------------------------------ * etu-per-update
    ;;				100
    ;;
    ;; In the following, we multiply the numerator and demoninator by 100
    ;; to avoid doing two divisions.
    ;;
    (setq workforce (empire-roundavg (f/ (f* (f+ (f+ (f* (f civ) (f work))
						     (f* (f mil) _f40))
						 (f* (f uw) _f100))
					     emp-sector-etu-per-update-float)
					 _f10000)))
    workforce
  )
)

(defun buildeff (workforce)
  "Build a sector's efficiency.  The cost to build efficiency is
returned, as an integer."
  (let ( work-cost
	 (total-cost 0)
	 n
       )
    (if (not (string= emp-sector-newdes "_"))
      (progn
	(if (> (setq work-cost (/ (+ emp-sector-eff 3) 4)) workforce)
	  (setq work-cost workforce))
	(if (< (setq n (- emp-sector-eff (* work-cost 4))) 0)
	  (setq n 0)
	)
	(setq emp-sector-eff n)
	(if (eq n 0)
	  (progn
	    (setq emp-sector-des emp-sector-newdes)
	    (setq emp-sector-newdes "_")
	  )
	)
	(setq workforce (- workforce work-cost))
	(setq total-cost (+ total-cost work-cost))
      )
    )
    (if (string= emp-sector-newdes "_")
      (progn
	(if (> (setq work-cost (- 100 emp-sector-eff)) workforce)
	  (setq work-cost workforce))
	(setq emp-sector-eff (+ emp-sector-eff work-cost))
	(setq total-cost (+ total-cost work-cost))
      )
    )
    total-cost
  )
)


(defun give-birth (current-pop birth-rate growth-limited)
  "Give birth to people."
  (let ( (newciv 0) new-birth new-food)
    (if (> current-pop 0)
      (progn
	(setq new-birth (empire-roundavg (f* (f* birth-rate (f current-pop))
					     emp-sector-etu-per-update-float)))
	(setq new-food (fint (f+ _f-point-5
				 (f/ (f emp-sector-food)
				     (f* _f2 emp-sector-babyeat)))))
	(setq newciv new-birth)
	(if (> newciv new-food)
	  (progn
	    (setq newciv new-food)
	    (set growth-limited t)
	  )
	)
      )
    )
    newciv
  )
)


(defun grow-people ()
  "Create new civ and uw, if possible.  Return the total number of new
civ and uw, as an integer."
  (let ( (newciv 0)
	 (newuw 0)
	 food-eaten
	 )
    ;;
    ;; Do civilians
    ;;
    (setq emp-sector-civ-growth-limited nil)
    (if (< emp-sector-civ 999)
      (progn
	(setq newciv (give-birth emp-sector-civ emp-sector-obrate
				 'emp-sector-civ-growth-limited))
	(setq emp-sector-civ (+ emp-sector-civ newciv))
      )
    )
    (if (> emp-sector-civ 999)
      (progn
	(setq emp-sector-civ 999)
	(setq emp-sector-civ-growth-limited t)
      )
    )
    ;;
    ;; Do uw
    ;;
    (setq emp-sector-uw-growth-limited nil)
    (if (< emp-sector-uw 999)
      (progn
	(setq newuw (give-birth emp-sector-uw emp-sector-uwbrate
				'emp-sector-uw-growth-limited))
	(setq emp-sector-uw (+ emp-sector-uw newuw))
      )
      (setq emp-sector-uw-growth-limited t)
    )
    (if (> emp-sector-uw 999)
      (progn
	(setq emp-sector-uw 999)
	(setq emp-sector-uw-growth-limited t)
      )
    )
    ;;
    ;; Babies eat food ...
    ;;
    (if (or (> newciv 0) (> newuw 0))
      (progn
	(setq food-eaten (empire-roundavg (f* (f (+ newciv newuw))
					      emp-sector-babyeat)))
	(setq emp-sector-food-eaten (+ emp-sector-food-eaten food-eaten))
	(setq emp-sector-food (- emp-sector-food food-eaten))
      )
    )
    (+ newciv newuw)
  )
)

(defun growfood (work fertil-float)
  "Grow emergency rations (food).  This function is called only if there
is insufficient food in the sector to feed the people.  Return the
amount of emergency rations grown."
  (let (food-workers food)
    (setq food-workers (f* (f work) emp-sector-fcrate))
    (setq food (f* (f* emp-sector-etu-per-update-float
		       fertil-float)
		   emp-sector-fgrate))
    (if (f> food food-workers)
      (setq food food-workers))
    (setq emp-sector-food (+ emp-sector-food (fint food)))
    (if (= emp-sector-food 0)
      (setq emp-sector-food 1))
    (if (> emp-sector-food 999)
      (setq emp-sector-food 999))
    (fint (f/ food emp-sector-fcrate))
  )
)

(defun feed-people (people)
  "Feed people.  If there is not enough food, starve the people.  Return
the number of people that starved."
  (let (food-eaten starved food-float people-left)
    (setq food-float (f emp-sector-food))
    (setq food-eaten (f* (f* emp-sector-etu-per-update-float
			     emp-sector-eatrate)
			 (f people)))
    (setq starved 0)
    (if (f> food-eaten food-float)
      (progn
	(setq people-left (f/ (f+ food-float _f-point-01)
			      (f+ food-eaten _f-point-01)))
	(setq starved people)
	(if (f< people-left _f-point-5)
	  (setq people-left _f-point-5))
	(setq emp-sector-civ (fint (f* (f emp-sector-civ) people-left)))
	(setq emp-sector-mil (fint (f* (f emp-sector-mil) people-left)))
	(setq emp-sector-uw (fint (f* (f emp-sector-uw) people-left)))
	(setq starved (- starved emp-sector-civ
			 emp-sector-mil emp-sector-uw))
	(setq emp-sector-food 0)
      )
      (progn
	(setq food-eaten (empire-roundavg food-eaten))
	(setq emp-sector-food-eaten (+ emp-sector-food-eaten food-eaten))
	(setq emp-sector-food (- emp-sector-food food-eaten))
      )
    )
    starved
  )
)

(defun starvation ()
  "Zap the variables that are zapped if starvation occurs."
  (let ()
    (setq emp-sector-work 0)
    (setq emp-sector-loyalty 0)
  )
)

(defun simulate-sector (x y)

  "Simulate the sector at X,Y.  Currently, the only sector variables
that are simulated are:

	efficiency
	civilians
	uncompensated workers
	food (the amount of food decreases if there is no distribution
		path; if there is a distribution path, it is assumed
		that infinite food exists, and this variable will not
		change.)
	cost (money used by this sector in increasing efficiency making
		babies)

Unfortunately, this code does not simulate production (food, iron, lcm,
etc.) as the necessary information is sometimes not kept in the sector
database.  Perhaps someday ....

The sector simulator iterates upon a sector, and each iteration
represents one update.  The initial sector data values are taken from
the sector database (see below for a list of variables).

The following hooks control the sector simulator (note that these hooks
can only contain *ONE* function; a list of functions are not allowed):

	emp-simulation-setup-hook
		Called just before the sector simulation begins.
		Optional (can be nil).  Used to initialize any necessary
		hook variables, and possibly display the initial
		starting values.
	emp-simulation-hook
		Called after each iteration (update).  Required (cannot
		be nil).  This function controls how long the simulation
		runs.  If this function returns non-nil, the simulation
		stops, and `simulate-sector' will return this non-nil
		value back to the caller.  If nil is returned, the
		simulation will continue.  This function may also want
		to display the current values of civ, uw, food, etc..
	emp-simulation-finish-hook
		Called just after the sector simulation has completed.
		Optional (can be nil).  Used to perform any required
		cleanup, and possibly summarize the results of the
		simulation.

It is the programmer's responsibility to properly set these hooks before
calling `simulate-sector'.

The following variables can be read when a hook is called (it is
strongly recommended that none of these variables be changed):

	emp-sector-iteration
		The iteration number of the current simulation
		iteration.  Iteration zero contains the initial starting
		values.  Iterations 1 and up are the actual sector
		simulation iterations.
	emp-sector-x
	emp-sector-y
		The X,Y location of the sector being simulated.
	emp-sector-des
  		The current designation of the sector being simulated.
	emp-sector-newdes
		The new designation of the sector being simulated (if
		this is \"_\", there is no new designation).  This
		variable is not \"_\" only if the user has redesignated
		the sector, and the old designation has not yet been
		torn down.
	emp-sector-eff
		The current efficiency of the sector being simulated.
	emp-sector-food
		The current amount of food of the sector being
		simulated.
	emp-sector-civ
		The current amount of civilians of the sector being
		simulated.
	emp-sector-uw
		The current amount of uncompensated workers of the
		sector being simulated.
	emp-sector-mil
		The current amount of military of the sector being
		simulated.  Does not change unless the sector starves.
	emp-sector-work
		The current level of work efficiency of the sector being
		simulated.
	emp-sector-money-used
		The amount of money spent in the current simulation
		interation.
	emp-sector-total-money-used
		The total amount of money spent in the current
		simulation.
	emp-sector-food-grown
		The amount of emergency food rations grown in the
		current simulation iteration.  This is nonzero only if
		there is insufficient food to feed the people in the
		sector.
	emp-sector-food-eaten
		The amount of food eaten in the current simulation
		iteration.
	emp-sector-starved
		The amount of people that starved in the current
		simulation iteration.
	emp-sector-civ-growth-limited
		Is non-nil, if the civilian population growth was
		limited for some reason (not enough food, population
		exceeds sector capacity, etc.).
	emp-sector-uw-growth-limited
		Is non-nil, if the uncompensated worker population
		growth was limited for some reason (not enough food,
		population exceeds sector capacity, etc.)


***** IMPORTANT NOTE *****

This sector simulator is based upon BSD Empire 1.1 code.  If the game in
which you are playing is using modified sector update routines, this
sector simulator will give the wrong results.  Note also that, as some
sector variables are probabilistic (they increase based upon a random
number generator), the numbers given by this simulator will not agree
exactly, although they should be close.  The numbers will become less
accurate as the number of iterations increase.

"

  (let (workforce cost people result fertility fertility-float return-val
		  distribution-path-exists dx dy)
    ;;
    (setq emp-sector-x x)
    (setq emp-sector-y y)
    (setq emp-sector-des (recall x y (position-of 'des)))
    (setq emp-sector-newdes (recall x y (position-of 'sdes)))
    (setq emp-sector-eff (recall x y (position-of 'eff)))
    (setq emp-sector-food (recall x y (position-of 'food)))
    (setq emp-sector-civ (recall x y (position-of 'civ)))
    (setq emp-sector-mil (recall x y (position-of 'mil)))
    (setq emp-sector-uw (recall x y (position-of 'uw)))
    (setq emp-sector-mob (recall x y (position-of 'mob)))
    (setq emp-sector-work (recall x y (position-of 'work)))
;;    (setq fertility-float (f (recall x y (position-of 'fert))))
    (setq fertility (recall x y (position-of 'fert)))
    ;;
    (setq emp-sector-etu-per-update-float (f etu-per-update))
    (setq emp-sector-food-grown 0)
    (setq emp-sector-food-eaten 0)
    (setq emp-sector-iteration 0)
    (setq emp-sector-money-used 0)
    (setq emp-sector-total-money-used 0)
    (setq emp-sector-civ-growth-limited nil)
    (setq emp-sector-uw-growth-limited nil)
    (setq dx (recall x y (position-of 'dist_x)))
    (setq dy (recall x y (position-of 'dist_y)))
    (if (not (and emp-sector-des
		  emp-sector-newdes
		  emp-sector-eff
		  emp-sector-food
		  emp-sector-civ
		  emp-sector-mil
		  emp-sector-uw
		  emp-sector-work
		  fertility
		  ))
	(error "Need `dump' information in database to simulate sector!"))
    (if (not emp-sector-mob)
	(setq emp-sector-mob 0))	; in case we want to simulate
					; enemy sectors
    (setq fertility-float (f fertility))
    (if (and dx dy
	     (not (and (eq dx x) (eq dx x)))
	)
      (setq distribution-path-exists t))
    (if emp-simulation-setup-hook
      (funcall emp-simulation-setup-hook))
    (setq return-val
	  (catch 'done
	    (while t				;; loop forever
	      ;;
	      ;; Reset variables that need to be reset before an iteration
	      ;;
	      (setq emp-sector-food-grown 0)
	      (setq emp-sector-food-eaten 0)
	      (setq emp-sector-civ-growth-limited t)
	      (setq emp-sector-uw-growth-limited t)
	      (setq emp-sector-starved 0)
	      (setq emp-sector-money-used 0)
	      ;;
	      ;; Quit if everyone is dead
	      ;;
	      (if (and (= emp-sector-civ 0)
		       (= emp-sector-mil 0)
		       (= emp-sector-uw 0))
		(throw 'done nil))
	      ;;
	      ;; Calculate workforce
	      ;;
	      (setq work (get-workforce emp-sector-civ emp-sector-mil
					emp-sector-uw emp-sector-work))
	      (if (or (< emp-sector-eff 100)
		      (not (string= emp-sector-newdes "_")))
		(progn
		  ;; build efficiency, if necessary
		  (setq cost (buildeff (/ work 2)))
		  (setq work (- work cost))
		  (setq emp-sector-money-used cost)
		  (setq emp-sector-total-money-used
			(+ emp-sector-total-money-used emp-sector-money-used))
		)
	      )
      
	      (setq people (+ emp-sector-civ emp-sector-mil emp-sector-uw))
	      (if (not (equal emp-sector-des "s"))
		(progn
		  ;;
		  ;; If this is not a sanctuary ....
		  ;;
		  ;; Do we have enough food?
		  (if (f< (f emp-sector-food)
			  (f+ (f* emp-sector-etu-per-update-float
				  (f* (f people) emp-sector-eatrate))
			      _f1))
		    (progn
		      ;; No -- Grow emergency rations
		      (setq work (- work
				    (setq emp-sector-food-grown
					  (growfood work fertility-float))))
		    )
		  )
		  ;;
		  ;; Feed the people and see if anyone starved
		  ;;
		  (setq emp-sector-starved (feed-people people))
		  (if (= emp-sector-starved 0)
		    (progn
		      ;;
		      ;; If no one starved, have babies
		      ;;
		      (if (< emp-sector-work 100)
			(setq emp-sector-work (+ emp-sector-work
						 (mod (empire-random) 15)
						 8))
		      )
		      (if (> emp-sector-work 100)
			(setq emp-sector-work 100))
		      (grow-people)
		    )
		    (progn
		      ;;
		      ;; People starved
		      ;;
		      (starvation)
		    )
		  )
		)
	      )
      
	      ;;
	      ;; If a distribution path exists, assume that the amount
	      ;; of food stays constant (ideally, we'd like to set the
	      ;; food to the amount set by the distribution thresholds,
	      ;; but this information is not stored in the database).
	      ;;
	      (if distribution-path-exists
		(setq emp-sector-food (recall x y (position-of 'food))))
      
	      (setq emp-sector-iteration (1+ emp-sector-iteration))
	      (if (setq result (funcall emp-simulation-hook))
		(throw 'done result))
	    )
	  )
    )
    (if emp-simulation-finish-hook
      (funcall emp-simulation-finish-hook))
    return-val
  )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; The following functions implement a crude user interface to the
;; sector simulator.
;;

(defvar empire-simulation-max-iterations 20
  "*The maximum number of iterations allowed before terminating the
simulation.")

(defun dump-simulation-output-setup ()
  "Insert, into the current temp buffer, header and starting value
information."
  (let ()
    (princ (format "***** Simulation of sector %s,%s (%s%% %s) *****\n\n"
		   x y emp-sector-eff emp-sector-des))
    (princ "                          eats  food   $$$\n")
    (princ "Update   eff   civ    uw  food  left  costs  total$  work%\n")
    (dump-simulation-output)
  )
)

(defun dump-simulation-output-finish ()
  "Insert a silly message saying that the simulation is complete."
  (let ()
    (princ "\n***** Simulation complete *****\n")
  )
)

(defun dump-simulation-output ()
  "Insert, into the current temp buffer, the current sector values.
This function is designed to be assigned to `emp-simulation-hook', and
will cause the simulation to stop if the sector starves or if 100
iterations have passed."
  (let ()
    (princ (format
	    "%5d   %3d%%   %3d%s  %3d%s  %3d  %4d  %4d   %5d   %3d%%\n"
	    emp-sector-iteration
	    emp-sector-eff
	    emp-sector-civ
	    (if emp-sector-civ-growth-limited
	      "*"
	      " "
	    )
	    emp-sector-uw
	    (if emp-sector-uw-growth-limited
	      "*"
	      " "
	    )
	    emp-sector-food-eaten
	    emp-sector-food
	    emp-sector-money-used
	    emp-sector-total-money-used
	    emp-sector-work
	    ))
    (sit-for 0)
    (if (or (> emp-sector-starved 0)
	    (>= emp-sector-iteration empire-simulation-max-iterations)
	)
      t
      nil
    )
  )
)

(defun empire-simulate-sector (x y max-iter)
  "Simulate the sector at X,Y for up to MAX-ITER iterations, displaying
the simulated values of each iteration in a temp buffer.  If called
interactively, the current map sector is simulated, and the current
value of `empire-simulation-max-iterations' determines the maximum
length of the simulation.  If the sector starves, the simulation will
terminate."
  (interactive (let ( (sect (get-map-sector)) )
		 (list (car sect) (cdr sect) empire-simulation-max-iterations)
		 )
	       )
  (let (output-buffer empire-simulation-max-iterations)
    (setq empire-simulation-max-iterations max-iter)
    (setq emp-tech-level-float (f empire-technology-level))
    (setq emp-edu-level-float (f empire-education-level))
    (setq emp-happiness-level-float (f empire-happiness-level))
    (setq emp-research-level-float (f empire-research-level))
    (setq output-buffer "*Sector_Simulation*")
    (setq emp-simulation-hook 'dump-simulation-output)
    (setq emp-simulation-setup-hook 'dump-simulation-output-setup)
    (setq emp-simulation-finish-hook 'dump-simulation-output-finish)
    (with-output-to-temp-buffer output-buffer
      (pop-to-buffer output-buffer)
      (empire-data-mode nil nil)
      (let ((buffer-read-only nil))
	(sit-for 0)
	(simulate-sector x y)
	))
  )
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Do important initializations, if necessary.
;;

(if (not sector-constants)
  (progn
    (if (not empire-batch-play)
	(message "Initializing emp-sector sector data ..."))
    (initialize-sector-data)
    (if (not empire-batch-play)
	(message ""))
  )
)

(if (not commodity-constants)
    (progn
      (if (not empire-batch-play)
	  (message "Initializing emp-sector commodity data ..."))
      (initialize-commodity-data)
      (if (not empire-batch-play)
	  (message ""))
      )
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'emp-sector)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; BUGS:
;;
;; * Coordinate "wrapping" does not occur.
;; * The mouse does not work in dynamic-mobility-mode.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
