;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-cmds.el -- Interactive commands for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright (c) 1990 Lynn Randolph Slater, Jr
;; 
;; Author          : Lynn Slater  (lrs@indetech.com), and
;;                   Darryl Okahata (darrylo%hpnmd@hpcea.hp.com)
;; Created On      : Thu Jan 31 09:03:39 1991
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:16:43 1991
;; Update Count    : 15
;; Status          : GEET General Release 2d Patch 0
;; 
;; PURPOSE
;; 	This file supplies most fo the interactive commands used to play
;; empire. the mouse bindings and related functions for empire.
;; HISTORY
;; 6-Feb-1991		Lynn Slater x2048	
;;    Last Modified: Tue Feb  5 19:35:53 1991 #13 (Lynn Slater x2048)
;;    let TAB commands pass a prefix arg per changes by Randll Smith
;; 31-Jan-1991		Lynn Slater x2048	
;;    Split out from emp-modes
;; TABLE OF CONTENTS
;;   map-execute-prompted-command -- Prompt for and execute an empire command.
;;   map-execute-empire-command -- Execute a simple empire COMMAND, and then execute FCN.  The sector
;;   map-execute-simple-empire-command -- Execute a simple empire COMMAND on sector(s) SECTS, and then
;;   map-execute-very-simple-empire-command -- On the sector X, Y, execute a simple empire COMMAND, and then
;;   map-execute-command -- From the map window, execute the command already typed in the
;;   empire-move-to-map -- Insert in empire-shell-buffer a path from the last sector mentioned in the
;;   empire-switch-to-map -- Moves to the map in another window. If prefix arg given, moves in the
;;   empire-switch-to-shell -- Moves to the empire shell in another window.
;;   empire-path-to-map -- Finds path from the last sector mentioned in the current buffer to the
;;   empire-insert-map-sector -- Inserts the current map sector.
;;   map-insert-location -- Insert the current x,y location into the empire shell buffer.
;;   map-unmove -- Undo the last map move.
;;   move-to-in-empire-map -- Moves in the map to a given sector
;;   move-in-empire-map-j -- Moves in the empire map in dir j.
;;   move-in-empire-map-g -- Moves in the empire map in dir g.
;;   move-in-empire-map-y -- Moves in the empire map in dir y.
;;   move-in-empire-map-u -- Moves in the empire map in dir u.
;;   move-in-empire-map-n -- Moves in the empire map in dir n.
;;   move-in-empire-map-b -- Moves in the empire map in dir b.
;;   move-in-empire-map-j-and-insert -- Moves in the empire map in dir j and inserts `j' in the empire shell
;;   move-in-empire-map-g-and-insert -- Moves in the empire map in dir g and inserts `g' in the empire shell
;;   move-in-empire-map-y-and-insert -- Moves in the empire map in dir y and inserts `y' in the empire shell
;;   move-in-empire-map-u-and-insert -- Moves in the empire map in dir u and inserts `u' in the empire shell
;;   move-in-empire-map-n-and-insert -- Moves in the empire map in dir n and inserts `n' in the empire shell
;;   move-in-empire-map-b-and-insert -- Moves in the empire map in dir b and inserts `b' in the empire shell
;;   map-insert-h -- Insert an `h' into the empire-shell-buffer.
;;   empire-des-in-empire-map -- Designates the sector at point
;;   empire-record-as-untaken -- Makes the sector not be reported as lost or taken
;;   scroll-right-half-page -- Is a scroll-right but by default uses window-width/2
;;   scroll-left-half-page -- Is a scroll-right but by default uses window-width/2
;;   display-map-sector-info -- Display the sector info for the current map sector.
;;   select-this-sector -- Calls selected-sector-hooks
;;   make-survey-readable -- Make the output of the survey command readable.
;;   empire-send-current-command -- Sends the command on the current line
;;   empire-display-this-sector -- Shows the sector first mentioned on this line
;;   reduce-production-reports -- Reduces the noise in a production report. Gets rid of strollers,
;;   really-reduce-production-reports -- Reduces the noise in a production report. Gets rid of strollers,
;;   take-out -- Removes this line from the rest of the buffer. Good for production
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; emp-install.el which should have been distributed with this file. These
;; terms constitute what the Free Software Foundation calls a COPYLEFT.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'emp-cmds)
(require 'emp-shell)
(require 'emp-plane)
(require 'emp-sail)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; User variables -- do not edit here, use empire-edit-options
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-map-history-directions-max 1000
  "*The maximum number of directions to keep in `empire-map-moved-directions'.
This variable is saved as part of the empire data file."
  )

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; System Variables
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-command-alist nil
  "Alist of empire commands that can be executed from the map buffer.
Never edit this list directly. Use the macro
empire-define-information-command or the call register-empire-command.
These commands may actually reside in any number of files, including some
local to your site.")

(defvar empire-map-moved-directions nil
  "A list of directions in which the user has moved via the juygbn keys."
  )

(defvar empire-last-sectors "#"
  "The last multi sector spec used. Affected by empire-read-sector-or-realm
and empire-mouse-insert-corner")

(defvar empire-last-condition ""
  "The last condition used, affected by empire-prompt-read-condition")

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Game mimic commands
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun register-empire-command (command name_or_exp)
  "Places an item on the empire-command-alist so that the user will see them."
  (if (assoc command empire-command-alist)
      (setcdr (assoc command empire-command-alist) name_or_exp)
    (setq empire-command-alist (cons (cons command name_or_exp)
				     empire-command-alist)))
  ;; sort as we insert it
  (setq empire-command-alist
	(sort empire-command-alist
	      '(lambda (x1 x2)
		 (string-lessp (car x1)
			       (car x2)))
	      ))
  (if (and (symbolp name_or_exp)
	   (not (string-match "Empire" (symbol-name name_or_exp))))
      (put name_or_exp 'empire 'info-command))
  )

(defmacro empire-define-information-command (doc name
						 sector-prompt initial-sectors
						 command fcn
						 &optional pass-prefix)
  (register-empire-command command name)
  (let ()
    (if sector-prompt
	(`
	 (defun (, name) (sects condition
				(,@ (if pass-prefix '(&optional prefix))))
	   (, doc)
	   (interactive (list
			 (empire-read-sector-or-realm (, sector-prompt)
						      (, initial-sectors)
						      )
			 (empire-prompt-read-condition)
			 (,@ (if pass-prefix '(current-prefix-arg)))
			 ))
	   (map-execute-empire-command (format "%s %s %s"
					       (, command) sects condition)
				       (, fcn)
				       (,@ (if pass-prefix '(prefix))))
	   )
	 )
      (if pass-prefix
	  (`
	   (defun (, name) (&optional prefix)
	     (, doc)
	     (interactive "P")
	     (map-execute-empire-command (format "%s"
						 (, command))
					 (, fcn) prefix)
	     )
	   )
	(`
	 (defun (, name) ()
	   (, doc)
	   (interactive)
	   (map-execute-empire-command (format "%s"
					       (, command))
				       (, fcn))
	   )
	 )
	)
      )
    )
  )


(defun map-execute-prompted-command ()
  "Prompt for and execute an empire command."
  (interactive)
  (let (command key prompt sects x y)
    (move-to-empire-shell-prompt)	
    (setq key (where-is-internal 'minibuffer-complete
				 minibuffer-local-must-match-map))
    (if key
	(progn
	  (setq key (car key))
	  (cond
	   ( (string= key "\t") (setq key "`TAB'") )
	   ( (string= key " ") (setq key "`space'") )
	   )
	  (setq prompt (format "Empire command (press %s for list)? " key))
	  )
      (setq prompt "Empire command? ")
      )

    (setq command
	  (completing-read prompt empire-command-alist
			   nil t))
    (if command
	(progn
	  (setq sects (get-map-sector))
	  (setq x (car sects))
	  (setq y (cdr sects))
	  (setq sects (format "%s,%s" x y))
	  (setq fcn (cdr (assoc command empire-command-alist)))
	  (if (symbolp fcn)
	      (call-interactively fcn)
	    (eval fcn)
	    )
	  (map-purge-directions)
	  )
      )
    )
  )
(put 'map-execute-prompted-command 'empire t)

(defun map-execute-empire-command (command fcn &optional args)
  "Execute a simple empire COMMAND, and then execute FCN.  The sector
info of the current map sector is then displayed."
  ;;(interactive)
  (let ()
    (if args
	(send-empire-command-and-parse-reply command fcn nil args)
      (send-empire-command-and-parse-reply command fcn))
    (display-map-sector-info)
    )
  )

(defun map-execute-simple-empire-command (command sects fcn)
  "Execute a simple empire COMMAND on sector(s) SECTS, and then
execute FCN.  The sector info of the current map sector is then
displayed."
  ;;(interactive)
  (let ()
    (send-empire-command-and-parse-reply (format "%s %s" command sects) fcn)
    (display-map-sector-info)
    )
  )

(defun map-execute-very-simple-empire-command (x y command fcn)
  "On the sector X, Y, execute a simple empire COMMAND, and then
execute FCN.  The sector info of that sector is then redisplayed."
  ;;(interactive)
  (map-execute-simple-empire-command command (format "%s,%s" x y) fcn)
  )


(defun map-execute-command ()
  "From the map window, execute the command already typed in the
empire-shell-buffer."
  (interactive)
  (switch-to-empire-buffer-if-necessary
   (empire-send-input)
   )
  (map-purge-directions)
  )
(put 'map-execute-command 'empire t)



;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Key equivilents to common mouse commands (for mouseless folks)
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-move-to-map () ;;key equivilent to middle, use on *Map*
  "Insert in empire-shell-buffer a path from the last sector mentioned in the
empire-shell-buffer to the sector clicked."
  (interactive)
  (if (not (eql (current-buffer) empire-map-buffer))
      (switch-to-buffer-other-window empire-map-buffer))
  (let ((x (map-x))
	(y (map-y))
	(shell-window (empire-get-window-to-change empire-shell-buffer))
	)
    (message "Moving to %s,%s" x y)
    (if shell-window
      (progn
	(select-window shell-window)
      )
      (switch-to-buffer-other-window empire-shell-buffer))
    (goto-char (point-max))
    (point-wysiwyg)
    (empire-path x y)
  ))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Mode support
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-switch-to-map (arg)
  "Moves to the map in another window. If prefix arg given, moves in the
map to the last sector mentioned in the current buffer."
  (interactive "P")
  (let (x y)
    (if arg
	(progn
	  (re-search-backward "\\(-*[0-9]+\\),\\(-*[0-9]+\\)")
	  (setq x (read-buffer-num (match-beginning 1) (match-end 1)))
	  (setq y (read-buffer-num (match-beginning 2) (match-end 2)))
	  ))
    (switch-to-buffer-other-window empire-map-buffer)
    (if arg
	(move-to-in-empire-map x y)
    (describe-sect (map-x) (map-y))	 
      )
    ))

(defun empire-switch-to-shell (arg)
  "Moves to the empire shell in another window.
If prefix arg given, inserts in the shell the current sector coords."
  (interactive "P")
  (let (x y (shell-window (empire-get-window-to-change empire-shell-buffer)) )
    (if arg
	(progn
	  (setq x (map-x))
	  (setq y (map-y))
	  ))
    ;;
    ;; If the empire shell window is already being displayed, switch to the
    ;; appropriate window (in case there is more than one window displaying
    ;; it).
    ;;
    (if shell-window
	(select-window shell-window)
      (switch-to-buffer-other-window empire-shell-buffer)
      )
    (point-wysiwyg)
    (sit-for 0)
    (if arg
	(progn
	  (describe-sect x y)
	  (insert-space-in-empire-buffer-if-necessary)
	  (insert (format "%s,%s " x y))
	  ))
    ))

(defun empire-path-to-map (arg)
  "Finds path from the last sector mentioned in the current buffer to the
current sector in the map.
   If arg is given, sector coords are prompted."
  (interactive "P")
  (if arg (call-interactively 'empire-path)
    (let ((cb (current-buffer))
	  x y)
      (set-buffer empire-map-buffer)
      (setq x (map-x) y (map-y))
      (set-buffer cb)
      (empire-path x y)
      (message "Path inserted to %s,%s" x y)
      )))

(defun empire-insert-map-sector (arg)
  "Inserts the current map sector."
  (interactive "P")
  (let ((cb (current-buffer))
	x y)
    (set-buffer empire-map-buffer)
    (setq x (map-x) y (map-y))
    (set-buffer cb)
    (visually-valid-sector x y)
    (insert-space-in-empire-buffer-if-necessary)
    (insert (format "%s,%s " x y))
    (describe-sect x y)
    ))

(defun map-insert-location () ; call from map window, seems same as empire-insert-map-sector
  "Insert the current x,y location into the empire shell buffer."
  (interactive)
  (fixup-map-cursor-location (map-x) (map-y))
  (let ((x (map-x))
	(y (map-y))
	)
    (show-map (point))
    (visible-insert-in-empire-buffer (format "%s,%s" x y) t)
  ))
(put 'map-insert-location 'empire t)




;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Commands to use on the map buffer
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Moving the cursor

(defun record-current-map-xy (x y)
  "Save the current map sector location."
  (setq empire-current-map-x x)
  (setq empire-current-map-y y)
  )

(defun fixup-map-cursor-location (x y)
  "Test to see if the given map coordinate is valid.  If it is, do
nothing.  If it is not, make it a valid map coordinate and update the
map cursor."
  (let ()
    (cond
     ( (> x empire-x-stop) (setq x empire-x-stop) )
     ( (< x empire-x-start) (setq x empire-x-start) )
     )
    (cond
     ( (> y empire-y-stop) (setq y empire-y-stop) )
     ( (< y empire-y-start) (setq y empire-y-start) )
     )
    (if (= (logxor (logand x 1) (logand y 1)) 1)
	(progn
	  (if empire-favor-left-coordinate
	      (setq x (1- x))
	    (setq x (1+ x))
	    )
	  (move-to-XY-in-empire-map x y t)
	  )
      )
    (cons x y)
    )
  )
  
(defun map-record-direction (direction)
  "Record movement in DIRECTION, which is one of the following symbols:
	j u y g b n J U Y G B N H
"
  (let (current)
    (setq empire-map-moved-directions (cons direction
					    empire-map-moved-directions))
    (if (> (length empire-map-moved-directions)
	   empire-map-history-directions-max)
	(progn
	  (setq previous nil)
	  (setq current empire-map-moved-directions)
	  (while (cdr current)
	    (setq previous current)
	    (setq current (cdr current))
	    )
	  (if previous
	      (setcdr previous nil)
	    )
	  )
      )
    )
  )

(defun map-get-last-direction ()
  "Get the last direction in which the user moved."
  (let (last-dir)
    (if empire-map-moved-directions
	(progn
	  (setq last-dir (car empire-map-moved-directions))
	  (setq empire-map-moved-directions (cdr empire-map-moved-directions))
	  last-dir
	  )
      nil
      )
    )
  )

(defun map-purge-directions ()
  "Purge the saved list of directions."
  (setq empire-map-moved-directions nil)
  )

(defun map-unmove ()
  "Undo the last map move."
  (interactive)
  (let (last-dir)
    (setq last-dir (map-get-last-direction))
    (if (not last-dir)
	(error "You can't backup any further")
      )
    (cond
     ( (or (eq last-dir 'j) (eq last-dir 'J))
       (move-in-empire-map -2 0)
       (if (eq last-dir 'J)
	   (backspace-in-empire-buffer)
	 )
       )
     ( (or (eq last-dir 'u) (eq last-dir 'U))
       (move-in-empire-map -1 1)
       (if (eq last-dir 'U)
	   (backspace-in-empire-buffer)
	 )
       )
     ( (or (eq last-dir 'y) (eq last-dir 'Y))
       (move-in-empire-map 1 1)
       (if (eq last-dir 'Y)
	   (backspace-in-empire-buffer)
	 )
       )
     ( (or (eq last-dir 'g) (eq last-dir 'G))
       (move-in-empire-map 2 0)
       (if (eq last-dir 'G)
	   (backspace-in-empire-buffer)
	 )
       )
     ( (or (eq last-dir 'b) (eq last-dir 'B))
       (move-in-empire-map 1 -1)
       (if (eq last-dir 'B)
	   (backspace-in-empire-buffer)
	 )
       )
     ( (or (eq last-dir 'n) (eq last-dir 'N))
       (move-in-empire-map -1 -1)
       (if (eq last-dir 'N)
	   (backspace-in-empire-buffer)
	 )
       )
     ( (eq last-dir 'H)
       (backspace-in-empire-buffer)
       )
     )
    )
  )
(put 'map-unmove 'empire t)

(defun move-to-in-empire-map (x y)
  "Moves in the map to a given sector"
  (interactive (let (sect)
		 (setq sect (empire-prompt-read-xy))
		 (list (car sect) (cdr sect))
	       ))
  (let (sect)
    ;;
    ;; Insure that X,Y is a valid displayed sector.
    ;;
    (setq sect (fixup-map-cursor-location x y))
    (setq x (car sect))
    (setq y (cdr sect))
    ;; Is this still needed?  No, but we leave it here just in case there are
    ;; bugs in `fixup-map-cursor-location'.
    (visually-valid-sector x y)
    ;; Save the current X,Y location.  We don't do this in `show-pt'
    ;; because `show-pt' can be called from many places, places where we
    ;; don't want the recorded XY location to change.
    (record-current-map-xy x y)
    (show-pt (cons x y))
    )
  ;; we do not want to mark just simple manauvering
  ;;(if (not (= (point) (mark))) (push-mark highlighted-map t))
  )

(defun move-to-XY-in-empire-map (x y &optional just-move)
  "Move map cursor to location X,Y.  This function will work from any buffer."
  (switch-to-map-buffer-if-necessary
   (visually-valid-sector x y)
   (goto-line (empire-y-to-map-row y))
   (move-to-column-force (empire-x-to-map-column x))
   ;; highlight manipulation?
   (if (not just-move)
       (setq highlighted-map (point))
     )
   )
  )

(defun move-in-empire-map (delta-x delta-y)
  "Moves point in empire map."
  (let ((sect (visually-normalize-sect (cons (+ delta-x (map-x))
					     (+ delta-y (map-y)))))
	)
    (move-to-in-empire-map (car sect) (cdr sect))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun move-in-empire-map-j ()
  "Moves in the empire map in dir j."
  (interactive)
  (map-record-direction 'j)
  (move-in-empire-map 2 0))
    
(defun move-in-empire-map-g ()
  "Moves in the empire map in dir g."
  (interactive)
  (map-record-direction 'g)
  (move-in-empire-map -2 0))
    
(defun move-in-empire-map-y ()
  "Moves in the empire map in dir y."
  (interactive)
  (map-record-direction 'y)
  (move-in-empire-map -1 -1))
    
(defun move-in-empire-map-u ()
  "Moves in the empire map in dir u."
  (interactive)
  (map-record-direction 'u)
  (move-in-empire-map 1 -1))
    
(defun move-in-empire-map-n ()
  "Moves in the empire map in dir n."
  (interactive)
  (map-record-direction 'n)
  (move-in-empire-map 1 1))
    
(defun move-in-empire-map-b ()
  "Moves in the empire map in dir b."
  (interactive)
  (map-record-direction 'b)
  (move-in-empire-map -1 1))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun move-in-empire-map-j-and-insert ()
  "Moves in the empire map in dir j and inserts `j' in the empire shell
buffer."
  (interactive)
  (move-in-empire-map 2 0)
  (map-record-direction 'J)
  (insert-in-empire-buffer "j"))

(defun move-in-empire-map-g-and-insert ()
  "Moves in the empire map in dir g and inserts `g' in the empire shell
buffer."
  (interactive)
  (move-in-empire-map -2 0)
  (map-record-direction 'G)
  (insert-in-empire-buffer "g"))

(defun move-in-empire-map-y-and-insert ()
  "Moves in the empire map in dir y and inserts `y' in the empire shell
buffer."
  (interactive)
  (move-in-empire-map -1 -1)
  (map-record-direction 'Y)
  (insert-in-empire-buffer "y"))

(defun move-in-empire-map-u-and-insert ()
  "Moves in the empire map in dir u and inserts `u' in the empire shell
buffer."
  (interactive)
  (move-in-empire-map 1 -1)
  (map-record-direction 'U)
  (insert-in-empire-buffer "u"))

(defun move-in-empire-map-n-and-insert ()
  "Moves in the empire map in dir n and inserts `n' in the empire shell
buffer."
  (interactive)
  (move-in-empire-map 1 1)
  (map-record-direction 'N)
  (insert-in-empire-buffer "n"))

(defun move-in-empire-map-b-and-insert ()
  "Moves in the empire map in dir b and inserts `b' in the empire shell
buffer."
  (interactive)
  (move-in-empire-map -1 1)
  (map-record-direction 'B)
  (insert-in-empire-buffer "b"))

(defun map-insert-h ()
  "Insert an `h' into the empire-shell-buffer."
  (interactive)
  (map-record-direction 'H)
  (insert-in-empire-buffer "h")
)
(put 'map-insert-h 'empire t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Other Map Commands

(defun empire-des-in-empire-map (new-des)
  "Designates the sector at point"
  (interactive "cNew Des: ")
  (let ((des (position-of 'des))
	(empire-map-des-interactively t))
    (record-des (map-x) (map-y) (make-string 1 new-des) t t)))

(defun empire-record-as-untaken (x y)
  "Makes the sector not be reported as lost or taken"
  (interactive (let (sect)
		 (setq sect (empire-prompt-read-xy))
		 (list (car sect) (cdr sect))
	       ))
  (record x y (position-of dist_x) nil)
  (message "%s,%s will no longer be reported as taken" x y))
  
(defun scroll-right-half-page (&optional arg)
  "Is a scroll-right but by default uses window-width/2"
  (interactive "P")
  ;;(message "p is %s %s" arg (prefix-numeric-value arg))
  ;;(sleep-for-millisecs 3000)
  (scroll-right (or arg (/ (window-width) 2)))	  
  (window-wysiwyg-point)
  )
(put 'scroll-right-half-page 'empire t)

(defun scroll-left-half-page (&optional arg)
  "Is a scroll-right but by default uses window-width/2"
  (interactive "P")
  ;;(message "p is %s %s" arg (prefix-numeric-value arg))
  ;;(sleep-for-millisecs 3000)
  (scroll-left (or arg (/ (window-width) 2)))
  (window-wysiwyg-point)
  )
(put 'scroll-left-half-page 'empire t)

(defun display-map-sector-info (&optional flag)
  "Display the sector info for the current map sector.
   If given an arg, describes the ideal levels"
  (interactive "P")
  (if empire-map-buffer
      (let ((sect (get-map-sector)))
	(switch-to-map-buffer-if-necessary
	 (if flag (empire-toggle-global-desc-fcn))
	 (move-to-in-empire-map (car sect) (cdr sect))
	 ))
    ))
(put 'display-map-sector-info 'empire t)

;; What to do when a sector is selected

(defvar selected-sector-hooks (list 'map-insert-location)
  "A function to call when a sector in the map is selected.")

(defun select-this-sector ()
  "Calls selected-sector-hooks"
  (interactive)
  (run-hooks 'selected-sector-hooks))

(put 'select-this-sector 'empire t)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Sector selector from fcn
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar last-selected-sector-returned nil)
(defun select-a-sector (&optional label)
  "Returns (cons X Y) from a selected sector"
  (if label (message "%s" label))
  (let ((selected-sector-hooks (list 'return-selected-sector)))
    (save-window-excursion (save-excursion (recursive-edit)))
    (message "")
    last-selected-sector-returned))

(defun return-selected-sector ()
  (let ((x (map-x))
	(y (map-y))
	)
    (show-map (point))
  (setq last-selected-sector-returned (cons x y))
  (throw 'exit nil)))

;; test fcn
;;(defun select-sector () (interactive) (message "sector %s" (select-a-sector)))




;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Commands from the shell
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun make-survey-readable ()
  "Make the output of the survey command readable.
This routine could be made much more efficient."
  (interactive)
  (let (char-list)
    (setq char-list '(
		      ("\240" . "~")
		      ("\260" . "0")
		      ("\261" . "1")
		      ("\262" . "2")
		      ("\263" . "3")
		      ("\264" . "4")
		      ("\265" . "5")
		      ("\266" . "6")
		      ("\267" . "7")
		      ("\270" . "8")
		      ("\271" . "9")
		      ("\244" . "$")
		      )
	  )
    (replace-chars-from-list char-list)
    )
  )
(put 'make-survey-readable 'empire t)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Data mode fcns
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun empire-send-current-command ()
  "Sends the command on the current line"
  (interactive)
  (beginning-of-line)
  (if (looking-at "[a-z]")
      (progn
	(show-shell)
	(send-empire-command (buffer-substring (point)
					       (progn
						 (end-of-line)
						 (point))))
	(beginning-of-line)
	(insert ";;")
	))
  (re-search-forward "^[a-z]" nil t)
  (beginning-of-line)
  )

(defun empire-display-this-sector ()
  "Shows the sector first mentioned on this line"
  (interactive)
  (let ((pt (point))
	(stop))
    (end-of-line)
    (setq stop (point))
    (beginning-of-line)
    (if (re-search-forward "\\(-*[0-9]+\\),\\(-*[0-9]+\\)" stop t)
	(show-pt (cons (read-match-num 1) (read-match-num 2)))
      (error "There is no sector on this line"))))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Misc stuff
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun reduce-production-reports ()
  "Reduces the noise in a production report. Gets rid of strollers,
medicine, tech, and graduates."
  (interactive)
  (delete-matching-lines "happy strollers (")
  (delete-matching-lines "medical discoveries (")
  (delete-matching-lines "a class of graduates (")
  (delete-matching-lines "technological breakthroughs (")
  )
(put 'reduce-production-reports 'empire t)

(defun really-reduce-production-reports ()
  "Reduces the noise in a production report. Gets rid of strollers,
medicine, tech, and graduates. Also gets rid of backlog and plague reports."
  (interactive)
  (delete-matching-lines "happy strollers (")
  (delete-matching-lines "medical discoveries (")
  (delete-matching-lines "a class of graduates (")
  (delete-matching-lines "technological breakthroughs (")
  (delete-matching-lines "backlog")
  (delete-matching-lines "plague")
  )
(put 'really-reduce-production-reports 'empire t)

(defun take-out ()
  "Removes this line from the rest of the buffer. Good for production
report problems that have been handled."
  (interactive)
  (beginning-of-line)
  (delete-matching-lines (regexp-quote
			  (buffer-substring
			   (point) (save-excursion (end-of-line) (point))))))
(put 'take-out 'empire t)

