;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Emacs-Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;
;; emp-auto.el -- Comands for automated play for Gnu Emacs Empire Tool (GEET)
;; 
;; Copyright (c) 1990 Lynn Randolph Slater, Jr
;; 
;; Author          : Lynn Slater (lrs@indetech.com)
;; Created On      : Fri Oct 26 16:02:47 1990
;; Last Modified By: Lynn Slater x2048
;; Last Modified On: Thu Feb 14 19:16:46 1991
;; Update Count    : 164
;; Status          : GEET General Release 2d Patch 0
;; 
;; HISTORY
;; 4-Jan-1991		Lynn Slater x2048	
;;    Last Modified: Wed Dec 26 18:53:56 1990 #126 (Lynn Slater x2048)
;;    added nova-from-explored-sectors by stevens@hplkss.hpl.hp.com
;; 26-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Thu Dec 13 13:13:14 1990 #125 (Lynn Slater x2048)
;;    nova keeps sects - until later
;; 13-Dec-1990		Lynn Slater x2048	
;;    Last Modified: Wed Dec 12 11:04:46 1990 #120 (Lynn Slater x2048)
;;    added empire-adjust-hooks
;; 25-Nov-1990		Lynn Slater x2048	
;;    Last Modified: Sat Nov 17 18:50:32 1990 #88 (Lynn Slater)
;;    darryl: made refresh output go to another buffer
;; 16-Nov-1990		Lynn Slater	
;;    Last Modified: Thu Nov 15 08:39:18 1990 #85 (Lynn Slater)
;;    took out rad *
;; 12-Nov-1990		Lynn Slater	
;;    Last Modified: Mon Nov 12 18:47:57 1990 #81 (Lynn Slater)
;;    added empire-fire-all-ships
;; 5-Nov-1990		Lynn Slater	
;;    Last Modified: Sun Nov  4 16:04:25 1990 #42 (Lynn Slater)
;;    added ping scan to extract all
;; 4-Nov-1990		Lynn Slater	
;;    Last Modified: Fri Nov  2 19:32:58 1990 #26 (Lynn Slater)
;;    added batch-adjust function and -adjust command line arg
;;    added *Proposed* buffer
;; 2-Nov-1990		Lynn Slater	
;;    Last Modified: Fri Nov  2 09:21:43 1990 #16 (Lynn Slater)
;;    refresh reads nation info
;; 30-Oct-1990		Lynn Slater	
;;    Last Modified: Mon Oct 29 12:50:50 1990 #8 (Lynn Slater)
;;    added automatic commands
;; PURPOSE
;; 	This file contains high level drivers of the functionality provided
;; by the other empire files. For example, in this file are commands that
;; cause commands to be issued to the empire client, the results scanned,
;; analysis performed, and actions taken.
;; TABLE OF CONTENTS
;;   empire-read-data -- Reads all readable things from here to end of buffer. Is useful for
;;   empire-read-unique-data -- Reads all readable things that cannot be recreated from here to end of
;;   empire-read-data-save -- Reads all readable things and stores the data. refresh-adjust-empire
;;   refresh-empire -- Finds out as much as can be found about the current empire state
;;   adjust-empire -- Scan nothing, do all possible analysis, make all possible conclusions
;;   refresh-adjust-empire -- Finds out as much as can be found about the current empire state and
;;   read-refresh-adjust-empire -- Scan all data from here to end of buffer (getting navs, explores, etc
;;   empire-get-new-world -- Starts empire up on a new world provided shell interaction works.
;;   empire-break-new-world -- Does a db init, sends realm to client, names country, breaks, explodes 1
;;   empire-des-new-world -- Gives default designations to new land. Normally designates sectors with
;;   empire-explode-from-sector -- Takes over all wilderness adjacient to a given sector
;;   empire-explode-all-sectors -- Takes over all adjacient unoccupied wilderness
;;   nova-from-sector -- Uses all but 1 civ and 5 mobility from a sector to grab via explore
;;   super-nova-from-sector -- Uses all but 1 civ and 5 mobility from a sector to grab via explore
;;   nova-from-explored-sectors -- Will nova from all sectors with < 5 & > 1 civ without mil.
;;   empire-fire-all-ships -- Fires from forts on all ships reported by coastwatch until they sink.
;;   import-to-sector -- This function emits the command to bring sector 2 up to its
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The contents of this file ARE copyrighted but permission to use, modify,
;; and distribute this code is granted as described in the file
;; emp-install.el which should have been distributed with this file. These
;; terms constitute what the Free Software Foundation calls a COPYLEFT.
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(provide 'emp-auto)
(require 'emp-shell)
(require 'emp-buffers)
(require 'emp-anal)
(require 'emp-sail)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Stand alone commands
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Mastor Extraction Drivers
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun forget-ownership ()
  "Forgets that we own each sector as some might have gone away"
  (let (
	(mob (position-of 'mob))
	(make (position-of 'make))
	)
    (walksects 
     (if (recall x y mob)
	 (progn
	   (record x y mob nil)
	   (record x y make nil)
	   ))
     "Forgetting ownership")))

(defun empire-read-data ()
  "Reads all readable things from here to end of buffer. Is useful for
manual operation."
  (interactive)
  (let ((limit (point)))
    (safe-excursion (empire-read-explores))
    (safe-excursion (progn
		      (goto-char (point-max))
		      (if (re-search-backward "^PRODUCTION SIMULATION" limit t)
			  (progn (beginning-of-line) (empire-read-production)))))
    (safe-excursion (progn
		      (goto-char (point-max))
		      (if (re-search-backward "^Dump Sector$" limit t)
			  (progn (beginning-of-line) (empire-read-dump)))))
    )
  (safe-excursion (empire-read-census))
  (safe-excursion (empire-read-radars))
  (safe-excursion (empire-read-spyreports))
  (let ((empire-map-des-interactively t))
    (safe-excursion (empire-read-looks))
    (safe-excursion (empire-read-flight))
    (safe-excursion (empire-read-navs))
    (safe-excursion (empire-read-pings))
    (safe-excursion (empire-read-coastwatch))
    (safe-excursion (empire-read-deployed-mines))
    (safe-excursion (empire-read-exploding-mines)))
  (end-of-buffer)
  (if (not empire-batch-play)
      (map-empire))
  )

(defun empire-read-unique-data ()
  "Reads all readable things that cannot be recreated from here to end of
buffer. Is useful for navs, spys, flights, explores, etc thay might not
have been automatically scanned already"
  (interactive)
  (let ((limit (point)))
    (safe-excursion (empire-read-explores))
    )
  (safe-excursion (empire-read-radars))
  (safe-excursion (empire-read-spyreports))
  (let ((empire-map-des-interactively t))
    (safe-excursion (empire-read-looks))
    (safe-excursion (empire-read-flight))
    (safe-excursion (empire-read-navs))
    (safe-excursion (empire-read-pings))
    (safe-excursion (empire-read-coastwatch))
    (safe-excursion (empire-read-deployed-mines))
    (safe-excursion (empire-read-exploding-mines)))
  (end-of-buffer)
  (if (not empire-batch-play)
      (map-empire))
  )

(defun empire-read-data-save (file)
  "Reads all readable things and stores the data. refresh-adjust-empire
does all this and performs analysys as well."
  (interactive 
   (list (read-file-name
	  (format "Save File Name (Default: %s): " empire-save-file-name) 
	  (if empire-save-file-name (file-name-directory empire-save-file-name))
	  empire-save-file-name)))

  (setq empire-save-file-name
	(expand-file-name file
	 (if empire-save-file-name
	     (file-name-directory empire-save-file-name))))
  (empire-read-data)
  (save-empire file)
  )

;;(defun empire-read-data-long-save (file)
;;  "Reads all readable things, stores the data, and performs the checks and
;;flow analysis. As I end a empire session, I dump #, prod #, and rad #. I
;;then go back to the start of the session and run this command."
;;  (interactive 
;;   (list (read-file-name
;;	  (format "Save File Name (Default: %s): " empire-save-file-name) 
;;	  (if empire-save-file-name (file-name-directory empire-save-file-name))
;;	  empire-save-file-name)))
;;
;;  (setq empire-save-file-name
;;	(expand-file-name file
;;	 (if empire-save-file-name
;;	     (file-name-directory empire-save-file-name))))
;;  (forget-ownership) ;; forget what we own as maybe we no longer own it
;;  (empire-read-data)
;;  (save-empire file)
;;  ;;(redistribute-food)
;;  ;;(redistribute-mil)
;;  (empire-flows)
;;  ;;(calc-defensive-air-umbrella)
;;  ;;(map-air-umbrella)
;;  (redistribute-civ)
;;  (check-empire)
;;  )


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Commands to work with the shell
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun refresh-empire (&optional doit)
  "Finds out as much as can be found about the current empire state
without exposing ships or taking any real actions."
  (interactive)
  ;;
  ;; Don't mess up any windows ...
  ;;
  (let ((empire-automatically-execute-commands
	 (or doit empire-automatically-execute-commands)))
    (save-window-excursion
      (save-excursion
	(switch-to-buffer (get-buffer-create "*Proposed*"))
	(empire-data-mode t t)
	(delete-region (point-min) (point-max))
	(insert "Suggested actions as of " (current-time-string) "\n")
	;;
	;; Here we set the mode of the empire-temp-buffer to
	;; `empire-data-mode'.  This is done so that the user can use
	;; the mouse in this buffer to move the map cursor.
	;;
	(switch-to-buffer (get-buffer-create empire-temp-buffer))
	(empire-data-mode t nil)
	))
    (set-buffer (get-buffer-create empire-temp-buffer))
    (delete-region (point-min) (point-max))
    (let ((start (point)))
      (message "Executing `map' command (%s)..." (current-hour-string))
      (empire-map-bounds-quietly empire-x-start empire-y-start
				 empire-x-stop  empire-y-stop
				 t)
      (execute-empire-command-with-buffer-output "nation" empire-temp-buffer)
      (sit-for 0)
      (execute-empire-command-with-buffer-output "dump #" empire-temp-buffer)
      (sit-for 0)
      (execute-empire-command-with-buffer-output "prod #" empire-temp-buffer)
      (sit-for 0)
      (execute-empire-command-with-buffer-output "coastwatch #"
						 empire-temp-buffer)
      (sit-for 0)
      (execute-empire-command-with-buffer-output "rad # ?des=)"
						 empire-temp-buffer)
      (sit-for 0)
      (execute-empire-command-with-buffer-output "sat *"
						 empire-temp-buffer)
      (sit-for 0)
      (execute-empire-command-with-buffer-output "plane *"
						 empire-temp-buffer)
      (sit-for 0)
      (execute-empire-command-with-buffer-output "ship *"
						 empire-temp-buffer)
      (sit-for 0)
      (execute-empire-command-with-buffer-output "cargo *"
						 empire-temp-buffer)
      (sit-for 0)
      (goto-char start)
      (forget-ownership)		; forget what we own as maybe we no longer own it
      (safe-excursion
       (progn
	 (empire-read-nation-report)
	 (empire-read-dump)
	 (empire-read-production)
	 (empire-read-coastwatch)
	 (empire-read-radars)
	 (empire-read-planes t)
	 (empire-read-ships t)
	 (empire-read-cargos)
	 ))
      (map-empire)
      ;;(ding)
      (message "Empire DB Refreshed")
      )))

(defun adjust-empire (doit)
  "Scan nothing, do all possible analysis, make all possible conclusions
from info already in the db.
  Uses the functions in empire-adjust-hooks in the order they appear."
  (interactive (list (y-or-n-p "Do you trust me? ")))
  ;; note dump conclusions already reached
  (let ((empire-automatically-execute-commands doit))
    (run-hooks 'empire-adjust-hooks)
    (message "Empire Adjusted %s" (current-hour-string))))
  
(defun refresh-adjust-empire (file quitp doit)
  "Finds out as much as can be found about the current empire state and
does all analysis. See empire-batch and the function batch-adjust in
emp-batch.el for a batch version of this.

See empire-adjust-hooks for a definition of what will be adjusted.
" 
  (interactive 
   (list (read-file-name
	  (format "Save File Name (Default: %s): " empire-save-file-name) 
	  (if empire-save-file-name (file-name-directory empire-save-file-name))
	  empire-save-file-name)
	 (y-or-n-p "Quit empire afterwards? " )
	 (y-or-n-p "Do you trust me? ")))
  (let ((external-auto empire-automatically-execute-commands)
	(empire-automatically-execute-commands doit))
    (if (not empire-batch-play)
	(if doit
	    (progn
	      (message "He he he he ....")
	      (sit-for 1))
	  (message "Ok, execute all the commands yourself if thats how you feel!")
	  (sit-for 1)))
    (switch-to-buffer empire-shell-buffer)
    (refresh-empire)
    (let ((empire-automatically-execute-commands external-auto))
      (save-empire file))
    (if (and quitp (not doit))
	(send-empire-command "quit"))
    (adjust-empire doit)
    (let ((empire-automatically-execute-commands external-auto))
      (save-empire file))
    (if (and quitp doit)
	(send-empire-command "quit"))
    (if (not (or quitp empire-batch-play))
	(progn
	 (ding)
	 (message "Refresh and adjustment done")))
    ))

(defun read-refresh-adjust-empire (file quitp doit)
  "Scan all data from here to end of buffer (getting navs, explores, etc
that might have been missed) doing no analysys. Then get fresh data from
empire and perform analysis and adjustments on this final state.

See empire-adjust-hooks for a definition of what will be adjusted.
"
  (interactive 
   (list (read-file-name
	  (format "Save File Name (Default: %s): " empire-save-file-name) 
	  (if empire-save-file-name (file-name-directory empire-save-file-name))
	  empire-save-file-name)
	 (y-or-n-p "Quit empire afterwards? " )
	 (y-or-n-p "Do you trust me? ")))
  (let ((empire-dump-hooks nil))	; suppress analysis
    (safe-excursion (empire-read-radars))
    (safe-excursion (empire-read-spyreports))
    (safe-excursion (empire-read-looks))
    (safe-excursion (empire-read-flight))
    (safe-excursion (empire-read-navs))
    (safe-excursion (empire-read-pings))
    (safe-excursion (empire-read-deployed-mines))
    (safe-excursion (empire-read-exploding-mines))
    )
  (end-of-buffer)
  (refresh-adjust-empire file quitp doit)
  (if (not empire-batch-play) (ding))
  )
	
(defun empire-get-new-world (file &optional okp)
  "Starts empire up on a new world provided shell interaction works."
  (interactive 
   (list (read-file-name
	  (format "Save File Name (Default: %s): " empire-save-file-name) 
	  (if empire-save-file-name (file-name-directory empire-save-file-name))
	  empire-save-file-name)
	 nil))
  (if (and empire-sects
	   (not (yes-or-no-p "All facts about the old world will be forgotten. Proceed? ")))
      (error "New world aborted"))
  (send-empire-command-wait "version")
  (sit-for 0)
  (save-excursion
    (pop-to-buffer empire-shell-buffer)
    (empire-new-world file))
  (send-empire-command-wait (format "realm 0 %s:%s,%s:%s"
				    empire-x-start empire-x-stop
				    empire-y-start empire-y-stop))  
  (refresh-empire)
  (if (not empire-batch-play)
      (map-empire))
  (check-empire))

(defun empire-break-new-world (file cname)
  "Does a db init, sends realm to client, names country, breaks, explodes 1
sector out, gets world, and consumes all mobility in 0,0 and 2,0 doing a
supernova. Designates all newly taken land.  

After using this, you should check the designations and correct then to
have a good road strategy and to place j's, k's, etc in the places that
were not designated anything else."
  (interactive 
   (list (read-file-name
	  (format "Save File Name (Default: %s): " empire-save-file-name) 
	  (if empire-save-file-name (file-name-directory empire-save-file-name))
	  empire-save-file-name)
	 (read-string "Country Name: ")))
  (if (and empire-sects
	   (not (yes-or-no-p "All facts about the old world will be forgotten. Proceed? ")))
      (error "New world aborted"))
  ;;(empire-map-default) bug <stub>
  (send-empire-command-wait (format "change country %s" cname))
  (send-empire-command-wait "break")
  (empire-get-new-world file t)
  (empire-explode-all-sectors)
  ;; it would be nice to get an ag going
  (super-nova-from-sector 2 0)
  (super-nova-from-sector 0 0)
  (empire-des-new-world)
  )

(defun empire-des-new-world (arg)
  "Gives default designations to new land. Normally designates sectors with
1 civ and 0 mobility, but if given an arg, will designate with criterion
you specofy."
  (interactive "p")
  (let ((cond (if arg (read-string "Condition: " "&civ=1&mob=0"))))

    (send-empire-command-wait (format "des # ?type=-%s +" cond))
    (send-empire-command-wait (format "des # ?type=+&ocontent>50%s o" cond))
    (send-empire-command-wait (format "des # ?type=+&gold>90%s g" cond))
    (send-empire-command-wait (format "des # ?type=+&min>90%s m" cond))
    (send-empire-command-wait (format "des # ?type=+&fert>90%s a" cond))
    (empire-census-sectors "#" nil)
    ))
  
(defun empire-explode-from-sector (x y)
  "Takes over all wilderness adjacient to a given sector"
  (interactive (let (sect)
		 (setq sect (empire-prompt-read-xy))
		 (list (car sect) (cdr sect))
		 )
	       )
  (if (oursp x y)
      (let ((empire-map-des-interactively t))
	(mapcar 
		 '(lambda (pair)
		   (if (and (> (recall x y (position-of 'civ)) 1)
			    (> (recall x y (position-of 'mob)) 1)
			    (landp (recall (car pair) (cdr pair)
					   (position-of 'des)))
			    )
		       (progn
			 (record x y (position-of 'civ)
				 (1- (recall x y (position-of 'civ))))
			 (move-to-in-empire-map (car pair) (cdr pair))
			 (send-empire-command
			  (format "exp civ %s,%s 1 %s" x y
				  (r-to-path (list (cons x y) pair))))
			 (send-empire-command
			  (format "des %s,%s +" (car pair) (cdr pair)))
			 (empire-census-sector)
			 )))
		(unowned-wilderness-near x y))))
  (message "Explosion from %s,%s done" x y))

(defun empire-explode-all-sectors ()
  "Takes over all adjacient unoccupied wilderness"
  (interactive)
  (mapsects 'empire-explode-from-sector "Exploding "))

(defun nova-explore (x y path)
  "This fcn is intended to be used with calc-nova-from-sector. It explores
from x y to the sector on the path if the sector is not already ours."
  (if (or (<= (recall x y (position-of 'civ)) 1)
	  (< (recall x y (position-of 'mob)) 5))
      t;; stop the loop as there is no more to do
    ;; else
    (if (and (string-equal (recall (car (car path)) (cdr (car path))
				   (position-of des))
			   "-")
	     (not (oursp (car (car path)) (cdr (car path))))
	     )
	(progn
	  (record x y (position-of 'civ)
		  (1- (recall x y (position-of 'civ))))
	  (move-to-in-empire-map (car (car path)) (cdr (car path)))
	  (if (eq (send-empire-command-wait
		   (format "exp civ %s,%s 1 %s" x y
			   (r-to-path (reverse path))))
		  t)
	      (progn
		;;
		;; Success!  The sector should now be ours.
		;;
		(setq nova-did-work t)
		(send-empire-command-wait
		 (format "des %s,%s +" (car (car path)) (cdr (car path))))
		;;(empire-census-sector)
		(record-des (car (car path)) (cdr (car path)) "+" nil)
		(if supernova
		    (empire-map-bounds (normalize-x (- (car (car path)) 2))
				       (normalize-x (+ (car (car path)) 2))
				       (normalize-y (- (cdr (car path)) 2))
				       (normalize-y (+ (cdr (car path)) 2))))
		(move-to-in-empire-map x y)
		(decriment-sector x y (position-of 'mob) 5) ; guess
		(if (< (recall x y (position-of mob)) 5)
		    (empire-census-sector))
		)
	    (progn
	      ;;
	      ;; We couldn't move there
	      ;;
	      (empire-send-input)	;; send a plain [Return] to get back
					;; to the main empire prompt.
	      (record-des (car (car path)) (cdr (car path)) "?" nil)
	      (send-empire-command-wait "h")
	      )
	    )
	  )
      )
    nil
    ))

(defun nova-from-sector (x y)
  "Uses all but 1 civ and 5 mobility from a sector to grab via explore
every known reachable wilderness sector. Grabbed sectors are designated +.
Does not do a map to see the newly visible wilderness sectors. For that use
\\[super-nova-from-sector]"
  (interactive (let (sect)
		 (setq sect (empire-prompt-read-xy))
		 (list (car sect) (cdr sect))
		 )
	       )
  (let ((nova-did-work t)
	(supernova nil))
    (switch-to-empire-buffer-if-necessary
     (calc-nova-from-sector x y 'nova-explore)))
  (message "Done with nova from %s,%s" x y))
(put 'nova-from-sector 'empire t)

(defun super-nova-from-sector (x y)
  "Uses all but 1 civ and 5 mobility from a sector to grab via explore
every known reachable wilderness sector. Grabbed sectors are designated +.
Does do a map to see the newly visible wilderness sectors.

Control the nova by setting empire bounds to a portion of the world
and by limiting civs and mob in the super nova source sector.

A whole country supernova is probably too dangerious."
  (interactive (let (sect)
		 (setq sect (empire-prompt-read-xy))
		 (list (car sect) (cdr sect))
		 )
	       )
  (pop-to-buffer empire-shell-buffer)
  (empire-map-bounds-quietly empire-x-start empire-y-start
			     empire-x-stop  empire-y-stop
			     t)
  (let ((nova-did-work t)
	(empire-map-des-interactively t)
	(supernova t))
    (while nova-did-work
      (setq nova-did-work nil)
      (calc-nova-from-sector x y 'nova-explore)
      (if nova-did-work (empire-map-bounds-quietly empire-x-start empire-y-start
						   empire-x-stop  empire-y-stop
						   t))
      ))
  (message "Done with SUPER nova from %s,%s" x y)  )
(put 'super-nova-from-sector 'empire t)

(defun nova-from-explored-sectors ()	; from stevens@hplkss.hpl.hp.com
  "Will nova from all sectors with < 5 & > 1 civ without mil.
   The theory here is to super-nova from the capitols when you
   break, and then expand using only the babies of the expansion
   sectors.  This permits expansion to continue concurrent with
   efficiency development in a new world.  Seems to be a good
   tradeoff of land grabbing and develpment.

   This uses the internal data base, so it MUST be up to date
   for this to work properly.  Run refresh-empire before running
   this to bring the database up to date."
  (interactive)
  (let ((expanded 0))
    (walksects
     (if (and (oursp x y)
	      (zerop (recall x y (position-of mil))))
	 (let ((civ-cnt (recall x y (position-of civ))))
	   (when (and (> civ-cnt 1) (< civ-cnt 5))
	     (incf expanded)
	     (nova-from-sector x y)))))
    (unless empire-batch-play
      (if (zerop expanded)
	  (message "No expansion done.")
	(empire-map-default)		; Sometimes nova damages screen,
	(map-empire)			;  so just repaint it.
	(message "Expansion from %s explored sectors complete!" expanded)))))
(put 'nova-from-explored-sectors 'empire t)

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Auto defense
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-fire-range 3
  "*Range forts can fire a shell.")
(defvar empire-ignorable-nations
  '(0)
  "*The Nations in this list are presumed to be non threats and no
automatic deamons will do hostile things to them, such as sinking their ships.")
(defun empire-fire-all-ships ()
  "Fires from forts on all ships reported by coastwatch until they sink.
   Does not prioritize ships."
  (interactive)
  (send-empire-command-and-parse-reply "coastwatch #"
				       'empire-fire-all-ships-doit
				       t
				       empire-fire-range
				       ))

(defun empire-fire-all-ships-doit (fire-range)
  (let ((ship-cnt 0)
	(bad-ship-cnt 0)
	(sunk-ship-cnt 0)
	(fire-cnt 0)
	(fort-cnt 0)
	(last-shipnum -1)
	(try-cnt 0)
	(hit-cnt 0)
	)
    (empire-read-coastwatch 'empire-find-fire-base)
    (if (= ship-cnt 0)
	(message "*** All is quiet along the coasts, sir ***\n")
      (message "************************************************************")
      (message "*** Reporting %s ships of which %s were bad guys."
	       ship-cnt
	       bad-ship-cnt)
      (if (not (= ship-cnt 0))
	  (progn
	    (message "*** Fired %s times from %s forts trying to get %s of them,"
	    fire-cnt fort-cnt try-cnt)
	    (message "***   hit %s times, sunk %s ships"
		     hit-cnt sunk-ship-cnt)
	    ))
      (message "************************************************************")
      )))

(defun empire-find-fire-base (country shipname shipnum shipx shipy)
  (setq ship-cnt (1+ ship-cnt))
  (if (memq country empire-ignorable-nations)
      (message "  Ship %s ship %s at %s,%s is allowed to pass as it is from %s"
	       shipname shipnum shipx shipy country)
    (setq bad-ship-cnt  (1+ bad-ship-cnt ))
    (message "Looking to sink %s ship %s from %s at %s %s"
	     shipname shipnum country shipx shipy)
    (show-pt (cons shipx shipy))
    (mapsects-active-in-range shipx shipy fire-range 'empire-sink-ship
			      "Looking to sink ship")))

(defun empire-sink-ship (x y)
  (if (and (string-equal "f" (recall x y (position-of des)))
	   shipx shipy)
      (save-excursion
	(switch-to-empire-buffer-if-necessary
	 (let (start
	       (cnt 0)
	       (empire-message-client-replies t) ; show results
	       )
	   (end-of-buffer)
	   (if (not (= last-shipnum shipnum))
	       (setq try-cnt (1+ try-cnt))
	     (setq last-shipnum shipnum))
	   (if (not (and  (< 3 (or (recall x y (position-of gun)) 0))
			  (< 3 (or (recall x y (position-of shell)) 0))
			  (< (* 3 (recall x y (position-of mil))
				(recall x y (position-of gun))))))
	       (message "Fort at %s,%s is not active. Gun: %s shell: %s mil %s"
			x y
			 (recall x y (position-of gun))
			 (recall x y (position-of shell))
			 (recall x y (position-of mil))))
	   (setq fort-cnt  (1+ fort-cnt ))
	   (while (and (< 3 (or (recall x y (position-of gun)) 0))
		       (< 3 (or (recall x y (position-of shell)) 0))
		       (< cnt 5))
	     (setq start (point))
	     (message "Try to sink %s range %s from f at %s,%s %s guns, shells %s"
		      shipnum
		      (empire-mapdist shipx shipy x y)
		      x y
		      (recall x y (position-of gun))
		      (recall x y (position-of shell)))
	     ;;(sit-for 3)
	     (setq fire-cnt (1+ fire-cnt))
	     (send-empire-command-wait
	      (format "fire %s %s,%s" shipnum x y))
	     (decriment-sector x y (position-of 'shell) 1)
	     (end-of-buffer)
	     (if (re-search-backward "sunk" start t)
		 (progn
		   (setq cnt 100
			 shipx nil
			 shipy nil)
		   (setq hit-cnt (1+ hit-cnt))
		   (setq sunk-ship-cnt (1+ sunk-ship-cnt))
		   (message "\n**** GOT THE BASTARD!, sir.\n")
		   )
	       (if (re-search-backward "Target ship out of range" start t)
		   (progn
		     (setq cnt 100)
		     (message "\n**** Sorry sir, he's too far away.\n")
		     )
		 (if (re-search-backward "shell hit" start t)
		     (progn
		       (setq cnt (1+ cnt))
		       (setq hit-cnt (1+ hit-cnt))
		       (message "\n*** The sucker's still afloat.\n")
		       )
		   (setq cnt 10)))))
	   )))))

;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Batch Play 
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar empire-client-command "(stty -inpck -icrnl imaxbel -onlcr -echo -echoprt -echoctl;setenv COUNTRY Nasty; setenv PLAYER Boy;setenv EMPIREHOST emerald;emp_client)"
  "*Names the command to issue when doing batch play. See emp-batch.el")


;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; import-to-sector
;; ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defconst first-dist-off (position-of 'u_dist)
  "the first distribution offset for import-to-sector")
(defconst first-comm-off (position-of 'uw)
  "the first comm offset for import-to-sector")

(defun import-to-sector (comm x1 y1 x2 y2)
  "This function emits the command to bring sector 2 up to its
  distribution threshold for COMModity by importing from the map point.
  It leaves at least etu_per_update mobility points behind in the source
  and will not drain enough of the resource to take the source below its
  threshold.
 
 Non-interactive users can use locations other than the map point.  The
  interactive interface really needs to be cleaned up."
  (interactive (let ( ( comm (prompt-for-item empire-complete-commodity-list))
		      ( source (cons (map-x) (map-y)))
		      ( dest (select-a-sector "click on dest") )
		      )
		 (list comm (car source) (cdr source) (car dest) (cdr dest))))
  (message "Move %s from %s,%s to %s,%s" comm x1 y1 x2 y2) (sit-for 3)
  
  (if (not (and (oursp x1 y1)
		(oursp x2 y2)))
      (progn
	(message "Sector is not ours!") ; maybe should be (error)
	nil)
    ( let* ( (cb (current-buffer))
	     ( comm-off (dynamic-position-of (read comm)) )
	     ( comm-dist-off (+ first-dist-off ( - comm-off first-comm-off) ) )
	     wanted
	     avail
	     really-avail
	     path
	     )
      (if (and comm-off comm-dist-off)
	  (progn
	    (setq wanted (recall x2 y2 comm-dist-off ))
	    (set-buffer empire-map-buffer)
	    (setq avail ( - (recall x1 y1 comm-off)
			    (recall x1 y1 comm-dist-off) ) )
	    (debug)
	    (setq path (concat (least-cost-path (cons x1 y1)
						(cons x2 y2) ) "h")
		  really-avail (min avail (calc-movable-amount (read comm)
							       x1 y1
							       path
							       etu-per-update)
				    ) )
	    (message "I want %d and can get %d(%d) over path %s"
		     wanted really-avail avail path)

	    (if (< 0 (setq really-avail (min wanted really-avail)))
		;; somebody PLEASE tell me what I want to do here
		;; (I want to make-automated-command but only have
		;; it show up in *Proposed*)
		;;   That is up to the setting of
		;;   empire-automatically-execute-commands -- lrs
		;; If the user trusts my function then it will be
		;; executed, but I don't want it to show up anywhere
		;; but *Proposed* and maybe *Empire*.
		;; make-automated-command also princs it into the
		;; current buffer.  I want the extra princ to disappear.
		(with-output-to-temp-buffer "/dev/null"
		  (make-automated-command (format
					   "move %s %d,%d %d %s"
					   comm x1 y1 really-avail path))
		  )
	      (message "I can't get enough there")
	      )
	    (set-buffer cb)
	    t
	    )
	(nil)
	))))

;;(import-to-sector "hcm" 0 0 7 -3) 
