
/*
 *  MANAGE.C
 *
 *  By Matthew Dillon, dillon@postgres.berkeley.edu
 *
 *  This independant program utilizes optimized best-path graph algorithms
 *  and sophisticated prioritized move generation code for any 
 *  commodity.  It generates move commands based on given information
 *  (census, commodities, level, milciv (awk generated), and plagued (awk
 *  generated) report).
 *
 * |----hand made civ/mil report:----  (may not contain tabs)
 * |xlev <garbage>
 * |<garbage-date>
 * |<garbage-hdr>
 * | sect       civ  mil
 * |sss,sss    nnnn nnnn
 * |sss,sss    nnnn nnnn
 * |  ... 
 *
 * |----hand made plague list:----    (may not contain tabs)
 * |plague <garbage>
 * |<garbage-date>
 * |<garbage-hdr>
 * | sect       dmy  dmy
 * |sss,sss       0    0
 * |sss,sss       0    0
 * |  ...
 *
 *  MANAGE [options] <reports>
 *
 *  The program automatically detects the type of report.  Multiple reports
 *  (of the same type) may be specified.  Later listings overide (where they
 *  overlap) newer listings.
 *
 *  -v		    Verbose.  Else just report statistics.
 *
 *  -c<comm>	    Specify the commodities to be considered.  Default is
 *		    to consider all commodities.
 *
 *  -r<comm>	    Specify the commodities NOT to be considered.  Default is
 *		    to consider all commodities.
 *
 *  -a		    Allow inefficient moves.  The default is to not allow
 *		    them (the moves simply are not made).
 *
 *  -o<file>	    Specify report outputfile... a verbose report is generated
 *		    to the specified file.  Moves are always written to stdout.
 *		    if -o not specified no report is generated.  if -o is
 *		    specified but without a filename, stderr is used.
 *
 *  -p<sect>,<sect> Request the path from one sector to another.  The first
 *		    sector must be a highway (+,=,#) or some sector adjacent
 *		    to a highway.  The second must be a highway.
 *
 *  -p		    Interactive.  Read pairs of sectors <sect>,<sect> and
 *		    output the paths.
 *
 *		    format:   cost <cost> <path>
 * 			      error   0   text-err-msg
 *
 *  -m<comm>,<lev>,<sect>
 *		    Generate move commands to place the specified amount
 *		    if the specified commodity in the specified sector
 *		    (which MUST be a highway sector).
 *
 *  -m		    Interactive.  Read triplets <comm>,<lev>,<sect> until
 *		    EOF and output the required move commands to accomplish
 *		    the feat.
 *
 *  -i		    Information only.  Report statistical information only,
 *		    do not generate any moves.
 *
 *  -w		    Give a warning message of a distribution's destination
 *		    is not a highway sector
 *		
 */

#include <stdio.h>

/*
 *  Note: Column numbering starts at 0.
 */
#define EMPFILES "EMPFILES"

#define COST(sect) (100 - (sect)->Holds[EFF] + (((sect)->IsHighway)?0:50))
#define ABS(x) (((x)>0)?(x):-(x))
#define NUMEN (sizeof(CommTxt)/sizeof(CommTxt[0]))  /* # of entries     */
#define XNUMEN (NUMEN-3)			    /* REAL commodities */
#define HEFFIC  85	/* highway sectors must be >=HEFFIC   */
#define MINTHR  50
#define MINFILL 80
#define MAXFILL 120	/* in percentage of request to fill	*/
#define NDIR	6
#define LEVID	"lev"
#define COMID	"com"
#define CENID	"cen"
#define XLEVID	"xlev"
#define AIDSID  "plague"
#define HIGHWAY(c)  ((c)=='+' || (c)=='#' || (c)=='=')

#define CIV	0
#define MIL	1
#define HCM     12
#define EFF	15
#define MOB	16

char *CommTxt[] = { "civ","mil","uw","food","sh","gun","pet","iron","dust",
		    "bar","oil","lcm","hcm","rad","work","eff","mob"
		  };
char  Consider[] ={     1,    1,   1,    1,    1,    1,    1,     1,     1,
			1,    1,    1,    1,    1,     1,    1,    1
		  };
char  Disted[]   ={     0,    0,   1,    1,    1,    1,    1,     1,     1,
			1,    1,    1,    1,    1,     0,    0,    0
		  };
short ColWidth[]= {	4,    4,   4,	 4,    4,    4,    4,	  4,	 4,
			4,    4,    4,	  4,	4,     3,    3,    4
		  };
short LevCols[] = {    -1,   -1,  28,	33,   37,   41,   45,	 50,	55,
		       59,   63,   67,	 71,   75,    -1,   -1,   -1
		  };
short ComCols[] = {    -1,   -1,  -1,	-1,   31,   37,   41,	 46,	51,
		       56,   61,   66,	 71,   75,    -1,   -1,   -1
		  };
short CenCols[] = {    29,   33,  37,	43,   -1,   -1,   -1,	 -1,	-1,
		       -1,   -1,   -1,	 -1,   -1,    72,   11,   15
		  };
short XLevCols[]= {    11,   16,  -1,	-1,   -1,   -1,   -1,	 -1,	-1,
		       -1,   -1,   -1,	 -1,   -1,    -1,   -1,   -1
		  };
short PlagCols[]= {    -1,   -1,  -1,	-1,   -1,   -1,   -1,	 -1,	-1,
		       -1,   -1,   -1,	 -1,   -1,    -1,   -1,   -1
		  };


char  DirText[NDIR] = { 'j', 'u', 'y', 'g', 'b', 'n' };
short DirX[NDIR]    = {  2 ,  1 , -1 , -2 , -1 ,  1  };
short DirY[NDIR]    = {  0 , -1 , -1 ,	0 ,  1 ,  1  };
short Back[NDIR]    = {  3 ,  4 ,  5 ,  0 ,  1 ,  2  };


/*
 *  These hold the current totals for various things for the current graph
 *  being worked on (since a country might contain several unconnected
 *  graphs... islands, we will work on one graph at a time).
 */

long  TtlComm[NUMEN];	    /*  Sum all commodities in the country	*/
long  TtlLevel[NUMEN];	    /*  Sum all thresholds in the country	*/

long  TtlAvail[NUMEN];	    /*  Sum Amounts free to be moved (in sectors
				on or adjacent to a highway		*/
long  TtlNeed[NUMEN];	    /*  Sum Amounts in need (adjusted levels in
				highway sectors				*/
long  TtlCanTake[NUMEN];    /*  Amounts available for taking.		*/
long  TtlMustFill[NUMEN];   /*  Amounts required 			*/

char Priority;
char Verbose;
char Path;
char CivAv;
char NoFiles = 1;	    /*  If no files specified, use env var.	*/
short PSx,PSy,PDx,PDy;

short tab = 1;

#define LEVDESTCOL  21
#define SKIPROWS    4
#define CENDES	    8

/*
 *  Each sector is stored in a graph joined together by pointers.  Only
 *  those pointers TO highway sectors exist (i.e. no pointers exist from
 *  a highway to a lcm factory, but pointers exist from the lcm factory
 *  to the highway).  The graph determines possible movement.
 */

#define SECTOR	struct _SECTOR

SECTOR {
    SECTOR  *Next;	    /*	Linked list (used to setup Directed */
    long    Holds[NUMEN];   /*	Current values		*/
    long    Needs[NUMEN];   /*	Current needs		*/
    short   Des;	    /*	Current Designation	*/
    short   X, Y;	    /*	Sector Position 	*/
    short   DX,DY;	    /*	Current Dist destination*/
    char    ChkOff[NUMEN];  /*  Commodity checked off   */
    char    M;		    /*  Used for graph traversal*/
    char    IsHighway;	    /*  Is a highway sector	*/
    char    Plag;	    /*  Has aids!		*/

    short   Away;	    /*  # sectors to dest	*/
    short   Dir;	    /*  Used for graph trav.	*/
    long    Best;	    /*  Best route to dest (len)*/

    SECTOR  *Dest;	    /*	Dist Dest.		*/
    SECTOR  *Directed[NDIR];/*	6 possible directions	*/
};

SECTOR *Sects;		    /*	Remaining sectors		    */
SECTOR *Bin;
char   *Error;

extern char *GetPathFrom();
extern char *getenv();
extern void optpath();
extern void incorporate_file();
extern char *AToSect();
extern char *OptPath();
extern SECTOR *FindSect();
extern char *malloc();

main(ac,av)
char **av;
{
    short i;
    short consider = 0;
    long count;

    if (av[0][0] == 'p') {
	Path = 1;
	Verbose = 1;

	i = (av[1]) ? av[1][0] : 0;
	if ((i >= '0' && i <= '9') || (i == '-')) {
	    AToSect(av[1], &PSx, &PSy);
	    AToSect(av[2], &PDx, &PDy);
	    av += 2;
	    ac -= 2;
	    Path = 2;
	}
    }

    for (i = 1; i < ac; ++i) {
	register char *ptr = av[i];
	if (*ptr != '-') {
	    incorporate_file(ptr);
	    NoFiles = 0;
	} else {
	    while (*++ptr) {
		switch(*ptr) {
		case 'T':
		    CivAv = 1;
		    puts("CivAv");
		    break;
		case 'p':
		    Path = 1;
		    Verbose = 0;
		    break;
		case 'P':
		    AToSect(ptr+1, &PSx, &PSy);
		    Priority = NUMEN;
		    ptr = "\0";
		    break;
		case 'o':
		    if (freopen(ptr+1, "w", stderr) == NULL) 
			exit(1);
		    ptr = "\0";
		    break;
		case 'r':
		    if (CAToI(ptr+1) >= 0) {
			Consider[CAToI(ptr+1)] = 0;
		    } else {
			fprintf(stderr, "-r option, illegal commodity\n");
			exit(1);
		    }
		    break;
		case 'V':
		    Verbose = 1;
		case 'v':
		    break;
		case 'c':
		    if (consider == 0) {
			consider = 1;
			bzero(Consider, sizeof(Consider));
		    }
		    if (CAToI(ptr+1) >= 0) {
			Consider[CAToI(ptr+1)] = 1;
		    } else {
			fprintf(stderr, "-c option, illegal commodity\n");
			exit(1);
		    }
		    ptr = "\0";
		    break;
		}
	    }
	}
	if (Error)
	    break;
    }
    if (NoFiles) {
	char *files = getenv(EMPFILES);
	char c;
	short i;
	if (files == NULL) {
	    fprintf(stderr, "No files specified and no env.var EMPFILES\n");
	} else {
	    while (*files) {
		while (*files == ' ' || *files == ',')
		    ++files;
		for (i=0; files[i] && files[i] != ' ' && files[i] != ',';++i);
		c = files[i];
		files[i] = 0;
		incorporate_file(files);
		files += i;
		*files = c;
	    }
	}
    };
    init_graph();	/* Directed: to and from highways, nonhws->highways */
    if (Error) {
	fprintf(stderr, "%s\n", Error);
	exit(1);
    }
    while (Path) {
	long loss;
	char *path;

	if (Path == 1) {
	    char buf[64];
	    if (Verbose) {
		printf("sx,sy dx,dy : ");
		fflush(stdout);
	    }
	    if (gets(buf) == NULL)
		 break;
	    AToSect(AToSect(buf, &PSx, &PSy), &PDx, &PDy);
	}
	path = OptPath(FindSect(PSx,PSy), FindSect(PDx,PDy), &loss);
	if (path) {
	    if (Verbose)
	        printf("path %3ld,%-3ld -> %3ld,%-3ld (loss %3ld) = ", 
		    PSx,PSy,PDx,PDy,loss
	        );
	    puts(path);
	} else {
	    printf("no path could be found %3ld,%-3ld -> %3ld,%-3ld\n", 
		PSx, PSy, PDx, PDy
	    );
	}
	if (Path == 2)
	    break;
    }
    if (Path)
	exit(0);
    country_sums();			    /*  TtlComm and TtlLevel*/
    Bin = Sects;
    Sects = NULL;
    while (count = get_subgraph())
        process_graph(count);
    if (Error) {                            /*  Report possible err.*/
	fprintf(stderr, "%s\n", Error);
	exit(1);
    }
    exit(0);
}

/*
 *  Incorporate a file by determining its type and picking out fields
 */

void
incorporate_file(name)
char *name;
{
    char buf[256];
    FILE *fi;
    short skip = 0;
    short *cols;

    fi = fopen(name, "r");
    if (fi == NULL) {
	Error = "Unable to open file";
	return;
    }

    if (Verbose)
        fprintf(stderr, "File: %s\n", name);

    /*
     *	Determine file type
     */

    Error = "Couldn't Figure out File";
    while (fgets(buf, 256, fi)) {
	SECTOR Sect;
	short type;

	Error = NULL;
	bzero(&Sect, sizeof(Sect));
	if (skip < SKIPROWS) {
	    if (skip == 0) {        /*  First line  */
		if (strncmp(buf, AIDSID,sizeof(AIDSID)-1) == 0) {
		    type = 5;
		    cols = PlagCols;
		}
		if (strncmp(buf, XLEVID,sizeof(XLEVID)-1) == 0) {
		    type = 4;
		    cols = XLevCols;
		}
		if (strncmp(buf, LEVID, sizeof(LEVID)-1) == 0) {
		    type = 1;
		    cols = LevCols;
		}
		if (strncmp(buf, COMID, sizeof(COMID)-1) == 0) {
		    type = 2;
		    cols = ComCols;
		}
		if (strncmp(buf, CENID, sizeof(CENID)-1) == 0) {
		    type = 3;
		    cols = CenCols;
		}
		if (!type)
		    break;
	    }
	    Error = "Unexpected EOF";
	    ++skip;
	    continue;
	}
	{
	    register short len = strlen(buf) - 1;
	    while (len && buf[len] == '\n' || buf[len] == 9 || buf[len] == ' ')
		--len;
	    buf[++len] = 0;
	    if (len < 17)       /*  end of scan */
		break;
	    AToSect(buf, &Sect.X, &Sect.Y);
	    Sect.DX = 9999;
	    if (type == 1)
		AToSect(buf+LEVDESTCOL, &Sect.DX, &Sect.DY);
	    if (type == 3) {
		Sect.Des = buf[CENDES];
		Sect.IsHighway = HIGHWAY(Sect.Des);
	    }
	    if (Error)
		break;
	}
	{
	    short i, j;
	    register SECTOR *s;

	    if ((s = FindSect(Sect.X, Sect.Y)) == NULL) {
		s = (SECTOR *)malloc(sizeof(SECTOR));
		bzero(s, sizeof(*s));
		s->DX = Sect.X;
		s->DY = Sect.Y;
		s->Next = Sects;
		Sects = s;
	    }
	    for (i = 0; i < NUMEN; ++i) {
		if ((j = cols[i]) >= 0) {         /*  process sector data */
		    long v = 0;
		    short n;
		    for (n = ColWidth[i]; n; --n) {
			if (buf[j] >= '0' && buf[j] <= '9')
			    v = v * 10 + buf[j] - '0';
			++j;
		    }
		    if (type == 1 || type == 4)
			Sect.Needs[i] = v;
		    else
			Sect.Holds[i] = v;
		    if (i == MIL && type == 3 && !s->Needs[i]) {
			if (Sect.Holds[i] > 100 && Sect.IsHighway)
			    Sect.Needs[i] = 100;
			else
			    Sect.Needs[i] = Sect.Holds[i];
		    }
		    if (i == CIV && type == 3 && !s->Needs[i]) {
			if (CivAv) {
			    Sect.Needs[i] = 800;
			} else if (Sect.IsHighway) {
			    if (Sect.Holds[i] > 800)
			        Sect.Needs[i] = 800;
			    else
			        Sect.Needs[i] = Sect.Holds[i];
			}
		    }
		}
	    }
	    if (type == 5) {	/* plague */
		s->Plag = 1;
	    }
	    for (i = 0; i < NUMEN; ++i) {
		if (Sect.Needs[i])
		    s->Needs[i] = Sect.Needs[i];
		if (Sect.Holds[i])
		    s->Holds[i] = Sect.Holds[i];
	    }
	    if (Sect.Des)
		s->Des = Sect.Des;
	    s->X = Sect.X;
	    s->Y = Sect.Y;
	    if (Sect.DX != 9999) {
		s->DX = Sect.DX;
		s->DY = Sect.DY;
	    }
	}
    }
    fclose(fi);
}


/*
 *  Fill in the graph arcs and distribution destination pointer, if
 *  possible.  Then, modify the levels in the dist dest sectors (if not h)
 *  to reflect requirements.
 */

init_graph()
{
    register SECTOR *s, *ss;
    register short i;


    for (s = Sects; s; s = s->Next) {
	s->IsHighway = HIGHWAY(s->Des) && s->Holds[EFF] >= HEFFIC; 
	if (s->DX != s->X || s->DY != s->Y) {  
	    if (s->Dest = FindSect(s->DX, s->DY)) {
		if (!s->IsHighway) {
		    for (i = 0; i < XNUMEN; ++i) {
			/*  note:  0 in s->Needs -> no dist.	*/
			if (Disted[i]) {
			    if (s->Needs[i] && s->Needs[i] - s->Holds[i] > 0) {
			        s->Dest->Needs[i] += s->Needs[i] - s->Holds[i];
				s->Needs[i] = s->Holds[i];
			    }
			}
		    }
		}
	    } else {
		if (Verbose && !Path) {
		    fprintf(stderr, "Bad Dist path: %ld,%ld -> %ld,%ld\n",
		        s->X, s->Y, s->DX, s->DY
		    );
		}
	    }
	}
	for (i = 0; i < NDIR; ++i) {
	    if (ss = FindSect(s->X + DirX[i], s->Y + DirY[i])) {
		s->Directed[i] = ss;
		ss->Directed[Back[i]] = s;
	    }
	}
    }
}

/*
 *  Determine country-wide commodity levels and requirements.  Generated for
 *  user information only.
 */

country_sums()
{
    register SECTOR *s;
    register short i;
    long count = 0;

    for (s = Sects; s; s = s->Next) {
	for (i = 0; i < XNUMEN; ++i) {
	    TtlComm[i] += s->Holds[i];
	    TtlLevel[i]+= s->Needs[i];
	}
	++count;
    }
    fprintf(stderr, "                               %ld sectors\n", count);
    fprintf(stderr, "   com    ttl  thrsh           (COUNTRY TOTALS)\n");
    for (i = 0; i < XNUMEN; ++i) {
        fprintf(stderr, "%6s %6ld %6ld (%6ld)\n",
	    CommTxt[i], TtlComm[i], TtlLevel[i], TtlComm[i] - TtlLevel[i]
	);
    }
    fprintf(stderr, "\n");
    return(count);
}

/*
 *    GET_SUBGRAPH()
 *
 *    (1) Throw away highway sectors in Sects
 *    (2) Move a highway sector from Bin into Sects
 *    (3) Move all related sectors to Sects
 *
 *    Note:  Any non-highway sectors is considered a barrier.  the barrier
 *           is included in both subgraphs but the two subgraphs are not
 *	     linked together.
 */

get_subgraph()
{
    register SECTOR *s, **ss;
    register long count = 0;

    if (Bin == NULL)
	return(0);
    for (s = Sects; s;) { 
	register SECTOR *sn = s->Next;
	if (!s->IsHighway) {
	    s->Next = Bin;
	    Bin = s;
	}
	s = sn;
    }
    Sects = NULL;			/* forget previous highways	*/
    for (s = Bin; s; s = s->Next)
	s->M = 0;
    for (s = Bin; s; s = s->Next) {	/* Find a highway to seed subg  */
	if (s->IsHighway)
	    break;
    }
    if (!s)
	return(0);
    fprintf(stderr, "@ %ld,%ld\n", s->X, s->Y);
    subgtraverse(s);			/* Traverse the subgraph	*/
    for (ss = &Bin; s = *ss;) {		/* And place into Sects		*/
	if (s->M) {
	    *ss = s->Next;
	    s->Next = Sects;
	    s->M = 0;
	    Sects = s;
	    ++count;
	} else {
	    ss = &s->Next;
	}
    }
    return(count);
}

subgtraverse(s)
register SECTOR *s;
{
    register short i;
    register SECTOR *to;

    s->M = 1;
    for (i = 0; i < NDIR; ++i) {
	if ((to = s->Directed[i]) && to->M == 0) {
	    if (to->IsHighway)
		subgtraverse(to);
	    else
	        to->M = 1;
	}
    }
}

/*
 *	PROCESS_GRAPH()
 *
 *	Process the current sub-graph.
 */

process_graph(count)
{
    register SECTOR *s;
    long percentage[NUMEN];
    long txfered[NUMEN];
    long nummoves;
    long sum;

    nummoves = 0;
    bzero(txfered, sizeof(txfered));

    bzero(TtlAvail, sizeof(TtlAvail));
    bzero(TtlNeed , sizeof(TtlNeed));
    bzero(TtlCanTake, sizeof(TtlCanTake));
    bzero(TtlMustFill, sizeof(TtlMustFill));

    for (s = Sects; s; s = s->Next) {
	register short i;
	long tmp;

	for (i = 0; i < XNUMEN; ++i) {
            TtlNeed[i] += s->Needs[i];
	    if (s->Plag)
		continue;
            TtlAvail[i] += s->Holds[i];
	    if (i == CIV || i == MIL)
	        tmp = s->Holds[i] - s->Needs[i];
	    else
	        tmp = s->Holds[i] - s->Needs[i] * MAXFILL / 100;
	    if (tmp > 0)
		TtlCanTake[i] += tmp;
	    tmp = s->Holds[i] - s->Needs[i];
	    if (tmp < 0)
		TtlMustFill[i] -= tmp;	
	}
    }
    {
	register short i;
        for (i = sum = 0; i < XNUMEN; ++i)
            sum += TtlAvail[i] + TtlNeed[i] + TtlCanTake[i] + TtlMustFill[i];
    }
    if (sum) {
	register short i;

        fprintf(stderr, "                     SUBGRAPH %ld sectors\n",count);
	fprintf(stderr, "commod    ttl  thrsh   free   need\n\n");
	for (i = 0; i < XNUMEN; ++i) {
	    fprintf(stderr, "%6s %6ld %6ld %6ld %6ld (%6ld)\n",
		CommTxt[i], TtlAvail[i], TtlNeed[i], TtlCanTake[i], 
	    	TtlMustFill[i], TtlCanTake[i] - TtlMustFill[i]
	    );
	}
	fprintf(stderr, "\n");
    }

    /*
     *	(2a). The totals are now known, calculate the percentage (percentage
     *	      of requested goods)  we can fill the sector, from 0 to 110%
     *	      when calculating moves below, a minimum of 1 unit of material
     *	      is moved no matter what the fill % is.
     */

    {
	register short i;
	for (i = 0; i < XNUMEN; ++i) {
	    if (TtlMustFill[i]) 
		percentage[i] = TtlCanTake[i] * 100 / TtlMustFill[i];
	    else
		percentage[i] = 100;
	    if (percentage[i] > MAXFILL)
		percentage[i] = MAXFILL;
	}
    }

    /*
     *  (2b).  Fill each sector according to need, starting with the most
     *	       needfull of sectors (have/need -> 0).  This is done by
     *	       finding the worse sector and commodity in the system,
     *	       generating the moves for it, updating commodity levels,
     *	       and checking it off.
     */

    {
	register short i;
	register short bestcom;
	register SECTOR *best;
	long  ratio;
	long  worstratio;
	short notdone = 1;
	short priority = (Priority) ? Priority + 1 : 0;

	while (notdone || priority) {
	    notdone = 0;
	    worstratio = 10000;
	    best = NULL;

	    s = Sects;
	    if (priority) {
		if (--priority) {
		    s = FindSect(PSx, PSy);
		    if (s == NULL)
		        s = Sects;
		}
	    }
	    for (; s; s = (priority) ? NULL : s->Next) {
		/*
	         * if (!s->IsHighway)
		 *    continue;
		 */
	        for (i = 0; i < XNUMEN; ++i) {
		    if (s->ChkOff[i] || s->Holds[i] >= s->Needs[i])
			continue;
		    if (!Consider[i])
			continue;
		    notdone = 1;
		    ratio=(s->Needs[i])?s->Holds[i]*10000/s->Needs[i]:10000;
		    if (ratio == 0) {
			best = s;
			bestcom = i;
			goto dblbreak;	/* smallest we can get! */
		    }
		    if (ratio < worstratio) {
		        best = s;
		        bestcom = i;
		        worstratio = ratio;
		    }
		}
	    }
dblbreak:
	    if (best) {
		long needs = best->Needs[bestcom];
		long holds = best->Holds[bestcom];
		long amount = needs - holds;
		long actual = amount * percentage[bestcom] / 100;

		if (actual < 0) {
		    fprintf(stderr, "ACTUAL < 0!: %ld,%ld (%ld)\n",
			amount, percentage[bestcom], bestcom
		    );
		    actual = 0;
		}
		if (actual == 0 && TtlCanTake[bestcom])
		    ++actual;
		if (Verbose) {
		    fprintf(stderr, "holds %3ld needs %3ld (%%=%3ld)",
		        holds, needs, percentage[bestcom]
		    );
		}

		/*
		 *  Optimization.  If we can't fill it with a reasonable amount
		 *  of the commodity (<100%) and it is already reasonably full
		 *  (>MINFILL) and has at least MINTHR products in it already,
		 *  don't bother to do the move (wait for the available supply
		 *  to go up or the sector to become more needy).
		 */

		best->ChkOff[bestcom] = 1;
		if (needs && holds > MINTHR) {
		    if (holds*100/needs > MINFILL && 
		     (holds+actual)*100/needs < 100) {
			if (Verbose) {
			 fprintf(stderr, 
		    	    "Could : %5ld(%5ld) %4s into %3ld,%-3ld ",
		            actual, amount, CommTxt[bestcom], best->X, best->Y
			 );
			 fprintf(stderr, " (but won't)\n");
			}
		        continue;
		    }
		}
		if (Verbose) {
		    fprintf(stderr, "Placing %5ld(%5ld) %4s into %3ld,%-3ld\n",
		        actual, amount, CommTxt[bestcom], best->X, best->Y
		    );
		}
		if (actual)
		    MoveCommodity(best, bestcom, actual, txfered);
	    }
	}
    }
    if (sum) {
        register short i;

        for (i = 0; i < XNUMEN; ++i) {
	    if (txfered[i])
	        break;
	}
	if (i != XNUMEN) {
	    fprintf(stderr, "commod    txfered\n\n");
	    for (i = 0; i < XNUMEN; ++i) {
	        if (txfered[i] == 0)
		    continue;
		fprintf(stderr, "%6s %6ld\n", CommTxt[i], txfered[i]);
	    }
   	    fprintf(stderr, "\n");
	}
    }
}

/*
 *   Move the specified amounts of the specified commodity TO the specified
 *   sector.  If it is possible to do in one move command, this is done.
 *   otherwise, multiple moves (from various areas) will be issued.  Priority:
 *
 *	(1) 0 mob cost in one move command
 *	(2) 0 mob cost in N move commands
 *	(3) N mob cost in N move commands
 *
 *   If no area has enough stuff, no move is issued.  A bredth-first search
 *   is initiated at the destination node trying to find a suitable source.
 */

MoveCommodity(to, com, amount, stats)
SECTOR *to;
short com;
long amount;
long *stats;
{
    short opt_notcalled = 1;

    /*
     *	  First try to avoid the general best-path graph traversal routine
     *    OptPath() by calling HWayPath(), which only traverses 100%
     *    highways.  If this fails, however, call OptPath().
     */

    HWayPath(to);
    while (amount) {
	long avail;
	long bavail;
	long best = -1;
	long loss;
	register SECTOR *pick;
	register SECTOR *s;
	char *path;

        for (s = Sects; s; s = s->Next) {
	    if (s->Best < 0 || (best != -1 && s->Best > best))
		continue;
	    if (s->Holds[com] <= s->Needs[com] || s->Plag || s->ChkOff[com])
		continue;
	    avail = s->Holds[com] - s->Needs[com];	/* for use */
	    {
		register long tavail;

		tavail = (com == CIV || com == MIL) ? avail :
		    s->Holds[com] - s->Needs[com] * MAXFILL / 100;
		if (tavail <= 0)
		    continue;
	    }
	    best = s->Best;
	    pick = s;
	    bavail = avail;
	    if (best == 0 && bavail >= amount)
		break;
	}
	if (best == -1) {
	    if (opt_notcalled) {	   /* highway-only failed      */
    		OptPath(NULL, to, NULL);   /* resort to general g.srch */
		opt_notcalled = 0;
		continue;
	    }
	    break;
	}
	if (bavail > amount)
	    bavail = amount;
	path = GetPathFrom(pick, to);
	if (!path) {
	    fprintf(stderr, "Error, unable to get path: %ld,%ld to %ld,%ld\n",
		pick->X, pick->Y, to->X, to->Y
	    );
	    continue;
	}
	if (pick->Best) {
	    if (opt_notcalled) {	   /* highway-only failed      */
    		OptPath(NULL, to, NULL);   /* resort to general g.srch */
		opt_notcalled = 0;
		continue;
	    }

	    if ((com != CIV && com != MIL) || pick->Best > 100) {
		fprintf(stderr, "Loss of %3ld (%3ld %4s from ",
		    pick->Best, bavail, CommTxt[com]
		);
		fprintf(stderr, "%3ld,%-3ld to %3ld,%-3ld) no move made\n",
		    pick->X, pick->Y, to->X, to->Y
		);
	        break;
	    }
	    if (pick->Holds[MOB] < 64+(bavail/7)) {
		fprintf(stderr, "Warning: moving %3ld %4s from %3ld,%-3ld ",
		    bavail, CommTxt[com], pick->X, pick->Y
		);
		fprintf(stderr, " Mobility @%-3ld (move made)\n", 
		    pick->Holds[MOB]
	 	);
	    }
	}
	amount -= bavail;
	if ((TtlCanTake[com] -= bavail) < 0)
	    TtlCanTake[com] = 0;
	pick->Holds[com] -= bavail;
	to->Holds[com] += bavail;
	printf("move %s %d,%d %d v%s\n\n", CommTxt[com], pick->X,
	    pick->Y, bavail, path
	);
	if (stats)
	    stats[com] += bavail;
    }
    if (amount) {
	fprintf(stderr,"Error: Unable to move remaining %ld %-6s into %d,%d\n",
	    amount, CommTxt[com], to->X, to->Y
        );
    }
}

char *
AToSect(buf, x, y)
char *buf;
short *x;
short *y;
{
    short v = 0;
    short n = 0;
    while (*buf == ' ')
	++buf;
    *x = atoi(buf);
    while (*buf == '-' || (*buf >= '0' && *buf <= '9'))
	++buf;
    if (*buf != ',') {
	Error = "Unable to get Sector Coordinates";
	return;
    }
    ++buf;
    while (*buf == ' ')
	++buf;
    if (*buf == '-') {
	n = 1;
	++buf;
    }
    while (*buf >= '0' && *buf <= '9') {
	v = v * 10 + *buf - '0';
	++buf;
    }
    *y = (n) ? -v : v;
    return(buf);
}

/*
 *  Search for the specified sector.  Very slow linked list but only
 *  done at initialization to link the graph together via pointers.
 */

SECTOR *
FindSect(x, y)
register short x, y;
{
    register SECTOR *s;

    for (s = Sects; s; s = s->Next) {
	if (s->X == x && s->Y == y)
	    return(s);
    }
    return(NULL);
}

/*
 *    Determine the optimal path from one sector to another.  The source
 *    sector must exist in the Graph list.  Two routines exist.  HWayPath()
 *    is almost linear and finds 100%-highway-only paths to the destination.
 *    OptPath() is general and finds the best (possibly not 100%) path.
 *
 *    HWayPath() does not necessarily find the physically shortest path,
 *    though always mob==0 paths.
 */

HWayPath(d)
SECTOR *d;
{
    register SECTOR *ss;
    for (ss = Sects; ss; ss = ss->Next) {
        ss->Best = -1;
	ss->M = 0;
    }
    if (!COST(d)) {
        d->Best = 0;
        d->Away = 0;
        hwaypath(d);
    }
}

hwaypath(d)
register SECTOR *d;
{
    register short i;

    d->M = 1;
    for (i = 0; i < NDIR; ++i) {
	register SECTOR *ss;
	
	if ((ss = d->Directed[i]) && ss->M == 0) {
	    ss->Away = d->Away + 1;
	    ss->Best = 0;
	    ss->Dir = Back[i];
	    if (ss->IsHighway && ss->Holds[EFF] == 100)
		hwaypath(ss);
	}
    }
}


char *
OptPath(s, d, loss)
SECTOR *s, *d;
long *loss;
{
    register SECTOR *ss;
    static char Path[1024];
    short max;

    for (ss = Sects; ss; ss = ss->Next) {
        ss->Best = -1;
	ss->M = 0;
    }
    if (d) {
	d->Best = COST(d);
	d->Away = 0;
	optpath(d);

	if (!s)
	    return(NULL);
	if (s->Best >= 0) {
	    register short i;
	    for (ss = s, i = 0; ss != d && i < sizeof(Path)-1; ++i) {
		if (ss->Best < 0) {
		    puts("OptPath, software error1");
		    exit(1);
		}
		Path[i] = DirText[ss->Dir];
		ss = ss->Directed[ss->Dir];
		if (ss == NULL) {
		    puts("OptPath, software error2");
		    exit(1);
		}
	    }
	    Path[i] = 0;
	    if (i == sizeof(Path)-1) {
		puts("OptPath, Path error");
		puts(Path);
		exit(1);
	    }
	    *loss = s->Best;
	    return(Path);
	}
    }
    return(NULL);
}

char *
GetPathFrom(s, d)
register SECTOR *s;
SECTOR *d;
{
    static char Path[1024];
    register SECTOR *ss;
    register short i;
    
    if (s->Best < 0)
	return(NULL);
    for (ss = s, i = 0; ss->Away && i < sizeof(Path)-1; ++i) {
	if (ss->Best < 0) {
	    fputs("GetPathFrom, soft error 1\n", stderr);
	    return(NULL);
	}
	Path[i] = DirText[ss->Dir];
	ss = ss->Directed[ss->Dir];
	if (ss == NULL) {
	    fputs("GetPathFrom, soft error 2\n", stderr);
	    return(NULL);
	}
    }
    if (ss != d) {
	fprintf(stderr, "GetPathFrom, Soft error, expected %ld,%ld\n",
	    d->X, d->Y
	);
	fprintf(stderr, " got instead: %ld,%ld\n", ss->X, ss->Y);

    }
    Path[i] = 0;
    if (i == sizeof(Path)-1) {
	fputs("GetPathFrom, path error\n", stderr);
	fprintf(stderr, "path: %s\n", Path);
	return(NULL);
    }
    return(Path);
}

/*
 *
 */

void
optpath(d)
register SECTOR *d;
{
    register short i;
    register SECTOR *ss;
    long obest;
    long best;

    if (d->Best > 100 || d->Away > 128)
	return;
loop:
    obest = d->Best;
    for (i = 0 ; i < NDIR; ++i) {
        best = obest + COST(d);
	if (ss = d->Directed[i]) {
	    if (ss->Best < 0) {
		ss->Best = best;
		ss->Dir = Back[i];
		ss->Away = d->Away + 1;
		optpath(ss);
	    } else if (best < ss->Best) {
		ss->Best = best;
		ss->Dir = Back[i];
		ss->Away = d->Away + 1;
		optpath(ss);
	    } else if (best == ss->Best) {
		if (d->Away+1 < ss->Away) {
		    ss->Best = best;
		    ss->Dir = Back[i];
		    ss->Away = d->Away + 1;
		    optpath(ss);
		}
	    }
	}
    }
    if (obest != d->Best) 
	goto loop;
}
	 

CAToI(str)
char *str;
{
    register short i;

    for (i = 0; i < XNUMEN; ++i) {
	if (strcmp(str, CommTxt[i]) == 0)
	    return(i);
    }
    return(-1);
}

PCToD(c)
{
    register short i;

    for (i = 0; i < NDIR; ++i) {
	if (DirText[i] == c)
	    return(i);
    }
    fprintf(stderr, "PCToD failed w/char (%c) %d\n", c, c);
    exit(1);
}

