/*
 *	File list.c
 *
 *	List InitList ()
 *		Allocates a new list
 *
 *	void AddGroup (list, group)
 *		Adds a new group
 *
 *	Group GetGroup (list, groupname)
 *		Returns the group "groupname"
 *
 *	void DeleteGroup (list, groupname)
 *		Deletes a group
 *
 *	char * FirstGroup (list)
 *		Returns the first group
 *
 *	char * NextGroup (list)
 *		Returns the next group
 *
 *	int NrGroups (list)
 *		Returns the number of groups
 *
 *	void AddUser (list, groupname, username, address)
 *		Adds a new user
 *
 *	User FirstUser (group)
 *		Returns the first user of a group
 *
 *	User NextUser (group)
 *		Returns the next user of a group
 *
 *	void DeleteUser (list, groupname, username)
 *		Deletes a user
 *
 *	char * UserName (user)
 *		Returns the name of the user
 *
 *	char * UserAddress (user)
 *		Returns the address of the user
 *
 *	char * UserFrom (from)
 *		Returns the address found in the from field of the message
 *
 *	int NrUsers (list, group)
 *		Returns the number of users in "group"
 *
 *	char * UserAddress (list, groupname, username)
 *		Returns the email-address of the user
 *
 *	void SaveList (list, filename)
 *		Saves list in "filename"
 *
 *	void LoadList (list, filename)
 *		Loads the list from "filename"
 *
 *	void FreeList (list)
 *		Frees up all memory
 */

#include "dlist.h"
#include "list.h"
#include <stdio.h>

#define MAX_USERNAMELEN		64
#define	MAX_ADDRESSLEN		64
#define	MAX_GROUPNAMELEN	64

typedef char * UserList;
typedef char * GroupList;

struct s_user
{
	char username [MAX_USERNAMELEN];
	char address [MAX_ADDRESSLEN];
};

struct s_group
{
	char groupname [MAX_GROUPNAMELEN];
	UserList ulist;
};

List InitList ()
{
	return (List) DLNew (0, sizeof (struct s_group));
}

int NrGroups (list)
List list;
{
	return DLElems (list);
}

void AddGroup (list, groupname)
List list;
char * groupname;
{
	Group group;

	group = (Group) DLAddElem (list, TAIL);
	(void) strncpy (group-> groupname, groupname, MAX_GROUPNAMELEN);
	group-> ulist = (UserList) DLNew (0, sizeof (struct s_user));
}

int UsersInGroup (group)
Group group;
{
	return DLElems (group);
}

Group GetGroup (list, groupname)
List list;
char * groupname;
{
	Group group;

	for (group = (Group) DLGet (list, HEAD);
	     group != (Group) 0;
	     group = (Group) DLGet (list, NEXT))
		if (strcmp (group-> groupname, groupname) == 0)
			return group;

	return (Group) 0;
}

void DeleteGroup (list, groupname)
List list;
char * groupname;
{
	Group group;

	group = GetGroup (list, groupname);
	if (group == (Group) 0)
		(void) fprintf (stderr,
			"DeleteGroup: Group \"%s\" not found.\n", groupname);
	else
		DLDelete (list, CURRENT, FREE);
}

char * FirstGroup (list)
List list;
{
	Group group;

	group = (Group) DLGet (list, HEAD);
	return group-> groupname;
}

char * NextGroup (list)
List list;
{
	Group group;

	group = (Group) DLGet (list, NEXT);
	return group-> groupname;
}


char * UserName (user)
User user;
{
	return user-> username;
}

char * UserAddress (user)
User user;
{
	return user-> address;
}

void AddUser (list, groupname, name, address)
List list;
char * groupname;
char * name, * address;
{
	Group group;
	User user;

	group = GetGroup (list, groupname);
	if (group != (Group) 0)
	{
		user = (User) DLAddElem (group-> ulist, TAIL);
		(void) strncpy (user-> username, name, MAX_USERNAMELEN);
		(void) strncpy (user-> address, address, MAX_ADDRESSLEN);
	}
	else
		(void) fprintf (stderr, "AddUser: Couldn't find \"%s\".\n",
					groupname);
}

User FirstUser (group)
Group group;
{
	return (User) DLGet (group-> ulist, HEAD);
}

User NextUser (group)
Group group;
{
	return (User) DLGet (group-> ulist, NEXT);
}

int UserExists (group, username)
Group group;
char * username;
{
	User user;

	for (user = FirstUser (group);
	     user != (User) 0;
	     user = NextUser (group))
		if (strcmp (user-> username, username) == 0)
			return 1;
	
	return 0;
}

void DeleteUser (list, groupname, username)
List list;
char * groupname, * username;
{
	Group group;
	User user;

	group = GetGroup  (list, groupname);
	if (group == (Group) 0)
	{
		(void) fprintf (stderr, "DeleteUser: Group \"%s\" not found.\n",
					groupname);
		return;
	}

	for (user = FirstUser (group);
	     user != (User) 0;
	     user = NextUser (group))
		if (strcmp (user-> username, username) == 0)
		{
			DLDelete (group-> ulist, CURRENT, FREE);
			return;
		}

	(void) fprintf (stderr, "DeleteUser: User \"%s\" not found.\n",
				username);
}

void SaveGroup (fp, group)
FILE * fp;
Group group;
{
	fwrite (group-> groupname, sizeof (group-> groupname), 1, fp);
	DLWrite (group-> ulist, fp, FWRITE);
}

void SaveList (list, filename)
List list;
char * filename;
{
	FILE * fp;

	fp = fopen (filename, "w");
	if (fp == (FILE *) 0)
	{
		(void) fprintf (stderr, "Couldn't open '%s'\n", filename);
		perror ("fopen");
		return;
	}

	DLWrite (list, fp, SaveGroup);

	fclose (fp);
}

void LoadGroup (fp, group)
FILE * fp;
Group group;
{
	fread (group-> groupname, sizeof (group-> groupname), 1, fp);
	group-> ulist = DLNew (0, sizeof (struct s_user));
	DLRead (group-> ulist, fp, FREAD);
}
	
void LoadList (list, filename)
List list;
char * filename;
{
	FILE * fp;

	fp = fopen (filename, "r");
	if (fp == (FILE *) 0)
	{
		(void) fprintf (stderr, "Couldn't open '%s'\n", filename);
		perror ("fopen");
		return;
	}

	DLRead (list, fp, LoadGroup);

	fclose (fp);
}
