#include "cs.h"                                             /*  LPANAL.C      */
#include "soundio.h"
#include "lpc.h"
#include <math.h>

/* LPC analysis, modified by BV 8'92 for linkage to audio files via soundin.c.
 * Currently set for maximum of 50 poles, & max anal segment of 1000 samples,
 * meaning that the frame slices cannot exceed 500 samples.
 * Program size expands linearly with slice size, and as square of npoles.
 */

#define DEFNPOLES 34        /* recommeded default (max 50 in lpc.h)    */
#define DEFSLICE 200	    /* <= MAXWINDIN/2 (currently 1000 in lpc.h) */
#define PITCHMIN	70.0
#define PITCHMAX	200.0	/* default limits in Hz for pitch search */

static  int     NPOLES, WINDIN, debug=0, verbose=0, doPitch = 1;
static  void    alpol(), gauss(), quit(), lpdieu(), usage();
extern  void    ptable();
extern  float   getpch();

#define FIND(MSG)   if (*s == '\0')  \
		        if (!(--argc) || (s = *++argv) && *s == '-')  \
			    lpdieu(MSG);

lpanal(argc, argv)
  int argc;
  char **argv;
{
        int 	slice, analframes, counter, channel;
	float	coef[NDATA+MAXPOLES], beg_time, input_dur, sr = 0.;
	char	*infilnam, *outfilnam;
	int     infd, ofd;
	double	errn, rms1, rms2, cc[MAXPOLES];
	float	*sigbuf, *sigbuf2;  	/* changed from short */
	long	n, nsamps;
	int 	nb, osiz, hsize;
	LPHEADER    *lph;
	char	*lpbuf, *tp;
	float	pchlow, pchhigh;
	SOUNDIN *p;             /* struct allocated by SAsndgetset */
extern	int     SAsndgetset();
extern	long    getsndin();
extern  char    *retfilnam;

	lpbuf = mcalloc((long)LPBUFSIZ);
	lph = (LPHEADER *) lpbuf;
	tp = lph->text;

	NPOLES = DEFNPOLES;	    /* DEFAULTS... */
	slice = DEFSLICE;
	channel = 1;
	beg_time = 0.0;
	input_dur = 0.0;		/* default duration is to length of ip */
	*tp = '\0';
	pchlow = PITCHMIN;
	pchhigh = PITCHMAX;

	if (!(--argc))  lpdieu("insufficient arguments");
	do {
	    register char *s = *++argv;
	    if (*s++ == '-')
	        switch (*s++) {
		case 's':       FIND("no sampling rate")
		    sscanf(s,"%f",&sr); break;
		case 'c':       FIND("no channel")
		    sscanf(s,"%d",&channel); break;
		case 'b':       FIND("no begin time")
		    sscanf(s,"%f",&beg_time); break;
		case 'd':       FIND("no duration time")
		    sscanf(s,"%f",&input_dur); break;
		case 'p':	FIND("no poles")
		    sscanf(s,"%d",&NPOLES); break;
		case 'h':	FIND("no hopsize")
		    sscanf(s,"%d",&slice); break; 
		case 'C':	FIND("no comment string")
		    strncat(tp,s,(LPBUFSIZ - sizeof(LPHEADER) + 4));
		    tp += strlen(tp);
		    break;
		case 'P':	FIND("no low frequency")
		    sscanf(s,"%f",&pchlow);
		    if (pchlow == 0.) doPitch = 0;	/* -P0 inhibits ptrack */
		    break;
		case 'Q':	FIND("no high frequency")
		    sscanf(s,"%f",&pchhigh); break;
		case 'v':	FIND("no verbose level")
		    sscanf(s,"%d",&verbose);
		    if (verbose > 1)  debug = 1;
		    break;
		default: sprintf(errmsg,"unrecognized flag -%c", *--s);
		         lpdieu(errmsg);
		}
	    else break;
	} while (--argc);

	if (argc != 2)  lpdieu("incorrect number of filenames");
	infilnam = *argv++;
	outfilnam = *argv;

	if (NPOLES > MAXPOLES)
	    quit("poles exceeds maximum allowed");
	if (slice < NPOLES * 5)
	    warning("hopsize may be too small, recommend at least npoles * 5");
	if ((WINDIN = slice * 2) > MAXWINDIN)
	    quit("input framesize (inter-frame-offset*2) exceeds maximum allowed");

	if (verbose) {
	    fprintf(stderr,"Reading sound from %s, writing lpfile to %s\n",
		    infilnam, outfilnam);
	    fprintf(stderr,"poles=%d hopsize=%d begin=%4.1f duration=%4.1f\n",
		    NPOLES, slice, beg_time, input_dur);
	    fprintf(stderr,"lpheader comment:\n%s\n", lph->text);
	    if (pchlow > 0.)
	        fprintf(stderr,"pch track range: %5.1f - %5.1f Hz\n",pchlow,pchhigh);
	    else fprintf(stderr,"pitch tracking inhibited\n");
	}
	if ((input_dur < 0) || (beg_time < 0))
	    quit("input and begin times cannot be less than zero");
		
	if (!(infd = SAsndgetset(infilnam,&p,&beg_time,&input_dur,&sr,channel))) {
	    sprintf(errmsg,"error while opening %s", retfilnam);
	    quit(errmsg);
	}

	if ((ofd = openout(outfilnam, 1)) < 0)  /* open output file */
	    quit("cannot create output file");
	lph->lpmagic = LP_MAGIC;                /* build output hdr */
	lph->npoles = NPOLES;
	lph->nvals = NPOLES + NDATA;
	lph->srate = p->sr;
	lph->framrate = (float) p->sr / slice;
	lph->duration = input_dur;
	hsize = tp - (char *) lph;      	/* header size including text */
	lph->headersize = (hsize + 3) & -4;     /* rounded up to 4 byte bndry */
	if (lph->headersize > LPBUFSIZ)    /* UNNECESSARY ?? */
	    lph->headersize = LPBUFSIZ;
	if ((nb = write(ofd,(char *)lph,(int)lph->headersize)) < lph->headersize)
	    quit("can't write header");
	osiz = (NPOLES + NDATA) * sizeof(float);

	sigbuf = (float *) mmalloc((long)WINDIN * sizeof(float));
	sigbuf2 = sigbuf + slice;
	if ((n = getsndin(infd, sigbuf, (long)WINDIN, p)) < WINDIN)
	    quit("soundfile read error, couldn't fill first frame");
	if (doPitch)
	    ptable(pchlow, pchhigh, (float) p->sr, WINDIN);
	counter = 0;
	analframes = (p->getframes - 1) / slice;
	do {
	    register float *fp1, *fp2;
	    register double *dfp;
	    counter++;
	    alpol(sigbuf, &errn, &rms1, &rms2, cc);
	    coef[0] = (float)rms2; 
	    coef[1] = (float)rms1; 
	    coef[2] = (float)errn; 
	    if (doPitch)
	        coef[3] = getpch(sigbuf);
	    else coef[3] = 0.0;
	    if (debug) fprintf(stderr,"%d\t%9.4f\t%9.4f\t%9.4f\t%9.4f\n",
			       counter, coef[0], coef[1], coef[2], coef[3]);
	    for (fp1=coef+NDATA, dfp=cc+NPOLES, n=NPOLES; n--; )
	        *fp1++ = - (float) *--dfp;        	 /* rev coefs & chng sgn */
	    if ((nb = write(ofd, (char *)coef, osiz)) != osiz)  /* wrt anal fram */
	        quit("write error");
	    for (fp1=sigbuf, fp2=sigbuf2, n=slice; n--; )  /* move slice forward */
	        *fp1++ = *fp2++;
	    if ((n = getsndin(infd, sigbuf2, (long)slice, p)) == 0)
	        break;						/* refil til EOF */
	} while (counter < analframes);				/* or nsmps done */
	printf("%d lpc frames written to %s\n", counter, outfilnam);
	close(infd);
	close(ofd);
	exit(0);
}

static void quit(msg)
 char *msg;
{
        printf("lpanal: %s\n", msg);
	die("analysis aborted");
}

static void lpdieu(msg)
 char *msg;
{
        printf("lpanal: %s\n", msg);
	usage();
	exit(0);
}

static void alpol(sig, errn, rms1, rms2, b)
     float *sig;					/* sig now float */
     double *errn, *rms1, *rms2, b[MAXPOLES];           /* b filled here */
{
	double a[MAXPOLES][MAXPOLES], v[MAXPOLES];
/*	double x[MAXWINDIN], y[MAXWINDIN];   */
	double *x, *xp;
	double sum, sumx, sumy;
	int i, j, k, limit;

	x = (double *) malloc(WINDIN * sizeof(double));  /* alloc a double array */
	for (xp=x; xp-x < WINDIN ;++xp,++sig)            /*   &  copy sigin into */
	    *xp = (double) *sig;
	for (i=0; i < NPOLES ;++i)  {
	    sum = (double) 0.0;
	    for (k=NPOLES; k < WINDIN ;++k)
		sum += x[k-(i+1)] * x[k];
	    v[i] = -sum;
	    if (i != NPOLES - 1)  {
		limit = NPOLES - (i+1);
		for (j=0; j < limit; j++)  {
		    sum += x[NPOLES-(i+1)-(j+1)] * x[NPOLES-(j+1)]
		         - x[WINDIN-(i+1)-(j+1)] * x[WINDIN-(j+1)];
		    a[(i+1)+j][j] = a[j][(i+1)+j] = sum;
		}
	    }
	}
	sum = (double) 0.0;
	for (k=NPOLES; k < WINDIN ;++k)
	    sum += pow(x[k], (double) 2.0);
	sumy = sumx = sum;
	for (j=0; j < NPOLES; j++)  {
	    sum += pow(x[NPOLES-(j+1)], (double) 2.0)
	         - pow(x[WINDIN-(j+1)], (double) 2.0);
	    a[j][j] = sum;
	}
	gauss(a, v, b);
/*	filtn(x, y, b);   */
	for (i=0; i < NPOLES ;++i)
	    sumy -= b[i]*v[i];
	*rms1 = sqrt(sumx / (WINDIN - NPOLES) );
	*rms2 = sqrt(sumy / (WINDIN - NPOLES) );
	*errn = pow(((*rms2)/(*rms1)), (double) 2.0);
	free(x);
}

static void gauss(aold, bold, b)
     double aold[MAXPOLES][MAXPOLES], *bold, *b;
{
	double amax, dum, pivot;
	double c[MAXPOLES], a[MAXPOLES][MAXPOLES];
	int i, j, k, l, istar, ii, lp;

	/* aold and bold untouched by this subroutine */
	for (i=0; i < NPOLES ;++i)  {
	    c[i] = bold[i];
	    for (j=0; j < NPOLES ;++j)
		a[i][j] = aold[i][j];
	}
	/* eliminate i-th unknown */
	for (i=0; i < NPOLES - 1 ;++i)  {        /* find largest pivot */
	    amax = 0.0;
	    for (ii=i; ii < NPOLES ;++ii)  {
		if (fabs(a[ii][i]) >= amax)  {
		    istar = ii;
		    amax = fabs(a[ii][i]);
		}
	    }
	    if (amax < 1e-20)
		die("gauss: ill-conditioned");
	    for (j=0; j < NPOLES ;++j)  {    /* switch rows */
		dum = a[istar][j];
		a[istar][j] = a[i][j];
		a[i][j] = dum;
	    }
	    dum = c[istar];
	    c[istar] = c[i];
	    c[i] = dum;
	    for (j=i+1; j < NPOLES ;++j)  {		/* pivot */
		pivot = a[j][i] / a[i][i];
		c[j] = c[j] - pivot * c[i];
		for (k=0; k < NPOLES ;++k)
		    a[j][k] = a[j][k] - pivot * a[i][k];
	    }
	}				/* return if last pivot is too small */
	if (fabs(a[NPOLES-1][NPOLES-1]) < 1e-20)
	    die("gauss: ill-conditioned");

	*(b+NPOLES-1) = c[NPOLES-1] / a[NPOLES-1][NPOLES-1];
	for (k=0; k<NPOLES-1; ++k)  {	/* back substitute */
	    l = NPOLES-1 -(k+1);
	    *(b+l) = c[l];
	    lp = l + 1;
	    for (j = lp; j<NPOLES; ++j)
		*(b+l) += -a[l][j] * *(b+j);
	    *(b+l) /= a[l][l];
	}
}

static void usage()
{
  fprintf(stderr,"USAGE:\tlpanal [flags] infilename outfilename\n");
  fprintf(stderr,"\twhere flag options are:\n");
  fprintf(stderr,"-s<srate>\tinput sample rate (defaults to header else %7.1f.)\n",
	  DFLT_SR);
  fprintf(stderr,"-c<chnlreq>\trequested channel of sound (default chan 1)\n");
  fprintf(stderr,"-b<begin>\tbegin time in seconds into soundfile (default 0.0)\n");
  fprintf(stderr,"-d<duration>\tseconds of sound to be analyzed (default: to EOF)\n");
  fprintf(stderr,"-p<npoles>\tnumber of poles for analysis (default %d)\n",
	  DEFNPOLES);
  fprintf(stderr,"-h<hopsize>\toffset between frames in samples (default %d)\n",
	  DEFSLICE);
  fprintf(stderr,"\t\t\t(framesize will be twice <hopsize>)\n");
  fprintf(stderr,"-C<string>\tcomment field of lp header (default empty)\n");
  fprintf(stderr,"-P<mincps>\tlower limit for pitch search (default %5.1f Hz)\n",
	  PITCHMIN);
  fprintf(stderr,"\t\t\t(-P0 inhibits pitch tracking)\n");
  fprintf(stderr,"-Q<maxcps>\tupper limit for pitch search (default %5.1f Hz)\n",
	  PITCHMAX);
  fprintf(stderr,"-v<verblevel>\tprinting verbosity: 0=none, 1=verbose, 2=debug.");
  fprintf(stderr," (default 0)\n");
  fprintf(stderr,"see also:  Csound Manual Appendix\n");
}

