/*
 * Linux Event Logging for the Enterprise
 * Copyright (c) International Business Machines Corp., 2001
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Please send e-mail to lkessler@users.sourceforge.net if you have
 *  questions or comments.
 *
 *  Project Website:  http://evlog.sourceforge.net/
 *
 */
 
#include <linux/module.h>
#include <linux/kernel.h> /* printk() */
#include <linux/slab.h> /* kmalloc() */
#include <linux/fs.h>     /* everything... */
#include <linux/spinlock.h>  /* spinlock_t */
#include <linux/time.h>
#include <linux/smp.h>
#include <linux/ptrace.h>
#include <linux/string.h>
#include <linux/config.h>
#include <linux/init.h>
#include <linux/nls.h>
#include <linux/sched.h>
#include <linux/ctype.h>
#include <linux/console.h>
#include <linux/smp_lock.h>
#include <linux/evl_log.h>

static char *copy_to_buf(char *bufp, char *data, size_t len, size_t *totalbytes);
static char *pack_string(char *d, char *s, size_t *reclen, char *bf_start,
				  char *bf_end, int * bufsz);
void evl_console_print(posix_log_facility_t facility, int event_type,
					   posix_log_severity_t severity, int format, uint rec_len, 
					   const char * databuf);
void _evl_console_print(const char * s);
static char *formatbytes(const char *dp, const char *dend, char *bp, size_t size);
static posix_log_facility_t get_linux_fac_code_by_name(const char *name);
static const char * get_linux_fac_name_by_code(posix_log_facility_t code);
static int gen_canonical_facility_name(const unsigned char *fac_name,
			unsigned char *canonical);
static uint32_t crc32(const unsigned char *data, int len);
static uint32_t crc32simple(const unsigned char *data, int len);

extern int evl_kwrite_buf(posix_log_facility_t, int, posix_log_severity_t,
						  int, unsigned char *, uint, uint, va_list);
extern unsigned char * cbufwrite(char *bufp, char *s, int len);
extern unsigned char * cbufwrap(char *bufp, int len, int dowraphdr);

extern int console_loglevel;
extern struct semaphore console_sem;
extern struct timezone sys_tz;

static unsigned char evl_pdata[POSIX_LOG_ENTRY_MAXLEN + 1]; /* Used only for posix_log_printf call */

spinlock_t pxlog_printf_lck = SPIN_LOCK_UNLOCKED;

typedef enum base_type {
		TY_NONE,        /* absence of value */
		TY_CHAR,
		TY_UCHAR,
		TY_SHORT,
		TY_USHORT,
		TY_INT,
		TY_UINT,
		TY_LONG,
		TY_ULONG,
		TY_LONGLONG,
		TY_ULONGLONG,
		TY_STRING,
		TY_ADDRESS
} base_type_t;

static struct type_info {
		size_t  ti_size;
		char    *ti_name;
} type_info[] = {
		{0,                     "none"},
		{sizeof(char),          "char"},
		{sizeof(unsigned char), "uchar"},
		{sizeof(short),         "short"},
		{sizeof(unsigned short),"ushort"},
		{sizeof(int),           "int"},
		{sizeof(unsigned int),  "uint"},
		{sizeof(long),          "long"},
		{sizeof(unsigned long), "ulong"},
		{sizeof(long long),     "longlong"},
		{sizeof(unsigned long long),    "ulonglong"},
		{0,                     "string"},
		{sizeof(void*),			"address"},
		{0,                     NULL}
};

struct att_type_info {
		base_type_t     at_type;    /* TY_INT for "int" or "int[]" or "2*int" */
		int             at_nelements;   /* 5 for "5*int */
		int             at_array;       /* 1 (true) for "int[]" */
};


/*
 *    These functions are used for logging events with log_format of
 *    POSIX_LOG_BINARY.  See event logging specification at 
 *    http://evlog.sourceforge.net/linuxEvlog.html 
 *    for details.
 */
int evl_writek(posix_log_facility_t facility, int event_type,
			   posix_log_severity_t severity, unsigned int flags, ...)
{
	va_list args;
	int ret = 0;

	va_start(args, flags);
	ret = evl_vwritek(facility, event_type, severity, flags, args);
	va_end(args);

	return ret;
}

int evl_vwritek(posix_log_facility_t facility, int event_type,
				posix_log_severity_t severity, unsigned int flags, va_list args)
{
	return evl_kwrite_buf(facility, event_type, severity, POSIX_LOG_BINARY,
						  (char *)NULL, 0, flags, args);
}

/*
 *    These functions are used for logging events with log_format of
 *	  POSIX_LOG_STRING.  See event logging specification at	
 *    http://evlog.sourceforge.net/linuxEvlog.html 
 *    for details.
 */
int posix_log_printf(posix_log_facility_t facility, int event_type,
	posix_log_severity_t severity, unsigned int flags, const char *fmt, ...)
{
	int ret = 0;
	va_list args;

	if (fmt == (char *)NULL) {
			
		ret = evl_kwrite_buf(facility, event_type, severity, POSIX_LOG_NODATA,
				(char *)NULL, 0, flags, (void *)NULL);
		if (severity < console_loglevel && console_drivers) {
			evl_console_print(facility, event_type, severity, POSIX_LOG_NODATA, 0, NULL);
		}
		return ret;
	}
	
	va_start(args, fmt);
	ret = posix_log_vprintf(facility, event_type, severity, flags, fmt, args);	
	va_end(args);
	return ret;
}

int posix_log_vprintf(posix_log_facility_t facility, int event_type,
	posix_log_severity_t severity, unsigned int flags, const char *fmt, va_list args)
{
	uint recsize = 0;
	int ret = 0;
	long iflags;

	if (fmt == (char *)NULL || args == NULL) {
			
		ret = evl_kwrite_buf(facility, event_type, severity, POSIX_LOG_NODATA,
				(char *)NULL, 0, flags, (void *)NULL);
		if (severity < console_loglevel && console_drivers) {
			evl_console_print(facility, event_type, severity, POSIX_LOG_NODATA, 0, NULL);
		}
		return ret;
	}
	
	spin_lock_irqsave(&pxlog_printf_lck, iflags);
	vsnprintf(evl_pdata, POSIX_LOG_ENTRY_MAXLEN + 1, fmt, args);	
	
	recsize = strlen(evl_pdata) + 1;
	
	ret = evl_kwrite_buf(facility, event_type, severity, POSIX_LOG_STRING,
				evl_pdata, recsize, flags, (void *)NULL);
	if (severity < console_loglevel && console_drivers) {
		evl_console_print(facility, event_type, severity, POSIX_LOG_STRING,
					  recsize, evl_pdata);
	}
	spin_unlock_irqrestore(&pxlog_printf_lck, iflags);
	return ret;
}
/*
 *	This is the standard POSIX function for writing events to the event log,
 *	See event logging specification at:
 *	http://evlog.sourceforge.net/linuxEvlog.html
 */
int posix_log_write(posix_log_facility_t facility, int event_type,
		posix_log_severity_t severity, const void *buf,
		size_t recsize, int format, unsigned int flags)
{
	int ret = 0;

	if ((buf == (void *)NULL) && (recsize > 0)) {
		return -EINVAL;
	}
	if (recsize == 0 && format != POSIX_LOG_NODATA) {
		return -EINVAL;
	}
	if (format == POSIX_LOG_STRING) {
                if (strlen((const char*)buf) != recsize-1) {
                        return -EBADMSG;
                }
        }


	
	ret = evl_kwrite_buf(facility, event_type, severity, format,
				(char *)buf, recsize, flags, (void *)NULL);
	if (severity < console_loglevel && console_drivers) {
		evl_console_print(facility, event_type, severity, format, recsize, buf);
	}

	return ret;
}

/*
 * Set *fcode to the canonical facility code that corresponds to the facility
 * name fname.  Return 0 on success.
 *
 * If fname is the upper or lower case name of a standard Linux facility,
 * return the corresponding facility code.
 *
 * Otherwise, compute the facility code as the CRC of the name.  If the
 * resulting code is the same as a reserved code (either a Linux code or
 * EVL_INVALID_FACILITY), we return -EEXIST.
 */
int
evl_gen_facility_code(const char *fname, posix_log_facility_t *fcode)
{
	posix_log_facility_t fac_num;
	size_t name_len;
	char canonical[POSIX_LOG_MEMSTR_MAXLEN];

	if (!fname || !fcode) {
		return -EINVAL;
	}

	name_len = strlen(fname);
	if (name_len == 0 || name_len >= POSIX_LOG_MEMSTR_MAXLEN) {
		return -EINVAL;
	}

	(void) gen_canonical_facility_name(fname, canonical);

	fac_num = get_linux_fac_code_by_name(canonical);
	if (fac_num != EVL_INVALID_FACILITY) {
		/* fname names a standard Linux facility. */
		*fcode = fac_num;
		return 0;
	}

	fac_num = crc32(canonical, strlen(canonical));
	if (fac_num == EVL_INVALID_FACILITY) {
		/* CRC happens to match EVL_INVALID_FACILITY */
		return -EEXIST;
	}
	if (get_linux_fac_name_by_code(fac_num) != NULL) {
		/* CRC happens to match a standard Linux facility code. */
		return -EEXIST;
	}
	*fcode = fac_num;
	return 0;
}

/*
 * Compute the appropriate facility code for the facility whose name is fname.
 * To ensure that this facility is registered in the facility registry, log
 * a special event with facility = LOG_LOGMGMT and event type =
 * EVLOG_REGISTER_FAC.  evlogd will intercept this event record and, if
 * the facility is not already registered, register it.
 *
 * If the facility code we compute for fname collides with a reserved code
 * (and fname is not the corresponding reserved name), we return -EEXIST
 * and set *fcode to EVL_INVALID_FACILITY.  But we still log the event.
 */
int
evl_register_facility(const char *fname, posix_log_facility_t *fcode)
{
	int status;
	int ret;
	struct evl_facreg_rq rq;
	posix_log_severity_t sev;

	/* evlogd will set this value eventually. */
	rq.fr_registry_fac_code = EVL_INVALID_FACILITY;
	
	status = evl_gen_facility_code(fname, fcode);
	if (status == -EINVAL) {
		return -EINVAL;
	} else if (status == 0) {
		rq.fr_kernel_fac_code = *fcode;
		rq.fr_rq_status = frst_kernel_ok;
		sev = LOG_INFO;
		ret = 0;
	} else {
		/* *fcode must conflict with a reserved facility code. */
		*fcode = EVL_INVALID_FACILITY;
		rq.fr_kernel_fac_code = EVL_INVALID_FACILITY;
		rq.fr_rq_status = frst_kernel_failed;
		sev = LOG_ERR;
		ret = -EEXIST;
	}

	/* Log the event no matter whether evl_gen_facility_code() succeeded. */
	status = evl_writek(LOG_LOGMGMT, EVLOG_REGISTER_FAC, sev, 0,
		"char[]", sizeof(rq), &rq,
		"string", fname,
		"endofdata");
	if (status != 0) {
		return status;
	}
	return ret;
}

/*
 * FUNCTION		: mk_rec_header 
 * ARGS			: rec_hdr - the record header stucture
 *			: facility - 
 *			: event_type -
 *			: severity -
 *			: recsize
 *			: flags -
 *			: format - indicates string or binary format or no data
 * RETURN	: void
 */
void mk_rec_header(struct posix_log_entry *rec_hdr, 
		   posix_log_facility_t   facility,
		   int 			  event_type, 
		   posix_log_severity_t   severity,
		   size_t		  recsize, 
		   uint 		  flags, 
		   int 		  format)
{

	rec_hdr->log_size	    	=  recsize;
	rec_hdr->log_format		=  format;
	rec_hdr->log_event_type 	=  event_type;
	rec_hdr->log_facility   	=  facility;
	rec_hdr->log_severity   	=  severity;
	rec_hdr->log_uid		=  current->uid;
	rec_hdr->log_gid		=  current->gid;
	rec_hdr->log_pid		=  current->pid;
	rec_hdr->log_pgrp		=  current->pgrp;
	rec_hdr->log_flags		=  flags;
	rec_hdr->log_thread		=  0;
	rec_hdr->log_processor		=  smp_processor_id();

	if (CURRENT_TIME == 0) {
		rec_hdr->log_flags |= EVL_INITIAL_BOOT_EVENT;
	} else {
#if defined(__i386__)
		if (sys_tz.tz_minuteswest == 0) { 
			/* localtime */
			rec_hdr->log_flags |= EVL_KERNTIME_LOCAL;
		} 
#endif
	}
	rec_hdr->log_time.tv_sec  = CURRENT_TIME;
}

#if 0
/* We do not support wide-character in the kernel */
/* Return length of Wide-character string S.  */
size_t
evl_wcslen (s)
	 wchar_t *s;
{
  size_t len = 0;

  while (s[len] != 0x00)
	{
	  if (s[++len] == 0x00)
		return len;
	  ++len;
	}

  return len;
}
#endif

static base_type_t
get_type_by_name(const char *name)
{
		base_type_t i;
		for (i = TY_NONE+1; type_info[i].ti_name; i++) {
				if (!strcmp(name, type_info[i].ti_name)) {
						return i;
				}
		}
		return TY_NONE;
}

/*
 * att_type should be a type spec such as "int", "int[]", or "5*int".  Parse it
 * and fill in *ti accordingly.  Returns 0 on success, -1 on failure.
 */
static int
parse_att_type(const char *att_type, struct att_type_info *ti)
{
		const char *s, *left_bracket;
		const char *type_name;
#define MAX_TYPE_NAME_LEN 20
		char name_buf[MAX_TYPE_NAME_LEN+1];

		if (isdigit(att_type[0])) {
				/* "5*int" */
				ti->at_nelements = (int) simple_strtoul(att_type, (char**) &s, 10);
				if (*s != '*') {
						return -1;
				}
				type_name = s+1;
				ti->at_array = 0;
		} else if ((left_bracket = strchr(att_type, '[')) != NULL) {
				/* int[] */
				size_t name_len;
				ti->at_array = 1;
				ti->at_nelements = 0;
				if (0 != strcmp(left_bracket, "[]")) {
						return -1;
				}
				/* Copy the name to name_buf and point type_name at it. */
				type_name = name_buf;
				name_len = left_bracket - att_type;
				if (name_len == 0 || name_len > MAX_TYPE_NAME_LEN) {
						return -1;
				}
				(void) memcpy(name_buf, att_type, name_len);
				name_buf[name_len] = '\0';
		} else {
				/* "int" */
				type_name = att_type;
				ti->at_array = 0;
				ti->at_nelements = 1;
		}
		ti->at_type = get_type_by_name(type_name);
		return (ti->at_type == TY_NONE ? -1 : 0);
}

/*
 * Pack a null-terminated string into the data buffer.  If there's room
 * for at least one byte, but not for the whole string, we copy in what
 * we can and null-terminate it.
 * param d	destination: The string will be copied to here.
 * param s	source: The string to be copied.
 * param reclen	points to the record length that is being accumulated by
 *		the caller.  It is updated to contain what the new
 *		record length would be if the buffer were sufficiently big.
 * return	the destination pointer, updated to point past the null
 *		at the end of the string we just added
 */
static char *
pack_string(char *d, char *s, size_t *reclen,
		   char *bf_start, char *bf_end, int * bufsz)
{
	size_t slen = strlen(s) + 1;
	size_t prevrlen = *reclen;
	
	d = copy_to_buf(d, s, slen, reclen);
	if (d == (char *)NULL) {
		return NULL;
	}
	if (*reclen > POSIX_LOG_ENTRY_MAXLEN) {
		/*
	 	 * if record length limit is not
	 	 * reached in the previous arg,
	 	 * copy the NULL character at
	 	 * the end the record.
	 	 */
		if (prevrlen >= POSIX_LOG_ENTRY_MAXLEN) {
			*bufsz = *reclen;
			return d;
		}
		if (d != bf_start) {
			*(d - 1) = '\0';
		} else {
			*(bf_end - 1) = '\0';
		}
	}
    return d;
}

/*
 * If the record length is not reached the MAX_REC_SIZE, determine
 * whether the circular buffer has space to copy. If so, copy it.
 */
static char *
copy_to_buf(char *dest, char *src, size_t n, size_t *reclen)
{
	size_t nb = n;
	size_t temp_rlen = *reclen + n;

	if (temp_rlen > POSIX_LOG_ENTRY_MAXLEN) {
		if (*reclen < POSIX_LOG_ENTRY_MAXLEN) {
			nb = POSIX_LOG_ENTRY_MAXLEN - *reclen;
		} else {
			*reclen = temp_rlen;
			return dest;
		}
	}
	*reclen = temp_rlen;
	if ((dest = cbufwrap(dest, nb, 0)) == (char *)NULL) {
		return NULL;
	}
	return cbufwrite(dest, src, nb);
}

/*
 * COPYARGS copies n args of type lt (little type) from the stack using
 * memcpyex.  bt (big type) is the type of the arg as it appears on the stack.
 */
#define COPYARGS(lt,bt) \
{ \
	while(n-- > 0) { \
		lt v=(lt)va_arg(args,bt); \
		d = copy_to_buf(d, (char *)&v, sizeof(lt), &reclen); \
		if (d == (char *)NULL) { \
			return NULL; \
		} \
	} \
}

/*
 * Packing attributes data onto a circular buffer if the space is available.
 *
 * ARGUMENTS:   args - the variable arguments list.
 *		bf_start - Starting pointer to the circular buffer
 *		bf_end   - End pointer to the circular buffer
 *              databuf - the event record buffer
 *              bufsz - Returns the size of the buffer copied.
 *		error - Return the errno if it fails.
 * return       pointer to the data buffer to write the next record.
 *              NULL for fail.
 */
char *
copy_attr_data(va_list args, char *bf_start, char *bf_end, 
		char *databuf, int *bufsz, int *error)
{
		char *att_type;
#if 0
		size_t wcharsz = sizeof(wchar_t);
#endif
		char *d = databuf;                      /* next data goes here */
		size_t reclen = 0;
	
		while ((att_type = va_arg(args, char*)) &&
			(0 != strcmp(att_type, "endofdata"))) {
			struct att_type_info ti;
			if (parse_att_type(att_type, &ti) == -1) {
				*error = -EINVAL;
					return NULL;
			}
			if (ti.at_array) {
				char *array, *a;
				size_t size = type_info[ti.at_type].ti_size;
				int n;

				/* Next arg is the array size. */
				n = va_arg(args, int);
				/* Next arg is the array address. */
				array = (char*) va_arg(args, void*);
				
				switch (ti.at_type) {
					case TY_STRING:
			    	{
						/* array points to an array of char* */
						char **sarray = (char**)array;
						int i;
						for (i = 0; i < n; i++) {
							d = pack_string(d, sarray[i], &reclen, bf_start, bf_end, bufsz);
						}
						break;
			    	}
					default:
                       	for (a = array; n > 0; a += size, n--) {
                           	d = copy_to_buf(d, a, size, &reclen);
							if (d == (char *)NULL) {
								return NULL;
							}
                       	}
                       	break;
				}
			} else {
				/*
				 * One or more args of the same type.
				 */
				int n = ti.at_nelements;
                switch (ti.at_type) {
					case TY_CHAR:
					case TY_UCHAR:
					#if defined(CONFIG_ARCH_S390X)   			/* 64bits s390x */
						COPYARGS(char, long)
					#else
						COPYARGS(char, int)
					#endif
						break;
					case TY_SHORT:
					case TY_USHORT:
					#if defined(CONFIG_ARCH_S390X)
						COPYARGS(short, long)
					#else
						COPYARGS(short, int)
					#endif
						break;
					case TY_INT:
					case TY_UINT:
					#if defined(CONFIG_ARCH_S390X)
						COPYARGS(int, long)
					#else
						COPYARGS(int, int)
					#endif
						break;
					case TY_LONG:
					case TY_ULONG:
						COPYARGS(long, long)
						break;
					case TY_LONGLONG:
					case TY_ULONGLONG:
						COPYARGS(long long, long long)

						break;
					case TY_ADDRESS:
						COPYARGS(void*, void*)
						break;
					case TY_STRING:
					{
						char *s;
						while (n-- > 0) {
							s = (char *) va_arg(args, char*);
							d = pack_string(d, s, &reclen, bf_start, bf_end, bufsz);
						}
					break;
					}
#if 0
					case TY_WCHAR:
						COPYARGS(wchar_t, wchar_t)
						break;
					case TY_WSTRING:
					{
						size_t slen;
						wchar_t *s;
						while (n-- > 0) {
							s = (wchar_t *) va_arg(args, wchar_t*);
							slen = wcslen(s) * wcharsz;
							d = copy_to_buf(d, s, slen, &reclen);
							if (d == (char *)NULL) {
								return NULL;
							}
						}
					} /* end of TY_WSTRING case */
#endif
			default:
				break;
			} /* end of switch */
		} /* not array */
	} /* next att_type */
		
	*bufsz = reclen;
	
	return d;
}

/*
 * FUNCTION	: printk_evlog_write - writes printk messages as events to event 
 *            buffer.
 * ARGS		: buf - pointer to printk message 
 *
 * RETURN	: return 0 for success, otherwise return a negative
 *			  error code.
 */
static char prtk_msgbuf[POSIX_LOG_ENTRY_MAXLEN + 1];
static int msgbuf_idx = 0;
int
printk_evlog_write(char *buf)
{
	char *recbuf= (char*)buf;
	char *p; 
	uint recsize = 0;
	static int sev = -1;
	int ret = 0;
	int format;
	int event_type;
	
	if (buf == (char *)NULL) {
		return -EINVAL;
	}
	
	/*
	 * If the message extends over multiple calls to printk, we consult
	 * only the beginning of the message to determine the severity.
	 *
	 * TODO: Decide what to do with calls such as
	 *	printk(KERN_EMERG "System on fire!\n"
	 *		KERN_WARN "Wear oven mitts!\n");
	 * Vanilla printk logs two lines:
	 *	<0>System on fire!\n
	 *	<4>Wear oven mitts!\n
	 * We currently log one message, with a severity of LOG_EMERG:
	 *	System on fire!\n<4>Wear oven mitts!\n
	 */
	if (sev == -1) {
		sev = LOG_NOTICE;
		if (strlen(buf) > 2 &&
		  buf[0] == '<' && buf[1] >= '0' && buf[1] <= '7' && buf[2] == '>') {
			sev = buf[1] - '0';
			recbuf += 3;
		}
	} 
	
	for (p = recbuf; *p; p++) {
		prtk_msgbuf[msgbuf_idx]= *p;
		msgbuf_idx++;
		if (msgbuf_idx >= POSIX_LOG_ENTRY_MAXLEN-2) {	
			 /* Message apparently bigger than the buffer */
			prtk_msgbuf[msgbuf_idx++] = '\n';	
			break;
		}	
	}

	if (msgbuf_idx == 0 || prtk_msgbuf[msgbuf_idx-1] != '\n') {
		/* Message not completed yet - return for more */
		return ret;
	}

	/*
	 * Message completed.  Change the terminating newline to a null.
	 * We remove the terminating newline to increase flexibility when
	 * formatting the record for viewing.
	 */
	prtk_msgbuf[msgbuf_idx-1] = '\0';
	recsize = msgbuf_idx;
	format = POSIX_LOG_STRING;
	event_type = EVL_PRINTK_MESSAGE;

	ret = evl_kwrite_buf(LOG_KERN, event_type, sev, format, prtk_msgbuf,
		recsize, EVL_PRINTK, (void *)NULL);

	msgbuf_idx = 0;
	sev = -1;
	return ret;
}

/*
 * print evlog message to the console
 */

void evl_console_print(posix_log_facility_t facility, int event_type,
					   posix_log_severity_t severity, int format, uint rec_len, 
					   const char * databuf)
{
	char s[256 + 41];		/* print buffer - only print first 256 chars  */
	char b[36+1];			/* (16 bytes x 2) + 4 spaces + null */ 	
	
	if (format == POSIX_LOG_STRING) {
		if (rec_len >= 256) {
			int h = 0;
			snprintf(s, sizeof(s) - 2, "F%04x:T%04x:S: %s", facility, event_type, databuf);
			strcat(s, "\n");
		} else {
			snprintf(s, sizeof(s), "F%04x:T%04x:S: %s\n", facility, 
					event_type, databuf);
		}
	} else if (format == POSIX_LOG_BINARY) {
		char *tmp = b;
		if (rec_len >= 16 ) {
			formatbytes(databuf, databuf+16, tmp, sizeof(b));
			snprintf(s, sizeof(s), "F%04x:T%04x:B: <%s...> %d more bytes\n", facility, 
				event_type, b, rec_len - 16);
		} else {
			formatbytes(databuf, databuf+rec_len, tmp, sizeof(b));
			snprintf(s, sizeof(s), "F%04x:T%04x:B: <%s>\n", facility, 
				event_type, b);
		}			
	} else {
		snprintf(s, sizeof(s), "F%04x:T%04x:N: <no data>\n", facility, 
				event_type);
	}
	_evl_console_print(s);
	
}

void _evl_console_print(const char * s) 
{
	struct console *con;

	down(&console_sem);
	for (con = console_drivers; con; con = con->next) {
		if ((con->flags & CON_ENABLED) && con->write)
			con->write(con, s, strlen(s));
	}
	up(&console_sem);	
}
	


static char *hexDigits = "0123456789ABCDEF";

static char *formatbytes(const char *dp, const char *dend, char *bp, size_t size)
{
	char *oldbp = bp, *bpend = bp + size - 1;
	int i;
	int nbytes = dend - dp;
	int n = 0;
	for (i = 1; i <= nbytes; i++, dp++, bp += 2) {
		if (dp <= dend && bp + 2 < bpend) {
			bp[0] = hexDigits[(*dp >> 4) & 0xF];
			bp[1] = hexDigits[*dp & 0xF];
			n++;
			if (n >= 4 && bp + 1 < bpend) {
				bp[2] = ' ';
				bp++;
				n = 0;
			} 	
		}
	}
	*bp='\0';
	return oldbp;
}

/*** Facility registration functions start here. ***/

struct nv_pair {
	int	nv_value;
	char	*nv_name;
};

/*
 * This should look a lot like the DEFAULT version of the facility registry,
 * /var/evlog/facility_registry.  DO NOT UPDATE THIS TABLE when you add a
 * new facility.
 */
static struct nv_pair linux_facilities[] = {
	{ 0, "kern" },
	{ 8, "user" },
	{ 16, "mail" },
	{ 24, "daemon" },
	{ 32, "auth" },
	{ 40, "syslog" },
	{ 48, "lpr" },
	{ 56, "news" },
	{ 64, "uucp" },
	{ 72, "cron" },
	{ 80, "authpriv" },
	{ 88, "ftp" },
	{ 96, "logmgmt" },
	{ 128, "local0" },
	{ 136, "local1" },
	{ 144, "local2" },
	{ 152, "local3" },
	{ 160, "local4" },
	{ 168, "local5" },
	{ 176, "local6" },
	{ 184, "local7" },
	{ -1, NULL }
};

static posix_log_facility_t
get_linux_fac_code_by_name(const char *name)
{
	struct nv_pair *nv;
	for (nv = linux_facilities; nv->nv_name; nv++) {
		if (!strcmp(name, nv->nv_name)) {
			return (posix_log_facility_t) nv->nv_value;
		}
	}
	return EVL_INVALID_FACILITY;
}

static const char *
get_linux_fac_name_by_code(posix_log_facility_t code)
{
	struct nv_pair *nv;
	for (nv = linux_facilities; nv->nv_name; nv++) {
		if (code == (posix_log_facility_t) (nv->nv_value)) {
			return nv->nv_name;
		}
	}
	return NULL;
}

/*
 * Copy fac_name to canonical, converting characters as necessary so that
 * canonical is the canonical version of facility name fac_name.  Returns
 * -EINVAL if fac_name is null or empty, or if canonical is null; 0 otherwise.
 *
 * Here are the rules for forming a canonical name:
 * 1. The following bytes are passed through unchanged: ASCII digits,
 * ASCII lowercase letters, period, underscore, and any bytes outside
 * the ASCII code set.
 * 2. Uppercase ASCII letters are converted to lowercase.
 * 3. A space character is converted to an underscore.
 * 4. Any other ASCII character is converted to a period.
 */
static int
gen_canonical_facility_name(const unsigned char *fac_name,
	unsigned char *canonical)
{
	const unsigned char *f;
	unsigned char *c;

	if (!fac_name || !canonical || fac_name[0] == '\0') {
		return -EINVAL;
	}

	for (f=fac_name, c=canonical; *f; f++, c++) {
		unsigned int uf = *f;
		if ('A' <= uf && uf <= 'Z') {
			*c = uf | 0x20;		/* ASCII toupper(uf) */
		} else if (uf > 0x7f
		    || ('a' <= uf && uf <= 'z')
		    || ('0' <= uf && uf <= '9')
		    || uf == '.'
		    || uf == '_') {
			*c = uf;
		} else if (uf == ' ') {
			*c = '_';
		} else {
			*c = '.';
		}
	}
	*c = '\0';

	/* "." and ".." are reserved directory names, so we convert them. */
	if (!strcmp(canonical, ".") || !strcmp(canonical, "..")) {
		canonical[0] = '_';
	}
	return 0;
}

/*
 * This is essentially CRC algorithm #3 from
 * http://www.cl.cam.ac.uk/Research/SRG/bluebook/21/crc/node6.html.
 * Some other CRC algorithms hard-wire crctab -- i.e.,
 *	uint32_t crctab[256] = { 256 strange numbers here };
 * We stick with the original code to make it at least a little easier to
 * understand what's going on.  Since crc32_init() typically gets called
 * very soon after boot, that's the only time the spinlock comes into play.
 * Even then, the spinlock may be unnecessary if you can guarantee that
 *	crctab[i] = crc
 * is an atomic operation.
 */
#define QUOTIENT 0x04c11db7
uint32_t crctab[256];
static spinlock_t crctab_lock = SPIN_LOCK_UNLOCKED;
static int crctab_initialized = 0;

static void
crc32_init(void)
{
	int i, j;
	uint32_t crc;
	unsigned long flags;

	spin_lock_irqsave(&crctab_lock, flags);
	if (crctab_initialized) {
		/* Somebody else beat us to it. */
		spin_unlock_irqrestore(&crctab_lock, flags);
		return;
	}

	for (i = 0; i < 256; i++) {
		crc = i << 24;
		for (j = 0; j < 8; j++) {
			if (crc & 0x80000000) {
				crc = (crc << 1) ^ QUOTIENT;
			} else {
				crc = crc << 1;
			}
		}
		crctab[i] = crc;
	}

	crctab_initialized = 1;
	spin_unlock_irqrestore(&crctab_lock, flags);
}

static uint32_t
crc32(const unsigned char *data, int len)
{
	uint32_t result;
	int i;

	if (len < 4) {
		return crc32simple(data, len);
	}

	if (!crctab_initialized) {
		crc32_init();
	}

	result = *data++ << 24;
	result |= *data++ << 16;
	result |= *data++ << 8;
	result |= *data++;
	result = ~ result;

	for (i=4; i<len; i++) {
		result = (result << 8 | *data++) ^ crctab[result >> 24];
	}

	for (i=0; i<4; i++) {
		result = (result << 8) ^ crctab[result >> 24];
	}

	return ~result;
}

/*
 * This is the code for CRC algorithm #1 from
 * http://www.cl.cam.ac.uk/Research/SRG/bluebook/21/crc/node6.html.
 * It is not as efficient as algorithm #3, but it can handle data lengths < 4.
 */
static uint32_t
crc32simple(const unsigned char *data, int len)
{
	uint32_t result;
	int i,j;
	unsigned char octet;
    
	result = -1;
    
	for (i=0; i<len; i++) {
		octet = *(data++);
		for (j=0; j<8; j++) {
			if ((octet >> 7) ^ (result >> 31)) {
				result = (result << 1) ^ QUOTIENT;
			} else {
				result = (result << 1);
			}
			octet <<= 1;
		}
	}
 
	return ~result;	/* The complement of the remainder */
}

EXPORT_SYMBOL(evl_writek);
EXPORT_SYMBOL(evl_vwritek);
EXPORT_SYMBOL(posix_log_write);
EXPORT_SYMBOL(posix_log_printf);
EXPORT_SYMBOL(posix_log_vprintf);
EXPORT_SYMBOL(evl_gen_facility_code);
EXPORT_SYMBOL(evl_register_facility);
