/* 
 * grsecurity/gracl.c
 * Copyright Brad Spengler 2001, 2002, 2003
 *
 */

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/tty.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/types.h>
#include <linux/capability.h>
#include <linux/sysctl.h>
#include <linux/gracl.h>
#include <linux/gralloc.h>
#include <linux/grsecurity.h>
#include <linux/grinternal.h>

#include <asm/uaccess.h>
#include <asm/errno.h>
#include <asm/mman.h>

static struct acl_role_db acl_role_set;
static struct acl_role_label *role_list_head;
static struct name_db name_set;
static struct name_db inodev_set;

/* for keeping track of userspace pointers used for subjects, so we
   can share references in the kernel as well
*/

static struct dentry *real_root;
static struct vfsmount *real_root_mnt;

static struct acl_subj_map_db subj_map_set;


static struct acl_role_label *default_role;

static u16 acl_sp_role_value;

static DECLARE_MUTEX(gr_dev_sem);
rwlock_t gr_inode_lock = RW_LOCK_UNLOCKED;

extern char *gr_shared_page[4][NR_CPUS];
struct gr_arg *gr_usermode;

static unsigned long gr_status = GR_STATUS_INIT;

extern int chkpw(struct gr_arg *entry, unsigned char *salt, unsigned char *sum);
extern void gr_clear_learn_entries(void);

#ifdef CONFIG_GRKERNSEC_RESLOG
extern void gr_log_resource(const struct task_struct *task,
			    const int res, const unsigned long wanted, const int gt);
#endif

unsigned char *gr_system_salt;
unsigned char *gr_system_sum;

static struct sprole_pw **acl_special_roles = NULL;
static __u16 num_sprole_pws = 0;

static struct acl_role_label *kernel_role = NULL;

/* The following are used to keep a place held in the hash table when we move
   entries around.  They can be replaced during insert. */

static struct acl_subject_label *deleted_subject;
static struct acl_object_label *deleted_object;
static struct name_entry *deleted_inodev;

/* for keeping track of the last and final allocated subjects, since
   nested subject parsing is tricky
*/
static struct acl_subject_label *s_last = NULL;
static struct acl_subject_label *s_final = NULL;

static unsigned int gr_auth_attempts = 0;
static unsigned long gr_auth_expires = 0UL;

extern int gr_init_uidset(void);
extern void gr_free_uidset(void);
extern void gr_remove_uid(uid_t uid);
extern int gr_find_uid(uid_t uid);

__inline__ int
gr_acl_is_enabled(void)
{
	return (gr_status & GR_READY);
}

char gr_roletype_to_char(void)
{
	switch (current->role->roletype &
		(GR_ROLE_DEFAULT | GR_ROLE_USER | GR_ROLE_GROUP |
		 GR_ROLE_SPECIAL)) {
	case GR_ROLE_DEFAULT:
		return 'D';
	case GR_ROLE_USER:
		return 'U';
	case GR_ROLE_GROUP:
		return 'G';
	case GR_ROLE_SPECIAL:
		return 'S';
	}

	return 'X';
}

__inline__ int
gr_acl_tpe_check(void)
{
	if (unlikely(!(gr_status & GR_READY)))
		return 0;
	if (current->role->roletype & GR_ROLE_TPE)
		return 1;
	else
		return 0;
}

int
gr_handle_rawio(const struct inode *inode)
{
#ifdef CONFIG_GRKERNSEC_CHROOT_CAPS
	if (inode && S_ISBLK(inode->i_mode) &&
	    grsec_enable_chroot_caps && proc_is_chrooted(current) &&
	    !capable(CAP_SYS_RAWIO))
		return 1;
#endif
	return 0;
}


static __inline__ int
gr_streq(const char *a, const char *b, const __u16 lena, const __u16 lenb)
{
	int i;
	unsigned long *l1;
	unsigned long *l2;
	unsigned char *c1;
	unsigned char *c2;
	int num_longs;

	if (likely(lena != lenb))
		return 0;

	l1 = (unsigned long *)a;
	l2 = (unsigned long *)b;

	num_longs = lena / sizeof(unsigned long);

	for (i = num_longs; i--; l1++, l2++) {
		if (unlikely(*l1 != *l2))
			return 0;
	}

	c1 = (unsigned char *) l1;
	c2 = (unsigned char *) l2;

	i = lena - (num_longs * sizeof(unsigned long));	

	for (; i--; c1++, c2++) {
		if (unlikely(*c1 != *c2))
			return 0;
	}

	return 1;
}
		
static char *
d_real_path(const struct dentry *dentry, const struct vfsmount *vfsmnt,
	    char *buf, int buflen)
{
	char *res;
	struct dentry *root;
	struct vfsmount *rootmnt;

	/* we can't use real_root, real_root_mnt, because they belong only to the RBAC system */
	read_lock(&child_reaper->fs->lock);
	root = dget(child_reaper->fs->root);
	rootmnt = mntget(child_reaper->fs->rootmnt);
	read_unlock(&child_reaper->fs->lock);

	spin_lock(&dcache_lock);
	res = __d_path((struct dentry *)dentry, (struct vfsmount *)vfsmnt, root, rootmnt, buf, buflen);
	spin_unlock(&dcache_lock);
	if (unlikely(IS_ERR(res)))
		res = strcpy(buf, "<path too long>");
	dput(root);
	mntput(rootmnt);
	return res;
}

static __inline__ char *
__d_real_path(const struct dentry *dentry, const struct vfsmount *vfsmnt,
	    char *buf, int buflen)
{
	char *res;
	struct dentry *root;
	struct vfsmount *rootmnt;

	/* we can't use real_root, real_root_mnt, because they belong only to the RBAC system */
	read_lock(&child_reaper->fs->lock);
	root = dget(child_reaper->fs->root);
	rootmnt = mntget(child_reaper->fs->rootmnt);
	read_unlock(&child_reaper->fs->lock);

	res = __d_path((struct dentry *)dentry, (struct vfsmount *)vfsmnt, root, rootmnt, buf, buflen);
	if (unlikely(IS_ERR(res)))
		res = strcpy(buf, "<path too long>");
	dput(root);
	mntput(rootmnt);
	return res;
}

char *
gr_to_filename_nolock(const struct dentry *dentry, const struct vfsmount *mnt)
{
	return __d_real_path(dentry, mnt, gr_shared_page[0][smp_processor_id()],
			   PAGE_SIZE);
}

char *
gr_to_filename(const struct dentry *dentry, const struct vfsmount *mnt)
{
	return d_real_path(dentry, mnt, gr_shared_page[0][smp_processor_id()],
			   PAGE_SIZE);
}

char *
gr_to_filename1(const struct dentry *dentry, const struct vfsmount *mnt)
{
	return d_real_path(dentry, mnt, gr_shared_page[1][smp_processor_id()],
			   PAGE_SIZE);
}

char *
gr_to_filename2(const struct dentry *dentry, const struct vfsmount *mnt)
{
	return d_real_path(dentry, mnt, gr_shared_page[2][smp_processor_id()],
			   PAGE_SIZE);
}

char *
gr_to_filename3(const struct dentry *dentry, const struct vfsmount *mnt)
{
	return d_real_path(dentry, mnt, gr_shared_page[3][smp_processor_id()],
			   PAGE_SIZE);
}

__inline__ __u32
to_gr_audit(const __u32 reqmode)
{
	__u32 retmode = 0;

	retmode |= (reqmode & GR_READ) ? GR_AUDIT_READ : 0;
	retmode |= (reqmode & GR_WRITE) ? GR_AUDIT_WRITE | GR_AUDIT_APPEND : 0;
	retmode |= (reqmode & GR_APPEND) ? GR_AUDIT_APPEND : 0;
	retmode |= (reqmode & GR_EXEC) ? GR_AUDIT_EXEC : 0;
	retmode |= (reqmode & GR_INHERIT) ? GR_AUDIT_INHERIT : 0;
	retmode |= (reqmode & GR_FIND) ? GR_AUDIT_FIND : 0;
	retmode |= (reqmode & GR_SETID) ? GR_AUDIT_SETID : 0;
	retmode |= (reqmode & GR_CREATE) ? GR_AUDIT_CREATE : 0;
	retmode |= (reqmode & GR_DELETE) ? GR_AUDIT_DELETE : 0;

	return retmode;
}

__inline__ struct acl_subject_label *
lookup_subject_map(const struct acl_subject_label *userp)
{
	unsigned long index = shash(userp, subj_map_set.s_size);
	struct subject_map *match;
	__u8 i = 0;

	match = subj_map_set.s_hash[index];

	while (match && match->user != userp) {
		index = (index + (1 << i)) % subj_map_set.s_size;
		match = subj_map_set.s_hash[index];
		i = (i + 1) % 32;
	}

	if (match)
		return match->kernel;
	else
		return NULL;
}

static void
insert_subj_map_entry(struct subject_map *subjmap)
{
	unsigned long index = shash(subjmap->user, subj_map_set.s_size);
	struct subject_map **curr;
	__u8 i = 0;

	curr = &subj_map_set.s_hash[index];

	while (*curr) {
		index = (index + (1 << i)) % subj_map_set.s_size;
		curr = &subj_map_set.s_hash[index];
		i = (i + 1) % 32;
	}

	*curr = subjmap;

	return;
}

__inline__ struct acl_role_label *
lookup_acl_role_label(const struct task_struct *task, const uid_t uid,
		      const gid_t gid)
{
	unsigned long index = rhash(uid, GR_ROLE_USER, acl_role_set.r_size);
	struct acl_role_label *match;
	struct role_allowed_ip *ipp;
	int x;
	__u8 i = 0;

	match = acl_role_set.r_hash[index];

	while (match) {
		if ((match->roletype & (GR_ROLE_DOMAIN | GR_ROLE_USER)) == (GR_ROLE_DOMAIN | GR_ROLE_USER)) {
			for (x = 0; x < match->domain_child_num; x++) {
				if (match->domain_children[x] == uid)
					goto found;
			}
		} else if (match->uidgid == uid && match->roletype & GR_ROLE_USER)
			break;
		index = (index + (1 << i)) % acl_role_set.r_size;
		match = acl_role_set.r_hash[index];
		i = (i + 1) % 32;
	}
found:
	if (match == NULL) {
	      try_group:
		index = rhash(gid, GR_ROLE_GROUP, acl_role_set.r_size);
		match = acl_role_set.r_hash[index];
		i = 0;

		while (match) {
			if ((match->roletype & (GR_ROLE_DOMAIN | GR_ROLE_GROUP)) == (GR_ROLE_DOMAIN | GR_ROLE_GROUP)) {
				for (x = 0; x < match->domain_child_num; x++) {
					if (match->domain_children[x] == gid)
						goto found2;
				}
			} else if (match->uidgid == gid && match->roletype & GR_ROLE_GROUP)
				break;
			index = (index + (1 << i)) % acl_role_set.r_size;
			match = acl_role_set.r_hash[index];
			i = (i + 1) % 32;
		}
found2:
		if (match == NULL)
			match = default_role;
		if (match->allowed_ips == NULL)
			return match;
		else {
			for (ipp = match->allowed_ips; ipp; ipp = ipp->next) {
				if (likely
				    ((ntohl(task->curr_ip) & ipp->netmask) ==
				     (ntohl(ipp->addr) & ipp->netmask)))
					return match;
			}
			match = default_role;
		}
	} else if (match->allowed_ips == NULL) {
		return match;
	} else {
		for (ipp = match->allowed_ips; ipp; ipp = ipp->next) {
			if (likely
			    ((ntohl(task->curr_ip) & ipp->netmask) ==
			     (ntohl(ipp->addr) & ipp->netmask)))
				return match;
		}
		goto try_group;
	}

	return match;
}

__inline__ struct acl_subject_label *
lookup_acl_subj_label(const ino_t ino, const kdev_t dev,
		      const struct acl_role_label *role)
{
	unsigned long subj_size = role->subj_hash_size;
	struct acl_subject_label **s_hash = role->subj_hash;
	unsigned long index = fhash(ino, dev, subj_size);
	struct acl_subject_label *match;
	__u8 i = 0;

	match = s_hash[index];

	while (match && (match->inode != ino || match->device != dev ||
	       (match->mode & GR_DELETED))) {
		index = (index + (1 << i)) % subj_size;
		match = s_hash[index];
		i = (i + 1) % 32;
	}

	if (match && (match != deleted_subject) && !(match->mode & GR_DELETED))
		return match;
	else
		return NULL;
}

static __inline__ struct acl_object_label *
lookup_acl_obj_label(const ino_t ino, const kdev_t dev,
		     const struct acl_subject_label *subj)
{
	unsigned long obj_size = subj->obj_hash_size;
	struct acl_object_label **o_hash = subj->obj_hash;
	unsigned long index = fhash(ino, dev, obj_size);
	struct acl_object_label *match;
	__u8 i = 0;

	match = o_hash[index];

	while (match && (match->inode != ino || match->device != dev ||
	       (match->mode & GR_DELETED))) {
		index = (index + (1 << i)) % obj_size;
		match = o_hash[index];
		i = (i + 1) % 32;
	}

	if (match && (match != deleted_object) && !(match->mode & GR_DELETED))
		return match;
	else
		return NULL;
}

static __inline__ struct acl_object_label *
lookup_acl_obj_label_create(const ino_t ino, const kdev_t dev,
		     const struct acl_subject_label *subj)
{
	unsigned long obj_size = subj->obj_hash_size;
	struct acl_object_label **o_hash = subj->obj_hash;
	unsigned long index = fhash(ino, dev, obj_size);
	struct acl_object_label *match;
	__u8 i = 0;

	match = o_hash[index];

	while (match && (match->inode != ino || match->device != dev ||
	       !(match->mode & GR_DELETED))) {
		index = (index + (1 << i)) % obj_size;
		match = o_hash[index];
		i = (i + 1) % 32;
	}

	if (match && (match != deleted_object) && (match->mode & GR_DELETED))
		return match;

	i = 0;
	index = fhash(ino, dev, obj_size);
	match = o_hash[index];

	while (match && (match->inode != ino || match->device != dev ||
	       (match->mode & GR_DELETED))) {
		index = (index + (1 << i)) % obj_size;
		match = o_hash[index];
		i = (i + 1) % 32;
	}

	if (match && (match != deleted_object) && !(match->mode & GR_DELETED))
		return match;
	else
		return NULL;
}

static __inline__ struct name_entry *
lookup_name_entry(const char *name)
{
	__u16 len = strlen(name);
	unsigned long index = nhash(name, len, name_set.n_size);
	struct name_entry *match;
	__u8 i = 0;

	match = name_set.n_hash[index];

	while (match && !gr_streq(match->name, name, match->len, len)) {
		index = (index + (1 << i)) % name_set.n_size;
		match = name_set.n_hash[index];
		i = (i + 1) % 32;
	}

	return match;
}

static __inline__ struct name_entry *
lookup_inodev_entry(const ino_t ino, const kdev_t dev)
{
	unsigned long index = fhash(ino, dev, inodev_set.n_size);
	struct name_entry *match;
	__u8 i = 0;

	match = inodev_set.n_hash[index];

	while (match && (match->inode != ino || match->device != dev)) {
		index = (index + (1 << i)) % inodev_set.n_size;
		match = inodev_set.n_hash[index];
		i = (i + 1) % 32;
	}

	if (match && (match != deleted_inodev))
		return match;
	else
		return NULL;
}

static void
insert_inodev_entry(struct name_entry *nentry)
{
	unsigned long index = fhash(nentry->inode, nentry->device,
				    inodev_set.n_size);
	struct name_entry **curr;
	__u8 i = 0;

	curr = &inodev_set.n_hash[index];

	while (*curr && *curr != deleted_inodev) {
		index = (index + (1 << i)) % inodev_set.n_size;
		curr = &inodev_set.n_hash[index];
		i = (i + 1) % 32;
	}

	*curr = nentry;

	return;
}

static void
__insert_acl_role_label(struct acl_role_label *role, uid_t uidgid)
{
	unsigned long index =
	    rhash(uidgid, role->roletype & (GR_ROLE_USER | GR_ROLE_GROUP), acl_role_set.r_size);
	struct acl_role_label **curr;
	__u8 i = 0;

	curr = &acl_role_set.r_hash[index];

	while (*curr) {
		index = (index + (1 << i)) % acl_role_set.r_size;
		curr = &acl_role_set.r_hash[index];
		i = (i + 1) % 32;
	}

	*curr = role;

	return;
}

static void
insert_acl_role_label(struct acl_role_label *role)
{
	int i;

	if (role->roletype & GR_ROLE_DOMAIN) {
		for (i = 0; i < role->domain_child_num; i++)
			__insert_acl_role_label(role, role->domain_children[i]);
	} else
		__insert_acl_role_label(role, role->uidgid);
}
					
static int
insert_name_entry(char *name, const ino_t inode, const kdev_t device)
{
	struct name_entry **curr;
	__u8 i = 0;
	__u16 len = strlen(name);
	unsigned long index = nhash(name, len, name_set.n_size);

	curr = &name_set.n_hash[index];

	while (*curr && !gr_streq((*curr)->name, name, (*curr)->len, len)) {
		index = (index + (1 << i)) % name_set.n_size;
		curr = &name_set.n_hash[index];
		i = (i + 1) % 32;
	}

	if (!(*curr)) {
		struct name_entry *nentry =
		    acl_alloc(sizeof (struct name_entry));
		if (!nentry)
			return 0;
		nentry->name = name;
		nentry->inode = inode;
		nentry->device = device;
		nentry->len = len;
		*curr = nentry;
		/* insert us into the table searchable by inode/dev */
		insert_inodev_entry(nentry);
	}

	return 1;
}

static void
insert_acl_obj_label(struct acl_object_label *obj,
		     struct acl_subject_label *subj)
{
	unsigned long index =
	    fhash(obj->inode, obj->device, subj->obj_hash_size);
	struct acl_object_label **curr;
	__u8 i = 0;

	curr = &subj->obj_hash[index];

	while (*curr && *curr != deleted_object) {
		index = (index + (1 << i)) % subj->obj_hash_size;
		curr = &subj->obj_hash[index];
		i = (i + 1) % 32;
	}

	*curr = obj;

	return;
}

static void
insert_acl_subj_label(struct acl_subject_label *obj,
		      struct acl_role_label *role)
{
	unsigned long subj_size = role->subj_hash_size;
	struct acl_subject_label **s_hash = role->subj_hash;
	unsigned long index = fhash(obj->inode, obj->device, subj_size);
	struct acl_subject_label **curr;
	__u8 i = 0;

	curr = &s_hash[index];

	while (*curr && *curr != deleted_subject) {
		index = (index + (1 << i)) % subj_size;
		curr = &s_hash[index];
		i = (i + 1) % 32;
	}

	*curr = obj;

	return;
}

static void **
create_table(__u32 * len)
{
	unsigned long table_sizes[] = {
		7, 13, 31, 61, 127, 251, 509, 1021, 2039, 4093, 8191, 16381,
		32749, 65521, 131071, 262139, 524287, 1048573, 2097143,
		4194301, 8388593, 16777213, 33554393, 67108859, 134217689,
		268435399, 536870909, 1073741789, 2147483647
	};
	void *newtable = NULL;
	unsigned int pwr = 0;

	while ((pwr < ((sizeof (table_sizes) / sizeof (table_sizes[0])) - 1)) &&
	       table_sizes[pwr] <= (2 * (*len)))
		pwr++;

	if (table_sizes[pwr] <= (2 * (*len)))
		return newtable;

	if ((table_sizes[pwr] * sizeof (void *)) <= PAGE_SIZE)
		newtable =
		    kmalloc(table_sizes[pwr] * sizeof (void *), GFP_KERNEL);
	else
		newtable = vmalloc(table_sizes[pwr] * sizeof (void *));

	*len = table_sizes[pwr];

	return newtable;
}

static int
init_variables(const struct gr_arg *arg)
{
	unsigned long stacksize;

	subj_map_set.s_size = arg->role_db.num_subjects;
	acl_role_set.r_size = arg->role_db.num_roles + arg->role_db.num_domain_children;
	name_set.n_size = arg->role_db.num_objects;
	inodev_set.n_size = arg->role_db.num_objects;

	if (!gr_init_uidset())
		return 1;

	/* set up the stack that holds allocation info */

	stacksize = arg->role_db.num_pointers + 5;

	if (!acl_alloc_stack_init(stacksize))
		return 1;

	/* create our empty, fake deleted acls */
	deleted_subject =
	    (struct acl_subject_label *)
	    acl_alloc(sizeof (struct acl_subject_label));
	deleted_object =
	    (struct acl_object_label *)
	    acl_alloc(sizeof (struct acl_object_label));
	deleted_inodev =
	    (struct name_entry *) acl_alloc(sizeof (struct name_entry));

	if (!deleted_subject || !deleted_object || !deleted_inodev)
		return 1;

	memset(deleted_subject, 0, sizeof (struct acl_subject_label));
	memset(deleted_object, 0, sizeof (struct acl_object_label));
	memset(deleted_inodev, 0, sizeof (struct name_entry));

	/* grab reference for the real root dentry and vfsmount */
	read_lock(&child_reaper->fs->lock);
	real_root_mnt = mntget(child_reaper->fs->rootmnt);
	real_root = dget(child_reaper->fs->root);
	read_unlock(&child_reaper->fs->lock);
	

	/* We only want 50% full tables for now */

	subj_map_set.s_hash =
	    (struct subject_map **) create_table(&subj_map_set.s_size);
	acl_role_set.r_hash =
	    (struct acl_role_label **) create_table(&acl_role_set.r_size);
	name_set.n_hash = (struct name_entry **) create_table(&name_set.n_size);
	inodev_set.n_hash =
	    (struct name_entry **) create_table(&inodev_set.n_size);

	if (!subj_map_set.s_hash || !acl_role_set.r_hash || 
	    !name_set.n_hash || !inodev_set.n_hash)
		return 1;

	memset(subj_map_set.s_hash, 0,
	       sizeof(struct subject_map *) * subj_map_set.s_size);
	memset(acl_role_set.r_hash, 0,
	       sizeof (struct acl_role_label *) * acl_role_set.r_size);
	memset(name_set.n_hash, 0,
	       sizeof (struct name_entry *) * name_set.n_size);
	memset(inodev_set.n_hash, 0,
	       sizeof (struct name_entry *) * inodev_set.n_size);

	return 0;
}

/* free information not needed after startup
   currently contains user->kernel pointer mappings for subjects
*/

static void
free_init_variables(void)
{
	__u32 i;

	if (subj_map_set.s_hash) {
		for (i = 0; i < subj_map_set.s_size; i++) {
			if (subj_map_set.s_hash[i]) {
				kfree(subj_map_set.s_hash[i]);
				subj_map_set.s_hash[i] = NULL;
			}
		}

		if ((subj_map_set.s_size * sizeof (struct subject_map *)) <=
		    PAGE_SIZE)
			kfree(subj_map_set.s_hash);
		else
			vfree(subj_map_set.s_hash);
	}

	return;
}

static void
free_variables(void)
{
	struct acl_subject_label *s;
	struct acl_role_label *r;
	struct task_struct *task;

	gr_clear_learn_entries();

	read_lock(&tasklist_lock);
	for_each_task(task) {
		task->acl_sp_role = 0;
		task->acl_role_id = 0;
		task->acl = NULL;
		task->role = NULL;
	}
	read_unlock(&tasklist_lock);

	/* release the reference to the real root dentry and vfsmount */
	if (real_root)
		dput(real_root);
	real_root = NULL;
	if (real_root_mnt)
		mntput(real_root_mnt);
	real_root_mnt = NULL;

	/* free all object hash tables */

	if (role_list_head) {
		for (r = role_list_head; r; r = r->next) {
			if (!r->subj_hash)
				break;
			for (s = r->hash->first; s; s = s->next) {
				if (!s->obj_hash)
					break;
				if ((s->obj_hash_size *
				     sizeof (struct acl_object_label *)) <=
				    PAGE_SIZE)
					kfree(s->obj_hash);
				else
					vfree(s->obj_hash);
			}
			if ((r->subj_hash_size *
			     sizeof (struct acl_subject_label *)) <= PAGE_SIZE)
				kfree(r->subj_hash);
			else
				vfree(r->subj_hash);
		}
	}

	acl_free_all();

	if (acl_role_set.r_hash) {
		if ((acl_role_set.r_size * sizeof (struct acl_role_label *)) <=
		    PAGE_SIZE)
			kfree(acl_role_set.r_hash);
		else
			vfree(acl_role_set.r_hash);
	}
	if (name_set.n_hash) {
		if ((name_set.n_size * sizeof (struct name_entry *)) <=
		    PAGE_SIZE)
			kfree(name_set.n_hash);
		else
			vfree(name_set.n_hash);
	}

	if (inodev_set.n_hash) {
		if ((inodev_set.n_size * sizeof (struct name_entry *)) <=
		    PAGE_SIZE)
			kfree(inodev_set.n_hash);
		else
			vfree(inodev_set.n_hash);
	}

	gr_free_uidset();

	memset(&name_set, 0, sizeof (struct name_db));
	memset(&inodev_set, 0, sizeof (struct name_db));
	memset(&acl_role_set, 0, sizeof (struct acl_role_db));
	memset(&subj_map_set, 0, sizeof (struct acl_subj_map_db));

	role_list_head = NULL;
	default_role = NULL;

	return;
}

static __u32
count_user_objs(struct acl_object_label *userp)
{
	struct acl_object_label o_tmp;
	__u32 num = 0;

	while (userp) {
		if (copy_from_user(&o_tmp, userp,
				   sizeof (struct acl_object_label)))
			break;

		userp = o_tmp.prev;
		num++;
	}

	return num;
}

static struct acl_subject_label *
do_copy_user_subj(struct acl_subject_label *userp, struct acl_role_label *role);

static int
copy_user_glob(struct acl_object_label *obj)
{
	struct acl_object_label *g_tmp, **guser, *glast = NULL;
	unsigned int len;
	char *tmp;

	if (obj->globbed == NULL)
		return 0;

	guser = &obj->globbed;
	while (*guser) {
		g_tmp = (struct acl_object_label *)
			acl_alloc(sizeof (struct acl_object_label));
		if (g_tmp == NULL)
			return -ENOMEM;

		if (copy_from_user(g_tmp, *guser,
				   sizeof (struct acl_object_label)))
			return -EFAULT;

		len = strnlen_user(g_tmp->filename, PATH_MAX);

		if (!len || len >= PATH_MAX)
			return -EINVAL;

		if ((tmp = (char *) acl_alloc(len)) == NULL)
			return -ENOMEM;

		if (copy_from_user(tmp, g_tmp->filename, len))
			return -EFAULT;

		g_tmp->filename = tmp;

		if (glast)
			glast->next = g_tmp;
		g_tmp->prev = glast;
		*guser = g_tmp;
		glast = g_tmp;
		guser = &((*guser)->next);
	}
		
	return 0;
}

static int
copy_user_objs(struct acl_object_label *userp, struct acl_subject_label *subj,
	       struct acl_role_label *role)
{
	struct acl_object_label *o_tmp;
	unsigned int len;
	int ret;
	char *tmp;

	while (userp) {
		if ((o_tmp = (struct acl_object_label *)
		     acl_alloc(sizeof (struct acl_object_label))) == NULL)
			return -ENOMEM;

		if (copy_from_user(o_tmp, userp,
				   sizeof (struct acl_object_label)))
			return -EFAULT;

		userp = o_tmp->prev;

		len = strnlen_user(o_tmp->filename, PATH_MAX);

		if (!len || len >= PATH_MAX)
			return -EINVAL;

		if ((tmp = (char *) acl_alloc(len)) == NULL)
			return -ENOMEM;

		if (copy_from_user(tmp, o_tmp->filename, len))
			return -EFAULT;

		o_tmp->filename = tmp;

		insert_acl_obj_label(o_tmp, subj);
		if (!insert_name_entry(o_tmp->filename, o_tmp->inode,
				       o_tmp->device))
			return -ENOMEM;

		ret = copy_user_glob(o_tmp);
		if (ret)
			return ret;

		if (o_tmp->nested) {
			o_tmp->nested = do_copy_user_subj(o_tmp->nested, role);
			if (IS_ERR(o_tmp->nested))
				return PTR_ERR(o_tmp->nested);

			s_final = o_tmp->nested;
		}
	}

	return 0;
}

static __u32
count_user_subjs(struct acl_subject_label *userp)
{
	struct acl_subject_label s_tmp;
	__u32 num = 0;

	while (userp) {
		if (copy_from_user(&s_tmp, userp,
				   sizeof (struct acl_subject_label)))
			break;

		userp = s_tmp.prev;
		/* do not count nested subjects against this count, since
		   they are not included in the hash table, but are
		   attached to objects.  We have already counted
		   the subjects in userspace for the allocation 
		   stack
		*/
		if (!(s_tmp.mode & GR_NESTED))
			num++;
	}

	return num;
}

static int
copy_user_allowedips(struct acl_role_label *rolep)
{
	struct role_allowed_ip *ruserip, *rtmp = NULL, *rlast;

	ruserip = rolep->allowed_ips;

	while (ruserip) {
		rlast = rtmp;

		if ((rtmp = (struct role_allowed_ip *)
		     acl_alloc(sizeof (struct role_allowed_ip))) == NULL)
			return -ENOMEM;

		if (copy_from_user(rtmp, ruserip,
				   sizeof (struct role_allowed_ip)))
			return -EFAULT;

		ruserip = rtmp->prev;

		if (!rlast) {
			rtmp->prev = NULL;
			rolep->allowed_ips = rtmp;
		} else {
			rlast->next = rtmp;
			rtmp->prev = rlast;
		}

		if (!ruserip)
			rtmp->next = NULL;
	}

	return 0;
}

static int
copy_user_transitions(struct acl_role_label *rolep)
{
	struct role_transition *rusertp, *rtmp = NULL, *rlast;
	unsigned int len;
	char *tmp;

	rusertp = rolep->transitions;

	while (rusertp) {
		rlast = rtmp;

		if ((rtmp = (struct role_transition *)
		     acl_alloc(sizeof (struct role_transition))) == NULL)
			return -ENOMEM;

		if (copy_from_user(rtmp, rusertp,
				   sizeof (struct role_transition)))
			return -EFAULT;

		rusertp = rtmp->prev;

		len = strnlen_user(rtmp->rolename, GR_SPROLE_LEN);

		if (!len || len >= GR_SPROLE_LEN)
			return -EINVAL;

		if ((tmp = (char *) acl_alloc(len)) == NULL)
			return -ENOMEM;

		if (copy_from_user(tmp, rtmp->rolename, len))
			return -EFAULT;

		rtmp->rolename = tmp;

		if (!rlast) {
			rtmp->prev = NULL;
			rolep->transitions = rtmp;
		} else {
			rlast->next = rtmp;
			rtmp->prev = rlast;
		}

		if (!rusertp)
			rtmp->next = NULL;
	}

	return 0;
}

static struct acl_subject_label *
do_copy_user_subj(struct acl_subject_label *userp, struct acl_role_label *role)
{
	struct acl_subject_label *s_tmp = NULL, *s_tmp2;
	unsigned int len;
	char *tmp;
	__u32 num_objs;
	struct acl_ip_label **i_tmp, *i_utmp2;
	struct gr_hash_struct ghash;
	struct subject_map *subjmap;
	unsigned long i_num;
	int err;

	s_tmp = lookup_subject_map(userp);

	/* we've already copied this subject into the kernel, just return
	   the reference to it, and don't copy it over again
	*/
	if (s_tmp)
		return(s_tmp);

	if ((s_tmp = (struct acl_subject_label *)
	    acl_alloc(sizeof (struct acl_subject_label))) == NULL)
		return ERR_PTR(-ENOMEM);

	subjmap = (struct subject_map *)kmalloc(sizeof (struct subject_map), GFP_KERNEL);
	if (subjmap == NULL)
		return ERR_PTR(-ENOMEM);

	subjmap->user = userp;
	subjmap->kernel = s_tmp;
	insert_subj_map_entry(subjmap);

	if (copy_from_user(s_tmp, userp,
			   sizeof (struct acl_subject_label)))
		return ERR_PTR(-EFAULT);

	if (!s_last) {
		s_tmp->prev = NULL;
		role->hash->first = s_tmp;
	} else {
		s_last->next = s_tmp;
		s_tmp->prev = s_last;
	}

	s_last = s_tmp;

	len = strnlen_user(s_tmp->filename, PATH_MAX);

	if (!len || len >= PATH_MAX)
		return ERR_PTR(-EINVAL);

	if ((tmp = (char *) acl_alloc(len)) == NULL)
		return ERR_PTR(-ENOMEM);

	if (copy_from_user(tmp, s_tmp->filename, len))
		return ERR_PTR(-EFAULT);

	s_tmp->filename = tmp;

	if (!strcmp(s_tmp->filename, "/"))
		role->root_label = s_tmp;

	if (copy_from_user(&ghash, s_tmp->hash, sizeof(struct gr_hash_struct)))
		return ERR_PTR(-EFAULT);

	/* copy user and group transition tables */

	if (s_tmp->user_trans_num) {
		uid_t *uidlist;

		uidlist = (uid_t *)acl_alloc(s_tmp->user_trans_num * sizeof(uid_t));
		if (uidlist == NULL)
			return ERR_PTR(-ENOMEM);
		if (copy_from_user(uidlist, s_tmp->user_transitions, s_tmp->user_trans_num * sizeof(uid_t)))
			return ERR_PTR(-EFAULT);

		s_tmp->user_transitions = uidlist;
	}

	if (s_tmp->group_trans_num) {
		gid_t *gidlist;

		gidlist = (gid_t *)acl_alloc(s_tmp->group_trans_num * sizeof(gid_t));
		if (gidlist == NULL)
			return ERR_PTR(-ENOMEM);
		if (copy_from_user(gidlist, s_tmp->group_transitions, s_tmp->group_trans_num * sizeof(gid_t)))
			return ERR_PTR(-EFAULT);

		s_tmp->group_transitions = gidlist;
	}

	/* set up object hash table */
	num_objs = count_user_objs(ghash.first);

	s_tmp->obj_hash_size = num_objs;
	s_tmp->obj_hash =
	    (struct acl_object_label **)
	    create_table(&(s_tmp->obj_hash_size));

	if (!s_tmp->obj_hash)
		return ERR_PTR(-ENOMEM);

	memset(s_tmp->obj_hash, 0,
	       s_tmp->obj_hash_size *
	       sizeof (struct acl_object_label *));

	/* copy before adding in objects, since a nested
	   acl could be found and be the final subject
	   copied
	*/

	s_final = s_tmp;

	/* add in objects */
	err = copy_user_objs(ghash.first, s_tmp, role);

	if (err)
		return ERR_PTR(err);

	/* set pointer for parent subject */
	if (s_tmp->parent_subject) {
		s_tmp2 = do_copy_user_subj(s_tmp->parent_subject, role);

		if (IS_ERR(s_tmp2))
			return s_tmp2;

		s_tmp->parent_subject = s_tmp2;
	}

	/* add in ip acls */

	if (!s_tmp->ip_num) {
		s_tmp->ips = NULL;
		goto insert;
	}

	i_tmp =
	    (struct acl_ip_label **) acl_alloc(s_tmp->ip_num *
					       sizeof (struct
						       acl_ip_label *));

	if (!i_tmp)
		return ERR_PTR(-ENOMEM);

	for (i_num = 0; i_num < s_tmp->ip_num; i_num++) {
		*(i_tmp + i_num) =
		    (struct acl_ip_label *)
		    acl_alloc(sizeof (struct acl_ip_label));
		if (!*(i_tmp + i_num))
			return ERR_PTR(-ENOMEM);

		if (copy_from_user
		    (&i_utmp2, s_tmp->ips + i_num,
		     sizeof (struct acl_ip_label *)))
			return ERR_PTR(-EFAULT);

		if (copy_from_user
		    (*(i_tmp + i_num), i_utmp2,
		     sizeof (struct acl_ip_label)))
			return ERR_PTR(-EFAULT);
	}

	s_tmp->ips = i_tmp;

insert:
	if (!insert_name_entry(s_tmp->filename, s_tmp->inode,
			       s_tmp->device))
		return ERR_PTR(-ENOMEM);

	return s_tmp;
}

static int
copy_user_subjs(struct acl_subject_label *userp, struct acl_role_label *role)
{
	struct acl_subject_label s_pre;
	struct acl_subject_label * ret;
	int err;

	while (userp) {
		if (copy_from_user(&s_pre, userp,
				   sizeof (struct acl_subject_label)))
			return -EFAULT;
		
		/* do not add nested subjects here, add
		   while parsing objects
		*/

		if (s_pre.mode & GR_NESTED) {
			userp = s_pre.prev;
			continue;
		}

		ret = do_copy_user_subj(userp, role);

		err = PTR_ERR(ret);
		if (IS_ERR(ret))
			return err;

		insert_acl_subj_label(ret, role);

		userp = s_pre.prev;
	}

	s_final->next = NULL;

	return 0;
}

static int
copy_user_acl(struct gr_arg *arg)
{
	struct acl_role_label *r_tmp = NULL, **r_utmp, *r_utmp2, *r_last;
	struct sprole_pw *sptmp;
	struct gr_hash_struct *ghash;
	uid_t *domainlist;
	unsigned long r_num;
	unsigned int len;
	char *tmp;
	int err = 0;
	__u16 i;
	__u32 num_subjs;

	/* we need a default and kernel role */
	if (arg->role_db.num_roles < 2)
		return -EINVAL;

	/* copy special role authentication info from userspace */

	num_sprole_pws = arg->num_sprole_pws;
	acl_special_roles = (struct sprole_pw **) acl_alloc(num_sprole_pws * sizeof(struct sprole_pw *));

	if (!acl_special_roles) {
		err = -ENOMEM;
		goto cleanup;
	}

	for (i = 0; i < num_sprole_pws; i++) {
		sptmp = (struct sprole_pw *) acl_alloc(sizeof(struct sprole_pw));
		if (!sptmp) {
			err = -ENOMEM;
			goto cleanup;
		}
		if (copy_from_user(sptmp, arg->sprole_pws + i,
				   sizeof (struct sprole_pw))) {
			err = -EFAULT;
			goto cleanup;
		}

		len =
		    strnlen_user(sptmp->rolename, GR_SPROLE_LEN);

		if (!len || len >= GR_SPROLE_LEN) {
			err = -EINVAL;
			goto cleanup;
		}

		if ((tmp = (char *) acl_alloc(len)) == NULL) {
			err = -ENOMEM;
			goto cleanup;
		}

		if (copy_from_user(tmp, sptmp->rolename, len)) {
			err = -EFAULT;
			goto cleanup;
		}

#ifdef CONFIG_GRKERNSEC_ACL_DEBUG
		printk(KERN_ALERT "Copying special role %s\n", tmp);
#endif
		sptmp->rolename = tmp;
		acl_special_roles[i] = sptmp;
	}

	r_utmp = (struct acl_role_label **) arg->role_db.r_table;

	for (r_num = 0; r_num < arg->role_db.num_roles; r_num++) {
		r_last = r_tmp;

		r_tmp = acl_alloc(sizeof (struct acl_role_label));

		if (!r_tmp) {
			err = -ENOMEM;
			goto cleanup;
		}

		if (copy_from_user(&r_utmp2, r_utmp + r_num,
				   sizeof (struct acl_role_label *))) {
			err = -EFAULT;
			goto cleanup;
		}

		if (copy_from_user(r_tmp, r_utmp2,
				   sizeof (struct acl_role_label))) {
			err = -EFAULT;
			goto cleanup;
		}

		if (!r_last) {
			r_tmp->prev = NULL;
			role_list_head = r_tmp;
		} else {
			r_last->next = r_tmp;
			r_tmp->prev = r_last;
		}

		if (r_num == (arg->role_db.num_roles - 1))
			r_tmp->next = NULL;

		len = strnlen_user(r_tmp->rolename, GR_SPROLE_LEN);

		if (!len || len >= PATH_MAX) {
			err = -EINVAL;
			goto cleanup;
		}

		if ((tmp = (char *) acl_alloc(len)) == NULL) {
			err = -ENOMEM;
			goto cleanup;
		}
		if (copy_from_user(tmp, r_tmp->rolename, len)) {
			err = -EFAULT;
			goto cleanup;
		}
		r_tmp->rolename = tmp;

		if (!strcmp(r_tmp->rolename, "default")
		    && (r_tmp->roletype & GR_ROLE_DEFAULT)) {
			default_role = r_tmp;
		} else if (!strcmp(r_tmp->rolename, ":::kernel:::")) {
			kernel_role = r_tmp;
		}

		if ((ghash = (struct gr_hash_struct *) acl_alloc(sizeof(struct gr_hash_struct))) == NULL) {
			err = -ENOMEM;
			goto cleanup;
		}
		if (copy_from_user(ghash, r_tmp->hash, sizeof(struct gr_hash_struct))) {
			err = -EFAULT;
			goto cleanup;
		}

		r_tmp->hash = ghash;

		num_subjs = count_user_subjs(r_tmp->hash->first);

		r_tmp->subj_hash_size = num_subjs;
		r_tmp->subj_hash =
		    (struct acl_subject_label **)
		    create_table(&(r_tmp->subj_hash_size));

		if (!r_tmp->subj_hash) {
			err = -ENOMEM;
			goto cleanup;
		}

		err = copy_user_allowedips(r_tmp);
		if (err)
			goto cleanup;

		/* copy domain info */
		if (r_tmp->domain_children != NULL) {
			domainlist = acl_alloc(r_tmp->domain_child_num * sizeof(uid_t));
			if (domainlist == NULL) {
				err = -ENOMEM;
				goto cleanup;
			}
			if (copy_from_user(domainlist, r_tmp->domain_children, r_tmp->domain_child_num * sizeof(uid_t))) {
				err = -EFAULT;
				goto cleanup;
			}
			r_tmp->domain_children = domainlist;
		}

		err = copy_user_transitions(r_tmp);
		if (err)
			goto cleanup;

		memset(r_tmp->subj_hash, 0,
		       r_tmp->subj_hash_size *
		       sizeof (struct acl_subject_label *));

		s_last = NULL;

		err = copy_user_subjs(r_tmp->hash->first, r_tmp);

		if (err)
			goto cleanup;

		insert_acl_role_label(r_tmp);
	}

      cleanup:
	return err;

}

static int
gracl_init(struct gr_arg *args)
{
	int error = 0;

	memcpy(gr_system_salt, args->salt, GR_SALT_LEN);
	memcpy(gr_system_sum, args->sum, GR_SHA_LEN);

	if (init_variables(args)) {
		security_alert_good(GR_INITF_ACL_MSG, GR_VERSION);
		error = -ENOMEM;
		free_variables();
		goto out;
	}

	error = copy_user_acl(args);
	free_init_variables();
	if (error) {
		free_variables();
		goto out;
	}

	if ((error = gr_set_acls(0))) {
		free_variables();
		goto out;
	}

	gr_status |= GR_READY;
      out:
	return error;
}

/* derived from glibc fnmatch() 0: match, 1: no match*/

static int
glob_match(const char *pattern, const char *string)
{
	char *p = pattern, *n = string;
	char c;

	while ((c = *p++) != '\0') {
	switch (c) {
		case '?':
			if (*n == '\0')
				return 1;
			else if (*n == '/')
				return 1;
			break;
		case '\\':
			if (*n != c)
				return 1;
			break;
		case '*':
			for (c = *p++; c == '?' || c == '*'; c = *p++) {
				if (*n == '/')
					return 1;
				else if (c == '?') {
					if (*n == '\0')
						return 1;
					else
						++n;
				}
			}
			if (c == '\0') {
				if (strchr(n, '/') != NULL)
					return 1;
				else
					return 0;
			} else {
				const char *endp;

				if ((endp = strchr(n, '/')) == NULL)
					endp = n + strlen(n);

				if (c == '[') {
					for (--p; n < endp; ++n)
						if (!glob_match(p, n))
							return 0;
				} else if (c == '/') {
					while (*n != '\0' && *n != '/')
						++n;
					if (*n == '/' && !glob_match(p, n + 1))
						return 0;
				} else {
					for (--p; n < endp; ++n)
						if (*n == c && !glob_match(p, n))
							return 0;
				}

				return 1;
			}
		case '[':
			{
			int not;
			char cold;

			if (*n == '\0' || *n == '/')
				return 1;

			not = (*p == '!' || *p == '^');
			if (not)
				++p;

			c = *p++;
			for (;;) {
				unsigned char fn = (unsigned char)*n;

				if (c == '\0')
					return 1;
				else {
					if (c == fn)
						goto matched;
					cold = c;
					c = *p++;

					if (c == '-' && *p != ']') {
						unsigned char cend = *p++;

						if (cend == '\0')
							return 1;

						if (cold <= fn && fn <= cend)
							goto matched;

						c = *p++;
					}
				}

				if (c == ']')
					break;
			}
			if (!not)
				return 1;
			break;
		matched:
			while (c != ']') {
				if (c == '\0')
					return 1;

				c = *p++;
			}
			if (not)
				return 1;
		}
		break;
	default:
		if (c != *n)
			return 1;
	}

	++n;
	}
				
	if (*n == '\0')
		return 0;

	if (*n == '/')
		return 0;

	return 1;
}

static struct acl_object_label *
chk_glob_label(struct acl_object_label *globbed,
		struct dentry *dentry, struct vfsmount *mnt, char **path)
{
	struct acl_object_label *tmp;

	if (*path == NULL)
		*path = gr_to_filename_nolock(dentry, mnt);

	tmp = globbed;

	while (tmp) {
		if (!glob_match(tmp->filename, *path))
			return tmp;
		tmp = tmp->next;
	}

	return NULL;
}

static __inline__ struct acl_object_label *
full_lookup(const struct dentry *orig_dentry, const struct vfsmount *orig_mnt,
	    struct dentry *curr_dentry,
	    const struct acl_subject_label *subj, char **path)
{
	struct acl_subject_label *tmpsubj;
	struct acl_object_label *retval;
	struct acl_object_label *retval2;

	tmpsubj = (struct acl_subject_label *) subj;
	read_lock(&gr_inode_lock);
	do {
		retval =
		    	lookup_acl_obj_label(curr_dentry->d_inode->i_ino,
					 curr_dentry->d_inode->i_dev, tmpsubj);
		if (retval) {
			if (retval->globbed) {
				retval2 = chk_glob_label(retval->globbed, (struct dentry *)orig_dentry,
						  (struct vfsmount *)orig_mnt, path);
				if (retval2)
					retval = retval2;
			}
			break;
		}
	} while ((tmpsubj = tmpsubj->parent_subject));
	read_unlock(&gr_inode_lock);

	return retval;
}

static struct acl_object_label *
chk_obj_label(const struct dentry *l_dentry, const struct vfsmount *l_mnt,
	      const struct acl_subject_label *subj)
{
	struct dentry *dentry = (struct dentry *) l_dentry;
	struct vfsmount *mnt = (struct vfsmount *) l_mnt;
	struct acl_object_label *retval;
	char *path = NULL;

	spin_lock(&dcache_lock);

	for (;;) {
		if (dentry == real_root && mnt == real_root_mnt)
			break;
		if (dentry == mnt->mnt_root || IS_ROOT(dentry)) {
			if (mnt->mnt_parent == mnt)
				break;

			retval = full_lookup(l_dentry, l_mnt, dentry, subj, &path);
			if (retval != NULL)
				goto out;

			dentry = mnt->mnt_mountpoint;
			mnt = mnt->mnt_parent;
			continue;
		}

		retval = full_lookup(l_dentry, l_mnt, dentry, subj, &path);
		if (retval != NULL)
			goto out;

		dentry = dentry->d_parent;
	}

	retval = full_lookup(l_dentry, l_mnt, dentry, subj, &path);

	if (retval == NULL)
		retval = full_lookup(l_dentry, l_mnt, real_root, subj, &path);
      out:
	spin_unlock(&dcache_lock);

	return retval;
}

static struct acl_object_label *
chk_obj_create_label(const struct dentry *l_dentry, const struct vfsmount *l_mnt,
	      const struct acl_subject_label *subj, char *path)
{
	struct dentry *dentry = (struct dentry *) l_dentry;
	struct vfsmount *mnt = (struct vfsmount *) l_mnt;
	struct acl_object_label *retval;

	spin_lock(&dcache_lock);

	for (;;) {
		if (dentry == real_root && mnt == real_root_mnt)
			break;
		if (dentry == mnt->mnt_root || IS_ROOT(dentry)) {
			if (mnt->mnt_parent == mnt)
				break;

			retval = full_lookup(l_dentry, l_mnt, dentry, subj, &path);
			if (retval != NULL)
				goto out;

			dentry = mnt->mnt_mountpoint;
			mnt = mnt->mnt_parent;
			continue;
		}

		retval = full_lookup(l_dentry, l_mnt, dentry, subj, &path);
		if (retval != NULL)
			goto out;

		dentry = dentry->d_parent;
	}

	retval = full_lookup(l_dentry, l_mnt, dentry, subj, &path);

	if (retval == NULL)
		retval = full_lookup(l_dentry, l_mnt, real_root, subj, &path);
      out:
	spin_unlock(&dcache_lock);

	return retval;
}

static struct acl_subject_label *
chk_subj_label(const struct dentry *l_dentry, const struct vfsmount *l_mnt,
	       const struct acl_role_label *role)
{
	struct dentry *dentry = (struct dentry *) l_dentry;
	struct vfsmount *mnt = (struct vfsmount *) l_mnt;
	struct acl_subject_label *retval;

	spin_lock(&dcache_lock);

	for (;;) {
		if (unlikely(dentry == real_root && mnt == real_root_mnt))
			break;
		if (unlikely(dentry == mnt->mnt_root || IS_ROOT(dentry))) {
			if (mnt->mnt_parent == mnt)
				break;

			read_lock(&gr_inode_lock);
			retval =
			    lookup_acl_subj_label(dentry->d_inode->i_ino,
						  dentry->d_inode->i_dev, role);
			read_unlock(&gr_inode_lock);
			if (unlikely(retval != NULL))
				goto out;

			dentry = mnt->mnt_mountpoint;
			mnt = mnt->mnt_parent;
			continue;
		}

		read_lock(&gr_inode_lock);
		retval =
		    lookup_acl_subj_label(dentry->d_inode->i_ino,
					  dentry->d_inode->i_dev, role);
		read_unlock(&gr_inode_lock);
		if (unlikely(retval != NULL))
			goto out;

		dentry = dentry->d_parent;
	}

	read_lock(&gr_inode_lock);
	retval =
	    lookup_acl_subj_label(dentry->d_inode->i_ino,
				  dentry->d_inode->i_dev, role);
	read_unlock(&gr_inode_lock);

	if (unlikely(retval == NULL)) {
		read_lock(&gr_inode_lock);
		retval =
		    lookup_acl_subj_label(real_root->d_inode->i_ino,
					  real_root->d_inode->i_dev, role);
		read_unlock(&gr_inode_lock);
	}
      out:
	spin_unlock(&dcache_lock);

	return retval;
}

static __inline__ void
gr_log_learn(const struct acl_role_label *role, const uid_t uid, const gid_t gid,
	     const struct task_struct *task, const char *pathname,
	     const __u32 mode)
{
	security_learn(GR_LEARN_AUDIT_MSG, role->rolename, role->roletype,
		       uid, gid, task->exec_file ? gr_to_filename1(task->exec_file->f_dentry,
		       task->exec_file->f_vfsmnt) : task->acl->filename, task->acl->filename,
		       1, 1, pathname, (unsigned long) mode, NIPQUAD(task->curr_ip));

	return;
}

__u32
gr_check_link(const struct dentry * new_dentry,
	      const struct dentry * parent_dentry,
	      const struct vfsmount * parent_mnt,
	      const struct dentry * old_dentry, const struct vfsmount * old_mnt)
{
	struct acl_object_label *obj;
	__u32 oldmode, newmode;

	if (unlikely(!(gr_status & GR_READY)))
		return (GR_WRITE | GR_CREATE);

	obj = chk_obj_label(old_dentry, old_mnt, current->acl);
	oldmode = obj->mode;

	if (current->acl->mode & GR_LEARN)
		oldmode |= (GR_WRITE | GR_CREATE);
	newmode =
	    gr_check_create(new_dentry, parent_dentry, parent_mnt,
			    oldmode | GR_CREATE | GR_AUDIT_CREATE |
			    GR_AUDIT_WRITE | GR_SUPPRESS);

	if ((newmode & oldmode) == oldmode)
		return newmode;
	else if (current->acl->mode & GR_LEARN) {
		gr_log_learn(current->role, current->uid, current->gid,
			current, gr_to_filename(old_dentry, old_mnt), oldmode);
		return (GR_WRITE | GR_CREATE);
	} else if (newmode & GR_SUPPRESS)
		return GR_SUPPRESS;
	else
		return 0;
}

__u32
gr_search_file(const struct dentry * dentry, const __u32 mode,
	       const struct vfsmount * mnt)
{
	__u32 retval = mode;
	struct acl_subject_label *curracl;
	struct acl_object_label *currobj;

	if (unlikely(!(gr_status & GR_READY)))
		return (mode & ~GR_AUDITS);

	curracl = current->acl;

	currobj = chk_obj_label(dentry, mnt, curracl);
	retval = currobj->mode & mode;

	if (unlikely
	    ((curracl->mode & GR_LEARN) && !(mode & GR_NOPTRACE)
	     && (retval != (mode & ~(GR_AUDITS | GR_SUPPRESS))))) {
		__u32 new_mode = mode;

		new_mode &= ~(GR_AUDITS | GR_SUPPRESS);

		retval = new_mode;

		if (!(mode & GR_NOLEARN))
			gr_log_learn(current->role, current->uid, current->gid,
				     current, gr_to_filename(dentry, mnt), new_mode);
	}

	return retval;
}

__u32
gr_check_create(const struct dentry * new_dentry, const struct dentry * parent,
		const struct vfsmount * mnt, const __u32 mode)
{
	struct name_entry *match;
	struct acl_object_label *matchpo;
	struct acl_subject_label *curracl;
	char *path;
	__u32 retval;

	if (unlikely(!(gr_status & GR_READY)))
		return (mode & ~GR_AUDITS);

	preempt_disable();
	path = gr_to_filename(new_dentry, mnt);
	match = lookup_name_entry(path);

	if (!match)
		goto check_parent;

	curracl = current->acl;

	read_lock(&gr_inode_lock);
	matchpo = lookup_acl_obj_label_create(match->inode, match->device, curracl);
	read_unlock(&gr_inode_lock);

	if (matchpo) {
		if ((matchpo->mode & mode) !=
		    (mode & ~(GR_AUDITS | GR_SUPPRESS))
		    && curracl->mode & GR_LEARN) {
			__u32 new_mode = mode;

			new_mode &= ~(GR_AUDITS | GR_SUPPRESS);

			gr_log_learn(current->role, current->uid, current->gid,
				     current, gr_to_filename(new_dentry, mnt), new_mode);

			preempt_enable();
			return new_mode;
		}
		preempt_enable();
		return (matchpo->mode & mode);
	}

	check_parent:
	curracl = current->acl;

	matchpo = chk_obj_create_label(parent, mnt, curracl, path);
	retval = matchpo->mode & mode;

	if ((retval != (mode & ~(GR_AUDITS | GR_SUPPRESS)))
	    && (curracl->mode & GR_LEARN)) {
		__u32 new_mode = mode;

		new_mode &= ~(GR_AUDITS | GR_SUPPRESS);

		gr_log_learn(current->role, current->uid, current->gid, 
			     current, gr_to_filename(new_dentry, mnt), new_mode);
		preempt_enable();
		return new_mode;
	}

	preempt_enable();
	return retval;
}

int
gr_check_hidden_task(const struct task_struct *task)
{
	if (unlikely(!(gr_status & GR_READY)))
		return 0;

	if (!(task->acl->mode & GR_FIND) && !(current->acl->mode & GR_VIEW))
		return 1;

	return 0;
}

int
gr_check_protected_task(const struct task_struct *task)
{
	if (unlikely(!(gr_status & GR_READY) || !task))
		return 0;

	if ((task->acl->mode & GR_PROTECTED) && !(current->acl->mode & GR_KILL))
		return 1;

	return 0;
}

__inline__ void
gr_copy_label(struct task_struct *tsk)
{
	tsk->used_accept = 0;
	tsk->acl_sp_role = 0;
	tsk->acl_role_id = current->acl_role_id;
	tsk->acl = current->acl;
	tsk->role = current->role;
	tsk->curr_ip = current->curr_ip;
	if (current->exec_file)
		get_file(current->exec_file);
	tsk->exec_file = current->exec_file;
	tsk->is_writable = current->is_writable;
	if (unlikely(current->used_accept))
		current->curr_ip = 0;

	return;
}

static __inline__ void
gr_set_proc_res(void)
{
	struct acl_subject_label *proc;
	unsigned short i;

	proc = current->acl;

	if (proc->mode & GR_LEARN)
		return;

	for (i = 0; i < RLIM_NLIMITS; i++) {
		if (!(proc->resmask & (1 << i)))
			continue;

		current->rlim[i].rlim_cur = proc->res[i].rlim_cur;
		current->rlim[i].rlim_max = proc->res[i].rlim_max;
	}

	return;
}

#ifdef CONFIG_GRKERNSEC_PAX_HAVE_ACL_FLAGS
void
pax_set_flags(struct linux_binprm *bprm)
{
	struct task_struct *task = current;
	struct acl_subject_label *proc;

	if (unlikely(!(gr_status & GR_READY)))
		return;

	proc = task->acl;

	if (proc->mode & GR_PAXPAGE)
		task->flags &= ~PF_PAX_PAGEEXEC;
	if (proc->mode & GR_PAXSEGM)
		task->flags &= ~PF_PAX_SEGMEXEC;
	if (proc->mode & GR_PAXGCC)
		task->flags |= PF_PAX_EMUTRAMP;
	if (proc->mode & GR_PAXMPROTECT)
		task->flags &= ~PF_PAX_MPROTECT;
	if (proc->mode & GR_PAXRANDMMAP)
		task->flags &= ~PF_PAX_RANDMMAP;
	if (proc->mode & GR_PAXRANDEXEC)
		task->flags |= PF_PAX_RANDEXEC;

	return;
}
#endif

static __inline__ void
do_set_role_label(struct task_struct *task, const uid_t uid, const gid_t gid)
{
	task->role = lookup_acl_role_label(task, uid, gid);

	return;
}

int
gr_check_user_change(int real, int effective, int fs)
{
	unsigned int i;
	__u16 num;
	uid_t *uidlist;
	int curuid;
	int realok = 0;
	int effectiveok = 0;
	int fsok = 0;

	if (unlikely(!(gr_status & GR_READY)))
		return 0;

	num = current->acl->user_trans_num;
	uidlist = current->acl->user_transitions;

	if (uidlist == NULL)
		return 0;

	if (real == -1)
		realok = 1;
	if (effective == -1)
		effectiveok = 1;
	if (fs == -1)
		fsok = 1;

	if (current->acl->user_trans_type & GR_ID_ALLOW) {
		for (i = 0; i < num; i++) {
			curuid = (int)uidlist[i];
			if (real == curuid)
				realok = 1;
			if (effective == curuid)
				effectiveok = 1;
			if (fs == curuid)
				fsok = 1;
		}
	} else if (current->acl->user_trans_type & GR_ID_DENY) {
		for (i = 0; i < num; i++) {
			curuid = (int)uidlist[i];
			if (real == curuid)
				break;
			if (effective == curuid)
				break;
			if (fs == curuid)
				break;
		}
		/* not in deny list */
		if (i == num) {
			realok = 1;
			effectiveok = 1;
			fsok = 1;
		}
	}

	if (realok && effectiveok && fsok)
		return 0;
	else {
		security_alert(GR_USRCHANGE_ACL_MSG,
			realok ? (effectiveok ? (fsok ? 0 : fs) : effective) : real, DEFAULTSECARGS);
		return 1;
	}
}

int
gr_check_group_change(int real, int effective, int fs)
{
	unsigned int i;
	__u16 num;
	gid_t *gidlist;
	int curgid;
	int realok = 0;
	int effectiveok = 0;
	int fsok = 0;

	if (unlikely(!(gr_status & GR_READY)))
		return 0;

	num = current->acl->group_trans_num;
	gidlist = current->acl->group_transitions;

	if (gidlist == NULL)
		return 0;

	if (real == -1)
		realok = 1;
	if (effective == -1)
		effectiveok = 1;
	if (fs == -1)
		fsok = 1;

	if (current->acl->group_trans_type & GR_ID_ALLOW) {
		for (i = 0; i < num; i++) {
			curgid = (int)gidlist[i];
			if (real == curgid)
				realok = 1;
			if (effective == curgid)
				effectiveok = 1;
			if (fs == curgid)
				fsok = 1;
		}
	} else if (current->acl->group_trans_type & GR_ID_DENY) {
		for (i = 0; i < num; i++) {
			curgid = (int)gidlist[i];
			if (real == curgid)
				break;
			if (effective == curgid)
				break;
			if (fs == curgid)
				break;
		}
		/* not in deny list */
		if (i == num) {
			realok = 1;
			effectiveok = 1;
			fsok = 1;
		}
	}

	if (realok && effectiveok && fsok)
		return 0;
	else {
		security_alert(GR_GRPCHANGE_ACL_MSG,
			realok ? (effectiveok ? (fsok ? 0 : fs) : effective) : real, DEFAULTSECARGS);
		return 1;
	}
}

void
gr_set_role_label(struct task_struct *task, const uid_t uid, const uid_t gid)
{
	struct acl_object_label *obj;
	struct file *filp;

	if (unlikely(!(gr_status & GR_READY)))
		return;

	filp = task->exec_file;

	/* kernel process, we'll give them the kernel role */
	if (unlikely(!filp)) {
		task->role = kernel_role;
		task->acl = kernel_role->root_label;
		return;
	} else if (!task->role || !(task->role->roletype & GR_ROLE_SPECIAL))
		do_set_role_label(task, uid, gid);

	task->acl =
	    chk_subj_label(filp->f_dentry, filp->f_vfsmnt, task->role);

	task->is_writable = 0;

	/* ignore additional mmap checks for processes that are writable 
	   by the default ACL */
	obj = chk_obj_label(filp->f_dentry, filp->f_vfsmnt, default_role->root_label);
	if (unlikely(obj->mode & GR_WRITE))
		task->is_writable = 1;
	obj = chk_obj_label(filp->f_dentry, filp->f_vfsmnt, task->role->root_label);
	if (unlikely(obj->mode & GR_WRITE))
		task->is_writable = 1;

#ifdef CONFIG_GRKERNSEC_ACL_DEBUG
	printk(KERN_ALERT "Set role label for (%s:%d): role:%s, subject:%s\n", task->comm, task->pid, task->role->rolename, task->acl->filename);
#endif

	gr_set_proc_res();

	return;
}

int
gr_set_proc_label(const struct dentry *dentry, const struct vfsmount *mnt)
{
	struct task_struct *task = current;
	struct acl_subject_label *newacl;
	struct acl_object_label *obj;
	__u32 retmode;

	if (unlikely(!(gr_status & GR_READY)))
		return 0;

	newacl = chk_subj_label(dentry, mnt, task->role);

	task_lock(task);
	if (((task->ptrace & PT_PTRACED) && !(task->acl->mode & 
	     GR_OVERRIDE) && (task->acl != newacl) && 
	     !(task->role->roletype & GR_ROLE_GOD) && 
	     !gr_search_file(dentry, GR_PTRACERD, mnt)) ||
	    (atomic_read(&task->fs->count) > 1 ||
	     atomic_read(&task->files->count) > 1 ||
	     atomic_read(&task->sig->count) > 1)) {
		task_unlock(task);
		security_alert(GR_PTRACE_EXEC_ACL_MSG,
			       gr_to_filename(dentry, mnt), DEFAULTSECARGS);
		return -EACCES;
	}
	obj = chk_obj_label(dentry, mnt, task->acl);
	retmode = obj->mode & (GR_INHERIT | GR_AUDIT_INHERIT);

	if ((newacl->mode & GR_LEARN) || !(retmode & GR_INHERIT)) {
		if (obj->nested)
			task->acl = obj->nested;
		else
			task->acl = newacl;
		task_unlock(task);
	} else if (retmode & GR_INHERIT && retmode & GR_AUDIT_INHERIT) {
		task_unlock(task);
		security_audit(GR_INHERIT_ACL_MSG, task->acl->filename,
			       gr_to_filename(dentry, mnt), DEFAULTSECARGS);
	} else
		task_unlock(task);

	task->is_writable = 0;

	/* ignore additional mmap checks for processes that are writable 
	   by the default ACL */
	obj = chk_obj_label(dentry, mnt, default_role->root_label);
	if (unlikely(obj->mode & GR_WRITE))
		task->is_writable = 1;
	obj = chk_obj_label(dentry, mnt, task->role->root_label);
	if (unlikely(obj->mode & GR_WRITE))
		task->is_writable = 1;

	gr_set_proc_res();

#ifdef CONFIG_GRKERNSEC_ACL_DEBUG
	printk(KERN_ALERT "Set subject label for (%s:%d): role:%s, subject:%s\n", task->comm, task->pid, task->role->rolename, task->acl->filename);
#endif
	return 0;
}

static __inline__ void
do_handle_delete(const ino_t ino, const kdev_t dev)
{
	struct acl_object_label *matchpo;
	struct acl_subject_label *matchps;
	struct acl_subject_label *i;
	struct acl_role_label *role;

	for (role = role_list_head; role; role = role->next) {
		for (i = role->hash->first; i; i = i->next) {
			if (unlikely((i->mode & GR_NESTED) &&
				     (i->inode == ino) &&
				     (i->device == dev)))
				i->mode |= GR_DELETED;
			if (unlikely((matchpo =
			     lookup_acl_obj_label(ino, dev, i)) != NULL))
				matchpo->mode |= GR_DELETED;
		}

		if (unlikely((matchps = lookup_acl_subj_label(ino, dev, role)) != NULL))
			matchps->mode |= GR_DELETED;
	}

	return;
}

void
gr_handle_delete(const ino_t ino, const kdev_t dev)
{
	if (unlikely(!(gr_status & GR_READY)))
		return;

	write_lock(&gr_inode_lock);
	if (unlikely((unsigned long)lookup_inodev_entry(ino, dev)))
		do_handle_delete(ino, dev);
	write_unlock(&gr_inode_lock);

	return;
}

static __inline__ void
update_acl_obj_label(const ino_t oldinode, const kdev_t olddevice,
		     const ino_t newinode, const kdev_t newdevice,
		     struct acl_subject_label *subj)
{
	unsigned long index = fhash(oldinode, olddevice, subj->obj_hash_size);
	struct acl_object_label **match;
	struct acl_object_label *tmp;
	__u8 i = 0;

	match = &subj->obj_hash[index];

	while (*match && ((*match)->inode != oldinode ||
	       (*match)->device != olddevice ||
	       !((*match)->mode & GR_DELETED))) {
		index = (index + (1 << i)) % subj->obj_hash_size;
		match = &subj->obj_hash[index];
		i = (i + 1) % 32;
	}

	if (*match && ((*match) != deleted_object)
	    && ((*match)->inode == oldinode)
	    && ((*match)->device == olddevice)
	    && ((*match)->mode & GR_DELETED)) {
		tmp = *match;
		tmp->inode = newinode;
		tmp->device = newdevice;
		tmp->mode &= ~GR_DELETED;

		*match = deleted_object;

		insert_acl_obj_label(tmp, subj);
	}

	return;
}

static __inline__ void
update_acl_subj_label(const ino_t oldinode, const kdev_t olddevice,
		      const ino_t newinode, const kdev_t newdevice,
		      struct acl_role_label *role)
{
	struct acl_subject_label **s_hash = role->subj_hash;
	unsigned long subj_size = role->subj_hash_size;
	unsigned long index = fhash(oldinode, olddevice, subj_size);
	struct acl_subject_label **match;
	struct acl_subject_label *tmp;
	__u8 i = 0;

	match = &s_hash[index];

	while (*match && ((*match)->inode != oldinode ||
	       (*match)->device != olddevice ||
	       !((*match)->mode & GR_DELETED))) {
		index = (index + (1 << i)) % subj_size;
		i = (i + 1) % 32;
		match = &s_hash[index];
	}

	if (*match && (*match != deleted_subject)
	    && ((*match)->inode == oldinode)
	    && ((*match)->device == olddevice)
	    && ((*match)->mode & GR_DELETED)) {
		tmp = *match;

		tmp->inode = newinode;
		tmp->device = newdevice;
		tmp->mode &= ~GR_DELETED;

		*match = deleted_subject;

		insert_acl_subj_label(tmp, role);
	}

	return;
}

static __inline__ void
update_inodev_entry(const ino_t oldinode, const kdev_t olddevice,
		    const ino_t newinode, const kdev_t newdevice)
{
	unsigned long index = fhash(oldinode, olddevice, inodev_set.n_size);
	struct name_entry **match;
	struct name_entry *tmp;
	__u8 i = 0;

	match = &inodev_set.n_hash[index];

	while (*match
	       && ((*match)->inode != oldinode
		   || (*match)->device != olddevice)) {
		index = (index + (1 << i)) % inodev_set.n_size;
		i = (i + 1) % 32;
		match = &inodev_set.n_hash[index];
	}

	if (*match && (*match != deleted_inodev)
	    && ((*match)->inode == oldinode)
	    && ((*match)->device == olddevice)) {
		tmp = *match;

		tmp->inode = newinode;
		tmp->device = newdevice;

		*match = deleted_inodev;

		insert_inodev_entry(tmp);
	}

	return;
}

static __inline__ void
do_handle_create(const struct name_entry *matchn, const struct dentry *dentry,
		 const struct vfsmount *mnt)
{
	struct acl_subject_label *i;
	struct acl_role_label *role;

	for (role = role_list_head; role; role = role->next) {
		update_acl_subj_label(matchn->inode, matchn->device,
				      dentry->d_inode->i_ino,
				      dentry->d_inode->i_dev, role);

		for (i = role->hash->first; i; i = i->next) {
			if (unlikely((i->mode & GR_NESTED) &&
				     (i->inode == dentry->d_inode->i_ino) &&
				     (i->device == dentry->d_inode->i_dev))) {
				i->inode = dentry->d_inode->i_ino;
				i->device = dentry->d_inode->i_dev;
			}
			update_acl_obj_label(matchn->inode, matchn->device,
					     dentry->d_inode->i_ino,
					     dentry->d_inode->i_dev, i);
		}
	}

	update_inodev_entry(matchn->inode, matchn->device,
			    dentry->d_inode->i_ino, dentry->d_inode->i_dev);

	return;
}

void
gr_handle_create(const struct dentry *dentry, const struct vfsmount *mnt)
{
	struct name_entry *matchn;

	if (unlikely(!(gr_status & GR_READY)))
		return;

	preempt_disable();
	matchn = lookup_name_entry(gr_to_filename(dentry, mnt));
	preempt_enable();

	if (unlikely((unsigned long)matchn)) {
		write_lock(&gr_inode_lock);
		do_handle_create(matchn, dentry, mnt);
		write_unlock(&gr_inode_lock);
	}

	return;
}

int
gr_handle_rename(struct inode *old_dir, struct inode *new_dir,
		 struct dentry *old_dentry,
		 struct dentry *new_dentry,
		 struct vfsmount *mnt, const __u8 replace)
{
	struct name_entry *matchn;
	int error = 0;

	preempt_disable();
	matchn = lookup_name_entry(gr_to_filename(new_dentry, mnt));
	preempt_enable();

	lock_kernel();
	error = vfs_rename(old_dir, old_dentry, new_dir, new_dentry);
	unlock_kernel();

	if (unlikely(error))
		return error;

	/* we wouldn't have to check d_inode if it weren't for
	   NFS silly-renaming
	 */

	write_lock(&gr_inode_lock);
	if (unlikely(replace && new_dentry->d_inode)) {
		if (unlikely(lookup_inodev_entry(new_dentry->d_inode->i_ino,
					new_dentry->d_inode->i_dev) &&
		    (old_dentry->d_inode->i_nlink <= 1)))
			do_handle_delete(new_dentry->d_inode->i_ino,
					 new_dentry->d_inode->i_dev);
	}

	if (unlikely(lookup_inodev_entry(old_dentry->d_inode->i_ino,
				old_dentry->d_inode->i_dev) &&
	    (old_dentry->d_inode->i_nlink <= 1)))
		do_handle_delete(old_dentry->d_inode->i_ino,
				 old_dentry->d_inode->i_dev);

	if (unlikely((unsigned long)matchn))
		do_handle_create(matchn, old_dentry, mnt);
	write_unlock(&gr_inode_lock);

	return error;
}

static int
lookup_special_role_auth(const char *rolename, unsigned char **salt,
			 unsigned char **sum)
{
	struct acl_role_label *r;
	struct role_allowed_ip *ipp;
	struct role_transition *trans;
	__u16 i;
	int found = 0;

	/* check transition table */

	for (trans = current->role->transitions; trans; trans = trans->next) {
		if (!strcmp(rolename, trans->rolename)) {
			found = 1;
			break;
		}
	}

	if (!found)
		return 0;

	/* handle special roles that do not require authentication 
	   and check ip */

	for (r = role_list_head; r; r = r->next) {
		if (!strcmp(rolename, r->rolename) &&
		    (r->roletype & GR_ROLE_SPECIAL)) {
			found = 0;
			if (r->allowed_ips != NULL) {
				for (ipp = r->allowed_ips; ipp; ipp = ipp->next) {
					if ((ntohl(current->curr_ip) & ipp->netmask) ==
					     (ntohl(ipp->addr) & ipp->netmask))
						found = 1;
				}
			} else
				found = 2;
			if (!found)
				return 0;

			if (r->roletype & GR_ROLE_NOPW) {
				*salt = NULL;
				*sum = NULL;
				return 1;
			}
		}
	}

	for (i = 0; i < num_sprole_pws; i++) {
		if (!strcmp(rolename, acl_special_roles[i]->rolename)) {
			*salt = acl_special_roles[i]->salt;
			*sum = acl_special_roles[i]->sum;
			return 1;
		}
	}

	return 0;
}

static void
assign_special_role(char *rolename)
{
	struct acl_object_label *obj;
	struct acl_role_label *r;
	struct acl_role_label *assigned = NULL;
	struct task_struct *tsk;
	struct file *filp;

	for (r = role_list_head; r; r = r->next)
		if (!strcmp(rolename, r->rolename) &&
		    (r->roletype & GR_ROLE_SPECIAL))
			assigned = r;

	if (!assigned)
		return;

	read_lock(&tasklist_lock);
	read_lock(&grsec_exec_file_lock);
	tsk = current->p_pptr;
	if (tsk == NULL) {
		read_unlock(&grsec_exec_file_lock);
		read_unlock(&tasklist_lock);
		return;
	}

	filp = tsk->exec_file;
	if (filp == NULL) {
		read_unlock(&grsec_exec_file_lock);
		read_unlock(&tasklist_lock);
		return;
	}

	tsk->is_writable = 0;

	acl_sp_role_value = (acl_sp_role_value % 65535) + 1;
	tsk->acl_sp_role = 1;
	tsk->acl_role_id = acl_sp_role_value;
	tsk->role = assigned;
	tsk->acl = chk_subj_label(filp->f_dentry, filp->f_vfsmnt, tsk->role);

	/* ignore additional mmap checks for processes that are writable 
	   by the default ACL */
	obj = chk_obj_label(filp->f_dentry, filp->f_vfsmnt, default_role->root_label);
	if (unlikely(obj->mode & GR_WRITE))
		tsk->is_writable = 1;
	obj = chk_obj_label(filp->f_dentry, filp->f_vfsmnt, tsk->role->root_label);
	if (unlikely(obj->mode & GR_WRITE))
		tsk->is_writable = 1;

#ifdef CONFIG_GRKERNSEC_ACL_DEBUG
	printk(KERN_ALERT "Assigning special role:%s subject:%s to process (%s:%d)\n", tsk->role->rolename, tsk->acl->filename, tsk->comm, tsk->pid);
#endif

	read_unlock(&grsec_exec_file_lock);
	read_unlock(&tasklist_lock);
	return;
}

ssize_t
write_grsec_handler(struct file *file, const char * buf, size_t count, loff_t *ppos)
{
	struct gr_arg_wrapper uwrap;
	unsigned char *sprole_salt;
	unsigned char *sprole_sum;
	int error = sizeof (struct gr_arg_wrapper);
	int error2 = 0;

	down(&gr_dev_sem);

	if (count != sizeof (struct gr_arg_wrapper)) {
		security_alert_good(GR_DEV_ACL_MSG, (int)count,
				    (int) sizeof (struct gr_arg_wrapper));
		error = -EINVAL;
		goto out;
	}

	if ((gr_auth_attempts >= CONFIG_GRKERNSEC_ACL_MAXTRIES)
	    && time_before_eq(gr_auth_expires, jiffies)) {
		gr_auth_expires = 0;
		gr_auth_attempts = 0;
	}

	if (copy_from_user(&uwrap, buf, sizeof (struct gr_arg_wrapper))) {
		error = -EFAULT;
		goto out;
	}

	if ((uwrap.version != GRSECURITY_VERSION) || (uwrap.size != sizeof(struct gr_arg))) {
		error = -EINVAL;
		goto out;
	}

	if (copy_from_user(gr_usermode, uwrap.arg, sizeof (struct gr_arg))) {
		error = -EFAULT;
		goto out;
	}

	if (gr_usermode->mode != SPROLE && time_after(gr_auth_expires, jiffies)) {
		error = -EBUSY;
		goto out;
	}

	/* if non-root trying to do anything other than use a special role,
	   do not attempt authentication, do not count towards authentication
	   locking
	 */

	if (gr_usermode->mode != SPROLE && current->uid) {
		error = -EPERM;
		goto out;
	}

	/* ensure pw and special role name are null terminated */

	gr_usermode->pw[GR_PW_LEN - 1] = '\0';
	gr_usermode->sp_role[GR_SPROLE_LEN - 1] = '\0';

	/* Okay. 
	 * We have our enough of the argument structure..(we have yet
	 * to copy_from_user the tables themselves) . Copy the tables
	 * only if we need them, i.e. for loading operations. */

	switch (gr_usermode->mode) {
	case STATUS:
			if (gr_status & GR_READY)
				error = 1;
			else
				error = 2;
			goto out;
	case SHUTDOWN:
		if ((gr_status & GR_READY)
		    && !(chkpw(gr_usermode, gr_system_salt, gr_system_sum))) {
			gr_status &= ~GR_READY;
			security_alert_good(GR_SHUTS_ACL_MSG, DEFAULTSECARGS);
			free_variables();
			memset(gr_usermode, 0, sizeof (struct gr_arg));
			memset(gr_system_salt, 0, GR_SALT_LEN);
			memset(gr_system_sum, 0, GR_SHA_LEN);
		} else if (gr_status & GR_READY) {
			security_alert(GR_SHUTF_ACL_MSG, DEFAULTSECARGS);
			error = -EPERM;
		} else {
			security_alert_good(GR_SHUTI_ACL_MSG, DEFAULTSECARGS);
			error = -EAGAIN;
		}
		break;
	case ENABLE:
		if (!(gr_status & GR_READY) && !(error2 = gracl_init(gr_usermode)))
			security_alert_good(GR_ENABLE_ACL_MSG, GR_VERSION);
		else {
			if (gr_status & GR_READY)
				error = -EAGAIN;
			else
				error = error2;
			security_alert(GR_ENABLEF_ACL_MSG, GR_VERSION,
				       DEFAULTSECARGS);
		}
		break;
	case RELOAD:
		if (!(gr_status & GR_READY)) {
			security_alert_good(GR_RELOADI_ACL_MSG);
			error = -EAGAIN;
		} else if (!(chkpw(gr_usermode, gr_system_salt, gr_system_sum))) {
			lock_kernel();
			gr_status &= ~GR_READY;
			free_variables();
			if (!(error2 = gracl_init(gr_usermode))) {
				unlock_kernel();
				security_alert_good(GR_RELOAD_ACL_MSG,
						    GR_VERSION);
			} else {
				unlock_kernel();
				error = error2;
				security_alert(GR_RELOADF_ACL_MSG, GR_VERSION,
					       DEFAULTSECARGS);
			}
		} else {
			security_alert(GR_RELOADF_ACL_MSG, GR_VERSION,
				       DEFAULTSECARGS);
			error = -EPERM;
		}
		break;
	case SEGVMOD:
		if (unlikely(!(gr_status & GR_READY))) {
			security_alert_good(GR_SEGVMODI_ACL_MSG,
					    DEFAULTSECARGS);
			error = -EAGAIN;
			break;
		}

		if (!(chkpw(gr_usermode, gr_system_salt, gr_system_sum))) {
			security_alert_good(GR_SEGVMODS_ACL_MSG,
					    DEFAULTSECARGS);
			if (gr_usermode->segv_device && gr_usermode->segv_inode) {
				struct acl_subject_label *segvacl;
				segvacl =
				    lookup_acl_subj_label(gr_usermode->segv_inode,
							  gr_usermode->segv_device,
							  current->role);
				if (segvacl) {
					segvacl->crashes = 0;
					segvacl->expires = 0;
				}
			} else if (gr_find_uid(gr_usermode->segv_uid) >= 0) {
				gr_remove_uid(gr_usermode->segv_uid);
			}
		} else {
			security_alert(GR_SEGVMODF_ACL_MSG, DEFAULTSECARGS);
			error = -EPERM;
		}
		break;
	case SPROLE:
		if (unlikely(!(gr_status & GR_READY))) {
			security_alert_good(GR_SPROLEI_ACL_MSG, DEFAULTSECARGS);
			error = -EAGAIN;
			break;
		}

		if ((current->role->auth_attempts >= CONFIG_GRKERNSEC_ACL_MAXTRIES)
		    && time_before_eq(current->role->expires, jiffies)) {
			current->role->expires = 0;
			current->role->auth_attempts = 0;
		}

		if (time_after(current->role->expires, jiffies)) {
			error = -EBUSY;
			goto out;
		}

		if (lookup_special_role_auth
		    (gr_usermode->sp_role, &sprole_salt, &sprole_sum)
		    && ((!sprole_salt && !sprole_sum)
			|| !(chkpw(gr_usermode, sprole_salt, sprole_sum)))) {
			assign_special_role(gr_usermode->sp_role);
			security_alert_good(GR_SPROLES_ACL_MSG,
					    (current->p_pptr) ? current->
					    p_pptr->role->rolename : "",
					    acl_sp_role_value, DEFAULTSECARGS);
		} else {
			security_alert(GR_SPROLEF_ACL_MSG, gr_usermode->sp_role,
				       DEFAULTSECARGS);
			error = -EPERM;
			current->role->auth_attempts++;
			if (current->role->auth_attempts >= CONFIG_GRKERNSEC_ACL_MAXTRIES) {
				current->role->expires =
				    jiffies + CONFIG_GRKERNSEC_ACL_TIMEOUT * HZ;
				security_alert(GR_MAXROLEPW_ACL_MSG,
				       CONFIG_GRKERNSEC_ACL_MAXTRIES,
				       gr_usermode->sp_role, DEFAULTSECARGS);
			}

			goto out;
		}
		break;
	case UNSPROLE:
		if (unlikely(!(gr_status & GR_READY))) {
			security_alert_good(GR_UNSPROLEI_ACL_MSG, DEFAULTSECARGS);
			error = -EAGAIN;
			break;
		}

		if (current->role->roletype & GR_ROLE_SPECIAL) {
			security_alert_good(GR_UNSPROLES_ACL_MSG,
					    (current->p_pptr) ? current->
					    p_pptr->role->rolename : "",
					    (current->p_pptr) ? current->
					    p_pptr->acl_role_id : 0, DEFAULTSECARGS);
			gr_set_acls(1);
		} else {
			security_alert(GR_UNSPROLEF_ACL_MSG, current->role->rolename,
				       DEFAULTSECARGS);
			error = -EPERM;
			goto out;
		}
		break;
	default:
		security_alert(GR_INVMODE_ACL_MSG, gr_usermode->mode,
			       DEFAULTSECARGS);
		error = -EINVAL;
		break;
	}

	if (error != -EPERM)
		goto out;

	gr_auth_attempts++;

	if (gr_auth_attempts >= CONFIG_GRKERNSEC_ACL_MAXTRIES) {
		security_alert(GR_MAXPW_ACL_MSG, CONFIG_GRKERNSEC_ACL_MAXTRIES);
		gr_auth_expires = jiffies + CONFIG_GRKERNSEC_ACL_TIMEOUT * HZ;
	}

      out:
	up(&gr_dev_sem);
	return error;
}

int
gr_set_acls(const int type)
{
	struct acl_object_label *obj;
	struct task_struct *task;
	struct file *filp;
	unsigned short i;

	read_lock(&tasklist_lock);
	read_lock(&grsec_exec_file_lock);
	for_each_task(task) {
		/* check to see if we're called from the exit handler,
		   if so, only replace ACLs that have inherited the admin
		   ACL */

		if (type && (task->role != current->role ||
			     task->acl_role_id != current->acl_role_id))
			continue;

		task->acl_role_id = 0;
		task->acl_sp_role = 0;

		if ((filp = task->exec_file)) {
			do_set_role_label(task, task->uid, task->gid);

			task->acl =
			    chk_subj_label(filp->f_dentry, filp->f_vfsmnt,
					   task->role);
			if (task->acl) {
				struct acl_subject_label *curr;
				curr = task->acl;

				task->is_writable = 0;
				/* ignore additional mmap checks for processes that are writable 
				   by the default ACL */
				obj = chk_obj_label(filp->f_dentry, filp->f_vfsmnt, default_role->root_label);
				if (unlikely(obj->mode & GR_WRITE))
					task->is_writable = 1;
				obj = chk_obj_label(filp->f_dentry, filp->f_vfsmnt, task->role->root_label);
				if (unlikely(obj->mode & GR_WRITE))
					task->is_writable = 1;

#ifdef CONFIG_GRKERNSEC_ACL_DEBUG
				printk(KERN_ALERT "gr_set_acls for (%s:%d): role:%s, subject:%s\n", task->comm, task->pid, task->role->rolename, task->acl->filename);
#endif
				if (!(curr->mode & GR_LEARN))
					for (i = 0; i < RLIM_NLIMITS; i++) {
						if (!(curr->resmask & (1 << i)))
							continue;

						task->rlim[i].rlim_cur =
						    curr->res[i].rlim_cur;
						task->rlim[i].rlim_max =
						    curr->res[i].rlim_max;
					}
			} else {
				read_unlock(&grsec_exec_file_lock);
				read_unlock(&tasklist_lock);
				security_alert_good(GR_DEFACL_MSG, task->comm,
						    task->pid);
				return 1;
			}
		} else {
			// it's a kernel process
			task->role = kernel_role;
			task->acl = kernel_role->root_label;
#ifdef CONFIG_GRKERNSEC_ACL_HIDEKERN
			task->acl->mode &= ~GR_FIND;
#endif
		}
	}
	read_unlock(&grsec_exec_file_lock);
	read_unlock(&tasklist_lock);
	return 0;
}

void
gr_learn_resource(const struct task_struct *task,
		  const int res, const unsigned long wanted, const int gt)
{
	struct acl_subject_label *acl;

	if (unlikely((gr_status & GR_READY) &&
		     task->acl && (task->acl->mode & GR_LEARN)))
		goto skip_reslog;

#ifdef CONFIG_GRKERNSEC_RESLOG
	gr_log_resource(task, res, wanted, gt);
#endif
      skip_reslog:

	if (unlikely(!(gr_status & GR_READY) || !wanted))
		return;

	acl = task->acl;

	if (likely(!acl || !(acl->mode & GR_LEARN) ||
		   !(acl->resmask & (1 << (unsigned short) res))))
		return;

	if (wanted >= acl->res[res].rlim_cur) {
		unsigned long res_add;

		res_add = wanted;
		switch (res) {
		case RLIMIT_CPU:
			res_add += GR_RLIM_CPU_BUMP;
			break;
		case RLIMIT_FSIZE:
			res_add += GR_RLIM_FSIZE_BUMP;
			break;
		case RLIMIT_DATA:
			res_add += GR_RLIM_DATA_BUMP;
			break;
		case RLIMIT_STACK:
			res_add += GR_RLIM_STACK_BUMP;
			break;
		case RLIMIT_CORE:
			res_add += GR_RLIM_CORE_BUMP;
			break;
		case RLIMIT_RSS:
			res_add += GR_RLIM_RSS_BUMP;
			break;
		case RLIMIT_NPROC:
			res_add += GR_RLIM_NPROC_BUMP;
			break;
		case RLIMIT_NOFILE:
			res_add += GR_RLIM_NOFILE_BUMP;
			break;
		case RLIMIT_MEMLOCK:
			res_add += GR_RLIM_MEMLOCK_BUMP;
			break;
		case RLIMIT_AS:
			res_add += GR_RLIM_AS_BUMP;
			break;
		case RLIMIT_LOCKS:
			res_add += GR_RLIM_LOCKS_BUMP;
			break;
		}

		acl->res[res].rlim_cur = res_add;

		if (wanted > acl->res[res].rlim_max)
			acl->res[res].rlim_max = res_add;

		security_learn(GR_LEARN_AUDIT_MSG, task->role->rolename,
			       task->role->roletype, acl->filename,
			       acl->res[res].rlim_cur, acl->res[res].rlim_max,
			       "", (unsigned long) res);
	}

	return;
}

#ifdef CONFIG_SYSCTL
extern struct proc_dir_entry *proc_sys_root;

__u32
gr_handle_sysctl(const struct ctl_table *table, const void *oldval,
		 const void *newval)
{
	struct proc_dir_entry *tmp;
	struct nameidata nd;
	const char *proc_sys = "/proc/sys";
	char *path = gr_shared_page[0][smp_processor_id()];
	struct acl_object_label *obj;
	unsigned short len = 0, pos = 0, depth = 0, i;
	__u32 err = 0;
	__u32 mode = 0;

	if (unlikely(!(gr_status & GR_READY)))
		return 1;

	if (oldval)
		mode |= GR_READ;
	if (newval)
		mode |= GR_WRITE;

	/* convert the requested sysctl entry into a pathname */

	for (tmp = table->de; tmp != proc_sys_root; tmp = tmp->parent) {
		len += strlen(tmp->name);
		len++;
		depth++;
	}

	if ((len + depth + strlen(proc_sys) + 1) > PAGE_SIZE)
		return 0;	// deny

	memset(path, 0, PAGE_SIZE);

	memcpy(path, proc_sys, strlen(proc_sys));

	pos += strlen(proc_sys);

	for (; depth > 0; depth--) {
		path[pos] = '/';
		pos++;
		for (i = 1, tmp = table->de; tmp != proc_sys_root;
		     tmp = tmp->parent) {
			if (depth == i) {
				memcpy(path + pos, tmp->name,
				       strlen(tmp->name));
				pos += strlen(tmp->name);
			}
			i++;
		}
	}

	if (path_init(path, LOOKUP_FOLLOW, &nd))
		err = path_walk(path, &nd);

	if (err)
		goto out;

	obj = chk_obj_label(nd.dentry, nd.mnt, current->acl);
	err = obj->mode & (mode | to_gr_audit(mode) | GR_SUPPRESS);

	if (unlikely((current->acl->mode & GR_LEARN) && ((err & mode) != mode))) {
		__u32 new_mode = mode;

		new_mode &= ~(GR_AUDITS | GR_SUPPRESS);

		err = new_mode;
		gr_log_learn(current->role, current->uid, current->gid,
			     current, path, new_mode);
	} else if ((err & mode) != mode && !(err & GR_SUPPRESS)) {
		security_alert(GR_SYSCTL_ACL_MSG, "denied", path,
			       (mode & GR_READ) ? " reading" : "",
			       (mode & GR_WRITE) ? " writing" : "",
			       DEFAULTSECARGS);
		err = 0;
	} else if ((err & mode) != mode) {
		err = 0;
	} else if (((err & mode) == mode) && (err & GR_AUDITS)) {
		security_audit(GR_SYSCTL_ACL_MSG, "successful",
			       path, (mode & GR_READ) ? " reading" : "",
			       (mode & GR_WRITE) ? " writing" : "",
			       DEFAULTSECARGS);
	}

	path_release(&nd);

      out:
	return err;
}
#endif

int
gr_handle_proc_ptrace(struct task_struct *task)
{
	struct file *filp;
	struct task_struct *tmp = task;
	struct task_struct *curtemp = current;
	__u32 retmode;

	if (unlikely(!(gr_status & GR_READY)))
		return 0;

	read_lock(&tasklist_lock);
	read_lock(&grsec_exec_file_lock);
	filp = task->exec_file;

	while (tmp->pid > 0) {
		if (tmp == curtemp)
			break;
		tmp = tmp->p_pptr;
	}

	if (!filp || (tmp->pid == 0 && !(current->acl->mode & GR_RELAXPTRACE))) {
		read_unlock(&grsec_exec_file_lock);
		read_unlock(&tasklist_lock);
		return 1;
	}

	retmode = gr_search_file(filp->f_dentry, GR_NOPTRACE, filp->f_vfsmnt);
	read_unlock(&grsec_exec_file_lock);
	read_unlock(&tasklist_lock);

	if (retmode & GR_NOPTRACE)
		return 1;

	if (!(current->acl->mode & GR_OVERRIDE) && !(current->role->roletype & GR_ROLE_GOD)
	    && (current->acl != task->acl || (current->acl != current->role->root_label
	    && current->pid != task->pid)))
		return 1;

	return 0;
}

int
gr_handle_ptrace(struct task_struct *task, const long request)
{
	struct task_struct *tmp = task;
	struct task_struct *curtemp = current;
	__u32 retmode;

	if (unlikely(!(gr_status & GR_READY)))
		return 0;

	read_lock(&tasklist_lock);
	while (tmp->pid > 0) {
		if (tmp == curtemp)
			break;
		tmp = tmp->p_pptr;
	}
	read_unlock(&tasklist_lock);

	if (tmp->pid == 0 && !(current->acl->mode & GR_RELAXPTRACE)) {
		security_alert(GR_PTRACE_ACL_MSG, task->exec_file ? 
			       gr_to_filename(task->exec_file->f_dentry, task->exec_file->f_vfsmnt)
			       : "(none)", task->comm, task->pid, 
			       DEFAULTSECARGS);
		return 1;
	}

	read_lock(&grsec_exec_file_lock);
	if (unlikely(!task->exec_file)) {
		read_unlock(&grsec_exec_file_lock);
		return 0;
	}

	retmode = gr_search_file(task->exec_file->f_dentry, GR_PTRACERD | GR_NOPTRACE, task->exec_file->f_vfsmnt);
	read_unlock(&grsec_exec_file_lock);

	if (retmode & GR_NOPTRACE) {
		security_alert(GR_PTRACE_ACL_MSG, gr_to_filename(task->exec_file->f_dentry, task->exec_file->f_vfsmnt),
			       task->comm, task->pid, DEFAULTSECARGS);
		return 1;
	}
		
	if (retmode & GR_PTRACERD) {
		switch (request) {
		case PTRACE_POKETEXT:
		case PTRACE_POKEDATA:
		case PTRACE_POKEUSR:
#if !defined(CONFIG_PPC32) && !defined(CONFIG_PARISC) && !defined(CONFIG_ALPHA)
		case PTRACE_SETREGS:
		case PTRACE_SETFPREGS:
#endif
#ifdef CONFIG_X86
		case PTRACE_SETFPXREGS:
#endif
#ifdef CONFIG_ALTIVEC
		case PTRACE_SETVRREGS:
#endif
			return 1;
		default:
			return 0;
		}
	} else if (!(current->acl->mode & GR_OVERRIDE) &&
		   !(current->role->roletype & GR_ROLE_GOD) &&
		   (current->acl != task->acl)) {
		security_alert(GR_PTRACE_ACL_MSG,
			       gr_to_filename(task->exec_file->f_dentry, task->exec_file->f_vfsmnt),
					      task->comm, task->pid, DEFAULTSECARGS);
		return 1;
	}

	return 0;
}

int
gr_handle_mmap(const struct file *filp, const unsigned long prot)
{
	struct acl_object_label *obj, *obj2;

	if (unlikely(!(gr_status & GR_READY) ||
		     (current->acl->mode & GR_OVERRIDE) || !filp ||
		     !(prot & PROT_EXEC)))
		return 0;

	if (unlikely(current->is_writable))
		return 0;

	obj = chk_obj_label(filp->f_dentry, filp->f_vfsmnt, default_role->root_label);
	obj2 = chk_obj_label(filp->f_dentry, filp->f_vfsmnt,
			     current->role->root_label);
	if (unlikely((obj->mode & GR_WRITE) || (obj2->mode & GR_WRITE))) {
		security_alert(GR_WRITLIB_ACL_MSG,
			       gr_to_filename(filp->f_dentry, filp->f_vfsmnt),
			       DEFAULTSECARGS);
		return 1;
	}

	return 0;
}

int
gr_acl_handle_mmap(const struct file *file, const unsigned long prot)
{
	__u32 mode;

	if (unlikely(!file || !(prot & PROT_EXEC)))
		return 1;

	mode =
	    gr_search_file(file->f_dentry,
			   GR_EXEC | GR_AUDIT_EXEC | GR_SUPPRESS,
			   file->f_vfsmnt);

	if (unlikely(!gr_tpe_allow(file) || (!(mode & GR_EXEC) && !(mode & GR_SUPPRESS)))) {
		security_alert(GR_MMAP_ACL_MSG, "denied",
			       gr_to_filename(file->f_dentry, file->f_vfsmnt),
			       DEFAULTSECARGS);
		return 0;
	} else if (unlikely(!gr_tpe_allow(file) || !(mode & GR_EXEC))) {
		return 0;
	} else if (unlikely(mode & GR_EXEC && mode & GR_AUDIT_EXEC)) {
		security_audit(GR_MMAP_ACL_MSG, "successful",
			       gr_to_filename(file->f_dentry, file->f_vfsmnt),
			       DEFAULTSECARGS);
		return 1;
	}

	return 1;
}

int
gr_acl_handle_mprotect(const struct file *file, const unsigned long prot)
{
	__u32 mode;

	if (unlikely(!file || !(prot & PROT_EXEC)))
		return 1;

	mode =
	    gr_search_file(file->f_dentry,
			   GR_EXEC | GR_AUDIT_EXEC | GR_SUPPRESS,
			   file->f_vfsmnt);

	if (unlikely(!gr_tpe_allow(file) || (!(mode & GR_EXEC) && !(mode & GR_SUPPRESS)))) {
		security_alert(GR_MPROTECT_ACL_MSG, "denied",
			       gr_to_filename(file->f_dentry, file->f_vfsmnt),
			       DEFAULTSECARGS);
		return 0;
	} else if (unlikely(!gr_tpe_allow(file) || !(mode & GR_EXEC))) {
		return 0;
	} else if (unlikely(mode & GR_EXEC && mode & GR_AUDIT_EXEC)) {
		security_audit(GR_MPROTECT_ACL_MSG, "successful",
			       gr_to_filename(file->f_dentry, file->f_vfsmnt),
			       DEFAULTSECARGS);
		return 1;
	}

	return 1;
}

void
gr_acl_handle_psacct(struct task_struct *task, const long code)
{
	unsigned long runtime;
	unsigned long cputime;
	unsigned int wday, cday;
	__u8 whr, chr;
	__u8 wmin, cmin;
	__u8 wsec, csec;
	char cur_tty[64] = { 0 };
	char parent_tty[64] = { 0 };

	if (unlikely(!(gr_status & GR_READY) || !task->acl ||
		     !(task->acl->mode & GR_PROCACCT)))
		return;

	runtime = (jiffies - task->start_time) / HZ;
	wday = runtime / (3600 * 24);
	runtime -= wday * (3600 * 24);
	whr = runtime / 3600;
	runtime -= whr * 3600;
	wmin = runtime / 60;
	runtime -= wmin * 60;
	wsec = runtime;

	cputime = (task->times.tms_utime + task->times.tms_stime) / HZ;
	cday = cputime / (3600 * 24);
	cputime -= cday * (3600 * 24);
	chr = cputime / 3600;
	cputime -= chr * 3600;
	cmin = cputime / 60;
	cputime -= cmin * 60;
	csec = cputime;

	security_audit(GR_ACL_PROCACCT_MSG, gr_task_fullpath(task), task->comm,
		       task->pid, NIPQUAD(task->curr_ip), tty_name(task->tty,
								   cur_tty),
		       task->uid, task->euid, task->gid, task->egid, wday, whr,
		       wmin, wsec, cday, chr, cmin, csec,
		       (task->
			flags & PF_SIGNALED) ? "killed by signal" : "exited",
		       code, gr_parent_task_fullpath(task), 
		       task->p_pptr->comm, task->p_pptr->pid,
		       NIPQUAD(task->p_pptr->curr_ip),
		       tty_name(task->p_pptr->tty, parent_tty),
		       task->p_pptr->uid, task->p_pptr->euid, task->p_pptr->gid,
		       task->p_pptr->egid);

	return;
}

void gr_set_kernel_label(struct task_struct *task)
{
	if (gr_status & GR_READY) {
		task->role = kernel_role;
		task->acl = kernel_role->root_label;
	}
	return;
}
