#define UwsEvent_c

// **********************************************************************
// * uwsevent.c                                                         *
// *                                                                    *
// * Written by Matthew DeLoera (deloera@us.ibm.com)                    *
// * (C) Copyright IBM Corporation, 1998-2001                           *
// *                                                                    *
// *  This software may be used and distributed according to the terms  *
// *  of the GNU Public License, incorporated herein by reference.      *
// **********************************************************************

// **********************************************************************
// * uwsevent.c - This source file provides all routines for event      *
// *              processing.                                           *
// *                                                                    *
// * Exported Functions                                                 *
// * ------------------                                                 *
// * sysSpRegisterEvent - blocking routine called by IOCtl handler in   *
// *                      response to a call to RegisterForEvents(),    *
// *                      which blocks the caller until an event has    *
// *                      been received, then copies the event's data   *
// *                      into the caller's data buffer                 *
// * sysSpRunEvent      - called by IRQ handler when an event message   *
// *                      received, so that when interrupt time ends,   *
// *                      all threads blocked on event registration can *
// *                      continue                                      *
// *                                                                    *
// * Changes                                                            *
// * ------------------------------------------------------------------ *
// * 01/29/2001 -            - Cleaned up for open source release.      *
// * ------------------------------------------------------------------ *
// * 06/04/2001 - PHR_162848 - Removed uresolved reference to           *
// *                           current->signal                          *
// * ------------------------------------------------------------------ *
// * 07/18/2002 - PHR_MULTI_4 - Added Multi Node support                *
// **********************************************************************


// ************
// * INCLUDES *
// ************
#include "uwscmd.h"

// ***********************
// * EXTERNAL PROTOTYPES *
// ***********************
extern void        sysSpDisplayQueue(void *pHead);
extern PWORK_BLOCK sysSpEnqueue(PDRIVER_INFO pDriverNode, void *pHead, PWORK_BLOCK pBlock);
extern PWORK_BLOCK sysSpDequeue(PDRIVER_INFO pDriverNode, void *pHead);

// ********************
// * LOCAL PROTOTYPES *
// ********************
void sysSpRegisterEvent(PDRIVER_INFO pDriverNode, PEVENT_BLOCK pEventBlk, PUSHORT pIoctlMagicNumber);


// ***********************************************************************************
// * sysSpRegisterEvent - This function is called when an application registers for  *
// *                      event notification. It then uses MagicNumber to determine  *
// *                      whether a new event has somehow managed to be received     *
// *                      in the brief time since the register call was made.        *
// *                      If so, that event's data is returned to the app, otherwise *
// *                      we block for a new event to arrive.                        *
// *                                                                                 *
// * Parameters:                                                                     *
// *    pEventBlk - Pointer to the event work block                                  *
// *    pIoctlMagicNumber - Pointer to the application's magic number                *
// ***********************************************************************************
void sysSpRegisterEvent(PDRIVER_INFO pDriverNode, PEVENT_BLOCK pEventBlk, PUSHORT pIoctlMagicNumber)
{
   unsigned long flags = 0;

   spin_lock_irqsave( &(pDriverNode->EventQueueLock), flags );

   // IF THIS IS THE FIRST REGISTER CALL FOR THIS PROCESSOR,
   // UPDATE THIS REGISTER CALL TO THE CURRENT EVENT INDEX
   if (*pIoctlMagicNumber == MAGIC_NUMBER_INIT)
   {
      *pIoctlMagicNumber = (USHORT)pDriverNode->MagicNumber;
   }

   // IF AN EVENT WAS MISSED SINCE REGISTRATION, RETURN THAT EVENT TO THE CALLER
   if (*pIoctlMagicNumber == pDriverNode->MagicNumber)
   {
      // ENQUEUE THIS REGISTRATION CALL ONTO THE EVENT QUEUE AND MARK AS PENDING
      sysSpEnqueue(pDriverNode, pDriverNode->EventQueue, pEventBlk);

      pEventBlk->completionCode = DDERR_EV_PENDING;

      init_waitqueue_head(&(pEventBlk->pBlock));

      spin_unlock_irqrestore( &(pDriverNode->EventQueueLock), flags );

      if (DBG_EVENTS)
      {
         printk(KERN_CRIT "ibmasm: DBG_EVENTS - Device driver blocking until an event occurs.\n");
      }

      // BLOCK UTIL EVENT IS RECEIVED OR INTERRUPTED BY A SIGNAL
      interruptible_sleep_on(&pEventBlk->pBlock);

      // IF THIS WAS UNBLOCKED BY A SIGNAL - RETURN TO THE CALLER 
      if (signal_pending(current))
      {
         pEventBlk->completionCode = DDERR_INTERRUPTED;
      }
      else
      {
         spin_lock_irqsave( &(pDriverNode->EventQueueLock), flags );
      }
   }
   else
   {
      // MARK THIS REQUEST AS SUCCESSFUL
      pEventBlk->completionCode = DDERR_SUCCESS;

      if (DBG_EVENTS)
      {
          printk(KERN_CRIT "ibmasm: DBG_EVENTS - Returning missed events to the caller, Blocking will not occur!\n");
      }
   }
  
   // DETERMINE WHETHER TO SEND EVENT DATA TO THE CALLING APPLICATION
   if (pEventBlk->completionCode == DDERR_SUCCESS)
   {
      int i, index=0;
      USHORT NxEventNum;
      int CopyRC;

      // SELECT THE APPROPRIATE EVENT BUFFER
      for (i = 0, NxEventNum = EV_MAXNUMBER; i < EV_BUFFERS; i++)
      {
         if ((pDriverNode->EvRcvd[i].EvNumber > *pIoctlMagicNumber) && 
             (pDriverNode->EvRcvd[i].EvNumber <= NxEventNum)        )
         {
              NxEventNum = pDriverNode->EvRcvd[i].EvNumber;
              index = i;
         }
      }

      // COPY THE EVENT DATA INTO THE IRP BUFFER AND UPDATE THE MAGIC NUMBER
      if (DBG_EVENTS)
      {
         printk(KERN_CRIT "ibmasm: DBG_EVENTS - Copying event data into callers buffer.\n");
      }

      CopyRC = 0;

      CopyRC = !access_ok(VERIFY_WRITE, pEventBlk->pBuffer, pDriverNode->EvRcvd[index].EvLength); 

      if (CopyRC)
      {
         if (DBG_EVENTS)
         {
            printk(KERN_CRIT "ibmasm: DBG_EVENTS - access_ok failed, Device returned more data than the callers buffer can handle.\n");
         }

         pEventBlk->completionCode= DDERR_ZALLOC_EVENT;
      }

      CopyRC = copy_to_user(pEventBlk->pBuffer, pDriverNode->EvRcvd[index].pEvBuffer, pDriverNode->EvRcvd[index].EvLength);

      if (CopyRC)
      {
         if (DBG_EVENTS)
         {
            printk(KERN_CRIT "ibmasm: DBG_EVENTS - Copy_to_user failed. Event failed.\n");
         }

         pEventBlk->completionCode= DDERR_ZALLOC_EVENT;
      }

      *pIoctlMagicNumber = NxEventNum;
   }

   if (pEventBlk->completionCode != DDERR_INTERRUPTED)
   {
     spin_unlock_irqrestore( &(pDriverNode->EventQueueLock), flags );
   }

   return;
}


// ***********************************************************************************
// * sysSpRunEvent - This function releases all blocked event registration threads,  *
// *                 and dequeues each node off the event queue. This is *only*      *
// *                 called by the interrupt handler, so there's no need to lock the *
// *                 event queue (here we're running at interrupt time).             *
// ***********************************************************************************
void sysSpRunEvent(PDRIVER_INFO pDriverNode)
{
   PEVENT_BLOCK pEventBlk;

   // DEQUEUE EACH REQUEST FROM THE EVENT QUEUE
   while ( (pEventBlk = sysSpDequeue(pDriverNode, pDriverNode->EventQueue)) )
   {
      // SET THIS REQUEST'S COMPLETION CODE TO SUCCESSFUL AND UNBLOCK THE ASSOCIATED PID
      pEventBlk->completionCode = DDERR_SUCCESS;
      wake_up_interruptible(&pEventBlk->pBlock);
   }
}


#undef UwsEvent_c


