/******************************************************************************
 *
 * Name:	skaddr.c
 * Project:	SysKonnect SK-9Dxx Gigabit Ethernet
 * Version:	$Revision: 1.4 $
 * Date:	$Date: 2001/09/20 12:08:19 $
 * Purpose:	Manage Addresses (Multicast and Unicast) and Promiscuous Mode.
 *
 ******************************************************************************/

/******************************************************************************
 *
 *	(C)Copyright 2001 SysKonnect GmbH.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	The information in this file is provided "AS IS" without warranty.
 *
 ******************************************************************************/

/******************************************************************************
 *
 * History:
 *
 *	$Log: skaddr.c,v $
 *	Revision 1.4  2001/09/20 12:08:19  mmoser
 *	typecasts for SK_U32 to SK_U8
 *	
 *	Revision 1.3  2001/08/08 13:27:27  rschmidt
 *	Editorial changes.
 *	
 *	Revision 1.2  2001/06/05 15:08:12  rassmann
 *	Editorial changes.
 *	
 *	Revision 1.1  2001/06/05 08:28:23  rassmann
 *	First public version.
 *	
 *
 ******************************************************************************/

/******************************************************************************
 *
 * Description:
 *
 * This module is intended to manage multicast addresses, address override,
 * and promiscuous mode on SysKonnect SK-9Dxx Gigabit Ethernet adapters.
 *
 * Include File Hierarchy:
 *
 *	"skdrv1st.h"
 *	"skdrv2nd.h"
 *
 ******************************************************************************/

#ifndef	lint
static const char SysKonnectFileId[] =
	"@(#) $Id: skaddr.c,v 1.4 2001/09/20 12:08:19 mmoser Exp $ (C) SysKonnect.";
#endif	/* !defined(lint) */

#define __SKADDR_C

#ifdef __cplusplus
#error C++ is not yet supported.
extern "C" {
#endif	/* cplusplus */

#include "h/skdrv1st.h"
#include "h/skdrv2nd.h"

/* defines ********************************************************************/

#define CRC32_POLY	0xEDB88320UL	/* CRC32-Poly - XMAC: Little Endian */
#define HASH_BITS	7				/* #bits in hash */
#define	SK_MC_BIT	0x01

/* Error numbers and messages. */

#define SKERR_ADDR_E001		(SK_ERRBASE_ADDR + 0)
#define SKERR_ADDR_E001MSG	"Bad Flags."
#define SKERR_ADDR_E002		(SKERR_ADDR_E001 + 1)
#define SKERR_ADDR_E002MSG	"New Error."

/* typedefs *******************************************************************/

/* None. */

/* global variables ***********************************************************/

/* hash values with all bits set. */

SK_U32	OnesHash[4] = {0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF};

/* local variables ************************************************************/

/* functions ******************************************************************/

/******************************************************************************
 *
 *	SkAddrInit - initialize data, set state to init
 *
 * Description:
 *
 *	SK_INIT_DATA
 *	============
 *
 *	This routine clears the multicast tables and resets promiscuous mode.
 *	Some entries are reserved for the "logical MAC address", the
 *	SK-RLMT multicast address, and the BPDU multicast address.
 *
 *
 *	SK_INIT_IO
 *	==========
 *
 *	All permanent MAC addresses are read from EPROM.
 *	If the current MAC addresses are not already set in software,
 *	they are set to the values of the permanent addresses.
 *	The current addresses are written to the corresponding XMAC.
 *
 *
 *	SK_INIT_RUN
 *	===========
 *
 *	Update multicast filter and MAC addresses.
 *
 * Context:
 *	init, pageable
 *
 * Returns:	N/A
 *
 */
void	SkAddrInit(
SK_AC	*pAC,	/* the adapter context */
SK_IOC	IoC,	/* I/O context */
int		Level)	/* initialization level */
{
	SK_ADDR_PORT	*pAPort;
	SK_U32			Data32;

	switch (Level) {
	case SK_INIT_DATA:
		SK_MEMSET((char *)&pAC->Addr, 0, sizeof(SK_ADDR));
		pAC->Addr.Port.RxMode = RX_MAC_ENAB | RX_ENAB_FLOW_CTRL;
		pAC->Addr.Port.PromMode = SK_PROM_MODE_NONE;
		break;

	case SK_INIT_IO:
		/* Get the MAC address. */
		SK_IN32(IoC, MAC_ADDR1_HIGH, &Data32);
		pAC->Addr.Net.PermanentMacAddress.a[0] = (SK_U8)((Data32 >> 8) & 0xff);
		pAC->Addr.Net.PermanentMacAddress.a[1] = (SK_U8)((Data32 >> 0) & 0xff);
		
		SK_IN32(IoC, MAC_ADDR1_LOW, &Data32);
		pAC->Addr.Net.PermanentMacAddress.a[2] = (SK_U8)((Data32 >> 24) & 0xff);
		pAC->Addr.Net.PermanentMacAddress.a[3] = (SK_U8)((Data32 >> 16) & 0xff);
		pAC->Addr.Net.PermanentMacAddress.a[4] = (SK_U8)((Data32 >>  8) & 0xff);
		pAC->Addr.Net.PermanentMacAddress.a[5] = (SK_U8)((Data32 >>  0) & 0xff);

		if (!pAC->Addr.Net.CurrentMacAddressSet) {
			/* Set the current logical MAC address to the permanent one. */
			pAC->Addr.Net.CurrentMacAddress =
				pAC->Addr.Net.PermanentMacAddress;
			pAC->Addr.Net.CurrentMacAddressSet = SK_TRUE;
		}

		pAPort = &pAC->Addr.Port;
		/* Set port address. */
		pAC->Addr.Port.PermanentMacAddress = pAC->Addr.Net.PermanentMacAddress;
		pAC->Addr.Port.PermanentMacAddress.a[5] |= 1;

		if (!pAPort->CurrentMacAddressSet) {
			/*
			 * Set the current and previous physical MAC address
			 * of this port to its permanent MAC address.
			 */
			pAPort->CurrentMacAddress = pAPort->PermanentMacAddress;
			pAPort->PreviousMacAddress = pAPort->PermanentMacAddress;
			pAPort->CurrentMacAddressSet = SK_TRUE;
		}
		break;

	case SK_INIT_RUN:
		(void)SkAddrMcUpdate(pAC, IoC);
		break;

	default:	/* error */
		break;
	}
}	/* SkAddrInit */


/******************************************************************************
 *
 *	SkAddrMcClear - clear the multicast table
 *
 * Description:
 *	This routine clears the multicast table (InexactFilter) of the given port.
 *	If not suppressed by Flag SK_MC_SW_ONLY, the hardware is updated
 *	immediately.
 *
 * Context:
 *	runtime, pageable
 *	may be called starting with SK_INIT_DATA with flag SK_MC_SW_ONLY
 *	may be called after SK_INIT_IO without limitation
 *
 * Returns:
 *	SK_ADDR_SUCCESS
 *	SK_ADDR_ILLEGAL_PORT
 */
int	SkAddrMcClear(
SK_AC	*pAC,		/* adapter context */
SK_IOC	IoC,		/* I/O context */
int		Flags)		/* permanent/non-perm, sw-only */
{
	int i;

	/* Clear InexactFilter. */
	for (i = 0; i < SK_HASH_BYTES / 4; i++) {
		pAC->Addr.Port.InexactFilter.Words[i] = 0;
	}

	if (!(Flags & SK_MC_SW_ONLY)) {
		(void)SkAddrMcUpdate(pAC, IoC);
	}

	return (SK_ADDR_SUCCESS);
}	/* SkAddrMcClear */


/*****************************************************************************
 *
 *	SkCrc32McHash - hash multicast address
 *
 * Description:
 *	This routine computes the hash value for a multicast address.
 *
 * Context:
 *	runtime, pageable
 *
 * Returns:
 *	Hash value of multicast address.
 */
static unsigned long SkCrc32McHash(
SK_U8	*pBuffer) {
	SK_U32	Crc;
	SK_U32	Idx;
	SK_U32	Bit;
	
	Crc = 0xFFFFFFFFUL;
	for (Idx = 0; Idx < SK_MAC_ADDR_LEN; Idx++) {
		Crc ^= *pBuffer++;
		for (Bit = 0; Bit < 8; Bit++) {
			Crc =  (Crc >> 1) ^ ((Crc & 1) ? CRC32_POLY : 0);
		}
	}
	return (Crc & ((1 << HASH_BITS) - 1));
} /* SkCrc32McHash */


/******************************************************************************
 *
 *	SkAddrMcAdd - add a multicast address to a port
 *
 * Description:
 *	This routine enables reception for a given address on the given port.
 *
 * Notes:
 *	The return code is only valid for SK_PROM_MODE_NONE.
 *
 *	In the current version, only RLMT may add addresses to the non-active
 *	port.
 *
 *	The multicast bit is only checked if there are no free exact match
 *	entries.
 *
 * Context:
 *	runtime, pageable
 *	may be called after SK_INIT_DATA
 *
 * Returns:
 *	SK_MC_FILTERING_EXACT
 *	SK_MC_FILTERING_INEXACT
 *	SK_MC_ILLEGAL_ADDRESS
 *	SK_MC_ILLEGAL_PORT
 *	SK_MC_RLMT_OVERFLOW
 */
int	SkAddrMcAdd(
SK_AC		*pAC,		/* adapter context */
SK_IOC		IoC,		/* I/O context */
SK_MAC_ADDR	*pMc,		/* multicast address to be added */
int			Flags)		/* permanent/non-permanent */
{
	unsigned	HashBit;

	if ((pMc->a[0] & SK_MC_BIT) == 0) {
		/* Hashing only possible with multicast addresses. */
		return (SK_MC_ILLEGAL_ADDRESS);
	}

	HashBit = SkCrc32McHash(&pMc->a[0]);

	/* Enable the multicast bit. */
	pAC->Addr.Port.InexactFilter.Words[HashBit / 32] |= 1 << (HashBit % 32);

	return (SK_MC_FILTERING_INEXACT);
}	/* SkAddrMcAdd */


/******************************************************************************
 *
 *	SkAddrMcUpdate - update the HW MC address table and set the MAC address
 *
 * Description:
 *	This routine enables reception of the addresses contained in a local
 *	table for a given port.
 *	It also programs the current MAC addresses.
 *
 * Notes:
 *	The return code is only valid for SK_PROM_MODE_NONE.
 *
 * Context:
 *	runtime, pageable
 *	may be called after SK_INIT_IO
 *
 * Returns:
 *	SK_MC_FILTERING_EXACT
 *	SK_MC_FILTERING_INEXACT
 *	SK_ADDR_ILLEGAL_PORT
 */
int	SkAddrMcUpdate(
SK_AC	*pAC,		/* adapter context */
SK_IOC	IoC)		/* I/O context */
{
	SK_ADDR_PORT	*pAPort;
	int				i;
	SK_U32			Data32;
	SK_U32			Inexact;

	SK_DBG_MSG(pAC, SK_DBGMOD_ADDR, SK_DBGCAT_CTRL, ("SkAddrMcUpdate.\n"));

	pAPort = &pAC->Addr.Port;

	SK_OUT32(IoC, RECEIVE_MODE, pAPort->RxMode);

	if (pAPort->PromMode == SK_PROM_MODE_NONE) {
		/* Set hash register to InexactFilter. */
		SK_OUTHASH(IoC, &pAPort->InexactFilter.Words[0]);
	}
	else {
		(void)SkAddrPromiscuousChange(pAC, IoC, pAPort->PromMode);
	}

	/* Set port's current logical MAC address. */
	Data32 =
		(pAC->Addr.Net.CurrentMacAddress.a[0] << 8) |
		(pAC->Addr.Net.CurrentMacAddress.a[1]);
	SK_OUT32(IoC, MAC_ADDR1_HIGH, Data32);

	Data32 =
		(pAC->Addr.Net.CurrentMacAddress.a[2] << 24) |
		(pAC->Addr.Net.CurrentMacAddress.a[3] << 16) |
		(pAC->Addr.Net.CurrentMacAddress.a[4] <<  8) |
		(pAC->Addr.Net.CurrentMacAddress.a[5]);
	SK_OUT32(IoC, MAC_ADDR1_LOW, Data32);

	/* Set port's current physical MAC address. */
	Data32 =
		(pAPort->CurrentMacAddress.a[0] << 8) |
		(pAPort->CurrentMacAddress.a[1]);
	SK_OUT32(IoC, MAC_ADDR2_HIGH, Data32);

	Data32 =
		(pAPort->CurrentMacAddress.a[2] << 24) |
		(pAPort->CurrentMacAddress.a[3] << 16) |
		(pAPort->CurrentMacAddress.a[4] <<  8) |
		(pAPort->CurrentMacAddress.a[5]);
	SK_OUT32(IoC, MAC_ADDR2_LOW, Data32);

	/* Delete the other addresses. */
	SK_OUT32(IoC, MAC_ADDR3_HIGH, 0);
	SK_OUT32(IoC, MAC_ADDR3_LOW, 0);
	SK_OUT32(IoC, MAC_ADDR4_HIGH, 0);
	SK_OUT32(IoC, MAC_ADDR4_LOW, 0);

	/* Receive broadcasts. */
	SK_OUT32(IoC, RX_RULES_CTRL0, 0xc2000000 & 0x7fffffff);
	SK_OUT32(IoC, RX_RULES_MASK_VAL0, 0xffffffff & 0x7fffffff);
	SK_OUT32(IoC, RX_RULES_CTRL1, 0x86000004 & 0x7fffffff);
	SK_OUT32(IoC, RX_RULES_MASK_VAL1, 0xffffffff & 0x7fffffff);

	/* Disable the rest of the rules. */
	for (i = 2; i < 16; i++) {
		SK_OUT32(IoC, RX_RULES_CTRL0 + i * 8, 0);
		SK_OUT32(IoC, RX_RULES_MASK_VAL0 + i * 8, 0);
	}

	/* Determine return value. */
	Inexact = 0;
	for (i = 0; i < SK_HASH_BYTES / 4; i++) {
		Inexact |= pAPort->InexactFilter.Words[i];
	}

	if (Inexact == 0 && pAPort->PromMode == 0) {
		return (SK_MC_FILTERING_EXACT);
	}
	else {
		return (SK_MC_FILTERING_INEXACT);
	}
}	/* SkAddrMcUpdate */


/******************************************************************************
 *
 *	SkAddrOverride - override a port's MAC address
 *
 * Description:
 *	This routine overrides the MAC address of one port.
 *
 * Context:
 *	runtime, pageable
 *	may be called after SK_INIT_IO
 *
 * Returns:
 *	SK_ADDR_SUCCESS if successful.
 *	SK_ADDR_DUPLICATE_ADDRESS if duplicate MAC address.
 *	SK_ADDR_MULTICAST_ADDRESS if multicast or broadcast address.
 *	SK_ADDR_TOO_EARLY if SK_INIT_IO was not executed before.
 */
int	SkAddrOverride(
SK_AC		*pAC,		/* adapter context */
SK_IOC		IoC,		/* I/O context */
SK_MAC_ADDR	*pNewAddr,	/* new MAC address */
int			Flags)		/* logical/physical MAC address */
{
//RARA	SK_EVPARA	Para;
	SK_U32	Data32;

	if (pNewAddr != NULL && (pNewAddr->a[0] & SK_MC_BIT) != 0) {
		return (SK_ADDR_MULTICAST_ADDRESS);
	}

	if (!pAC->Addr.Net.CurrentMacAddressSet) {
		return (SK_ADDR_TOO_EARLY);
	}

	if (Flags & SK_ADDR_PHYSICAL_ADDRESS) {	/* Physical MAC address. */
		if (SK_ADDR_EQUAL(pNewAddr->a, pAC->Addr.Net.CurrentMacAddress.a)) {
			return (SK_ADDR_DUPLICATE_ADDRESS);
		}

		if (!pAC->Addr.Port.CurrentMacAddressSet) {
			return (SK_ADDR_TOO_EARLY);
		}

		if (SK_ADDR_EQUAL(pNewAddr->a, pAC->Addr.Port.CurrentMacAddress.a)) {
			return (SK_ADDR_SUCCESS);
		}

		pAC->Addr.Port.PreviousMacAddress = pAC->Addr.Port.CurrentMacAddress;
		pAC->Addr.Port.CurrentMacAddress = *pNewAddr;

#if 0
//RARA
		/* Report address change to RLMT. */
		Para.Para32[0] = 0;
		Para.Para32[0] = -1;
		SkEventQueue(pAC, SKGE_RLMT, SK_RLMT_PORT_ADDR, Para);
#endif	/* 0 */

		/* Change physical MAC address in HW. */
		Data32 = (pNewAddr->a[0] << 8) | (pNewAddr->a[1]);
		SK_OUT32(IoC, MAC_ADDR2_HIGH, Data32);

		Data32 = (pNewAddr->a[2] << 24) | (pNewAddr->a[3] << 16) |
				 (pNewAddr->a[4] <<  8) | (pNewAddr->a[5]);
		SK_OUT32(IoC, MAC_ADDR2_LOW, Data32);
	}
	else {	/* Logical MAC address. */
		if (!pAC->Addr.Port.CurrentMacAddressSet) {
			return (SK_ADDR_TOO_EARLY);
		}

		if (SK_ADDR_EQUAL(pNewAddr->a, pAC->Addr.Port.CurrentMacAddress.a)) {
			return (SK_ADDR_DUPLICATE_ADDRESS);
		}

		pAC->Addr.Net.CurrentMacAddress = *pNewAddr;

		/* Change logical MAC address in HW. */
		Data32 = (pNewAddr->a[0] << 8) | (pNewAddr->a[1]);
		SK_OUT32(IoC, MAC_ADDR1_HIGH, Data32);

		Data32 = (pNewAddr->a[2] << 24) | (pNewAddr->a[3] << 16) |
				 (pNewAddr->a[4] <<  8) | (pNewAddr->a[5]);
		SK_OUT32(IoC, MAC_ADDR1_LOW, Data32);
	}

	return (SK_ADDR_SUCCESS);
}	/* SkAddrOverride */


/******************************************************************************
 *
 *	SkAddrPromiscuousChange - set promiscuous mode for given port
 *
 * Description:
 *	This routine manages promiscuous mode:
 *	- none
 *	- all LLC frames
 *	- all MC frames
 *
 * Context:
 *	runtime, pageable
 *	may be called after SK_INIT_IO
 *
 * Returns:
 *	SK_ADDR_SUCCESS
 *	SK_ADDR_ILLEGAL_PORT
 */
int	SkAddrPromiscuousChange(
SK_AC	*pAC,			/* adapter context */
SK_IOC	IoC,			/* I/O context */
int		NewPromMode)	/* new promiscuous mode */
{
	int		i;
	int		CurPromMode;
	SK_HASH	HwInexactFilter;
	SK_U32	HwInexact;
	SK_U32	Inexact;
	SK_U32	Mode;

	Inexact = 0xFFFFFFFF;
	for (i = 0; i < SK_HASH_BYTES / 4; i++) {
		Inexact &= pAC->Addr.Port.InexactFilter.Words[i];
	}

	if (Inexact == 0xFFFFFFFF) {
		NewPromMode |= (pAC->Addr.Port.PromMode & SK_PROM_MODE_ALL_MC);
	}

	pAC->Addr.Port.PromMode = NewPromMode;

	CurPromMode = SK_PROM_MODE_NONE;

	/* Read CurPromMode from Hardware. */
	SK_IN32(IoC, RECEIVE_MODE, &Mode);
	if ((Mode & PROMISC_MODE) != 0) {
		CurPromMode |= SK_PROM_MODE_LLC;
	}

	/* Read hash register from HW. */
	SK_INHASH(IoC, &HwInexactFilter.Words[0]);

	HwInexact = 0xFFFFFFFF;
	for (i = 0; i < SK_HASH_BYTES / 4; i++) {
		HwInexact &= HwInexactFilter.Words[i];
	}

	if (HwInexact == 0xFFFFFFFF) {
		CurPromMode |= SK_PROM_MODE_ALL_MC;
	}

	if (NewPromMode == CurPromMode) {
		return (SK_ADDR_SUCCESS);
	}

	if ((NewPromMode & SK_PROM_MODE_ALL_MC) != 0 &&
		(CurPromMode & SK_PROM_MODE_ALL_MC) == 0) {	/* All MC. */

		/* Set all bits in hash register. */
		SK_OUTHASH(IoC, &OnesHash);
	}
	else if ((CurPromMode & SK_PROM_MODE_ALL_MC) != 0 &&
		(NewPromMode & SK_PROM_MODE_ALL_MC) == 0) {	/* Norm MC. */

		/* Set hash register to InexactFilter. */
		SK_OUTHASH(IoC, &pAC->Addr.Port.InexactFilter.Words[0]);
	}

	if ((NewPromMode & SK_PROM_MODE_LLC) != 0 &&
		(CurPromMode & SK_PROM_MODE_LLC) == 0) {	/* Prom. LLC */

		/* Set promiscuous bit in mode register. */
		SK_OUT32(IoC, RECEIVE_MODE, pAC->Addr.Port.RxMode | PROMISC_MODE);
	}
	else if ((CurPromMode & SK_PROM_MODE_LLC) != 0 &&
		(NewPromMode & SK_PROM_MODE_LLC) == 0) {	/* Norm. LLC. */

		/* Clear promiscuous bit in mode register. */
		SK_OUT32(IoC, RECEIVE_MODE, pAC->Addr.Port.RxMode & ~PROMISC_MODE);
	}
	
	return (SK_ADDR_SUCCESS);
}	/* SkAddrPromiscuousChange */

#ifdef __cplusplus
}
#endif	/* __cplusplus */

